/* cy8cmbr31xx.c: Cypress CY8CMBR31xx Driver.
 *
 * Copyright 2016, 2017, 2019, 2021, 2022 Sony Corporation
 *
 * This package is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * THIS PACKAGE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/slab.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/miscdevice.h>
#include <linux/uaccess.h>
#include <linux/input.h>
#include <linux/of_irq.h>
#include <linux/i2c.h>
#include <linux/gpio.h>
#include <linux/gpio/consumer.h>
#include <linux/regulator/consumer.h>
#include <linux/input/cy8cmbr31xx.h>

#define DRV_NAME		"cy8cmbr31xx"

#define ERRLOG(fmt, ...)	pr_err(DRV_NAME ": Error: " fmt, \
					## __VA_ARGS__)
#define WRNLOG(fmt, ...)	pr_warn(DRV_NAME ": Warning: " fmt, \
					## __VA_ARGS__)
#define MSGLOG(fmt, ...)	pr_info(DRV_NAME ": " fmt, ## __VA_ARGS__)
#define DBGLOG(fmt, ...)	pr_debug(DRV_NAME ": %s(%d): " fmt, \
					__func__, __LINE__, ## __VA_ARGS__)
#define TRACE(fmt, ...)		if (p_priv->boot_trace) \
					pr_info(DRV_NAME ": %s(%d): " fmt, \
					__func__, __LINE__, ## __VA_ARGS__)

/* cy8cmbr31xx registers */
enum {
	DATA_OFFSET			=	0x00,
	SENSOR_EN			=	DATA_OFFSET,
	FSS_EN				=	0x02,
	TOGGLE_EN			=	0x04,
	LED_ON_EN			=	0x06,
	SENSITIVITY0			=	0x08,
	SENSITIVITY1			=	0x09,
	SENSITIVITY2			=	0x0a,
	SENSITIVITY3			=	0x0b,
	BASE_THRESHOLD0			=	0x0c,
	BASE_THRESHOLD1			=	0x0d,
	FINGER_THRESHOLD2		=	0x0e,
	FINGER_THRESHOLD3		=	0x0f,
	FINGER_THRESHOLD4		=	0x10,
	FINGER_THRESHOLD5		=	0x11,
	FINGER_THRESHOLD6		=	0x12,
	FINGER_THRESHOLD7		=	0x13,
	FINGER_THRESHOLD8		=	0x14,
	FINGER_THRESHOLD9		=	0x15,
	FINGER_THRESHOLD10		=	0x16,
	FINGER_THRESHOLD11		=	0x17,
	FINGER_THRESHOLD12		=	0x18,
	FINGER_THRESHOLD13		=	0x19,
	FINGER_THRESHOLD14		=	0x1a,
	FINGER_THRESHOLD15		=	0x1b,
	SENSOR_DEBOUNCE			=	0x1c,
	BUTTON_HYS			=	0x1d,
	BUTTON_LBR			=	0x1f,
	BUTTON_NNT			=	0x20,
	BUTTON_NT			=	0x21,
	PROX_EN				=	0x26,
	PROX_CFG			=	0x27,
	PROX_CFG2			=	0x28,
	PROX_TOUCH_TH0			=	0x2a,
	PROX_TOUCH_TH1			=	0x2c,
	PROX_RESOLUTION0		=	0x2e,
	PROX_RESOLUTION1		=	0x2f,
	PROX_HYS			=	0x30,
	PROX_LBR			=	0x32,
	PROX_NNT			=	0x33,
	PROX_NT				=	0x34,
	PROX_POSITIVE_TH0		=	0x35,
	PROX_POSITIVE_TH1		=	0x36,
	PROX_NEGATIVE_TH0		=	0x39,
	PROX_NEGATIVE_TH1		=	0x3a,
	LED_ON_TIME			=	0x3d,
	BUZZER_CFG			=	0x3e,
	BUZZER_ON_TIME			=	0x3f,
	GPO_CFG				=	0x40,
	PWM_DUTYCYCLE_CFG0		=	0x41,
	PWM_DUTYCYCLE_CFG1		=	0x42,
	PWM_DUTYCYCLE_CFG2		=	0x43,
	PWM_DUTYCYCLE_CFG3		=	0x44,
	PWM_DUTYCYCLE_CFG4		=	0x45,
	PWM_DUTYCYCLE_CFG5		=	0x46,
	PWM_DUTYCYCLE_CFG6		=	0x47,
	PWM_DUTYCYCLE_CFG7		=	0x48,
	SPO_CFG				=	0x4c,
	DEVICE_CFG0			=	0x4d,
	DEVICE_CFG1			=	0x4e,
	DEVICE_CFG2			=	0x4f,
	DEVICE_CFG3			=	0x50,
	I2C_ADDR			=	0x51,
	REFRESH_CTRL			=	0x52,
	STATE_TIMEOUT			=	0x55,
	SLIDER_CFG			=	0x5d,
	SLIDER1_CFG			=	0x61,
	SLIDER1_RESOLUTION		=	0x62,
	SLIDER1_THRESHOLD		=	0x63,
	SLIDER2_CFG			=	0x67,
	SLIDER2_RESOLUTION		=	0x68,
	SLIDER2_THRESHOLD		=	0x69,
	SLIDER_LBR			=	0x71,
	SLIDER_NNT			=	0x72,
	SLIDER_NT			=	0x73,
	CONFIG_CRC			=	0x7e,
	GPO_OUTPUT_STATE		=	0x80,
	SENSOR_ID			=	0x82,
	CTRL_CMD			=	0x86,
		CMD_OP_CODE_0		=	0,
		SAVE_CHECK_CRC		=	2,
		DBEUG_CHECK_CRC		=	3,
		POWER_SAVE_MODE		=	7,
		RESET_STAT		=	8,
		RESET_LPF_PS0		=	9,
		RESET_LPF_PS1		=	10,
		SW_RESET		=	255,
	CTRL_CMD_STATUS			=	0x88,
	CTRL_CMD_ERR			=	0x89,
	SYSTEM_STATUS			=	0x8a,
	PREV_CTRL_CMD_CODE		=	0x8c,
	FAMILY_ID			=	0x8f,
	DEVICE_ID			=	0x90,
	DEVICE_REV			=	0x92,
	CALC_CRC			=	0x94,
	TOTAL_WORKING_SNS		=	0x97,
	SNS_CP_HIGH			=	0x98,
	SNS_VDD_SHORT			=	0x9a,
	SNS_GND_SHORT			=	0x9c,
	SNS_SNS_SHORT			=	0x9e,
	CMOD_SHIELD_TEST		=	0xa0,
	BUTTON_STAT			=	0xaa,
	LATCHED_BUTTON_STAT		=	0xac,
	PROX_STAT			=	0xae,
	LATCHED_PROX_STAT		=	0xaf,
	SLIDER1_POSITION		=	0xb0,
	LIFTOFF_SLIDER1_POSITION	=	0xb1,
	SLIDER2_POSITION		=	0xb2,
	LIFTOFF_SLIDER2_POSITION	=	0xb3,
	SYNC_COUNTER0			=	0xb9,
	DIFFERENCE_COUNT_SENSOR0	=	0xba,
	DIFFERENCE_COUNT_SENSOR1	=	0xbc,
	DIFFERENCE_COUNT_SENSOR2	=	0xbe,
	DIFFERENCE_COUNT_SENSOR3	=	0xc0,
	DIFFERENCE_COUNT_SENSOR4	=	0xc2,
	DIFFERENCE_COUNT_SENSOR5	=	0xc4,
	DIFFERENCE_COUNT_SENSOR6	=	0xc6,
	DIFFERENCE_COUNT_SENSOR7	=	0xc8,
	DIFFERENCE_COUNT_SENSOR8	=	0xca,
	DIFFERENCE_COUNT_SENSOR9	=	0xcc,
	DIFFERENCE_COUNT_SENSOR10	=	0xce,
	DIFFERENCE_COUNT_SENSOR11	=	0xd0,
	DIFFERENCE_COUNT_SENSOR12	=	0xd2,
	DIFFERENCE_COUNT_SENSOR13	=	0xd4,
	DIFFERENCE_COUNT_SENSOR14	=	0xd6,
	DIFFERENCE_COUNT_SENSOR15	=	0xd8,
	GPO_DATA			=	0xda,
	SYNC_COUNTER1			=	0xdb,
	DEBUG_SENSOR_ID			=	0xdc,
	DEBUG_CP			=	0xdd,
	DEBUG_DIFFERENCE_COUNT0		=	0xde,
	DEBUG_BASELINE0			=	0xe0,
	DEBUG_RAW_COUNT0		=	0xe2,
	DEBUG_AVG_RAW_COUNT0		=	0xe4,
	SYNC_COUNTER2			=	0xe7
};

enum {
/* flag */
	RES_NORM		= 0,		/* Normal reset             */
	RES_RETRY		= 1,		/* Reset due to retry       */
/* timer unit:microsecond */
	TI2CBOOT		= 15000,	/* Start-up time            */
	TBOOT_SYS_WF		= 1350000,	/* System diag,EMC boottime */
	TXRES			= 5000,		/* External rst pulse width */
	PCONTRES		= 50000,	/* External rst pulse width */
	TI2C_LATENCY_MAX	= 50000,	/* Time to exec I2C command */
	TI2C_LATENCY_SAVE	= 220000,	/* SAVE_CHECK_CRC time      */
	I2C_SPEED		= 400000,	/* I2C communication speed  */
	BOOT_INTERVAL		= 20000,	/* Start wait interval      */
	WRITE_CONFIG_INTERVAL	= 300000,	/* Config write interval    */
	POLLING_INTERVAL	= 50,		/* Polling Interval (ms)　　　　*/
/* limiter */
	I2C_RETRY		= 10,		/* I2C retry count          */
	BOOT_RETRY		= TBOOT_SYS_WF/BOOT_INTERVAL,
};

enum {
	TYPE_CY8CMBR3102 = 0,
	TYPE_CY8CMBR3108,
	TYPE_CY8CMBR3110,
	TYPE_CY8CMBR3116,
	TYPE_MAX
};

/* key info */
struct key_info {
	uint16_t	key_bit;	/* Key Bits */
	uint16_t	key_code;	/* Key Code */
	char		name[16];	/* Key name */
};

/* device config */
struct key_conf {
	uint8_t		*p_regs;	/* Configration data */
	struct key_info	*p_info;	/* key information */
	int		count;		/* Valid keys */
};

/* private data */
struct priv {
	struct i2c_client	*p_client;	/* I2C client */
	struct miscdevice	misc;		/* MISC device */
	struct input_dev	*p_input;	/* Input device */
	struct delayed_work	dwork;		/* Workqueue */
	struct key_conf		key_config;	/* Config Data */
	struct gpio_desc	*gpio_xrst;	/* Reset-GPIO */
	struct gpio_desc	*gpio_intr;	/* INT-GPIO */
	struct regulator	*power;		/* Power Supply */
	bool			power_save;	/* Powersave enable */
	bool			show_crc;	/* Enable crc check */
	bool			boot_trace;	/* Enable boot-trace */
	uint8_t			type;		/* Device Type */
	uint16_t		mask;		/* Key-mask */
	uint16_t		state;		/* Ket-state */
	long			sensor_val;	/* Sensor Value */
};

static const uint16_t deviceId[TYPE_MAX] = {
	2561,		/* TYPE_CY8CMBR3102  */
	2563,		/* TYPE_CY8CMBR3108  */
	2562,		/* TYPE_CY8CMBR3110  */
	2565,		/* TYPE_CY8CMBR3116  */
};

/* Standard register setting */
static const uint8_t default_conf[] = {
	0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x80, 0x80, 0x80, 0x80,
	0x80, 0x80, 0x80, 0x80, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x03, 0x0C, 0x00, 0x32,
	0x33, 0x33, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x05, 0x00, 0x00, 0x02, 0x00, 0x02, 0x00, 0x00,
	0x05, 0x00, 0x32, 0x14, 0x14, 0x1E, 0x1E, 0x00,
	0x00, 0x1E, 0x1E, 0x00, 0x00, 0x00, 0x01, 0x01,
	0x01, 0x0F, 0x0F, 0x0F, 0x0F, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x14, 0x02, 0x01, 0x28,
	0x00, 0x37, 0x06, 0x00, 0x00, 0x0A, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x90, 0xEB
};

#define KEY_DISABLE	0xFFFF

static const struct key_info key_table[] = {
	{ 0x0001,	KEY_DISABLE,	"DISABLE"  },
	{ 0x0002,	KEY_DISABLE,	"DISABLE"  },
	{ 0x0004,	KEY_DISABLE,	"DISABLE"  },
	{ 0x0008,	KEY_DISABLE,	"DISABLE"  },
	{ 0x0010,	KEY_DISABLE,	"DISABLE"  },
	{ 0x0020,	KEY_DISABLE,	"DISABLE"  },
	{ 0x0040,	KEY_DISABLE,	"DISABLE"  },
	{ 0x0080,	KEY_DISABLE,	"DISABLE"  },
	{ 0x0100,	KEY_DISABLE,	"DISABLE"  },
	{ 0x0200,	KEY_DISABLE,	"DISABLE"  },
	{ 0x0400,	KEY_DISABLE,	"DISABLE"  },
	{ 0x0800,	KEY_DISABLE,	"DISABLE"  },
	{ 0x1000,	KEY_DISABLE,	"DISABLE"  },
	{ 0x2000,	KEY_DISABLE,	"DISABLE"  },
	{ 0x4000,	KEY_DISABLE,	"DISABLE"  },
	{ 0x8000,	KEY_DISABLE,	"DISABLE"  }
};
#define KEY_COUNT	(sizeof(key_table) / sizeof(struct key_info))

/*----------------------------------------------------------------------*/
/*	Calc CRC16-CCITT(x^16+x^12*x^5+1,left)				*/
/*----------------------------------------------------------------------*/
static uint16_t calc_crc16(uint8_t *data, int size)
{
	uint16_t	crc;
	int		i, h;

	crc = 0xFFFF;
	for (i = 0; i < size; i++) {
		crc ^= ((uint16_t)data[i] << 8);
		for (h = 0; h < 8; h++) {
			if (crc & 0x8000)
				crc = (crc << 1) ^ 0x1021;
			else
				crc <<= 1;
		}
	}
	return crc;
}

/*----------------------------------------------------------------------*/
/*	Chip I/O functions						*/
/*----------------------------------------------------------------------*/
static int read_regs(struct priv *p_priv, uint8_t cmd, void *p_data, int size)
{
	struct i2c_msg	msg[2];
	int		result;
	unsigned int	index;

	memset(&msg[0], 0, sizeof(msg));
	msg[0].addr   = p_priv->p_client->addr;
	msg[0].flags  = 0;
	msg[0].len    = 1;
	msg[0].buf    = &cmd;
	msg[1].addr   = p_priv->p_client->addr;
	msg[1].flags  = I2C_M_RD;
	msg[1].len    = size;
	msg[1].buf    = p_data;

	for (index = 0; index < I2C_RETRY; index++) {
		result = i2c_transfer(p_priv->p_client->adapter,
					&msg[0], ARRAY_SIZE(msg));
		if (result == ARRAY_SIZE(msg))
			return 0;
		usleep_range(10000, 11000);
	}

	return -EIO;
}

static inline int read_reg16(struct priv *p_priv, uint8_t cmd,
			uint16_t *p_data)
{
	uint8_t	data[2];
	int	result;

	result = read_regs(p_priv, cmd, &data[0], sizeof(data));
	if (result)
		return -EIO;

	*p_data = ((uint16_t)data[1] << 8) | data[0];

	return 0;
}

static int write_regs(struct priv *p_priv, uint8_t cmd,
		const void *p_data, int size)
{
	struct i2c_msg	msg;
	uint8_t		buff[256];
	int		result;
	unsigned int	index;

	buff[0] = cmd;
	memcpy(&buff[1], p_data, size);

	memset(&msg, 0, sizeof(msg));
	msg.addr   = p_priv->p_client->addr;
	msg.flags  = 0;
	msg.len    = 1 + size;
	msg.buf    = buff;

	for (index = 0; index < I2C_RETRY; index++) {
		result = i2c_transfer(p_priv->p_client->adapter, &msg, 1);
		if (result == 1)
			break;
		usleep_range(10000, 11000);
	}
	if (result != 1)
		return -EIO;

	return 0;
}

static inline int write_reg8(struct priv *p_priv, uint8_t cmd,
					uint8_t value)
{
	return write_regs(p_priv, cmd, &value, 1);
}

static int reset_chip(struct priv *p_priv, int restype)
{
	int	result;

	if (p_priv->gpio_xrst) {
		gpiod_set_value_cansleep(p_priv->gpio_xrst, 1);
		if (restype == RES_NORM) {
			TRACE("Hard Reset\n");
			usleep_range(TXRES, TXRES + 100);
		} else {
			usleep_range(PCONTRES, PCONTRES + 100);
		}
		gpiod_set_value_cansleep(p_priv->gpio_xrst, 0);
	} else {
		if (restype == RES_NORM)
			TRACE("Soft Reset\n");
		usleep_range(TI2CBOOT, TI2CBOOT + 100);
		result = write_reg8(p_priv, CTRL_CMD, SW_RESET);
		if (result) {
			ERRLOG("Soft Reset(%d)\n", result);
			return result;
		}
		usleep_range(TI2C_LATENCY_MAX, TI2C_LATENCY_MAX + 100);
	}

	usleep_range(TI2CBOOT, TI2CBOOT + 100);

	return 0;
}

static int wait_bootup(struct priv *p_priv)
{
	uint16_t	value;
	int		result, i;

	TRACE("Enter\n");
	for (i = 1; i < BOOT_RETRY; i++) {
		result = read_reg16(p_priv, DEVICE_ID, &value);
		if (!result) {
			if (value == deviceId[p_priv->type])
				break;
			else
				result = -EIO;
		}

		WRNLOG("Retry:%d\n", i);
		usleep_range(BOOT_INTERVAL, BOOT_INTERVAL + 100);
	}

	return result;
}

/*----------------------------------------------------------------------*/
/*	program register settings					*/
/*----------------------------------------------------------------------*/
static int initial_setting(struct priv *p_priv, uint8_t *p_conf)
{
	uint16_t	value, crc;
	int		result;

	TRACE("Enter\n");
	reset_chip(p_priv, RES_NORM);
	result = wait_bootup(p_priv);
	if (result) {
		ERRLOG("Busy Chip boot(%d)\n", result);
		return result;
	}

	crc = calc_crc16(p_conf, 126);
	TRACE("Setting CRC16=0x%4x\n", crc);
	p_conf[CONFIG_CRC]     = (uint8_t)(crc & 0xFF);
	p_conf[CONFIG_CRC + 1] = (uint8_t)((crc & 0xFF00) >> 8);

	result = read_reg16(p_priv, CONFIG_CRC, &value);
	if (result) {
		ERRLOG("CRC read(%d)\n", result);
		return result;
	}
	TRACE("Writted CRC16=0x%4x\n", value);
	if (crc == value) {
		MSGLOG("Skip write configdata.\n");
		return 0;
	}

	/* Set configration data */
	DBGLOG("Set configration data\n");
	result = write_regs(p_priv, DATA_OFFSET, p_conf, 128);
	if (result) {
		ERRLOG("Config data write(%d)\n", result);
		return result;
	}
	usleep_range(TI2C_LATENCY_MAX, TI2C_LATENCY_MAX + 100);

	/* Save configration data */
	TRACE("Save configration data\n");
	result = write_reg8(p_priv, CTRL_CMD, SAVE_CHECK_CRC);
	if (result) {
		ERRLOG("Flush write(%d)\n", result);
		return result;
	}
	usleep_range(TI2C_LATENCY_SAVE, TI2C_LATENCY_SAVE + 100);

	if (p_priv->show_crc) {
		TRACE("Show calculate crc value\n");
		result = write_reg8(p_priv, CTRL_CMD, DBEUG_CHECK_CRC);
		MSGLOG("DBEUG_CHECK_CRC(%d)\n", result);
		usleep_range(TI2C_LATENCY_MAX, TI2C_LATENCY_MAX + 100);

		result = read_reg16(p_priv, CALC_CRC, &value);
		MSGLOG("CALC_CRC_VALUE(%d), 0x%02x,0x%02x\n", result,
				(uint8_t)value, (uint8_t)(value >> 8));

		ERRLOG("Show calc crc mode\n");
		return -EIO;
	}

	/* Get status */
	TRACE("Get Status Start\n");
	result = read_reg16(p_priv, CTRL_CMD_STATUS, &value);
	if (result) {
		ERRLOG("Get Status(%d)\n", result);
		return result;
	}
	if (value & 0x0001) {
		ERRLOG("ERROR CODE(%d)\n", (value >> 8));
		return -EIO;
	}

	usleep_range(WRITE_CONFIG_INTERVAL, WRITE_CONFIG_INTERVAL + 100);

	/* Soft-Reset */
	TRACE("Soft Reset Start\n");
	result = write_reg8(p_priv, CTRL_CMD, SW_RESET);
	if (result) {
		ERRLOG("Soft Reset(%d)\n", result);
		return result;
	}
	usleep_range(TI2C_LATENCY_MAX, TI2C_LATENCY_MAX + 100);

	DBGLOG("update configdata done\n");
	return 0;
}

/*----------------------------------------------------------------------*/
/*	key check proc							*/
/*----------------------------------------------------------------------*/
static void cy8cmbr31xx_workqueue_proc(struct work_struct *p_work)
{
	struct delayed_work *p_data;
	struct priv	*p_priv;
	struct key_conf	*p_conf;
	struct key_info	*p_info;
	uint16_t	state, update, regAddr = 0;
	uint16_t	key_code, value;
	int		result;
	unsigned int	index;

	DBGLOG("Enter\n");
	p_data = container_of(p_work, struct delayed_work, work);
	p_priv = container_of(p_data, struct priv, dwork);
	p_conf = &p_priv->key_config;

	do {
		/* Get button status */
		result = read_reg16(p_priv, BUTTON_STAT, &state);
		if (result) {
			WRNLOG("Communication failed\n");
			reset_chip(p_priv, RES_RETRY);
			break;
		}
		state &= p_priv->mask;

		/* Ignore if there is no change in state */
		update = state ^ p_priv->state;
		if (!update)
			break;
		p_priv->state = state;

		/* Change key status */
		for (index = 0; index < KEY_COUNT; index++) {
			p_info = &p_conf->p_info[index];
			key_code = p_info->key_code;

			if (update & p_info->key_bit) {
				regAddr =
					DIFFERENCE_COUNT_SENSOR0 + (index << 1);
				result = read_reg16(p_priv,
							regAddr, &value);
				if (!result)
					p_priv->sensor_val = (value & 0xFF);

				if ((state & p_info->key_bit) != 0) {
					MSGLOG("key down key_code=0x%x (%s)\n",
						key_code, p_info->name);
					input_event(p_priv->p_input,
						EV_KEY, key_code, true);
				} else {
					MSGLOG("key up key_code=0x%x (%s)\n",
						key_code, p_info->name);
					input_event(p_priv->p_input,
						EV_KEY, key_code, false);
				}
				update &= ~p_info->key_bit;
				if (!update)
					break;
			}
		}
		input_sync(p_priv->p_input);
	} while (0);

	/* Restart delayed work queue */
	if (!p_priv->gpio_intr)
		schedule_delayed_work(&p_priv->dwork,
			msecs_to_jiffies(POLLING_INTERVAL));
}

/*----------------------------------------------------------------------*/
/*	IRQ entry							*/
/*----------------------------------------------------------------------*/
static irqreturn_t cy8cmbr31xx_irq(int irq_num, void *data)
{
	struct priv	*p_priv = data;

	DBGLOG("IRQ entry\n");
	schedule_delayed_work(&p_priv->dwork, 0);

	return IRQ_HANDLED;
}

/*----------------------------------------------------------------------*/
/*	misc device open						*/
/*----------------------------------------------------------------------*/
static int cy8cmbr31xx_open(struct inode *p_inode, struct file *p_file)
{
	DBGLOG("Enter\n");
	p_file->private_data = container_of(p_file->private_data,
						struct priv, misc);

	return 0;
}

/*----------------------------------------------------------------------*/
/*	IOCTL functions							*/
/*----------------------------------------------------------------------*/
static long cy8cmbr31xx_ioctl(struct file *p_file,
				 unsigned int cmd, unsigned long arg)
{
	struct priv	*p_priv = p_file->private_data;
	struct key_conf	*p_conf = &p_priv->key_config;
	struct key_info	*p_info;
	uint32_t	buff[KEY_COUNT];
	unsigned int	index, count;

	DBGLOG("Enter\n");
	switch (cmd) {
	case ICX_KEY_IOCTL_CMD_GETCOUNT:
		return p_conf->count;

	case ICX_KEY_IOCTL_CMD_GETINFO:
		for (index = count = 0; index < KEY_COUNT; index++) {
			p_info = &p_conf->p_info[index];
			if (p_info->key_code != KEY_DISABLE)
				buff[count++] = p_info->key_code;
		}

		if (copy_to_user((void __user *)arg,
			buff, sizeof(u32) * count))
			return -EFAULT;
		break;

	case ICX_KEY_IOCTL_CMD_GET_TOUCH_SENSOR_VAL:
		return p_priv->sensor_val;

	default:
		return -EINVAL;
	}

	return 0;
}

static const struct file_operations cy8cmbr31xx_fops = {
	.owner		= THIS_MODULE,
	.open		= cy8cmbr31xx_open,
	.unlocked_ioctl	= cy8cmbr31xx_ioctl,
	.compat_ioctl	= cy8cmbr31xx_ioctl,
};

/*----------------------------------------------------------------------*/
/*     parse device tree						*/
/*----------------------------------------------------------------------*/
static void cy8cmbr31xx_parse_dt(struct device_node *node,
					struct key_conf *p_conf)
{
	const uint8_t *prop, *end_prop;
	const __be32 *prop32;
	unsigned int proplen;
	int index, len;
	uint16_t key_code;
	char *name;

	/* Get device-tree property */
	prop = of_get_property(node, "key-map", &proplen);
	if (!prop)
		return;

	end_prop = prop + proplen;
	while (prop < (end_prop - 9)) {
		prop32 = (__be32 *)prop;
		index = be32_to_cpup(prop32);
		key_code = (uint16_t)be32_to_cpup(prop32 + 1);
		prop += sizeof(__be32) * 2;
		len = strnlen(prop, 16);
		name = (char *)prop;
		prop += (len + 1);
		if (index < KEY_COUNT) {
			p_conf->p_info[index].key_code = key_code;
			strncpy(p_conf->p_info[index].name, name,
				sizeof(p_conf->p_info[index].name)-1);
			DBGLOG("index:0x%04x key_code:0x%x name:%s\n",
				index, key_code, name);
		}
	}
}

/*----------------------------------------------------------------------*/
/*	platform device Probe						*/
/*----------------------------------------------------------------------*/
static int cy8cmbr31xx_probe(struct i2c_client *p_client,
				const struct i2c_device_id *p_id)
{
	struct device	*dev = &p_client->dev;
	struct key_conf	*p_conf;
	struct key_info	*p_info;
	struct priv	*p_priv;
	struct device_node *node;
	struct regulator *power;
	int		result;
	unsigned int	index;

	DBGLOG("Start cy8cmbr31xx touch-key driver\n");

	power = devm_regulator_get(dev, "power");
	if (IS_ERR(power)) {
		WRNLOG("no power-supply(%ld)\n", PTR_ERR(power));
		if (PTR_ERR(power) == -EPROBE_DEFER)
			return -EPROBE_DEFER;
		power = NULL;
	}

	/* Allocate provate data memory */
	p_priv = devm_kzalloc(dev, sizeof(struct priv), GFP_KERNEL);
	p_conf = &p_priv->key_config;

	/* Allocate config data memory */
	p_conf->p_regs = devm_kzalloc(dev, sizeof(default_conf), GFP_KERNEL);
	if (p_conf->p_regs == NULL) {
		ERRLOG("Allocate config data memory\n");
		return -ENOMEM;
	}
	/* Allocate key table memory */
	p_conf->p_info = devm_kzalloc(dev, sizeof(key_table), GFP_KERNEL);
	if (p_conf->p_info == NULL) {
		ERRLOG("Allocate key table memory\n");
		return -ENOMEM;
	}

	/* Initialize provate data */
	i2c_set_clientdata(p_client, p_priv);
	p_priv->p_client = p_client;
	p_priv->gpio_xrst = NULL;
	p_priv->power    = power;
	p_priv->type     = (int)p_id->driver_data;
	p_priv->mask     = 0;
	p_priv->state    = 0;
	memcpy(p_conf->p_regs, default_conf, sizeof(default_conf));
	memcpy(p_conf->p_info, key_table, sizeof(key_table));
	p_conf->count    = 0;

	/* Get device-tree property */
	node = dev->of_node;
	p_priv->boot_trace = of_property_read_bool(node, "boot-trace");
	if (p_priv->boot_trace)
		DBGLOG("boot-trace enable\n");

	cy8cmbr31xx_parse_dt(node, p_conf);

	p_priv->show_crc = of_property_read_bool(node, "crc-check");
	if (p_priv->show_crc)
		DBGLOG("CRC check mode\n");

	p_priv->power_save = of_property_read_bool(node, "power-save");
	if (p_priv->power_save) {
		DBGLOG("power-save enable\n");
		if (device_may_wakeup(&p_client->dev))
			WRNLOG("wakeup-source cannot use with power-save\n");
	}

	p_priv->gpio_xrst = devm_gpiod_get(dev, "xrst", GPIOD_OUT_LOW);
	if (IS_ERR(p_priv->gpio_xrst)) {
		result = PTR_ERR(p_priv->gpio_xrst);
		WRNLOG("xrst_gpios pin is disabled\n");
		p_priv->gpio_xrst = NULL;
	}

	p_priv->gpio_intr = devm_gpiod_get(dev, "intr", GPIOD_IN);
	if (IS_ERR(p_priv->gpio_intr)) {
		result = PTR_ERR(p_priv->gpio_intr);
		WRNLOG("intr-gpios pin is disabled, Works in polling-mode\n");
		p_priv->gpio_intr = NULL;
		if (device_may_wakeup(&p_client->dev))
			WRNLOG("wakeup-source cannot use with polling-mode\n");
	}

	/* Create misd decice */
	TRACE("Create misc device\n");
	p_priv->misc.minor = MISC_DYNAMIC_MINOR;
	p_priv->misc.name  = DRV_NAME;
	p_priv->misc.fops  = &cy8cmbr31xx_fops;
	result             = misc_register(&p_priv->misc);
	if (result) {
		ERRLOG("Create misc device(%d)\n", result);
		return result;
	}

	/* Allocate input device */
	TRACE("Allocate input device\n");
	p_priv->p_input = input_allocate_device();
	if (p_priv->p_input == NULL) {
		ERRLOG("Allocate input device\n");
		result = -ENOMEM;
		goto exit_misc;
	}

	/* Register input device */
	TRACE("Register input device\n");
	p_priv->p_input->name       = DRV_NAME;
	p_priv->p_input->phys       = DRV_NAME;
	p_priv->p_input->id.bustype = BUS_HOST;
	p_priv->p_input->id.vendor  = 0x0001;
	p_priv->p_input->id.product = 0x0001;
	p_priv->p_input->id.version = 0x0100;
	p_priv->p_input->dev.parent = &p_client->dev;
	set_bit(EV_KEY, p_priv->p_input->evbit);
	for (index = 0; index < KEY_COUNT; index++) {
		p_info = &p_conf->p_info[index];
		if (p_info->key_code != KEY_DISABLE) {
			set_bit(p_info->key_code, p_priv->p_input->keybit);
			p_priv->mask |= p_info->key_bit;
			p_conf->count++;
		}
	}
	p_conf->p_regs[SENSOR_EN] = (p_priv->mask & 0xFF);
	p_conf->p_regs[SENSOR_EN + 1] = ((p_priv->mask >> 8) & 0xFF);
	TRACE("SENSOR_EN:0x%04x\n", p_conf->p_regs[SENSOR_EN]);

	result = input_register_device(p_priv->p_input);
	if (result) {
		input_free_device(p_priv->p_input);
		ERRLOG("Register input device(%d)\n", result);
		goto exit_misc;
	}

	/* Create delayed Workqueue */
	INIT_DELAYED_WORK(&p_priv->dwork, cy8cmbr31xx_workqueue_proc);

	if (p_priv->power) {
		MSGLOG("power-supply enable\n");
		if (regulator_enable(p_priv->power))
			WRNLOG("Regulator Failed\n");
	}

	/* Initial setting for */
	TRACE("Initialize cy8cmbr31xx\n");
	result = initial_setting(p_priv, &p_conf->p_regs[0]);
	if (result)	/* Retry */
		result = initial_setting(p_priv, &p_conf->p_regs[0]);
	if (result) {
		ERRLOG("Program config data\n");
		goto exit_input;
	}

	/* Set irq handler or Start polling */
	if (p_priv->gpio_intr) {
		MSGLOG("Set irq handler\n");
		p_client->irq = gpiod_to_irq(p_priv->gpio_intr);
		TRACE("chip_irq=%d\n", p_client->irq);
		result = devm_request_irq(dev, p_client->irq,
				cy8cmbr31xx_irq,
				IRQF_TRIGGER_FALLING | IRQF_SHARED,
				dev_name(dev), p_priv);
		if (result) {
			ERRLOG("Request irq handler(%d)\n", result);
			goto exit_input;
		}
	} else {
		MSGLOG("Start polling\n");
		schedule_delayed_work(&p_priv->dwork, 0);
	}

	DBGLOG("probe done\n");
	return 0;

exit_input:
	input_unregister_device(p_priv->p_input);
exit_misc:
	misc_deregister(&p_priv->misc);

	ERRLOG("probe error:%d\n", result);
	return result;
}

/*----------------------------------------------------------------------*/
/*	platform device  Remove						*/
/*----------------------------------------------------------------------*/
static int cy8cmbr31xx_remove(struct i2c_client *p_client)
{
	struct device	*dev = &p_client->dev;
	struct priv	*p_priv = i2c_get_clientdata(p_client);

	DBGLOG("Enter\n");
	devm_free_irq(dev, p_client->irq, p_priv);
	cancel_delayed_work_sync(&p_priv->dwork);
	input_unregister_device(p_priv->p_input);
	misc_deregister(&p_priv->misc);

	if (p_priv->power) {
		MSGLOG("power-supply disable\n");
		if (regulator_disable(p_priv->power))
			WRNLOG("Regulator Failed\n");
	}

	MSGLOG("remove done\n");
	return 0;
}

#ifdef CONFIG_PM_SLEEP
/* -------------------------------------------------------------------- */
/*	platform device suspend						*/
/* -------------------------------------------------------------------- */
static int cy8cmbr31xx_suspend(struct device *dev)
{
	struct i2c_client *p_client = to_i2c_client(dev);
	struct priv	  *p_priv = i2c_get_clientdata(p_client);
	int		  result;

	DBGLOG("Enter\n");
	cancel_delayed_work_sync(&p_priv->dwork);

	if (p_priv->power_save) {
		result = write_reg8(p_priv, CTRL_CMD, POWER_SAVE_MODE);
		if (!result)
			MSGLOG("enter power save mode\n");
		else
			ERRLOG("enter power save mode\n");
	}

	if (p_priv->gpio_intr) {
		if (device_may_wakeup(&p_client->dev)) {
			enable_irq_wake(p_client->irq);
			MSGLOG("enable_irq_wake(%d)\n", p_client->irq);
		}
	}

	MSGLOG("suspend done\n");
	return 0;
}

/* -------------------------------------------------------------------- */
/*	platform device resume						*/
/* -------------------------------------------------------------------- */
static int cy8cmbr31xx_resume(struct device *dev)
{
	struct i2c_client *p_client = to_i2c_client(dev);
	struct priv	  *p_priv = i2c_get_clientdata(p_client);
	int		  result;

	DBGLOG("Enter\n");

	if (p_priv->gpio_intr) {
		if (device_may_wakeup(&p_client->dev)) {
			disable_irq_wake(p_client->irq);
			MSGLOG("disable_irq_wake(%d)\n", p_client->irq);
		}
	}

	if (p_priv->power_save) {
		result = write_reg8(p_priv, CTRL_CMD, CMD_OP_CODE_0);
		if (!result)
			MSGLOG("cancel power save mode\n");
		else
			ERRLOG("cancel power save mode\n");
	}

	/* Restart work queue */
	if (!p_priv->gpio_intr)
		schedule_delayed_work(&p_priv->dwork, 0);

	MSGLOG("resume done\n");
	return 0;
}
#endif /* CONFIG_PM_SLEEP */

/* -------------------------------------------------------------------- */
/*	Configration i2c device driver					*/
/* -------------------------------------------------------------------- */
static SIMPLE_DEV_PM_OPS(cy8cmbr31xx_pm_ops,
			 cy8cmbr31xx_suspend, cy8cmbr31xx_resume);

static const struct of_device_id cy8cmbr31xx_dt_ids[] = {
	{ .compatible = "cypress,cy8cmbr3102", },
	{ .compatible = "cypress,cy8cmbr3108", },
	{ .compatible = "cypress,cy8cmbr3110", },
	{ .compatible = "cypress,cy8cmbr3116", },
	{}
};
MODULE_DEVICE_TABLE(of, cy8cmbr31xx_dt_ids);

static const struct i2c_device_id cy8cmbr31xx_id[] = {
	{ "cy8cmbr3102",  TYPE_CY8CMBR3102  },
	{ "cy8cmbr3108",  TYPE_CY8CMBR3108  },
	{ "cy8cmbr3110",  TYPE_CY8CMBR3110  },
	{ "cy8cmbr3116",  TYPE_CY8CMBR3116  },
	{}
};
MODULE_DEVICE_TABLE(i2c, cy8cmbr31xx_id);

static struct i2c_driver cy8cmbr31xx_driver = {
	.probe     = cy8cmbr31xx_probe,
	.remove    = cy8cmbr31xx_remove,
	.id_table  = cy8cmbr31xx_id,
	.driver    = {
		.name  = DRV_NAME,
		.owner = THIS_MODULE,
		.pm	= &cy8cmbr31xx_pm_ops,
		.of_match_table = cy8cmbr31xx_dt_ids,
	},
};

module_i2c_driver(cy8cmbr31xx_driver);
MODULE_DESCRIPTION("CY8CMBR31xx Driver");
MODULE_AUTHOR("Sony Home Entertainment & Sound Products Inc.");
MODULE_LICENSE("GPL");
MODULE_VERSION("v1.00");
