/* SPDX-License-Identifier: GPL-2.0 */
/*
 * cec-driver-adapter.c - CecDriverAdapter
 *
 * Copyright 2019 Sony Home Entertainment & Sound Products Inc.
 */

#include <linux/kernel.h>
#include <linux/module.h>           // THIS_MODULE
#include <linux/slab.h>             // kfree, kzalloc
#include <linux/err.h>
#include <linux/cdev.h>             // cdev_init, cdev_add, cdev_del
#include <linux/platform_device.h>  // platform_device
#include <linux/uaccess.h>          // copy_to_user
#include <linux/semaphore.h>        // DEFINE_SEMAPHORE
#include <linux/cm4.h>
#include <linux/mod_devicetable.h>  // of_device_id
#include <media/cec-driver-adapter.h>

#define _CEC_DRIVER_ADAPTER_C_

#define DRV_NAME                "cec-driver-adapter"
#define ELOG(fmt, ...)          pr_err(DRV_NAME ": Error: %s(%d): " fmt, __func__, __LINE__, ## __VA_ARGS__)
#define WLOG(fmt, ...)          pr_warn(DRV_NAME ": Warning: %s(%d): " fmt, __func__, __LINE__, ## __VA_ARGS__)
#define ILOG(fmt, ...)          pr_info(DRV_NAME ": %s(%d): " fmt, __func__, __LINE__, ## __VA_ARGS__)
#define DLOG(fmt, ...)          pr_debug(DRV_NAME ": %s(%d): " fmt, __func__, __LINE__, ## __VA_ARGS__)
//#define DLOG(fmt, ...)          pr_err(DRV_NAME ": %s(%d): " fmt, __func__, __LINE__, ## __VA_ARGS__)



/*******************************************************************************
 * Definition
*******************************************************************************/
#define CEC_RET_OK                  (0)

#define CEC_DRIVER_NAME             "cec-driver-adapter"
#define CEC_DRIVER_CLASS            "cec-driver-adapter_class"
#define CEC_DRIVER_MINOR_BASE       (0)
#define CEC_DRIVER_MINOR_NUM        (1)

#define CEC_RPMSG_CM4_TYPE_REQUEST  (1)
#define CEC_RPMSG_CM4_TYPE_RESPONSE (2)

// CM4<->A53 communication command
#define CEC_HANDSHAKE_REQ               (0x00)  /* CM4<->A53 */
#define CEC_NOTIFY_CEC_IN               (0x01)  /* CM4-->A53 */
#define CEC_CEC_TRAN_REQ                (0x02)  /* CM4<--A53 */
#define CEC_NOTIFY_CEC_OUT_RESULT       (0x03)  /* CM4-->A53 */
#define CEC_SET_RETRY_COUNT             (0x04)  /* CM4<--A53 */
#define CEC_SET_LOGICAL_ADDRESS         (0x05)  /* CM4<--A53 */
#define CEC_NOTIFY_HDMI_CONTROL_CHANGE  (0x06)  /* CM4<--A53 */
#define CEC_SET_FILTER_CEC_IN           (0x07)  /* CM4<--A53 */

// CM4<->A53 communication command size
#define CEC_CM4_A53_HEADER_SIZE         (2)
#define CEC_HANDSHAKE_SIZE              (CEC_CM4_A53_HEADER_SIZE)
#define CEC_SEQUENCE_ID_SIZE            (1)
#define CEC_HEADER_SIZE                 (1)
#define CEC_OPCODE_SIZE                 (1)
#define CEC_COMMAND_SIZE_MAX            (CEC_HEADER_SIZE + CEC_OPCODE_SIZE + CEC_OPERAND_SIZE_MAX)
#define CEC_RECEIVE_COMMAND_SIZE_MIN    (CEC_CM4_A53_HEADER_SIZE + CEC_HEADER_SIZE)
#define CEC_SEND_COMMAND_SIZE_MIN       (CEC_CM4_A53_HEADER_SIZE + CEC_SEQUENCE_ID_SIZE + CEC_HEADER_SIZE)
#define CEC_SEND_COMMAND_SIZE_MAX       (CEC_CM4_A53_HEADER_SIZE + CEC_SEQUENCE_ID_SIZE + CEC_COMMAND_SIZE_MAX)
#define CEC_LOGICAL_ADDRESS_SIZE        (2)
#define CEC_LOGICAL_ADDRESS_SEND_SIZE   (CEC_CM4_A53_HEADER_SIZE + CEC_LOGICAL_ADDRESS_SIZE)
#define CEC_HDMI_CONTROL_SIZE           (1)
#define CEC_HDMI_CONTROL_SEND_SIZE      (CEC_CM4_A53_HEADER_SIZE + CEC_HDMI_CONTROL_SIZE)
#define CEC_SET_RETRY_CNT_SIZE          (1)
#define CEC_SET_RETRY_CNT_SEND_SIZE     (CEC_CM4_A53_HEADER_SIZE + CEC_SET_RETRY_CNT_SIZE)
#define CEC_SEND_COMMAND_RESULT_SIZE    (1)
#define CEC_SEND_COMMAND_RET_SIZE_MIN   (CEC_CM4_A53_HEADER_SIZE + CEC_SEQUENCE_ID_SIZE + CEC_SEND_COMMAND_RESULT_SIZE)
#define CEC_SET_FILTER_CEC_IN_SIZE      (1)
#define CEC_SET_FILTER_CEC_IN_SEND_SIZE (CEC_CM4_A53_HEADER_SIZE + CEC_SET_FILTER_CEC_IN_SIZE)

// CM4<->A53 communication command parameter
#define CEC_SEND_CMD_RET_ACK_OK           (0)
#define CEC_SEND_CMD_RET_NO_ACK_ERROR     (1)
#define CEC_SEND_CMD_RET_ACK_NG_ERROR     (2)
#define CEC_SEND_CMD_RET_ARBITRATION_LOST (3)
#define CEC_SEND_CMD_RET_TIME_OVER_ERROR  (4)


#define CEC_COMMAND_BUFFER_SIZE         (10)
#define CEC_LOGICAL_ADDRESS_MEMORY_SIZE (3)


static const char* cec_env_state_get_cmd = "CEC_STATE=3";   // HDMI_CEC_STATE_GET_CMD
static const char* cec_env_state_sent_cmd = "CEC_STATE=2";  // HDMI_CEC_STATE_TX_STS

struct cec_driver_adapter_dev {
  dev_t                   dev_id;  // kdev_t.hƁAxxxxxxxxxxxxyyyyyyyyyyyyyyyyyyyybŃW[=xC}Ci[=y
  struct cdev             cdev;
  struct class            *class;
  struct device           *dev;
};


typedef struct CEC_RECEIVE_CEC_BUFFER {
  unsigned char           read_ite;
  unsigned char           write_ite;
  bool                    is_full;
  CEC_FRAME_BLOCK_IO      buf[CEC_COMMAND_BUFFER_SIZE];
} cec_receive_cec_buffer;

typedef struct CEC_SAVE_LA_MEMORY {
  unsigned char           save_ite;
  unsigned char           logical_address[CEC_LOGICAL_ADDRESS_MEMORY_SIZE];  // FILOő삷
} cec_save_la_memory;

// ŊǗ
typedef struct CEC_DRIVER_ADAPTER_MANAGE_INFO {
  bool                    handshake_completed;
  u8                      seq;
  cec_receive_cec_buffer  cec_in;
  CEC_SEND_MSG            cec_out;
  CEC_ACK_INFO            cec_ack;
  cec_save_la_memory      la;
} cec_driver_adapter_manage_info;


static int cec_register_probe(struct platform_device *pdev);
static long cec_driver_adapter_ioctl(struct file *filp, unsigned int cmd, unsigned long arg);
static long cec_driver_adapter_compat_ioctl(struct file *filp, unsigned int cmd, unsigned long arg);

int cec_cm4_send(u8 type, u8 length, u8 *data);
void cec_cm4_callback(u8 type, u8 length, u8 seq, u8 *data);
static void cec_cm4_cb_cec_in(const char **envp, u8 length, u8 *data);
static void cec_cm4_cb_cec_out_result(const char **envp, u8 length, u8 *data);
static void cec_notify_to_service(const char *envp);

static signed int cec_ioctl_nop( unsigned long arg );
static signed int cec_ioctl_get_cec_status( unsigned long arg );
static signed int cec_ioctl_set_cec_command( unsigned long arg );
static signed int cec_ioctl_set_logical_address( unsigned long arg );
static signed int cec_ioctl_get_cec_command( unsigned long arg );
static signed int cec_ioctl_hdmi_control_enable( unsigned long arg );
static signed int cec_ioctl_set_retry_cnt( unsigned long arg );
static signed int cec_ioctl_set_filter_cec_in( unsigned long arg );

/*******************************************************************************
 * Variables
*******************************************************************************/

static const struct of_device_id cec_dt_ids[] = {
    { .compatible = "sony,cec-driver-adapter", },
    { },
};
MODULE_DEVICE_TABLE(of, cec_dt_ids);

static struct platform_driver cec_driver_register = {
    .driver = {
        .name = DRV_NAME,
        .owner = THIS_MODULE,
        .of_match_table = cec_dt_ids,
    },
    .probe = cec_register_probe,
};

static struct cec_driver_adapter_dev *cec_driver_adapter_dev_info;
static struct device *cec_dev = NULL;
static cec_driver_adapter_manage_info cec_manage_info;

static const struct file_operations cec_driver_adapter_fops = {
  .owner          = THIS_MODULE,
  .unlocked_ioctl = cec_driver_adapter_ioctl,
  .compat_ioctl   = cec_driver_adapter_compat_ioctl,
};


static DEFINE_SEMAPHORE(cec_mutex);


/*******************************************************************************
 * Functions
*******************************************************************************/

/* *****************************************************************
 *  ֐̏ڍ  Fprobe
 *  param       F[IN]  pdev vbgtH[foCX̃|C^
 *  return      FCEC_RET_OK/ERR ID
 * ***************************************************************** */
static int cec_register_probe(struct platform_device *pdev)
{
  int ret;
  u8 handshake[2];

  DLOG("%s 4 \n", __func__);

  ret = rpmsg_cm4_register_callback(CM4_MESSAGE_ID_CEC, cec_cm4_callback);
  if (ret != 0) {
    ELOG("regist callback failed(%d)\n", ret);
    return -EINVAL;
  }

  cec_dev = &pdev->dev;
  if (cec_dev == NULL) {
    ELOG("dev is NULL \n");
    return -EINVAL;
  }

  // handshakeĂ݂
  handshake[0] = CEC_HANDSHAKE_REQ;
  handshake[1] = 0;
  ret = cec_cm4_send(CEC_RPMSG_CM4_TYPE_REQUEST, CEC_HANDSHAKE_SIZE, handshake);
  if (ret != CEC_RET_OK) {
    ELOG("cec_cm4_send failed(%d)\n", ret);
    return ret;
  }

  return CEC_RET_OK;
}

/* *****************************************************************
 *  ֐̏ڍ  Funlocked_ioctl
 *  param       F[IN]  *file 
 *              F[IN]  cmd
 *              F[IN]  arg
 *  return      FCEC_RET_OK/ERR ID
 * ***************************************************************** */
static long cec_driver_adapter_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
  int ret = CEC_RET_OK;
  
  if (down_interruptible(&cec_mutex))
  {
      ELOG("[CEC]warning: down err\n!");
  }
  // HandshakẽgC͂ł͂ȂʓrKv ::TBD::
  if (false == cec_manage_info.handshake_completed) {
    u8 handshake[CEC_HANDSHAKE_SIZE];
    // handshakeĂ݂
    handshake[0] = CEC_HANDSHAKE_REQ;
    handshake[1] = 0;
    ret = cec_cm4_send(CEC_RPMSG_CM4_TYPE_REQUEST, CEC_HANDSHAKE_SIZE, handshake);
    if (ret != CEC_RET_OK) {
      ELOG("cec_cm4_send failed(%d)\n", ret);
      // return ret;
    }
    DLOG("%s send handshake", __func__);
  }

  DLOG("%s cmd=%d cmd&0xff=%d", __func__, cmd, (cmd & 0xff));

  switch(cmd) {
  case CEC_POWER_ENABLE:          ret = cec_ioctl_nop(arg);                   break;  // 
  case CEC_GET_CECSTS:            ret = cec_ioctl_get_cec_status(arg);        break;  // CEC_NOTIFY_CEC_OUT_RESULT
  case CEC_SET_CECCMD:            ret = cec_ioctl_set_cec_command(arg);       break;  // CEC_CEC_TRAN_REQ
  case CEC_SETLA:                 ret = cec_ioctl_set_logical_address(arg);   break;  // CEC_SET_LOGICAL_ADDRESS
  case CEC_GET_CECCMD:            ret = cec_ioctl_get_cec_command(arg);       break;  // CEC_NOTIFY_CEC_IN
  case CEC_HDMI_CONTROL_ENABLE:   ret = cec_ioctl_hdmi_control_enable(arg);   break;  // CEC_NOTIFY_HDMI_CONTROL_CHANGE
  case CEC_SET_RETRY_CNT:         ret = cec_ioctl_set_retry_cnt(arg);         break;  // CEC_SET_RETRY_COUNT
  case CEC_FILTER_CEC_RECEIVE:    ret = cec_ioctl_set_filter_cec_in(arg);     break;  // CEC_SET_FILTER_CEC_IN
  default:
    ELOG("unknown command %d(0x%02x) \n", (cmd & 0xff), (cmd & 0xff));
    ret = -EFAULT;
    break;
  }
  
  up(&cec_mutex);
  return ret;
}

/* *****************************************************************
 *  ֐̏ڍ  Fcompat_ioctl
 *  param       F[IN]  *file 
 *              F[IN]  cmd
 *              F[IN]  arg
 *  return      FCEC_RET_OK/ERR ID
 * ***************************************************************** */
static long cec_driver_adapter_compat_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
  DLOG("%s \n", __func__);
  cec_driver_adapter_ioctl(filp, cmd, arg);
  return 0;
}

/* *****************************************************************
 *  ֐̏ڍ  Fcm4ւ̒ʒm
 *  param       F[IN]  type   R}h^Cv(REQUEST/RESPONSE)
 *              F[IN]  length datãTCY
 *              F[IN]  data   f[^
 *  return      FCEC_RET_OK/ERR ID
 * ***************************************************************** */
int cec_cm4_send(u8 type, u8 length, u8 *data)
{
  int ret;
  //int i = 0;

  if ((data == NULL) || (length < 2)){
    ELOG("param error data=%p length=%d \n", data, length);
    return -EINVAL;
  }

  DLOG("send msg type=%d length=%d seq=%d\n", type, length, cec_manage_info.seq);
  //for (i = 0; i < length; i+=8) {
  //  switch(length - i) {
  //  case 0:                                                                                                   break;
  //  case 1:   DLOG("data[%2d] = %02x  \n", i, data[i+0]);                                                     break;
  //  case 2:   DLOG("data[%2d] = %02x %02x \n", i, data[i+0], data[i+1]);                                      break;
  //  case 3:   DLOG("data[%2d] = %02x %02x %02x \n", i, data[i+0], data[i+1], data[i+2]);                      break;
  //  case 4:   DLOG("data[%2d] = %02x %02x %02x %02x \n", i, data[i+0], data[i+1], data[i+2], data[i+3]);      break;
  //  case 5:   DLOG("data[%2d] = %02x %02x %02x %02x   %02x \n", i,
  //                  data[i+0], data[i+1], data[i+2], data[i+3], data[i+4]);                                   break;
  //  case 6:   DLOG("data[%2d] = %02x %02x %02x %02x   %02x %02x \n", i,
  //                  data[i+0], data[i+1], data[i+2], data[i+3], data[i+4], data[i+5]);                        break;
  //  case 7:   DLOG("data[%2d] = %02x %02x %02x %02x   %02x %02x %02x \n", i,
  //                  data[i+0], data[i+1], data[i+2], data[i+3], data[i+4], data[i+5], data[i+6]);             break;
  //  default:  DLOG("data[%2d] = %02x %02x %02x %02x   %02x %02x %02x %02x \n", i,
  //                  data[i+0], data[i+1], data[i+2], data[i+3], data[i+4], data[i+5], data[i+6], data[i+7]);  break;
  //  }
  //}

  ret = rpmsg_cm4_send(CM4_MESSAGE_ID_CEC, type, length, cec_manage_info.seq, data);
  if (ret != 0) {
    ELOG("rpmsg_cm4_send failed(%d)\n", ret);
    return -EINVAL;
  }
  cec_manage_info.seq++;
  return CEC_RET_OK;
}

/* *****************************************************************
 *  ֐̏ڍ  Fcm4̒ʒm
 *  param       F[IN]  type   R}h^Cv(REQUEST/RESPONSE)
 *              F[IN]  length datãTCY
 *              F[IN]  seq    V[PXԍ
 *              F[IN]  data   f[^
 *  return      FȂ
 * ***************************************************************** */
void cec_cm4_callback(u8 type, u8 length, u8 seq, u8 *data)
{
  //int i = 0;
  const char *envp = NULL;

  if ((data == NULL) || (length <= 0)){
    ELOG("param error data=%p length=%d \n", data, length);
    return;
  }

  DLOG("recv msg type=%d length=%d seq=%d\n", type, length, seq);
  //for (i = 0; i < length; i+=8) {
  //  switch(length - i) {
  //  case 0:                                                                                                   break;
  //  case 1:   DLOG("data[%2d] = %02x  \n", i, data[i+0]);                                                     break;
  //  case 2:   DLOG("data[%2d] = %02x %02x \n", i, data[i+0], data[i+1]);                                      break;
  //  case 3:   DLOG("data[%2d] = %02x %02x %02x \n", i, data[i+0], data[i+1], data[i+2]);                      break;
  //  case 4:   DLOG("data[%2d] = %02x %02x %02x %02x \n", i, data[i+0], data[i+1], data[i+2], data[i+3]);      break;
  //  case 5:   DLOG("data[%2d] = %02x %02x %02x %02x   %02x \n", i,
  //                  data[i+0], data[i+1], data[i+2], data[i+3], data[i+4]);                                   break;
  //  case 6:   DLOG("data[%2d] = %02x %02x %02x %02x   %02x %02x \n", i,
  //                  data[i+0], data[i+1], data[i+2], data[i+3], data[i+4], data[i+5]);                        break;
  //  case 7:   DLOG("data[%2d] = %02x %02x %02x %02x   %02x %02x %02x \n", i,
  //                  data[i+0], data[i+1], data[i+2], data[i+3], data[i+4], data[i+5], data[i+6]);             break;
  //  default:  DLOG("data[%2d] = %02x %02x %02x %02x   %02x %02x %02x %02x \n", i,
  //                  data[i+0], data[i+1], data[i+2], data[i+3], data[i+4], data[i+5], data[i+6], data[i+7]);  break;
  //  }
  //}

  if (down_interruptible(&cec_mutex))
  {
    ELOG("[CEC]warning: down err\n!");
  }

  switch(data[0]) {
  case CEC_HANDSHAKE_REQ:           cec_manage_info.handshake_completed = true;      break;
  case CEC_NOTIFY_CEC_IN:           cec_cm4_cb_cec_in(&envp, length, data);          break;
  case CEC_NOTIFY_CEC_OUT_RESULT:   cec_cm4_cb_cec_out_result(&envp, length, data);  break;
  //case CEC_CEC_TRAN_REQ:
  //case CEC_SET_RETRY_COUNT:
  //case CEC_SET_LOGICAL_ADDRESS:
  //case CEC_NOTIFY_HDMI_CONTROL_CHANGE:
  default:
    ELOG("invalid command id =0x%02x \n", data[0]);
    break;
  }

  if (envp != NULL) {
    cec_notify_to_service(envp);
  }

  up(&cec_mutex);
  return;
}

/* *****************************************************************
 *  ֐̏ڍ  FCEC_NOTIFY_CEC_IN
 *  param       F[OUT] **envp ueventŒʒm镶̃|C^
 *              F[IN]  length datãTCY
 *              F[IN]  data   f[^
 *  return      FȂ
 * ***************************************************************** */
static void cec_cm4_cb_cec_in(const char **envp, u8 length, u8 *data)
{
  unsigned char write = cec_manage_info.cec_in.write_ite;
  unsigned char header;
  unsigned char opcode;
  unsigned char command_len;
  unsigned char operand_len;
  unsigned char i;

  if ((data == NULL) || (length < CEC_RECEIVE_COMMAND_SIZE_MIN)){
    ELOG("param error data=%p length=%d \n", data, length);
    return;
  }

  data++;                 // CM4<->A53Ԃ̃R}hID
  command_len = *data;    // CECR}h̒
  data++;
  header = *data;
  data++;

  // p[^TCYHEADERTCY͂肦Ȃ
  if (command_len < CEC_HEADER_SIZE) {
    ELOG("param error data[1]=0x%x \n", *data);
    return;
  }
  // p[^TCYCOMANDTCYőlȂۂ߂
  if (command_len > CEC_COMMAND_SIZE_MAX) {
    command_len = CEC_COMMAND_SIZE_MAX;
  }

  if(command_len > CEC_HEADER_SIZE) {
    // OpcodeCECR}h(<Polling>ȊÕR}h)
    opcode = *data;
    data++;
    // HEADEROPCODẼTCYCECR}h̃p[^TCYɂȂ
    operand_len = (command_len - (CEC_HEADER_SIZE + CEC_OPCODE_SIZE));
  } else {
    // <Polling>̏ꍇOpcode
    opcode = 0;
    operand_len = 0;

    // <Polling>bZ[W͎擾LÂݒʒm
    for(i=0; i<CEC_LOGICAL_ADDRESS_MEMORY_SIZE; i++) {
      if (cec_manage_info.la.logical_address[i] == (header & 0x0F)) {
        break;
      }
    }
    if (i >= CEC_LOGICAL_ADDRESS_MEMORY_SIZE) {
      // ĈȊO<Polling>͉Ȃ
      return;
    }
  }

  // CecServiceCECMʒm
  *envp = cec_env_state_get_cmd;

  // OMCECR}h擾ĂȂCecServiceɒǉŒʒmo
  if(write != cec_manage_info.cec_in.read_ite) {
    DLOG("add notify service \n");
    cec_notify_to_service(*envp);
  }

  // 
  cec_manage_info.cec_in.buf[write].is_no_data = false;
  for(i = 0; i < CEC_OPERAND_SIZE_MAX; i++){
    cec_manage_info.cec_in.buf[write].operand[i] = 0;
  }

  cec_manage_info.cec_in.buf[write].header = header;
  cec_manage_info.cec_in.buf[write].opcode = opcode;

  // operandRs[
  if(operand_len > 0) {
    memcpy(&cec_manage_info.cec_in.buf[write].operand, data, operand_len);
  }
  cec_manage_info.cec_in.buf[write].operand_size = operand_len;

  if(cec_manage_info.cec_in.write_ite >= (CEC_COMMAND_BUFFER_SIZE - 1)) {
    cec_manage_info.cec_in.write_ite = 0;
  } else {
    cec_manage_info.cec_in.write_ite ++;
  }

  if(cec_manage_info.cec_in.is_full != false) {
    // buffer fullԂōXCECR}hMReaditerationwrite_iteƓlɂ(Oobt@)
    cec_manage_info.cec_in.read_ite = cec_manage_info.cec_in.write_ite;
    DLOG("%s", "buffer loop");
  } else if(cec_manage_info.cec_in.write_ite == cec_manage_info.cec_in.read_ite) {
    // bufferfullɂȂ
    cec_manage_info.cec_in.is_full = true;
    DLOG("%s", "buffer full");
  } else {
    // nop
  }
  DLOG("size=%d is_full=%d write_ite=%d read_ite=%d header=%d opcode=%d cmd=%02x %02x %02x %02x  %02x %02x %02x %02x ", command_len,
    cec_manage_info.cec_in.is_full, cec_manage_info.cec_in.write_ite, cec_manage_info.cec_in.read_ite,
    cec_manage_info.cec_in.buf[write].header, cec_manage_info.cec_in.buf[write].opcode,
    cec_manage_info.cec_in.buf[write].operand[0], cec_manage_info.cec_in.buf[write].operand[1],
    cec_manage_info.cec_in.buf[write].operand[2], cec_manage_info.cec_in.buf[write].operand[3],
    cec_manage_info.cec_in.buf[write].operand[4], cec_manage_info.cec_in.buf[write].operand[5],
    cec_manage_info.cec_in.buf[write].operand[6], cec_manage_info.cec_in.buf[write].operand[7]);

  return;
}

/* *****************************************************************
 *  ֐̏ڍ  FCEC_NOTIFY_CEC_OUT_RESULT
 *  param       F[OUT] **envp ueventŒʒm镶̃|C^
 *              F[IN]  length datãTCY
 *              F[IN]  data   f[^
 *  return      FȂ
 * ***************************************************************** */
static void cec_cm4_cb_cec_out_result(const char **envp, u8 length, u8 *data)
{
  if ((data == NULL) || (length < CEC_SEND_COMMAND_RET_SIZE_MIN)){
    ELOG("param error data=%p length=%d \n", data, length);
    return;
  }

  *envp = cec_env_state_sent_cmd;

  data++;                               // CM4<->A53Ԃ̃R}hID
  data++;                               // CM4<->A53Ԃ̃R}hTCY
  cec_manage_info.cec_ack.tag = *data;  // Sequence_ID
  data++;
  if (*data == CEC_SEND_CMD_RET_ACK_OK) {
    cec_manage_info.cec_ack.e_ack_cond = CEC_ACK_COND_OK;
  } else {
    // ARBITRATION_LOST̏ꍇ͖
    if (*data == CEC_SEND_CMD_RET_ARBITRATION_LOST) {
        *envp = NULL;
    }
    cec_manage_info.cec_ack.e_ack_cond = CEC_ACK_COND_NO_RESPONSE;
  }
  data++;
  DLOG("tag=%d e_ack_cond=%d \n", cec_manage_info.cec_ack.tag, cec_manage_info.cec_ack.e_ack_cond);

  return;
}

/* *****************************************************************
 *  ֐̏ڍ  FCEC_NOTIFY_CEC_IN
 *  param       F[IN]  *str ueventŒʒm镶̃|C^
 *  return      FȂ
 * ***************************************************************** */
static void cec_notify_to_service(const char *str)
{
  char *envp[] = { NULL, NULL };

  if (str == NULL) {
    ELOG("param error str=%p \n", str);
    return;
  }
  if (cec_dev == NULL) {
    ELOG("cec_dev is NULL \n");
    return;
  }

  /* This cast is safe because kokobject_uevent_env() takes a array of
     non-const char pointer but does not modify its stuff. */
  envp[0] = (char *)str;
  kobject_uevent_env(&cec_dev->kobj, KOBJ_CHANGE, envp);

  return;
}

/* *****************************************************************
 *  ֐̏ڍ  Fʒm
 *  param       F[IN]  arg p[^
 *  return      FCEC_RET_OK/ERR ID
 * ***************************************************************** */
static signed int cec_ioctl_nop( unsigned long arg )
{
  (void)arg;
  return CEC_RET_OK;
}

/* *****************************************************************
 *  ֐̏ڍ  Fʒm(CEC_GET_CECSTS)
 *  param       F[IN]  arg p[^
 *  return      FCEC_RET_OK/ERR ID
 * ***************************************************************** */
static signed int cec_ioctl_get_cec_status( unsigned long arg ) {

  if (copy_to_user((void __user *)arg, &cec_manage_info.cec_ack, sizeof(CEC_ACK_INFO))) {
     ELOG("[%d]copy_to_user failed! arg=%ld(0x%08lx) \n", __LINE__, arg, arg);
     return -EFAULT;
  }
  DLOG("tag=%d e_ack_cond=%d \n", cec_manage_info.cec_ack.tag, cec_manage_info.cec_ack.e_ack_cond);
  return CEC_RET_OK;
}

/* *****************************************************************
 *  ֐̏ڍ  Fʒm(CEC_SET_CECCMD)
 *  param       F[IN]  arg p[^
 *  return      FCEC_RET_OK/ERR ID
 * ***************************************************************** */
static signed int cec_ioctl_set_cec_command( unsigned long arg ) {

  signed int ret = CEC_RET_OK;
  u8 length;
  u8 send_command[CEC_SEND_COMMAND_SIZE_MAX];
  unsigned char i;

  if (copy_from_user(&cec_manage_info.cec_out, (void __user *)arg, sizeof(CEC_SEND_MSG))) {
    ELOG("[%d]copy_to_user failed! arg=%ld(0x%08lx) \n", __LINE__, arg, arg);
    return -EFAULT;
  }

  for(i = 0; i < CEC_SEND_COMMAND_SIZE_MAX; i++){
    send_command[i] = 0;
  }
  
  send_command[0] = CEC_CEC_TRAN_REQ;
  send_command[2] = cec_manage_info.cec_out.tag;
  send_command[3] = cec_manage_info.cec_out.command.header;
  length = CEC_SEND_COMMAND_SIZE_MIN;  // Œ̌Œ蒷TCY
  if(cec_manage_info.cec_out.is_polling == CEC_BOOL_FALSE) {
    send_command[4] = cec_manage_info.cec_out.command.opcode;
    if(cec_manage_info.cec_out.command.operand_size > 0) {
      if(cec_manage_info.cec_out.command.operand_size > CEC_OPERAND_SIZE_MAX) {
        cec_manage_info.cec_out.command.operand_size = CEC_OPERAND_SIZE_MAX;  // Ö
      }
      memcpy(&send_command[5], cec_manage_info.cec_out.command.operand, cec_manage_info.cec_out.command.operand_size);
    }
    length += (cec_manage_info.cec_out.command.operand_size + CEC_OPCODE_SIZE);  // opcode  operand ̃TCY
  }
  send_command[1] = length - CEC_CM4_A53_HEADER_SIZE;  // header2byte炷

  ret = cec_cm4_send(CEC_RPMSG_CM4_TYPE_REQUEST, length, send_command);
  if (ret != CEC_RET_OK) {
    ELOG("cec_cm4_send failed(%d)\n", ret);
    return ret;
  }
  DLOG("tag=0x%02x size=%d  header=%d opcode=%d command=0x%02x %02x %02x %02x  %02x %02x %02x %02x \n", cec_manage_info.cec_out.tag, send_command[1],
    cec_manage_info.cec_out.command.header, cec_manage_info.cec_out.command.opcode,
    cec_manage_info.cec_out.command.operand[0], cec_manage_info.cec_out.command.operand[1],
    cec_manage_info.cec_out.command.operand[2], cec_manage_info.cec_out.command.operand[3],
    cec_manage_info.cec_out.command.operand[4], cec_manage_info.cec_out.command.operand[5],
    cec_manage_info.cec_out.command.operand[6], cec_manage_info.cec_out.command.operand[7]);
  return CEC_RET_OK;
}

/* *****************************************************************
 *  ֐̏ڍ  Fʒm(CEC_SETLA)
 *  param       F[IN]  arg p[^
 *  return      FCEC_RET_OK/ERR ID
 * ***************************************************************** */
static signed int cec_ioctl_set_logical_address( unsigned long arg ) {

  signed int ret = CEC_RET_OK;
  u8 send_command[CEC_LOGICAL_ADDRESS_SEND_SIZE];
  unsigned char write = cec_manage_info.la.save_ite;
  unsigned char read = ((write == 0) ? 0 : (write - 1));
  unsigned char i;
  CEC_DRV_ADDR_CFG set_la;

  if (copy_from_user(&set_la, (void __user *)arg, sizeof(CEC_DRV_ADDR_CFG))) {
    ELOG("[%d]copy_to_user failed! arg=%ld(0x%08lx) \n", __LINE__, arg, arg);
    return -EFAULT;
  }

  // NȀꍇ0x0FĂ̂ŁALAoĂꍇ͂ǂ̂킩Ȃ
  // TuAhX̕NA
  if (set_la.set_req == CEC_LOGICAL_ADDRESS_CLEAR) {
    set_la.la = cec_manage_info.la.logical_address[read];  // NALAݒ肷
    cec_manage_info.la.save_ite = read;
    cec_manage_info.la.logical_address[read] = 0x0F;
    DLOG("LA Clear!! la=%x  save_ite=%d \n", set_la.la , cec_manage_info.la.save_ite);
  } else {
    // łɐݒ肳ĂLAȂVKɕۑȂ
    for(i=0; i<CEC_LOGICAL_ADDRESS_MEMORY_SIZE; i++) {
      if (cec_manage_info.la.logical_address[i] == set_la.la) {
        write = i;
        break;
      }
    }
    if (write == cec_manage_info.la.save_ite) {
      if (write >= CEC_LOGICAL_ADDRESS_MEMORY_SIZE) {
        ELOG("cec_save_la_memory over flaw!! write=%d \n", write);
        return -EFAULT;
      }
      DLOG("LA Save!! la=%x  save_ite=%d \n", set_la.la , cec_manage_info.la.save_ite);
      cec_manage_info.la.logical_address[write] = set_la.la;
      cec_manage_info.la.save_ite++;
    }
  }
  
  send_command[0] = CEC_SET_LOGICAL_ADDRESS;
  send_command[1] = CEC_LOGICAL_ADDRESS_SIZE;
  send_command[2] = set_la.la;
  send_command[3] = set_la.set_req;
  // ݒ肷_AhX0x0F̏ꍇ͉Ȃ
  if (set_la.la != 0x0F) {
    DLOG("LA Set!! la=%x  save_ite=%d req=%d \n", set_la.la , cec_manage_info.la.save_ite, set_la.set_req);
    ret = cec_cm4_send(CEC_RPMSG_CM4_TYPE_REQUEST, CEC_LOGICAL_ADDRESS_SEND_SIZE, send_command);
    if (ret != CEC_RET_OK) {
      ELOG("cec_cm4_send failed(%d)\n", ret);
      return ret;
    }
  }
  //for(i=0; i<CEC_LOGICAL_ADDRESS_MEMORY_SIZE; i++) {
  //  ELOG("cec_manage_info.la.logical_address[%d] = %02x", i, cec_manage_info.la.logical_address[i]);
  //}
  return CEC_RET_OK;
}

/* *****************************************************************
 *  ֐̏ڍ  Fʒm(CEC_GET_CECCMD)
 *  param       F[IN]  arg p[^
 *  return      FCEC_RET_OK/ERR ID
 * ***************************************************************** */
static signed int cec_ioctl_get_cec_command( unsigned long arg ) {

  unsigned char read = cec_manage_info.cec_in.read_ite;
  unsigned char i;

  if((cec_manage_info.cec_in.write_ite == read) && (cec_manage_info.cec_in.is_full == false)) {
    DLOG("buffer empty \n");
    // CECR}hƂno_datatO𗧂Ă(tOȊO̓[߂Ă)
    cec_manage_info.cec_in.buf[read].operand_size = 0;
    cec_manage_info.cec_in.buf[read].is_no_data = true;
    cec_manage_info.cec_in.buf[read].header = 0;
    cec_manage_info.cec_in.buf[read].opcode = 0;
    for(i = 0; i < CEC_OPERAND_SIZE_MAX; i++){
      cec_manage_info.cec_in.buf[read].operand[i] = 0;
    }
    if (copy_to_user((void __user *)arg, &cec_manage_info.cec_in.buf[read], sizeof(CEC_FRAME_BLOCK_IO))) {
      ELOG("[%d]copy_to_user failed! arg=%ld(0x%08lx) \n", __LINE__, arg, arg);
      return -EFAULT;
    }
    return CEC_RET_OK;
  }

  if (copy_to_user((void __user *)arg, &cec_manage_info.cec_in.buf[read], sizeof(CEC_FRAME_BLOCK_IO))) {
    ELOG("[%d]copy_to_user failed! arg=%ld(0x%08lx) \n", __LINE__, arg, arg);
    return -EFAULT;
  }
  
  if(cec_manage_info.cec_in.read_ite >= (CEC_COMMAND_BUFFER_SIZE - 1)) {
    cec_manage_info.cec_in.read_ite = 0;
  } else {
    cec_manage_info.cec_in.read_ite ++;
  }
  cec_manage_info.cec_in.is_full = false;
  DLOG("size=%d is_full=%d write_ite=%d read_ite=%d header=%d opcode=%d cmd=%02x %02x %02x %02x  %02x %02x %02x %02x ",
    cec_manage_info.cec_in.buf[read].operand_size + (CEC_HEADER_SIZE + CEC_OPCODE_SIZE),
    cec_manage_info.cec_in.is_full, cec_manage_info.cec_in.write_ite, cec_manage_info.cec_in.read_ite,
    cec_manage_info.cec_in.buf[read].header, cec_manage_info.cec_in.buf[read].opcode,
    cec_manage_info.cec_in.buf[read].operand[0], cec_manage_info.cec_in.buf[read].operand[1],
    cec_manage_info.cec_in.buf[read].operand[2], cec_manage_info.cec_in.buf[read].operand[3],
    cec_manage_info.cec_in.buf[read].operand[4], cec_manage_info.cec_in.buf[read].operand[5],
    cec_manage_info.cec_in.buf[read].operand[6], cec_manage_info.cec_in.buf[read].operand[7]);

  return CEC_RET_OK;
}

/* *****************************************************************
 *  ֐̏ڍ  Fʒm(CEC_HDMI_CONTROL_ENABLE)
 *  param       F[IN]  arg p[^
 *  return      FCEC_RET_OK/ERR ID
 * ***************************************************************** */
static signed int cec_ioctl_hdmi_control_enable( unsigned long arg )
{

  signed int ret = CEC_RET_OK;
  u8 send_command[CEC_HDMI_CONTROL_SEND_SIZE];

  send_command[0] = CEC_NOTIFY_HDMI_CONTROL_CHANGE;
  send_command[1] = CEC_HDMI_CONTROL_SIZE;
  send_command[2] = arg;
  ret = cec_cm4_send(CEC_RPMSG_CM4_TYPE_REQUEST, CEC_HDMI_CONTROL_SEND_SIZE, send_command);
  if (ret != CEC_RET_OK) {
    ELOG("cec_cm4_send failed(%d)\n", ret);
    return ret;
  }

  return CEC_RET_OK;
}

/* *****************************************************************
 *  ֐̏ڍ  Fʒm(CEC_SET_RETRY_CNT)
 *  param       F[IN]  arg p[^
 *  return      FCEC_RET_OK/ERR ID
 * ***************************************************************** */
static signed int cec_ioctl_set_retry_cnt( unsigned long arg )
{

  signed int ret = CEC_RET_OK;
  u8 send_command[CEC_SET_RETRY_CNT_SEND_SIZE];

  send_command[0] = CEC_SET_RETRY_COUNT;
  send_command[1] = CEC_SET_RETRY_CNT_SIZE;
  send_command[2] = arg;
  ret = cec_cm4_send(CEC_RPMSG_CM4_TYPE_REQUEST, CEC_SET_RETRY_CNT_SEND_SIZE, send_command);
  if (ret != CEC_RET_OK) {
    ELOG("cec_cm4_send failed(%d)\n", ret);
    return ret;
  }

  return CEC_RET_OK;
}

/* *****************************************************************
 *  ֐̏ڍ  Fʒm(CEC_FILTER_CEC_RECEIVE)
 *  param       F[IN]  arg p[^
 *  return      FCEC_RET_OK/ERR ID
 * ***************************************************************** */
static signed int cec_ioctl_set_filter_cec_in( unsigned long arg )
{

  signed int ret = CEC_RET_OK;
  u8 send_command[CEC_SET_FILTER_CEC_IN_SEND_SIZE];

  send_command[0] = CEC_SET_FILTER_CEC_IN;
  send_command[1] = CEC_SET_FILTER_CEC_IN_SIZE;
  send_command[2] = arg;
  ret = cec_cm4_send(CEC_RPMSG_CM4_TYPE_REQUEST, CEC_SET_FILTER_CEC_IN_SEND_SIZE, send_command);
  if (ret != CEC_RET_OK) {
    ELOG("cec_cm4_send failed(%d)\n", ret);
    return ret;
  }

  return CEC_RET_OK;
}

/* *****************************************************************
 *  ֐̏ڍ  FVXeNꂽۂɌĂ΂B
 *  param       FȂ
 *  return      FCEC_RET_OK/ERR ID
 * ***************************************************************** */
static int __init cec_init(void)
{
  int ret;
  unsigned char i, j;

  DLOG("%s \n", __func__);

  cec_manage_info.handshake_completed = false;
  cec_manage_info.seq = 0;
  cec_manage_info.cec_in.read_ite = 0;
  cec_manage_info.cec_in.write_ite = 0;
  cec_manage_info.cec_in.is_full = false;
  for(i = 0; i < CEC_COMMAND_BUFFER_SIZE; i++){
    cec_manage_info.cec_in.buf[i].operand_size = 0;
    cec_manage_info.cec_in.buf[i].is_no_data = 0;
    cec_manage_info.cec_in.buf[i].header = 0;
    cec_manage_info.cec_in.buf[i].opcode = 0;
    for(j = 0; j < CEC_OPERAND_SIZE_MAX; j++){
      cec_manage_info.cec_in.buf[i].operand[j] = 0;
    }
  }
  cec_manage_info.cec_out.is_polling = 0;
  cec_manage_info.cec_out.tag = 0;
  cec_manage_info.cec_out.command.operand_size = 0;
  cec_manage_info.cec_out.command.is_no_data = 0;
  cec_manage_info.cec_out.command.header = 0;
  cec_manage_info.cec_out.command.opcode = 0;
  for(i = 0; i < CEC_OPERAND_SIZE_MAX; i++){
    cec_manage_info.cec_out.command.operand[i] = 0;
  }
  cec_manage_info.cec_ack.tag = 0;
  cec_manage_info.cec_ack.e_ack_cond = CEC_ACK_COND_OK;
  cec_manage_info.la.save_ite = 0;
  for(i = 0; i < CEC_LOGICAL_ADDRESS_MEMORY_SIZE; i++){
    cec_manage_info.la.logical_address[i] = 0x0F;
  }

  ret = platform_driver_register(&cec_driver_register);
  if (ret != 0) {
    ELOG("regist driver failed(%d)\n", ret);
    return ret;
  }


  DLOG("cec driver adapter initializing...\n");
  cec_driver_adapter_dev_info = (struct cec_driver_adapter_dev *)kzalloc(sizeof(struct cec_driver_adapter_dev), GFP_KERNEL);
  if (!cec_driver_adapter_dev_info) {
    pr_err("Can not allocate memory for multicore dev %p\n", cec_driver_adapter_dev_info);
    return -ENOMEM;
  }

  // 󂢂ĂfoCXԍ(W[ԍƃ}Ci[ԍ)mۂ
  ret = alloc_chrdev_region(&cec_driver_adapter_dev_info->dev_id, CEC_DRIVER_MINOR_BASE, CEC_DRIVER_MINOR_NUM, CEC_DRIVER_NAME);
  if (ret) {
    pr_err("alloc_chrdev_region() error:0x%x\n", ret);
    goto out_free;
  }

  // foCX̃NXo^(/sys/class/cec-driver-adapter_class 쐬)
  cec_driver_adapter_dev_info->class = class_create(THIS_MODULE, CEC_DRIVER_CLASS);
  if (IS_ERR(cec_driver_adapter_dev_info->class)) {
    ret = PTR_ERR(cec_driver_adapter_dev_info->class);
    ELOG("Can not create class: 0x%x\n", ret);
    goto out_cdev_del;
  }

  // cdev\̂̏ƃVXeR[nhe[u̓o^
  cdev_init(&cec_driver_adapter_dev_info->cdev, &cec_driver_adapter_fops);
  cec_driver_adapter_dev_info->cdev.owner = THIS_MODULE;

  // foCX̃NXo^(/sys/class/cec-driver-adapter_class 쐬)
  ret = cdev_add(&cec_driver_adapter_dev_info->cdev, cec_driver_adapter_dev_info->dev_id, CEC_DRIVER_MINOR_NUM);
  if (ret) {
    ELOG("Can not add cdev for cec driver adapter: 0x%x\n", ret);
    goto out_unregister;
  }

  /* foCXm[h쐬(/sys/class/cec-driver-adapter_class/cec-driver-adapter) */
  /* 쐬m[h /dev/cec-driver-adapter ŃANZX\ */
  cec_driver_adapter_dev_info->dev = device_create(cec_driver_adapter_dev_info->class, NULL,
                          cec_driver_adapter_dev_info->dev_id, NULL, CEC_DRIVER_NAME);
  if (IS_ERR(cec_driver_adapter_dev_info->dev)) {
    ret = PTR_ERR(cec_driver_adapter_dev_info->dev);
    pr_err("Can not create device: 0x%x\n", ret);
    goto out_class_destroy;
  }

  return ret;


out_class_destroy:
  class_destroy(cec_driver_adapter_dev_info->class);
out_cdev_del:
  cdev_del(&cec_driver_adapter_dev_info->cdev);
out_unregister:
  unregister_chrdev_region(cec_driver_adapter_dev_info->dev_id, CEC_DRIVER_MINOR_NUM);
out_free:
  kfree(cec_driver_adapter_dev_info);

  platform_driver_unregister(&cec_driver_register);

  return 0;
}

/* *****************************************************************
 *  ֐̏ڍ  FVXeIۂɌĂ΂B
 *  param       FȂ
 *  return      FȂ
 * ***************************************************************** */
static void __exit cec_exit(void)
{
  DLOG("exit\n");

  device_destroy(cec_driver_adapter_dev_info->class, cec_driver_adapter_dev_info->dev_id);
  class_destroy(cec_driver_adapter_dev_info->class);
  cdev_del(&cec_driver_adapter_dev_info->cdev);
  unregister_chrdev_region(cec_driver_adapter_dev_info->dev_id, CEC_DRIVER_MINOR_NUM);
  kfree(cec_driver_adapter_dev_info);

  platform_driver_unregister(&cec_driver_register);

}



//module_platform_driver(cec_driver_register);
late_initcall(cec_init);
module_exit(cec_exit);


MODULE_AUTHOR("Sony Home Entertainment & Sound Products Inc.");
MODULE_DESCRIPTION("cec-driver-adapter");
MODULE_LICENSE("GPL");

