// SPDX-License-Identifier: GPL-2.0
/*
 * icx_pm_helper.c : ICX power management helper driver
 *
 *  Copyright 2019, 2020, 2022 Sony Corporation
 */

#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/err.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/io.h>
#include <linux/semaphore.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/suspend.h>

#include <misc/icx_pm_helper.h>

/*
 ***********************
 * @ global_definitions
 ***********************
 */
#define CONFIG_ICX_PM_HELPER_RESUME_LOCK_MS (20*1000)

static DEFINE_SEMAPHORE(icxpmh_sem);

/*
 *******************************************
 * @ module_parameters (set by boot loader)
 *******************************************
 */
int icx_pm_helper_boot_reason = ICX_PM_HELPER_BOOT_REASON_UNKNOWN;
EXPORT_SYMBOL_GPL(icx_pm_helper_boot_reason);
module_param_named(
	ICX_PM_HELPER_BOOT_REASON_KERNEL_PARAM,
	icx_pm_helper_boot_reason,
	int,
	S_IWUSR | S_IRUGO);

#ifdef CONFIG_ICX_USE_BID_HA
int icx_pm_helper_bid = -1;
EXPORT_SYMBOL_GPL(icx_pm_helper_bid);
module_param_named(icx_bid, icx_pm_helper_bid, int, S_IWUSR | S_IRUGO);
#endif

/*
 ****************
 * @ sysfs_macro
 ****************
 */
#define ICXPMH_INT_SHOW(_name_) \
static ssize_t icxpmh_##_name_##_show( \
	struct device *dev, \
	struct device_attribute *attr, \
	char *buf) \
{ \
	struct icxpmh_context *context; \
	int result = 0; \
	int val; \
\
	down(&icxpmh_sem); \
\
	context = dev_get_drvdata(dev); \
	if (context == NULL) { \
		result = -ENODEV; \
		goto out; \
	} \
\
	val = context->_name_; \
	result = snprintf(buf, PAGE_SIZE, "%d\n", val); \
\
out: \
	up(&icxpmh_sem); \
	return result; \
}

#define ICXPMH_KT_SHOW(_name_) \
static ssize_t icxpmh_##_name_##_show( \
	struct device *dev, \
	struct device_attribute *attr, \
	char *buf) \
{ \
	struct icxpmh_context *context; \
	int result = 0; \
	struct timespec64 ts; \
\
	down(&icxpmh_sem); \
\
	context = dev_get_drvdata(dev); \
	if (context == NULL) { \
		result = -ENODEV; \
		goto out; \
	} \
\
	ts = ktime_to_timespec64(context->_name_); \
	result = snprintf( \
		buf, \
		PAGE_SIZE, \
		"%llu.%09lu\n", \
		(unsigned long long)(ts.tv_sec), \
		(unsigned long)     (ts.tv_nsec)); \
\
out: \
	up(&icxpmh_sem); \
	return result; \
}

#define ICXPMH_TS_SHOW(_name_) \
static ssize_t icxpmh_##_name_##_show( \
	struct device *dev, \
	struct device_attribute *attr, \
	char *buf) \
{ \
	struct icxpmh_context *context; \
	int result = 0; \
\
	down(&icxpmh_sem); \
\
	context = dev_get_drvdata(dev); \
	if (context == NULL) { \
		result = -ENODEV; \
		goto out; \
	} \
\
	result = snprintf( \
		buf, \
		PAGE_SIZE, \
		"%llu.%09lu\n", \
		(unsigned long long)(context->_name_.tv_sec), \
		(unsigned long)     (context->_name_.tv_nsec)); \
\
out: \
	up(&icxpmh_sem); \
	return result; \
}

/*
 *********************
 * @ boot_information
 *********************
 */
ICXPMH_INT_SHOW(boot_powerkey);

static DEVICE_ATTR(boot_powerkey, S_IRUGO, icxpmh_boot_powerkey_show, NULL);

/*
 *******************************
 * @ suspend_resume_information
 *******************************
 */
ICXPMH_KT_SHOW(suspend_kt);
ICXPMH_TS_SHOW(suspend_ts);
ICXPMH_KT_SHOW(resume_kt);
ICXPMH_TS_SHOW(resume_ts);

static DEVICE_ATTR(suspend_kt,         S_IRUGO, icxpmh_suspend_kt_show,  NULL);
static DEVICE_ATTR(suspend_ts,         S_IRUGO, icxpmh_suspend_ts_show,  NULL);
static DEVICE_ATTR(resume_kt,          S_IRUGO, icxpmh_resume_kt_show,   NULL);
static DEVICE_ATTR(resume_ts,          S_IRUGO, icxpmh_resume_ts_show,   NULL);

/*
 *******************
 * @ resume_lock_ms
 *******************
 */
static ssize_t icxpmh_resume_lock_ms_show(
	struct device *dev,
	struct device_attribute *attr,
	char *buf)
{
	struct icxpmh_context *context;
	ssize_t result = 0;

	down(&icxpmh_sem);

	context = dev_get_drvdata(dev);
	if (context == NULL) {
		result = -ENODEV;
		goto out;
	}

	result = snprintf(buf, PAGE_SIZE, "%u\n", context->resume_lock_ms);

out:
	up(&icxpmh_sem);
	return result;
}

static ssize_t icxpmh_resume_lock_ms_store(
	struct device *dev,
	struct device_attribute *attr,
	const char *buf,
	size_t size)
{
	struct icxpmh_context *context;
	ssize_t result;
	unsigned long val;
	int ret;

	/* force cast */
	result = (ssize_t)size;

	down(&icxpmh_sem);

	context = dev_get_drvdata(dev);
	if (context == NULL) {
		result = -ENODEV;
		goto out;
	}

	ret = (unsigned int)kstrtoul(buf, 10, &val);
	if (ret != 0) {
		pr_err("%s: Conversion failed(%d)\n", __func__, ret);
		result = -EINVAL;
		goto out;
	}

	context->resume_lock_ms = val;

out:
	up(&icxpmh_sem);
	return result;
}

static DEVICE_ATTR(
	resume_lock_ms,
	S_IWUSR | S_IRUGO,
	icxpmh_resume_lock_ms_show,
	icxpmh_resume_lock_ms_store);

/*
 ***********************
 * @ resume_lock_cancel
 ***********************
 */
static ssize_t icxpmh_resume_lock_cancel_store(
	struct device *dev,
	struct device_attribute *attr,
	const char *buf,
	size_t size)
{
	struct icxpmh_context *context;
	ssize_t result;
	long val;
	int ret;

	/* force cast */
	result = (ssize_t)size;

	down(&icxpmh_sem);

	context = dev_get_drvdata(dev);
	if (context == NULL) {
		result = -ENODEV;
		goto out;
	}

	ret = kstrtol(buf, 10, &val);
	if (ret != 0) {
		pr_err("%s: Conversion failed(%d)\n", __func__, ret);
		result = -EINVAL;
		goto out;
	}

	switch (val) {
	case ICX_PM_HELPER_RESUME_LOCK_CANCEL_CANCEL:
		pm_wakeup_event(context->dev, 0);
		break;
	case ICX_PM_HELPER_RESUME_LOCK_CANCEL_DO_NOTHING:
		/* do nothing. */
		break;
	default:
		result = -EINVAL;
		break;
	}

out:
	up(&icxpmh_sem);
	return result;
}

static DEVICE_ATTR(
	resume_lock_cancel,
	S_IWUSR | S_IRUGO,
	NULL,
	icxpmh_resume_lock_cancel_store
);

/*
 ***********************
 * @ sysfs_global_value
 ***********************
 */
static struct attribute *icxpmh_attributes[] = {
	&dev_attr_boot_powerkey.attr,

	&dev_attr_suspend_kt.attr,
	&dev_attr_suspend_ts.attr,
	&dev_attr_resume_kt.attr,
	&dev_attr_resume_ts.attr,

	&dev_attr_resume_lock_ms.attr,
	&dev_attr_resume_lock_cancel.attr,

	NULL,
};

static const struct attribute_group icxpmh_attr_group = {
	.attrs = icxpmh_attributes,
};

/*
 *************************
 * @ suspend_resume_entry
 *************************
 */
static int icxpmh_raise_resume_uevent(struct icxpmh_context *context)
{
	char event[] = "EVENT=resume";
	char *envp[2] = {event, NULL};

	struct device *dev;
	int result = 0;
	int ret;

	pr_debug("%s()\n", __func__);

	dev = context->dev;
	if (dev == NULL) {
		pr_err("%s: context->dev is NULL.\n", __func__);
		result = -ENODEV;
		goto out;
	}

	ret = kobject_uevent_env(&dev->kobj, KOBJ_CHANGE, envp);
	if (ret != 0) {
		pr_err("%s: kobject_uevent_env() failed.\n", __func__);
		result = ret;
		goto out;
	}

out:
	return result;
}

static int icxpmh_notifier_call(
	struct notifier_block *nb,
	unsigned long event,
	void *ignore)
{
	struct icxpmh_context *context;

	down(&icxpmh_sem);

	pr_debug("%s(event=%lu)\n", __func__, event);

	context = container_of(nb, struct icxpmh_context, pm_nb);

	context->pm_state = event;

	if (event == PM_POST_SUSPEND)
		icxpmh_raise_resume_uevent(context);

	up(&icxpmh_sem);
	return NOTIFY_OK;
}

static int icxpmh_suspend(struct platform_device *pdev,  pm_message_t state)
{
	struct icxpmh_context *context;

	down(&icxpmh_sem);

	pr_debug("%s(event=%d)\n", __func__, state.event);

	context = platform_get_drvdata(pdev);
	if (context == NULL) {
		pr_warn("%s: context is NULL.\n", __func__);
		goto out;
	}

	ktime_get_real_ts64(&(context->suspend_ts));
	context->suspend_kt = ktime_get();

out:
	up(&icxpmh_sem);
	return 0;
}

static int icxpmh_resume(struct platform_device *pdev)
{
	struct icxpmh_context *context;
	int result = 0;
	unsigned int lock_ms;

	down(&icxpmh_sem);

	pr_debug("%s()\n", __func__);

	context = platform_get_drvdata(pdev);
	if (context == NULL) {
		pr_warn("%s: context is NULL.\n", __func__);
		goto out;
	}

	ktime_get_real_ts64(&(context->resume_ts));
	context->resume_kt = ktime_get();

	lock_ms = context->resume_lock_ms;
	if (lock_ms > 0) {
		pr_info("get icx_pm_helper wake_lock for %u ms.", lock_ms);
		pm_wakeup_event(context->dev, lock_ms);
	}

out:
	up(&icxpmh_sem);
	return result;
}

/*
 **********************
 * @ load_unload_entry
 **********************
 */
static void icxpmh_probe_boot_reasons(struct icxpmh_context *context)
{
	long reason;

	reason = icx_pm_helper_boot_reason;
	if (reason != ICX_PM_HELPER_BOOT_REASON_UNKNOWN) {
		/* boot loader set boot reason. */
		if (reason & ICX_PM_HELPER_BOOT_REASON_POWER_BUTTON)
			context->boot_powerkey = ICX_PM_HELPER_BOOT_X_TRUE;
		else
			context->boot_powerkey = ICX_PM_HELPER_BOOT_X_FALSE;
	} else {
		context->boot_powerkey = ICX_PM_HELPER_BOOT_X_TRUE;
	}
}

static int icxpmh_probe(struct platform_device *pdev)
{
	struct icxpmh_context *context = NULL;
	int result = 0;
	int wlret = -1;
	int sfret = -1;
	int pnret = -1;

	down(&icxpmh_sem);

	pr_debug("%s()\n", __func__);

	context = kzalloc(sizeof(*context), GFP_NOIO);
	if (!context) {
		result = -ENOMEM;
		goto out;
	}

	platform_set_drvdata(pdev, context);
	context->dev = &(pdev->dev);

	icxpmh_probe_boot_reasons(context);

	wlret = device_init_wakeup(context->dev, true);
	if (wlret != 0) {
		pr_err("%s: device_init_wakeup() failed.\n", __func__);
		result = wlret;
		goto out;
	}
	context->resume_lock_ms = CONFIG_ICX_PM_HELPER_RESUME_LOCK_MS;
	wlret = 0;

	sfret = sysfs_create_group(&pdev->dev.kobj, &icxpmh_attr_group);
	if (sfret != 0) {
		pr_err("%s: sysfs_create_group() failed.\n", __func__);
		result = sfret;
		goto out;
	}

	context->pm_nb.notifier_call = icxpmh_notifier_call;
	context->pm_nb.priority = 0;
	register_pm_notifier(&context->pm_nb);
	pnret = 0;

	pr_info("loaded ICX power management helper.\n");

out:
	if (result != 0) {
		if (pnret == 0)
			unregister_pm_notifier(&context->pm_nb);
		if (sfret == 0)
			sysfs_remove_group(&pdev->dev.kobj, &icxpmh_attr_group);
		if (wlret == 0) {
			pm_relax(context->dev);
			device_init_wakeup(context->dev, false);
		}
		if (context != NULL)
			kfree(context);
	}
	up(&icxpmh_sem);
	return result;
}

static int icxpmh_remove(struct platform_device *pdev)
{
	struct icxpmh_context *context;

	down(&icxpmh_sem);

	pr_debug("%s()\n", __func__);

	context = platform_get_drvdata(pdev);
	if (context == NULL) {
		pr_err("%s: context is NULL.\n", __func__);
		goto out;
	}

	unregister_pm_notifier(&context->pm_nb);
	sysfs_remove_group(&pdev->dev.kobj, &icxpmh_attr_group);
	pm_relax(context->dev);
	device_init_wakeup(context->dev, false);
	context->dev = NULL;
	platform_set_drvdata(pdev, NULL);
	kfree(context);

out:
	up(&icxpmh_sem);
	return 0;
}

static const struct of_device_id icxpmh_of_match[] = {
	{
		.compatible = "sony,icx_pm_helper",
	},
	{
	},
};

static struct platform_driver icxpmh_driver = {
	.probe      = icxpmh_probe,
	.remove     = icxpmh_remove,
	.suspend    = icxpmh_suspend,
	.resume     = icxpmh_resume,
	.driver     = {
		.name   = "icx_pm_helper",
		.owner  = THIS_MODULE,
		.of_match_table = icxpmh_of_match,
	},
};

static int __init icxpmh_init(void)
{
	platform_driver_register(&icxpmh_driver);
	return 0;
}
module_init(icxpmh_init);

static void __exit icxpmh_exit(void)
{
	platform_driver_unregister(&icxpmh_driver);
}
module_exit(icxpmh_exit);

MODULE_AUTHOR("SONY");
MODULE_DESCRIPTION("ICX power management helper.");
MODULE_LICENSE("GPL");
