/* SPDX-License-Identifier: GPL-2.0 */
/*
 * keepalive_cm4.c - keepalive_ driver
 *
 * Copyright 2023 Sony Corporation
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/virtio.h>
#include <linux/rpmsg.h>
#include <linux/err.h>
#include <linux/cdev.h>
#include <linux/platform_device.h>
#include <linux/property.h>
#include <linux/cm4.h>
#include <linux/kthread.h>

#define DRV_NAME		"keepalive_cm4"
#define ERRLOG(fmt, ...)	pr_err(DRV_NAME ": Error: " fmt, ## __VA_ARGS__)
#define WRNLOG(fmt, ...)	pr_warn(DRV_NAME ": Warning: " fmt, ## __VA_ARGS__)
#define MSGLOG(fmt, ...)	pr_info(DRV_NAME ": " fmt, ## __VA_ARGS__)
#define DBGLOG(fmt, ...)	pr_debug(DRV_NAME ": %s(%d): " fmt, __func__, __LINE__, ## __VA_ARGS__)

/*******************************************************************************
 * Definition
*******************************************************************************/
#define KPAV_CM4_STS_REQ	(0x01)
#define KPAV_CM4_STS_RESP	(0x81)

struct keepalive_cm4_priv {
	struct device		*dev;
	struct task_struct *kthread_tsk;
	struct workqueue_struct *wq_res_timer;
	struct delayed_work work_res_timer;
	u32 keepalive_interval;
	u32 response_wait_timer;
	u8	mng_info_seq;
};

static struct keepalive_cm4_priv *keepalive_cm4_drvdata;

/*******************************************************************************
 * Variables
*******************************************************************************/

/*******************************************************************************
 * Functions
*******************************************************************************/
static int keepalive_core_cm4_send(struct keepalive_cm4_priv *priv,
						u8 type, u8 length, u8 *data)
{
	int ret;

	if ((data == NULL) || (length < 4)) {
		ERRLOG("param error data=%p length=%d\n", data, length);
		return -EINVAL;
	}

	ret = rpmsg_cm4_send(CM4_MESSAGE_ID_KPAV, type, length,
						priv->mng_info_seq, data);
	if (ret != 0) {
		ERRLOG("rpmsg_cm4_send failed(%d)\n", ret);
		return -EINVAL;
	}
	priv->mng_info_seq++;
	return 0;
}

static void keepalive_core_res_timer(struct work_struct *work)
{
	WRNLOG("cm4 has no response. It might be hung up\n");
}

static void keepalive_core_sts_request(struct keepalive_cm4_priv *priv)
{
	int ret;
	u8 data[4] = {0x01, 0x00, KPAV_CM4_STS_REQ, 0x00};

	queue_delayed_work(priv->wq_res_timer,
		&priv->work_res_timer,
		msecs_to_jiffies(priv->response_wait_timer * 1000));
	ret = keepalive_core_cm4_send(priv, CM4_MESSAGE_TYPE_REQ, 4, data);
	if (ret != 0)
		cancel_delayed_work(&priv->work_res_timer);
}

void keepalive_core_callback(u8 type, u8 length, u8 seq, u8 *data)
{
	struct keepalive_cm4_priv *priv = keepalive_cm4_drvdata;

	if ((data == NULL) || (length <= 0)) {
		ERRLOG("param error data=%p length=%d\n", data, length);
		return;
	}

	switch (data[0]) {
	case KPAV_CM4_STS_RESP:
		cancel_delayed_work(&priv->work_res_timer);
		break;
	default:
		ERRLOG("invalid command id =0x%02x\n", data[0]);
		break;
	}
}

static void kthread_keepalive_core_main(struct keepalive_cm4_priv *priv)
{
	keepalive_core_sts_request(priv);
}

static int kthread_keepalive_core_func(void *arg)
{
	struct keepalive_cm4_priv *priv = keepalive_cm4_drvdata;

	MSGLOG("kthread_func\n");

	while (!kthread_should_stop()) {
		set_current_state(TASK_INTERRUPTIBLE);
		schedule_timeout(msecs_to_jiffies(priv->keepalive_interval * 1000));
		if (kthread_should_stop())
			break;

		kthread_keepalive_core_main(priv);
	}

	return 0;
}
static int keepalive_core_probe(struct platform_device *pdev)
{
	struct keepalive_cm4_priv *priv;
	int ret = 0;

	DBGLOG("probe\n");

	priv = devm_kzalloc(&pdev->dev,
		sizeof(struct keepalive_cm4_priv), GFP_KERNEL);
	if (!priv) {
		ret = -ENOMEM;
		ERRLOG("allocate failed\n");
		goto err_exit;
	}

	priv->dev = &pdev->dev;
	keepalive_cm4_drvdata = priv;

	if (device_property_read_u32(&pdev->dev, "keepalive-interval",
			&priv->keepalive_interval))
		priv->keepalive_interval = 50;

	if (device_property_read_u32(&pdev->dev, "response-wait-timer",
			&priv->response_wait_timer))
		priv->response_wait_timer = 2;

	priv->wq_res_timer =
		create_singlethread_workqueue("keepalive_res_timer");
	if (!priv->wq_res_timer) {
		ret = -ENOMEM;
		ERRLOG("unable to alloc workqueue response timer\n");
		goto err_kfree;
	}
	INIT_DELAYED_WORK(&priv->work_res_timer, keepalive_core_res_timer);

	priv->kthread_tsk = kthread_run(kthread_keepalive_core_func,
				(void *)priv, "keepalive_cm4_tsk");
	if (IS_ERR(priv->kthread_tsk)) {
		ret = PTR_ERR(priv->kthread_tsk);
		ERRLOG("kthread_run() error:0x%x\n", ret);
		goto err_workqueue;
	}

	ret = rpmsg_cm4_register_callback(CM4_MESSAGE_ID_KPAV,
					keepalive_core_callback);
	if (ret != 0) {
		ret = -EPROBE_DEFER;
		ERRLOG("regist callback failed(%d)\n", ret);
		goto err_kthread_stop;
	}

	platform_set_drvdata(pdev, priv);

	return 0;

err_kthread_stop:
	kthread_stop(priv->kthread_tsk);
err_workqueue:
	destroy_workqueue(priv->wq_res_timer);
err_kfree:
	devm_kfree(&pdev->dev, priv);
	keepalive_cm4_drvdata = NULL;
err_exit:
	return ret;
}

static int keepalive_core_remove(struct platform_device *pdev)
{
	struct keepalive_cm4_priv *priv = dev_get_drvdata(&pdev->dev);

	DBGLOG("remove\n");

	cancel_delayed_work(&priv->work_res_timer);
	flush_workqueue(priv->wq_res_timer);
	destroy_workqueue(priv->wq_res_timer);
	kthread_stop(priv->kthread_tsk);
	devm_kfree(&pdev->dev, priv);

	keepalive_cm4_drvdata = NULL;

	return 0;
}

static const struct of_device_id keepalive_core_dt_ids[] = {
	{ .compatible = "sony,keepalive_cm4", },
	{ },
};
MODULE_DEVICE_TABLE(of, keepalive_core_dt_ids);

static struct platform_driver keepalive_core_driver = {
	.driver = {
		.name = DRV_NAME,
		.owner = THIS_MODULE,
		.of_match_table = keepalive_core_dt_ids,
	},
	.probe = keepalive_core_probe,
	.remove = keepalive_core_remove,
};

static int __init keepalive_core_init(void)
{
	int ret = 0;

	DBGLOG("init\n");

	ret = platform_driver_register(&keepalive_core_driver);
	if (ret != 0) {
		ERRLOG("register driver failed(%d)\n", ret);
		return ret;
	}

	return 0;
}

static void __exit keepalive_core_exit(void)
{
	DBGLOG("exit\n");

	platform_driver_unregister(&keepalive_core_driver);
}

module_init(keepalive_core_init);
module_exit(keepalive_core_exit);


MODULE_AUTHOR("Sony Corporation");
MODULE_DESCRIPTION("keepalive_cm4");
MODULE_LICENSE("GPL");
