/* SPDX-License-Identifier: GPL-2.0 */
/*
 * keys_cm4.c - key driver
 *
 * Copyright 2022 Sony Corporation
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/virtio.h>
#include <linux/rpmsg.h>
#include <linux/err.h>
#include <linux/cdev.h>
#include <linux/platform_device.h>
#include <linux/cm4.h>
#include <misc/keys_cm4.h>

#define DRV_NAME		"keys_cm4"
#define ERRLOG(fmt, ...)	pr_err(DRV_NAME ": Error: " fmt, ## __VA_ARGS__)
#define WRNLOG(fmt, ...)	pr_warn(DRV_NAME ": Warning: " fmt, ## __VA_ARGS__)
#define MSGLOG(fmt, ...)	pr_info(DRV_NAME ": " fmt, ## __VA_ARGS__)
#define DBGLOG(fmt, ...)	pr_debug(DRV_NAME ": %s(%d): " fmt, __func__, __LINE__, ## __VA_ARGS__)

/*******************************************************************************
 * Definition
*******************************************************************************/
#define KEYS_CM4_RECIVE_RESPONSE	(1)
#define KEYS_CM4_REQ_GET_STATUS		(2)
#define KEYS_CM4_NOTIFY_KEY			(0x81)
#define KEYS_CM4_RES_GET_STATUS		(0x82)

u8	keys_cm4_mng_info_seq;

const char keys_cm4_str_action_button[] = "action button";
const char keys_cm4_str_mic_onoff[] = "mic onoff";
/*******************************************************************************
 * Variables
*******************************************************************************/

/*******************************************************************************
 * Functions
*******************************************************************************/
static int keys_core_cm4_send(u8 type, u8 length, u8 *data)
{
	int ret;

	if ((data == NULL) || (length < 4)) {
		ERRLOG("param error data=%p length=%d\n", data, length);
		return -EINVAL;
	}

	ret = rpmsg_cm4_send(CM4_MESSAGE_ID_KEY, type, length,
						keys_cm4_mng_info_seq, data);
	if (ret != 0) {
		ERRLOG("rpmsg_cm4_send failed(%d)\n", ret);
		return -EINVAL;
	}
	keys_cm4_mng_info_seq++;
	return 0;
}

static ssize_t keys_core_get_status(u8 keyid)
{
	ssize_t length;
	u8 data[4];

	if ((keyid == KEYS_CM4_ID_NONE) || (keyid >= KEYS_CM4_ID_NUM) ) {
		ERRLOG("illegal key id:%d\n", keyid);
		return -EINVAL;
	}

	length = sizeof(data);
	data[0] = KEYS_CM4_REQ_GET_STATUS;
	data[1] = 0x00;
	data[2] = keyid;
	data[3] = 0x00;
	keys_core_cm4_send(CM4_MESSAGE_TYPE_REQ, (u8)length, data);

	return length;
}

static ssize_t keys_core_recive_response(struct device *dev,
				struct device_attribute *attr,
				char *buf)
{
	ssize_t length;
	u8 data[4];

	length = sizeof(data);
	data[0] = KEYS_CM4_RECIVE_RESPONSE;
	data[1] = buf[1];
	data[2] = buf[2];
	data[3] = buf[3];

	keys_core_cm4_send(CM4_MESSAGE_TYPE_RES, (u8)length, data);

	return length;
}

static ssize_t keys_core_get_status_store(struct device *dev,
				struct device_attribute *attr,
				const char *buf, size_t size)
{
	u8 keyid;

	if (kstrtou8(buf, 0, &keyid))
		return -EINVAL;

	(void)keys_core_get_status(keyid);

	return size;
}

static DEVICE_ATTR(status, 0660, NULL, keys_core_get_status_store);

static struct attribute *attributes[] = {
	&dev_attr_status.attr,
	NULL,
};

static const struct attribute_group attr_group = {
	.attrs	= (struct attribute **)attributes,
};

void keys_core_callback(u8 type, u8 length, u8 seq, u8 *data)
{
	const char* key_str;

	DBGLOG("callback\n");

	if ((data == NULL) || (length <= 0)) {
		ERRLOG("param error data=%p length=%d\n", data, length);
		return;
	}

	switch (data[2]) {
	case KEYS_CM4_ID_ACTION_BUTTON:
		key_str = &keys_cm4_str_action_button[0];
		break;
	case KEYS_CM4_ID_MIC_ONOFF:
		key_str = &keys_cm4_str_mic_onoff[0];
		break;
	default:
		ERRLOG("callback illegal keyid:%d\n", data[2]);
		return;
	}

	switch (data[0]) {
	case KEYS_CM4_NOTIFY_KEY:
		MSGLOG("notify %s level=%d\n", key_str, data[3]);
		keys_core_recive_response(NULL, NULL, data);
		break;
	case KEYS_CM4_RES_GET_STATUS:
		MSGLOG("get status response result=%d %s level=%d\n",
			data[1], key_str, data[3]);
		break;
	default:
		ERRLOG("invalid command id =0x%02x\n", data[0]);
		break;
	}

}

static int keys_core_probe(struct platform_device *pdev)
{
	int ret = 0;

	DBGLOG("probe\n");

	ret = rpmsg_cm4_register_callback(CM4_MESSAGE_ID_KEY,
							keys_core_callback);
	if (ret != 0) {
		ERRLOG("regist callback failed(%d)\n", ret);
		return -EPROBE_DEFER;
	}

	ret = sysfs_create_group(&pdev->dev.kobj, &attr_group);
	if (ret) {
		ERRLOG("sysfs create failed(%d)\n", ret);
		return ret;
	}

	return 0;
}

static int keys_core_remove(struct platform_device *pdev)
{
	DBGLOG("remove\n");

	sysfs_remove_group(&pdev->dev.kobj, &attr_group);

	return 0;
}

static const struct of_device_id keys_core_dt_ids[] = {
	{ .compatible = "sony,keys_cm4", },
	{ },
};
MODULE_DEVICE_TABLE(of, keys_core_dt_ids);

static struct platform_driver keys_core_driver = {
	.driver = {
		.name = DRV_NAME,
		.owner = THIS_MODULE,
		.of_match_table = keys_core_dt_ids,
	},
	.probe = keys_core_probe,
	.remove = keys_core_remove,
};

static int __init keys_core_init(void)
{
	int result = 0;

	DBGLOG("init\n");

	result = platform_driver_register(&keys_core_driver);
	if (result != 0) {
		ERRLOG("register driver failed(%d)\n", result);
		return result;
	}

	keys_cm4_mng_info_seq = 0;

	return 0;
}

static void __exit keys_core_exit(void)
{
	DBGLOG("exit\n");

	platform_driver_unregister(&keys_core_driver);
}

module_init(keys_core_init);
module_exit(keys_core_exit);


MODULE_AUTHOR("Sony Corporation");
MODULE_DESCRIPTION("keys_cm4");
MODULE_LICENSE("GPL");
