/*
 * Copyright (C) 2019 MediaTek Inc.
 * Copyright 2022 Sony Corporation
 *
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License version 2 as published by the
 * Free Software Foundation.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program.
 * If not, see <http://www.gnu.org/licenses/>.
 */

#include <linux/device.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/of.h>
#include <linux/platform_device.h>
#include <linux/printk.h>
#include <soc/mediatek/hifi4dsp/hifi4dsp_eint.h>

#define DRV_PREFIX "mediatek"
#define DRV_NAME "mt8570_sleep"

struct hifi4dsp_sleep_priv {
	struct device *dev;
	int state;
	struct mutex lock;
	int refcount;
};

enum hifi4dsp_sleep_state_enum {
	HIFI4DSP_SLEEP_STATE_SLEPT = 0,
	HIFI4DSP_SLEEP_STATE_ACTIVED,
	HIFI4DSP_SLEEP_STATE_MAX,
};

static const char * const hifi4dsp_sleep_state_str[] = {
	[HIFI4DSP_SLEEP_STATE_SLEPT] = "slept",
	[HIFI4DSP_SLEEP_STATE_ACTIVED] = "actived",
};

enum hifi4dsp_sleep_cmd_enum {
	HIFI4DSP_SLEEP_CMD_SLEEP = 0,
	HIFI4DSP_SLEEP_CMD_ACTIVE,
	HIFI4DSP_SLEEP_CMD_MAX,
};

static const char * const hifi4dsp_sleep_cmd_str[] = {
	[HIFI4DSP_SLEEP_CMD_SLEEP] = "sleep",
	[HIFI4DSP_SLEEP_CMD_ACTIVE] = "active",
};

static bool hifi4dsp_sleep_state_run_cmd(int state_new)
{
	bool ret = true;

	pr_debug("%s state: %d", __func__, state_new);

	switch (state_new) {
	case HIFI4DSP_SLEEP_STATE_SLEPT:
		ret = trigger_dsp_suspend();
		break;
	case HIFI4DSP_SLEEP_STATE_ACTIVED:
		ret = trigger_dsp_resume();
		break;
	default:
		break;
	}

	return ret;
}

static bool hifi4dsp_sleep_is_cmd_matched(const char * const cmd_expected[],
	const char *cmd_pattern,
	unsigned int cmd_id)
{
	if (cmd_id < HIFI4DSP_SLEEP_CMD_MAX)
		return !strncmp(cmd_expected[cmd_id],
			cmd_pattern, strlen(cmd_expected[cmd_id]));
	else
		return false;
}

static void hifi4dsp_sleep_update_state(struct device *dev,
	int state, bool apply)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct hifi4dsp_sleep_priv *priv = platform_get_drvdata(pdev);

	if (apply) {
		priv->state = state;
	} else {
		if (state == HIFI4DSP_SLEEP_STATE_SLEPT)
			priv->refcount++;
		else if (state == HIFI4DSP_SLEEP_STATE_ACTIVED)
			priv->refcount--;
	}
}

/*
 * refcount for active request
 *    needs only one active request to leave the sleep mode
 *    needs the same amount of sleep request to leave the active mode
 */
static int hifi4dsp_sleep_get_next_state(struct device *dev,
	const char *buf)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct hifi4dsp_sleep_priv *priv = platform_get_drvdata(pdev);
	int state_new = -EINVAL;

	if (hifi4dsp_sleep_is_cmd_matched(hifi4dsp_sleep_cmd_str,
			buf, HIFI4DSP_SLEEP_CMD_SLEEP)) {
		priv->refcount--;
		if (priv->refcount == 0)
			state_new = HIFI4DSP_SLEEP_STATE_SLEPT;
		else if (priv->refcount < 0)
			priv->refcount = 0;
	} else if (hifi4dsp_sleep_is_cmd_matched(hifi4dsp_sleep_cmd_str,
			buf, HIFI4DSP_SLEEP_CMD_ACTIVE)) {
		priv->refcount++;
		if (priv->refcount == 1)
			state_new = HIFI4DSP_SLEEP_STATE_ACTIVED;
	} else {
		pr_notice("%s no matching cmd\n", __func__);
		goto exit;
	}

	if (state_new == priv->state)
		goto exit;

exit:
	return state_new;
}

static ssize_t state_show(struct device *dev,
	struct device_attribute *attr, char *buf)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct hifi4dsp_sleep_priv *priv = platform_get_drvdata(pdev);
	int ret = 0;

	mutex_lock(&priv->lock);

	if (priv->state >= HIFI4DSP_SLEEP_STATE_MAX)
		goto exit;

	ret = scnprintf(buf, PAGE_SIZE, "%s\n",
			hifi4dsp_sleep_state_str[priv->state]);

exit:
	mutex_unlock(&priv->lock);

	return ret;
}

static ssize_t state_store(struct device *dev,
	struct device_attribute *attr,
	const char *buf, size_t count)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct hifi4dsp_sleep_priv *priv = platform_get_drvdata(pdev);
	int state_new = 0;
	bool ret = false;

	pr_debug("%s", __func__);

	mutex_lock(&priv->lock);

	state_new = hifi4dsp_sleep_get_next_state(dev, buf);

	if (IS_ERR_VALUE(state_new))
		goto exit;

	ret = hifi4dsp_sleep_state_run_cmd(state_new);
	hifi4dsp_sleep_update_state(dev, state_new, ret);

exit:
	mutex_unlock(&priv->lock);

	return count;
}
static DEVICE_ATTR_RW(state);

static int hifi4dsp_sleep_probe(struct platform_device *pdev)
{
	struct hifi4dsp_sleep_priv *priv;

	pr_notice("%s", __func__);

	priv = devm_kzalloc(&pdev->dev,
		sizeof(struct hifi4dsp_sleep_priv), GFP_KERNEL);
	if (!priv)
		return -ENOMEM;

	priv->dev = &pdev->dev;

	platform_set_drvdata(pdev, priv);

	priv->state = HIFI4DSP_SLEEP_STATE_ACTIVED;
	priv->refcount = 1;

	mutex_init(&priv->lock);

	return device_create_file(priv->dev, &dev_attr_state);
}

static int hifi4dsp_sleep_remove(struct platform_device *pdev)
{
	struct hifi4dsp_sleep_priv *priv = platform_get_drvdata(pdev);

	pr_notice("%s", __func__);

	device_remove_file(priv->dev, &dev_attr_state);

	mutex_destroy(&priv->lock);

	return 0;
}

#ifdef CONFIG_OF
static const struct of_device_id hifi4dsp_sleep_of_ids[] = {
	{ .compatible = DRV_PREFIX, DRV_NAME, },
	{}
};
#endif

static struct platform_driver hifi4dsp_sleep_driver = {
	.probe = hifi4dsp_sleep_probe,
	.remove = hifi4dsp_sleep_remove,
	.driver = {
		.name = DRV_NAME,
		.owner = THIS_MODULE,
#ifdef CONFIG_OF
		.of_match_table = hifi4dsp_sleep_of_ids,
#endif
	},
};

static int __init hifi4dsp_sleep_driver_init(void)
{
	int ret = 0;

	pr_notice("%s", __func__);

	ret = platform_driver_register(&hifi4dsp_sleep_driver);
	if (ret) {
		pr_notice("%s driver register fail, ret %d\n", __func__, ret);
		goto err_drv;
	}

	return ret;

err_drv:
	return ret;
}

static void __exit hifi4dsp_sleep_driver_exit(void)
{
	pr_notice("%s", __func__);

	platform_driver_unregister(&hifi4dsp_sleep_driver);
}

module_init(hifi4dsp_sleep_driver_init);
module_exit(hifi4dsp_sleep_driver_exit);

MODULE_DESCRIPTION("MT8570 Sleep Driver");
