// SPDX-License-Identifier: GPL-2.0

/* notify_hightemp.c: Notify High Temp Driver
 * Copyright 2020 Sony Home Entertainment & Sound Products Inc.
 * Copyright 2022 Sony Corporation
 */
#include <linux/types.h>
#include <linux/atomic.h>
#include <linux/semaphore.h>
#include <linux/kernel.h>
#include <linux/notifier.h>
#include <linux/kobject.h>
#include <linux/module.h>
#include <linux/device.h>
#include <linux/of_device.h>
#include <linux/platform_device.h>
#include <linux/reboot.h>
#include <linux/device_cooling.h>
#include <linux/string.h>
#include <linux/sysfs.h>
#include <linux/init.h>

#define DRV_NAME "notify_hightemp"

struct notify_hightemp {
	struct semaphore	sem;
	struct device		*dev;
	struct kobject		*kobj;
	bool			is_hot;
};

static struct notify_hightemp notify_hightemp = {
	.sem =  __SEMAPHORE_INITIALIZER(notify_hightemp.sem, 1),
};

static int thermal_hot_pm_notify(struct notifier_block *nb,
	unsigned long event, void *dummy)
{
	struct notify_hightemp	*ht = &(notify_hightemp);
	struct device		*dev;
	struct kobject		*kobj;

	/* In atomic context, down() and sysfs_notify() will dump back trace.
	 * down() and sysfs_notify() wants called from non-atomic context.
	 * NOTE: Strictly say, notifier_call_chain() may run in
	 * atomic context. But termal_zone_* calls
	 * notifier_call_chain() in work queue context when
	 * using periodic polling.
	 */

	down(&(ht->sem));

	dev = ht->dev;
	kobj = ht->kobj;

	if ((!dev) || (!kobj)) {
		/* dev or kobj is(are) NULL,
		 * We might run in race condition with probe() or remove().
		 */
		pr_err(
			"%s: DROP NOTIFY, no device or no kobject context. event=%lu, pdev=%s, kobj=%s\n",
			__func__,
			event,
			(dev ?  "VALID" : "NULL"),
			(kobj ? "VALID" : "NULL")
		);
		goto out;
	}

	if (event) {
		if (!(ht->is_hot)) {
			dev_warn(dev, "NOTIFY hot.\n");
			ht->is_hot = true;
			sysfs_notify(kobj, NULL, "is_hot");
		}
	} else {
		if (ht->is_hot) {
			dev_warn(dev, "NOTIFY cool.\n");
			ht->is_hot = false;
			sysfs_notify(kobj, NULL, "is_hot");
		}
	}
out:
	up(&(ht->sem));
	return NOTIFY_OK;
}

static struct notifier_block thermal_hot_pm_notifier = {
	.notifier_call = thermal_hot_pm_notify,
};

static ssize_t hot_show(struct kobject *kobj,
	struct kobj_attribute *attr, char *buf)
{
	struct notify_hightemp	*ht = &(notify_hightemp);
	int	ret;
	int	hot;

	ret = down_interruptible(&(ht->sem));
	if (ret != 0) {
		pr_err("%s: Can not down semaphore. ret=%d\n",
			__func__,
			ret
		);
		return ret;
	}
	hot = ht->is_hot;
	up(&(ht->sem));
	return snprintf(buf, PAGE_SIZE, "%d\n", hot);
}

static struct kobj_attribute hot_attribute =
	__ATTR(is_hot, 0444, hot_show, NULL);

static struct attribute *attrs[] = {
	&hot_attribute.attr,
	NULL,
};

static struct attribute_group attr_group = {
	.attrs = attrs,
};

static int notify_hightemp_probe(struct platform_device *pdev)
{
	int ret;
	struct notify_hightemp	*ht = &(notify_hightemp);
	struct device	*dev;
	struct kobject	*kobj;

	dev = &(pdev->dev);
	down(&(ht->sem));

	if ((ht->dev) || (ht->kobj)) {
		dev_err(dev,
			"Can not register twice or more. dev=%s, kobj=%s\n",
			(ht->dev  ? "EXIST" : "NULL"),
			(ht->kobj ? "EXIST" : "NULL")
		);
		ret = -EEXIST;
		goto out;
	}

	kobj = kobject_create_and_add("notify_hightemp", kernel_kobj);
	if (!kobj) {
		ret = -ENOMEM;
		dev_err(dev,
			"Can not create notify_hightemp kernel object. ret=%d\n",
			ret
		);
		goto out;
	}

	ht->dev = dev;
	ht->kobj = kobj;

	ret = sysfs_create_group(kobj, &attr_group);
	if (ret) {
		dev_err(dev, "Can not create hot_show node. ret=%d\n",
			ret
		);
		goto out_put_kobj;
	}

	/* Release lock here,
	 * May register_devfreq_cooling_notifier() calls back us?
	 * Currently, register_devfreq_cooling_notifier() does only
	 * link function into call back chain.
	 */
	up(&(ht->sem));

	ret = register_devfreq_cooling_notifier(&thermal_hot_pm_notifier);
	if (ret != 0) {
		dev_err(dev,
			"Can not register notifier call back. ret=%d\n",
			ret
		);
		down(&(ht->sem));
		goto out_remove_group;
	}
	dev_info(dev, "Register is_hot notifier.\n");
	return 0;

out_remove_group:
	sysfs_remove_group(kobj, &attr_group);
out_put_kobj:
	kobject_put(kobj);
	ht->dev = NULL;
	ht->kobj = NULL;
out:
	up(&(ht->sem));
	return ret;
}

static int notify_hightemp_remove(struct platform_device *pdev)
{
	struct notify_hightemp	*ht = &(notify_hightemp);
	struct kobject	*kobj;

	dev_info(&(pdev->dev), "Unregister is_hot notifier.\n");

	unregister_devfreq_cooling_notifier(&thermal_hot_pm_notifier);

	down(&(ht->sem));
	kobj = ht->kobj;
	if (kobj) {
		/* We have node to notify. */
		ht->kobj = NULL;
		sysfs_remove_group(kobj, &attr_group);
		kobject_put(kobj);
	}

	ht->dev = NULL;
	up(&(ht->sem));
	return 0;
}

static const struct of_device_id notify_hightemp_dt_ids[] = {
	{
	.compatible = "sony,notify_hightemp",
	},
	{}};
MODULE_DEVICE_TABLE(of, notify_hightemp_dt_ids);

static struct platform_driver notify_hightemp_driver = {
	.probe = notify_hightemp_probe,
	.remove = notify_hightemp_remove,
	.driver = {
		.name = DRV_NAME,
		.owner = THIS_MODULE,
		.of_match_table = notify_hightemp_dt_ids,
	},
};

module_platform_driver(notify_hightemp_driver);
MODULE_DESCRIPTION("Notify High Temp Driver");
MODULE_AUTHOR("Sony Home Entertainment & Sound Products Inc.");
MODULE_LICENSE("GPL");
MODULE_VERSION("v1.01");
