// SPDX-License-Identifier: GPL-2.0

/* sony_ac_cut.c: Sony AC-Cut Driver
 * Copyright 2022 Sony Corporation
 */
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/delay.h>
#include <linux/gpio.h>
#include <linux/interrupt.h>
#include <linux/of_irq.h>
#include <linux/of_gpio.h>
#include <linux/of_device.h>
#include <linux/platform_device.h>
#include <linux/reboot.h>
#include <linux/cm4.h>

#define DRV_NAME		"sony_ac_cut"

#define PINCTR_IOMUXC_BASE	0x30330000
#define PINCTR_IOMUXC_REG_SIZE	0xFFFF

#define ERRLOG(fmt, ...)	pr_err(DRV_NAME ": Error: " fmt, \
					## __VA_ARGS__)
#define WRNLOG(fmt, ...)	pr_warn(DRV_NAME ": Warning: " fmt, \
					## __VA_ARGS__)
#define MSGLOG(fmt, ...)	pr_info(DRV_NAME ": " fmt, ## __VA_ARGS__)
#define DBGLOG(fmt, ...)	pr_debug(DRV_NAME ": " fmt, ## __VA_ARGS__)

struct pinctrl_detail {
	void __iomem		*mux_addr;	/* Pad Mux Register        */
	void __iomem		*ctrl_addr;	/* Pad Control Register    */
	void __iomem		*sel_addr;	/* SelectInput Register    */
	int			alt;
	int			ctrl;
	int			sel;
};

struct pinctrl_group {
	struct pinctrl_detail	*pins;
	int 			count;
};

struct gpio_param {
	int			pin;
	enum of_gpio_flags	flags;
};

struct gpio_group {
	struct gpio_param	*pins;
	int			count;
};

/* private data */
struct priv {
	struct gpio_desc	*accut_det;	/* detect AC-Cut GPIO      */
	int			irq;		/* AC-Cut IRQ number       */
	unsigned int		det_delay;	/* AC-Cut detect delay     */
	struct gpio_group	mute_gpios;	/* Mute gpio group         */
	unsigned int		mute_delay;	/* after mute delay        */
	struct pinctrl_group	sai_pins;	/* SAI pinctrl group       */
	unsigned int		stage_cnt;	/* sequence stage counter  */
	unsigned int 		*stage_delay;	/* sequence control(delay) */
	struct gpio_group	*stage_gpios;	/* sequence control(gpio)  */
	bool			no_restart;	/* no restart after AC-Cut */
};
struct priv *save_priv;

/*----------------------------------------------------------------------*/
/*	control function						*/
/*----------------------------------------------------------------------*/
static void set_gpio_pin(struct gpio_param *gpio, int active)
{
	active = (gpio->flags & OF_GPIO_ACTIVE_LOW) ? !active : active;
	DBGLOG("set gpio%d=%d\n", gpio->pin, active);
	if (gpio_cansleep(gpio->pin))
		gpio_set_value_cansleep(gpio->pin, active);
	else
		gpio_set_value(gpio->pin, active);
}

static void set_gpio_group(struct gpio_group *gpios, int active)
{
	int i;

	if (!gpios)
		return;

	for (i = 0; i < gpios->count; i++)
		set_gpio_pin(&gpios->pins[i], active);
}

static void set_pinctrl_group(struct pinctrl_group *pingrp)
{
	int i;

	if (!pingrp)
		return;

	for (i = 0; i < pingrp->count; i++) {
		iowrite32(pingrp->pins[i].alt, pingrp->pins[i].mux_addr);
		if (pingrp->pins[i].sel_addr)
			iowrite32(pingrp->pins[i].sel, pingrp->pins[i].sel_addr);
		iowrite32(pingrp->pins[i].ctrl, pingrp->pins[i].ctrl_addr);
	}
}

static int get_gpio_group(struct device *dev,
			  struct gpio_group *gpios, char *name)
{
	enum of_gpio_flags	flags;
	int			i, cnt, pin, result;

	cnt = of_gpio_named_count(dev->of_node, name);
	if (cnt > 0) {
		MSGLOG("detect %s(%d) gpio-pins\n", name, cnt);
		gpios->count = cnt;
		gpios->pins = devm_kzalloc(dev,
				(size_t)cnt * sizeof(struct gpio_param),
				 GFP_KERNEL);
		if (IS_ERR(gpios->pins)) {
			result = PTR_ERR(gpios->pins);
			ERRLOG("failed to allocate gpio %s(%d)\n",
				name, result);
			return result;
		}

		for (i = 0; i < gpios->count; i++) {
			pin = of_get_named_gpio_flags(dev->of_node, name,
						      i, &flags);
			if (pin < 0) {
				if (pin == -EPROBE_DEFER) {
					ERRLOG("gpio %s(i) is busy(%d)\n",
						name, i);
					return -EPROBE_DEFER;
				}
				ERRLOG("get_gpio %s(%d) failed(%d) \n",
					name, i, pin);
				return pin;
			} else {
				gpios->pins[i].pin = pin;
				gpios->pins[i].flags = flags;
			}
		}
	}

	return 0;
}

static int get_pinctrl_group(struct device *dev,
			     struct pinctrl_group *pingrp, char *name)
{
	void __iomem	*iomuxc_base;
	int		i, cnt, result;
	unsigned int	mux_off, ctrl_off, sel_off, alt, ctrl, sel;

	iomuxc_base = ioremap(PINCTR_IOMUXC_BASE, PINCTR_IOMUXC_REG_SIZE);

	cnt = of_property_count_u32_elems(dev->of_node, name);
	if (cnt > 0) {
		MSGLOG("detect %s(%d) pinctrls\n", name, cnt / 6);
		pingrp->count = cnt / 6;
		pingrp->pins = devm_kzalloc(dev,
				(size_t)cnt * sizeof(struct pinctrl_detail),
				GFP_KERNEL);
		if (IS_ERR(pingrp->pins)) {
			result = PTR_ERR(pingrp->pins);
			ERRLOG("failed to allocate pinctrl %s(%d)\n",
				name, result);
			return result;
		}

		for (i = 0; i < pingrp->count; i++) {
			if (!of_property_read_u32_index(dev->of_node,
						name, i * 6, &mux_off)
			 && !of_property_read_u32_index(dev->of_node,
						name, i * 6 + 1, &ctrl_off)
			 && !of_property_read_u32_index(dev->of_node,
						name, i * 6 + 2, &sel_off)
			 && !of_property_read_u32_index(dev->of_node,
						name, i * 6 + 3, &alt)
			 && !of_property_read_u32_index(dev->of_node,
						name, i * 6 + 4, &sel)
			 && !of_property_read_u32_index(dev->of_node,
						name, i * 6 + 5, &ctrl)) {
				pingrp->pins[i].mux_addr =iomuxc_base + mux_off;
				pingrp->pins[i].ctrl_addr =iomuxc_base + ctrl_off;
				if (sel_off)
					pingrp->pins[i].sel_addr =
							iomuxc_base + sel_off;
				else
					pingrp->pins[i].sel_addr = 0x00;
				pingrp->pins[i].alt = alt;
				pingrp->pins[i].sel = sel;
				pingrp->pins[i].ctrl = ctrl;
			} else {
				ERRLOG("get_pinctrl %s(%d) failed\n", name, i);
				return -EINVAL;
			}
		}
	}

	return 0;
}

/*----------------------------------------------------------------------*/
/*	AC-Cut IRQ thread entry						*/
/*----------------------------------------------------------------------*/
static void ac_cut_irq_proc(struct priv *p_priv)
{
	int i;

	/* Chattering check */
	if (p_priv->det_delay) {
		usleep_range(p_priv->det_delay * 1000,
			     p_priv->det_delay * 1000 + 10);

		/* Detect AC-Cut again to remove noise */
		if (gpiod_get_value(p_priv->accut_det) == 0)
			return;
	}

	MSGLOG("Start force shutdown sequence\n");

	/* Set Mute GPIOs */
	set_gpio_group(&p_priv->mute_gpios, 1);
	if (p_priv->mute_delay)
		usleep_range(p_priv->mute_delay * 1000,
			     p_priv->mute_delay * 1000 + 10);

	/* Set SAI PINs */
	set_pinctrl_group(&p_priv->sai_pins);

	/* Set PCONT GPIOs */
	for (i = 0; i < p_priv->stage_cnt; i++) {
		if (p_priv->stage_delay[i])
			usleep_range(p_priv->stage_delay[i] * 1000,
				     p_priv->stage_delay[i] * 1000 + 10);
		set_gpio_group(&p_priv->stage_gpios[i], 0);
	}

	rpmsg_cm4_accut_notifier();

	MSGLOG("end force shutdown sequence\n");

	/* Execute emergency restart */
	if (!p_priv->no_restart) {
		MSGLOG("Restart after 3s\n");
		usleep_range(3000000, 3001000);
		emergency_restart();
	}
}

static irqreturn_t ac_cut_irq_thread(int irq_num, void *data)
{
	ac_cut_irq_proc((struct priv *)data);

	return IRQ_HANDLED;
}

/*----------------------------------------------------------------------*/
/*	debug function							*/
/*----------------------------------------------------------------------*/
static int dummy_detect_set(const char *buf, const struct kernel_param *kp)
{
	if (!save_priv)
		return -1;

	if (sysfs_streq(buf, "run\n")) {
		save_priv->no_restart = 0;
		ac_cut_irq_proc(save_priv);
	} else if (sysfs_streq(buf, "debug\n")) {
		save_priv->no_restart = 1;
		ac_cut_irq_proc(save_priv);
	}

	return 0;
}

static struct kernel_param_ops dummy_detect_ops = {
	.set = dummy_detect_set,
};
module_param_cb(dummy_detect, &dummy_detect_ops, NULL, 0644);

/*----------------------------------------------------------------------*/
/*	platform device probe						*/
/*----------------------------------------------------------------------*/
static int ac_cut_probe(struct platform_device *p_dev)
{
	struct device	*dev = &p_dev->dev;
	struct priv	*p_priv;
	int		i, result;
	unsigned int	dly;
	char		name[32];

	/* Allocate memory for private data */
	p_priv = devm_kzalloc(dev, sizeof(struct priv), GFP_KERNEL);
	if (IS_ERR(p_priv)) {
		result = PTR_ERR(p_priv);
		ERRLOG("private memory allocate(%d)\n", result);
		return result;
	}

	/* Get sequence control stages */
	do {
		snprintf(name, 32, "stage%d-gpios", p_priv->stage_cnt + 1);
		result = of_gpio_named_count(dev->of_node, name);
		if (result == -ENOENT) {
			break;
		} else if (result < 0) {
			ERRLOG("invalid stage(%d)(%d)\n",
				p_priv->stage_cnt, result);
			return result;
		}
		p_priv->stage_cnt++;
	} while(result);

	/* Allocate memory for stage delay */
	p_priv->stage_delay = devm_kzalloc(dev,
				p_priv->stage_cnt * sizeof(unsigned int),
				GFP_KERNEL);
	if (IS_ERR(p_priv->stage_delay)) {
		result = PTR_ERR(p_priv->stage_delay);
		ERRLOG("stage delay memory allocate(%d)\n", result);
		return result;
	}

	/* Allocate memory for stage gpios */
	p_priv->stage_gpios = devm_kzalloc(dev,
				p_priv->stage_cnt * sizeof(struct gpio_group),
				GFP_KERNEL);
	if (IS_ERR(p_priv->stage_gpios)) {
		result = PTR_ERR(p_priv->stage_gpios);
		ERRLOG("stage gpios memory allocate(%d)\n", result);
		return result;
	}

	/* Get AC-Cut detection gpio */
	p_priv->accut_det = devm_gpiod_get(dev, "accut", GPIOD_IN);
	if (IS_ERR(p_priv->accut_det)) {
		result = PTR_ERR(p_priv->accut_det);
		ERRLOG("AC-Cut gpio is not detect(%d)\n", result);
		return result;
	}

	/* Get Mute-GPIOs */
	result = get_gpio_group(dev, &p_priv->mute_gpios, "mute-gpios");
	if (result) {
		ERRLOG("get mute-gpios(%d)\n", result);
		return result;
	}

	/* Get SAI-PINs */
	result = get_pinctrl_group(dev, &p_priv->sai_pins, "sai-pins");
	if (result) {
		ERRLOG("get sai-pins(%d)\n", result);
		return result;
	}

	/* Get PCONT-GPIOs with sequence control */
	for (i = 0; i < p_priv->stage_cnt; i++) {
		snprintf(name, 32, "stage%d-gpios", i + 1);
		result = get_gpio_group(dev, &p_priv->stage_gpios[i], name);
		if (result) {
			ERRLOG("get sequence control gpios(%d)\n", result);
			return result;
		}
	}

	/* Get AC-Cut Re-detect delay value */
	p_priv->det_delay = 0;
	result = device_property_read_u32(dev, "detection-delay", &dly);
	if (!result) {
		p_priv->det_delay = dly;
		MSGLOG("detect detection-delay(%d mSec)\n", dly);
	}

	/* Get Mute delay value */
	p_priv->mute_delay = 0;
	result = device_property_read_u32(dev, "mute-delay", &dly);
	if (!result) {
		p_priv->mute_delay = dly;
		MSGLOG("detect mute-delay(%d mSec)\n", dly);
	}

	/* Get sequence delay value */
	for (i = 0; i < p_priv->stage_cnt; i++) {
		p_priv->stage_delay[i] = 0;
		snprintf(name, 32, "stage%d-delay", i + 1);
		result = device_property_read_u32(dev, name, &dly);
		if (!result) {
			p_priv->stage_delay[i] = dly;
			MSGLOG("detect stage%d-delay(%d mSec)\n", i + 1, dly);
		}
	}

	/* Get other settings */
	p_priv->no_restart = of_property_read_bool(
				dev->of_node, "no-restart");

	/* Set irq handler */
	p_priv->irq = gpiod_to_irq(p_priv->accut_det);
	DBGLOG("AC-Cut irq=%d\n", p_priv->irq);
	result = devm_request_threaded_irq(dev, p_priv->irq,
			NULL, ac_cut_irq_thread,
			IRQF_SHARED | IRQF_TRIGGER_FALLING | IRQF_ONESHOT,
			dev_name(dev), p_priv);
	if (result) {
		ERRLOG("request irq handler(%d)\n", result);
		return result;
	}

	/* Set private data */
	dev_set_drvdata(dev, p_priv);
	save_priv = p_priv;

	return 0;
}

/*----------------------------------------------------------------------*/
/*	platform device remove						*/
/*----------------------------------------------------------------------*/
static int ac_cut_remove(struct platform_device *p_dev)
{
	struct device	*dev = &p_dev->dev;
	struct priv	*p_priv = dev_get_drvdata(dev);

	devm_free_irq(dev, p_priv->irq, p_priv);

	return 0;
}

/* -------------------------------------------------------------------- */
/*	Configration platform device driver				*/
/* -------------------------------------------------------------------- */
static const struct of_device_id ac_cut_dt_ids[] = {
	{ .compatible = "sony,sony_ac_cut", },
	{}
};
MODULE_DEVICE_TABLE(of, ac_cut_dt_ids);

static struct platform_driver ac_cut_driver = {
	.probe     = ac_cut_probe,
	.remove    = ac_cut_remove,
	.driver    = {
		.name  = DRV_NAME,
		.owner = THIS_MODULE,
		.of_match_table = ac_cut_dt_ids,
	},
};

module_platform_driver(ac_cut_driver);
MODULE_DESCRIPTION("Sony AC-Cut Detect Driver");
MODULE_AUTHOR("Sony Corporation");
MODULE_LICENSE("GPL");
MODULE_VERSION("v1.00");
