// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright 2020, 2021, 2022, 2023 Sony Corporation
 */

#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/of_gpio.h>

#include <misc/sw_pll.h>
#include <sound/sony-asoc-card.h>
#include <sound/tas5825.h>
#include <misc/simple_consumer.h>
#include <linux/gpio/consumer.h>

/* #define TRACE_PRINT_ON */
#define TRACE_TAG "####### "

/* trace print macro */
#ifdef TRACE_PRINT_ON
	#define print_trace(fmt, args...) pr_info(TRACE_TAG "" fmt, ##args)
#else
	#define print_trace(fmt, args...) pr_debug(TRACE_TAG "" fmt, ##args)
#endif

#define SONY_HW_MUTE_ON       1
#define SONY_HW_MUTE_OFF      0

enum mute_factor_index {
	FS_CHANGE,
	ALSA_TRIG,
	FCT_NUM,
};

struct mute_factor {
	unsigned int mute;
	unsigned int enable;
};

struct sony_hw_mute_priv {
	struct device *dev;
	struct mutex mute_mutex;
	struct mute_factor fct[FCT_NUM];
	struct workqueue_struct *mute_wq;
	struct delayed_work mute_off_work;
	struct gpio_desc *gpio_soc_mute;
	struct gpio_desc *gpio_fs_change_det;

	unsigned int swpll_mute_req;
	unsigned int mute_off_delay;
};

/* only used in callback */
static struct sony_hw_mute_priv *sony_hw_mute_drvdata;

static void mute_off_work_func(struct work_struct *work)
{
	struct delayed_work *dwork = to_delayed_work(work);
	struct sony_hw_mute_priv *priv = container_of(
		dwork, struct sony_hw_mute_priv, mute_off_work);

	gpiod_set_value(priv->gpio_soc_mute, 0);
	print_trace("%s(): mute=0\n", __func__);
}

static void sony_hw_mute_update(struct sony_hw_mute_priv *priv)
{
	unsigned char i;
	unsigned int mute = 0;

	print_trace("%s()\n", __func__);

	for (i = 0; i < FCT_NUM; i++)
		if (priv->fct[i].enable)
			mute |= priv->fct[i].mute;

	if (mute == SONY_HW_MUTE_ON) {
		cancel_delayed_work(&priv->mute_off_work);
		gpiod_set_value(priv->gpio_soc_mute, 1);
	} else {
		queue_delayed_work(priv->mute_wq,
			&priv->mute_off_work,
			msecs_to_jiffies(priv->mute_off_delay));
	}
	print_trace("%s(): mute=%u\n", __func__, mute);
}

static ssize_t status_show(struct device *dev,
				   struct device_attribute *attr,
				   char *buf)
{
	struct sony_hw_mute_priv *priv = dev_get_drvdata(dev);
	ssize_t rb = 0;

	rb += snprintf(buf + rb, PAGE_SIZE - rb,
			"[fs_change] mute:%d, enable:%d\n",
			priv->fct[FS_CHANGE].mute, priv->fct[FS_CHANGE].enable);
	rb += snprintf(buf + rb, PAGE_SIZE - rb,
			"[alsa_trig] mute:%d, enable:%d\n",
			priv->fct[ALSA_TRIG].mute, priv->fct[ALSA_TRIG].enable);
	rb += snprintf(buf + rb, PAGE_SIZE - rb,
			"swpll_mute_req:%d\n",
			priv->swpll_mute_req);
	rb += snprintf(buf + rb, PAGE_SIZE - rb,
			"soc_mute:%d\n",
			gpiod_get_value(priv->gpio_soc_mute));
	if (priv->gpio_fs_change_det) {
		rb += snprintf(buf + rb, PAGE_SIZE - rb,
				"fs_change_det:%d\n",
				gpiod_get_value(priv->gpio_fs_change_det));
	}

	return rb;
}

static ssize_t fs_change_enable_store(struct device *dev,
				      struct device_attribute *attr,
				      const char *buf, size_t size)
{
	struct sony_hw_mute_priv *priv = dev_get_drvdata(dev);
	long enable;

	if (kstrtol(buf, 0, &enable))
		return -EINVAL;

	priv->fct[FS_CHANGE].enable = enable;
	sony_hw_mute_update(priv);

	return size;
}

static ssize_t alsa_trigger_enable_store(struct device *dev,
					 struct device_attribute *attr,
					 const char *buf, size_t size)
{
	struct sony_hw_mute_priv *priv = dev_get_drvdata(dev);
	long enable;

	if (kstrtol(buf, 0, &enable))
		return -EINVAL;

	priv->fct[ALSA_TRIG].enable = enable;
	sony_hw_mute_update(priv);

	return size;
}

static ssize_t soc_mute_store(struct device *dev,
			      struct device_attribute *attr,
			      const char *buf, size_t size)
{
	struct sony_hw_mute_priv *priv = dev_get_drvdata(dev);
	long mute;

	if (kstrtol(buf, 0, &mute))
		return -EINVAL;

	if (gpiod_get_value(priv->gpio_soc_mute) != mute)
		gpiod_set_value(priv->gpio_soc_mute, mute);

	return size;
}

static ssize_t fs_change_det_store(struct device *dev,
			      struct device_attribute *attr,
			      const char *buf, size_t size)
{
	struct sony_hw_mute_priv *priv = dev_get_drvdata(dev);
	long fs_change_det;

	if (kstrtol(buf, 0, &fs_change_det))
		return -EINVAL;

	if (priv->gpio_fs_change_det &&
		gpiod_get_value(priv->gpio_fs_change_det) != fs_change_det)
		gpiod_set_value(priv->gpio_fs_change_det, fs_change_det);

	return size;
}

static DEVICE_ATTR_RO(status);
static DEVICE_ATTR_WO(fs_change_enable);
static DEVICE_ATTR_WO(alsa_trigger_enable);
static DEVICE_ATTR_WO(soc_mute);
static DEVICE_ATTR_WO(fs_change_det);

static struct attribute *attributes[] = {
	&dev_attr_status.attr,
	&dev_attr_fs_change_enable.attr,
	&dev_attr_alsa_trigger_enable.attr,
	&dev_attr_soc_mute.attr,
	&dev_attr_fs_change_det.attr,
	NULL,
};

static const struct attribute_group attr_group = {
	.attrs  = (struct attribute **)attributes,
};

/* callback function which is called by sw_pll driver */
static int hw_mute_fs_change_notify(struct notifier_block *self,
				unsigned long notused, void *event)
{
	SW_PLL_STAT *stat = (SW_PLL_STAT *)event;
	struct sony_hw_mute_priv *priv = sony_hw_mute_drvdata;

	print_trace("%s() Mute_Req=%d\n", __func__, stat->Mute_Req);

	priv->swpll_mute_req = stat->Mute_Req;
	mutex_lock(&priv->mute_mutex);
	if (stat->Mute_Req == SONY_HW_MUTE_ON) {
		priv->fct[FS_CHANGE].mute = 1;
		if (priv->gpio_fs_change_det && priv->fct[FS_CHANGE].enable)
			gpiod_set_value(priv->gpio_fs_change_det, 1);
	}
	sony_hw_mute_update(priv);
	mutex_unlock(&priv->mute_mutex);

	return NOTIFY_OK;
}

static struct notifier_block hw_mute_swpll_nb = {
	.notifier_call = hw_mute_fs_change_notify
};

/* callback function which is called by sony-asoc-card and tas5825 */
static int hw_mute_alsa_trigger_notify(struct notifier_block *self,
				    unsigned long mute, void *notused)
{
	struct sony_hw_mute_priv *priv = sony_hw_mute_drvdata;

	print_trace("%s() mute=%lu\n", __func__, mute);

	if (mute == SONY_HW_MUTE_ON) {
		priv->fct[ALSA_TRIG].mute = 1;
		if (priv->gpio_fs_change_det)
			gpiod_set_value(priv->gpio_fs_change_det, 1);
	} else {
		priv->fct[ALSA_TRIG].mute = 0;
		priv->fct[FS_CHANGE].mute = 0;
		if (priv->gpio_fs_change_det)
			gpiod_set_value(priv->gpio_fs_change_det, 0);
	}
	sony_hw_mute_update(priv);

	return NOTIFY_OK;
}

static struct notifier_block hw_mute_alsa_trigger_nb = {
	.notifier_call = hw_mute_alsa_trigger_notify
};

/* callback function which is called by simple_consumer */
static int hw_mute_simple_consumer_notify(struct notifier_block *self,
				unsigned long mute, void *notused)
{
	struct sony_hw_mute_priv *priv = sony_hw_mute_drvdata;

	print_trace("%s() mute=%lu\n", __func__, mute);

	mutex_lock(&priv->mute_mutex);
	if (mute == SONY_HW_MUTE_ON)
		gpiod_set_value(priv->gpio_soc_mute, 1);
	else
		gpiod_set_value(priv->gpio_soc_mute, 0);

	mutex_unlock(&priv->mute_mutex);

	return NOTIFY_OK;
}

static struct notifier_block hw_mute_simple_consumer_nb = {
	.notifier_call = hw_mute_simple_consumer_notify
};

/* platform driver */
static int sony_hw_mute_probe(struct platform_device *pdev)
{
	struct sony_hw_mute_priv *priv;
	int rv;

	print_trace("%s()\n", __func__);

	priv = devm_kzalloc(&pdev->dev, sizeof(struct sony_hw_mute_priv),
			     GFP_KERNEL);
	if (!priv)
		return -ENOMEM;

	mutex_init(&priv->mute_mutex);
	priv->dev = &pdev->dev;

	priv->swpll_mute_req = 0;
	priv->fct[FS_CHANGE].mute = 0;
	priv->fct[FS_CHANGE].enable = 1;
	priv->fct[ALSA_TRIG].mute = 1;
	priv->fct[ALSA_TRIG].enable = 1;
	priv->gpio_soc_mute = devm_gpiod_get(&pdev->dev,
						"hw_mute",
						GPIOD_OUT_HIGH);
	if (IS_ERR(priv->gpio_soc_mute)) {
		dev_err(&pdev->dev, "Failed to get hw_mute-gpios\n");
		return PTR_ERR(priv->gpio_soc_mute);
	}

	sony_hw_mute_update(priv);
	if (of_property_read_u32(pdev->dev.of_node,
				 "mute-off-delay",
				 &priv->mute_off_delay))
		priv->mute_off_delay = 0;

	sony_hw_mute_drvdata = priv;

	priv->gpio_fs_change_det = devm_gpiod_get(&pdev->dev,
						  "fs_change_det",
						  GPIOD_OUT_HIGH);
	if (IS_ERR(priv->gpio_fs_change_det)) {
		rv = PTR_ERR(priv->gpio_fs_change_det);
		if (rv != -ENOENT) {
			dev_err(&pdev->dev,
				"Failed to get fs_change_det-gpio(%d)\n",
				rv);
			return rv;
		}
		priv->gpio_fs_change_det = NULL;
	} else {
		dev_info(&pdev->dev, "Enable fs_change_det\n");
	}

	platform_set_drvdata(pdev, priv);

	rv = sysfs_create_group(&pdev->dev.kobj, &attr_group);
	if (rv) {
		dev_err(&pdev->dev,
			"Failed to sysfs_create_group(%d)\n",
			rv);
		goto out1;
	}

	priv->mute_wq = create_singlethread_workqueue("sony_hw_mute_wq");
	if (!priv->mute_wq) {
		dev_err(&pdev->dev,
			"Failed to create_singlethread_workqueue\n");
		rv = -ENOMEM;
		goto out2;
	}
	INIT_DELAYED_WORK(&priv->mute_off_work, mute_off_work_func);

	rv = register_sw_pll_notifier(&hw_mute_swpll_nb);
	if (rv) {
		dev_err(&pdev->dev,
			"Failed to register_sw_pll_notifier(%d)\n",
			rv);
		goto out3;
	}

	rv = register_sony_asoc_card_notifier(&hw_mute_alsa_trigger_nb);
	if (rv) {
		dev_err(&pdev->dev,
			"Failed to register_sony_asoc_card_notifier(%d)\n",
			rv);
		goto out4;
	}

	rv = register_tas5825_notifier(&hw_mute_alsa_trigger_nb);
	if (rv) {
		dev_err(&pdev->dev,
			"Failed to register_tas5825_notifier(%d)\n",
			rv);
		goto out5;
	}

	rv = register_simple_consumer_notifier(&hw_mute_simple_consumer_nb);
	if (rv) {
		dev_err(&pdev->dev,
			"Failed to register_simple_consumer_notifier(%d)\n",
			rv);
		goto out6;
	}

	return 0;
out6:
	unregister_tas5825_notifier(&hw_mute_alsa_trigger_nb);
out5:
	unregister_sony_asoc_card_notifier(&hw_mute_alsa_trigger_nb);
out4:
	unregister_sw_pll_notifier(&hw_mute_swpll_nb);
out3:
	destroy_workqueue(priv->mute_wq);
out2:
	sysfs_remove_group(&pdev->dev.kobj, &attr_group);
out1:
	return rv;
}

static int sony_hw_mute_remove(struct platform_device *pdev)
{
	struct sony_hw_mute_priv *priv = dev_get_drvdata(&pdev->dev);

	print_trace("%s()\n", __func__);

	gpiod_set_value(priv->gpio_soc_mute, 0);
	if (priv->gpio_fs_change_det)
		gpiod_set_value(priv->gpio_fs_change_det, 1);

	cancel_delayed_work_sync(&priv->mute_off_work);
	destroy_workqueue(priv->mute_wq);

	unregister_sw_pll_notifier(&hw_mute_swpll_nb);
	unregister_sony_asoc_card_notifier(&hw_mute_alsa_trigger_nb);
	unregister_tas5825_notifier(&hw_mute_alsa_trigger_nb);
	unregister_simple_consumer_notifier(&hw_mute_simple_consumer_nb);

	sysfs_remove_group(&pdev->dev.kobj, &attr_group);

	return 0;
}

static int sony_hw_mute_suspend(struct device *dev)
{
	struct sony_hw_mute_priv *priv = dev_get_drvdata(dev);

	print_trace("%s()\n", __func__);

	gpiod_set_value(priv->gpio_soc_mute, 0);
	if (priv->gpio_fs_change_det)
		gpiod_set_value(priv->gpio_fs_change_det, 1);

	return 0;
}

static int sony_hw_mute_resume(struct device *dev)
{
	print_trace("%s()\n", __func__);

	return 0;
}

static const struct dev_pm_ops sony_hw_mute_pm_ops = {
	.suspend = sony_hw_mute_suspend,
	.resume = sony_hw_mute_resume,
};

#if defined(CONFIG_OF)
static const struct of_device_id sony_hw_mute_ids[] = {
	{.compatible = "sony,hw-mute",},
	{}
};
#endif

static struct platform_driver sony_hw_mute_driver = {
	.driver = {
		.name = "sony-hw-mute",
		.owner = THIS_MODULE,
		.pm = &sony_hw_mute_pm_ops,
#if defined(CONFIG_OF)
		.of_match_table = sony_hw_mute_ids,
#endif
	},
	.probe = sony_hw_mute_probe,
	.remove = sony_hw_mute_remove
};

module_platform_driver(sony_hw_mute_driver);

/* Module information */
MODULE_DESCRIPTION("sony hw mute driver");
MODULE_AUTHOR("Sony Corporation");
MODULE_LICENSE("GPL");
MODULE_VERSION("v1.00");
