// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright 2019, 2021, 2022, 2023 Sony Corporation
 *
 */

#include <linux/module.h>
#include <linux/string.h>
#include <linux/err.h>
#include <linux/platform_device.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/regulator/driver.h>
#include <linux/regulator/machine.h>
#include <linux/regulator/of_regulator.h>
#include <linux/pm_runtime.h>
#include <linux/device.h>
#include <linux/i2c.h>
#include <linux/spi/spi.h>
#include <linux/pinctrl/consumer.h>

/* Max support pm-sync devices count */
#define MAX_PM_SYNC_DEV_COUNT	(4)

struct srcon_data {
#ifdef CONFIG_REGULATOR_INDEPEND
	struct device *dev;
	int num_independ;
	struct regulator_bulk_data *independs;
	bool independ_sync;
#endif /* CONFIG_REGULATOR_INDEPEND */
	const char *srcon_name;
	int num_supplies;
	struct regulator_bulk_data *supplies;
	bool suspend_off;
	struct pinctrl *pinctrl;
	struct pinctrl_state *pins_state, *pins_state_inactive;
	struct device *pm_dev[MAX_PM_SYNC_DEV_COUNT];
	int pm_dev_count;
};

static BLOCKING_NOTIFIER_HEAD(simple_consumer_notify_list);

static struct bus_type *bus_list[] = {
	&platform_bus_type,
	&i2c_bus_type,
	&spi_bus_type
};
#define BUS_TYPE_COUNT (sizeof(bus_list) / sizeof(bus_list[0]))

static int of_dev_node_match(struct device *dev, const void *data)
{
	return dev->of_node == data;
}

static struct device *of_find_bus_device_by_node(struct device_node *np)
{
	struct device *dev = NULL;
	int bus;

	for (bus = 0; bus < BUS_TYPE_COUNT; bus++) {
		dev = bus_find_device(bus_list[bus], NULL,
					 np, of_dev_node_match);
		if (dev)
			break;
	}
	if (dev)
		pr_debug("%s: bus(%d): %p\n", __func__, bus, dev);
	return dev;
}

int register_simple_consumer_notifier(struct notifier_block *nb)
{
	return blocking_notifier_chain_register(&simple_consumer_notify_list, nb);
}
EXPORT_SYMBOL_GPL(register_simple_consumer_notifier);

int unregister_simple_consumer_notifier(struct notifier_block *nb)
{
	return blocking_notifier_chain_unregister(&simple_consumer_notify_list, nb);
}
EXPORT_SYMBOL_GPL(unregister_simple_consumer_notifier);

#ifdef CONFIG_REGULATOR_INDEPEND
static ssize_t srcon_independ_store(struct device *dev,
	 struct device_attribute *attr, const char *buf, size_t count)
{
	struct srcon_data *p_con_data = dev_get_drvdata(dev);
	struct regulator *consumer;
	bool enable;
	int i;

	if (sysfs_streq(buf, "enable\n")) {
		enable = true;
	} else if (sysfs_streq(buf, "disable\n")) {
		enable = false;
	} else {
		dev_err(dev, "Configuring invalid mode\n");
		return count;
	}

	for (i = 0; i <  p_con_data->num_independ; i++) {
		consumer = p_con_data->independs[i].consumer;
		regulator_set_independ(consumer, enable);
	}

	return count;
}
static DEVICE_ATTR(independ, 0664, NULL, srcon_independ_store);

static struct attribute *srcon_dev_attrs[] = {
	&dev_attr_independ.attr,
	NULL
};

static const struct attribute_group srcon_dev_group = {
	.attrs = srcon_dev_attrs,
};

static const struct attribute_group *srcon_dev_groups[] = {
	&srcon_dev_group,
	NULL
};
#endif /* CONFIG_REGULATOR_INDEPEND */

static int srcon_probe(struct platform_device *pdev)
{
	int idx, ret, i;
	struct device *dev = &pdev->dev;
	struct device_node *np = dev->of_node;
	const char *name;
	struct srcon_data *p_con_data;
	struct device_node *devn;
	struct device *sdev;

	dev_dbg(dev, "consumer probe\n");
	p_con_data = devm_kzalloc(dev, sizeof(struct srcon_data), GFP_KERNEL);
	if (p_con_data == NULL) {
		dev_err(dev, "Failed to alloc mem\n");
		return -ENOMEM;
	}
	p_con_data->srcon_name = np->name;

	ret = of_property_count_strings(np, "supply-names");
	if (ret > 0) {
		p_con_data->num_supplies = ret;
		p_con_data->supplies = devm_kzalloc(dev,
				sizeof(struct regulator_bulk_data) * ret,
				GFP_KERNEL);
		if (p_con_data->supplies == NULL) {
			dev_err(dev, "Failed to alloc mem\n");
			return -ENOMEM;
		}
		for (i = 0; i < p_con_data->num_supplies; i++) {
			ret = of_property_read_string_index(np, "supply-names",
							    i, &name);
			if (ret < 0) {
				dev_err(dev,
					"Failed to parse supply-names (%d)\n",
					ret);
				return ret;
			}
			p_con_data->supplies[i].supply = name;
		}
	}

#ifdef CONFIG_REGULATOR_INDEPEND
	ret = of_property_count_strings(np, "independ_control");
	if (ret > 0) {
		p_con_data->num_independ = ret;
		p_con_data->independs = devm_kzalloc(dev,
				sizeof(struct regulator_bulk_data) * ret,
				GFP_KERNEL);
		if (p_con_data->independs == NULL) {
			dev_err(dev, "Failed to alloc mem\n");
			return -ENOMEM;
		}
		for (i = 0; i < p_con_data->num_independ; i++) {
			ret = of_property_read_string_index(np,
							    "independ_control",
							    i, &name);
			if (ret < 0) {
				dev_err(dev,
					"Failed to parse supply-names (%d)\n",
					ret);
				return ret;
			}
			p_con_data->independs[i].supply = name;
		}
	}

#endif /* CONFIG_REGULATOR_INDEPEND */
	for (idx = 0; idx < MAX_PM_SYNC_DEV_COUNT; idx++) {
		devn = of_parse_phandle(dev->of_node, "pm-sync-dev", idx);
		if (devn) {
			dev_dbg(dev, "pm-sync node-%d: %p\n", idx, devn);
			sdev = of_find_bus_device_by_node(devn);
			dev_dbg(dev, "pm-sync dev: %p\n", sdev);
			p_con_data->pm_dev[idx] = sdev;
			if (!sdev)
				return -EPROBE_DEFER;
			else
				dev_dbg(dev, "pm-sync data: %p\n",
						sdev->driver_data);
		} else {
			break;
		}
	}
	p_con_data->pm_dev_count = idx;

	ret = devm_regulator_bulk_get(dev, p_con_data->num_supplies, 
						p_con_data->supplies);
	if (ret) {
		dev_err(dev, "failed to get regulator (%d)\n", ret);
		if (ret == -EPROBE_DEFER)
			dev_info(dev, "probe defered\n");
		return ret;
	}

#ifdef CONFIG_REGULATOR_INDEPEND
	ret = devm_regulator_bulk_get(dev, p_con_data->num_independ,
						p_con_data->independs);
	if (ret) {
		dev_err(dev, "failed to get regulator (%d)\n", ret);
		if (ret == -EPROBE_DEFER)
			dev_info(dev, "probe defered\n");
		return ret;
	}

	p_con_data->independ_sync = of_property_read_bool(np, "independ_sync");
	if (p_con_data->independ_sync)
		dev_dbg(dev, "enable independ_sync\n");

#endif /* CONFIG_REGULATOR_INDEPEND */
	p_con_data->suspend_off = of_property_read_bool(np, "suspend_off");
	if (p_con_data->suspend_off)
		dev_dbg(dev, "enable suspend-off\n");

	p_con_data->pinctrl = devm_pinctrl_get(dev);
	if (!IS_ERR_OR_NULL(p_con_data->pinctrl)){
		p_con_data->pins_state =
			pinctrl_lookup_state(p_con_data->pinctrl, "default");
		p_con_data->pins_state_inactive =
			pinctrl_lookup_state(p_con_data->pinctrl, "inactive");
	}

	platform_set_drvdata(pdev, p_con_data);

#ifdef CONFIG_REGULATOR_INDEPEND
	/* register with sysfs */
	p_con_data->dev = dev;
	p_con_data->dev->groups = srcon_dev_groups;
	ret = sysfs_create_groups(&dev->kobj, p_con_data->dev->groups);
	if (ret)
		dev_err(dev, "failed to sysfs create(%d)\n", ret);
#endif /* CONFIG_REGULATOR_INDEPEND */

	pm_runtime_enable(dev);

	if (of_property_read_bool(np, "boot_on")) {
		dev_dbg(dev, "enable regulators\n");
		pm_runtime_forbid(dev);

		if (!IS_ERR_OR_NULL(p_con_data->pinctrl) &&
			!IS_ERR_OR_NULL(p_con_data->pins_state)){
			ret = pinctrl_select_state(p_con_data->pinctrl, p_con_data->pins_state);
			if (ret)
				dev_err(dev,
				"failed to set proper pins state: %d\n", ret);
		}
	}

	return ret;
}

static int srcon_remove(struct platform_device *pdev)
{
	dev_dbg(&pdev->dev,"%s called\n", __func__);
	if (pm_runtime_active(&pdev->dev)) {
		dev_err(&pdev->dev, "runtime still active\n");
		pm_runtime_allow(&pdev->dev);
	}

	pm_runtime_disable(&pdev->dev);
	return 0;
}

static void srcon_shutdown(struct platform_device *pdev)
{
	struct srcon_data *p_con_data = platform_get_drvdata(pdev);
	int ret;
#ifdef CONFIG_REGULATOR_INDEPEND
	int i;
	struct regulator *consumer;
#endif /* CONFIG_REGULATOR_INDEPEND */

	dev_dbg(&pdev->dev,"%s called\n", __func__);

	if (pm_runtime_active(&pdev->dev)) {
		dev_dbg(&pdev->dev, "runtime still active\n");
		if (!IS_ERR_OR_NULL(p_con_data->pins_state_inactive) &&
			!IS_ERR_OR_NULL(p_con_data->pinctrl)) {
			ret = pinctrl_select_state(p_con_data->pinctrl, p_con_data->pins_state_inactive);
			if (ret)
				dev_err(&pdev->dev,
					"failed to set proper pins state: %d\n", ret);
		}

		ret = regulator_bulk_disable(p_con_data->num_supplies,
				p_con_data->supplies);
		if (ret)
			dev_err(&pdev->dev, "Failed to disable regulators (%d)\n",
				ret);
#ifdef CONFIG_REGULATOR_INDEPEND
		if (p_con_data->independ_sync) {
			for (i = 0; i <  p_con_data->num_independ; i++) {
				consumer = p_con_data->independs[i].consumer;
				regulator_set_independ(consumer, 0);
			}
		}
#endif /* CONFIG_REGULATOR_INDEPEND */

		pm_runtime_allow(&pdev->dev);
	}

	if (pm_runtime_enabled(&pdev->dev)) {
		dev_dbg(&pdev->dev, "runtime still enable\n");
		pm_runtime_disable(&pdev->dev);
	}
	return;
}

static int srcon_suspend(struct device *dev)
{
	int idx, ret = 0;
	struct srcon_data *p_con_data = dev_get_drvdata(dev);
#ifdef CONFIG_REGULATOR_INDEPEND
	int i;
	struct regulator *consumer;
#endif /* CONFIG_REGULATOR_INDEPEND */

	dev_dbg(dev, "%s called\n", __func__);

	if (p_con_data
	 && p_con_data->suspend_off
	 && pm_runtime_active(dev)) {
		for (idx = 0; idx < p_con_data->pm_dev_count; idx++) {
			if (p_con_data->pm_dev[idx]->driver_data)
				pm_runtime_allow(p_con_data->pm_dev[idx]);
		}

		if (!IS_ERR_OR_NULL(p_con_data->pins_state_inactive) &&
			!IS_ERR_OR_NULL(p_con_data->pinctrl)) {
			ret = pinctrl_select_state(p_con_data->pinctrl, p_con_data->pins_state_inactive);
			if (ret) {
				dev_err(dev,
					"failed to set proper pins state: %d\n", ret);
				return ret;
			}
		}

		ret = regulator_bulk_disable(p_con_data->num_supplies,
					     p_con_data->supplies);
		if (ret)
			dev_err(dev, "Failed to disable regulators (%d)\n",
					ret);
#ifdef CONFIG_REGULATOR_INDEPEND
		if (p_con_data->independ_sync) {
			for (i = 0; i <  p_con_data->num_independ; i++) {
				consumer = p_con_data->independs[i].consumer;
				regulator_set_independ(consumer, 0);
			}
		}
#endif /* CONFIG_REGULATOR_INDEPEND */
	}

	return ret;
}

static int srcon_resume(struct device *dev)
{
	int idx, ret = 0;
	struct srcon_data *p_con_data = dev_get_drvdata(dev);
#ifdef CONFIG_REGULATOR_INDEPEND
	int i;
	struct regulator *consumer;
#endif /* CONFIG_REGULATOR_INDEPEND */

	dev_dbg(dev, "%s called\n", __func__);
	if (p_con_data
	 && p_con_data->suspend_off
	 && pm_runtime_active(dev)) {
#ifdef CONFIG_REGULATOR_INDEPEND
		if (p_con_data->independ_sync) {
			for (i = 0; i <  p_con_data->num_independ; i++) {
				consumer = p_con_data->independs[i].consumer;
				regulator_set_independ(consumer, 1);
			}
		}
#endif /* CONFIG_REGULATOR_INDEPEND */
		ret = regulator_bulk_enable(p_con_data->num_supplies,
					    p_con_data->supplies);
		if (ret)
			dev_err(dev, "Failed to enable regulators (%d)\n",
					ret);

		if (!IS_ERR_OR_NULL(p_con_data->pins_state) &&
			!IS_ERR_OR_NULL(p_con_data->pinctrl)) {
			ret = pinctrl_select_state(p_con_data->pinctrl, p_con_data->pins_state);
			if (ret) {
				dev_err(dev,
					"failed to set proper pins state: %d\n", ret);
				return ret;
			}
		}

		for (idx = 0; idx < p_con_data->pm_dev_count; idx++) {
			if (p_con_data->pm_dev[idx]->driver_data)
				pm_runtime_forbid(p_con_data->pm_dev[idx]);
		}
	}

	return ret;
}

static int srcon_runtime_suspend(struct device *dev)
{
	int idx, ret = 0;
	struct srcon_data *p_con_data = dev_get_drvdata(dev);
#ifdef CONFIG_REGULATOR_INDEPEND
	int i;
	struct regulator *consumer;
#endif /* CONFIG_REGULATOR_INDEPEND */

	dev_dbg(dev, "%s called\n", __func__);

	if (p_con_data) {
		for (idx = 0; idx < p_con_data->pm_dev_count; idx++) {
			if (p_con_data->pm_dev[idx]->driver_data)
				pm_runtime_allow(p_con_data->pm_dev[idx]);
		}

		if (!IS_ERR_OR_NULL(p_con_data->pins_state_inactive) &&
			!IS_ERR_OR_NULL(p_con_data->pinctrl)) {
			ret = pinctrl_select_state(p_con_data->pinctrl, p_con_data->pins_state_inactive);
			if (ret) {
				dev_err(dev,
					"failed to set proper pins state: %d\n", ret);
				return ret;
			}
		}

		ret = regulator_bulk_disable(p_con_data->num_supplies,
					     p_con_data->supplies);
		if (ret)
			dev_err(dev, "Failed to disable regulators (%d)\n",
					ret);
#ifdef CONFIG_REGULATOR_INDEPEND
		if (p_con_data->independ_sync) {
			for (i = 0; i <  p_con_data->num_independ; i++) {
				consumer = p_con_data->independs[i].consumer;
				regulator_set_independ(consumer, 0);
			}
		}
#endif /* CONFIG_REGULATOR_INDEPEND */
		blocking_notifier_call_chain(&simple_consumer_notify_list, 0, NULL);
	}

	return ret;
}

static int srcon_runtime_resume(struct device *dev)
{
	int idx, ret = 0;
	struct srcon_data *p_con_data = dev_get_drvdata(dev);
#ifdef CONFIG_REGULATOR_INDEPEND
	int i;
	struct regulator *consumer;
#endif /* CONFIG_REGULATOR_INDEPEND */

	dev_dbg(dev, "%s called\n", __func__);
	if (p_con_data) {
		blocking_notifier_call_chain(&simple_consumer_notify_list, 1, NULL);
#ifdef CONFIG_REGULATOR_INDEPEND
		if (p_con_data->independ_sync) {
			for (i = 0; i <  p_con_data->num_independ; i++) {
				consumer = p_con_data->independs[i].consumer;
				regulator_set_independ(consumer, 1);
			}
		}
#endif /* CONFIG_REGULATOR_INDEPEND */
		ret = regulator_bulk_enable(p_con_data->num_supplies,
					     p_con_data->supplies);
		if (ret)
			dev_err(dev, "Failed to enable regulators (%d)\n",
					ret);

		if (!IS_ERR_OR_NULL(p_con_data->pins_state) &&
			!IS_ERR_OR_NULL(p_con_data->pinctrl)) {
			ret = pinctrl_select_state(p_con_data->pinctrl, p_con_data->pins_state);
			if (ret) {
				dev_err(dev,
					"failed to set proper pins state: %d\n", ret);
				return ret;
			}
		}

		for (idx = 0; idx < p_con_data->pm_dev_count; idx++) {
			if (p_con_data->pm_dev[idx]->driver_data)
				pm_runtime_forbid(p_con_data->pm_dev[idx]);
		}
	}

	return ret;
}


static const struct dev_pm_ops srcon_pm_ops = {
	.suspend = srcon_suspend,
	.resume = srcon_resume,
	.runtime_suspend = srcon_runtime_suspend,
	.runtime_resume = srcon_runtime_resume,
};


static const struct of_device_id srcon_of_match[] = {
	{ .compatible = "sony,simple_regulator_consumer", },
	{},
};
MODULE_DEVICE_TABLE(of, srcon_of_match);

static struct platform_driver srcon_driver = {
	.probe		= srcon_probe,
	.remove		= srcon_remove,
	.shutdown       = srcon_shutdown,
	.driver  = {
		.name  = "simple_consumer",
		.owner = THIS_MODULE,
		.pm = &srcon_pm_ops,
		.of_match_table = of_match_ptr(srcon_of_match),
	},
};

static int __init srcon_init(void)
{
	return platform_driver_register(&srcon_driver);
}
module_init(srcon_init);

static void __exit srcon_exit(void)
{
	platform_driver_unregister(&srcon_driver);
}
module_exit(srcon_exit)

MODULE_DESCRIPTION("Simple regulator consumer driver");
MODULE_LICENSE("GPL");
