/* SPDX-License-Identifier: GPL-2.0 */
/*
 * cm4.c - CM4 communication driver
 *
 * Copyright 2019, 2020, 2022 Sony Corporation
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/err.h>
#include <linux/cdev.h>
#include <linux/notifier.h>
#include <linux/reboot.h>
#include <linux/moduleparam.h>
#include <linux/slab.h>

#include <linux/virtio.h>
#include <linux/rpmsg.h>
#include <linux/cm4.h>

/*******************************************************************************
 * Definition
 ******************************************************************************/
#define CM4_CALLBACK_TABLE_MAX_NUM 30
#define HANDSHAKE_MSG	"HelloFromA53"
#define CM4_CONSOLE_ON 1
#define CM4_CONSOLE_OFF 0

#define CM4_LOG_ERR(fmt, args...) pr_err("[%s]:[%d]" fmt, __func__, __LINE__, \
									##args)
#define CM4_LOG_WAR(fmt, args...) pr_warn("[%s]:[%d]" fmt, __func__, __LINE__, \
									##args)
#define CM4_LOG_INFO(fmt, args...) pr_info(fmt, ##args)
#define CM4_LOG_DBG(fmt, args...) pr_debug(fmt, ##args)


enum cm4_system_msg_id {
	CM4_SHUTDOWN_REQ = 0x01,
	CM4_ACCUT_REQ = 0x02,
};
enum cm4_console_msg_id {
	CM4_CONSOLE_ON_REQ = 0x01,
	CM4_CONSOLE_OFF_REQ = 0x02,
};

struct notification_info {
	u8 id;
	cm4_rx_cb_t func;
};

struct rpmsg_cm4_dev {
	struct rpmsg_device	*rpmsg_dev;
	struct mutex		state_lock;
};

/*******************************************************************************
 * Variables
 ******************************************************************************/
static int16_t callback_table_index;
static struct notification_info callback_table[CM4_CALLBACK_TABLE_MAX_NUM];
static struct rpmsg_cm4_dev *rpmsg_cm4_dev_info;

/*******************************************************************************
 * module paramater (set by bootloader)
 ******************************************************************************/
static int cm4_console_enabled = -1;
module_param_named(cm4_console, cm4_console_enabled, int, 0444);

/*******************************************************************************
 * Functions
 ******************************************************************************/

static char *create_cm4_message(enum cm4_message_id id, u8 type, u8 length,
							u8 seq, u8 *data)
{
	char *message_data = kmalloc(CM4_MSG_HEADER_LEN + length, GFP_ATOMIC);

	if (message_data != NULL) {
		message_data[0] = id;
		message_data[1] = type;
		message_data[2] = length;
		message_data[3] = seq;
		memcpy(&message_data[4], data, length);
	}
	return message_data;
}

static int search_regidtered_callback(u8 id)
{
	int i;

	for (i = 0; i < callback_table_index; i++) {
		if (callback_table[i].id == id)
			return i;
	}
	CM4_LOG_WAR("No match callback id\n");
	return -ENXIO;
}

int rpmsg_cm4_send(enum cm4_message_id id, u8 type, u8 length, u8 seq, u8 *data)
{
	int total_len;
	char *msg;
	int result;

	if (rpmsg_cm4_dev_info == NULL)
		return -ENODEV;

	if (rpmsg_cm4_dev_info->rpmsg_dev == NULL)
		return -ENODEV;

	total_len = CM4_MSG_HEADER_LEN + length;
	msg = create_cm4_message(id, type, length, seq, data);

	if (msg == NULL)
		return -ENOMEM;

	mutex_lock(&rpmsg_cm4_dev_info->state_lock);
	result = rpmsg_send(rpmsg_cm4_dev_info->rpmsg_dev->ept, msg, total_len);
	mutex_unlock(&rpmsg_cm4_dev_info->state_lock);
	kfree(msg);

	return result;
}
EXPORT_SYMBOL(rpmsg_cm4_send);

int rpmsg_cm4_register_callback(enum cm4_message_id id, cm4_rx_cb_t func)
{
	if (rpmsg_cm4_dev_info == NULL)
		return -ENODEV;

	if (callback_table_index >= CM4_CALLBACK_TABLE_MAX_NUM)
		return -ENOMEM;

	if (id < 0 || func == NULL)
		return -EINVAL;

	mutex_lock(&rpmsg_cm4_dev_info->state_lock);
	callback_table[callback_table_index].id = id;
	callback_table[callback_table_index].func = func;
	callback_table_index++;
	mutex_unlock(&rpmsg_cm4_dev_info->state_lock);
	CM4_LOG_INFO("Register_callback -> id %d\n", id);
	return 0;
}
EXPORT_SYMBOL(rpmsg_cm4_register_callback);

static int rpmsg_cm4_cb(struct rpmsg_device *dev, void *data, int len,
						void *priv, u32 src)
{
	unsigned char  *message = (unsigned char *)data;

	u8 id = message[0];
	u8 type = message[1];
	u8 length = message[2];
	u8 seq = message[3];
	int index = 0;

	CM4_LOG_DBG("Message receive from cm4\n");
	CM4_LOG_DBG("id: %x, type: %x, length: %d, seq: %d\n", id, type,
							length, seq);

	index = search_regidtered_callback(id);
	if (index >= 0) {
		if (callback_table[index].func == NULL)
			CM4_LOG_ERR("No callback func\n");
		else
			callback_table[index].func(type, length, seq,
							&message[4]);
	}

	return 0;
}

static int rpmsg_cm4_probe(struct rpmsg_device *rpmsg)
{
	int result;
	u8 msg_data, type, length, seq;
	u8 data[1] = {0};
	enum cm4_message_id id;

	CM4_LOG_DBG("rpmsg cm4 probe\n");
	rpmsg_cm4_dev_info = devm_kzalloc(&rpmsg->dev,
		sizeof(struct rpmsg_cm4_dev), GFP_KERNEL);

	if (!rpmsg_cm4_dev_info) {
		CM4_LOG_ERR("Can NOT allocate memory for rpmsg cm4\n");
		return -ENOMEM;
	}

	mutex_init(&rpmsg_cm4_dev_info->state_lock);

	CM4_LOG_INFO("New channel: 0x%x -> 0x%x!\n",
			rpmsg->src, rpmsg->dst);

	result = rpmsg_send(rpmsg->ept, HANDSHAKE_MSG, strlen(HANDSHAKE_MSG));

	if (result != 0) {
		CM4_LOG_ERR("Handshake message send failed: %d\n", result);
		return result;
	}
	rpmsg_cm4_dev_info->rpmsg_dev = rpmsg;

	msg_data = CM4_CONSOLE_OFF_REQ;

	if (cm4_console_enabled == CM4_CONSOLE_ON)
		msg_data = CM4_CONSOLE_ON_REQ;

	id = CM4_MESSAGE_ID_CONSOLE;
	type = CM4_MESSAGE_TYPE_REQ;
	length = 1;
	seq = 0;
	data[0] = msg_data;

	rpmsg_cm4_send(id, type, length, seq, data);

	return result;
}

static void rpmsg_cm4_remove(struct rpmsg_device *dev)
{
	CM4_LOG_DBG("Remove rpmsg device\n");
}

void rpmsg_cm4_accut_notifier(void)
{
	/* Send the accut proc request to CM4 */
	u8 id = CM4_MESSAGE_ID_SYSTEM;
	u8 type = CM4_MESSAGE_TYPE_REQ;
	u8 length = 1;
	u8 seq = 0;
	u8 data[1] = {CM4_ACCUT_REQ};

	rpmsg_cm4_send((enum cm4_message_id)id, type, length, seq, data);
}
EXPORT_SYMBOL(rpmsg_cm4_accut_notifier);

static int rpmsg_cm4_reboot_notifier(struct notifier_block *this,
					unsigned long code, void *unused)
{
	if (code == SYS_RESTART || code == SYS_POWER_OFF  || code == SYS_HALT) {
		enum cm4_message_id id = CM4_MESSAGE_ID_SYSTEM;
		u8 type = CM4_MESSAGE_TYPE_REQ;
		u8 length = 1;
		u8 seq = 0;
		u8 data[1] = {CM4_SHUTDOWN_REQ};

		CM4_LOG_INFO("Send cm4 shutdown req\n");
		rpmsg_cm4_send(id, type, length, seq, data);
	}
	return NOTIFY_DONE;
}

static struct notifier_block rpmsg_cm4_notifier = {
	.notifier_call = rpmsg_cm4_reboot_notifier,
};

static struct rpmsg_device_id rpmsg_cm4_id_table[] = {
	{ .name	= "multicore-cm4-channel" },
	{ },
};
MODULE_DEVICE_TABLE(rpmsg, rpmsg_cm4_id_table);

static const struct of_device_id rpmsg_cm4_dt_ids[] = {
	{ .compatible = "sony,rpmsg-cm4", },
	{ },
};
MODULE_DEVICE_TABLE(of, rpmsg_cm4_dt_ids);

static struct rpmsg_driver cm4_rpmsg_driver = {
	.drv = {
		.name =			"cm4_rpmsg_drv",
		.owner =		THIS_MODULE,
		.of_match_table =	rpmsg_cm4_dt_ids,
	},
	.id_table =	rpmsg_cm4_id_table,
	.probe =	rpmsg_cm4_probe,
	.callback =	rpmsg_cm4_cb,
	.remove =	rpmsg_cm4_remove,
};

static int __init rpmsg_cm4_init(void)
{
	int result;

	CM4_LOG_INFO("Register rpmsg cm4 driver\r\n");
	result = register_rpmsg_driver(&cm4_rpmsg_driver);
	register_reboot_notifier(&rpmsg_cm4_notifier);
	callback_table_index = 0;

	CM4_LOG_INFO("rpmsg cm4 event dispatcher initialized: %d\n", result);

	return result;
}

static void __exit rpmsg_cm4_exit(void)
{
	CM4_LOG_DBG("rpmsg cm4 exit process...\n");
	unregister_reboot_notifier(&rpmsg_cm4_notifier);
	unregister_rpmsg_driver(&cm4_rpmsg_driver);
	CM4_LOG_DBG("exit\n");
}

subsys_initcall(rpmsg_cm4_init);
module_exit(rpmsg_cm4_exit);

MODULE_AUTHOR("Sony Corporation");
MODULE_DESCRIPTION("rpmsg_cm4");
MODULE_LICENSE("GPL v2");
