// SPDX-License-Identifier: GPL-2.0
/*
 * host.c - ChipIdea USB host controller driver
 *
 * Copyright (c) 2012 Intel Corporation
 * Copyright 2022 Sony Corporation
 *
 * Author: Alexander Shishkin
 */

#include <linux/kernel.h>
#include <linux/io.h>
#include <linux/usb.h>
#include <linux/usb/hcd.h>
#include <linux/usb/chipidea.h>
#include <linux/regulator/consumer.h>
#include <linux/pinctrl/consumer.h>
#include <linux/semaphore.h>

#include "../host/ehci.h"

#include "ci.h"
#include "bits.h"
#include "host.h"
#include "otg.h"

static struct hc_driver __read_mostly ci_ehci_hc_driver;
static int (*orig_bus_suspend)(struct usb_hcd *hcd);
static int (*orig_bus_resume)(struct usb_hcd *hcd);

struct ehci_ci_priv {
	/* NOTE: We don't use reg_vbus and enabled,
	 * but keeps here.
	 */
	struct regulator *reg_vbus;
	bool enabled;
};

struct ci_hdrc_dma_aligned_buffer {
	void *kmalloc_ptr;
	void *old_xfer_buffer;
	u8 data[];
};


static int usb_vbus_event_send_direct(struct ci_hdrc *ci, int port1,
	enum ci_vbus_event event
)
{
	struct device	*dev;

	static const char var_usb_vbus[]  = "USB_VBUS=%s";
	static const char var_usb_port1[] = "USB_PORT1=%d";

	enum {
		USB_POWER_EVENT_VBUS = 0,
		USB_POWER_EVENT_PORT1,
		USB_POWER_EVENT_NULL,
		USB_POWER_EVENT_NUMS,
	};

	struct env_vars {
		/* note: sizeof(state) should be longer than
		 * sizeof(var_usb_vbus)
		 * + Max(for e in each enum vbus_event {
		 *         strlen(ci_vbus_event_to_str(e))
		 *      }
		 *   )
		 */
		char	state[sizeof(var_usb_vbus) + 32];
		char	port1[sizeof(var_usb_port1) + sizeof(int) * 3];
		char	*envp[USB_POWER_EVENT_NUMS];
	};

	struct	env_vars	*temp;
	int		result;
	int		ret;
	const char	*state;

	state = ci_vbus_event_to_str(event);
	if (!state) {
		/* Nothing to send. */
		ci_vbus_event_debug(ci->dev,
			"%s: No VBUS uevent. event=%s, port1=%d\n",
			__func__,
			ci_vbus_event_to_printable(event),
			port1
		);
		return 0;
	}
	ci_vbus_event_debug(ci->dev,
		"%s: Send VBUS uevent. event=%s, port1=%d\n",
		__func__,
		ci_vbus_event_to_printable(event),
		port1
	);

	result = 0;

	dev = ci->dev;
	temp = kzalloc(sizeof(*temp), GFP_KERNEL);
	if (temp == NULL) {
		dev_err(dev,
			"Not enough memory to build USB_VBUS uevent message.\n"
		);
		return -ENOMEM;
	}

	snprintf(&(temp->state[0]), sizeof(temp->state),
		var_usb_vbus, state);
	snprintf(&(temp->port1[0]), sizeof(temp->port1),
		var_usb_port1, port1);
	/* build envp table. */
	temp->envp[USB_POWER_EVENT_VBUS]  = &(temp->state[0]);
	temp->envp[USB_POWER_EVENT_PORT1] = &(temp->port1[0]);
	temp->envp[USB_POWER_EVENT_NULL]  = NULL;

	ret = kobject_uevent_env(&(dev->kobj), KOBJ_CHANGE,
		temp->envp);
	if (ret != 0) {
		dev_err(dev,
			"Not enough memory to send USB_VBUS. ret=%d\n",
			ret
		);
		result = ret;
	}

	kfree(temp);
	return result;
}

static int usb_vbus_event_send_stored(struct ci_hdrc *ci, int port1)
{	enum ci_vbus_event event;

	event = ci_vbus_event_update(ci, CI_VBUS_EVENT_SENT);
	return usb_vbus_event_send_direct(ci, port1, event);
}

#if (defined(CONFIG_USB_CHIPIDEA_HOST))

bool ci_hdrc_reg_vbus_is_enabled(struct ci_hdrc *ci)
{
	bool	result;

	down(&(ci->reg_vbus_sem));
	result = ci->reg_vbus_enabled;
	up(&(ci->reg_vbus_sem));

	return result;
}

/* Switch VBUS
 * Note: sysfs vbus_store calls me.
 */
int ci_hdrc_reg_vbus(struct ci_hdrc *ci, bool enable)
{
	struct device			*dev;
	struct ci_hdrc_platform_data	*platdata;
	struct regulator		*reg_vbus;

	int ret = 0;

	if (ci == NULL) {
		/* driver is not initialized. */
		pr_err("%s: No ci_hdrc context.\n",
			__func__
		);
		return -ENODEV;
	}

	dev = ci->dev;
	platdata = ci->platdata;

	if (!platdata) {
		/* Probed without platform data. */
		dev_err(dev, "No platform data.\n");
		return -ENODEV;
	}

	reg_vbus = platdata->reg_vbus;

	if (!reg_vbus) {
		/* No VBUS regulator, nothing to do without error. */
		return 0;
	}

	down(&(ci->reg_vbus_sem));
	/* NOTE: We only support one root hub port per host controller. */
	if (enable != ci->reg_vbus_enabled) {
		/* Turn VBUS switch. */
		if (enable) {
			/* Turn ON */
			ret = regulator_enable(reg_vbus);
		} else {
			/* Turn OFF */
			ret = regulator_disable(reg_vbus);
		}
	}

	if (ret != 0) {
		dev_err(dev,
			"VBUS regulator rejects request. enable=%d\n",
			(int)(enable)
		);
	} else {
		/* Update VBUS regulator state */
		ci->reg_vbus_enabled = enable;
	}
	up(&(ci->reg_vbus_sem));
	return ret;
}

/* Turn On/Off Root HUB VBUS power.
 * @ci chipidea driver context.
 * @rh_vbus root hub VBUS power, true: on, false: off
 *
 * @pre Lock vbus_power_sem except during probe, start, stop,
 *      and remove process.
 */
static int ci_hdrc_reg_vbus_root_power(struct ci_hdrc *ci, bool rh_vbus)
{	struct device	*dev;
	int		ret = 0;

	switch (ci->vbus_power) {
	case CI_VBUS_ON:
		/* Simply control VBUS by USB control request. */
		ret = ci_hdrc_reg_vbus(ci, rh_vbus);
		break;
	case CI_VBUS_OFF:
		ret = ci_hdrc_reg_vbus(ci, false);
		break;
	case CI_VBUS_AUTO:
		/* Simply control VBUS by USB control request. */
		ret = ci_hdrc_reg_vbus(ci, rh_vbus);
		break;
	default:
		dev = ci->dev;
		if (dev) {
			dev_err(dev,
				"Invalid value. vbus_power=%d\n",
				(int)(ci->vbus_power)
			);
		} else {
			pr_err("%s: Invalid value. vbus_power=%d\n",
				__func__,
				(int)(ci->vbus_power)
			);
		}
		break;
	}
	/* Update Root HUB VBUS power state. */
	ci->rh_vbus = rh_vbus;
	return ret;
}

#endif /* (defined(CONFIG_USB_CHIPIDEA_HOST)) */

static int ehci_ci_portpower_disable_vbus_auto(struct ci_hdrc *ci,
	int portnum, enum ci_role role)
{
	int	result = 1;

	switch (role) {
	case CI_ROLE_HOST:
		/* Host role, do only Root HUB operation. */
		/* Igonre error. */
		(void) ci_hdrc_reg_vbus_root_power(ci, false);
		break;
	case CI_ROLE_GADGET:
		/* OTGSC(ID pin) shows Gadget. */
		/* Igonre error. */
		(void) ci_hdrc_reg_vbus_root_power(ci, false);
		ci_vbus_event_update(ci, CI_VBUS_EVENT_ROLE_TO_GADGET);
		ci_vbus_event_debug(ci->dev,
			"portpower(clear).auto: Update VBUS uevent. update=TO_GADGET, event=%s\n",
			ci_vbus_event_read_printable(ci)
		);
		usb_vbus_event_send_stored(ci, portnum);
		break;
	default:
		/* Not Host, and not Gadget. */
		/* It may not happen, OTGSC always shows Host or Gadget. */
		break;
	}
	return result;
}

static int ehci_ci_portpower_disable_vbus_on(struct ci_hdrc *ci,
	int portnum, enum ci_role role)
{
	int	result = 1;

	switch (role) {
	case CI_ROLE_HOST:
		/* Host role, do only Root HUB operation. */
		/* When come here, one of the following situations is
		 * satisfied.
		 *  * USB core/hub fails enumeration and begin
		 *    retry loop with power off/on. It's a control request
		 *    to HUB port. see hub_port_connect().
		 *  * Userland application submits a USB control request
		 *    USB_REQ_CLEAR_FEATURE(USB_RT_PORT, USB_PORT_FEAT_POWER)
		 *    via libusb.
		 * Here, we should do simply off Root HUB power,
		 * and don't send any uevent to userland.
		 */
		/* Igonre error. */
		(void) ci_hdrc_reg_vbus_root_power(ci, false);
		break;
	case CI_ROLE_GADGET:
		/* OTGSC(ID pin) shows Gadget. */
		/* ON to OFF */
		ci->vbus_power = CI_VBUS_OFF;
		/* Igonre error. */
		(void) ci_hdrc_reg_vbus_root_power(ci, false);
		ci_vbus_event_update(ci, CI_VBUS_EVENT_ROLE_TO_GADGET);
		ci_vbus_event_debug(ci->dev,
			"portpower(clear).on: Update VBUS uevent. update=TO_GADGET, event=%s\n",
			ci_vbus_event_read_printable(ci)
		);
		usb_vbus_event_send_stored(ci, portnum);
		break;
	default:
		/* Not Host, and not Gadget. */
		/* It may not happen, OTGSC always shows Host or Gadget. */
		break;
	}
	return result;
}

static int ehci_ci_portpower(struct usb_hcd *hcd, int portnum, bool enable)
{
	struct ehci_hcd *ehci = hcd_to_ehci(hcd);
	struct device *dev = hcd->self.controller;
	struct ci_hdrc *ci = dev_get_drvdata(dev);
	int ret = 0;
	int port = HCS_N_PORTS(ehci->hcs_params);
	enum ci_role role;

	if (port > 1) {
		dev_warn(dev,
			"Not support multi-port regulator control\n");
		return 0;
	}

	/* To access CHIPIDEA IP block, feed power to block. */
	pm_runtime_get_sync(ci->dev);
	role = ci_otg_role(ci);
	if (ci->platdata->dr_mode == USB_DR_MODE_HOST) {
		/* Host only mode. */
		/* Override OTGSC_ID (ID pin) reads. */
		role = CI_ROLE_HOST;
	}

	ci_vbus_event_debug(ci->dev, "Root HUB port power. portnum=%d, enable=%d, role=%d\n",
		portnum,
		(int)enable,
		(int)role
	);

	down(&(ci->vbus_power_sem));
	if (enable) {
		/* Turn Port power into ON. */
		ci_hdrc_reg_vbus_root_power(ci, enable);
	} else {
		/* Turn Port power into OFF. */
		switch (ci->vbus_power) {
		case CI_VBUS_AUTO:
			/* AUTO to OFF */
			(void) ehci_ci_portpower_disable_vbus_auto(ci,
				portnum, role);
			break;
		case CI_VBUS_OFF:
			/* OFF to OFF */
			ci_vbus_event_debug(ci->dev,
				"portpower(clear).off: Off to off. event=%s\n",
				ci_vbus_event_read_printable(ci)
			);
			ci_hdrc_reg_vbus_root_power(ci, false);
			/* Note:
			 * When come here, following two conditions are true.
			 *  * written "off" to vbus_power node.
			 *  * USB EHCI driver offs VBUS by ehci_port_power().
			 * Writing "off" to vbus_power is handled in
			 * chipidea/core.c vbus_power_store(), this
			 * function does following process,
			 *  * Updates uevent as CI_VBUS_EVENT_SYSFS_OFF,
			 *    this uevent will be sent here.
			 *  * Remove Chipidea Host controller.
			 * Removing host controller does
			 * off Root Hub VBUS power in following call path.
			 *  ci_role_stop() -> host_stop() -> usb_remove_hcd()
			 *  -> usb_stop_hcd() -(indirect)-> ehci_stop()
			 *  -> ehci_turn_off_all_ports()
			 *  -> ehci_port_power(ehci, portnum, false)
			 *  -(indirect)-> ehci_ci_portpower()
			 */
			usb_vbus_event_send_stored(ci, portnum);
			break;
		case CI_VBUS_ON:
			/* ON to OFF */
			(void) ehci_ci_portpower_disable_vbus_on(ci, portnum,
				role);
			break;
		default:
			break;
		}
	}
	up(&(ci->vbus_power_sem));

	if (enable && (ci->platdata->phy_mode == USBPHY_INTERFACE_MODE_HSIC)) {
		/*
		 * Marvell 28nm HSIC PHY requires forcing the port to HS mode.
		 * As HSIC is always HS, this should be safe for others.
		 */
		hw_port_test_set(ci, 5);
		hw_port_test_set(ci, 0);
	}
	pm_runtime_put(ci->dev);
	return ret;
};

static int ehci_ci_reset(struct usb_hcd *hcd)
{
	struct device *dev = hcd->self.controller;
	struct ci_hdrc *ci = dev_get_drvdata(dev);
	struct ehci_hcd *ehci = hcd_to_ehci(hcd);
	int ret;

	ret = ehci_setup(hcd);
	if (ret)
		return ret;

	ehci->need_io_watchdog = 0;

	if (ci->platdata->notify_event) {
		ret = ci->platdata->notify_event(ci,
				CI_HDRC_CONTROLLER_RESET_EVENT);
		if (ret)
			return ret;
	}

	ci_platform_configure(ci);

	return ret;
}

static const struct ehci_driver_overrides ehci_ci_overrides = {
	.extra_priv_size = sizeof(struct ehci_ci_priv),
	.port_power	 = ehci_ci_portpower,
	.reset		 = ehci_ci_reset,
};

static irqreturn_t host_irq(struct ci_hdrc *ci)
{
	return usb_hcd_irq(ci->irq, ci->hcd);
}

static int host_start(struct ci_hdrc *ci)
{
	struct usb_hcd *hcd;
	struct ehci_hcd *ehci;
	struct ehci_ci_priv *priv;
	int ret;

	if (usb_disabled())
		return -ENODEV;

	/* Reset VBUS uevent. */
	ci_vbus_event_update(ci, CI_VBUS_EVENT_NOTHING);
	ci_vbus_event_debug(ci->dev,
		"%s: Clear VBUS uevent. event=%s\n",
		__func__,
		ci_vbus_event_read_printable(ci)
	);
	ci->rh_vbus = false;

	/* note: On VBUS power after host started. */
	clear_bit(CI_VBUS_OC_FLAGS_DETECTED, &(ci->vbus_oc_flags));

	hcd = __usb_create_hcd(&ci_ehci_hc_driver, ci->dev->parent,
			       ci->dev, dev_name(ci->dev), NULL);
	if (!hcd)
		return -ENOMEM;

	dev_set_drvdata(ci->dev, ci);
	hcd->rsrc_start = ci->hw_bank.phys;
	hcd->rsrc_len = ci->hw_bank.size;
	hcd->regs = ci->hw_bank.abs;
	hcd->has_tt = 1;

	hcd->power_budget = ci->platdata->power_budget;
	hcd->tpl_support = ci->platdata->tpl_support;
	if (ci->phy || ci->usb_phy) {
		hcd->skip_phy_initialization = 1;
		if (ci->usb_phy)
			hcd->usb_phy = ci->usb_phy;
	}

	ehci = hcd_to_ehci(hcd);
	ehci->caps = ci->hw_bank.cap;
	ehci->has_hostpc = ci->hw_bank.lpm;
	ehci->has_tdi_phy_lpm = ci->hw_bank.lpm;
	ehci->imx28_write_fix = ci->imx28_write_fix;

	priv = (struct ehci_ci_priv *)ehci->priv;
	priv->reg_vbus = NULL;

	if (ci->platdata->reg_vbus && !ci_otg_is_fsm_mode(ci)) {
		if (ci->platdata->flags & CI_HDRC_TURN_VBUS_EARLY_ON) {
			ret = ci_hdrc_reg_vbus_root_power(ci, true);
			if (ret) {
				dev_err(ci->dev,
				"Failed to enable vbus regulator, ret=%d\n",
									ret);
				goto put_hcd;
			}
		} else {
			priv->reg_vbus = ci->platdata->reg_vbus;
		}
	}

	if (ci->platdata->pins_host)
		pinctrl_select_state(ci->platdata->pctl,
				     ci->platdata->pins_host);

	ci->hcd = hcd;

	ret = usb_add_hcd(hcd, 0, 0);
	if (ret) {
		ci->hcd = NULL;
		goto disable_reg;
	} else {
		struct usb_otg *otg = &ci->otg;

		if (ci_otg_is_fsm_mode(ci)) {
			otg->host = &hcd->self;
			hcd->self.otg_port = 1;
		}

		if (ci->platdata->notify_event &&
			(ci->platdata->flags & CI_HDRC_IMX_IS_HSIC))
			ci->platdata->notify_event
				(ci, CI_HDRC_IMX_HSIC_ACTIVE_EVENT);
	}

	down(&(ci->vbus_oc_sem));
	ci->vbus_oc_jiffies = get_jiffies_64();
	ci->vbus_oc_continue = 0;
	ci_vbus_oc_irq_enable(ci);
	up(&(ci->vbus_oc_sem));

	return ret;

disable_reg:
	if (ci->platdata->reg_vbus && !ci_otg_is_fsm_mode(ci) &&
			(ci->platdata->flags & CI_HDRC_TURN_VBUS_EARLY_ON)) {
		/* note: On 8MMini EVK and Sony HA platform,
		 * This block will not run, the "flags" doesn't
		 * have CI_HDRC_TURN_VBUS_EARLY_ON.
		 */
		down(&(ci->vbus_oc_sem));
		ci_vbus_oc_irq_disable(ci, false);
		up(&(ci->vbus_oc_sem));

		down(&(ci->vbus_power_sem));
		switch (ci->vbus_power) {
		case CI_VBUS_ON:
			/* Requested force on, refrect off state. */
			ci->vbus_power = CI_VBUS_OFF;
			break;
		default:
			/* OFF or AUTO */
			/* Do nothing. */
			break;
		}
		ci_hdrc_reg_vbus(ci, false);
		ci->rh_vbus = false;
		up(&(ci->vbus_power_sem));
	}
put_hcd:
	usb_put_hcd(hcd);

	return ret;
}

static void host_stop(struct ci_hdrc *ci)
{
	struct usb_hcd *hcd = ci->hcd;

	ci_vbus_event_debug(ci->dev,
		"%s: Host stop. hcd=0x%lx, role=%d, event=%s\n",
		__func__,
		(ulong)(hcd),
		ci->role,
		ci_vbus_event_read_printable(ci)
	);

	down(&(ci->vbus_oc_sem));
	ci_vbus_oc_irq_disable(ci, false);
	up(&(ci->vbus_oc_sem));

	if (hcd) {
		if (ci->platdata->notify_event)
			ci->platdata->notify_event(ci,
				CI_HDRC_CONTROLLER_STOPPED_EVENT);
		usb_remove_hcd(hcd);
		ci->role = CI_ROLE_END;
		synchronize_irq(ci->irq);
		if (ci->platdata->reg_vbus && !ci_otg_is_fsm_mode(ci) &&
			(ci->platdata->flags & CI_HDRC_TURN_VBUS_EARLY_ON)) {
			/* NOTE for i.MX8MMini platform:
			 * On i.MX8MMini, codes in this block will not
			 * be executed. Because i.MX8MMini doesn't set
			 * CI_HDRC_TURN_VBUS_EARLY_ON in flags.
			 */
			struct usb_device *roothub = hcd->self.root_hub;

			if (roothub) {
				/* We have root hub context. */
				down(&(ci->vbus_power_sem));
				switch (ci->vbus_power) {
				case CI_VBUS_ON:
					/* Requested ON, turn off state. */
					ci->vbus_power = CI_VBUS_OFF;
					break;
				default:
					/* OFF or AUTO. */
					/* Do nothing. */
					break;
				}
				ci_hdrc_reg_vbus(ci, false);
				ci->rh_vbus = false;
				ci_vbus_event_update(ci,
					CI_VBUS_EVENT_ROLE_TO_GADGET);
				ci_vbus_event_debug(ci->dev,
					"%s: Update VBUS uevent. update=TO_GADGET, event=%s\n",
					__func__,
					ci_vbus_event_read_printable(ci)
				);
				usb_vbus_event_send_stored(ci, roothub->portnum);
				up(&(ci->vbus_power_sem));
			}
		} else {
			ci_vbus_event_debug(ci->dev,
				"%s: Skip VBUS control. reg_vbus=0x%lx, fsm_mode()=%d, EARLY_ON=0x%lx, event=%s\n",
				__func__,
				(long)(ci->platdata->reg_vbus),
				ci_otg_is_fsm_mode(ci),
				(ulong)(ci->platdata->flags & CI_HDRC_TURN_VBUS_EARLY_ON),
				ci_vbus_event_read_printable(ci)
			);
		}
		/* Release HCD context after contolling VBUS,
		 * HCD(EHC) context will be destroyed.
		 */
		usb_put_hcd(hcd);
	}
	ci->hcd = NULL;
	ci->otg.host = NULL;

	if (ci->platdata->pins_host && ci->platdata->pins_default)
		pinctrl_select_state(ci->platdata->pctl,
				     ci->platdata->pins_default);
}


void ci_hdrc_host_destroy(struct ci_hdrc *ci)
{
	if (ci->role == CI_ROLE_HOST && ci->hcd)
		host_stop(ci);
}

/* The below code is based on tegra ehci driver */
static int ci_ehci_hub_control(
	struct usb_hcd	*hcd,
	u16		typeReq,
	u16		wValue,
	u16		wIndex,
	char		*buf,
	u16		wLength
)
{
	struct ehci_hcd	*ehci = hcd_to_ehci(hcd);
	unsigned int	ports = HCS_N_PORTS(ehci->hcs_params);
	u32 __iomem	*status_reg;
	u32		temp, port_index, suspend_line_state;
	unsigned long	flags;
	int		retval = 0;
	bool		done = false;
	struct device *dev = hcd->self.controller;
	struct ci_hdrc *ci = dev_get_drvdata(dev);

	port_index = wIndex & 0xff;
	port_index -= (port_index > 0);
	status_reg = &ehci->regs->port_status[port_index];

	spin_lock_irqsave(&ehci->lock, flags);

	if (ci->platdata->hub_control) {
		retval = ci->platdata->hub_control(ci, typeReq, wValue, wIndex,
						   buf, wLength, &done, &flags);
		if (done)
			goto done;
	}

	if (typeReq == SetPortFeature && wValue == USB_PORT_FEAT_SUSPEND) {
		if (!wIndex || wIndex > ports) {
			retval = -EPIPE;
			goto done;
		}

		temp = ehci_readl(ehci, status_reg);
		if ((temp & PORT_PE) == 0 || (temp & PORT_RESET) != 0) {
			retval = -EPIPE;
			goto done;
		}

		temp &= ~(PORT_RWC_BITS | PORT_WKCONN_E);
		temp |= PORT_WKDISC_E | PORT_WKOC_E;
		ehci_writel(ehci, temp | PORT_SUSPEND, status_reg);

		/*
		 * If a transaction is in progress, there may be a delay in
		 * suspending the port. Poll until the port is suspended.
		 */
		if (ehci_handshake(ehci, status_reg, PORT_SUSPEND,
			PORT_SUSPEND, 5000))
			ehci_err(ehci, "timeout waiting for SUSPEND\n");

		if (ci->platdata->flags & CI_HDRC_HOST_SUSP_PHY_LPM) {
			if (PORT_SPEED_LOW(temp))
				suspend_line_state = PORTSC_LS_K;
			else
				suspend_line_state = PORTSC_LS_J;
			if (!ehci_handshake(ehci, status_reg, PORTSC_LS,
					   suspend_line_state, 5000))
				ci_hdrc_enter_lpm(ci, true);
		}


		if (ci->platdata->flags & CI_HDRC_IMX_IS_HSIC) {
			if (ci->platdata->notify_event)
				ci->platdata->notify_event(ci,
					CI_HDRC_IMX_HSIC_SUSPEND_EVENT);

			temp = ehci_readl(ehci, status_reg);
			temp &= ~(PORT_WKDISC_E | PORT_WKCONN_E);
			ehci_writel(ehci, temp, status_reg);
		}

		spin_unlock_irqrestore(&ehci->lock, flags);
		if (ehci_port_speed(ehci, temp) ==
				USB_PORT_STAT_HIGH_SPEED && hcd->usb_phy) {
			/* notify the USB PHY */
			usb_phy_notify_suspend(hcd->usb_phy, USB_SPEED_HIGH);
		}
		spin_lock_irqsave(&ehci->lock, flags);

		set_bit(port_index, &ehci->suspended_ports);
		goto done;
	}

	/*
	 * After resume has finished, it needs do some post resume
	 * operation for some SoCs.
	 */
	else if (typeReq == ClearPortFeature &&
		wValue == USB_PORT_FEAT_C_SUSPEND) {
		/* Make sure the resume has finished, it should be finished */
		if (ehci_handshake(ehci, status_reg, PORT_RESUME, 0, 25000))
			ehci_err(ehci, "timeout waiting for resume\n");

		temp = ehci_readl(ehci, status_reg);

		if (ehci_port_speed(ehci, temp) ==
				USB_PORT_STAT_HIGH_SPEED && hcd->usb_phy) {
			/* notify the USB PHY */
			usb_phy_notify_resume(hcd->usb_phy, USB_SPEED_HIGH);
		}
	}

	spin_unlock_irqrestore(&ehci->lock, flags);

	/* Handle the hub control events here */
	return ehci_hub_control(hcd, typeReq, wValue, wIndex, buf, wLength);
done:
	spin_unlock_irqrestore(&ehci->lock, flags);
	return retval;
}
static int ci_ehci_bus_suspend(struct usb_hcd *hcd)
{
	struct ehci_hcd *ehci = hcd_to_ehci(hcd);
	struct device *dev = hcd->self.controller;
	struct ci_hdrc *ci = dev_get_drvdata(dev);
	int port;
	u32 tmp;

	int ret = orig_bus_suspend(hcd);

	if (ret)
		return ret;

	port = HCS_N_PORTS(ehci->hcs_params);
	while (port--) {
		u32 __iomem *reg = &ehci->regs->port_status[port];
		u32 portsc = ehci_readl(ehci, reg);

		if (portsc & PORT_CONNECT) {
			/*
			 * For chipidea, the resume signal will be ended
			 * automatically, so for remote wakeup case, the
			 * usbcmd.rs may not be set before the resume has
			 * ended if other resume paths consumes too much
			 * time (~24ms), in that case, the SOF will not
			 * send out within 3ms after resume ends, then the
			 * high speed device will enter full speed mode.
			 */

			tmp = ehci_readl(ehci, &ehci->regs->command);
			tmp |= CMD_RUN;
			ehci_writel(ehci, tmp, &ehci->regs->command);
			/*
			 * It needs a short delay between set RS bit and PHCD.
			 */
			usleep_range(150, 200);
			/*
			 * If a transaction is in progress, there may be
			 * a delay in suspending the port. Poll until the
			 * port is suspended.
			 */
			if (test_bit(port, &ehci->bus_suspended) &&
					ehci_handshake(ehci, reg, PORT_SUSPEND,
							PORT_SUSPEND, 5000))
				ehci_err(ehci, "timeout waiting for SUSPEND\n");
			/*
			 * Need to clear WKCN and WKOC for imx HSIC,
			 * otherwise, there will be wakeup event.
			 */
			if (ci->platdata->flags & CI_HDRC_IMX_IS_HSIC) {
				tmp = ehci_readl(ehci, reg);
				tmp &= ~(PORT_WKDISC_E | PORT_WKCONN_E);
				ehci_writel(ehci, tmp, reg);
			}

			if (hcd->usb_phy && test_bit(port, &ehci->bus_suspended)
				&& (ehci_port_speed(ehci, portsc) ==
					USB_PORT_STAT_HIGH_SPEED))
				/*
				 * notify the USB PHY, it is for global
				 * suspend case.
				 */
				usb_phy_notify_suspend(hcd->usb_phy,
					USB_SPEED_HIGH);
			break;
		}
	}

	return 0;
}

static void ci_hdrc_free_dma_aligned_buffer(struct urb *urb)
{
	struct ci_hdrc_dma_aligned_buffer *temp;
	size_t length;

	if (!(urb->transfer_flags & URB_ALIGNED_TEMP_BUFFER))
		return;

	temp = container_of(urb->transfer_buffer,
			    struct ci_hdrc_dma_aligned_buffer, data);

	if (usb_urb_dir_in(urb)) {
		if (usb_pipeisoc(urb->pipe))
			length = urb->transfer_buffer_length;
		else
			length = urb->actual_length;

		memcpy(temp->old_xfer_buffer, temp->data, length);
	}
	urb->transfer_buffer = temp->old_xfer_buffer;
	kfree(temp->kmalloc_ptr);

	urb->transfer_flags &= ~URB_ALIGNED_TEMP_BUFFER;
}

static int ci_hdrc_alloc_dma_aligned_buffer(struct urb *urb, gfp_t mem_flags)
{
	struct ci_hdrc_dma_aligned_buffer *temp, *kmalloc_ptr;
	const unsigned int ci_hdrc_usb_dma_align = 32;
	size_t kmalloc_size;

	if (urb->num_sgs || urb->sg || urb->transfer_buffer_length == 0 ||
	    !((uintptr_t)urb->transfer_buffer & (ci_hdrc_usb_dma_align - 1)))
		return 0;

	/* Allocate a buffer with enough padding for alignment */
	kmalloc_size = urb->transfer_buffer_length +
		       sizeof(struct ci_hdrc_dma_aligned_buffer) +
		       ci_hdrc_usb_dma_align - 1;

	kmalloc_ptr = kmalloc(kmalloc_size, mem_flags);
	if (!kmalloc_ptr)
		return -ENOMEM;

	/* Position our struct dma_aligned_buffer such that data is aligned */
	temp = PTR_ALIGN(kmalloc_ptr + 1, ci_hdrc_usb_dma_align) - 1;
	temp->kmalloc_ptr = kmalloc_ptr;
	temp->old_xfer_buffer = urb->transfer_buffer;
	if (usb_urb_dir_out(urb))
		memcpy(temp->data, urb->transfer_buffer,
		       urb->transfer_buffer_length);
	urb->transfer_buffer = temp->data;

	urb->transfer_flags |= URB_ALIGNED_TEMP_BUFFER;

	return 0;
}

static int ci_hdrc_map_urb_for_dma(struct usb_hcd *hcd, struct urb *urb,
				   gfp_t mem_flags)
{
	int ret;

	ret = ci_hdrc_alloc_dma_aligned_buffer(urb, mem_flags);
	if (ret)
		return ret;

	ret = usb_hcd_map_urb_for_dma(hcd, urb, mem_flags);
	if (ret)
		ci_hdrc_free_dma_aligned_buffer(urb);

	return ret;
}

static void ci_hdrc_unmap_urb_for_dma(struct usb_hcd *hcd, struct urb *urb)
{
	usb_hcd_unmap_urb_for_dma(hcd, urb);
	ci_hdrc_free_dma_aligned_buffer(urb);
}

static void ci_hdrc_host_save_for_power_lost(struct ci_hdrc *ci)
{
	struct ehci_hcd *ehci;

	if (!ci->hcd)
		return;

	ehci = hcd_to_ehci(ci->hcd);
	/* save EHCI registers */
	ci->pm_usbmode = ehci_readl(ehci, &ehci->regs->usbmode);
	ci->pm_command = ehci_readl(ehci, &ehci->regs->command);
	ci->pm_command &= ~CMD_RUN;
	ci->pm_status  = ehci_readl(ehci, &ehci->regs->status);
	ci->pm_intr_enable  = ehci_readl(ehci, &ehci->regs->intr_enable);
	ci->pm_frame_index  = ehci_readl(ehci, &ehci->regs->frame_index);
	ci->pm_segment  = ehci_readl(ehci, &ehci->regs->segment);
	ci->pm_frame_list  = ehci_readl(ehci, &ehci->regs->frame_list);
	ci->pm_async_next  = ehci_readl(ehci, &ehci->regs->async_next);
	ci->pm_configured_flag  =
			ehci_readl(ehci, &ehci->regs->configured_flag);
	ci->pm_portsc = ehci_readl(ehci, &ehci->regs->port_status[0]);
}

static void ci_hdrc_host_restore_from_power_lost(struct ci_hdrc *ci)
{
	struct ehci_hcd *ehci;
	unsigned long   flags;
	u32 tmp;
	int step_ms;
	/*
	 * If the vbus is off during system suspend, most of devices will pull
	 * DP up within 200ms when they see vbus, set 1000ms for safety.
	 */
	int timeout_ms = 1000;

	if (!ci->hcd)
		return;

	hw_controller_reset(ci);

	ehci = hcd_to_ehci(ci->hcd);
	spin_lock_irqsave(&ehci->lock, flags);
	/* Restore EHCI registers */
	ehci_writel(ehci, ci->pm_usbmode, &ehci->regs->usbmode);
	ehci_writel(ehci, ci->pm_portsc, &ehci->regs->port_status[0]);
	ehci_writel(ehci, ci->pm_command, &ehci->regs->command);
	ehci_writel(ehci, ci->pm_intr_enable, &ehci->regs->intr_enable);
	ehci_writel(ehci, ci->pm_frame_index, &ehci->regs->frame_index);
	ehci_writel(ehci, ci->pm_segment, &ehci->regs->segment);
	ehci_writel(ehci, ci->pm_frame_list, &ehci->regs->frame_list);
	ehci_writel(ehci, ci->pm_async_next, &ehci->regs->async_next);
	ehci_writel(ehci, ci->pm_configured_flag,
					&ehci->regs->configured_flag);
	/* Restore the PHY's connect notifier setting */
	if (ci->pm_portsc & PORTSC_HSP)
		usb_phy_notify_connect(ci->usb_phy, USB_SPEED_HIGH);

	tmp = ehci_readl(ehci, &ehci->regs->command);
	tmp |= CMD_RUN;
	ehci_writel(ehci, tmp, &ehci->regs->command);
	spin_unlock_irqrestore(&ehci->lock, flags);

	if (!(ci->pm_portsc & PORTSC_CCS))
		return;

	for (step_ms = 0; step_ms < timeout_ms; step_ms += 25) {
		if (ehci_readl(ehci, &ehci->regs->port_status[0]) & PORTSC_CCS)
			break;
		msleep(25);
	}
}

static void ci_hdrc_host_suspend(struct ci_hdrc *ci)
{
	struct usb_hcd	  *hcd = ci->hcd;

	/* Notify EHCI driver OTG controller will enter suspend. */
	ehci_otg_suspend(hcd);

	ci_hdrc_host_save_for_power_lost(ci);

	if (!ci_otg_is_fsm_mode(ci)) {
		/* Not supported OTG FSM (Finite State Machine)
		 * Note: HA PF doesn't support OTG FSM mode,
		 * CONFIG_USB_OTG_FSM is not set.
		 */
		struct usb_device *roothub;

		roothub = hcd ? hcd->self.root_hub : NULL;
		if (roothub) {
			/* Root hub alive. */
			down(&(ci->vbus_power_sem));
			switch (ci->vbus_power) {
			case CI_VBUS_ON:
				ci->vbus_power = CI_VBUS_OFF;
				break;
			default:
				/* AUTO or OFF */
				break;
			}
			ci_hdrc_reg_vbus(ci, false);
			ci_vbus_event_update(ci,
				CI_VBUS_EVENT_SUSPEND);
			ci_vbus_event_debug(ci->dev,
				"%s: Update VBUS uevent. update=SUSPEND, event=%s\n",
				__func__,
				ci_vbus_event_read_printable(ci)
			);
			usb_vbus_event_send_stored(ci, roothub->portnum);
			up(&(ci->vbus_power_sem));
		}
	}
}

static void ci_hdrc_host_resume(struct ci_hdrc *ci, bool power_lost)
{
	struct usb_hcd	  *hcd = ci->hcd;

	if (power_lost)
		ci_hdrc_host_restore_from_power_lost(ci);


	/* Notify EHCI driver OTG controller exits suspend. */
	ehci_otg_resume(hcd);

	if (!ci_otg_is_fsm_mode(ci)) {
		/* Not supported OTG FSM (Finite State Machine)
		 * Note: HA PF doesn't support OTG FSM mode,
		 * CONFIG_USB_OTG_FSM is not set.
		 */
		struct usb_device *roothub;

		roothub = hcd ? hcd->self.root_hub : NULL;
		if (roothub) {
			/* Root hub alive. */
			down(&(ci->vbus_power_sem));
			switch (ci->vbus_power) {
			case CI_VBUS_OFF:
				/* We may off VBUS again. */
				ci_hdrc_reg_vbus(ci, false);
				break;
			case CI_VBUS_AUTO:
				/* Reset VBUS uevent. */
				ci_vbus_event_update(ci,
					CI_VBUS_EVENT_NOTHING);
				ci_vbus_event_debug(ci->dev,
					"%s: Clear VBUS uevent. event=%s\n",
					__func__,
					ci_vbus_event_read_printable(ci)
				);
				ci_hdrc_reg_vbus(ci, ci->rh_vbus);
				break;
			default:
				/* NOTE: may not be ON,
				 * turn into OFF at suspend.
				 */
				break;
			}
			up(&(ci->vbus_power_sem));
		}
	}
}

static int ci_ehci_bus_resume(struct usb_hcd *hcd)
{
	struct ehci_hcd *ehci = hcd_to_ehci(hcd);
	int port;

	int ret = orig_bus_resume(hcd);

	if (ret)
		return ret;

	port = HCS_N_PORTS(ehci->hcs_params);
	while (port--) {
		u32 __iomem *reg = &ehci->regs->port_status[port];
		u32 portsc = ehci_readl(ehci, reg);
		/*
		 * Notify PHY after resume signal has finished, it is
		 * for global suspend case.
		 */
		if (hcd->usb_phy
			&& test_bit(port, &ehci->bus_suspended)
			&& (portsc & PORT_CONNECT)
			&& (ehci_port_speed(ehci, portsc) ==
				USB_PORT_STAT_HIGH_SPEED))
			/* notify the USB PHY */
			usb_phy_notify_resume(hcd->usb_phy, USB_SPEED_HIGH);
	}

	return 0;
}

int ci_hdrc_host_init(struct ci_hdrc *ci)
{
	struct ci_role_driver *rdrv;

	if (!hw_read(ci, CAP_DCCPARAMS, DCCPARAMS_HC))
		return -ENXIO;

	rdrv = devm_kzalloc(ci->dev, sizeof(struct ci_role_driver), GFP_KERNEL);
	if (!rdrv)
		return -ENOMEM;

	rdrv->start	= host_start;
	rdrv->stop	= host_stop;
	rdrv->suspend	= ci_hdrc_host_suspend;
	rdrv->resume	= ci_hdrc_host_resume;
	rdrv->irq	= host_irq;
	rdrv->name	= "host";
	ci->roles[CI_ROLE_HOST] = rdrv;

	if (ci->platdata->flags & CI_HDRC_REQUIRES_ALIGNED_DMA) {
		ci_ehci_hc_driver.map_urb_for_dma = ci_hdrc_map_urb_for_dma;
		ci_ehci_hc_driver.unmap_urb_for_dma = ci_hdrc_unmap_urb_for_dma;
	}

	return 0;
}

void ci_hdrc_host_driver_init(void)
{
	ehci_init_driver(&ci_ehci_hc_driver, &ehci_ci_overrides);
	orig_bus_suspend = ci_ehci_hc_driver.bus_suspend;
	orig_bus_resume = ci_ehci_hc_driver.bus_resume;
	ci_ehci_hc_driver.bus_resume = ci_ehci_bus_resume;
	ci_ehci_hc_driver.bus_suspend = ci_ehci_bus_suspend;
	ci_ehci_hc_driver.hub_control = ci_ehci_hub_control;
}
