// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright 2020, 2021, 2022 Sony Corporation
 */

#include <linux/module.h>
#include <linux/platform_device.h>
#include <sound/soc.h>
#include <sound/soc-dai.h>
#include <linux/delay.h>

#include <sound/tas5825.h>

struct tas5825_priv {
	unsigned int mute_busy;
};

static BLOCKING_NOTIFIER_HEAD(tas5825_notify_list);

int register_tas5825_notifier(struct notifier_block *nb)
{
	return blocking_notifier_chain_register(&tas5825_notify_list, nb);
}
EXPORT_SYMBOL_GPL(register_tas5825_notifier);

int unregister_tas5825_notifier(struct notifier_block *nb)
{
	return blocking_notifier_chain_unregister(&tas5825_notify_list, nb);
}
EXPORT_SYMBOL_GPL(unregister_tas5825_notifier);

static int tas5825_trigger(struct snd_pcm_substream *substream,
			      int cmd, struct snd_soc_dai *dai)
{
	if (cmd == SNDRV_PCM_TRIGGER_STOP)
		blocking_notifier_call_chain(&tas5825_notify_list, 1, NULL);

	return 0;
}

static int tas5825_digital_mute(struct snd_soc_dai *dai, int mute, int stream)
{
	struct snd_soc_component *component = dai->component;
	struct tas5825_priv *priv = snd_soc_component_get_drvdata(component);

	if (mute == 1)
		usleep_range(priv->mute_busy, priv->mute_busy + 100);

	return 0;
}

static const struct snd_soc_dai_ops tas5825_dai_ops = {
	.trigger = tas5825_trigger,
	.mute_stream = tas5825_digital_mute,
};

static struct snd_soc_dai_driver tas5825_dai_driver = {
	.name = "tas5825-dai",
	.playback = {
		.stream_name = "Playback",
		.channels_min = 1,
		.channels_max = 16,
		.rates = SNDRV_PCM_RATE_8000_192000,
		.formats = SNDRV_PCM_FMTBIT_S16_LE |
			   SNDRV_PCM_FMTBIT_S24_LE |
			   SNDRV_PCM_FMTBIT_S32_LE,
	},
	.ops = &tas5825_dai_ops,
};

static const struct snd_soc_component_driver tas5825_driver = {
};

static int tas5825_probe(struct platform_device *pdev)
{
	struct tas5825_priv *priv;
	struct device *dev = &pdev->dev;
	unsigned int val;

	priv = devm_kzalloc(dev, sizeof(struct tas5825_priv), GFP_KERNEL);
	if (priv == NULL)
		return -ENOMEM;
	dev_set_drvdata(dev, priv);

	/* Default values */
	priv->mute_busy = 7000;

	if (!of_property_read_u32(dev->of_node, "mute-busy", &val))
		priv->mute_busy = val;

	return snd_soc_register_component(&pdev->dev, &tas5825_driver,
					  &tas5825_dai_driver, 1);
}

static int tas5825_remove(struct platform_device *pdev)
{
	snd_soc_unregister_component(&pdev->dev);

	return 0;
}

#ifdef CONFIG_OF
static const struct of_device_id tas5825_device_id[] = {
	{ .compatible = "sony,tas5825" },
	{}
};
MODULE_DEVICE_TABLE(of, tas5825_device_id);
#endif

static struct platform_driver tas5825_platform_driver = {
	.driver = {
		.name = "tas5825",
		.owner = THIS_MODULE,
#if defined(CONFIG_OF)
		.of_match_table = of_match_ptr(tas5825_device_id),
#endif
	},
	.probe	= tas5825_probe,
	.remove = tas5825_remove,
};
module_platform_driver(tas5825_platform_driver);

/* Module information */
MODULE_DESCRIPTION("tas5825 codec driver");
MODULE_AUTHOR("Sony Corporation");
MODULE_LICENSE("GPL");
MODULE_VERSION("v1.00");
