// SPDX-License-Identifier: GPL-2.0+
// Copyright 2017-2021 NXP

#include <linux/module.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/gpio/consumer.h>
#include <linux/of_device.h>
#include <linux/i2c.h>
#include <linux/of_gpio.h>
#include <linux/clk.h>
#include <sound/soc.h>
#include <sound/pcm_params.h>
#include <sound/pcm.h>
#include <sound/soc-dapm.h>
#include <sound/simple_card_utils.h>

/*
 * Mapping LRCK fs and frame width, table 3 & 4 in datasheet
 * @rmin: min rate
 * @rmax: max rate
 * @wmin: min frame ratio
 * @wmax: max frame ratio
 */
struct imx_akcodec_fs_mul {
	unsigned int rmin;
	unsigned int rmax;
	unsigned int wmin;
	unsigned int wmax;
};

/*
 * Mapping TDM mode and frame width
 */
struct imx_akcodec_tdm_fs_mul {
	unsigned int min;
	unsigned int max;
	unsigned int mul;
};

/*
 * struct imx_card_plat_data - specific info for codecs
 *
 * @fs_mul: ratio of mclk/fs for normal mode
 * @tdm_fs_mul: ratio of mclk/fs for tdm mode
 * @support_rates: supported sample rate
 * @support_tdm_rates: supported sample rate for tdm mode
 * @support_channels: supported channels
 * @support_tdm_channels: supported channels for tdm mode
 * @num_fs_mul: ARRAY_SIZE of fs_mul
 * @num_tdm_fs_mul: ARRAY_SIZE of tdm_fs_mul
 * @num_rates: ARRAY_SIZE of support_rates
 * @num_tdm_rates: ARRAY_SIZE of support_tdm_rates
 * @num_channels: ARRAY_SIZE of support_channels
 * @num_tdm_channels: ARRAY_SIZE of support_tdm_channels
 * @type: codec type
 */
struct imx_card_plat_data {
	struct imx_akcodec_fs_mul  *fs_mul;
	struct imx_akcodec_tdm_fs_mul  *tdm_fs_mul;
	const u32 *support_rates;
	const u32 *support_tdm_rates;
	const u32 *support_channels;
	const u32 *support_tdm_channels;
	unsigned int num_fs_mul;
	unsigned int num_tdm_fs_mul;
	unsigned int num_rates;
	unsigned int num_tdm_rates;
	unsigned int num_channels;
	unsigned int num_tdm_channels;
	unsigned int num_codecs;
};

/*
 * struct dai_link_data - specific info for dai link
 *
 * @slots: slot number
 * @slot_width: slot width value
 * @cpu_sysclk_id: sysclk id for cpu dai
 */
struct dai_link_data {
	unsigned int slots;
	unsigned int slot_width;
	unsigned int cpu_sysclk_id;
};

/*
 * struct imx_card_data - platform device data
 *
 * @plat_data: pointer of imx_card_plat_data
 * @dapm_routes: pointer of dapm_routes
 * @link_data: private data for dai link
 * @card: card instance
 * @num_dapm_routes: number of dapm_routes
 * @asrc_rate: asrc rates
 * @asrc_format: asrc format
 */
struct imx_card_data {
	struct imx_card_plat_data *plat_data;
	struct snd_soc_dapm_route *dapm_routes;
	struct dai_link_data *link_data;
	struct snd_soc_card card;
	int num_dapm_routes;
	u32 asrc_rate;
	u32 asrc_format;
};


static bool format_is_tdm(struct dai_link_data *link_data)
{
	if (link_data->slots > 2)
		return true;
	else
		return false;
}

static int imx_aif_hw_params(struct snd_pcm_substream *substream,
			     struct snd_pcm_hw_params *params)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_dai *cpu_dai = asoc_rtd_to_cpu(rtd, 0);
	struct snd_soc_card *card = rtd->card;
	struct imx_card_data *data = snd_soc_card_get_drvdata(card);
	struct dai_link_data *link_data = &data->link_data[rtd->num];
	struct device *dev = card->dev;
	struct snd_soc_dai *codec_dai;
	unsigned long mclk_freq;
	unsigned int fmt = rtd->dai_link->dai_fmt;
	unsigned int slots, slot_width;
	int ret, i;

	slots = link_data->slots;
	slot_width = link_data->slot_width;

	if (!format_is_tdm(link_data)) {
		slots = 2;
		slot_width = params_physical_width(params);
		fmt = (rtd->dai_link->dai_fmt & ~SND_SOC_DAIFMT_FORMAT_MASK) |
			SND_SOC_DAIFMT_I2S;
	}

	ret = snd_soc_dai_set_fmt(cpu_dai, fmt);
	if (ret && ret != -ENOTSUPP) {
		dev_err(dev, "failed to set cpu dai fmt: %d\n", ret);
		return ret;
	}
	ret = snd_soc_dai_set_tdm_slot(cpu_dai,
				       BIT(slots) - 1,
				       BIT(slots) - 1,
				       slots, slot_width);
	if (ret && ret != -ENOTSUPP) {
		dev_err(dev, "failed to set cpu dai tdm slot: %d\n", ret);
		return ret;
	}

	for_each_rtd_codec_dais(rtd, i, codec_dai) {
		ret = snd_soc_dai_set_fmt(codec_dai, fmt);
		if (ret && ret != -ENOTSUPP) {
			dev_err(dev, "failed to set codec dai[%d] fmt: %d\n", i, ret);
			return ret;
		}

		ret = snd_soc_dai_set_tdm_slot(codec_dai,
					       BIT(slots) - 1,
					       BIT(slots) - 1,
					       slots, slot_width);
		if (ret && ret != -ENOTSUPP) {
			dev_err(dev, "failed to set codec dai[%d] tdm slot: %d\n", i, ret);
			return ret;
		}
	}

	/* Set MCLK freq */
	mclk_freq = params_rate(params) * slots * slot_width;

	ret = snd_soc_dai_set_sysclk(cpu_dai, link_data->cpu_sysclk_id, mclk_freq,
				     SND_SOC_CLOCK_OUT);
	if (ret && ret != -ENOTSUPP) {
		dev_err(dev, "failed to set cpui dai mclk1 rate (%lu): %d\n", mclk_freq, ret);
		return ret;
	}

	return 0;
}

static int imx_aif_startup(struct snd_pcm_substream *substream)
{
	struct snd_pcm_runtime *runtime = substream->runtime;
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_card *card = rtd->card;
	struct imx_card_data *data = snd_soc_card_get_drvdata(card);
	struct dai_link_data *link_data = &data->link_data[rtd->num];
	static struct snd_pcm_hw_constraint_list constraint_rates;
	static struct snd_pcm_hw_constraint_list constraint_channels;
	int ret = 0;

	if (format_is_tdm(link_data)) {
		constraint_channels.list = data->plat_data->support_tdm_channels;
		constraint_channels.count = data->plat_data->num_tdm_channels;
		constraint_rates.list = data->plat_data->support_tdm_rates;
		constraint_rates.count = data->plat_data->num_tdm_rates;
	} else {
		constraint_channels.list = data->plat_data->support_channels;
		constraint_channels.count = data->plat_data->num_channels;
		constraint_rates.list = data->plat_data->support_rates;
		constraint_rates.count = data->plat_data->num_rates;
	}

	if (constraint_channels.count) {
		ret = snd_pcm_hw_constraint_list(runtime, 0,
						 SNDRV_PCM_HW_PARAM_CHANNELS,
						 &constraint_channels);
		if (ret)
			return ret;
	}

	if (constraint_rates.count) {
		ret = snd_pcm_hw_constraint_list(runtime, 0,
						 SNDRV_PCM_HW_PARAM_RATE,
						 &constraint_rates);
		if (ret)
			return ret;
	}

	return ret;
}

static struct snd_soc_ops imx_aif_ops = {
	.hw_params = imx_aif_hw_params,
	.startup = imx_aif_startup,
};

static int imx_card_parse_of(struct imx_card_data *data)
{
	struct imx_card_plat_data *plat_data = data->plat_data;
	struct snd_soc_card *card = &data->card;
	struct snd_soc_dai_link_component *dlc;
	struct device_node *platform = NULL;
	struct device_node *codec = NULL;
	struct device_node *cpu = NULL;
	struct device_node *np;
	struct device *dev = card->dev;
	struct snd_soc_dai_link *link;
	struct dai_link_data *link_data;
	struct of_phandle_args args;
	int ret, num_links;

	ret = snd_soc_of_parse_card_name(card, "model");
	if (ret) {
		dev_err(dev, "Error parsing card name: %d\n", ret);
		return ret;
	}

	/* DAPM routes */
	if (of_property_read_bool(dev->of_node, "audio-routing")) {
		ret = snd_soc_of_parse_audio_routing(card, "audio-routing");
		if (ret)
			return ret;
	}

	/* Populate links */
	num_links = of_get_child_count(dev->of_node);

	/* Allocate the DAI link array */
	card->dai_link = devm_kcalloc(dev, num_links, sizeof(*link), GFP_KERNEL);
	if (!card->dai_link)
		return -ENOMEM;

	data->link_data = devm_kcalloc(dev, num_links, sizeof(*link), GFP_KERNEL);
	if (!data->link_data)
		return -ENOMEM;

	card->num_links = num_links;
	link = card->dai_link;
	link_data = data->link_data;

	for_each_child_of_node(dev->of_node, np) {
		dlc = devm_kzalloc(dev, 2 * sizeof(*dlc), GFP_KERNEL);
		if (!dlc) {
			ret = -ENOMEM;
			goto err_put_np;
		}

		link->cpus	= &dlc[0];
		link->platforms	= &dlc[1];

		link->num_cpus		= 1;
		link->num_platforms	= 1;

		ret = of_property_read_string(np, "link-name", &link->name);
		if (ret) {
			dev_err(card->dev, "error getting codec dai_link name\n");
			goto err_put_np;
		}

		cpu = of_get_child_by_name(np, "cpu");
		if (!cpu) {
			dev_err(dev, "%s: Can't find cpu DT node\n", link->name);
			ret = -EINVAL;
			goto err;
		}

		ret = of_parse_phandle_with_args(cpu, "sound-dai",
						 "#sound-dai-cells", 0, &args);
		if (ret) {
			dev_err(card->dev, "%s: error getting cpu phandle\n", link->name);
			goto err;
		}

		link->cpus->of_node = args.np;
		link->platforms->of_node = link->cpus->of_node;
		link->id = args.args[0];

		ret = snd_soc_of_get_dai_name(cpu, &link->cpus->dai_name);
		if (ret) {
			if (ret != -EPROBE_DEFER)
				dev_err(card->dev, "%s: error getting cpu dai name: %d\n",
					link->name, ret);
			goto err;
		}

		codec = of_get_child_by_name(np, "codec");
		if (codec) {
			ret = snd_soc_of_get_dai_link_codecs(dev, codec, link);
			if (ret < 0) {
				if (ret != -EPROBE_DEFER)
					dev_err(dev, "%s: codec dai not found: %d\n",
						link->name, ret);
				goto err;
			}

			plat_data->num_codecs = link->num_codecs;

		} else {
			dlc = devm_kzalloc(dev, sizeof(*dlc), GFP_KERNEL);
			if (!dlc) {
				ret = -ENOMEM;
				goto err;
			}

			link->codecs	 = dlc;
			link->num_codecs = 1;

			link->codecs->dai_name = "snd-soc-dummy-dai";
			link->codecs->name = "snd-soc-dummy";
		}

		link->ops = &imx_aif_ops;

		if (link->no_pcm || link->dynamic)
			snd_soc_dai_link_set_capabilities(link);

		/* Get dai fmt */
		ret = asoc_simple_parse_daifmt(dev, np, codec,
					       NULL, &link->dai_fmt);
		if (ret)
			link->dai_fmt = SND_SOC_DAIFMT_NB_NF |
					SND_SOC_DAIFMT_CBS_CFS |
					SND_SOC_DAIFMT_I2S;

		/* Get tdm slot */
		snd_soc_of_parse_tdm_slot(np, NULL, NULL,
					  &link_data->slots,
					  &link_data->slot_width);
		/* default value */
		if (!link_data->slots)
			link_data->slots = 2;

		if (!link_data->slot_width)
			link_data->slot_width = 32;

		link->ignore_pmdown_time = 1;
		link->stream_name = link->name;
		link++;
		link_data++;

		of_node_put(cpu);
		of_node_put(codec);
		of_node_put(platform);
	}

	return 0;
err:
	of_node_put(cpu);
	of_node_put(codec);
	of_node_put(platform);
err_put_np:
	of_node_put(np);
	return ret;
}

static int imx_card_probe(struct platform_device *pdev)
{
	struct snd_soc_dai_link *link_be = NULL, *link;
	struct imx_card_plat_data *plat_data;
	struct imx_card_data *data;
	int ret, i;

	data = devm_kzalloc(&pdev->dev, sizeof(*data), GFP_KERNEL);
	if (!data)
		return -ENOMEM;

	plat_data = devm_kzalloc(&pdev->dev, sizeof(*plat_data), GFP_KERNEL);
	if (!plat_data)
		return -ENOMEM;

	data->plat_data = plat_data;
	data->card.dev = &pdev->dev;

	dev_set_drvdata(&pdev->dev, &data->card);
	snd_soc_card_set_drvdata(&data->card, data);
	ret = imx_card_parse_of(data);
	if (ret)
		return ret;

	data->num_dapm_routes = plat_data->num_codecs + 1;
	data->dapm_routes = devm_kcalloc(&pdev->dev, data->num_dapm_routes,
					 sizeof(struct snd_soc_dapm_route),
					 GFP_KERNEL);
	if (!data->dapm_routes)
		return -ENOMEM;

	/* with asrc as front end */
	if (data->card.num_links == 3) {
		data->card.dapm_routes = data->dapm_routes;
		data->card.num_dapm_routes = data->num_dapm_routes;
		for_each_card_prelinks(&data->card, i, link) {
			if (link->no_pcm == 1)
				link_be = link;
		}
		for_each_card_prelinks(&data->card, i, link) {
			if (link->dynamic == 1 && link_be) {
				link->dpcm_playback = link_be->dpcm_playback;
				link->dpcm_capture = link_be->dpcm_capture;
			}
		}
	}

	ret = devm_snd_soc_register_card(&pdev->dev, &data->card);
	if (ret) {
		if (ret != -EPROBE_DEFER)
			dev_err(&pdev->dev, "snd_soc_register_card failed (%d)\n", ret);
		return ret;
	}

	return 0;
}

static const struct of_device_id imx_card_dt_ids[] = {
	{ .compatible = "sony,imx-audio-card", },
	{ },
};
MODULE_DEVICE_TABLE(of, imx_card_dt_ids);

static struct platform_driver imx_card_driver = {
	.driver = {
		.name = "imx-sony-card",
		.pm = &snd_soc_pm_ops,
		.of_match_table = imx_card_dt_ids,
	},
	.probe = imx_card_probe,
};
module_platform_driver(imx_card_driver);

MODULE_DESCRIPTION("Freescale i.MX ASoC Machine Driver");
MODULE_LICENSE("GPL v2");
MODULE_ALIAS("platform:imx-sony-card");
