/*
 * Copyright 2017 NXP
 * Copyright 2019 Sony Home Entertainment & Sound Products Inc.
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

#include <linux/module.h>
#include <sound/soc.h>
#include <sound/pcm_params.h>
#include <linux/of_device.h>
#include <linux/of_gpio.h>

#include "fsl_sai.h"

static const u32 mt8570_rates[] = {
	8000, 11025, 16000, 22050,
	32000, 44100, 48000, 88200,
	96000, 176400, 192000, 352800,
	384000,
};

static const u32 mt8570_channels[] = {
	1, 2, 4, 6, 8,
};



static const u32 mt8570_channels_tdm[] = {
        1, 2, 4, 6, 8, 10, 12, 14, 16,
};

#define FIXED_MCLK     24576000        /* 24MHz. This also used MCLK for opt-in */

struct sony_va_in_data {
	bool tdm_mode;
	unsigned int slots;
	unsigned int slot_width;
};

static int sony_mt8570_hw_params(struct snd_pcm_substream *substream,
		 struct snd_pcm_hw_params *params)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_dai *cpu_dai = asoc_rtd_to_cpu(rtd, 0);
	struct snd_soc_card *card = rtd->card;
	struct device *dev = card->dev;
	unsigned int channels = params_channels(params);
	unsigned int fmt = SND_SOC_DAIFMT_NB_NF | SND_SOC_DAIFMT_CBS_CFS;
	unsigned long mclk_freq;
	int ret;
	struct sony_va_in_data *data = snd_soc_card_get_drvdata(card);

	if (data->tdm_mode) {
		data->slots = 8;
		data->slot_width = 32;
		fmt |= SND_SOC_DAIFMT_DSP_A;
	} else {
		data->slots = 2;
		data->slot_width = params_physical_width(params);
		fmt |= SND_SOC_DAIFMT_I2S;
	}

	ret = snd_soc_dai_set_fmt(cpu_dai, fmt);
	if (ret) {
		dev_err(dev, "failed to set cpu dai fmt: %d\n", ret);
		return ret;
	}

	ret = snd_soc_dai_set_tdm_slot(cpu_dai,
			BIT(channels) - 1, BIT(channels) - 1,
			data->slots, data->slot_width);

	if (ret) {
		dev_err(dev, "failed to set cpu dai tdm slot: %d\n", ret);
		return ret;
	}

	mclk_freq = FIXED_MCLK;
	ret = snd_soc_dai_set_sysclk(cpu_dai, FSL_SAI_CLK_MAST1, mclk_freq,
				     SND_SOC_CLOCK_OUT);
	if (ret < 0)
		dev_err(dev, "failed to set cpui dai mclk1 rate (%lu): %d\n",
			mclk_freq, ret);

	return ret;
}


static int sony_mt8570_startup(struct snd_pcm_substream *substream)
{
	struct snd_pcm_runtime *runtime = substream->runtime;
	static struct snd_pcm_hw_constraint_list constraint_rates;
	static struct snd_pcm_hw_constraint_list constraint_channels;
	int ret;
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_card *card = rtd->card;
	struct sony_va_in_data *data = snd_soc_card_get_drvdata(card);

	if (data->tdm_mode) {
		constraint_channels.list = mt8570_channels_tdm;
		constraint_channels.count = ARRAY_SIZE(mt8570_channels_tdm);
	} else {
		constraint_channels.list = mt8570_channels;
		constraint_channels.count = ARRAY_SIZE(mt8570_channels);
	}

	constraint_rates.list = mt8570_rates;
	constraint_rates.count = ARRAY_SIZE(mt8570_rates);

	ret = snd_pcm_hw_constraint_list(runtime, 0, SNDRV_PCM_HW_PARAM_CHANNELS,
					 &constraint_channels);
	if (ret)
		return ret;

	ret = snd_pcm_hw_constraint_list(runtime, 0, SNDRV_PCM_HW_PARAM_RATE,
					 &constraint_rates);
	if (ret)
		return ret;

	return 0;
}

static void sony_mt8570_shutdown(struct snd_pcm_substream *substream)
{
	return;
}


static struct snd_soc_ops sony_mt8570_ops = {
	.hw_params = sony_mt8570_hw_params,
	.startup = sony_mt8570_startup,
	.shutdown = sony_mt8570_shutdown,
};

static struct snd_soc_dai_link sony_mt8570_dai = {
	.name = "mt8570--in",
	.stream_name = "mt8570-in",
	.ops = &sony_mt8570_ops,
	.capture_only = 1,
};

static struct snd_soc_card sony_va_in_card = {
	.dai_link = &sony_mt8570_dai,
	.num_links = 1,
};

static int sony_va_mic_probe(struct platform_device *pdev)
{
	int ret;
	struct device_node *cpu_np;
	struct platform_device *cpu_pdev;
	struct sony_va_in_data *priv;
	struct snd_soc_dai_link_component *comp;

	pr_debug("%s()\n", __func__);

	priv = devm_kzalloc(&pdev->dev, sizeof(*priv), GFP_KERNEL);
	if (!priv)
		return -ENOMEM;

	comp = devm_kzalloc(&pdev->dev, 3 * sizeof(*comp), GFP_KERNEL);
	if (!comp)
		return -ENOMEM;

	cpu_np = of_parse_phandle(pdev->dev.of_node, "audio-cpu", 0);
	if (!cpu_np) {
		dev_err(&pdev->dev, "audio dai phandle missing or invalid\n");
		ret = -EINVAL;
		goto fail;
	}

	cpu_pdev = of_find_device_by_node(cpu_np);
	if (!cpu_pdev) {
		dev_err(&pdev->dev, "failed to find SAI platform device\n");
		ret = -EINVAL;
		goto fail;
	}

	sony_mt8570_dai.cpus = &comp[0];
	sony_mt8570_dai.num_cpus = 1;
	sony_mt8570_dai.codecs = &comp[1];
	sony_mt8570_dai.num_codecs = 1;
	sony_mt8570_dai.platforms = &comp[2];
	sony_mt8570_dai.num_platforms = 1;

	sony_mt8570_dai.cpus->dai_name = dev_name(&cpu_pdev->dev);
	sony_mt8570_dai.cpus->of_node = cpu_np;
	sony_mt8570_dai.platforms->of_node = cpu_np;

	sony_mt8570_dai.codecs->dai_name = "snd-soc-dummy-dai";
	sony_mt8570_dai.codecs->name = "snd-soc-dummy";

	sony_va_in_card.dev = &pdev->dev;
	sony_va_in_card.owner = THIS_MODULE;
	ret = snd_soc_of_parse_card_name(&sony_va_in_card, "model");
	if (ret)
		goto fail;

	if (of_find_property(pdev->dev.of_node, "fsl,tdm", NULL))
		priv->tdm_mode = true;

	snd_soc_card_set_drvdata(&sony_va_in_card, priv);

	ret = snd_soc_register_card(&sony_va_in_card);
	if (ret) {
		dev_err(&pdev->dev, "failed to find SAI platform device\n");
		goto fail;
	}

	return 0;

fail:
	if (cpu_np)
		of_node_put(cpu_np);

	return ret;
}

static int sony_va_mic_remove(struct platform_device *pdev)
{
	struct snd_soc_card *card = platform_get_drvdata(pdev);

	return snd_soc_unregister_card(card);
}

#ifdef CONFIG_OF
static const struct of_device_id sony_va_in_dt_match[] = {
	{.compatible = "fsl,sony-va-in",},
	{}
};
#endif

static struct platform_driver sony_va_mic = {
	.driver = {
		.name = "sony-va-mic",
		.owner = THIS_MODULE,
                #ifdef CONFIG_OF
		.of_match_table = sony_va_in_dt_match,
                #endif
	},
	.probe = sony_va_mic_probe,
	.remove = sony_va_mic_remove
};

module_platform_driver(sony_va_mic);

/* Module information */
MODULE_DESCRIPTION("sony va mic driver");
MODULE_LICENSE("GPL");
MODULE_ALIAS("sony va mic card");
