/*
 * mt8570-adsp-controls.c  --  Mediatek 8570 adsp controls
 *
 * Copyright (c) 2019 MediaTek Inc.
 * Author: Hidalgo Huang <hidalgo.huang@mediatek.com>
 * Copyright 2022 Sony Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <soc/mediatek/mt8570-adsp-controls.h>
#include <soc/mediatek/mt8570-adsp-common.h>
#include <linux/string.h>
#include <sound/soc.h>


static int mt8570_adsp_keyword_detect_get(struct snd_kcontrol *kcontrol,
	struct snd_ctl_elem_value *ucontrol)
{
#ifdef SARD_4_19_35	
	struct snd_soc_component *plat = snd_soc_kcontrol_component(kcontrol);
	struct mt8570_adsp_pcm_priv *priv =
		snd_soc_component_get_drvdata(plat);
#else
	struct snd_soc_platform *plat = snd_soc_kcontrol_platform(kcontrol);
	struct mt8570_adsp_pcm_priv *priv =
		snd_soc_platform_get_drvdata(plat);
#endif
		
	struct mt8570_adsp_control_data *data = &priv->ctrl_data;

	ucontrol->value.integer.value[0] = data->wakeword_detect_en;

	return 0;
}

static int mt8570_adsp_keyword_detect_put(struct snd_kcontrol *kcontrol,
	struct snd_ctl_elem_value *ucontrol)
{
#ifdef SARD_4_19_35	
	struct snd_soc_component *plat = snd_soc_kcontrol_component(kcontrol);
	struct mt8570_adsp_pcm_priv *priv =
		snd_soc_component_get_drvdata(plat);
#else
	struct snd_soc_platform *plat = snd_soc_kcontrol_platform(kcontrol);
	struct mt8570_adsp_pcm_priv *priv =
		snd_soc_platform_get_drvdata(plat);
#endif
		
	struct mt8570_adsp_control_data *data = &priv->ctrl_data;

	data->wakeword_detect_en = (ucontrol->value.integer.value[0]) ?
				   true : false;

	return 0;
}

#if defined(CONFIG_SND_SOC_MT8570_ADSP_PCM_PLAYBACK) || \
	defined(CONFIG_SND_SOC_MT8570_COMPRESS)
static int mt8570_adsp_master_volume_get(struct snd_kcontrol *kcontrol,
	struct snd_ctl_elem_value *ucontrol)
{
#ifdef SARD_4_19_35	
	struct snd_soc_component *plat = snd_soc_kcontrol_component(kcontrol);
	struct mt8570_adsp_pcm_priv *priv =
		snd_soc_component_get_drvdata(plat);
#else
	struct snd_soc_platform *plat = snd_soc_kcontrol_platform(kcontrol);
	struct mt8570_adsp_pcm_priv *priv =
		snd_soc_platform_get_drvdata(plat);
#endif
		
	struct mt8570_adsp_control_data *data = &priv->ctrl_data;

	ucontrol->value.integer.value[0] = data->master_vol;

	return 0;
}

static int mt8570_adsp_master_volume_put(struct snd_kcontrol *kcontrol,
	struct snd_ctl_elem_value *ucontrol)
{
#ifdef SARD_4_19_35	
	struct snd_soc_component *plat = snd_soc_kcontrol_component(kcontrol);
	struct mt8570_adsp_pcm_priv *priv =
		snd_soc_component_get_drvdata(plat);
#else
	struct snd_soc_platform *plat = snd_soc_kcontrol_platform(kcontrol);
	struct mt8570_adsp_pcm_priv *priv =
		snd_soc_platform_get_drvdata(plat);
#endif		
	struct dsp_vol_info *vol_info = &priv->adsp_vol.master;
	struct mt8570_adsp_control_data *data = &priv->ctrl_data;
	struct ipi_msg_t ipi_msg;
	int level = ucontrol->value.integer.value[0];

	if (level > HOST_VOLUME_MAX_LEVEL || level < 0)
		return -EINVAL;

	mt8570_adsp_host_to_dsp_vol(level, vol_info);

	memset(&ipi_msg, 0, sizeof(ipi_msg));

	dev_dbg(priv->dev, "%s level %d vol %d\n",
		__func__, level, vol_info->master_vol);

	mt8570_adsp_send_ipi_cmd(&ipi_msg,
		TASK_SCENE_AUDIO_CONTROLLER,
		AUDIO_IPI_LAYER_TO_DSP,
		AUDIO_IPI_MSG_ONLY,
		AUDIO_IPI_MSG_NEED_ACK,
		MSG_TO_DSP_DSP_SET_VOLUME,
		vol_info->master_vol,
		TASK_SCENE_MIXER_PRIMARY, NULL);

	data->master_vol = level;

	return 0;
}
#endif

#ifdef CONFIG_SND_SOC_MT8570_COMPRESS
static int get_compress_volume_id(const char *ctl_name)
{
	int id = -1;

#if CONFIG_SND_SOC_COMPRESS_NR_PLAYBACK_STREAMS > 0
	if (strstr(ctl_name, "Volume 1"))
		return MT8570_ADSP_FE_COMPR_PLAYBACK1;
#endif

#if CONFIG_SND_SOC_COMPRESS_NR_PLAYBACK_STREAMS > 1
	if (strstr(ctl_name, "Volume 2"))
		return MT8570_ADSP_FE_COMPR_PLAYBACK2;
#endif

#if CONFIG_SND_SOC_COMPRESS_NR_PLAYBACK_STREAMS > 2
	if (strstr(ctl_name, "Volume 3"))
		return MT8570_ADSP_FE_COMPR_PLAYBACK3;
#endif

	return id;
}

static int mt8570_adsp_compress_volume_get(struct snd_kcontrol *kcontrol,
	struct snd_ctl_elem_value *ucontrol)
{
#ifdef SARD_4_19_35	
	struct snd_soc_component *plat = snd_soc_kcontrol_component(kcontrol);
	struct mt8570_adsp_pcm_priv *priv =
		snd_soc_component_get_drvdata(plat);
#else
	struct snd_soc_platform *plat = snd_soc_kcontrol_platform(kcontrol);
	struct mt8570_adsp_pcm_priv *priv =
		snd_soc_platform_get_drvdata(plat);
#endif		
	struct mt8570_adsp_control_data *data = &priv->ctrl_data;
	int index, id;

	id = get_compress_volume_id(kcontrol->id.name);
	if (id < 0)
		return -EINVAL;

	index = mt8570_adsp_get_compr_index(id);
	if (index < 0)
		return -EINVAL;

	ucontrol->value.integer.value[0] = data->compress_vol[index];

	return 0;
}

static int mt8570_adsp_compress_volume_put(struct snd_kcontrol *kcontrol,
	struct snd_ctl_elem_value *ucontrol)
{
#ifdef SARD_4_19_35	
	struct snd_soc_component *plat = snd_soc_kcontrol_component(kcontrol);
	struct mt8570_adsp_pcm_priv *priv =
		snd_soc_component_get_drvdata(plat);
#else
	struct snd_soc_platform *plat = snd_soc_kcontrol_platform(kcontrol);
	struct mt8570_adsp_pcm_priv *priv =
		snd_soc_platform_get_drvdata(plat);
#endif	
	struct mt8570_adsp_control_data *data = &priv->ctrl_data;
	struct dsp_vol_info *vol_info;
	struct ipi_msg_t ipi_msg;
	int level = ucontrol->value.integer.value[0];
	int index, id, scene;

	if (level > HOST_VOLUME_MAX_LEVEL || level < 0)
		return -EINVAL;

	id = get_compress_volume_id(kcontrol->id.name);
	if (id < 0)
		return -EINVAL;

	index = mt8570_adsp_get_compr_index(id);
	if (index < 0)
		return -EINVAL;

	vol_info = &priv->adsp_vol.compress[index];

	mt8570_adsp_host_to_dsp_vol(level, vol_info);

	scene = mt8570_adsp_get_scene_by_dai_id(id);

	memset(&ipi_msg, 0, sizeof(ipi_msg));

	dev_dbg(priv->dev, "%s [#%d] level %d vol %d\n",
		__func__, scene, level, vol_info->input_vol);

	mt8570_adsp_send_ipi_cmd(&ipi_msg,
		TASK_SCENE_AUDIO_CONTROLLER,
		AUDIO_IPI_LAYER_TO_DSP,
		AUDIO_IPI_MSG_ONLY,
		AUDIO_IPI_MSG_NEED_ACK,
		MSG_TO_DSP_HOST_SET_VOLUME,
		vol_info->input_vol,
		scene, NULL);

	data->compress_vol[index] = level;

	return 0;
}
#endif


#ifdef CONFIG_SND_SOC_MT8570_ADSP_PCM_PLAYBACK
static int get_pcm_volume_index(const char *ctl_name)
{
	int index = -1;

	if (strstr(ctl_name, "Volume 1")) {
		index = MT8570_ADSP_FE_PCM_PLAYBACK1 -
			MT8570_ADSP_FE_PCM_BASE;
	}

	return index;
}

static int mt8570_adsp_pcm_volume_get(struct snd_kcontrol *kcontrol,
	struct snd_ctl_elem_value *ucontrol)
{
#ifdef SARD_4_19_35	
	struct snd_soc_component *plat = snd_soc_kcontrol_component(kcontrol);
	struct mt8570_adsp_pcm_priv *priv =
		snd_soc_component_get_drvdata(plat);
#else
	struct snd_soc_platform *plat = snd_soc_kcontrol_platform(kcontrol);
	struct mt8570_adsp_pcm_priv *priv =
		snd_soc_platform_get_drvdata(plat);
#endif
		
	struct mt8570_adsp_control_data *data = &priv->ctrl_data;
	int index;

	index = get_pcm_volume_index(kcontrol->id.name);

	if (index < 0 || index > MT8570_ADSP_FE_PCM_CNT)
		return -EINVAL;

	ucontrol->value.integer.value[0] = data->pcm_vol[index];

	return 0;
}

static int mt8570_adsp_pcm_volume_put(struct snd_kcontrol *kcontrol,
	struct snd_ctl_elem_value *ucontrol)
{
#ifdef SARD_4_19_35	
	struct snd_soc_component *plat = snd_soc_kcontrol_component(kcontrol);
	struct mt8570_adsp_pcm_priv *priv =
		snd_soc_component_get_drvdata(plat);
#else
	struct snd_soc_platform *plat = snd_soc_kcontrol_platform(kcontrol);
	struct mt8570_adsp_pcm_priv *priv =
		snd_soc_platform_get_drvdata(plat);
#endif
		
	struct mt8570_adsp_control_data *data = &priv->ctrl_data;
	struct dsp_vol_info *vol_info;
	struct ipi_msg_t ipi_msg;
	int level = ucontrol->value.integer.value[0];
	int index;
	int scene;

	if (level > HOST_VOLUME_MAX_LEVEL || level < 0)
		return -EINVAL;

	index = get_pcm_volume_index(kcontrol->id.name);

	if (index < 0 || index > MT8570_ADSP_FE_PCM_CNT)
		return -EINVAL;

	vol_info = &priv->adsp_vol.pcm[index];

	mt8570_adsp_host_to_dsp_vol(level, vol_info);

	scene =	mt8570_adsp_get_scene_by_dai_id(MT8570_ADSP_FE_PCM_BASE +
		index);

	memset(&ipi_msg, 0, sizeof(ipi_msg));

	dev_dbg(priv->dev, "%s [#%d] level %d vol %d\n",
		__func__, scene, level, vol_info->input_vol);

	mt8570_adsp_send_ipi_cmd(&ipi_msg,
		TASK_SCENE_AUDIO_CONTROLLER,
		AUDIO_IPI_LAYER_TO_DSP,
		AUDIO_IPI_MSG_ONLY,
		AUDIO_IPI_MSG_NEED_ACK,
		MSG_TO_DSP_HOST_SET_VOLUME,
		vol_info->input_vol,
		scene, NULL);

	data->pcm_vol[index] = level;

	return 0;
}
#endif

static int mt8570_adsp_tic_ns_get(struct snd_kcontrol *kcontrol,
	struct snd_ctl_elem_value *ucontrol)
{
#ifdef SARD_4_19_35	
	struct snd_soc_component *plat = snd_soc_kcontrol_component(kcontrol);
	struct mt8570_adsp_pcm_priv *priv =
		snd_soc_component_get_drvdata(plat);
#else
	struct snd_soc_platform *plat = snd_soc_kcontrol_platform(kcontrol);
	struct mt8570_adsp_pcm_priv *priv =
		snd_soc_platform_get_drvdata(plat);
#endif	
	struct mt8570_adsp_control_data *data = &priv->ctrl_data;

	ucontrol->value.integer.value[0] = data->dsp_tic_ns;

	return 0;
}

static int mt8570_adsp_tic_ns_put(struct snd_kcontrol *kcontrol,
	struct snd_ctl_elem_value *ucontrol)
{
#ifdef SARD_4_19_35	
	struct snd_soc_component *plat = snd_soc_kcontrol_component(kcontrol);
	struct mt8570_adsp_pcm_priv *priv =
		snd_soc_component_get_drvdata(plat);
#else
	struct snd_soc_platform *plat = snd_soc_kcontrol_platform(kcontrol);
	struct mt8570_adsp_pcm_priv *priv =
		snd_soc_platform_get_drvdata(plat);
#endif	
	struct mt8570_adsp_control_data *data = &priv->ctrl_data;
	int dsp_tic_ns = ucontrol->value.integer.value[0];

	dev_dbg(priv->dev, "%s change from %d to %d\n",
		__func__, data->dsp_tic_ns, dsp_tic_ns);

	if (dsp_tic_ns) {
		data->dsp_tic_ns = 1;
		/* already get tick info when dsp boot done */
	} else{
		data->dsp_tic_ns = 0;
	}

	return 0;
}

#ifdef CONFIG_SND_SOC_MT8570_ADSP_VOICE_ASSIST
static int mt8570_adsp_hotword_trigger_get(struct snd_kcontrol *kcontrol,
	struct snd_ctl_elem_value *ucontrol)
{
#ifdef SARD_4_19_35	
	struct snd_soc_component *plat = snd_soc_kcontrol_component(kcontrol);
	struct mt8570_adsp_pcm_priv *priv =
		snd_soc_component_get_drvdata(plat);
#else
	struct snd_soc_platform *plat = snd_soc_kcontrol_platform(kcontrol);
	struct mt8570_adsp_pcm_priv *priv =
		snd_soc_platform_get_drvdata(plat);
#endif		
	struct mt8570_adsp_control_data *data = &priv->ctrl_data;

	ucontrol->value.integer.value[0] = data->hotword_trig;

	return 0;
}

static int mt8570_adsp_hotword_trigger_put(struct snd_kcontrol *kcontrol,
	struct snd_ctl_elem_value *ucontrol)
{
#ifdef SARD_4_19_35	
	struct snd_soc_component *plat = snd_soc_kcontrol_component(kcontrol);
	struct mt8570_adsp_pcm_priv *priv =
		snd_soc_component_get_drvdata(plat);
#else
	struct snd_soc_platform *plat = snd_soc_kcontrol_platform(kcontrol);
	struct mt8570_adsp_pcm_priv *priv =
		snd_soc_platform_get_drvdata(plat);
#endif	
	struct mt8570_adsp_control_data *data = &priv->ctrl_data;
	int trigger = ucontrol->value.integer.value[0];

	dev_dbg(priv->dev, "%s change from %d to %d\n",
		__func__, data->hotword_trig, trigger);

	if (trigger) {
		mt8570_adsp_send_ipi_cmd(NULL,
			TASK_SCENE_VA_HOSTLESS,
			AUDIO_IPI_LAYER_TO_DSP,
			AUDIO_IPI_MSG_ONLY,
			AUDIO_IPI_MSG_DIRECT_SEND,
			MSG_TO_DSP_VA_FORCE_KEYWORD_PASS,
			0, 0, NULL);

		data->hotword_trig = 1;
	} else {
		data->hotword_trig = 0;
	}

	return 0;
}

static int mt8570_adsp_voice_trigger_get(struct snd_kcontrol *kcontrol,
	struct snd_ctl_elem_value *ucontrol)
{
#ifdef SARD_4_19_35	
	struct snd_soc_component *plat = snd_soc_kcontrol_component(kcontrol);
	struct mt8570_adsp_pcm_priv *priv =
		snd_soc_component_get_drvdata(plat);
#else
	struct snd_soc_platform *plat = snd_soc_kcontrol_platform(kcontrol);
	struct mt8570_adsp_pcm_priv *priv =
		snd_soc_platform_get_drvdata(plat);
#endif		
	struct mt8570_adsp_control_data *data = &priv->ctrl_data;

	ucontrol->value.integer.value[0] = data->voice_trig;

	return 0;
}

static int mt8570_adsp_voice_trigger_put(struct snd_kcontrol *kcontrol,
	struct snd_ctl_elem_value *ucontrol)
{
#ifdef SARD_4_19_35	
	struct snd_soc_component *plat = snd_soc_kcontrol_component(kcontrol);
	struct mt8570_adsp_pcm_priv *priv =
		snd_soc_component_get_drvdata(plat);
#else
	struct snd_soc_platform *plat = snd_soc_kcontrol_platform(kcontrol);
	struct mt8570_adsp_pcm_priv *priv =
		snd_soc_platform_get_drvdata(plat);
#endif
		
	struct mt8570_adsp_control_data *data = &priv->ctrl_data;
	int trigger = ucontrol->value.integer.value[0];

	dev_dbg(priv->dev, "%s change from %d to %d\n",
		__func__, data->voice_trig, trigger);

	if (ucontrol->value.integer.value[0]) {
		mt8570_adsp_send_ipi_cmd(NULL,
			TASK_SCENE_VA_HOSTLESS,
			AUDIO_IPI_LAYER_TO_DSP,
			AUDIO_IPI_MSG_ONLY,
			AUDIO_IPI_MSG_DIRECT_SEND,
			MSG_TO_DSP_VA_FORCE_VAD_STATE,
			0, 0, NULL);

		data->voice_trig = 1;
	} else {
		data->voice_trig = 0;
	}

	return 0;
}
#endif

static const struct snd_kcontrol_new mt8570_adsp_controls[] = {
	SOC_SINGLE_BOOL_EXT(WWE_DETECT_STS_CTL_NAME,
		0,
		mt8570_adsp_keyword_detect_get,
		mt8570_adsp_keyword_detect_put),
#if defined(CONFIG_SND_SOC_MT8570_ADSP_PCM_PLAYBACK) || \
	defined(CONFIG_SND_SOC_MT8570_COMPRESS)
	SOC_SINGLE_EXT("ADSP Master Volume",
		SND_SOC_NOPM, 0, HOST_VOLUME_MAX_LEVEL, 0,
		mt8570_adsp_master_volume_get,
		mt8570_adsp_master_volume_put),
#endif
#ifdef CONFIG_SND_SOC_MT8570_ADSP_PCM_PLAYBACK
	SOC_SINGLE_EXT("ADSP PCM Volume 1",
		SND_SOC_NOPM, 0, HOST_VOLUME_MAX_LEVEL, 0,
		mt8570_adsp_pcm_volume_get,
		mt8570_adsp_pcm_volume_put),
#endif
#ifdef CONFIG_SND_SOC_MT8570_COMPRESS
#if CONFIG_SND_SOC_COMPRESS_NR_PLAYBACK_STREAMS > 0
	SOC_SINGLE_EXT("ADSP Compress Volume 1",
		SND_SOC_NOPM, 0, HOST_VOLUME_MAX_LEVEL, 0,
		mt8570_adsp_compress_volume_get,
		mt8570_adsp_compress_volume_put),
#endif
#if CONFIG_SND_SOC_COMPRESS_NR_PLAYBACK_STREAMS > 1
	SOC_SINGLE_EXT("ADSP Compress Volume 2",
		SND_SOC_NOPM, 0, HOST_VOLUME_MAX_LEVEL, 0,
		mt8570_adsp_compress_volume_get,
		mt8570_adsp_compress_volume_put),
#endif
#if CONFIG_SND_SOC_COMPRESS_NR_PLAYBACK_STREAMS > 2
	SOC_SINGLE_EXT("ADSP Compress Volume 3",
		SND_SOC_NOPM, 0, HOST_VOLUME_MAX_LEVEL, 0,
		mt8570_adsp_compress_volume_get,
		mt8570_adsp_compress_volume_put),
#endif
#endif
	SOC_SINGLE_EXT("Get DSP Tic NS", SND_SOC_NOPM, 0, 1, 0,
		mt8570_adsp_tic_ns_get,
		mt8570_adsp_tic_ns_put),
#ifdef CONFIG_SND_SOC_MT8570_ADSP_VOICE_ASSIST
	SOC_SINGLE_EXT("Hotword Trigger", SND_SOC_NOPM, 0, 1, 0,
		mt8570_adsp_hotword_trigger_get,
		mt8570_adsp_hotword_trigger_put),
	SOC_SINGLE_EXT("Voice Trigger", SND_SOC_NOPM, 0, 1, 0,
		mt8570_adsp_voice_trigger_get,
		mt8570_adsp_voice_trigger_put),
#endif
};

#ifdef SARD_4_19_35
int mt8570_adsp_add_controls(struct snd_soc_component *platform)
{
	return snd_soc_add_component_controls(platform,
		mt8570_adsp_controls,
		ARRAY_SIZE(mt8570_adsp_controls));
}
#else
int mt8570_adsp_add_controls(struct snd_soc_platform *platform)
{
	return snd_soc_add_platform_controls(platform,
		mt8570_adsp_controls,
		ARRAY_SIZE(mt8570_adsp_controls));
}
#endif


