/*
 * mt8570-adsp-ppc.c  --  Mediatek 8570 adsp ppc function
 *
 * Copyright (c) 2019 MediaTek Inc.
 * Author: Hidalgo Huang <hidalgo.huang@mediatek.com>
 * Copyright 2022 Sony Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/dma-mapping.h>
#include <linux/miscdevice.h>
#include <linux/slab.h>
#include <linux/uaccess.h>
#include <sound/core.h>
#include <soc/mediatek/mtk-adsp-ppc-def.h>
#include <soc/mediatek/mt8570-adsp-utils.h>
#include <soc/mediatek/hifi4dsp/hifi4dsp_load.h>

struct mt8570_adsp_ppc_dev_priv {
	struct miscdevice misc_dev;
	struct device *dev;
	struct mutex dev_lock;
	void *cmd_buffer_virt;
	dma_addr_t cmd_buffer_phys;
	struct dsp_host_ppc_shared_mem_info ppc_mem_info;
};

#define miscdev_to_ppc_priv(d) \
	container_of(d, struct mt8570_adsp_ppc_dev_priv, misc_dev)

static int mt8570_adsp_verify_ppc_shared_mem_info(struct ipi_msg_t *ack_msg,
	struct dsp_host_ppc_shared_mem_info *ppc_mem_info)
{
	int ret = 0;

	if (!ack_msg)
		return -EINVAL;

	if (ack_msg->ack_type != AUDIO_IPI_MSG_ACK_BACK) {
		pr_info("%s unexpected ack type %u\n",
			__func__, ack_msg->ack_type);
		return -EINVAL;
	}

	if (!ppc_mem_info)
		return ret;

	memcpy(ppc_mem_info, ack_msg->payload,
	       sizeof(struct dsp_host_ppc_shared_mem_info));

	return ret;
}

static int mt8570_adsp_ppc_push_cmd(struct mt8570_adsp_ppc_dev_priv *priv,
	uint32_t cmd_type)
{
	int ret;

	dsp_spi_write_ex(priv->ppc_mem_info.start_addr,
		priv->cmd_buffer_virt,
		sizeof(struct dsp_host_ppc_cli_cmd),
		SPI_SPEED_HIGH);

	ret = mt8570_adsp_send_ipi_cmd(NULL,
		TASK_SCENE_PPC_CMD_HANDLER,
		AUDIO_IPI_LAYER_TO_DSP,
		AUDIO_IPI_MSG_ONLY,
		AUDIO_IPI_MSG_NEED_ACK,
		MSG_TO_DSP_PPC_SET,
		cmd_type, 0, NULL);

	return ret;
}

static int mt8570_adsp_ppc_pull_cmd(struct mt8570_adsp_ppc_dev_priv *priv,
	uint32_t cmd_type)
{
	int ret;

	dsp_spi_write_ex(priv->ppc_mem_info.start_addr,
		priv->cmd_buffer_virt,
		sizeof(struct dsp_host_ppc_cli_cmd),
		SPI_SPEED_HIGH);

	ret = mt8570_adsp_send_ipi_cmd(NULL,
		TASK_SCENE_PPC_CMD_HANDLER,
		AUDIO_IPI_LAYER_TO_DSP,
		AUDIO_IPI_MSG_ONLY,
		AUDIO_IPI_MSG_NEED_ACK,
		MSG_TO_DSP_PPC_GET,
		cmd_type, 0, NULL);
	if (ret)
		return ret;

	dsp_spi_read_ex(priv->ppc_mem_info.start_addr,
		priv->cmd_buffer_virt,
		sizeof(struct dsp_host_ppc_cli_cmd),
		SPI_SPEED_HIGH);

	return ret;
}

static int mt8570_adsp_ppc_open(struct inode *inode,
	struct file *filp)
{
	struct mt8570_adsp_ppc_dev_priv *priv =
		miscdev_to_ppc_priv(filp->private_data);
	struct ipi_msg_t ipi_msg;
	int ret;

	memset(&ipi_msg, 0, sizeof(ipi_msg));

	mt8570_adsp_notify_power_state(TASK_SCENE_PPC_CMD_HANDLER);

	ret = mt8570_adsp_send_ipi_cmd(&ipi_msg,
		TASK_SCENE_PPC_CMD_HANDLER,
		AUDIO_IPI_LAYER_TO_DSP,
		AUDIO_IPI_MSG_ONLY,
		AUDIO_IPI_MSG_NEED_ACK,
		MSG_TO_DSP_PPC_GET,
		PPC_GET_TYPE_GET_SHARED_MEM_ADDR,
		0, NULL);
	if (ret)
		goto ppc_open_clear_pwr_state;

	ret = mt8570_adsp_verify_ppc_shared_mem_info(&ipi_msg,
		&priv->ppc_mem_info);
	if (ret)
		goto ppc_open_clear_pwr_state;

	return 0;

ppc_open_clear_pwr_state:
	dev_info(priv->dev, "%s ret %d\n", __func__, ret);
	mt8570_adsp_clear_power_state(TASK_SCENE_PPC_CMD_HANDLER);

	return ret;
}

static int mt8570_adsp_ppc_release(struct inode *inode,
	struct file *filp)
{
	return 0;
}

static long mt8570_adsp_ppc_ioctl(struct file *filp,
	unsigned int cmd, unsigned long arg)
{
	struct mt8570_adsp_ppc_dev_priv *priv =
		miscdev_to_ppc_priv(filp->private_data);
	int ret = 0;

	mutex_lock(&priv->dev_lock);

	mt8570_adsp_notify_power_state(TASK_SCENE_PPC_CMD_HANDLER);

	switch (cmd) {
	case PPC_IOCTL_ENABLE:
	{
		int enable;

		ret = get_user(enable, (int __user *)arg);
		if (ret) {
			ret = -EFAULT;
			goto ppc_ioctl_exit;
		}

		dev_dbg(priv->dev, "%s set_enable %d\n",
			__func__, enable);

		ret = mt8570_adsp_send_ipi_cmd(NULL,
			TASK_SCENE_PPC_CMD_HANDLER,
			AUDIO_IPI_LAYER_TO_DSP,
			AUDIO_IPI_MSG_ONLY,
			AUDIO_IPI_MSG_NEED_ACK,
			enable ? MSG_TO_DSP_PPC_ON : MSG_TO_DSP_PPC_OFF,
			0, 0, NULL);
		break;
	}
	case PPC_IOCTL_SET_PARA:
	{
		struct ppc_para para;
		struct dsp_host_ppc_cli_cmd *ppc_cmd;
		int i;

		ret = copy_from_user(&para,
			(void __user *)arg,
			sizeof(struct ppc_para));
		if (ret) {
			ret = -EFAULT;
			goto ppc_ioctl_exit;
		}

		ppc_cmd = priv->cmd_buffer_virt;

		if (para.num > ARRAY_SIZE(ppc_cmd->para.param)) {
			ret = -EINVAL;
			goto ppc_ioctl_exit;
		}

		ppc_cmd->cmd_type = PPC_SET_TYPE_SET_PARA;
		ppc_cmd->para.element_id = para.element_id;
		ppc_cmd->para.param_id = para.para_id;
		ppc_cmd->para.param_num = para.num;

		ret = copy_from_user_toio(&ppc_cmd->para.param[0],
			para.value,
			sizeof(int64_t) * para.num);
		if (ret) {
			ret = -EFAULT;
			goto ppc_ioctl_exit;
		}

#if defined(CONFIG_DYNAMIC_DEBUG) || defined(DEBUG)
		for (i = 0; i < ppc_cmd->para.param_num; i++)
			dev_dbg(priv->dev, "%s set_para #%d %lld\n",
				__func__, i, ppc_cmd->para.param[i]);
#endif

		ret = mt8570_adsp_ppc_push_cmd(priv, ppc_cmd->cmd_type);
		break;
	}
	case PPC_IOCTL_GET_PARA:
	{
		struct ppc_para para;
		struct dsp_host_ppc_cli_cmd *ppc_cmd;
		int i;

		ret = copy_from_user(&para,
			(void __user *)arg,
			sizeof(struct ppc_para));
		if (ret) {
			ret = -EFAULT;
			goto ppc_ioctl_exit;
		}

		ppc_cmd = priv->cmd_buffer_virt;

		if (para.num > ARRAY_SIZE(ppc_cmd->para.param)) {
			ret = -EINVAL;
			goto ppc_ioctl_exit;
		}

		ppc_cmd->cmd_type = PPC_GET_TYPE_GET_PARA;
		ppc_cmd->para.element_id = para.element_id;
		ppc_cmd->para.param_id = para.para_id;
		ppc_cmd->para.param_num = para.num;

		ret = mt8570_adsp_ppc_pull_cmd(priv, ppc_cmd->cmd_type);
		if (ret)
			goto ppc_ioctl_exit;

		ret = copy_to_user_fromio(para.value,
			&ppc_cmd->para.param[0],
			sizeof(int64_t) * para.num);

#if defined(CONFIG_DYNAMIC_DEBUG) || defined(DEBUG)
		for (i = 0; i < ppc_cmd->para.param_num; i++)
			dev_dbg(priv->dev, "%s get_para #%d %lld\n",
				__func__, i, ppc_cmd->para.param[i]);
#endif
		break;
	}
	case PPC_IOCTL_SET_MODE:
	{
		int mode;
		struct dsp_host_ppc_cli_cmd *ppc_cmd;

		ret = get_user(mode, (int __user *)arg);
		if (ret) {
			ret = -EFAULT;
			goto ppc_ioctl_exit;
		}

		ppc_cmd = priv->cmd_buffer_virt;
		ppc_cmd->cmd_type = PPC_SET_TYPE_SET_MODE;
		ppc_cmd->mode = mode;

		dev_dbg(priv->dev, "%s set_mode %d\n",
			__func__, mode);

		ret = mt8570_adsp_ppc_push_cmd(priv, ppc_cmd->cmd_type);
		break;
	}
	case PPC_IOCTL_GET_MODE:
	{
		int mode;
		struct dsp_host_ppc_cli_cmd *ppc_cmd;

		ppc_cmd = priv->cmd_buffer_virt;
		ppc_cmd->cmd_type = PPC_GET_TYPE_GET_MODE;

		ret = mt8570_adsp_ppc_pull_cmd(priv, ppc_cmd->cmd_type);
		if (ret)
			goto ppc_ioctl_exit;

		mode = ppc_cmd->mode;

		dev_dbg(priv->dev, "%s get_mode %d\n",
			__func__, mode);

		ret = put_user(mode, (int __user *)arg);
		if (ret)
			ret = -EFAULT;

		break;
	}
	case PPC_IOCTL_SET_ELEM_BYP:
	{
		struct ppc_elem_byp elem_byp;
		int enable;
		struct dsp_host_ppc_cli_cmd *ppc_cmd;

		ret = copy_from_user(&elem_byp,
			(void __user *)arg,
			sizeof(struct ppc_elem_byp));
		if (ret) {
			ret = -EFAULT;
			goto ppc_ioctl_exit;
		}

		ret = get_user(enable, elem_byp.enable);
		if (ret) {
			ret = -EFAULT;
			goto ppc_ioctl_exit;
		}

		ppc_cmd = priv->cmd_buffer_virt;
		ppc_cmd->cmd_type = PPC_SET_TYPE_SET_ELEMENT_BYPASS;
		ppc_cmd->bypass.element_id = elem_byp.element_id;
		ppc_cmd->bypass.enabled = enable;

		dev_dbg(priv->dev, "%s set_elem_byp id 0x%x enable %d\n",
			__func__, elem_byp.element_id, enable);

		ret = mt8570_adsp_ppc_push_cmd(priv, ppc_cmd->cmd_type);
		break;
	}
	case PPC_IOCTL_GET_ELEM_BYP:
	{
		struct ppc_elem_byp elem_byp;
		int enable;
		struct dsp_host_ppc_cli_cmd *ppc_cmd;

		ret = copy_from_user(&elem_byp,
			(void __user *)arg,
			sizeof(struct ppc_elem_byp));
		if (ret) {
			ret = -EFAULT;
			goto ppc_ioctl_exit;
		}

		ppc_cmd = priv->cmd_buffer_virt;
		ppc_cmd->cmd_type = PPC_GET_TYPE_GET_ELEMENT_BYPASS;
		ppc_cmd->bypass.element_id = elem_byp.element_id;

		ret = mt8570_adsp_ppc_pull_cmd(priv, ppc_cmd->cmd_type);
		if (ret)
			goto ppc_ioctl_exit;

		enable = ppc_cmd->bypass.enabled;

		dev_dbg(priv->dev, "%s get_elem_byp id 0x%x enable %d\n",
			__func__, elem_byp.element_id, enable);

		ret = put_user(enable, elem_byp.enable);
		if (ret)
			ret = -EFAULT;
		break;
	}
	case PPC_IOCTL_TOOL_VER:
	{
		break;
	}
	case PPC_IOCTL_SWIP_VER:
	{
		break;
	}
	case PPC_IOCTL_SET_ELEM_DUMP:
	{
		struct ppc_elem_dump elem_dump;
		int enable;
		struct dsp_host_ppc_cli_cmd *ppc_cmd;

		ret = copy_from_user(&elem_dump,
			(void __user *)arg,
			sizeof(struct ppc_elem_dump));
		if (ret) {
			ret = -EFAULT;
			goto ppc_ioctl_exit;
		}

		ret = get_user(enable, elem_dump.enable);
		if (ret) {
			ret = -EFAULT;
			goto ppc_ioctl_exit;
		}

		ppc_cmd = priv->cmd_buffer_virt;
		ppc_cmd->cmd_type = PPC_SET_TYPE_DUMP_ELEMENT_DATA;
		ppc_cmd->dump.element_id = elem_dump.element_id;
		ppc_cmd->dump.enabled = enable;

		dev_dbg(priv->dev, "%s set_elem_dump id 0x%x enable %d\n",
			__func__, elem_dump.element_id, enable);

		ret = mt8570_adsp_ppc_push_cmd(priv, ppc_cmd->cmd_type);
		break;
	}
	default:
		ret = -ENOTTY;
		break;
	}

ppc_ioctl_exit:
	mt8570_adsp_clear_power_state(TASK_SCENE_PPC_CMD_HANDLER);

	mutex_unlock(&priv->dev_lock);

	if (ret)
		dev_info(priv->dev, "%s cmd 0x%x ret %d\n",
			 __func__, cmd, ret);

	return ret;
}

#ifdef CONFIG_COMPAT
static long mt8570_adsp_ppc_ioctl_compat(struct file *filp,
	unsigned int cmd, unsigned long arg)
{
	return mt8570_adsp_ppc_ioctl(filp, cmd,
		(unsigned long)compat_ptr(arg));
}
#endif

static const struct file_operations mt8570_adsp_ppc_fops = {
	.open = mt8570_adsp_ppc_open,
	.release = mt8570_adsp_ppc_release,
	.llseek = noop_llseek,
	.unlocked_ioctl = mt8570_adsp_ppc_ioctl,
#ifdef CONFIG_COMPAT
	.compat_ioctl = mt8570_adsp_ppc_ioctl_compat,
#endif
};

static int mt8570_adsp_ppc_probe(struct platform_device *pdev)
{
	struct mt8570_adsp_ppc_dev_priv *priv;
	struct device *dev = &pdev->dev;
	int ret;

	priv = devm_kzalloc(dev, sizeof(struct mt8570_adsp_ppc_dev_priv),
		GFP_KERNEL);
	if (!priv)
		return -ENOMEM;

	priv->cmd_buffer_virt = dma_alloc_coherent(dev,
		sizeof(struct dsp_host_ppc_cli_cmd),
		&priv->cmd_buffer_phys, GFP_KERNEL);
	if (!priv->cmd_buffer_virt)
		return -ENOMEM;

	priv->misc_dev.minor = MISC_DYNAMIC_MINOR;
	priv->misc_dev.name = "mtk_adsp_ppc";
	priv->misc_dev.fops = &mt8570_adsp_ppc_fops;

	ret = misc_register(&priv->misc_dev);
	if (ret < 0) {
		dev_info(dev, "%s misc_register failed\n", __func__);
		goto err_free_dma;
	}

	mutex_init(&priv->dev_lock);

	priv->dev = dev;

	platform_set_drvdata(pdev, priv);

	return 0;

err_free_dma:
	dma_free_coherent(dev,
		sizeof(struct dsp_host_ppc_cli_cmd),
		priv->cmd_buffer_virt,
		priv->cmd_buffer_phys);

	return ret;
}

static int mt8570_adsp_ppc_remove(struct platform_device *pdev)
{
	struct mt8570_adsp_ppc_dev_priv *priv = platform_get_drvdata(pdev);
	struct device *dev = &pdev->dev;

	misc_deregister(&priv->misc_dev);

	dma_free_coherent(dev,
		sizeof(struct dsp_host_ppc_cli_cmd),
		priv->cmd_buffer_virt,
		priv->cmd_buffer_phys);

	return 0;
}

static const struct of_device_id mt8570_adsp_ppc_dt_match[] = {
	{ .compatible = "mediatek,mt8570-adsp-ppc", },
	{ }
};
MODULE_DEVICE_TABLE(of, mt8570_adsp_ppc_dt_match);

static struct platform_driver mt8570_adsp_ppc_driver = {
	.driver = {
		.name = "mt8570-adsp-ppc",
		.of_match_table = mt8570_adsp_ppc_dt_match,
	},
	.probe = mt8570_adsp_ppc_probe,
	.remove = mt8570_adsp_ppc_remove,
};

module_platform_driver(mt8570_adsp_ppc_driver);

