// SPDX-License-Identifier: GPL-2.0+
/*
 * Copyright 2018 NXP
 * Copyright 2023 Sony Corporation
 */
#include <common.h>
#include <env.h>
#include <init.h>
#include <miiphy.h>
#include <netdev.h>
#include <asm/global_data.h>
#include <asm/mach-imx/iomux-v3.h>
#include <asm-generic/gpio.h>
#include <asm/arch/imx8mm_pins.h>
#include <asm/arch/clock.h>
#include <asm/arch/sys_proto.h>
#include <asm/mach-imx/gpio.h>
#include <asm/mach-imx/mxc_i2c.h>
#include <i2c.h>
#include <asm/io.h>
#include "../common/tcpc.h"
#include <usb.h>
#include <imx_sip.h>
#include <linux/arm-smccc.h>
#include <linux/io.h>
#include <asm/arch-imx8m/ddr.h>
#include <board_id.h>
#include <nvp.h>
#include <memalign.h>
#include <fsl_sec.h>
#if defined(CONFIG_SONY_PORT_INIT)
#include <sony_port_init.h>
#endif /* defined(CONFIG_SONY_PORT_INIT) */
#include <stdlib.h>
#include <mmc.h>
#include <command.h>

DECLARE_GLOBAL_DATA_PTR;

#define UART_PAD_CTRL	(PAD_CTL_DSE6 | PAD_CTL_FSEL1)
#define WDOG_PAD_CTRL	(PAD_CTL_DSE6 | PAD_CTL_ODE | PAD_CTL_PUE | PAD_CTL_PE)

static iomux_v3_cfg_t const uart_pads[] = {
	IMX8MM_PAD_UART2_RXD_UART2_RX | MUX_PAD_CTRL(UART_PAD_CTRL),
	IMX8MM_PAD_UART2_TXD_UART2_TX | MUX_PAD_CTRL(UART_PAD_CTRL),
};

static iomux_v3_cfg_t const wdog_pads[] = {
	IMX8MM_PAD_GPIO1_IO02_WDOG1_WDOG_B  | MUX_PAD_CTRL(WDOG_PAD_CTRL),
};

#ifdef CONFIG_NAND_MXS
#ifdef CONFIG_SPL_BUILD
#define NAND_PAD_CTRL	(PAD_CTL_DSE6 | PAD_CTL_FSEL2 | PAD_CTL_HYS)
#define NAND_PAD_READY0_CTRL (PAD_CTL_DSE6 | PAD_CTL_FSEL2 | PAD_CTL_PUE)
static iomux_v3_cfg_t const gpmi_pads[] = {
	IMX8MM_PAD_NAND_ALE_RAWNAND_ALE | MUX_PAD_CTRL(NAND_PAD_CTRL),
	IMX8MM_PAD_NAND_CE0_B_RAWNAND_CE0_B | MUX_PAD_CTRL(NAND_PAD_CTRL),
	IMX8MM_PAD_NAND_CLE_RAWNAND_CLE | MUX_PAD_CTRL(NAND_PAD_CTRL),
	IMX8MM_PAD_NAND_DATA00_RAWNAND_DATA00 | MUX_PAD_CTRL(NAND_PAD_CTRL),
	IMX8MM_PAD_NAND_DATA01_RAWNAND_DATA01 | MUX_PAD_CTRL(NAND_PAD_CTRL),
	IMX8MM_PAD_NAND_DATA02_RAWNAND_DATA02 | MUX_PAD_CTRL(NAND_PAD_CTRL),
	IMX8MM_PAD_NAND_DATA03_RAWNAND_DATA03 | MUX_PAD_CTRL(NAND_PAD_CTRL),
	IMX8MM_PAD_NAND_DATA04_RAWNAND_DATA04 | MUX_PAD_CTRL(NAND_PAD_CTRL),
	IMX8MM_PAD_NAND_DATA05_RAWNAND_DATA05	| MUX_PAD_CTRL(NAND_PAD_CTRL),
	IMX8MM_PAD_NAND_DATA06_RAWNAND_DATA06	| MUX_PAD_CTRL(NAND_PAD_CTRL),
	IMX8MM_PAD_NAND_DATA07_RAWNAND_DATA07	| MUX_PAD_CTRL(NAND_PAD_CTRL),
	IMX8MM_PAD_NAND_RE_B_RAWNAND_RE_B | MUX_PAD_CTRL(NAND_PAD_CTRL),
	IMX8MM_PAD_NAND_READY_B_RAWNAND_READY_B | MUX_PAD_CTRL(NAND_PAD_READY0_CTRL),
	IMX8MM_PAD_NAND_WE_B_RAWNAND_WE_B | MUX_PAD_CTRL(NAND_PAD_CTRL),
	IMX8MM_PAD_NAND_WP_B_RAWNAND_WP_B | MUX_PAD_CTRL(NAND_PAD_CTRL),
};
#endif

static void setup_gpmi_nand(void)
{
#ifdef CONFIG_SPL_BUILD
	imx_iomux_v3_setup_multiple_pads(gpmi_pads, ARRAY_SIZE(gpmi_pads));
#endif

	init_nand_clk();
}
#endif

#if defined(TARGET_BUILD_VARIANT_USER) || !defined(UBOOT_DEBUG)
#define EARLY_SILENT
#endif

int board_early_init_f(void)
{
	struct wdog_regs *wdog = (struct wdog_regs *)WDOG1_BASE_ADDR;

#if defined(EARLY_SILENT) && defined(CONFIG_SILENT_CONSOLE)
	gd->flags |= GD_FLG_SILENT;
#endif

	imx_iomux_v3_setup_multiple_pads(wdog_pads, ARRAY_SIZE(wdog_pads));

	set_wdog_reset(wdog);

	imx_iomux_v3_setup_multiple_pads(uart_pads, ARRAY_SIZE(uart_pads));

	init_uart_clk(1);

#ifdef CONFIG_NAND_MXS
	setup_gpmi_nand(); /* SPL will call the board_early_init_f */
#endif

	return 0;
}

#if IS_ENABLED(CONFIG_FEC_MXC)
static int setup_fec(void)
{
	struct iomuxc_gpr_base_regs *gpr =
		(struct iomuxc_gpr_base_regs *)IOMUXC_GPR_BASE_ADDR;

	/* Use 125M anatop REF_CLK1 for ENET1, not from external */
	clrsetbits_le32(&gpr->gpr[1], 0x2000, 0);

	return 0;
}

int board_phy_config(struct phy_device *phydev)
{
	if (phydev->drv->config)
		phydev->drv->config(phydev);

#ifndef CONFIG_DM_ETH
	/* enable rgmii rxc skew and phy mode select to RGMII copper */
	phy_write(phydev, MDIO_DEVAD_NONE, 0x1d, 0x1f);
	phy_write(phydev, MDIO_DEVAD_NONE, 0x1e, 0x8);

	phy_write(phydev, MDIO_DEVAD_NONE, 0x1d, 0x00);
	phy_write(phydev, MDIO_DEVAD_NONE, 0x1e, 0x82ee);
	phy_write(phydev, MDIO_DEVAD_NONE, 0x1d, 0x05);
	phy_write(phydev, MDIO_DEVAD_NONE, 0x1e, 0x100);
#endif

	return 0;
}
#endif

#ifdef CONFIG_USB_TCPC
struct tcpc_port port1;
struct tcpc_port port2;

static int setup_pd_switch(uint8_t i2c_bus, uint8_t addr)
{
	struct udevice *bus;
	struct udevice *i2c_dev = NULL;
	int ret;
	uint8_t valb;

	ret = uclass_get_device_by_seq(UCLASS_I2C, i2c_bus, &bus);
	if (ret) {
		printf("%s: Can't find bus\n", __func__);
		return -EINVAL;
	}

	ret = dm_i2c_probe(bus, addr, 0, &i2c_dev);
	if (ret) {
		printf("%s: Can't find device id=0x%x\n",
			__func__, addr);
		return -ENODEV;
	}

	ret = dm_i2c_read(i2c_dev, 0xB, &valb, 1);
	if (ret) {
		printf("%s dm_i2c_read failed, err %d\n", __func__, ret);
		return -EIO;
	}
	valb |= 0x4; /* Set DB_EXIT to exit dead battery mode */
	ret = dm_i2c_write(i2c_dev, 0xB, (const uint8_t *)&valb, 1);
	if (ret) {
		printf("%s dm_i2c_write failed, err %d\n", __func__, ret);
		return -EIO;
	}

	/* Set OVP threshold to 23V */
	valb = 0x6;
	ret = dm_i2c_write(i2c_dev, 0x8, (const uint8_t *)&valb, 1);
	if (ret) {
		printf("%s dm_i2c_write failed, err %d\n", __func__, ret);
		return -EIO;
	}

	return 0;
}

int pd_switch_snk_enable(struct tcpc_port *port)
{
	if (port == &port1) {
		debug("Setup pd switch on port 1\n");
		return setup_pd_switch(1, 0x72);
	} else if (port == &port2) {
		debug("Setup pd switch on port 2\n");
		return setup_pd_switch(1, 0x73);
	} else
		return -EINVAL;
}

struct tcpc_port_config port1_config = {
	.i2c_bus = 1, /*i2c2*/
	.addr = 0x50,
	.port_type = TYPEC_PORT_UFP,
	.max_snk_mv = 5000,
	.max_snk_ma = 3000,
	.max_snk_mw = 40000,
	.op_snk_mv = 9000,
	.switch_setup_func = &pd_switch_snk_enable,
};

struct tcpc_port_config port2_config = {
	.i2c_bus = 1, /*i2c2*/
	.addr = 0x52,
	.port_type = TYPEC_PORT_UFP,
	.max_snk_mv = 9000,
	.max_snk_ma = 3000,
	.max_snk_mw = 40000,
	.op_snk_mv = 9000,
	.switch_setup_func = &pd_switch_snk_enable,
};

static int setup_typec(void)
{
	int ret;

	debug("tcpc_init port 2\n");
	ret = tcpc_init(&port2, port2_config, NULL);
	if (ret) {
		printf("%s: tcpc port2 init failed, err=%d\n",
		       __func__, ret);
	} else if (tcpc_pd_sink_check_charging(&port2)) {
		/* Disable PD for USB1, since USB2 has priority */
		port1_config.disable_pd = true;
		printf("Power supply on USB2\n");
	}

	debug("tcpc_init port 1\n");
	ret = tcpc_init(&port1, port1_config, NULL);
	if (ret) {
		printf("%s: tcpc port1 init failed, err=%d\n",
		       __func__, ret);
	} else {
		if (!port1_config.disable_pd)
			printf("Power supply on USB1\n");
		return ret;
	}

	return ret;
}

int board_usb_init(int index, enum usb_init_type init)
{
	int ret = 0;
	struct tcpc_port *port_ptr;

	debug("board_usb_init %d, type %d\n", index, init);

	if (index == 0)
		port_ptr = &port1;
	else
		port_ptr = &port2;

	imx8m_usb_power(index, true);

	if (init == USB_INIT_HOST)
		tcpc_setup_dfp_mode(port_ptr);
	else
		tcpc_setup_ufp_mode(port_ptr);

	return ret;
}

int board_usb_cleanup(int index, enum usb_init_type init)
{
	int ret = 0;

	debug("board_usb_cleanup %d, type %d\n", index, init);

	if (init == USB_INIT_HOST) {
		if (index == 0)
			ret = tcpc_disable_src_vbus(&port1);
		else
			ret = tcpc_disable_src_vbus(&port2);
	}

	imx8m_usb_power(index, false);
	return ret;
}

int board_ehci_usb_phy_mode(struct udevice *dev)
{
	int ret = 0;
	enum typec_cc_polarity pol;
	enum typec_cc_state state;
	struct tcpc_port *port_ptr;

	if (dev_seq(dev) == 0)
		port_ptr = &port1;
	else
		port_ptr = &port2;

	tcpc_setup_ufp_mode(port_ptr);

	ret = tcpc_get_cc_status(port_ptr, &pol, &state);
	if (!ret) {
		if (state == TYPEC_STATE_SRC_RD_RA || state == TYPEC_STATE_SRC_RD)
			return USB_INIT_HOST;
	}

	return USB_INIT_DEVICE;
}

#endif

static int set_append_bootargs(char *append_cmdline, unsigned int size)
{
	char *buf;
	const char *env_val;
	char *cmdline = env_get("append_bootargs");

	debug("before append_bootarg fix-up: %s\n", cmdline);

	if (cmdline && (cmdline[0] != '\0')) {
		/* Allocate space for maximum possible new command line */
		buf = malloc(strlen(cmdline) + 1 + size + 1);
		if (!buf) {
			debug("%s: out of memory\n", __func__);
			return -1;
		}
		snprintf(buf, (strlen(cmdline) + 1 + size + 1), "%s %s", cmdline, append_cmdline);
		env_val = buf;
	} else {
		buf = NULL;
		env_val = append_cmdline;
	}

	env_set("append_bootargs", env_val);
	debug("after append_bootarg: %s\n", env_val);
	free(buf);
	return 0;
}

#ifdef CONFIG_SYSTEMD_FIXED_MACHINE_ID
#define OCOTP_HW_OCOTP_TESTER0 0x30350410
#define OCOTP_HW_OCOTP_TESTER1 0x30350420
static int create_systemd_machine_id(char *systemd_machine_id, unsigned int size)
{
	void __iomem *machine_id_base;
	u32 tester0, tester1;

	if (size < 64)
		return -1;

	machine_id_base = ioremap(OCOTP_HW_OCOTP_TESTER0,0x20);
	tester0 = ioread32(machine_id_base);
	tester1 = ioread32(machine_id_base + 0x10);
	snprintf(systemd_machine_id, 64,"systemd.machine_id=0000000000000000%08x%08x",
						tester0, tester1);
	iounmap(machine_id_base);
	return 0;
}

static int fixed_systemd_machine_id(void)
{
	char unique_id[64];

	if (create_systemd_machine_id(unique_id, sizeof(unique_id)) < 0)
		return -1;

	if (set_append_bootargs(unique_id, sizeof(unique_id)) < 0)
		return -1;

	return 0;
}
#endif

#define DISPMIX				9
#define MIPI				10

int board_init(void)
{
	struct arm_smccc_res res;

#if defined(CONFIG_SONY_PORT_INIT)
	initr_sony_port_init();
#endif /* defined(CONFIG_SONY_PORT_INIT) */

#ifdef CONFIG_USB_TCPC
	setup_typec();
#endif

#if IS_ENABLED(CONFIG_FEC_MXC)
	setup_fec();
#endif

	arm_smccc_smc(IMX_SIP_GPC, IMX_SIP_GPC_PM_DOMAIN,
		      DISPMIX, true, 0, 0, 0, 0, &res);
	arm_smccc_smc(IMX_SIP_GPC, IMX_SIP_GPC_PM_DOMAIN,
		      MIPI, true, 0, 0, 0, 0, &res);

#ifdef CONFIG_BOARD_ID
	initr_board_id();
#endif

	return 0;
}

#ifdef CONFIG_ENV_IS_IN_MMC
#define KERNEL_DTB_PARTITION	1
#define ROOT_FS_PARTITION	2

static void sony_late_mmc_env_init(void)
{
	char *autodetect = env_get("mmcautodetect");

	if (autodetect && strncmp(autodetect, "yes", strlen("yes") + 1) == 0) {
		char cmd[32];
		char mmcblk[32];
		u32 dev_no = mmc_get_env_dev();

		env_set_ulong("mmcdev", dev_no);
		env_set_ulong("cm4_load_mmcdev", dev_no);

		env_set_ulong("mmcpart", KERNEL_DTB_PARTITION);
		env_set_ulong("cm4_load_mmcpart", KERNEL_DTB_PARTITION);

		/* Set mmcblk env */
		snprintf(mmcblk, sizeof(mmcblk), "/dev/mmcblk%dp%d rootwait ro",
			mmc_map_to_kernel_blk(dev_no), ROOT_FS_PARTITION);
		env_set("mmcroot", mmcblk);

		snprintf(cmd, sizeof(cmd), "mmc dev %d", dev_no);
		run_command(cmd, 0);
	}
}
#endif /* CONFIG_ENV_IS_IN_MMC */

#define FSL_CAAM_MP_PUBK_BYTES		64
#define MAX_SIZE_BUF			256

/** Get mppubkey of mfgprot command
 *
 *  \param buf 	The string for output
 *  \return	Success(0), fail(-1)
 */
static int get_mfgprot_pubk(char *buf)
{
	int i, ret;
	u8 *dst_ptr;
	char tmp_buf[3] = {'\0'};
	char key_buf[MAX_SIZE_BUF] = {'\0'};

	dst_ptr = malloc_cache_aligned(FSL_CAAM_MP_PUBK_BYTES);
	if (!dst_ptr)
		return -1;

	ret = gen_mppubk(dst_ptr);
	if (ret) {
		free(dst_ptr);
		return -1;
	}

	/* Output results */
	for (i = 0; i < FSL_CAAM_MP_PUBK_BYTES; i++) {
		snprintf(tmp_buf, sizeof(tmp_buf), "%02X", (dst_ptr)[i]);
		if (strlen(key_buf) + strlen(tmp_buf) < FSL_CAAM_MP_PUBK_BYTES*2 + 1)
			strncat(key_buf, tmp_buf, 2);
	}
	snprintf(buf, MAX_SIZE_BUF, "pubk=%s", key_buf);
	free(dst_ptr);

	return 0;
}

int board_late_init(void)
{
#ifdef CONFIG_ICX_NVP_EMMC
	unsigned char mac[6];
	char buf[64];
	int bootmode;
	int gpio_diag = 1;
#endif
	char cmd_buf[MAX_SIZE_BUF];

#ifdef CONFIG_SILENT_CONSOLE
	bool force_console_on = false;
#endif

#ifdef CONFIG_ENV_IS_IN_MMC
	sony_late_mmc_env_init();
#endif

#ifdef CONFIG_ICX_NVP_EMMC
	(void) icx_nvp_init();

	bootmode = icx_nvp_get_bootmode();
#if defined(CONFIG_SONY_PORT_INIT)
	gpio_diag = gpio_get_value(BAR_GPIO_XDIAG);
#endif
	if ((gpio_diag == 0) ||
	    (bootmode == NVP_BOOTMODE_DIAG) ||
	    (bootmode == NVP_BOOTMODE_FWUPDATE)) {
		env_set_ulong("mmcpart", 4); /* RECOVERY */
		set_append_bootargs("bootkernel=recovery", sizeof("bootkernel=recovery"));
	} else {
		if (bootmode == NVP_BOOTMODE_SERVICE)
			set_append_bootargs("service_mode=1", sizeof("service_mode=1"));
		set_append_bootargs("bootkernel=normal", sizeof("bootkernel=normal"));
	}

	/* Only gpio_diag */
	if (gpio_diag == 0) {
		if (get_mfgprot_pubk(cmd_buf) == 0)
			set_append_bootargs(cmd_buf, sizeof(cmd_buf));
	}

	icx_nvp_get_mac(mac, 6);
	snprintf(buf, sizeof(buf), "fec.macaddr=0x%.2x,0x%.2x,0x%.2x,0x%.2x,0x%.2x,0x%.2x",
		mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);
	set_append_bootargs(buf, strlen(buf));

#ifdef CONFIG_SILENT_CONSOLE
	force_console_on = gpio_diag == 0 || bootmode == NVP_BOOTMODE_DIAG;
#endif
#endif

#ifdef CONFIG_ENV_VARS_UBOOT_RUNTIME_CONFIG
	env_set("board_name", "HA-BB-3-3");
	env_set("board_rev", "iMX8MM");
#endif

	/* enable DDRC PWRCTL: en_dfi_dram_clk_disable, powerdown_en, selfref_en */
	writel(readl(DDRC_PWRCTL(0)) | 0xb, DDRC_PWRCTL(0));

#if !defined(EARLY_SILENT) && defined(CONFIG_SILENT_CONSOLE)
	if (gd->flags & GD_FLG_ENV_DEFAULT)
		env_set("silent", "1");
#endif

#ifdef CONFIG_SILENT_CONSOLE
	if (!force_console_on && env_get("silent") && !nvp_emmc_check_printk())
		gd->flags |= GD_FLG_SILENT;
	else
		gd->flags &= ~GD_FLG_SILENT;

	if (!force_console_on  && !nvp_emmc_check_printk())
		set_append_bootargs(CM4_CONSOLE_OFF, sizeof(CM4_CONSOLE_OFF));
	else
		set_append_bootargs(CM4_CONSOLE_ON, sizeof(CM4_CONSOLE_ON));

#endif

#ifdef CONFIG_SYSTEMD_FIXED_MACHINE_ID
	if (fixed_systemd_machine_id() < 0)
		return -1;
#endif
	set_append_bootargs("init_fatal_sh=1", sizeof("init_fatal_sh=1"));

	printf("[DRAM] Limit RAM to 992MB in this environment.\n");
	set_append_bootargs("mem=992M", sizeof("mem=992M"));

	return 0;
}

#ifdef CONFIG_ANDROID_SUPPORT
bool is_power_key_pressed(void) {
	return (bool)(!!(readl(SNVS_HPSR) & (0x1 << 6)));
}
#endif

#ifdef CONFIG_FSL_FASTBOOT
#ifdef CONFIG_ANDROID_RECOVERY
int is_recovery_key_pressing(void)
{
	return 0; /* TODO */
}
#endif /* CONFIG_ANDROID_RECOVERY */
#endif /* CONFIG_FSL_FASTBOOT */
