// SPDX-License-Identifier: GPL-2.0+
/*
 * Copyright 2019 NXP
 * Copyright 2023 Sony Corporation
 */

#include <common.h>
#include <command.h>
#include <cpu_func.h>
#include <hang.h>
#include <image.h>
#include <init.h>
#include <log.h>
#include <spl.h>
#include <asm/global_data.h>
#include <asm/io.h>
#include <asm/mach-imx/iomux-v3.h>
#include <asm/arch/clock.h>
#include <asm/arch/imx8mm_pins.h>
#include <asm/arch/sys_proto.h>
#include <asm/mach-imx/boot_mode.h>
#include <asm/arch/ddr.h>

#include <power/pmic.h>
#ifdef CONFIG_POWER_PCA9450
#include <power/pca9450.h>
#else
#include <power/bd71837.h>
#endif
#include <asm/mach-imx/gpio.h>
#include <asm/mach-imx/mxc_i2c.h>
#include <fsl_esdhc_imx.h>
#include <fsl_sec.h>
#include <mmc.h>
#include <linux/delay.h>

DECLARE_GLOBAL_DATA_PTR;

int spl_board_boot_device(enum boot_device boot_dev_spl)
{
	switch (boot_dev_spl) {
	case SD2_BOOT:
	case MMC2_BOOT:
		return BOOT_DEVICE_MMC1;
	case SD3_BOOT:
	case MMC3_BOOT:
		return BOOT_DEVICE_MMC2;
	case QSPI_BOOT:
		return BOOT_DEVICE_NOR;
	case NAND_BOOT:
		return BOOT_DEVICE_NAND;
	case USB_BOOT:
		return BOOT_DEVICE_BOARD;
	default:
		return BOOT_DEVICE_NONE;
	}
}

#define DRAM_ID1_GPIO IMX_GPIO_NR(3, 14)
#define DRAM_ID2_GPIO IMX_GPIO_NR(3, 8)
#define PAD_CTRL_DRAM_GPIO_DS0 MUX_PAD_CTRL(PAD_CTL_DSE1 | PAD_CTL_PE)
static iomux_v3_cfg_t const dramid_pads[] = {
	IMX8MM_PAD_NAND_DQS_GPIO3_IO14 | PAD_CTRL_DRAM_GPIO_DS0, /* DRAM_ID1 */
	IMX8MM_PAD_NAND_DATA02_GPIO3_IO8 | PAD_CTRL_DRAM_GPIO_DS0,   /* DRAM_ID2 */
};

int get_dramid_capacity(void)
{
	int dram_id;
	int dramid_capacity = 2;

	imx_iomux_v3_setup_multiple_pads(dramid_pads,
					 ARRAY_SIZE(dramid_pads));
	gpio_request(DRAM_ID1_GPIO, "DRAMID1");
	gpio_request(DRAM_ID2_GPIO, "DRAMID2");
	gpio_direction_input(DRAM_ID1_GPIO);
	gpio_direction_input(DRAM_ID2_GPIO);
	dram_id = gpio_get_value(DRAM_ID1_GPIO)
		 | (gpio_get_value(DRAM_ID2_GPIO) << 1);

	switch (dram_id) {
	case 0:
		dramid_capacity = 2;
		break;
	case 1:
		dramid_capacity = 1;
		break;
	default:
		printf("Warning: dram_id:%d is not unexpected.\n", dram_id);
		dramid_capacity = 2;
	}

	printf("%s: dram_id:%x dramid capacity:%dGB\n", __func__, dram_id, dramid_capacity);

	return dramid_capacity;
}

int get_dram_density(void)
{
	unsigned int ddr_info = 0;
	static const unsigned int regs = 8; //density in MR8
	int dram_density;
	int dram_capacity;

	ddr_info = lpddr4_mr_read(0xF, regs);
	dram_density = (ddr_info >> 2) & 0xf;

	switch (dram_density) {
	case 0x4:
		dram_capacity = 2;
		break;
	case 0x2:
		dram_capacity = 1;
		break;
	default:
		printf("Warning: dram_density:%d is not unexpected.\n", dram_density);
		dram_capacity = 0;
	}

	printf("DRAM Density = 0x%x dram capacity:%dGB\n", dram_density, dram_capacity);

	return dram_capacity;
}

void spl_dram_init(void)
{
	int dramid_capacity;
	int mr_capacity;

	dramid_capacity = get_dramid_capacity();
	if (dramid_capacity == 1) {
		printf("start 1GB traning\n");
		ddr_init(&dram_timing_1g);
		//check dram mr register
		mr_capacity = get_dram_density();
		if (mr_capacity && (dramid_capacity != mr_capacity)) {
			printf("Warning:difference in capacity between dramid_capacity:%dGB and mr_capacity:%dGB\n", dramid_capacity, mr_capacity);
			printf("start 2GB traning\n");
			ddr_init(&dram_timing_2g);
		}

	}
	else { //dramid_capacity 2gb and else
		printf("start 2GB traning\n");
		ddr_init(&dram_timing_2g);
		//check dram mr register
		mr_capacity = get_dram_density();
		if (mr_capacity && (dramid_capacity != mr_capacity)) {
			printf("Warning:difference in capacity between dramid_capacity:%dGB and mr_capacity:%dGB\n", dramid_capacity, mr_capacity);
			printf("start 1GB traning\n");
			ddr_init(&dram_timing_1g);
		}
	}
}

#define I2C_PAD_CTRL	(PAD_CTL_DSE6 | PAD_CTL_HYS | PAD_CTL_PUE | PAD_CTL_PE)
#define PC MUX_PAD_CTRL(I2C_PAD_CTRL)
struct i2c_pads_info i2c_pad_info1 = {
	.scl = {
		.i2c_mode = IMX8MM_PAD_I2C1_SCL_I2C1_SCL | PC,
		.gpio_mode = IMX8MM_PAD_I2C1_SCL_GPIO5_IO14 | PC,
		.gp = IMX_GPIO_NR(5, 14),
	},
	.sda = {
		.i2c_mode = IMX8MM_PAD_I2C1_SDA_I2C1_SDA | PC,
		.gpio_mode = IMX8MM_PAD_I2C1_SDA_GPIO5_IO15 | PC,
		.gp = IMX_GPIO_NR(5, 15),
	},
};

#define USDHC2_CD_GPIO	IMX_GPIO_NR(2, 18)
#define USDHC2_PWR_GPIO IMX_GPIO_NR(2, 19)

#define USDHC_PAD_CTRL	(PAD_CTL_DSE6 | PAD_CTL_HYS | PAD_CTL_PUE |PAD_CTL_PE | \
			 PAD_CTL_FSEL2)
#define USDHC_GPIO_PAD_CTRL (PAD_CTL_HYS | PAD_CTL_DSE1)

static iomux_v3_cfg_t const usdhc3_pads[] = {
	IMX8MM_PAD_NAND_WE_B_USDHC3_CLK | MUX_PAD_CTRL(USDHC_PAD_CTRL),
	IMX8MM_PAD_NAND_WP_B_USDHC3_CMD | MUX_PAD_CTRL(USDHC_PAD_CTRL),
	IMX8MM_PAD_NAND_DATA04_USDHC3_DATA0 | MUX_PAD_CTRL(USDHC_PAD_CTRL),
	IMX8MM_PAD_NAND_DATA05_USDHC3_DATA1 | MUX_PAD_CTRL(USDHC_PAD_CTRL),
	IMX8MM_PAD_NAND_DATA06_USDHC3_DATA2 | MUX_PAD_CTRL(USDHC_PAD_CTRL),
	IMX8MM_PAD_NAND_DATA07_USDHC3_DATA3 | MUX_PAD_CTRL(USDHC_PAD_CTRL),
	IMX8MM_PAD_NAND_RE_B_USDHC3_DATA4 | MUX_PAD_CTRL(USDHC_PAD_CTRL),
	IMX8MM_PAD_NAND_CE2_B_USDHC3_DATA5 | MUX_PAD_CTRL(USDHC_PAD_CTRL),
	IMX8MM_PAD_NAND_CE3_B_USDHC3_DATA6 | MUX_PAD_CTRL(USDHC_PAD_CTRL),
	IMX8MM_PAD_NAND_CLE_USDHC3_DATA7 | MUX_PAD_CTRL(USDHC_PAD_CTRL),
};

static iomux_v3_cfg_t const usdhc2_pads[] = {
	IMX8MM_PAD_SD2_CLK_USDHC2_CLK | MUX_PAD_CTRL(USDHC_PAD_CTRL),
	IMX8MM_PAD_SD2_CMD_USDHC2_CMD | MUX_PAD_CTRL(USDHC_PAD_CTRL),
	IMX8MM_PAD_SD2_DATA0_USDHC2_DATA0 | MUX_PAD_CTRL(USDHC_PAD_CTRL),
	IMX8MM_PAD_SD2_DATA1_USDHC2_DATA1 | MUX_PAD_CTRL(USDHC_PAD_CTRL),
	IMX8MM_PAD_SD2_DATA2_USDHC2_DATA2 | MUX_PAD_CTRL(USDHC_PAD_CTRL),
	IMX8MM_PAD_SD2_DATA3_USDHC2_DATA3 | MUX_PAD_CTRL(USDHC_PAD_CTRL),
	IMX8MM_PAD_SD2_RESET_B_GPIO2_IO19 | MUX_PAD_CTRL(USDHC_GPIO_PAD_CTRL),
};

/*
 * The evk board uses DAT3 to detect CD card plugin,
 * in u-boot we mux the pin to GPIO when doing board_mmc_getcd.
 */
static iomux_v3_cfg_t const usdhc2_cd_pad =
	IMX8MM_PAD_SD2_DATA3_GPIO2_IO18 | MUX_PAD_CTRL(USDHC_GPIO_PAD_CTRL);

static iomux_v3_cfg_t const usdhc2_dat3_pad =
	IMX8MM_PAD_SD2_DATA3_USDHC2_DATA3 |
	MUX_PAD_CTRL(USDHC_PAD_CTRL);


static struct fsl_esdhc_cfg usdhc_cfg[2] = {
	{USDHC2_BASE_ADDR, 0, 4},
	{USDHC3_BASE_ADDR, 0, 8},
};

int board_mmc_init(struct bd_info *bis)
{
	int i, ret;
	/*
	 * According to the board_mmc_init() the following map is done:
	 * (U-Boot device node)    (Physical Port)
	 * mmc0                    USDHC1
	 * mmc1                    USDHC2
	 */
	for (i = 0; i < CONFIG_SYS_FSL_USDHC_NUM; i++) {
		switch (i) {
		case 0:
			init_clk_usdhc(1);
			usdhc_cfg[0].sdhc_clk = mxc_get_clock(MXC_ESDHC2_CLK);
			imx_iomux_v3_setup_multiple_pads(
				usdhc2_pads, ARRAY_SIZE(usdhc2_pads));
			gpio_request(USDHC2_PWR_GPIO, "usdhc2_reset");
			gpio_direction_output(USDHC2_PWR_GPIO, 0);
			udelay(500);
			gpio_direction_output(USDHC2_PWR_GPIO, 1);
			break;
		case 1:
			init_clk_usdhc(2);
			usdhc_cfg[1].sdhc_clk = mxc_get_clock(MXC_ESDHC3_CLK);
			imx_iomux_v3_setup_multiple_pads(
				usdhc3_pads, ARRAY_SIZE(usdhc3_pads));
			break;
		default:
			printf("Warning: you configured more USDHC controllers"
				"(%d) than supported by the board\n", i + 1);
			return -EINVAL;
		}

		ret = fsl_esdhc_initialize(bis, &usdhc_cfg[i]);
		if (ret)
			return ret;
	}

	return 0;
}

int board_mmc_getcd(struct mmc *mmc)
{
	struct fsl_esdhc_cfg *cfg = (struct fsl_esdhc_cfg *)mmc->priv;
	int ret = 0;

	switch (cfg->esdhc_base) {
	case USDHC3_BASE_ADDR:
		ret = 1;
		break;
	case USDHC2_BASE_ADDR:
		imx_iomux_v3_setup_pad(usdhc2_cd_pad);
		gpio_request(USDHC2_CD_GPIO, "usdhc2 cd");
		gpio_direction_input(USDHC2_CD_GPIO);

		/*
		 * Since it is the DAT3 pin, this pin is pulled to
		 * low voltage if no card
		 */
		ret = gpio_get_value(USDHC2_CD_GPIO);

		imx_iomux_v3_setup_pad(usdhc2_dat3_pad);
		return ret;
	}

	return 1;
}

#ifdef CONFIG_POWER
#define I2C_PMIC	0
#ifdef CONFIG_POWER_PCA9450
int power_init_board(void)
{
	struct pmic *p;
	int ret;

	ret = power_pca9450_init(I2C_PMIC);
	if (ret)
		printf("power init failed");
	p = pmic_get("PCA9450");
	pmic_probe(p);

	/* BUCKxOUT_DVS0/1 control BUCK123 output */
	pmic_reg_write(p, PCA9450_BUCK123_DVS, 0x29);

	/* Buck 1 DVS control through PMIC_STBY_REQ */
	pmic_reg_write(p, PCA9450_BUCK1CTRL, 0x59);

	/* Set DVS1 to 0.8v for suspend */
	pmic_reg_write(p, PCA9450_BUCK1OUT_DVS1, 0x10);

	/* increase VDD_DRAM to 0.95v for 3Ghz DDR */
	pmic_reg_write(p, PCA9450_BUCK3OUT_DVS0, 0x1C);

	/* VDD_DRAM needs off in suspend, set B1_ENMODE=10 (ON by PMIC_ON_REQ = H && PMIC_STBY_REQ = L) */
	pmic_reg_write(p, PCA9450_BUCK3CTRL, 0x4a);

	/* set VDD_SNVS_0V8 from default 0.85V */
	pmic_reg_write(p, PCA9450_LDO2CTRL, 0xC0);

	/* set WDOG_B_CFG to cold reset */
	pmic_reg_write(p, PCA9450_RESET_CTRL, 0xA1);

	return 0;
}
#else
int power_init_board(void)
{
	struct pmic *p;
	int ret;
	u32 val = 0x0;

	ret = power_bd71837_init(I2C_PMIC);
	if (ret)
		printf("power init failed");

	p = pmic_get("BD71837");
	pmic_probe(p);

	/* BUCK 5 Auto mode workaround */
	ret = pmic_reg_read(p, 0x09, &val);
	if (ret != 0)
		printf("--> Fail to read buck5 register(0x09)\n");

	printf("buck5 0x%x\n", val);

	if (val == 0x02 /* PDVPUGPUDRAM */) {
		ret = pmic_reg_write(p, 0x09, 0x00);
		if (ret != 0)
			printf("--> Fail to write buck5 register\n");
		udelay(1000); /* wait a stable of switching regulator */
	}

#ifdef CONFIG_ICX_ERRATA_BD71837_REVA0
	/* BD71837 bug workaround for rev.A0 */
  {
	unsigned char addr_org = pmic_i2c_addr;
	u32 regval;
	int rval;

	printf("BD71837: i2c_addr = 0x%x\n", addr_org);

	if (pmic_reg_read(p, BD718XX_REV, &regval) != 0)
		printf("power_init_board: cannot read PMIC REV register.\n");

	printf("BD71837: revision(0x%x)\n", regval);
	do {
		if (regval == 0xa0) {
			printf("BD71837 need a workaround\n");

			/* Try to enable slave addr 0x5c/5d/5e */
			rval = pmic_reg_write(p, 0xfe, 0x8c);
			if (rval != 0) {
				printf("--> Fail to enter debug mode(1)\n");
				break;
			}

			/*
			 * Try to enter test mode with alternate slave addr
			 * BUS(0):ADDR(0x5e):REG(0) enter into test mode
			 */
			pmic_i2c_addr = 0x5e;
			rval = pmic_reg_write(p, 0x0, 0x1);
			if (rval != 0) {
				printf("--> Fail to enter debug mode(2)\n");
				break;
			}

			/*
			 * write REG[0x5E:7F] to 0x57;
			 */
			rval = pmic_reg_read(p, 0x7f, &regval);
			if (rval != 0) {
				printf("--> Fail to read debug register(0x5E:0x7F)\n");
				break;
			}
			printf("REG[0x5E:0x7F]: 0x%x ->", regval);
			rval = pmic_reg_write(p, 0x7f, 0x57);
			if (rval != 0) {
				printf("\n--> Fail to write workaround (0x5E:0x7F)=0x57\n");
				break;
			}
			rval = pmic_reg_read(p, 0x7f, &regval);
			if (rval != 0) {
				printf("--> Fail to read debug register(0x5E:0x7F) again\n");
				break;
			}
			printf("0x%x\n", regval);

			if (regval != 0x57) {
				printf("--> Fail to verify workaround.\n");
				break;
			}
			printf("BD71837 applied a workaround\n");

			/* recover normal i2c slave address */
			pmic_i2c_addr = addr_org;
			(void) pmic_reg_write(p, 0xfe, 0);
			(void) pmic_reg_read(p, BD718XX_REV, &regval);
			if (regval != 0xa0) {
				printf("--> Fail to recover normal i2c slave address.\n");
			}
		} else {
			printf("BD71837 don't need a workaround\n");
		}
	} while (0);

	/* make sure normal slave addr in case of break */
	pmic_i2c_addr = addr_org;
  }
#endif /* ICX_ERRATA_BD71837_REVA0 */

	/* Increase RESET key long push time from the default 10ms to 3s */
	pmic_reg_write(p, BD718XX_PWRONCONFIG1, 0x03);

	/* unlock the PMIC regs */
	pmic_reg_write(p, BD718XX_REGLOCK, 0x1);

	/* increase VDD_SOC to typical value 0.82v before first DRAM access */
	pmic_reg_write(p, BD718XX_BUCK1_VOLT_RUN, 0x0c);
	pmic_reg_write(p, BD718XX_BUCK1_VOLT_IDLE, 0x0c);
	pmic_reg_write(p, BD718XX_BUCK1_VOLT_SUSP, 0x0c);

	/* increase VDD_DRAM to 0.975v for 3Ghz DDR */
	pmic_reg_write(p, BD718XX_1ST_NODVS_BUCK_VOLT, 0x83);

#ifndef CONFIG_IMX8M_LPDDR4
	/* increase NVCC_DRAM_1V2 to 1.2v for DDR4 */
	pmic_reg_write(p, BD718XX_4TH_NODVS_BUCK_VOLT, 0x28);
#endif

	/* lock the PMIC regs */
	pmic_reg_write(p, BD718XX_REGLOCK, 0x11);

	return 0;
}
#endif
#endif

#ifdef CONFIG_SPL_POWER_CONTROL
#define BOARD_ID1_GPIO		IMX_GPIO_NR(4, 24)
#define BOARD_ID2_GPIO		IMX_GPIO_NR(4, 26)
#define BOARD_ID3_GPIO		IMX_GPIO_NR(1, 15)
#define IOEXP_XRST_GPIO		IMX_GPIO_NR(1,  5)
#define IOEXP_INT_GPIO		IMX_GPIO_NR(1, 13)
#define PCONT_AMP2_GPIO		IMX_GPIO_NR(2,  7)
#define PCONT_AMP_GPIO		IMX_GPIO_NR(2, 20)
#define PCONT_ON_GPIO		IMX_GPIO_NR(3,  0)
#define PCONT_STANDBY_GPIO	IMX_GPIO_NR(3,  1)
#define BD_REQ_GPIO		IMX_GPIO_NR(3,  7)
#define IFCON_RST_GPIO		IMX_GPIO_NR(3,  9)
#define ECSPI1_SCLK		IMX_GPIO_NR(5,  6)
#define ECSPI1_MISO		IMX_GPIO_NR(5,  7)
#define ECSPI1_MOSI		IMX_GPIO_NR(5,  8)
#define ECSPI1_SS0		IMX_GPIO_NR(5,  9)

#define PAD_CTRL_GPIO_DS1 MUX_PAD_CTRL(PAD_CTL_FSEL2 | PAD_CTL_DSE1)
#define PAD_CTRL_GPIO_DS6 MUX_PAD_CTRL(PAD_CTL_FSEL2 | PAD_CTL_DSE6)
#define PAD_CTRL_GPIO_PU_DS1 MUX_PAD_CTRL(PAD_CTL_PE | PAD_CTL_PUE | PAD_CTL_FSEL2 | PAD_CTL_DSE1)
#define PAD_CTRL_SPI_OUT MUX_PAD_CTRL(PAD_CTL_FSEL2 | PAD_CTL_DSE1)
#define PAD_CTRL_SPI_INP MUX_PAD_CTRL(PAD_CTL_PE | PAD_CTL_FSEL2 | PAD_CTL_DSE1)
#define PAD_CTRL_I2C MUX_MODE_SION | MUX_PAD_CTRL(PAD_CTL_FSEL2 | PAD_CTL_DSE1)
static iomux_v3_cfg_t const bb_pcont_pads[] = {
	IMX8MM_PAD_GPIO1_IO05_GPIO1_IO5    | PAD_CTRL_GPIO_PU_DS1,	/* A53_Exp_RST */
	IMX8MM_PAD_GPIO1_IO13_GPIO1_IO13   | PAD_CTRL_GPIO_DS1,		/* A53_Exp_INT */
	IMX8MM_PAD_SD2_WP_GPIO2_IO20       | PAD_CTRL_GPIO_DS6,		/* PCONT_AMP */
	IMX8MM_PAD_NAND_ALE_GPIO3_IO0      | PAD_CTRL_GPIO_DS6,		/* PCONT_ON */
	IMX8MM_PAD_NAND_DATA01_GPIO3_IO7   | PAD_CTRL_GPIO_DS6,		/* BD_REQ */
	IMX8MM_PAD_NAND_DATA03_GPIO3_IO9   | PAD_CTRL_GPIO_DS6,		/* IF_RESET */
	IMX8MM_PAD_ECSPI1_SCLK_GPIO5_IO6   | PAD_CTRL_SPI_OUT,		/* ECSPI1 SCLK */
	IMX8MM_PAD_ECSPI1_MOSI_GPIO5_IO7   | PAD_CTRL_SPI_OUT,		/* ECSPI1 MOSI */
	IMX8MM_PAD_ECSPI1_MISO_GPIO5_IO8   | PAD_CTRL_SPI_INP,		/* ECSPI1 MISO */
	IMX8MM_PAD_ECSPI1_SS0_GPIO5_IO9    | PAD_CTRL_SPI_OUT,		/* ECSPI1 SS0 */
};

static iomux_v3_cfg_t const sf_pcont_pads[] = {
	IMX8MM_PAD_GPIO1_IO05_GPIO1_IO5    | PAD_CTRL_GPIO_PU_DS1,	/* A53_Exp_RST */
	IMX8MM_PAD_GPIO1_IO13_GPIO1_IO13   | PAD_CTRL_GPIO_DS1,		/* A53_Exp_INT */
	IMX8MM_PAD_SD1_DATA5_GPIO2_IO7     | PAD_CTRL_GPIO_DS6,		/* PCONT_AMP2 */
	IMX8MM_PAD_SD2_WP_GPIO2_IO20       | PAD_CTRL_GPIO_DS6,		/* PCONT_AMP1 */
	IMX8MM_PAD_NAND_ALE_GPIO3_IO0      | PAD_CTRL_GPIO_DS6,		/* PCONT_ON */
	IMX8MM_PAD_NAND_CE0_B_GPIO3_IO1    | PAD_CTRL_GPIO_DS6,		/* PCONT_STANDBY */
	IMX8MM_PAD_NAND_DATA01_GPIO3_IO7   | PAD_CTRL_GPIO_DS6,		/* BD_REQ */
	IMX8MM_PAD_NAND_DATA03_GPIO3_IO9   | PAD_CTRL_GPIO_DS6,		/* IF_RESET */
	IMX8MM_PAD_ECSPI1_SCLK_GPIO5_IO6   | PAD_CTRL_SPI_OUT,		/* ECSPI1 SCLK */
	IMX8MM_PAD_ECSPI1_MOSI_GPIO5_IO7   | PAD_CTRL_SPI_OUT,		/* ECSPI1 MOSI */
	IMX8MM_PAD_ECSPI1_MISO_GPIO5_IO8   | PAD_CTRL_SPI_INP,		/* ECSPI1 MISO */
	IMX8MM_PAD_ECSPI1_SS0_GPIO5_IO9    | PAD_CTRL_SPI_OUT,		/* ECSPI1 SS0 */
};

static iomux_v3_cfg_t const bid_mid_pads[] = {
	IMX8MM_PAD_SAI2_TXFS_GPIO4_IO24    | PAD_CTRL_GPIO_DS1,		/* BOARD_ID1 */
	IMX8MM_PAD_SAI2_TXD0_GPIO4_IO26    | PAD_CTRL_GPIO_DS1,		/* BOARD_ID2 */
	IMX8MM_PAD_GPIO1_IO15_GPIO1_IO15   | PAD_CTRL_GPIO_DS1,		/* BOARD_ID3 */
};

struct i2c_pads_info i2c_pad_info4 = {
	.scl = {
		.i2c_mode = IMX8MM_PAD_I2C4_SCL_I2C4_SCL    | PAD_CTRL_I2C,
		.gpio_mode = IMX8MM_PAD_I2C4_SCL_GPIO5_IO20 | PAD_CTRL_GPIO_DS1,
		.gp = IMX_GPIO_NR(5, 20),
	},
	.sda = {
		.i2c_mode = IMX8MM_PAD_I2C4_SDA_I2C4_SDA    | PAD_CTRL_I2C,
		.gpio_mode = IMX8MM_PAD_I2C4_SDA_GPIO5_IO21 | PAD_CTRL_GPIO_DS1,
		.gp = IMX_GPIO_NR(5, 21),
	},
};

#define TCA64XX_BUS (3)		/* i2c4 */
#define TCA6416_ADR (0x20)
#define TCA6416_OUT_REG (0x02)
#define TCA6416_DIR_REG (0x06)
#define TCA6424_ADR (0x22)
#define TCA6424_OUT_REG (0x84)
#define TCA6424_DIR_REG (0x8C)
#define TCA64XX_I2C_PROBE_RETRY_CNT (3)

/* For BB: TCA6424ARGJR(24bit) */
/* Initialize direction */
static unsigned char tca6424_dir_init[] = { 0x00, 0x4b, 0xff };
/* Output all OFF */
static unsigned char tca6424_out_init[] = { 0x00, 0x00, 0x00 };
/* Stage-1 Set: PCONT1 + PCONT_AMP + PCONT_TS + PCONT_MIC */
static unsigned char tca6424_out_pwr1[] = { 0x27, 0x00, 0x00 };
/* Stage-2 Set: PCONT1 + PCONT_AMP + PCONT_TS + PCONT_MIC + ENET_RST */
static unsigned char tca6424_out_pwr2[] = { 0x27, 0x10, 0x00 };

/* For Set: TCA6416ARTWR(16bit) */
/* Initialize direction */
static unsigned char tca6416_dir_init[] = { 0x2e, 0xaf };
/* Output all OFF */
static unsigned char tca6416_out_init[] = { 0x00, 0x00 };
/* Stage-1 Set: PCONT1 */
static unsigned char tca6416_out_pwr1[] = { 0x01, 0x00 };

#define GPIO_OUT_LOW (0)
#define GPIO_OUT_HIGH (1)
#define GPIO_IN (2)

void ioexp_write(u8 adr, unsigned int reg, unsigned char *vals, int len)
{
	if (i2c_write(adr, reg, 1, vals, len))
		puts("i2c_write error\n");
}

static void spl_i2c_probe(int bus, u8 address)
{
	int i;

	if (i2c_set_bus_num(bus))
		puts("i2c_set_bus_num error\n");

	for (i = 0; i < TCA64XX_I2C_PROBE_RETRY_CNT; i++) {
		if (!i2c_probe(address))
			break;
	}

	/* Continue the power sequence even if an error occurs in I2C communication. */
	if (i == TCA64XX_I2C_PROBE_RETRY_CNT)
		puts("i2c_probe error\n");
}

static void spl_gpio_init(unsigned int gpio, const char *label, int state)
{
	gpio_request(gpio, label);
	switch (state) {
	case GPIO_OUT_LOW:
	case GPIO_OUT_HIGH:
		gpio_direction_output(gpio, state);
		break;
	case GPIO_IN:
		gpio_direction_input(gpio);
		break;
	default:
		break;
	}
}

void sf_pcont_setup(void)
{
	u8 adr = TCA6416_ADR;

	imx_iomux_v3_setup_multiple_pads(sf_pcont_pads, ARRAY_SIZE(sf_pcont_pads));
	if (setup_i2c(TCA64XX_BUS, CONFIG_SYS_I2C_SPEED, 0x7f, &i2c_pad_info4))
		puts("setup_i2c error\n");

	spl_gpio_init(IOEXP_XRST_GPIO, "A53_Exp_RST", GPIO_OUT_LOW);
	spl_gpio_init(IOEXP_INT_GPIO, "A53_Exp_INT", GPIO_IN);
	spl_gpio_init(IFCON_RST_GPIO, "IF_RESET", GPIO_OUT_HIGH);
	spl_gpio_init(PCONT_STANDBY_GPIO, "PCONT_STANDBY", GPIO_OUT_LOW);
	spl_gpio_init(PCONT_ON_GPIO, "PCONT_ON", GPIO_OUT_LOW);
	spl_gpio_init(PCONT_AMP_GPIO, "PCONT_AMP1", GPIO_OUT_LOW);
	spl_gpio_init(PCONT_AMP2_GPIO, "PCONT_AMP2", GPIO_OUT_LOW);
	spl_gpio_init(BD_REQ_GPIO, "BD_REQ", GPIO_OUT_LOW);
	spl_gpio_init(ECSPI1_SCLK, "ECSPI1_SCLK", GPIO_OUT_LOW);
	spl_gpio_init(ECSPI1_MOSI, "ECSPI1_MOSI", GPIO_OUT_LOW);
	spl_gpio_init(ECSPI1_MISO, "ECSPI1_MISO", GPIO_IN);
	spl_gpio_init(ECSPI1_SS0, "ECSPI1_SS0", GPIO_OUT_LOW);
	udelay(10000);
	gpio_set_value(IOEXP_XRST_GPIO, 1);
	gpio_set_value(IFCON_RST_GPIO, 0);

	spl_i2c_probe(TCA64XX_BUS, adr);

	puts("PCONT startup with TCA6416(SF Set)\n");
	ioexp_write(adr, TCA6416_OUT_REG, tca6416_out_init, 2);
	ioexp_write(adr, TCA6416_DIR_REG, tca6416_dir_init, 2);
	gpio_set_value(PCONT_ON_GPIO, 1);
	gpio_set_value(PCONT_AMP_GPIO, 1);
	udelay(10000);
	ioexp_write(adr, TCA6416_OUT_REG, tca6416_out_pwr1, 2);
	gpio_set_value(PCONT_AMP2_GPIO, 1);
	udelay(10000);
	gpio_set_value(BD_REQ_GPIO, 1);
}

void bb_pcont_setup(void)
{
	u8 adr = TCA6424_ADR;

	imx_iomux_v3_setup_multiple_pads(bb_pcont_pads, ARRAY_SIZE(bb_pcont_pads));
	if (setup_i2c(TCA64XX_BUS, CONFIG_SYS_I2C_SPEED, 0x7f, &i2c_pad_info4))
		puts("setup_i2c error\n");

	spl_gpio_init(IOEXP_XRST_GPIO, "A53_Exp_RST", GPIO_OUT_LOW);
	spl_gpio_init(IOEXP_INT_GPIO, "A53_Exp_INT", GPIO_IN);
	spl_gpio_init(IFCON_RST_GPIO, "IF_RESET", GPIO_OUT_HIGH);
	spl_gpio_init(PCONT_ON_GPIO, "PCONT_ON", GPIO_OUT_LOW);
	spl_gpio_init(PCONT_AMP_GPIO, "PCONT_AMP", GPIO_OUT_LOW);
	spl_gpio_init(BD_REQ_GPIO, "BD_REQ", GPIO_OUT_LOW);
	spl_gpio_init(ECSPI1_SCLK, "ECSPI1_SCLK", GPIO_OUT_LOW);
	spl_gpio_init(ECSPI1_MOSI, "ECSPI1_MOSI", GPIO_OUT_LOW);
	spl_gpio_init(ECSPI1_MISO, "ECSPI1_MISO", GPIO_IN);
	spl_gpio_init(ECSPI1_SS0, "ECSPI1_SS0", GPIO_OUT_LOW);
	udelay(10000);
	gpio_set_value(IOEXP_XRST_GPIO, 1);
	gpio_set_value(IFCON_RST_GPIO, 0);

	spl_i2c_probe(TCA64XX_BUS, adr);

	puts("PCONT startup with TCA6424(BB)\n");
	ioexp_write(adr, TCA6424_OUT_REG, tca6424_out_init, 3);
	ioexp_write(adr, TCA6424_DIR_REG, tca6424_dir_init, 3);
	gpio_set_value(PCONT_ON_GPIO, 1);
	udelay(100000);
	ioexp_write(adr, TCA6424_OUT_REG, tca6424_out_pwr1, 3);
	gpio_set_value(PCONT_AMP_GPIO, 1);
	udelay(10000);
	ioexp_write(adr, TCA6424_OUT_REG, tca6424_out_pwr2, 3);
	gpio_set_value(BD_REQ_GPIO, 1);
}

int pcont_setup(void)
{
	int bid;

	imx_iomux_v3_setup_multiple_pads(bid_mid_pads,
					 ARRAY_SIZE(bid_mid_pads));
	spl_gpio_init(BOARD_ID1_GPIO, "BOARDID1", GPIO_IN);
	spl_gpio_init(BOARD_ID2_GPIO, "BOARDID2", GPIO_IN);
	spl_gpio_init(BOARD_ID3_GPIO, "BOARDID3", GPIO_IN);
	bid = gpio_get_value(BOARD_ID1_GPIO)
	    | (gpio_get_value(BOARD_ID2_GPIO) << 1)
	    | (gpio_get_value(BOARD_ID3_GPIO) << 2);

	if (bid > 0)
		sf_pcont_setup();
	else
		bb_pcont_setup();

	return 0;
}
#endif

void spl_board_init(void)
{
#ifdef CONFIG_FSL_CAAM
	if (sec_init()) {
		printf("\nsec_init failed!\n");
	}
#endif
#ifndef CONFIG_SPL_USB_SDP_SUPPORT
	/* Serial download mode */
	if (is_usb_boot()) {
		puts("Back to ROM, SDP\n");
		restore_boot_params();
	}
#endif
	puts("Normal Boot\n");
}

#ifdef CONFIG_SPL_LOAD_FIT
int board_fit_config_name_match(const char *name)
{
	/* Just empty function now - can't decide what to choose */
	debug("%s: %s\n", __func__, name);

	return 0;
}
#endif

void lpddr4_print_info(void)
{
	int i;
	unsigned int ddr_info = 0;
	static const unsigned int regs[] = {5, 6, 7, 8};

	for (i = 0; i < ARRAY_SIZE(regs); i++) {
		ddr_info = lpddr4_mr_read(0xF, regs[i]);
		switch (i) {
		case 0:
			printf("DRAM INFO : Manufacturer ID = 0x%x", ddr_info);
			if (ddr_info == 0Xff)
				printf(", Micron");
			if (ddr_info == 0X06)
				printf(", SK hynix");
			if (ddr_info == 0X01)
				printf(", Samsung");
			if (ddr_info == 0X05)
				printf(", Nanya");
			printf("\n");
			break;
		case 1:
			printf("DRAM INFO : Revision ID1 = 0x%x\n", ddr_info);
			break;
		case 2:
			printf("DRAM INFO : Revision ID2 = 0x%x\n", ddr_info);
			break;
		case 3:
			printf("DRAM INFO : I/O Width and Density = 0x%x\n", ddr_info);
			break;
		default:
			break;
		}
	}
}

void board_init_f(ulong dummy)
{
	int ret;

	/* Clear the BSS. */
	memset(__bss_start, 0, __bss_end - __bss_start);

	arch_cpu_init();

	board_early_init_f();

	timer_init();

	preloader_console_init();

	ret = spl_init();
	if (ret) {
		debug("spl_init() failed: %d\n", ret);
		hang();
	}

	enable_tzc380();

	/* Adjust pmic voltage to 1.0V for 800M */
	setup_i2c(0, CONFIG_SYS_I2C_SPEED, 0x7f, &i2c_pad_info1);

	power_init_board();
#ifdef CONFIG_SPL_POWER_CONTROL
	pcont_setup();
#endif
	/* DDR initialization */
	spl_dram_init();

	lpddr4_print_info();

	board_init_r(NULL, 0);
}
