#!/bin/sh
# Copyright (c) 2016 Google, Inc
# Written by Simon Glass <sjg@chromium.org>
# Copyright 2023 Sony Corporation
#
# Check that the u-boot.cfg file provided does not introduce any new
# ad-hoc CONFIG options
#
# Use scripts/build-whitelist.sh to generate the list of current ad-hoc
# CONFIG options (those which are not in Kconfig).

# Usage
#    check-config.sh <path to u-boot.cfg> <path to whitelist file> <source dir>
#
# For example:
#   scripts/check-config.sh b/chromebook_link/u-boot.cfg kconfig_whitelist.txt .

set -e
set -u

PROG_NAME="${0##*/}"

usage() {
	echo "$PROG_NAME <path to u-boot.cfg> <path to whitelist file> <source dir>"
	exit 1
}

[ $# -ge 3 ] || usage

path="$1"
whitelist="$2"
srctree="$3"

# Temporary files
configs="${path}.configs"
suspects="${path}.suspects"
ok="${path}.ok"
new_adhoc="${path}.adhoc"

export LC_ALL=C
export LC_COLLATE=C

cat ${path} |sed -nr 's/^#define (CONFIG_[A-Za-z0-9_]*).*/\1/p' |sort |uniq \
	>${configs}

comm -23 ${configs} ${whitelist} > ${suspects}

cat `find ${srctree} -name "Kconfig*"` |sed -nr \
	-e 's/^[[:blank:]]*config *([A-Za-z0-9_]*).*$/CONFIG_\1/p' \
	-e 's/^[[:blank:]]*menuconfig ([A-Za-z0-9_]*).*$/CONFIG_\1/p' \
	|sort |uniq > ${ok}
comm -23 ${suspects} ${ok} >${new_adhoc}
if [ -s ${new_adhoc} ]; then
	# Ignore errors, we continue anyway. Get all dump outputs.
	set +e
	echo >&2 "Error: You must add new CONFIG options using Kconfig"
	echo >&2 "The following new ad-hoc CONFIG options were detected:"
	cat >&2 ${new_adhoc}
	echo >&2
	echo >&2 "Please add these via Kconfig instead. Find a suitable Kconfig"
	echo >&2 "file and add a 'config' or 'menuconfig' option."

	# Check running system

	on_build_server=n

	if [ -d /mnt/slave ]
	then
		# We are on build server.
		on_build_server=y
	fi

	if env | grep -q '^GIT_AUTHOR_NAME=TE-ro[a-zA-Z]*[0-9]\+-iz[a-zA-Z]\+integ'
	then
		# We are on build server.
		on_build_server=y
	fi

if [ "${on_build_server}" = "y" ]
then
	# Server build.
	# Report file system stats.
	echo >&2 "$0: BEGIN df."
	/bin/df >&2
	echo >&2 "$0: END df."

	# Report dmesg.
	echo >&2 "$0: BEGIN dmesg."
	/bin/dmesg | tail -n 20 >&2
	echo >&2 "$0: END dmesg."

	# Report stat.
	echo >&2 "$0: BEGIN stat."
	stat >&2 "${path}"
	stat >&2 "${srctree}"
	stat >&2 "${configs}"
	stat >&2 "${whitelist}"
	stat >&2 "${suspects}"
	stat >&2 "${ok}"
	echo >&2 "$0: END stat."

	# Report stat -L
	echo >&2 "$0: BEGIN stat -L."
	stat >&2 -L "${path}"
	stat >&2 -L "${srctree}"
	stat >&2 -L "${configs}"
	stat >&2 -L "${whitelist}"
	stat >&2 -L "${suspects}"
	stat >&2 -L "${ok}"
	echo >&2 "$0: END stat -L."

	# Report readlink -f
	echo >&2 "$0: BEGIN readlink -f."
	readlink >&2 -f "${path}"
	readlink >&2 -f "${srctree}"
	readlink >&2 -f "${configs}"
	readlink >&2 -f "${whitelist}"
	readlink >&2 -f "${suspects}"
	readlink >&2 -f "${ok}"
	echo >&2 "$0: END readlink -f."

	# Dump path
	echo >&2 "${path}: BEGIN Dump path file."
	cat >&2 "${path}"
	echo >&2 "${path}: END Dump path file."

	# Dump configs
	echo >&2 "${configs}: BEGIN Dump configs file."
	cat >&2 "${configs}"
	echo >&2 "${configs}: END Dump configs file."

	# Dump whitelist
	echo >&2 "${whitelist}: BEGIN Dump whitelist file."
	cat >&2 "${whitelist}"
	echo >&2 "${whitelist}: END Dump whitelist file."

	# Dump suspects
	echo >&2 "${suspects}: BEGIN Dump suspects file."
	cat >&2 "${suspects}"
	echo >&2 "${suspects}: END Dump suspects file."

	# Dump ok
	echo >&2 "${ok}: BEGIN Dump ok file."
	cat >&2 "${ok}"
	echo >&2 "${ok}: END Dump ok file."
else
	# Local build.
	time_stamp=`date +%y%m%d%H%M%S-$$`
	build_dir=`dirname "${configs}"`
	echo >&2 "Backup related files."

	copy_to="${path}.${time_stamp}"
	echo >&2 "${path}: Copy to ${path}"
	cp -p "${path}" "${copy_to}"

	copy_to="${configs}.${time_stamp}"
	echo >&2 "${configs}: Copy to ${copy_to}"
	cp -p "${configs}" "${copy_to}"

	whitelist_name=`basename "${whitelist}"`
	copy_to="${build_dir}/${whitelist_name}.${time_stamp}"
	echo >&2 "${whitelist}: Copy to ${copy_to}"
	cp -p "${whitelist}" "${copy_to}"

	copy_to="${suspects}.${time_stamp}"
	echo >&2 "${suspects}: Copy to ${copy_to}"
	cp -p "${suspects}" "${copy_to}"

	copy_to="${ok}.${time_stamp}"
	echo >&2 "${ok}: Copy to ${copy_to}"
	cp -p "${ok}" "${copy_to}"
fi
	# Don't delete the temporary files in case they are useful
	exit 1
else
	rm ${suspects} ${ok} ${new_adhoc}
fi
