/*
 * Analog Devices ADV7511 HDMI transmitter driver
 *
 * Copyright 2012 Analog Devices Inc.
 * Copyright 2020,2021 Sony Home Entertainment & Sound Products Inc.
 * Copyright 2021 Sony Corporation
 *
 * Licensed under the GPL-2.
 */

#include <linux/device.h>
#include <linux/gpio/consumer.h>
#include <linux/module.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/of_graph.h>
#include <linux/slab.h>
#include <linux/sysfs.h>

#include <drm/drmP.h>
#include <drm/drm_atomic.h>
#include <drm/drm_atomic_helper.h>
#include <drm/drm_edid.h>

#include "adv7511.h"
#include "hd_define.h"
#include "hd_edid_ram.h"

static struct adv7511 *g_adv7511;

/* ADI recommended values for proper operation. */
static const struct reg_sequence adv7511_fixed_registers[] = {
	{ 0x98, 0x03 },
	{ 0x9a, 0xe0 },
	{ 0x9c, 0x30 },
	{ 0x9d, 0x61 },
	{ 0xa2, 0xa4 },
	{ 0xa3, 0xa4 },
	{ 0xe0, 0xd0 },
	{ 0xf9, 0x00 },
	{ 0x55, 0x02 },
};

static const uint8_t adv7511_fixed_edid_first_block[] = {
	0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00,
	0x4d, 0xd9, 0xd1, 0x05, 0x01, 0x01, 0x01, 0x01,
	0x31, 0x1e, 0x01, 0x03, 0x80, 0xba, 0x69, 0x78,
	0x0a, 0x0d, 0xc9, 0xa0, 0x57, 0x47, 0x98, 0x27,
	0x12, 0x48, 0x4c, 0x20, 0x00, 0x00, 0x01, 0x01,
	0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
	0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0xd6, 0x09,
	0x80, 0xa0, 0x20, 0xe0, 0x2d, 0x10, 0x08, 0x60,
	0x22, 0x00, 0x12, 0x8e, 0x21, 0x08, 0x08, 0x18,
	0x00, 0x00, 0x00, 0xfe, 0x00, 0x0a, 0x20, 0x20,
	0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
	0x20, 0x20, 0x00, 0x00, 0x00, 0xfc, 0x00, 0x53,
	0x4f, 0x4e, 0x59, 0x20, 0x54, 0x56, 0x20, 0x20,
	0x2a, 0x30, 0x33, 0x0a, 0x00, 0x00, 0x00, 0xfd,
	0x00, 0x30, 0x3e, 0x0e, 0x46, 0x0f, 0x00, 0x0a,
	0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x01, 0x2f,
};

static const uint8_t adv7511_fixed_edid_second_block[] = {
	0x23, 0x09, 0x07, 0x07,		// Audio Data Block
	0x83, 0x5f, 0x00, 0x00,		// Speaker Allocation Data Block
	0x68, 0x03, 0x0c, 0x00, 0x10, 
	0x00, 0xb8, 0x3c, 0x0f,		// VSDB(HDMI LLC)
	0xe2, 0x00, 0xfb,		// Ext Video Capability Data Block
};

/* Sorted in ascending order of priority */
static const uint8_t adv7511_support_vic[] = {
	0x01, 0x11, 0x12, 0x02, 0x03, 0x04, 0x10
};

/* -----------------------------------------------------------------------------
 * Register access
 */

static const uint8_t adv7511_register_defaults[] = {
	0x12, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 00 */
	0x00, 0x00, 0x01, 0x0e, 0xbc, 0x18, 0x01, 0x13,
	0x25, 0x37, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 10 */
	0x46, 0x62, 0x04, 0xa8, 0x00, 0x00, 0x1c, 0x84,
	0x1c, 0xbf, 0x04, 0xa8, 0x1e, 0x70, 0x02, 0x1e, /* 20 */
	0x00, 0x00, 0x04, 0xa8, 0x08, 0x12, 0x1b, 0xac,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 30 */
	0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0xb0,
	0x00, 0x50, 0x90, 0x7e, 0x79, 0x70, 0x00, 0x00, /* 40 */
	0x00, 0xa8, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x02, 0x0d, 0x00, 0x00, 0x00, 0x00, /* 50 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 60 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x01, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 70 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 80 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, /* 90 */
	0x0b, 0x02, 0x00, 0x18, 0x5a, 0x60, 0x00, 0x00,
	0x00, 0x00, 0x80, 0x80, 0x08, 0x04, 0x00, 0x00, /* a0 */
	0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x40, 0x14,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* b0 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* c0 */
	0x00, 0x03, 0x00, 0x00, 0x02, 0x00, 0x01, 0x04,
	0x30, 0xff, 0x80, 0x80, 0x80, 0x00, 0x00, 0x00, /* d0 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x01,
	0x80, 0x75, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, /* e0 */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x75, 0x11, 0x00, /* f0 */
	0x00, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
};

static bool adv7511_register_volatile(struct device *dev, unsigned int reg)
{
	switch (reg) {
	case ADV7511_REG_CHIP_REVISION:
	case ADV7511_REG_SPDIF_FREQ:
	case ADV7511_REG_CTS_AUTOMATIC1:
	case ADV7511_REG_CTS_AUTOMATIC2:
	case ADV7511_REG_VIC_DETECTED:
	case ADV7511_REG_VIC_SEND:
	case ADV7511_REG_AUX_VIC_DETECTED:
	case ADV7511_REG_STATUS:
	case ADV7511_REG_GC(1):
	case ADV7511_REG_INT(0):
	case ADV7511_REG_INT(1):
	case ADV7511_REG_PLL_STATUS:
	case ADV7511_REG_AN(0):
	case ADV7511_REG_AN(1):
	case ADV7511_REG_AN(2):
	case ADV7511_REG_AN(3):
	case ADV7511_REG_AN(4):
	case ADV7511_REG_AN(5):
	case ADV7511_REG_AN(6):
	case ADV7511_REG_AN(7):
	case ADV7511_REG_HDCP_STATUS:
	case ADV7511_REG_BCAPS:
	case ADV7511_REG_BKSV(0):
	case ADV7511_REG_BKSV(1):
	case ADV7511_REG_BKSV(2):
	case ADV7511_REG_BKSV(3):
	case ADV7511_REG_BKSV(4):
	case ADV7511_REG_DDC_STATUS:
	case ADV7511_REG_EDID_READ_CTRL:
	case ADV7511_REG_BSTATUS(0):
	case ADV7511_REG_BSTATUS(1):
	case ADV7511_REG_CHIP_ID_HIGH:
	case ADV7511_REG_CHIP_ID_LOW:
		return true;
	}

	return false;
}

static const struct regmap_config adv7511_regmap_config = {
	.reg_bits = 8,
	.val_bits = 8,

	.max_register = 0xff,
	.cache_type = REGCACHE_RBTREE,
	.reg_defaults_raw = adv7511_register_defaults,
	.num_reg_defaults_raw = ARRAY_SIZE(adv7511_register_defaults),

	.volatile_reg = adv7511_register_volatile,
};


/* ADV75xx IRQ control */

static void
adv7511_enable_irq(struct adv7511 *adv)
{
        if (!adv || !adv->i2c_main || !adv->i2c_main->irq) {
                pr_err("%s: error: NULL\n", __func__); /* FATAL */
                return;
        }

	if (!adv->irq_en)
		enable_irq(adv->i2c_main->irq);
	else
		pr_info("%s: IRQ has already been enabled\n", __func__);

	adv->irq_en = true;
}

static void
adv7511_disable_irq(struct adv7511 *adv)
{
        if (!adv || !adv->i2c_main || !adv->i2c_main->irq) {
                pr_err("%s: error: NULL\n", __func__); /* FATAL */
                return;
        }

	if (adv->irq_en)
		disable_irq(adv->i2c_main->irq);
	else
		pr_info("%s: IRQ has already been disabled\n", __func__);

	adv->irq_en = false;
}

/* -----------------------------------------------------------------------------
 * pcont_1 power control
 *
 * Power is supplied to avdd, dvdd, pvdd, a2vdd, v3p3, v1p8
 * by enabling pcont1.
 */

/* adv7511_is_enabled_pcont1_regulator
 * @brief Check if other consumers grab PCONT_1 handle or not
 * @return
 *   true; PCONT_1 is enabled (On EVK, no need to care)
 *  false; PCONT_1 is disabled
 */
static bool
adv7511_is_enabled_pcont1_regulator(struct adv7511 *adv)
{
       if (adv->power_pcont1 == NULL)
               return true;

       return regulator_is_enabled(adv->power_pcont1) > 0;
}

static void
adv7511_enable_pcont1_regulator(struct adv7511 *adv)
{
	struct device *dev = &adv->i2c_main->dev;
	int ret;

	if (adv->power_pcont1 == NULL)
		return;

	if (!adv->pcont1_en) {
		dev_info(dev, "pcont1 power-supply enable\n");
		ret = regulator_enable(adv->power_pcont1);
		WARN_ON(ret);
		adv->pcont1_en = true;
	} else
		dev_warn(dev, "pcont1 power-supply is already enabled\n");
}

static void
adv7511_disable_pcont1_regulator(struct adv7511 *adv)
{
	struct device *dev = &adv->i2c_main->dev;

	if (adv->power_pcont1 == NULL)
		return;

	if (adv->pcont1_en) {
		dev_info(dev, "pcont1 power-supply disable\n");
		regulator_disable(adv->power_pcont1);
		adv->pcont1_en = false;

		adv7511_clear_raw_edid_buf(adv);
	} else
		dev_warn(dev, "pcont1 power-supply is already disabled\n");
}

static void
adv7511_init_pcont1_regulator(struct adv7511 *adv)
{
	struct device *dev = &adv->i2c_main->dev;

	/* Check Device Tree at first */
	if (!of_find_property(dev->of_node, "pcont1-supply", NULL)) {
		adv->power_pcont1 = NULL;
		return; /* In case of EVK, intentional */
	}

	/* Try to get PCONT_1 regulator handle */
	adv->power_pcont1 = devm_regulator_get(dev, "pcont1");

	/* If pcont1 does not exist, DummyRegulator is controlled */
	if (IS_ERR(adv->power_pcont1)) {
		adv->power_pcont1 = NULL;
		dev_warn(dev, "no power-supply (pcont1)\n");
		return;
	}
}

/* I2C bus check */
/* It's needed after turning on PCONT_1 regulator. */
static int
adv7511_check_i2c_bus(struct adv7511 *adv7511)
{
	unsigned int val;
	int ret, i;

	if (!adv7511 || !adv7511->regmap) {
		pr_err("%s: error: NULL\n", __func__);
		return -1;
	}

	/* According to adv7533 hardware user's guide,
	 * we have to wait 200ms for serial communication
	 * with adv75xx when initially powered up. */
	msleep(50);

	/* It's better to treat the first I2C communication carefully.
	 * So, introduced retry loop here. */
	for (i = 0; i < 10; i++) {
		ret = regmap_read(adv7511->regmap,
				ADV7511_REG_CHIP_REVISION, &val);
		if (!ret) /* OK */
			break;
		pr_err("%s: I2C read error (%d times): ret=%d\n",
			__func__, i + 1, ret);
		/* wait for 50ms before retry */
		msleep(50);
	}
	if (i < 10) {
		pr_info("%s: I2C bus line is ready\n", __func__);
		ret = 0;
	} else {
		pr_err("%s: I2C read retry over\n", __func__);
		ret = -1;
	}

	return ret;
}

static void
adv7511_send_disconnected_state_to_drm(struct adv7511 *adv7511)
{
	/* If HPD state is still 'On', send fake state 'Off' to DRM */
	/* in order to avoid state confliction after that.	    */
	if (adv7511->connector.status == connector_status_connected) {
		dev_info(&adv7511->i2c_main->dev,
			"[PCONT_1 OFF] notify hotplug event:%d\n",
			connector_status_disconnected);
		adv7511->connector.status = connector_status_disconnected;
		drm_kms_helper_hotplug_event(adv7511->connector.dev);
		msleep(100); /* wait DRM state is stabled */
	}
}

/* called by imx-drm core */
void adv7511_regulator_on(void)
{
	if (!g_adv7511) {
		pr_err("%s: the instance has been freed\n", __func__);
		return;
	}

	adv7511_enable_pcont1_regulator(g_adv7511);

	adv7511_check_i2c_bus(g_adv7511);

	adv7511_enable_irq(g_adv7511);
}

void adv7511_regulator_off(void)
{
	if (!g_adv7511) {
		pr_err("%s: the instance has been freed\n", __func__);
		return;
	}

	adv7511_disable_irq(g_adv7511);
	flush_scheduled_work(); /* wait all queued work issued by IRQ */

	adv7511_send_disconnected_state_to_drm(g_adv7511); /* disconnected */

	adv7511_disable_pcont1_regulator(g_adv7511);
}

/* -----------------------------------------------------------------------------
 * Hardware configuration (added by Sony)
 */

static void adv7511_update_tmds_power_state(struct adv7511 *adv)
{
	u8 is_tmds_signal_on = adv->is_tmds_signal_on;

	/* Setting power state of            */
	/* HDMI Channel 0-2 and Clock Driver */
	/*   ADV7511_REG_HDMI_POWER [5:2]    */
	if (is_tmds_signal_on) {
		if (adv->is_hdmi) 
			regmap_update_bits(adv->regmap, 0xAF, 0x02, 0x02);
		else
			regmap_update_bits(adv->regmap, 0xAF, 0x02, 0x00);

		regmap_update_bits(adv->regmap,
				   ADV7511_REG_HDMI_POWER,
				   ADV7511_REG_TMDS_PD_MASK,
				   0); /* power up all CHs and clk */
		if (adv->replace_edid && !adv->is_hdcp_authenticated) {
			pr_info("%s: Start HDCP Authenticate\n", __func__);
			regmap_update_bits(adv->regmap, ADV7511_REG_MAIN_0XD5,
				ADV7511_REG_BLACK_IMAGE_MASK, ADV7511_REG_BLACK_IMAGE_ENABLED);
			
			/* Before starting HDCP authentication,
			 * It is necessary to add a delay after sending TMDS. 
			 * As an actual value of the past model, 
			 * add a 300ms delay. */
			msleep(300);
			regmap_update_bits(adv->regmap, ADV7511_REG_HDCP_HDMI_CFG,
				ADV7511_HDCP_MASK, ADV7511_HDCP_ENABLE);
		}
	}
	else {
		regmap_update_bits(adv->regmap,
				   ADV7511_REG_MAIN_0XD5,
				   ADV7511_REG_BLACK_IMAGE_MASK,
				   ADV7511_REG_BLACK_IMAGE_ENABLED);
		/* 16ms: wait at least 1 frame @ 60fps */
		usleep_range(16667, 20000);
		regmap_update_bits(adv->regmap,
				   ADV7511_REG_HDMI_POWER,
				   ADV7511_REG_TMDS_PD_MASK,
				   ADV7511_REG_TMDS_PD_ALL);
		regmap_update_bits(adv->regmap,
				   ADV7511_REG_MAIN_0XD5,
				   ADV7511_REG_BLACK_IMAGE_MASK,
				   ADV7511_REG_BLACK_IMAGE_DISABLED);
		
		if (adv->replace_edid && adv->is_hdcp_authenticated) {
			pr_info("%s: Stop HDCP Authenticate\n", __func__);
			regmap_update_bits(adv->regmap, ADV7511_REG_HDCP_HDMI_CFG,
				ADV7511_HDCP_MASK, ADV7511_HDCP_DISABLE);
			adv->is_hdcp_authenticated = false;
		}
	}
}

static int adv7511_read_tmds_power_state(struct adv7511 *adv)
{
	unsigned int val;
	int ret;

	if (!adv7511_is_enabled_pcont1_regulator(adv))
		return -1;

	ret = regmap_read(adv->regmap, ADV7511_REG_HDMI_POWER, &val);
	if (ret >= 0)
		return val & 0x3d ? 0 : 1;

	return -1;
}

static void adv7511_update_black_image(struct adv7511 *adv)
{
	u8 is_black_image_on = adv->is_black_image_on;

	regmap_update_bits(adv->regmap,
			   ADV7511_REG_MAIN_0XD5,
			   ADV7511_REG_BLACK_IMAGE_MASK,
			   is_black_image_on ?
			     ADV7511_REG_BLACK_IMAGE_ENABLED :
			     ADV7511_REG_BLACK_IMAGE_DISABLED);
}

static int adv7511_read_black_image(struct adv7511 *adv)
{
	unsigned int val;
	int ret;

	if (!adv7511_is_enabled_pcont1_regulator(adv))
		return -1;

	ret = regmap_read(adv->regmap, ADV7511_REG_MAIN_0XD5,
			  &val);
	if (ret >= 0)
		return val & ADV7511_REG_BLACK_IMAGE_MASK ? 1 : 0;

	return -1;
}

static void adv7511_update_picture_aspect(struct adv7511 *adv)
{
	u8 picture_aspect = adv->picture_aspect;

	if (picture_aspect > 0x3)
		return; /* Invalid value */

	regmap_update_bits(adv->regmap, ADV7511_REG_MAIN_0X17,
			   ADV7511_REG_ASPECT_RATIO_MASK,
			   picture_aspect == 1 ? /* 1; "4:3" */
			     ADV7511_REG_ASPECT_RATIO_4_3 :
			     ADV7511_REG_ASPECT_RATIO_16_9);
	regmap_update_bits(adv->regmap, ADV7511_REG_AVI_INFOFRAME(1),
		0x3f, (picture_aspect << 4) | 0x8);
}

static int adv7511_read_picture_aspect(struct adv7511 *adv)
{
	unsigned int val;
	int ret;

	if (!adv7511_is_enabled_pcont1_regulator(adv))
		return -1;

	ret = regmap_read(adv->regmap, ADV7511_REG_AVI_INFOFRAME(1), &val);
	if (ret >= 0)
		return (int)(val & 0x30) >> 4;

	return -1;
}

static int adv7511_read_vic_to_rx(struct adv7511 *adv)
{
	unsigned int val;
	int ret;

	if (!adv7511_is_enabled_pcont1_regulator(adv))
		return -1;

	ret = regmap_read(adv->regmap, ADV7511_REG_VIC_SEND, &val);
	if (ret >= 0)
		return (int)(val & 0x3f); /* Upper 2-bits are reserved */

	return -1;
}

static void adv7511_update_vhsync_polarity(struct adv7511 *adv)
{
	regmap_update_bits(adv->regmap, ADV7511_REG_MAIN_0X17,
			   ADV7511_REG_VSYNC_POLARITY_MASK |
			     ADV7511_REG_HSYNC_POLARITY_MASK,
			   (adv->mode_vsync_polarity << 6) |
			     (adv->mode_hsync_polarity << 5));
}

static void adv7511_update_sysfs_settings(struct adv7511 *adv)
{
	mutex_lock(&adv->sysfs_lock);

	adv7511_update_tmds_power_state(adv);
	adv7511_update_black_image(adv);
	adv7511_update_picture_aspect(adv);

	mutex_unlock(&adv->sysfs_lock);
}

/* -----------------------------------------------------------------------------
 * sysfs implementation
 */

static ssize_t
adv7511_sysfs_black_image_show(struct device *dev,
				struct device_attribute *attr, char *buf)
{
	struct adv7511 *adv = dev_get_drvdata(dev);
	int val;
	int ret;

	ret = mutex_lock_interruptible(&adv->sysfs_lock);
	if (ret)
		return ret;

	val = adv7511_read_black_image(adv);

	mutex_unlock(&adv->sysfs_lock);

	return snprintf(buf, 8, "%d\n", val);
}

static ssize_t
adv7511_sysfs_black_image_store(struct device *dev,
				struct device_attribute *attr,
				const char *buf, size_t count)
{
	struct adv7511 *adv = dev_get_drvdata(dev);
	bool val;
	int ret;

	ret = mutex_lock_interruptible(&adv->sysfs_lock);
	if (ret)
		return ret;

	pr_debug("%s: %s\n", __func__, buf);

	if (kstrtobool(buf, &val) == 0)
		adv->is_black_image_on = val;

	pr_info("%s: %d\n", __func__, adv->is_black_image_on);
	adv7511_update_black_image(adv);

	mutex_unlock(&adv->sysfs_lock);

	return count;
}

static DEVICE_ATTR(black_image, 0644, \
	adv7511_sysfs_black_image_show, \
	adv7511_sysfs_black_image_store);

static ssize_t
adv7511_sysfs_tmds_signal_show(struct device *dev,
				struct device_attribute *attr, char *buf)
{
	struct adv7511 *adv = dev_get_drvdata(dev);
	int val;
	int ret;

	ret = mutex_lock_interruptible(&adv->sysfs_lock);
	if (ret)
		return ret;

	val = adv7511_read_tmds_power_state(adv);

	mutex_unlock(&adv->sysfs_lock);

	return snprintf(buf, 8, "%d\n", val);
}

static ssize_t
adv7511_sysfs_tmds_signal_store(struct device *dev,
				struct device_attribute *attr,
				const char *buf, size_t count)
{
	struct adv7511 *adv = dev_get_drvdata(dev);
	bool val;
	int ret;

	ret = mutex_lock_interruptible(&adv->sysfs_lock);
	if (ret)
		return ret;

	pr_debug("%s: %s\n", __func__, buf);

	if (kstrtobool(buf, &val) == 0)
		adv->is_tmds_signal_on = val;

	pr_info("%s: %d\n", __func__, adv->is_tmds_signal_on);
	adv7511_update_tmds_power_state(adv);

	mutex_unlock(&adv->sysfs_lock);

	return count;
}

static DEVICE_ATTR(tmds_signal, 0644, \
	adv7511_sysfs_tmds_signal_show, \
	adv7511_sysfs_tmds_signal_store);

static ssize_t
adv7511_sysfs_picture_aspect_show(struct device *dev,
				struct device_attribute *attr, char *buf)
{
	struct adv7511 *adv = dev_get_drvdata(dev);
	static const char * const aspect_str[] = {
		"no data", "4:3", "16:9", "none"
	};
	const char *ret_str;
	int val;
	int ret;

	ret = mutex_lock_interruptible(&adv->sysfs_lock);
	if (ret)
		return ret;

	val = adv7511_read_picture_aspect(adv);

	mutex_unlock(&adv->sysfs_lock);

	ret_str = val < 0 ? "-1" : aspect_str[val & 0x3];
	pr_debug("%s: %s\n", __func__, ret_str);

	return snprintf(buf, 16, "%s\n", ret_str);
}

static ssize_t
adv7511_sysfs_picture_aspect_store(struct device *dev,
				struct device_attribute *attr,
				const char *buf, size_t count)
{
	struct adv7511 *adv = dev_get_drvdata(dev);
	u8 val;
	int ret;

	ret = mutex_lock_interruptible(&adv->sysfs_lock);
	if (ret)
		return ret;

	pr_debug("%s: %s\n", __func__, buf);

	if (kstrtou8(buf, 10, &val) == 0)
		adv->picture_aspect = val & 0x3;

	pr_info("%s: %u\n", __func__, adv->picture_aspect);
	adv7511_update_picture_aspect(adv);

	mutex_unlock(&adv->sysfs_lock);

	return count;
}

static DEVICE_ATTR(picture_aspect, 0644, \
	adv7511_sysfs_picture_aspect_show, \
	adv7511_sysfs_picture_aspect_store);

static ssize_t
adv7511_sysfs_vic_to_rx_show(struct device *dev,
				struct device_attribute *attr, char *buf)
{
	struct adv7511 *adv = dev_get_drvdata(dev);
	int val;
	int ret;

	ret = mutex_lock_interruptible(&adv->sysfs_lock);
	if (ret)
		return ret;

	val = adv7511_read_vic_to_rx(adv);

	mutex_unlock(&adv->sysfs_lock);

	return snprintf(buf, 8, "%d\n", val);
}

static DEVICE_ATTR(vic_to_rx, 0444, \
	adv7511_sysfs_vic_to_rx_show, NULL);

static ssize_t
adv7511_sysfs_raw_edid_show(struct device *dev,
				struct device_attribute *attr, char *buf)
{
	struct adv7511 *adv = dev_get_drvdata(dev);
	int ret;

	ret = mutex_lock_interruptible(&adv->sysfs_lock);
	if (ret)
		return ret;

	if (adv->edid_size > 0 &&
	    adv->edid_size <= ADV7511_EDID_MAX_LENGTH &&
		adv->edid_size % ADV7511_EDID_LENGTH == 0) {
		ret = adv->edid_size;
		memcpy(buf, adv->raw_edid, adv->edid_size);
	}
	else
		ret = -1;

	mutex_unlock(&adv->sysfs_lock);

	pr_debug("%s: edid_size=%d\n", __func__, adv->edid_size);

	return ret;
}

static DEVICE_ATTR(raw_edid, 0444, \
	adv7511_sysfs_raw_edid_show, NULL);

static int adv7511_read_hdcp_key(struct adv7511 *adv, char *hdcp_key)
{
	int ret;
	int val;

	if (!adv7511_is_enabled_pcont1_regulator(adv))
		return -1;
	
	ret = regmap_read(adv->regmap, 0xBA, &val);
	if (ret < 0)
		return -1;

	/* AKSV Config */
	regmap_update_bits(adv->regmap, 0xBA, 0x04, 0x04);

	ret = regmap_bulk_read(adv->regmap, 0xB0, hdcp_key, 5);

	regmap_update_bits(adv->regmap, 0xBA, 0x04, val & 0x04);

	return ret;
}

static ssize_t
adv7511_sysfs_hdcp_key_show(struct device *dev,
				struct device_attribute *attr, char *buf)
{
	struct adv7511 *adv = dev_get_drvdata(dev);
	char hdcp_key[5];
	int ret;
	
	ret = mutex_lock_interruptible(&adv->sysfs_lock);
	if (ret)
		return ret;

	ret = adv7511_read_hdcp_key(adv, hdcp_key);

	mutex_unlock(&adv->sysfs_lock);

	if (ret < 0)
		return ret;

	return snprintf(buf, 16, "%02x %02x %02x %02x %02x\n",
		hdcp_key[0], hdcp_key[1], hdcp_key[2],
		hdcp_key[3], hdcp_key[4]);
}

static DEVICE_ATTR(hdcp_key, 0444, \
	adv7511_sysfs_hdcp_key_show, NULL);

static struct attribute *adv7511_sysfs_entries[] = {
	&dev_attr_tmds_signal.attr,
	&dev_attr_black_image.attr,
	&dev_attr_picture_aspect.attr,
	&dev_attr_vic_to_rx.attr,
	&dev_attr_raw_edid.attr,
	&dev_attr_hdcp_key.attr,
	NULL,
};

static const struct attribute_group adv7511_sysfs_grp = {
	.name = NULL, /* put in device directory */
	.attrs = adv7511_sysfs_entries,
};

static void adv7511_sysfs_register(struct adv7511 *adv)
{
	int err;

	adv->sysfs_dev = root_device_register("adv7511");
	dev_set_drvdata(adv->sysfs_dev, adv);
	err = sysfs_create_group(&adv->sysfs_dev->kobj, &adv7511_sysfs_grp);
	if (err)
		dev_info(&adv->i2c_main->dev,
			"sysfs_create_group failed\n");
}

static void adv7511_sysfs_unregister(struct adv7511 *adv)
{
	if (adv->sysfs_dev)
		root_device_unregister(adv->sysfs_dev);

	adv->sysfs_dev = NULL;
}

/* -----------------------------------------------------------------------------
 * Hardware configuration
 */

static void adv7511_set_colormap(struct adv7511 *adv7511, bool enable,
				 const uint16_t *coeff,
				 unsigned int scaling_factor)
{
	unsigned int i;

	regmap_update_bits(adv7511->regmap, ADV7511_REG_CSC_UPPER(1),
			   ADV7511_CSC_UPDATE_MODE, ADV7511_CSC_UPDATE_MODE);

	if (enable) {
		for (i = 0; i < 12; ++i) {
			regmap_update_bits(adv7511->regmap,
					   ADV7511_REG_CSC_UPPER(i),
					   0x1f, coeff[i] >> 8);
			regmap_write(adv7511->regmap,
				     ADV7511_REG_CSC_LOWER(i),
				     coeff[i] & 0xff);
		}
	}

	if (enable)
		regmap_update_bits(adv7511->regmap, ADV7511_REG_CSC_UPPER(0),
				   0xe0, 0x80 | (scaling_factor << 5));
	else
		regmap_update_bits(adv7511->regmap, ADV7511_REG_CSC_UPPER(0),
				   0x80, 0x00);

	regmap_update_bits(adv7511->regmap, ADV7511_REG_CSC_UPPER(1),
			   ADV7511_CSC_UPDATE_MODE, 0);
}

static int adv7511_packet_enable(struct adv7511 *adv7511, unsigned int packet)
{
	if (packet & 0xff)
		regmap_update_bits(adv7511->regmap, ADV7511_REG_PACKET_ENABLE0,
				   packet, 0xff);

	if (packet & 0xff00) {
		packet >>= 8;
		regmap_update_bits(adv7511->regmap, ADV7511_REG_PACKET_ENABLE1,
				   packet, 0xff);
	}

	return 0;
}

static int adv7511_packet_disable(struct adv7511 *adv7511, unsigned int packet)
{
	if (packet & 0xff)
		regmap_update_bits(adv7511->regmap, ADV7511_REG_PACKET_ENABLE0,
				   packet, 0x00);

	if (packet & 0xff00) {
		packet >>= 8;
		regmap_update_bits(adv7511->regmap, ADV7511_REG_PACKET_ENABLE1,
				   packet, 0x00);
	}

	return 0;
}

/* Coefficients for adv7511 color space conversion */
static const uint16_t adv7511_csc_ycbcr_to_rgb[] = {
	0x0734, 0x04ad, 0x0000, 0x1c1b,
	0x1ddc, 0x04ad, 0x1f24, 0x0135,
	0x0000, 0x04ad, 0x087c, 0x1b77,
};

static void adv7511_set_config_csc(struct adv7511 *adv7511,
				   struct drm_connector *connector,
				   bool rgb)
{
	struct adv7511_video_config config;
	bool output_format_422, output_format_ycbcr;
	unsigned int mode;
	uint8_t infoframe[17];

	if (adv7511->edid)
		config.hdmi_mode = drm_detect_hdmi_monitor(adv7511->edid);
	else
		config.hdmi_mode = false;

	hdmi_avi_infoframe_init(&config.avi_infoframe);

	config.avi_infoframe.scan_mode = HDMI_SCAN_MODE_UNDERSCAN;

	if (rgb) {
		config.csc_enable = false;
		config.avi_infoframe.colorspace = HDMI_COLORSPACE_RGB;
	} else {
		config.csc_scaling_factor = ADV7511_CSC_SCALING_4;
		config.csc_coefficents = adv7511_csc_ycbcr_to_rgb;

		if ((connector->display_info.color_formats &
		     DRM_COLOR_FORMAT_YCRCB422) &&
		    config.hdmi_mode) {
			config.csc_enable = false;
			config.avi_infoframe.colorspace =
				HDMI_COLORSPACE_YUV422;
		} else {
			config.csc_enable = true;
			config.avi_infoframe.colorspace = HDMI_COLORSPACE_RGB;
		}
	}

	if (config.hdmi_mode) {
		mode = ADV7511_HDMI_CFG_MODE_HDMI;

		switch (config.avi_infoframe.colorspace) {
		case HDMI_COLORSPACE_YUV444:
			output_format_422 = false;
			output_format_ycbcr = true;
			break;
		case HDMI_COLORSPACE_YUV422:
			output_format_422 = true;
			output_format_ycbcr = true;
			break;
		default:
			output_format_422 = false;
			output_format_ycbcr = false;
			break;
		}
	} else {
		mode = ADV7511_HDMI_CFG_MODE_DVI;
		output_format_422 = false;
		output_format_ycbcr = false;
	}

	adv7511_packet_disable(adv7511, ADV7511_PACKET_ENABLE_AVI_INFOFRAME);

	adv7511_set_colormap(adv7511, config.csc_enable,
			     config.csc_coefficents,
			     config.csc_scaling_factor);

	regmap_update_bits(adv7511->regmap, ADV7511_REG_VIDEO_INPUT_CFG1, 0x81,
			   (output_format_422 << 7) | output_format_ycbcr);

	regmap_update_bits(adv7511->regmap, ADV7511_REG_HDCP_HDMI_CFG,
			   ADV7511_HDMI_CFG_MODE_MASK, mode);

	hdmi_avi_infoframe_pack(&config.avi_infoframe, infoframe,
				sizeof(infoframe));

	/* The AVI infoframe id is not configurable */
	regmap_bulk_write(adv7511->regmap, ADV7511_REG_AVI_INFOFRAME_VERSION,
			  infoframe + 1, sizeof(infoframe) - 1);

	adv7511_update_sysfs_settings(adv7511);

	adv7511_packet_enable(adv7511, ADV7511_PACKET_ENABLE_AVI_INFOFRAME);
}

static void adv7511_set_link_config(struct adv7511 *adv7511,
				    const struct adv7511_link_config *config)
{
	/*
	 * The input style values documented in the datasheet don't match the
	 * hardware register field values :-(
	 */
	static const unsigned int input_styles[4] = { 0, 2, 1, 3 };

	unsigned int clock_delay;
	unsigned int color_depth;
	unsigned int input_id;

	clock_delay = (config->clock_delay + 1200) / 400;
	color_depth = config->input_color_depth == 8 ? 3
		    : (config->input_color_depth == 10 ? 1 : 2);

	/* TODO Support input ID 6 */
	if (config->input_colorspace != HDMI_COLORSPACE_YUV422)
		input_id = config->input_clock == ADV7511_INPUT_CLOCK_DDR
			 ? 5 : 0;
	else if (config->input_clock == ADV7511_INPUT_CLOCK_DDR)
		input_id = config->embedded_sync ? 8 : 7;
	else if (config->input_clock == ADV7511_INPUT_CLOCK_2X)
		input_id = config->embedded_sync ? 4 : 3;
	else
		input_id = config->embedded_sync ? 2 : 1;

	regmap_update_bits(adv7511->regmap, ADV7511_REG_I2C_FREQ_ID_CFG, 0xf,
			   input_id);
	regmap_update_bits(adv7511->regmap, ADV7511_REG_VIDEO_INPUT_CFG1, 0x7e,
			   (color_depth << 4) |
			   (input_styles[config->input_style] << 2));
	regmap_write(adv7511->regmap, ADV7511_REG_VIDEO_INPUT_CFG2,
		     config->input_justification << 3);
	regmap_write(adv7511->regmap, ADV7511_REG_TIMING_GEN_SEQ,
		     config->sync_pulse << 2);

	regmap_write(adv7511->regmap, ADV7511_REG_CLOCK_DELAY,
		     clock_delay << 5);

	adv7511->embedded_sync = config->embedded_sync;
	adv7511->hsync_polarity = config->hsync_polarity;
	adv7511->vsync_polarity = config->vsync_polarity;
	adv7511->rgb = config->input_colorspace == HDMI_COLORSPACE_RGB;
}

static void __adv7511_power_on(struct adv7511 *adv7511)
{
	adv7511->current_edid_segment = -1;

	regmap_update_bits(adv7511->regmap, ADV7511_REG_POWER,
			   ADV7511_POWER_POWER_DOWN, 0);
	if (adv7511->i2c_main->irq) {
		u8 int0_enable, int1_enable;
		/*
		 * Documentation says the INT_ENABLE registers are reset in
		 * POWER_DOWN mode. My 7511w preserved the bits, however.
		 * Still, let's be safe and stick to the documentation.
		 */		
		if (adv7511->replace_edid) {
			int0_enable = ADV7511_INT0_EDID_READY | ADV7511_INT0_HPD |
			     ADV7511_INT0_HDCP_AUTHENTICATED | ADV7511_INT0_RI_READY | ADV7511_INT0_RSEN;
			int1_enable = ADV7511_INT1_DDC_ERROR | ADV7511_INT1_BKSV;
		} else {
			int0_enable = ADV7511_INT0_EDID_READY | ADV7511_INT0_HPD;
			int1_enable = ADV7511_INT1_DDC_ERROR;
		}

		regmap_write(adv7511->regmap, ADV7511_REG_INT_ENABLE(0), int0_enable);
		regmap_write(adv7511->regmap, ADV7511_REG_INT_ENABLE(1), int1_enable);
	}

	/*
	 * Per spec it is allowed to pulse the HPD signal to indicate that the
	 * EDID information has changed. Some monitors do this when they wakeup
	 * from standby or are enabled. When the HPD goes low the adv7511 is
	 * reset and the outputs are disabled which might cause the monitor to
	 * go to standby again. To avoid this we ignore the HPD pin for the
	 * first few seconds after enabling the output.
	 */
	regmap_update_bits(adv7511->regmap, ADV7511_REG_POWER2,
			   ADV7511_REG_POWER2_HPD_SRC_MASK,
			   ADV7511_REG_POWER2_HPD_SRC_NONE);
}

static void adv7511_power_on(struct adv7511 *adv7511)
{
	__adv7511_power_on(adv7511);

	/*
	 * Most of the registers are reset during power down or when HPD is low.
	 */
	mutex_lock(&adv7511->sysfs_lock);
	regcache_sync(adv7511->regmap);
	mutex_unlock(&adv7511->sysfs_lock);

	if (adv7511->type == ADV7533 || adv7511->type == ADV7535)
		adv7533_dsi_power_on(adv7511);
	adv7511->powered = true;
}

static void __adv7511_power_off(struct adv7511 *adv7511)
{
	/* TODO: setup additional power down modes */
	regmap_update_bits(adv7511->regmap, ADV7511_REG_POWER,
			   ADV7511_POWER_POWER_DOWN,
			   ADV7511_POWER_POWER_DOWN);
	regcache_mark_dirty(adv7511->regmap);
}

static void adv7511_power_off(struct adv7511 *adv7511)
{
	__adv7511_power_off(adv7511);
	if (adv7511->type == ADV7533 || adv7511->type == ADV7535)
		adv7533_dsi_power_off(adv7511);
	adv7511->powered = false;
}

/* -----------------------------------------------------------------------------
 * Interrupt and hotplug detection
 */

static bool adv7511_hpd(struct adv7511 *adv7511)
{
	unsigned int irq0;
	int ret;

	ret = regmap_read(adv7511->regmap, ADV7511_REG_INT(0), &irq0);
	if (ret < 0)
		return false;

	if (irq0 & ADV7511_INT0_HPD) {
		regmap_write(adv7511->regmap, ADV7511_REG_INT(0),
			     ADV7511_INT0_HPD);
		return true;
	}

	return false;
}

static void adv7511_hpd_work(struct work_struct *work)
{
	struct adv7511 *adv7511 = container_of(work, struct adv7511, hpd_work);
	enum drm_connector_status status = connector_status_disconnected;
	unsigned int val;
	int ret;

	if (!adv7511_is_pcont1_and_irq_enabled(adv7511)) {
		dev_info(&adv7511->i2c_main->dev,
			"%s: IRQ or PCONT_1 is disabled after "
			"scheduling this wq, do nothing\n", __func__);
		return;
	}

	ret = regmap_read(adv7511->regmap, ADV7511_REG_STATUS, &val);
	if (ret >= 0 && (val & ADV7511_STATUS_HPD))
		status = connector_status_connected;

	DRM_DEV_DEBUG_DRIVER(adv7511->connector.kdev, "HDMI HPD event: %s\n",
		drm_get_connector_status_name(status));

	/*
	 * The bridge resets its registers on unplug. So when we get a plug
	 * event and we're already supposed to be powered, cycle the bridge to
	 * restore its state.
	 */
	if (status == connector_status_connected &&
	    adv7511->connector.status == connector_status_disconnected &&
	    adv7511->powered) {
		regcache_mark_dirty(adv7511->regmap);
		adv7511_power_on(adv7511);
	}

	if (adv7511->connector.status != status) {
		dev_info(&adv7511->i2c_main->dev,
			"notify hotplug event:%d\n", status);
		adv7511->connector.status = status;

		if (status == connector_status_disconnected)
			adv7511_clear_raw_edid_buf(adv7511);

		drm_kms_helper_hotplug_event(adv7511->connector.dev);
	}
}

static int adv7511_irq_process(struct adv7511 *adv7511, bool process_hpd)
{
	unsigned int irq0, irq1;
	unsigned int val;
	int ret;

	if (!adv7511_is_pcont1_and_irq_enabled(adv7511))
		return -EINVAL;

	ret = regmap_read(adv7511->regmap, ADV7511_REG_INT(0), &irq0);
	if (ret < 0)
		return ret;

	ret = regmap_read(adv7511->regmap, ADV7511_REG_INT(1), &irq1);
	if (ret < 0)
		return ret;

	regmap_write(adv7511->regmap, ADV7511_REG_INT(0), irq0);
	regmap_write(adv7511->regmap, ADV7511_REG_INT(1), irq1);

	if (process_hpd && irq0 & ADV7511_INT0_HPD && adv7511->bridge.encoder)
		schedule_work(&adv7511->hpd_work);

	if (irq0 & ADV7511_INT0_EDID_READY || irq1 & ADV7511_INT1_DDC_ERROR) {
		if (irq1 & ADV7511_INT1_DDC_ERROR) {
			unsigned int status;

			ret = regmap_read(adv7511->regmap,
				ADV7511_REG_DDC_STATUS, &status);
			if (ret < 0)
				return ret;

			pr_debug("%s: ADV7511_INT1_DDC_ERROR\n",
				__func__);
			pr_debug("%s:   DDC error: 0x%x\n",
				__func__, status >> 4);
			pr_debug("%s:   DDC state: 0x%x\n",
				__func__, status & 0xf);
			
			/* Enable Force Black Video Data when DDC Error Occer */
			if (adv7511->replace_edid && ((status & 0xf0) > 0))
				regmap_update_bits(adv7511->regmap, ADV7511_REG_MAIN_0XD5, 
					ADV7511_REG_BLACK_IMAGE_MASK, ADV7511_REG_BLACK_IMAGE_ENABLED);
		}
		adv7511->edid_read = true;

		if (adv7511->i2c_main->irq)
			wake_up_all(&adv7511->wq);
	}

	if (irq0 & ADV7511_INT0_HDCP_AUTHENTICATED) {
		pr_info("%s: Interrupt about HDCP Authenticated!: 0x96=0x%02x\n", __func__, irq0);
		/* Disable Force Black Video Data when HDCP Authenticated */
		regmap_update_bits(adv7511->regmap, ADV7511_REG_MAIN_0XD5,
			ADV7511_REG_BLACK_IMAGE_MASK, ADV7511_REG_BLACK_IMAGE_DISABLED);
		adv7511->is_hdcp_authenticated = true;
	}
	
	if (irq0 & ADV7511_INT0_RI_READY) {
		pr_debug("%s: Interrupt about Ri Ready: 0x96=0x%02x\n", __func__, irq0);
		regmap_read(adv7511->regmap, ADV7511_REG_HDCP_STATUS, &val);
		if (val & ADV7511_HDCP_ENCRYPTION_STATUS)
			regmap_update_bits(adv7511->regmap, ADV7511_REG_MAIN_0XD5, 
				ADV7511_REG_BLACK_IMAGE_ENABLED, ADV7511_REG_BLACK_IMAGE_DISABLED);
		else
			regmap_update_bits(adv7511->regmap, ADV7511_REG_MAIN_0XD5,
				ADV7511_REG_BLACK_IMAGE_ENABLED, ADV7511_REG_BLACK_IMAGE_ENABLED);
	}

	if (irq0 & ADV7511_INT0_RSEN) {
		unsigned int rx_sense;

		regmap_read(adv7511->regmap, ADV7511_REG_STATUS, &val);
		rx_sense = (val & ADV7511_RSEN_STATUS) >> 5;
		pr_info("%s: Interrupt About RSEN: rx_sense=%d", __func__, rx_sense);
	}
	
	if (irq1 & ADV7511_INT1_BKSV) {
		pr_info("%s: Interrupt about BKSV: 0x97=0x%02x\n", __func__, irq1);
		regmap_update_bits(adv7511->regmap, ADV7511_REG_INT(1), 
			ADV7511_INT1_BKSV, ADV7511_HDCP_BKSV_CLEAR);
	}

	return 0;
}

static irqreturn_t adv7511_irq_handler(int irq, void *devid)
{
	struct adv7511 *adv7511 = devid;
	int ret;

	pr_debug("%s: IRQ received\n", __func__);

	ret = adv7511_irq_process(adv7511, true);
	if (ret < 0)
		pr_err("%s: I2C read error (ret: %d)\n", __func__, ret);

	return ret < 0 ? IRQ_NONE : IRQ_HANDLED;
}

/* -----------------------------------------------------------------------------
 * EDID retrieval
 */

static int adv7511_wait_for_edid(struct adv7511 *adv7511, int timeout)
{
	int ret;

	if (adv7511->i2c_main->irq) {
		ret = wait_event_interruptible_timeout(adv7511->wq,
				adv7511->edid_read, msecs_to_jiffies(timeout));
	} else {
		for (; timeout > 0; timeout -= 25) {
			ret = adv7511_irq_process(adv7511, false);
			if (ret < 0)
				break;

			if (adv7511->edid_read)
				break;

			msleep(25);
		}
	}

	return adv7511->edid_read ? 0 : -EIO;
}

static int adv7511_get_edid_block(void *data, u8 *buf, unsigned int block,
				  size_t len)
{
	struct adv7511 *adv7511 = data;
	struct i2c_msg xfer[2];
	uint8_t offset;
	unsigned int i;
	int ret;

	if (len > 128)
		return -EINVAL;

	if (adv7511->current_edid_segment != block / 2) {
		unsigned int status;

		ret = regmap_read(adv7511->regmap, ADV7511_REG_DDC_STATUS,
				  &status);
		if (ret < 0)
			return ret;

		if (adv7511->replace_edid || status != 2) {
			adv7511->edid_read = false;
			regmap_write(adv7511->regmap, ADV7511_REG_EDID_SEGMENT,
				     block / 2);
			ret = adv7511_wait_for_edid(adv7511, 200);
			if (ret < 0)
				return ret;
		}

		/* Break this apart, hopefully more I2C controllers will
		 * support 64 byte transfers than 256 byte transfers
		 */

		xfer[0].addr = adv7511->i2c_edid->addr;
		xfer[0].flags = 0;
		xfer[0].len = 1;
		xfer[0].buf = &offset;
		xfer[1].addr = adv7511->i2c_edid->addr;
		xfer[1].flags = I2C_M_RD;
		xfer[1].len = 64;
		xfer[1].buf = adv7511->edid_buf;

		offset = 0;

		for (i = 0; i < 4; ++i) {
			ret = i2c_transfer(adv7511->i2c_edid->adapter, xfer,
					   ARRAY_SIZE(xfer));
			if (ret < 0)
				return ret;
			else if (ret != 2)
				return -EIO;

			xfer[1].buf += 64;
			offset += 64;
		}

		adv7511->current_edid_segment = block / 2;
	}

	if (block % 2 == 0)
		memcpy(buf, adv7511->edid_buf, len);
	else
		memcpy(buf, adv7511->edid_buf + 128, len);

	return 0;
}

/* -----------------------------------------------------------------------------
 * ADV75xx helpers
 */

static int adv7511_is_support_vic(uint8_t vic)
{
	int i;

	for (i = 0; i < sizeof(adv7511_support_vic); i++)
		if (vic == adv7511_support_vic[i]) 
			return i + 1;

	return 0;
}

static bool adv7511_is_duplicate_vic(uint8_t vic, 
			     struct adv7511_can_output_resolution *reso)
{
	int i;

	for (i = 0; i < reso->size; i++)
		if (vic == reso->vic_list[i])
			return true;
	
	return false;
}

static void adv7511_extract_support_vic(uint8_t *vic_list,
			     struct adv7511_can_output_resolution *reso)
{
	int i;

	for (i = 0; vic_list[i]; i++) {
		if (adv7511_is_support_vic(vic_list[i]) &&
			!adv7511_is_duplicate_vic(vic_list[i], reso)) {
			reso->vic_list[reso->size] = vic_list[i]; 
			reso->size++;
			pr_debug("support %d\n", vic_list[i]);
		}
		else
			pr_debug("unsupport %d\n", vic_list[i]);
	}
}

/* Checksum must be written so that the total value for each block(128byte) is 0x00 */
static int adv7511_calculate_edid_checksum(u8 *edid, int block_num)
{
	int i;
	int start, end;
	int sum = 0;

	start = block_num * ADV7511_EDID_LENGTH;
	end = (block_num + 1) * ADV7511_EDID_LENGTH - 1;

	for (i = start;  i < end; i++)
		sum += edid[i];

	return 0x100 - (sum & 0xff);
}

static void adv7511_replace_edid(struct adv7511 *adv7511, struct edid *edid,
			     u8 *replaced_edid)
{
	uint8_t dtd[ADV7511_EDID_DTD_MAX_SIZE];
	uint8_t svd[ADV7511_EDID_VIC_MAX_SIZE];
	struct adv7511_can_output_resolution reso;
	int i;
	int offset;

	reso.size = 0;
	memset(dtd, '\0', sizeof(dtd));
	memset(svd, '\0', sizeof(svd));

	/* Analyze EDID from Sony Module */
	hd_edid_RamInitialize();
	hd_edid_RamInitializeData();
	hd_edid_SetEdidRamBlock0(HD_MODULE_PORT_TX_0, (u8 *)edid);
	hd_edid_SetTxEdidBlockSize(HD_MODULE_PORT_TX_0, edid->extensions + 1);
	hd_edid_Analyze(HD_MODULE_PORT_TX_0);
	hd_edid_GetSinkResolution(HD_MODULE_PORT_TX_0, svd, dtd);
	adv7511->is_hdmi = hd_edid_IsHdmi(HD_MODULE_PORT_TX_0);
	
	adv7511_extract_support_vic(dtd, &reso);
	adv7511_extract_support_vic(svd, &reso);

	/* If Sink EDID doesn't write VIC that ADV7535 can output,
	 * write VGA VIC.
	 */
	if (reso.size == 0) {
		reso.vic_list[0] = ADV7511_EDID_VIC_VGA;
		reso.size++;
	}
	
	/* 1st block replace */
	memcpy(&replaced_edid[0], adv7511_fixed_edid_first_block, 
		sizeof(adv7511_fixed_edid_first_block));

	/* 2nd block replace */
	replaced_edid[ADV7511_EDID_2ND_BLOCK_HEADER(0)] = ADV7511_EDID_CTA_EXT_TAG;
	replaced_edid[ADV7511_EDID_2ND_BLOCK_HEADER(1)] = ADV7511_EDID_CTA_EXT_REV;
	replaced_edid[ADV7511_EDID_2ND_BLOCK_HEADER(3)] = ADV7511_EDID_CTA_EXT_GD;

	/* 2nd block Video Data Block replace */
	replaced_edid[ADV7511_EDID_START_BLOCK_HEADER(0)] = 
		(ADV7511_EDID_VDB_TAG_CODE << 5) | reso.size;
	offset = ADV7511_EDID_START_BLOCK_HEADER(1);
	memcpy(&replaced_edid[offset], reso.vic_list,
		reso.size);
	offset += reso.size;

	/* 2nd block other Data Block replace */
	memcpy(&replaced_edid[offset], adv7511_fixed_edid_second_block,
		sizeof(adv7511_fixed_edid_second_block));
	offset += sizeof(adv7511_fixed_edid_second_block);

	/* 2nd block header replace */
	replaced_edid[ADV7511_EDID_2ND_BLOCK_OFFSET] = offset - ADV7511_EDID_LENGTH;

	/* 2nd block checksum replace */
	replaced_edid[255] = 
		adv7511_calculate_edid_checksum(replaced_edid, ADV7511_EDID_BLOCK1);

	/* Print */
	pr_debug("    00 01 02 03 04 05 06 07 08 09 0A 0B 0C 0D 0E 0F\n");
	pr_debug("-----------------------------------------------------\n");
	for (i = 0; i < 16; i++) {
		pr_debug("%02x: %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x\n",
			i * 16,
			replaced_edid[16 * i],
			replaced_edid[16 * i + 1],
			replaced_edid[16 * i + 2],
			replaced_edid[16 * i + 3],
			replaced_edid[16 * i + 4],
			replaced_edid[16 * i + 5],
			replaced_edid[16 * i + 6],
			replaced_edid[16 * i + 7],
			replaced_edid[16 * i + 8],
			replaced_edid[16 * i + 9],
			replaced_edid[16 * i + 10],
			replaced_edid[16 * i + 11],
			replaced_edid[16 * i + 12],
			replaced_edid[16 * i + 13],
			replaced_edid[16 * i + 14],
			replaced_edid[16 * i + 15]
		);
	}
}

static int adv7511_get_modes(struct adv7511 *adv7511,
			     struct drm_connector *connector)
{
	struct edid *edid = NULL;
	unsigned int count;
	u32 bus_format = MEDIA_BUS_FMT_RGB888_1X24;
	int ret;
	int try_count = 0;
	unsigned int val;
	u8 replaced_edid[ADV7511_EDID_REPLACE_LENGTH] = {0};

	/* Reading the EDID only works if the device is powered */
	if (!adv7511_is_pcont1_and_irq_enabled(adv7511))
		goto skip_edid_detect;

	if (!adv7511->powered) {
		unsigned int edid_i2c_addr;
		of_property_read_u32((&(adv7511->i2c_main)->dev)->of_node, 
			"adi,addr-edid", &edid_i2c_addr);
		edid_i2c_addr = edid_i2c_addr << 1;

		__adv7511_power_on(adv7511);

		/* Reset the EDID_I2C_ADDR register as it might be cleared */
		regmap_write(adv7511->regmap, ADV7511_REG_EDID_I2C_ADDR,
			     edid_i2c_addr);
	}

	do {
		if (!adv7511_is_pcont1_and_irq_enabled(adv7511)) {
			pr_info("%s: HPD: IRQ or PCONT_1 is disabled during "
				"DDC communication, abort\n", __func__);
			edid = NULL;
			break; /* EDID: NULL */
		}

		ret = regmap_read(adv7511->regmap, ADV7511_REG_STATUS, &val);
		if (ret < 0 || !(val & ADV7511_STATUS_HPD)) {
			pr_info("%s: HPD: disconnected during DDC communication, abort\n", 
				__func__);
			edid = NULL;
			break; /* EDID: NULL */
		}
		try_count++;
		pr_info("%s: wait for EDID (try: %d)\n", __func__,
			try_count);

		edid = drm_do_get_edid(connector, adv7511_get_edid_block,
					adv7511);

		/* Retry reading the EDID when 2nd block or later 
		 * checksum error occur. */
		if (edid && adv7511->replace_edid) {
			u8 checksum;
			int i;
			bool is_checksum_correct;

			is_checksum_correct = true;
			for (i = 1; i <= edid->extensions; i++) {
				checksum = adv7511_calculate_edid_checksum((u8 *)edid, i);

				if (checksum != ((u8 *)edid)[(i + 1) * EDID_LENGTH - 1]) {
					pr_info("%s: %d Block checksum error\n", __func__, i + 1);
					is_checksum_correct = false;
					break;
				}
			}

			if (!is_checksum_correct) {
				edid = NULL;
				continue;
			}
		}

		if (edid) /* EDID READY */
			break;

		/* Failed, retry after reset the chip */
		__adv7511_power_off(adv7511);
		usleep_range(15000, 20000);
		__adv7511_power_on(adv7511);
		regcache_sync(adv7511->regmap);
	} while (try_count <= 100);

	if (!adv7511_is_pcont1_and_irq_enabled(adv7511))
		goto skip_edid_detect;

	if (!adv7511->powered)
		__adv7511_power_off(adv7511);

	if (edid) {
		adv7511->edid_size = ADV7511_EDID_LENGTH * (edid->extensions + 1);
		memcpy(adv7511->raw_edid, (u8 *)edid, adv7511->edid_size);
		
		if (adv7511->replace_edid) {
			struct edid *new_edid = NULL;
			pr_info("replace EDID\n");
			adv7511_replace_edid(adv7511, edid, replaced_edid);
			
			new_edid = krealloc(edid, ADV7511_EDID_REPLACE_LENGTH, GFP_KERNEL);
			if (!new_edid)
				goto skip_edid_detect;
			edid = new_edid;
			memcpy(edid, (struct edid *)replaced_edid, 
				ADV7511_EDID_REPLACE_LENGTH);
		}
	}

skip_edid_detect: /* It also passes in the normal case here. */
	kfree(adv7511->edid);
	adv7511->edid = edid;

	drm_mode_connector_update_edid_property(connector, edid);
	if (!edid) /* retry count over */
		return -EINVAL; /* return error code via connector */

	count = drm_add_edid_modes(connector, edid);

	adv7511_set_config_csc(adv7511, connector, adv7511->rgb);

	connector->display_info.bus_flags = DRM_BUS_FLAG_DE_LOW |
					    DRM_BUS_FLAG_PIXDATA_NEGEDGE;

	ret = drm_display_info_set_bus_formats(&connector->display_info,
					       &bus_format, 1);
	if (ret)
		return ret;

	return count;
}

static enum drm_connector_status
adv7511_detect(struct adv7511 *adv7511, struct drm_connector *connector)
{
	enum drm_connector_status status;
	unsigned int val;
	bool hpd;
	int ret;

	ret = regmap_read(adv7511->regmap, ADV7511_REG_STATUS, &val);
	if (ret < 0)
		return connector_status_disconnected;

	if (val & ADV7511_STATUS_HPD)
		status = connector_status_connected;
	else
		status = connector_status_disconnected;

	hpd = adv7511_hpd(adv7511);

	/* The chip resets itself when the cable is disconnected, so in case
	 * there is a pending HPD interrupt and the cable is connected there was
	 * at least one transition from disconnected to connected and the chip
	 * has to be reinitialized. */
	if (status == connector_status_connected && hpd && adv7511->powered) {
		regcache_mark_dirty(adv7511->regmap);
		adv7511_power_on(adv7511);
		ret = adv7511_get_modes(adv7511, connector);
		if (ret < 0 || adv7511->status == connector_status_connected)
			status = connector_status_disconnected;
	} else {
		/* Renable HPD sensing */
		regmap_update_bits(adv7511->regmap, ADV7511_REG_POWER2,
				   ADV7511_REG_POWER2_HPD_SRC_MASK,
				   ADV7511_REG_POWER2_HPD_SRC_BOTH);
	}

	if (!adv7511->is_initialized) {
		pr_info("%s: HPD: %s on booting\n", __func__,
			(status == connector_status_connected)
			? "connected" : "disconnected");
		adv7511->is_initialized = true;
	}

	adv7511->status = status;
	return status;
}

static int adv7511_mode_valid(struct adv7511 *adv7511,
			      struct drm_display_mode *mode)
{
	/* _good_mode_combo[]:			     */
	/*   This is the whitelist of display mode.  */
	static const struct _good_mode_combo_t {
		int hdisplay;
		int vdisplay;
		int hsync_start;
		int hsync_end;
		int htotal;
		int vrefresh;
		int clock;
	} _good_mode_combo[] = {
	     /* {hdisp, vdisp,  hss,  hse, htot, vref, clock(kHz) } */
		{  640,   480,  656,  752,  800,   60,   25175 }, // VIC1     59.94Hz
		{  640,   480,  656,  752,  800,   60,   25200 }, // VIC1     60.00Hz
		{  720,   480,  736,  798,  858,   60,   27000 }, // VIC2,3   59.94Hz
		{  720,   480,  736,  798,  858,   60,   27027 }, // VIC2,3   60.00Hz
		{  720,   576,  732,  796,  864,   50,   27000 }, // VIC17,18 50.00Hz
		{ 1280,   720, 1390, 1430, 1650,   60,   74176 }, // VIC4     59.94Hz
		{ 1280,   720, 1390, 1430, 1650,   60,   74250 }, // VIC4     60.00Hz
		{ 1920,  1080, 2008, 2052, 2200,   60,  148500 }, // VIC16    60.00Hz
		{ 1920,  1080, 2008, 2052, 2200,   60,  148352 }, // VIC16    59.94Hz
	};
	int i;

	if (mode->clock > 165000)
		return MODE_CLOCK_HIGH;

	for (i = 0; i < ARRAY_SIZE(_good_mode_combo); i++)
		if ((mode->hdisplay    == _good_mode_combo[i].hdisplay)    &&
		    (mode->vdisplay    == _good_mode_combo[i].vdisplay)    &&
		    (mode->hsync_start == _good_mode_combo[i].hsync_start) &&
		    (mode->hsync_end   == _good_mode_combo[i].hsync_end)   &&
		    (mode->htotal      == _good_mode_combo[i].htotal)      &&
		    (mode->clock       == _good_mode_combo[i].clock)) {
			/* valid: The mode is found in the table */
			pr_debug("%s: MODE VALID (%d x %d @ %d, %d Hz)\n",
				__func__, mode->hdisplay, mode->vdisplay,
				mode->vrefresh, mode->clock);
			return MODE_OK;
		}

	/* invalid: The mode is not found in the table */
	pr_debug("%s: MODE INVALID (%d x %d @ %d, %d Hz)\n",
		__func__, mode->hdisplay, mode->vdisplay,
		mode->vrefresh, mode->clock);
	return MODE_BAD;
}

static void adv7511_mode_set(struct adv7511 *adv7511,
			     struct drm_display_mode *mode,
			     struct drm_display_mode *adj_mode)
{
	unsigned int low_refresh_rate;
	unsigned int hsync_polarity = 0;
	unsigned int vsync_polarity = 0;

	if (adv7511->embedded_sync) {
		unsigned int hsync_offset, hsync_len;
		unsigned int vsync_offset, vsync_len;

		hsync_offset = adj_mode->crtc_hsync_start -
			       adj_mode->crtc_hdisplay;
		vsync_offset = adj_mode->crtc_vsync_start -
			       adj_mode->crtc_vdisplay;
		hsync_len = adj_mode->crtc_hsync_end -
			    adj_mode->crtc_hsync_start;
		vsync_len = adj_mode->crtc_vsync_end -
			    adj_mode->crtc_vsync_start;

		/* The hardware vsync generator has a off-by-one bug */
		vsync_offset += 1;

		regmap_write(adv7511->regmap, ADV7511_REG_HSYNC_PLACEMENT_MSB,
			     ((hsync_offset >> 10) & 0x7) << 5);
		regmap_write(adv7511->regmap, ADV7511_REG_SYNC_DECODER(0),
			     (hsync_offset >> 2) & 0xff);
		regmap_write(adv7511->regmap, ADV7511_REG_SYNC_DECODER(1),
			     ((hsync_offset & 0x3) << 6) |
			     ((hsync_len >> 4) & 0x3f));
		regmap_write(adv7511->regmap, ADV7511_REG_SYNC_DECODER(2),
			     ((hsync_len & 0xf) << 4) |
			     ((vsync_offset >> 6) & 0xf));
		regmap_write(adv7511->regmap, ADV7511_REG_SYNC_DECODER(3),
			     ((vsync_offset & 0x3f) << 2) |
			     ((vsync_len >> 8) & 0x3));
		regmap_write(adv7511->regmap, ADV7511_REG_SYNC_DECODER(4),
			     vsync_len & 0xff);

		hsync_polarity = !(adj_mode->flags & DRM_MODE_FLAG_PHSYNC);
		vsync_polarity = !(adj_mode->flags & DRM_MODE_FLAG_PVSYNC);
	} else {
		enum adv7511_sync_polarity mode_hsync_polarity;
		enum adv7511_sync_polarity mode_vsync_polarity;

		/**
		 * If the input signal is always low or always high we want to
		 * invert or let it passthrough depending on the polarity of the
		 * current mode.
		 **/
		if (adj_mode->flags & DRM_MODE_FLAG_NHSYNC)
			mode_hsync_polarity = ADV7511_SYNC_POLARITY_LOW;
		else
			mode_hsync_polarity = ADV7511_SYNC_POLARITY_HIGH;

		if (adj_mode->flags & DRM_MODE_FLAG_NVSYNC)
			mode_vsync_polarity = ADV7511_SYNC_POLARITY_LOW;
		else
			mode_vsync_polarity = ADV7511_SYNC_POLARITY_HIGH;

		if (adv7511->hsync_polarity != mode_hsync_polarity &&
		    adv7511->hsync_polarity !=
		    ADV7511_SYNC_POLARITY_PASSTHROUGH)
			hsync_polarity = 1;

		if (adv7511->vsync_polarity != mode_vsync_polarity &&
		    adv7511->vsync_polarity !=
		    ADV7511_SYNC_POLARITY_PASSTHROUGH)
			vsync_polarity = 1;
	}

	if (mode->vrefresh <= 24)
		low_refresh_rate = ADV7511_LOW_REFRESH_RATE_24HZ;
	else if (mode->vrefresh <= 25)
		low_refresh_rate = ADV7511_LOW_REFRESH_RATE_25HZ;
	else if (mode->vrefresh <= 30)
		low_refresh_rate = ADV7511_LOW_REFRESH_RATE_30HZ;
	else
		low_refresh_rate = ADV7511_LOW_REFRESH_RATE_NONE;

	if (adv7511->type == ADV7535)
		regmap_update_bits(adv7511->regmap,
				   ADV7511_REG_INFOFRAME_UPDATE,
				   0xc, low_refresh_rate << 2);
	else
		regmap_update_bits(adv7511->regmap,
				   ADV7511_REG_LOW_REFRESH_RATE_BITS,
				   0x6, low_refresh_rate << 1);

	/* Apply current V/H Sync polarity */
	adv7511->mode_vsync_polarity = mode->flags & DRM_MODE_FLAG_NVSYNC;
	adv7511->mode_hsync_polarity = mode->flags & DRM_MODE_FLAG_NHSYNC;
	adv7511_update_vhsync_polarity(adv7511);
	
	if (adv7511->replace_edid) {
		regmap_update_bits(adv7511->regmap, ADV7511_REG_HDCP_HDMI_CFG,
			ADV7511_HDCP_MASK, ADV7511_HDCP_DISABLE);
		regmap_update_bits(adv7511->regmap, ADV7511_REG_HDCP_CONFIG,
			ADV7511_RI_FREQUENCY_MASK, ADV7511_RI_FREQUENCY_64);
	}

	if (adv7511->type == ADV7533 || adv7511->type == ADV7535)
		adv7533_mode_set(adv7511, adj_mode);

	drm_mode_copy(&adv7511->curr_mode, adj_mode);

	/*
	 * TODO Test first order 4:2:2 to 4:4:4 up conversion method, which is
	 * supposed to give better results.
	 */

	adv7511->f_tmds = mode->clock;
}

/* Connector funcs */
static struct adv7511 *connector_to_adv7511(struct drm_connector *connector)
{
	return container_of(connector, struct adv7511, connector);
}

static int adv7511_connector_get_modes(struct drm_connector *connector)
{
	struct adv7511 *adv = connector_to_adv7511(connector);

	return adv7511_get_modes(adv, connector);
}

static enum drm_mode_status
adv7511_connector_mode_valid(struct drm_connector *connector,
			     struct drm_display_mode *mode)
{
	struct adv7511 *adv = connector_to_adv7511(connector);

	return adv7511_mode_valid(adv, mode);
}

static struct drm_connector_helper_funcs adv7511_connector_helper_funcs = {
	.get_modes = adv7511_connector_get_modes,
	.mode_valid = adv7511_connector_mode_valid,
};

static enum drm_connector_status
adv7511_connector_detect(struct drm_connector *connector, bool force)
{
	struct adv7511 *adv = connector_to_adv7511(connector);

	return adv7511_detect(adv, connector);
}

static const struct drm_connector_funcs adv7511_connector_funcs = {
	.fill_modes = drm_helper_probe_single_connector_modes,
	.detect = adv7511_connector_detect,
	.destroy = drm_connector_cleanup,
	.reset = drm_atomic_helper_connector_reset,
	.atomic_duplicate_state = drm_atomic_helper_connector_duplicate_state,
	.atomic_destroy_state = drm_atomic_helper_connector_destroy_state,
};

/* Bridge funcs */
static struct adv7511 *bridge_to_adv7511(struct drm_bridge *bridge)
{
	return container_of(bridge, struct adv7511, bridge);
}

static void adv7511_bridge_enable(struct drm_bridge *bridge)
{
	struct adv7511 *adv = bridge_to_adv7511(bridge);

	adv7511_power_on(adv);
}

static void adv7511_bridge_disable(struct drm_bridge *bridge)
{
	struct adv7511 *adv = bridge_to_adv7511(bridge);

	adv7511_power_off(adv);
}

static void adv7511_bridge_mode_set(struct drm_bridge *bridge,
				    struct drm_display_mode *mode,
				    struct drm_display_mode *adj_mode)
{
	struct adv7511 *adv = bridge_to_adv7511(bridge);

	adv7511_mode_set(adv, mode, adj_mode);
}

static bool adv7511_bridge_mode_fixup(struct drm_bridge *bridge,
				      const struct drm_display_mode *mode,
				      struct drm_display_mode *adjusted_mode)
{
	struct adv7511 *adv = bridge_to_adv7511(bridge);

	/* Overwrite AVI InfoFrame */

	/* In case of some resolutions, fix aspect ratio to 4:3 or 16:9 forcibly. */
	pr_debug("%s: resolution %d x %d\n", __func__, mode->hdisplay, mode->vdisplay);
	if (mode->hdisplay == 640 && mode->vdisplay == 480) {
		pr_debug("%s: fix up VGA aspect ratio to 4:3\n", __func__);
		adv->picture_aspect = DRM_MODE_PICTURE_ASPECT_4_3;
	} else if (mode->vdisplay == 720) {
		pr_debug("%s: fix up 720p aspect ratio to 16:9\n", __func__);
		adv->picture_aspect = DRM_MODE_PICTURE_ASPECT_16_9;
	} else if (mode->vdisplay == 1080) {
		pr_debug("%s: fix up 1080p aspect ratio to 16:9\n", __func__);
		adv->picture_aspect = DRM_MODE_PICTURE_ASPECT_16_9;
	}

	adv7511_update_sysfs_settings(adv);

	/* V/H Sync Polarity Fix Up (Using current flags instead of adj) */
	adv7511_update_vhsync_polarity(adv);

	if (adv->type == ADV7533 || adv->type == ADV7535)
		return adv7533_mode_fixup(adv, adjusted_mode);

	return true;
}

static int adv7511_bridge_attach(struct drm_bridge *bridge)
{
	struct adv7511 *adv = bridge_to_adv7511(bridge);
	int ret;

	if (!bridge->encoder) {
		DRM_ERROR("Parent encoder object not found");
		return -ENODEV;
	}

	adv->connector.polled = DRM_CONNECTOR_POLL_HPD;

	ret = drm_connector_init(bridge->dev, &adv->connector,
				 &adv7511_connector_funcs,
				 DRM_MODE_CONNECTOR_HDMIA);
	if (ret) {
		DRM_ERROR("Failed to initialize connector with drm\n");
		return ret;
	}
	drm_connector_helper_add(&adv->connector,
				 &adv7511_connector_helper_funcs);
	drm_mode_connector_attach_encoder(&adv->connector, bridge->encoder);

	if (adv->type == ADV7533 || adv->type == ADV7535)
		ret = adv7533_attach_dsi(adv);

	if (adv->i2c_main->irq)
		regmap_write(adv->regmap, ADV7511_REG_INT_ENABLE(0),
			     ADV7511_INT0_HPD);

	return ret;
}

static const struct drm_bridge_funcs adv7511_bridge_funcs = {
	.enable = adv7511_bridge_enable,
	.disable = adv7511_bridge_disable,
	.mode_set = adv7511_bridge_mode_set,
	.mode_fixup = adv7511_bridge_mode_fixup,
	.attach = adv7511_bridge_attach,
};

/* -----------------------------------------------------------------------------
 * Probe & remove
 */

static const char * const adv7511_supply_names[] = {
	"avdd",
	"dvdd",
	"pvdd",
	"bgvdd",
	"dvdd-3v",
};

static const char * const adv7533_supply_names[] = {
	"avdd",
	"dvdd",
	"pvdd",
	"a2vdd",
	"v3p3",
	"v1p2",
};

static int adv7511_init_regulators(struct adv7511 *adv)
{
	struct device *dev = &adv->i2c_main->dev;
	const char * const *supply_names;
	unsigned int i;
	int ret;

	adv7511_init_pcont1_regulator(adv);
	adv7511_enable_pcont1_regulator(adv);

	if (adv->power_pcont1 != NULL)
		return 0;

	if (adv->type == ADV7511) {
		supply_names = adv7511_supply_names;
		adv->num_supplies = ARRAY_SIZE(adv7511_supply_names);
	} else {
		supply_names = adv7533_supply_names;
		adv->num_supplies = ARRAY_SIZE(adv7533_supply_names);
	}

	adv->supplies = devm_kcalloc(dev, adv->num_supplies,
				     sizeof(*adv->supplies), GFP_KERNEL);
	if (!adv->supplies)
		return -ENOMEM;

	for (i = 0; i < adv->num_supplies; i++)
		adv->supplies[i].supply = supply_names[i];

	ret = devm_regulator_bulk_get(dev, adv->num_supplies, adv->supplies);
	if (ret)
		return ret;

	return regulator_bulk_enable(adv->num_supplies, adv->supplies);
}

static void adv7511_uninit_regulators(struct adv7511 *adv)
{

	if (adv->power_pcont1 == NULL)
		regulator_bulk_disable(adv->num_supplies, adv->supplies);

	adv7511_disable_pcont1_regulator(adv);
}

static int adv7511_parse_dt(struct device_node *np,
			    struct adv7511_link_config *config)
{
	const char *str;
	int ret;

	of_property_read_u32(np, "adi,input-depth", &config->input_color_depth);
	if (config->input_color_depth != 8 && config->input_color_depth != 10 &&
	    config->input_color_depth != 12)
		return -EINVAL;

	ret = of_property_read_string(np, "adi,input-colorspace", &str);
	if (ret < 0)
		return ret;

	if (!strcmp(str, "rgb"))
		config->input_colorspace = HDMI_COLORSPACE_RGB;
	else if (!strcmp(str, "yuv422"))
		config->input_colorspace = HDMI_COLORSPACE_YUV422;
	else if (!strcmp(str, "yuv444"))
		config->input_colorspace = HDMI_COLORSPACE_YUV444;
	else
		return -EINVAL;

	ret = of_property_read_string(np, "adi,input-clock", &str);
	if (ret < 0)
		return ret;

	if (!strcmp(str, "1x"))
		config->input_clock = ADV7511_INPUT_CLOCK_1X;
	else if (!strcmp(str, "2x"))
		config->input_clock = ADV7511_INPUT_CLOCK_2X;
	else if (!strcmp(str, "ddr"))
		config->input_clock = ADV7511_INPUT_CLOCK_DDR;
	else
		return -EINVAL;

	if (config->input_colorspace == HDMI_COLORSPACE_YUV422 ||
	    config->input_clock != ADV7511_INPUT_CLOCK_1X) {
		ret = of_property_read_u32(np, "adi,input-style",
					   &config->input_style);
		if (ret)
			return ret;

		if (config->input_style < 1 || config->input_style > 3)
			return -EINVAL;

		ret = of_property_read_string(np, "adi,input-justification",
					      &str);
		if (ret < 0)
			return ret;

		if (!strcmp(str, "left"))
			config->input_justification =
				ADV7511_INPUT_JUSTIFICATION_LEFT;
		else if (!strcmp(str, "evenly"))
			config->input_justification =
				ADV7511_INPUT_JUSTIFICATION_EVENLY;
		else if (!strcmp(str, "right"))
			config->input_justification =
				ADV7511_INPUT_JUSTIFICATION_RIGHT;
		else
			return -EINVAL;

	} else {
		config->input_style = 1;
		config->input_justification = ADV7511_INPUT_JUSTIFICATION_LEFT;
	}

	of_property_read_u32(np, "adi,clock-delay", &config->clock_delay);
	if (config->clock_delay < -1200 || config->clock_delay > 1600)
		return -EINVAL;

	config->embedded_sync = of_property_read_bool(np, "adi,embedded-sync");

	/* Hardcode the sync pulse configurations for now. */
	config->sync_pulse = ADV7511_INPUT_SYNC_PULSE_NONE;
	config->vsync_polarity = ADV7511_SYNC_POLARITY_PASSTHROUGH;
	config->hsync_polarity = ADV7511_SYNC_POLARITY_PASSTHROUGH;

	return 0;
}

static int adv7511_probe(struct i2c_client *i2c, const struct i2c_device_id *id)
{
	struct adv7511_link_config link_config;
	struct adv7511 *adv7511;
	struct device *dev = &i2c->dev;
#if IS_ENABLED(CONFIG_OF_DYNAMIC)
	struct device_node *remote_node = NULL, *endpoint = NULL;
	struct of_changeset ocs;
	struct property *prop;
#endif
	unsigned int main_i2c_addr = i2c->addr << 1;
	unsigned int edid_i2c_addr = main_i2c_addr + 4;
	unsigned int cec_i2c_addr = main_i2c_addr - 2;
	unsigned int pkt_i2c_addr = main_i2c_addr - 0xa;
	int ret;

	if (!dev->of_node)
		return -EINVAL;

	adv7511 = devm_kzalloc(dev, sizeof(*adv7511), GFP_KERNEL);
	if (!adv7511)
		return -ENOMEM;
	g_adv7511 = adv7511;

	mutex_init(&adv7511->sysfs_lock);

	adv7511->i2c_main = i2c;
	adv7511->powered = false;
	adv7511->status = connector_status_disconnected;
	adv7511->power_pcont1 = NULL;
	adv7511->pcont1_en  = false;
	adv7511->irq_en = false;
	adv7511->sysfs_dev = NULL;
	adv7511->mode_vsync_polarity = false;
	adv7511->mode_hsync_polarity = false;
	adv7511->is_tmds_signal_on = false;
	adv7511->is_black_image_on = false;
	adv7511->picture_aspect = DRM_MODE_PICTURE_ASPECT_16_9;
	adv7511->is_initialized = false;
	adv7511->edid_size = 0;
	adv7511->replace_edid = false;
	adv7511->is_hdmi = true;
	adv7511->is_hdcp_authenticated = false;

	if (dev->of_node)
		adv7511->type = (enum adv7511_type)of_device_get_match_data(dev);
	else
		adv7511->type = id->driver_data;

	memset(&link_config, 0, sizeof(link_config));

	if (adv7511->type == ADV7511)
		ret = adv7511_parse_dt(dev->of_node, &link_config);
	else
		ret = adv7533_parse_dt(dev->of_node, adv7511);
	if (ret)
		return ret;

	ret = adv7511_init_regulators(adv7511);
	if (ret) {
		dev_err(dev, "failed to init regulators\n");
		return ret;
	}

	if (adv7511->addr_cec != 0)
		cec_i2c_addr = adv7511->addr_cec << 1;
	else
		adv7511->addr_cec = cec_i2c_addr >> 1;

	if (adv7511->addr_edid != 0)
		edid_i2c_addr = adv7511->addr_edid << 1;
	else
		adv7511->addr_edid = edid_i2c_addr >> 1;

	if (adv7511->addr_pkt != 0)
		pkt_i2c_addr = adv7511->addr_pkt << 1;
	else
		adv7511->addr_pkt = pkt_i2c_addr >> 1;

	/*
	 * The power down GPIO is optional. If present, toggle it from active to
	 * inactive to wake up the encoder.
	 */
	adv7511->gpio_pd = devm_gpiod_get_optional(dev, "pd", GPIOD_OUT_HIGH);
	if (IS_ERR(adv7511->gpio_pd)) {
		ret = PTR_ERR(adv7511->gpio_pd);
		goto uninit_regulators;
	}

	if (adv7511->gpio_pd) {
		mdelay(5);
		gpiod_set_value_cansleep(adv7511->gpio_pd, 0);
	}

	adv7511->regmap = devm_regmap_init_i2c(i2c, &adv7511_regmap_config);
	if (IS_ERR(adv7511->regmap)) {
		ret = PTR_ERR(adv7511->regmap);
		goto uninit_regulators;
	}

	ret = adv7511_check_i2c_bus(adv7511);
	if (ret)
		goto uninit_regulators;

	if (adv7511->type == ADV7511)
		ret = regmap_register_patch(adv7511->regmap,
					    adv7511_fixed_registers,
					    ARRAY_SIZE(adv7511_fixed_registers));
	else
		ret = adv7533_patch_registers(adv7511);
	if (ret)
		goto uninit_regulators;

	regmap_write(adv7511->regmap, ADV7511_REG_EDID_I2C_ADDR,
			edid_i2c_addr);
	regmap_write(adv7511->regmap, ADV7511_REG_PACKET_I2C_ADDR,
			pkt_i2c_addr);
	regmap_write(adv7511->regmap, ADV7511_REG_CEC_I2C_ADDR,
			cec_i2c_addr);

	adv7511_packet_disable(adv7511, 0xffff);

	adv7511->i2c_edid = i2c_new_dummy(i2c->adapter, edid_i2c_addr >> 1);
	if (!adv7511->i2c_edid) {
		ret = -ENOMEM;
		goto uninit_regulators;
	}

	if (adv7511->type == ADV7533 || adv7511->type == ADV7535) {
		ret = adv7533_init_cec(adv7511);
		if (ret)
			goto err_i2c_unregister_edid;
	}

	INIT_WORK(&adv7511->hpd_work, adv7511_hpd_work);

	if (i2c->irq) {
		init_waitqueue_head(&adv7511->wq);

		ret = devm_request_threaded_irq(dev, i2c->irq, NULL,
						adv7511_irq_handler,
						IRQF_ONESHOT, dev_name(dev),
						adv7511);
		if (ret)
			goto err_unregister_cec;

		adv7511->irq_en = true;
	}

	/* CEC is unused for now */
	regmap_write(adv7511->regmap, ADV7511_REG_CEC_CTRL,
		     ADV7511_CEC_CTRL_POWER_DOWN);

	adv7511_power_off(adv7511);

	i2c_set_clientdata(i2c, adv7511);

	if (adv7511->type == ADV7511)
		adv7511_set_link_config(adv7511, &link_config);

	/* Initialize AVI InfoFrame */
	adv7511_update_sysfs_settings(adv7511);

	/* Initialize V/H Sync Polarity */
	adv7511_update_vhsync_polarity(adv7511);

	adv7511->bridge.funcs = &adv7511_bridge_funcs;
	adv7511->bridge.of_node = dev->of_node;

	drm_bridge_add(&adv7511->bridge);

	adv7511_audio_init(dev, adv7511);

	adv7511_sysfs_register(adv7511);

	return 0;

err_unregister_cec:
	adv7533_uninit_cec(adv7511);
err_i2c_unregister_edid:
	i2c_unregister_device(adv7511->i2c_edid);
uninit_regulators:
	adv7511_uninit_regulators(adv7511);
#if IS_ENABLED(CONFIG_OF_DYNAMIC)
	endpoint = of_graph_get_next_endpoint(dev->of_node, NULL);
	if (endpoint)
		remote_node = of_graph_get_remote_port_parent(endpoint);

	if (remote_node) {
		int num_endpoints = 0;

		/*
		 * Remote node should have two endpoints (input and output: us)
		 * If remote node has more than two endpoints, probably that it
		 * has more outputs, so there is no need to disable it.
		 */
		endpoint = NULL;
		while ((endpoint = of_graph_get_next_endpoint(remote_node,
							      endpoint)))
			num_endpoints++;

		if (num_endpoints > 2) {
			of_node_put(remote_node);
			return ret;
		}

		prop = devm_kzalloc(dev, sizeof(*prop), GFP_KERNEL);
		prop->name = devm_kstrdup(dev, "status", GFP_KERNEL);
		prop->value = devm_kstrdup(dev, "disabled", GFP_KERNEL);
		prop->length = 9;
		of_changeset_init(&ocs);
		of_changeset_update_property(&ocs, remote_node, prop);
		ret = of_changeset_apply(&ocs);
		if (!ret)
			dev_warn(dev,
				"Probe failed. Remote port '%s' disabled\n",
				remote_node->full_name);

		of_node_put(remote_node);
	};
#endif

	return ret;
}

static int adv7511_remove(struct i2c_client *i2c)
{
	struct adv7511 *adv7511 = i2c_get_clientdata(i2c);

	adv7511_sysfs_unregister(adv7511);

	if (adv7511->type == ADV7533 || adv7511->type == ADV7535) {
		adv7533_detach_dsi(adv7511);
		adv7533_uninit_cec(adv7511);
	}

	adv7511_uninit_regulators(adv7511);

	drm_bridge_remove(&adv7511->bridge);

	adv7511_audio_exit(adv7511);

	if (adv7511->i2c_edid) {
		i2c_unregister_device(adv7511->i2c_edid);
		kfree(adv7511->edid);
	}

	mutex_destroy(&adv7511->sysfs_lock);

	kfree(adv7511);

	g_adv7511 = NULL;

	return 0;
}

static const struct i2c_device_id adv7511_i2c_ids[] = {
	{ "adv7511", ADV7511 },
	{ "adv7511w", ADV7511 },
	{ "adv7513", ADV7511 },
#ifdef CONFIG_DRM_I2C_ADV7533
	{ "adv7533", ADV7533 },
	{ "adv7535", ADV7535 },
#endif
	{ }
};
MODULE_DEVICE_TABLE(i2c, adv7511_i2c_ids);

static const struct of_device_id adv7511_of_ids[] = {
	{ .compatible = "adi,adv7511", .data = (void *)ADV7511 },
	{ .compatible = "adi,adv7511w", .data = (void *)ADV7511 },
	{ .compatible = "adi,adv7513", .data = (void *)ADV7511 },
#ifdef CONFIG_DRM_I2C_ADV7533
	{ .compatible = "adi,adv7533", .data = (void *)ADV7533 },
	{ .compatible = "adi,adv7535", .data = (void *)ADV7535 },
#endif
	{ }
};
MODULE_DEVICE_TABLE(of, adv7511_of_ids);

static struct mipi_dsi_driver adv7533_dsi_driver = {
	.driver.name = "adv7533",
};

static struct i2c_driver adv7511_driver = {
	.driver = {
		.name = "adv7511",
		.of_match_table = adv7511_of_ids,
	},
	.id_table = adv7511_i2c_ids,
	.probe = adv7511_probe,
	.remove = adv7511_remove,
};

static int __init adv7511_init(void)
{
	if (IS_ENABLED(CONFIG_DRM_MIPI_DSI))
		mipi_dsi_driver_register(&adv7533_dsi_driver);

	return i2c_add_driver(&adv7511_driver);
}
module_init(adv7511_init);

static void __exit adv7511_exit(void)
{
	i2c_del_driver(&adv7511_driver);

	if (IS_ENABLED(CONFIG_DRM_MIPI_DSI))
		mipi_dsi_driver_unregister(&adv7533_dsi_driver);
}
module_exit(adv7511_exit);

MODULE_AUTHOR("Lars-Peter Clausen <lars@metafoo.de>");
MODULE_DESCRIPTION("ADV7511 HDMI transmitter driver");
MODULE_LICENSE("GPL");
