/* SPDX-License-Identifier: GPL-2.0 */
/*
 * Copyright 2015, 2016, 2021 Sony Corporation
 *
 *
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/timer.h>
#include <linux/interrupt.h>
#include <linux/platform_device.h>
#include <linux/i2c.h>
#include <linux/of_platform.h>
#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/soc.h>

#include <asm/dma.h>
#include <asm/mach-types.h>

#include <linux/edma.h>
#include <linux/of_device.h>

/******************************************************************************/
/* symbolic constant definition which are refered only this file              */
/******************************************************************************/
/* Debuf log setting */
#define RICOH5B_SOUNDCARD_DEBUG RICOH5B_CONF_DEBUG

#define DIV_ID_MCLK_TO_BCK (0)
#define DIV_ID_BCK_TO_LRCK (1)
#define SLAVE_USE_ASRC_YES (1U << 31)

static int bbb_hw_params(struct snd_pcm_substream *substream,
			 struct snd_pcm_hw_params *params)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_dai *codec_dai = rtd->codec_dai;
	struct snd_soc_dai *cpu_dai = rtd->cpu_dai;
	/* codec slave, mt8590 master */
	unsigned int fmt = SND_SOC_DAIFMT_I2S | SND_SOC_DAIFMT_CBM_CFM
						| SND_SOC_DAIFMT_CONT;
	unsigned int mclk_rate;
	unsigned int rate = params_rate(params); /* data rate */
	unsigned int div_mclk_to_bck = rate > 192000 ? 2 : 4;
	unsigned int div_bck_to_lrck = 64;

#ifdef RICOH5B_SOUNDCARD_DEBUG
	pr_info("%s() rate = %d\n", __func__, rate);
#endif
	mclk_rate = rate * div_bck_to_lrck * div_mclk_to_bck;
	/* codec mclk */
	snd_soc_dai_set_sysclk(codec_dai, 0, mclk_rate, SND_SOC_CLOCK_IN);
	/* codec slave */
	snd_soc_dai_set_fmt(codec_dai, fmt);
	/* mt8590 mclk */
	snd_soc_dai_set_sysclk(cpu_dai, 0, mclk_rate, SND_SOC_CLOCK_OUT);
	/* mt8590 bck */
	snd_soc_dai_set_clkdiv(cpu_dai, DIV_ID_MCLK_TO_BCK, div_mclk_to_bck);
	/* mt8590 lrck */
	snd_soc_dai_set_clkdiv(cpu_dai, DIV_ID_BCK_TO_LRCK, div_bck_to_lrck);
	/* mt8590 master */
	snd_soc_dai_set_fmt(cpu_dai, fmt);
	/* mt8590 slave */
	snd_soc_dai_set_fmt(cpu_dai, fmt | SLAVE_USE_ASRC_YES);
						/* slave PCM can use asrc */
	return 0;
}

static struct snd_soc_ops ricoh5b_soc_ops = {
	.hw_params = bbb_hw_params
};

/*
 * The struct is used as place holder. It will be completely
 * filled with data from dt node.
 */
#define RICOH5B_CODEC "ricoh5b.1-0064"
static struct snd_soc_dai_link bbb_dai_ricoh5b[] = {
	{
	.name = "ricoh5b-pcm-in6",
	.stream_name = "ricoh5b-codec-in6",
	.platform_name = "mt8590-audio",
	.cpu_dai_name = "mt8590-i2s6",
	.codec_dai_name = "ricoh5b-i2s",
	.codec_name = RICOH5B_CODEC,
		.dai_fmt = SND_SOC_DAIFMT_I2S
				 | SND_SOC_DAIFMT_CBS_CFS
				 | SND_SOC_DAIFMT_GATED,
	.ops = &ricoh5b_soc_ops,
	},
};

/* ricoh5b bbb audio machine driver */
static struct snd_soc_card bbb_soc_card = {
	.name = "ricoh5b-soc-card",
	.dai_link = bbb_dai_ricoh5b,
	.num_links = ARRAY_SIZE(bbb_dai_ricoh5b),
};

static int ricoh5b_bbb_probe(struct platform_device *pdev)
{
	int ret = 0;
	struct snd_soc_card *drvdata = &bbb_soc_card;

#ifdef RICOH5B_SOUNDCARD_DEBUG
	pr_info("%s : Start\n", __func__);
#endif

	drvdata->dev = &pdev->dev;
	ret = snd_soc_register_card(drvdata);

	return ret;
}

static int ricoh5b_bbb_remove(struct platform_device *pdev)
{
	struct snd_soc_card *card = platform_get_drvdata(pdev);

#ifdef RICOH5B_SOUNDCARD_DEBUG
	pr_info("%s : Start\n", __func__);
#endif

	snd_soc_unregister_card(card);

	return 0;
}

static struct platform_driver ricoh5b_bbb_driver = {
	.probe = ricoh5b_bbb_probe,
	.remove = ricoh5b_bbb_remove,
	.driver = {
	    .name = "ricoh5b_bbb",
	    .owner = THIS_MODULE,
	    .pm = &snd_soc_pm_ops,
	    .of_match_table = of_match_ptr(ricoh5b_bbb_dt_ids),
	},
};

static struct platform_device *bbb_snd_device;

static int __init bbb_init(void)
{

#ifdef RICOH5B_SOUNDCARD_DEBUG
	pr_info("%s : Start\n", __func__);
#endif

	return platform_driver_register(&ricoh5b_bbb_driver);
}

static void __exit bbb_exit(void)
{

#ifdef RICOH5B_SOUNDCARD_DEBUG
	pr_info("%s : Start\n", __func__);
#endif
	platform_driver_unregister(&ricoh5b_bbb_driver);

	platform_device_unregister(bbb_snd_device);
}

module_init(bbb_init);
module_exit(bbb_exit);

MODULE_DESCRIPTION("RICOH 5B Tuner soundcard driver");
MODULE_LICENSE("GPL");
