/*
 * Copyright (C) 2019 MediaTek Inc.
 *
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License version 2 as published by the
 * Free Software Foundation.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program.
 * If not, see <http://www.gnu.org/licenses/>.
 */

#include <adsp_ipi.h>
#include <linux/device.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/of.h>
#include <linux/platform_device.h>
#include <linux/printk.h>

struct hifi4dsp_audio_jack_priv {
	struct device *dev;
	bool plugged;
};

enum {
	IPI_MSG_AUDIO_JACK_PLUGGED = 0,
	IPI_MSG_AUDIO_JACK_UNPLUGGED,
};

static struct hifi4dsp_audio_jack_priv *g_priv;

static void audio_jack_ipi_msg_dispatcher(int _id, void *data,
	unsigned int len)
{
	u8 msg = *((u8 *)data);

	switch (msg) {
	case IPI_MSG_AUDIO_JACK_PLUGGED:
		pr_debug("%s plugged\n", __func__);
		if (g_priv)
			g_priv->plugged = true;
		break;
	case IPI_MSG_AUDIO_JACK_UNPLUGGED:
		pr_debug("%s unplugged\n", __func__);
		if (g_priv)
			g_priv->plugged = false;
		break;
	default:
		break;
	}
}

static ssize_t plugged_show(struct device *dev,
	struct device_attribute *attr, char *buf)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct hifi4dsp_audio_jack_priv *priv = platform_get_drvdata(pdev);

	return scnprintf(buf, PAGE_SIZE, "%u\n", priv->plugged);
}

static DEVICE_ATTR_RO(plugged);

static int hifi4dsp_audio_jack_probe(struct platform_device *pdev)
{
	struct hifi4dsp_audio_jack_priv *priv;

	pr_notice("%s", __func__);

	priv = devm_kzalloc(&pdev->dev,
		sizeof(struct hifi4dsp_audio_jack_priv), GFP_KERNEL);
	if (!priv)
		return -ENOMEM;

	priv->dev = &pdev->dev;
	platform_set_drvdata(pdev, priv);

	g_priv = priv;

	adsp_ipi_registration(ADSP_IPI_AUDIO_JACK,
		audio_jack_ipi_msg_dispatcher, "audio_jack_ipi");

	return device_create_file(priv->dev, &dev_attr_plugged);
}

static int hifi4dsp_audio_jack_remove(struct platform_device *pdev)
{
	struct hifi4dsp_audio_jack_priv *priv = platform_get_drvdata(pdev);

	dev_dbg(priv->dev, "%s", __func__);

	device_remove_file(priv->dev, &dev_attr_plugged);

	return 0;
}

#ifdef CONFIG_OF
static const struct of_device_id hifi4dsp_audio_jack_of_ids[] = {
	{ .compatible = "mediatek,mt8570-audio-jack", },
	{}
};
#endif

static struct platform_driver hifi4dsp_audio_jack_driver = {
	.probe = hifi4dsp_audio_jack_probe,
	.remove = hifi4dsp_audio_jack_remove,
	.driver = {
		.name = "mt8570-audio-jack",
		.owner = THIS_MODULE,
#ifdef CONFIG_OF
		.of_match_table = hifi4dsp_audio_jack_of_ids,
#endif
	},
};

static int __init hifi4dsp_audio_jack_driver_init(void)
{
	return platform_driver_register(&hifi4dsp_audio_jack_driver);
}

static void __exit hifi4dsp_audio_jack_driver_exit(void)
{
	platform_driver_unregister(&hifi4dsp_audio_jack_driver);
}

module_init(hifi4dsp_audio_jack_driver_init);
module_exit(hifi4dsp_audio_jack_driver_exit);

MODULE_DESCRIPTION("HIFI4DSP Audio Jack Driver");
