/*
 * Copyright (C) 2019 MediaTek Inc.
 *
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License version 2 as published by the
 * Free Software Foundation.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program.
 * If not, see <http://www.gnu.org/licenses/>.
 */

#include <linux/device.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/of.h>
#include <linux/platform_device.h>
#include <linux/printk.h>
#include <linux/of_gpio.h>
#include <linux/of_irq.h>
#include <linux/interrupt.h>
#include <linux/wait.h>

#define DRV_PREFIX "mediatek"
#define DRV_NAME "mt8570_sleep_eint"

struct hifi4dsp_sleep_eint_priv {
	struct device *dev;
	struct mutex lock;
	int suspend_resume_gpio;
	int resume_irq;
	int gpiopin;
	int trigger_type;
};

static struct hifi4dsp_sleep_eint_priv *priv;
static wait_queue_head_t wait_dsp_resume_queue;
static wait_queue_head_t wait_dsp_suspend_queue;
static bool waitResumeDone;
static bool waitSuspendNotDone;

static long WAIT_TIMEOUT_RESUME = HZ / 2;
static long WAIT_TIMEOUT_SUSPEND = HZ / 50;

static int triggerSuspendFlag;
static int triggerResumeFlag;

#define GPIO_HIGH_LEVEL 1
#define GPIO_LOW_LEVEL 0

#define SLEEP_EINT_TAG "[DSP Sleep/EINT] "
#define SLEEP_EINT_NOTICE(fmt, args...) pr_notice(SLEEP_EINT_TAG fmt, ##args)
#define SLEEP_EINT_INFO(fmt, args...) pr_info(SLEEP_EINT_TAG fmt, ##args)


#ifdef CONFIG_OF
static const struct of_device_id hifi4dsp_sleep_eint_of_ids[] = {
	{ .compatible = "mediatek,mt8570-sleep-eint", },
	{}
};
#endif

static bool trigger_dsp_suspend_nolock(void)
{
	int ret = 0;

	if (priv->suspend_resume_gpio) {
		gpio_direction_output(priv->suspend_resume_gpio,
							GPIO_HIGH_LEVEL);

		ret = wait_event_interruptible_timeout(wait_dsp_suspend_queue,
				waitSuspendNotDone, WAIT_TIMEOUT_SUSPEND);

		waitSuspendNotDone = false;
		if (ret > 0) {
			SLEEP_EINT_NOTICE("wait dsp suspend fail\n");
			return false;
		}
	} else
		return false;

	return true;
}

static bool trigger_dsp_resume_nolock(void)
{
	int ret = 0;

	if (priv->suspend_resume_gpio) {
		gpio_direction_output(priv->suspend_resume_gpio,
							GPIO_LOW_LEVEL);

		ret = wait_event_interruptible_timeout(wait_dsp_resume_queue,
				waitResumeDone, WAIT_TIMEOUT_RESUME);

		waitResumeDone = false;
		if (ret <= 0) {
			SLEEP_EINT_NOTICE("wait dsp resume fail\n");
			return false;
		}
	} else {
		SLEEP_EINT_NOTICE("resume gpio err\n");
		return false;
	}

	return true;
}

bool trigger_dsp_suspend(void)
{
	bool ret = true;

	mutex_lock(&priv->lock);

	triggerSuspendFlag = 1;
	SLEEP_EINT_INFO("AP will trigger dsp enter suspend\n");
	ret = trigger_dsp_suspend_nolock();

	triggerSuspendFlag = 0;
	mutex_unlock(&priv->lock);

	return true;
}

bool trigger_dsp_resume(void)
{
	bool ret = true;

	mutex_lock(&priv->lock);

	triggerResumeFlag = 1;
	SLEEP_EINT_INFO("AP will trigger dsp exit suspend\n");
	ret = trigger_dsp_resume_nolock();

	triggerResumeFlag = 0;
	mutex_unlock(&priv->lock);
	return ret;
}

static irqreturn_t dsp_resume_state_handler(int irq, void *data)
{
	disable_irq_nosync(priv->resume_irq);
	if (priv->trigger_type == IRQ_TYPE_LEVEL_HIGH) {
		irq_set_irq_type(priv->resume_irq, IRQ_TYPE_LEVEL_LOW);
		priv->trigger_type = IRQ_TYPE_LEVEL_LOW;
	} else {
		irq_set_irq_type(priv->resume_irq, IRQ_TYPE_LEVEL_HIGH);
		priv->trigger_type = IRQ_TYPE_LEVEL_HIGH;
	}

	if (triggerResumeFlag) {
		waitResumeDone = true;
		wake_up_interruptible(&wait_dsp_resume_queue);
	}

	if (triggerSuspendFlag) {
		waitSuspendNotDone = true;
		wake_up_interruptible(&wait_dsp_suspend_queue);
	}

	SLEEP_EINT_INFO("dsp has resume\n");
	enable_irq(priv->resume_irq);
	return IRQ_HANDLED;
}

static int hifi4dsp_sleep_eint_probe(struct platform_device *pdev)
{

	struct device_node *node = NULL;
	int ret = 0;

	SLEEP_EINT_INFO("%s() enter.\n", __func__);

	priv = devm_kzalloc(&pdev->dev,
		sizeof(struct hifi4dsp_sleep_eint_priv), GFP_KERNEL);
	if (!priv)
		return -ENOMEM;

	priv->dev = &pdev->dev;

	node = of_find_matching_node(node, hifi4dsp_sleep_eint_of_ids);

	if (node) {
		priv->suspend_resume_gpio =
				of_get_named_gpio(node,
				"suspend-resume-gpio", 0);
		ret = gpio_request(priv->suspend_resume_gpio,
				"suspend-resume_pin");

		if (ret) {
			SLEEP_EINT_NOTICE("gpio_request fail, ret(%d)\n", ret);
			return -ENODEV;
		}

		gpio_direction_output(priv->suspend_resume_gpio,
				GPIO_LOW_LEVEL);

		priv->resume_irq = irq_of_parse_and_map(node, 0);
		if (priv->resume_irq < 0)
			SLEEP_EINT_NOTICE("hifi4dsp not find irq\n");
		priv->trigger_type = irq_get_trigger_type(priv->resume_irq);

		priv->gpiopin = of_get_named_gpio(node, "resume-gpio", 0);
		if (priv->gpiopin < 0) {
			SLEEP_EINT_NOTICE("hifi4dsp not find gpio\n");
			return -ENODEV;
		}

		ret = gpio_request(priv->gpiopin, "resume-gpio");

		if (ret) {
			SLEEP_EINT_NOTICE("gpio_request fail, ret(%d)\n", ret);
			return -ENODEV;
		}

		ret = request_irq(priv->resume_irq, dsp_resume_state_handler,
				IRQ_TYPE_NONE, "Sleep-eint", NULL);
		if (ret) {
			SLEEP_EINT_NOTICE("EINT IRQ LINE NOT AVAILABLE\n");
			return -ENODEV;
		}
		gpio_direction_input(priv->gpiopin);
	} else {
		SLEEP_EINT_NOTICE("can't get node from dts\n");
		return -ENODEV;
	}

	mutex_init(&priv->lock);
	init_waitqueue_head(&wait_dsp_resume_queue);
	init_waitqueue_head(&wait_dsp_suspend_queue);
	platform_set_drvdata(pdev, priv);
	return 0;
}

static int hifi4dsp_sleep_eint_remove(struct platform_device *pdev)
{
	mutex_destroy(&priv->lock);
	return 0;
}


static struct platform_driver hifi4dsp_sleep_eint_driver = {
	.probe = hifi4dsp_sleep_eint_probe,
	.remove = hifi4dsp_sleep_eint_remove,
	.driver = {
		.name = DRV_NAME,
		.owner = THIS_MODULE,
#ifdef CONFIG_OF
		.of_match_table = hifi4dsp_sleep_eint_of_ids,
#endif
	},
};

static int __init hifi4dsp_sleep_eint_driver_init(void)
{
	int ret = 0;

	pr_notice("%s\n", __func__);

	ret = platform_driver_register(&hifi4dsp_sleep_eint_driver);
	if (ret) {
		pr_notice("%s driver register fail, ret %d\n", __func__, ret);
		goto err_drv;
	}

	return ret;

err_drv:
	return ret;
}

static void __exit hifi4dsp_sleep_eint_driver_exit(void)
{
	pr_notice("%s", __func__);

	platform_driver_unregister(&hifi4dsp_sleep_eint_driver);
}

module_init(hifi4dsp_sleep_eint_driver_init);
module_exit(hifi4dsp_sleep_eint_driver_exit);

MODULE_DESCRIPTION("MT8570 Sleep Eint Driver");

