// SPDX-License-Identifier: GPL-2.0

/* notify_hightemp.c: Notify High Temp Driver
 * Copyright 2020 Sony Home Entertainment & Sound Products Inc.
 */
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/delay.h>
#include <linux/gpio.h>
#include <linux/interrupt.h>
#include <linux/of_irq.h>
#include <linux/of_gpio.h>
#include <linux/of_device.h>
#include <linux/platform_device.h>
#include <linux/reboot.h>
#include <linux/device_cooling.h>
#include <linux/kobject.h>
#include <linux/string.h>
#include <linux/sysfs.h>
#include <linux/init.h>

#define DRV_NAME "notify_hightemp"

static struct kobject *hightemp_notify_kobj;
static int is_hot_state = false;

static int thermal_hot_pm_notify(struct notifier_block *nb, unsigned long event,
				 void *dummy)
{
	static bool is_already_too_hot = false;

	if (event && !is_already_too_hot)
	{
		pr_warn("%s:%d  NOTIFY hot\n", __func__, __LINE__);
		is_already_too_hot = true;
		is_hot_state = true;
		sysfs_notify(hightemp_notify_kobj, NULL, "is_hot");
	}
	else if (!event && is_already_too_hot)
	{
		pr_warn("%s:%d  NOTIFY cool\n", __func__, __LINE__);
		is_already_too_hot = false;
		is_hot_state = false;
		sysfs_notify(hightemp_notify_kobj, NULL, "is_hot");
	}

	return NOTIFY_OK;
}

static struct notifier_block thermal_hot_pm_notifier = {
	.notifier_call = thermal_hot_pm_notify,
};

static ssize_t hot_show(struct kobject *kobj, struct kobj_attribute *attr, char *buf)
{
	return sprintf(buf, "%d\n", is_hot_state);
}

static struct kobj_attribute hot_attribute = __ATTR(is_hot, 0444, hot_show, NULL);

static struct attribute *attrs[] = {
	&hot_attribute.attr,
	NULL,
};

static struct attribute_group attr_group = {
	.attrs = attrs,
};

static int notify_hightemp_probe(struct platform_device *p_dev)
{
	int ret;

	hightemp_notify_kobj = kobject_create_and_add("notify_hightemp", kernel_kobj);
	if (!hightemp_notify_kobj)
		return -ENOMEM;

	ret = sysfs_create_group(hightemp_notify_kobj, &attr_group);
	if (ret)
		kobject_put(hightemp_notify_kobj);

	register_devfreq_cooling_notifier(&thermal_hot_pm_notifier);

	return 0;
}

static int notify_hightemp_remove(struct platform_device *p_dev)
{
	unregister_devfreq_cooling_notifier(&thermal_hot_pm_notifier);
	if (hightemp_notify_kobj)
		kobject_put(hightemp_notify_kobj);

	return 0;
}

static const struct of_device_id notify_hightemp_dt_ids[] = {
	{
	.compatible = "sony,notify_hightemp",
	},
	{}};
MODULE_DEVICE_TABLE(of, notify_hightemp_dt_ids);

static struct platform_driver notify_hightemp_driver = {
	.probe = notify_hightemp_probe,
	.remove = notify_hightemp_remove,
	.driver = {
		.name = DRV_NAME,
		.owner = THIS_MODULE,
		.of_match_table = notify_hightemp_dt_ids,
	},
};

module_platform_driver(notify_hightemp_driver);
MODULE_DESCRIPTION("Notify High Temp Driver");
MODULE_AUTHOR("Sony Home Entertainment & Sound Products Inc.");
MODULE_LICENSE("GPL");
MODULE_VERSION("v1.00");
