// SPDX-License-Identifier: GPL-2.0
/*
 * sony_sw_pll.c : Sony software pll driver
 *
 *  Copyright 2020,2021 Sony Home Entertainment & Sound Products Inc.
 */

#include <linux/module.h>
#include <linux/kthread.h>
#include <linux/sched.h>
#include <linux/jiffies.h>
#include <linux/of_device.h>
#include <linux/platform_device.h>
#include <linux/io.h>
#include <linux/slab.h>
#include <misc/sw_pll.h>

#define DRV_NAME "sony_sw_pll"

//temporary defines
#define SAI6_BASE_ADDR 0x30060000
#define ANATOP_BASE_ADDR 0x30360000
#define FSL_SAI_TBCTN  0x78 /* SAI Transmit Bit Counter Register */
#define FSL_SAI_TTCAP  0x7C /* SAI Transmit Timestamp Capture */
#define FSL_SAI_RBCTN  0xf8 /* SAI Receive Bit Counter Register */
#define FSL_SAI_RTCAP  0xfc /* SAI Receive Timestamp Capture */
#define FSL_SAI_TTSR   0x74 /* SAI Transmit Timestamp Register */
/* SPDIF interrupt mask define */
#define INT_DPLL_LOCKED			(1 << 20)
#define INT_CNEW			(1 << 17)
#define INT_BIT_ERR			(1 << 14)
#define INT_LOSS_LOCK			(1 << 2)
/* sw pll interrupt mask define */
#define INT_SW_PLL_FS			(0x1)
#define INT_SW_PLL_STATE		(0x2)
#define INT_SW_PLL_ALERT		(0x4)
#define INT_SW_PLL_MUTE			(0x8)
#define INT_SW_PLL_SPDIF		(0x10)
#define INT_SW_PLL_UNLOCK		(0x20)
#define INT_SW_PLL_READ_CBITS		(0x40)
#define INT_SW_PLL_READ_SPDIF_STAT	(0x80)
#define INT_SW_PLL_LOCK			(0x100)
#define INT_SW_PLL_UNLOCK_2TIMES	(0x200)
#define INT_SW_PLL_UNLOCK_SPDIF		(0x400)
#define INT_SW_PLL_SPDIF_BITERR		(0x800)
#define INT_SW_PLL_SPDIF_RESET		(0x1000)

//for long logging
//#define LONG_TIME_LOG

#ifdef LONG_TIME_LOG
typedef struct SW_PLL_Logging {
	uint32_t  RBCR;
	uint32_t  RBCTR;
} SW_PLL_LOGGING;
#define LOG_MAX_SIZE (1000000)
#define LOG_SIZE_PER_LINE 25
#define LOG_SIZE_TOTAL 3001
#define LOG_READ_SIZE 100
#else
typedef struct SW_PLL_Logging {
	uint32_t  RBCR;
	uint32_t  RBCTR;
	uint32_t  TBCR;
	uint32_t  TBCTR;
	uint8_t   State;
	int32_t   PLL2_Freq;
	int32_t   Phase_Err;
	uint8_t   Detect_fs;
	uint32_t  Alert;
	uint8_t   Mute_Req;
	uint8_t   Mute_Ack;
	uint8_t   Fixed_fs;
	uint8_t   Unlock;
	uint64_t  spdif_cbits;
	int32_t   spdif_dpll_lock;
	int32_t   spdif_srpc_lock;
	int32_t   spdif_open_stat;
	uint32_t  spdif_irq_stat;
	uint8_t   opt_i2s_mode;
	uint32_t  spdif_event_log;
} SW_PLL_LOGGING;
#define LOG_MAX_SIZE (250000)
#define LOG_SIZE_PER_LINE 150
#define LOG_SIZE_TOTAL 1600
#define LOG_READ_SIZE 10
#endif
#define LOG_LEVEL_INTERVAL 3

typedef struct SW_PLL_Info {
	int16_t   PLL_DSM;
	uint8_t   Detect_fs;
	uint8_t   State;
	uint8_t   Fixed_fs;
} SW_PLL_INFO;

struct sony_spdif_priv {
	void __iomem *sai6_base;
	void __iomem *ccm_base;
	struct platform_device *pdev;
	int is_enable;
	int is_log_trigger;
	struct task_struct *kthread_tsk;
	struct kobject *sw_pll_kobj;
	struct kobject *sw_pll_isr_kobject;
	struct workqueue_struct *wq;
	struct work_struct work;
	atomic_t irq_event;
	SW_PLL_STAT send_data;
	int log_counter;
	int log_read_counter;
	int log_read_size;
	int log_size;
	SW_PLL_LOGGING *logdata;
	atomic_t spdif_event;
	spinlock_t sw_pll_lock;
	struct mutex dsm_lock;
	atomic_t fixed_on;
	atomic_t fixed_off;
	struct mutex fixed_lock;
	uint8_t prev_fixed_fs;
	uint8_t unlock_log;
	u64(*spdif_cbits_read_func)(int, int*);
	int(*spdif_lock_read_func)(int*, int*, int*, uint32_t*);
	int (*spdif_softreset_func)(void);
	uint64_t  spdif_cbits;
	uint8_t opt_i2s_mode;
	SW_PLL_INFO transport_data;
	struct mutex transport_lock;
	int32_t   spdif_dpll_lock;
	int32_t   spdif_srpc_lock;
	int32_t   spdif_open_stat;
	uint32_t  spdif_irq_stat;
	uint32_t  spdif_event_log;
	struct mutex sw_pll_workque_lock;
};
static struct sony_spdif_priv *sw_pll_priv;
static SW_PLL_PARA Para;
static BLOCKING_NOTIFIER_HEAD(sw_pll_notify_list);

int register_sw_pll_notifier(struct notifier_block *nb)
{
	return blocking_notifier_chain_register(&sw_pll_notify_list, nb);
}
EXPORT_SYMBOL_GPL(register_sw_pll_notifier);

int unregister_sw_pll_notifier(struct notifier_block *nb)
{
	return blocking_notifier_chain_unregister(&sw_pll_notify_list, nb);
}
EXPORT_SYMBOL_GPL(unregister_sw_pll_notifier);

int sw_pll_set_spdif_event(uint32_t spdif_event_flag)
{
	u32 tmp_event;

	tmp_event = atomic_read(&sw_pll_priv->spdif_event);
	atomic_set(&sw_pll_priv->spdif_event, tmp_event | spdif_event_flag);

	return 0;
}

EXPORT_SYMBOL_GPL(sw_pll_set_spdif_event);

int sw_pll_set_fixed(uint8_t is_fixed)
{
	mutex_lock(&sw_pll_priv->fixed_lock);
	if (is_fixed) {
		atomic_set(&sw_pll_priv->fixed_on, 1);
		pr_debug("sw_pll: fixed_on\n");

	}
	else {
		atomic_set(&sw_pll_priv->fixed_off, 1);
		pr_debug("sw_pll: fixed_off\n");
	}
	mutex_unlock(&sw_pll_priv->fixed_lock);

	return 0;
}

EXPORT_SYMBOL_GPL(sw_pll_set_fixed);

static int sw_pll_set_mute_ack(uint8_t mute_ack)
{
	Para.Mute_Ack = mute_ack;
	return 0;
}

EXPORT_SYMBOL_GPL(sw_pll_set_mute_ack);

int sw_pll_get_dsm_lock(void)
{
	mutex_lock(&sw_pll_priv->dsm_lock);

	return 0;
}

EXPORT_SYMBOL_GPL(sw_pll_get_dsm_lock);

int sw_pll_get_dsm_unlock(void)
{
	mutex_unlock(&sw_pll_priv->dsm_lock);
	return 0;
}

EXPORT_SYMBOL_GPL(sw_pll_get_dsm_unlock);

int sw_pll_register_cbits_read_func(u64 (*cbits_read)(int, int*))
{
	sw_pll_priv->spdif_cbits_read_func = cbits_read;

	return 0;
}

EXPORT_SYMBOL_GPL(sw_pll_register_cbits_read_func);

int sw_pll_register_lock_read_func(int (*lock_read)(int*, int*, int*, uint32_t*))
{
	sw_pll_priv->spdif_lock_read_func = lock_read;

	return 0;
}

EXPORT_SYMBOL_GPL(sw_pll_register_lock_read_func);

int sw_pll_register_softreset_func(int (*softreset)(void))
{
	sw_pll_priv->spdif_softreset_func = softreset;

	return 0;
}
EXPORT_SYMBOL_GPL(sw_pll_register_softreset_func);

int sw_pll_set_mode(uint8_t opt_i2s_mode)
{
	sw_pll_priv->opt_i2s_mode = opt_i2s_mode;
	pr_debug("sw_pll: set mode %d\n", opt_i2s_mode);

	return 0;
}

EXPORT_SYMBOL_GPL(sw_pll_set_mode);

static void PkAve1(uint32_t BCK, uint32_t* Pk, uint32_t* Cent, int Init)
{
	static uint32_t  fMin=0, fMax=0, Pending=0;
	if (Init == 1) {
		fMin = BCK;
		fMax = BCK;
	}
	if (Pending == 0 && (fMin > BCK || fMax < BCK)) {
		Pending = 1;
		*Cent = (fMax + fMin) >> 1;
		if (fMin > BCK)
			fMin = BCK;
		if (fMax < BCK)
			fMax = BCK;
	} else {
		Pending = 0;
		if (fMin > BCK)
			fMin = BCK;
		if (fMax < BCK)
			fMax = BCK;
		*Cent = (fMax + fMin) >> 1;
	}
	*Pk = fMax - fMin;
}

static void PkAveN(uint32_t* BCK_Freq, uint32_t* Pk, uint32_t* Cent, int posBgn, int N)
{
	uint32_t  fMin, fMax, *BCK, BCK_i;
	int  i;

	BCK = BCK_Freq+posBgn-1;
	fMin = fMax = BCK[1];
	for (i = 2; i <= N; i++) {
		BCK_i = BCK[i];
		if (fMin > BCK_i)
			fMin = BCK_i;
		else if (fMax < BCK_i)
			fMax = BCK_i;
	}
	*Pk = fMax - fMin;
	*Cent = (fMax + fMin) >> 1;
}

static void CalcThrWid(uint32_t Pk, int* Thr, int32_t* ThrWid)
{
	int  dev;

	if ((Thr[0]&0xf)==2) dev = 1790; // nBCK=2
	else                 dev = 3579; // nBCK=1

	if ((int)Pk <= Thr[3]) // Pk1
		*ThrWid = ((Pk * Thr[11]) >> 13) + Thr[2]; // RatWid, ThrNar
	else if (dev < (int)Pk) // dev
		*ThrWid = ((Pk - dev) >> 1) + Thr[2];
	else
		*ThrWid = ((dev - Pk) >> 1) + Thr[2];
}

static void CompIntvl(uint32_t* BCK_in, uint32_t* BCK_out, uint32_t* RBCTR, uint32_t Cent, int N)
{
	int  p;
	int64_t  Intvl_df;

	for (p=1; p<=N; p++) {
		Intvl_df = ((int)RBCTR[p]-2400000);
		Intvl_df *= (int)(BCK_in[p]-Cent);
		BCK_out[p] = Intvl_df/2400000+BCK_in[p];
	}
}

static void pol2ok(int8_t* pol, int8_t* ok, int* EstCent, int N)
{
	int  p;

	for (p=0; p<=N; p++)
		ok[p] = 0;
	for (p=1; p<=N; p++) {
		if (pol[p]==1 && pol[p+1]==-1 && p<=N-1)
			ok[p] = 1;
		if (pol[p]==1 && pol[p+1]==0 && pol[p+2]==-1 && p<=N-2)
			ok[p] = ok[p+1] = 2;
		if (pol[p]==1 && pol[p+1]==0 && pol[p+2]==0 && pol[p+3]==-1 && p<=N-3)
			ok[p] = ok[p+1] = ok[p+2] = 3;
		if (pol[p]==1 && pol[p+1]==0 && pol[p+2]==0 && pol[p+3]==0 && p<=N-3)
			ok[p] = ok[p+1] = ok[p+2] = ok[p+3] = *EstCent = 4;
		if (pol[p]==0 && pol[p+1]==0 && pol[p+2]==0 && pol[p+3]==-1 && p<=N-3)
			ok[p-1] = ok[p] = ok[p+1] = ok[p+2] = *EstCent = 4;
	}
	if (pol[N]==1) ok[N] = 1;
	if (pol[1]==-1) ok[0] = 1;
	if (pol[N-1]==1 && pol[N]==0) ok[N-1] = ok[N] = 1;
	if (pol[1]==0 && pol[2]==-1) ok[0] = ok[1] = 1;
	if (pol[N-1]==1 && pol[N]==1) ok[N] = 2;
	if (pol[1]==-1 && pol[2]==-1) ok[0] = 2;
	if (pol[N-2]==1 && pol[N-1]==0 && pol[N]==0) ok[N-2] = ok[N-1] = ok[N] = 1;
	if (pol[1]==0 && pol[2]==0 && pol[3]==-1) ok[0] = ok[1] = ok[2] = 1;
}

static void CalcAve(int* gpend, int* gpnum, int32_t* slp, int grp, int32_t* slpAve)
{
	int32_t  sum=0;
	int  idx;

	for (idx=gpend[grp]-gpnum[grp]+1; idx<=gpend[grp]; idx++)
		sum += slp[idx];
	*slpAve = sum/gpnum[grp];
}

static void proc0(uint32_t* BCK_Freq, uint32_t* BCK_Comp, uint32_t* RBCTR_df, uint32_t* Pk, uint32_t Cent0, uint32_t* Cent1, int* EstCent, int posBgn, int posN, int* Thr)
{
	int32_t  ThrNar, ThrWid, Thrsh;
	uint32_t  *BCK, *RBCTR;
	int  near[posMax+1], near2[posMax+1], p, i, brk=0;
	int  pos, difMin, difSum, posCnt, posMin, difpos;

	BCK = BCK_Freq+posBgn-1;
	RBCTR = RBCTR_df+posBgn-1;

	CalcThrWid(*Pk, Thr, &ThrWid);
	ThrNar = Thr[2];

	do {
		if (*EstCent==4) {
			brk = 40;
			break;
		}

		near[0] = near[1] = near2[0] = near2[1] = 0;
		for (p=2; p<=posN; p++) {
			if (abs((int)BCK[p]-(int)BCK[p-1])<ThrNar && abs((int)BCK[p]-(int)Cent0)<ThrWid)
				near[p] = near[p-1]+1;
			else
				near[p] = 0;
			if (abs((int)BCK[p]-(int)BCK[p-1])<ThrNar)
				near2[p] = near2[p-1]+1;
			else
				near2[p] = 0;
		}

		for (p=posN; p>=2; p--) {
			if ((near[p]>=2 && p-near[p]>1 && BCK[p-near[p]-1]>BCK[p-near[p]]+ThrNar && p<posN && BCK[p+1]<BCK[p]-ThrNar) || near[p]>=5) {
				*EstCent = 4;
				for (*Cent1=0, i=p-near[p]; i<=p; i++) {
					*Cent1 += BCK[i];
				}
				*Cent1 /= near[p]+1;
				brk = 43;
				break;
			}
		}
		if (brk!=0) break;

		for (p=posN; p>=2; p--) {
			if ((near2[p]>=3 && p-near2[p]>1 && BCK[p-near2[p]-1]>BCK[p-near2[p]]+ThrNar && p<posN && BCK[p+1]<BCK[p]-ThrNar) || near2[p]>=5) {
				*EstCent = 4;
				for (*Cent1=0, i=p-near2[p]; i<=p; i++) {
					*Cent1 += BCK[i];
				}
				*Cent1 /= near2[p]+1;
				brk = 46;
				break;
			}
		}
		if (brk!=0) break;

		for (p=posN; p>=2; p--) {
			if (near[p]>=2 && ((p-near[p]>1 && BCK[p-near[p]-1]>BCK[p-near[p]]+ThrNar && p==posN) || (p-near[p]<=1 && p<posN && BCK[p+1]<BCK[p]-ThrNar))) {
				*EstCent = 3;
				for (*Cent1=0, i=p-near[p]; i<=p; i++) {
					*Cent1 += BCK[i];
				}
				*Cent1 /= near[p]+1;
				brk = 30;
				break;
			}
		}
		if (brk!=0) break;

		for (p=posN; p>=2; p--) {
			if (near2[p]>=3 && ((p-near2[p]>1 && BCK[p-near2[p]-1]>BCK[p-near2[p]]+ThrNar && p==posN) || (p-near2[p]<=1 && p<posN && BCK[p+1]<BCK[p]-ThrNar))) {
				*EstCent = 3;
				for (*Cent1=0, i=p-near2[p]; i<=p; i++) {
					*Cent1 += BCK[i];
				}
				*Cent1 /= near2[p]+1;
				brk = 33;
				break;
			}
		}
		if (brk!=0) break;

		for (p=posN; p>=2; p--) {
			if (p>2 && p<posN && near[p-1]==0 && near[p]==1 && near[p+1]==0 && (p==3 || near[p-2]==0) && (p==posN-1 || near[p+2]==0) && BCK[p-2]>BCK[p-1]+ThrNar && BCK[p]>BCK[p+1]+ThrNar) {
				*EstCent = 2;
				*Cent1 = (BCK[p-1]+BCK[p])/2;
				brk = 20;
				break;
			}
		}
		if (brk!=0) break;

		for (p=posN; p>=2; p--) {
			if (p>4 && p<posN && near[p-2]==1 && near[p-1]==0 && near[p]==1 && BCK[p-2]>BCK[p-1]+ThrNar && BCK[p-4]>BCK[p-3]+ThrNar && (BCK[p]<BCK[p+1]-ThrNar || (BCK[p]>BCK[p+1]+ThrNar && abs((int)BCK[p-2]-(int)Cent0)<abs((int)BCK[p]-(int)Cent0)))) {
				*EstCent = 2;
				*Cent1 = (BCK[p-3]+BCK[p-2])/2;
				brk = 23;
				break;
			}
			if (p>4 && p<posN && near[p-2]==1 && near[p-1]==0 && near[p]==1 && BCK[p-2]>BCK[p-1]+ThrNar && BCK[p]>BCK[p+1]+ThrNar && (BCK[p-4]<BCK[p-3]-ThrNar || (BCK[p-4]>BCK[p-3]+ThrNar && abs((int)BCK[p-2]-(int)Cent0)>=abs((int)BCK[p]-(int)Cent0)))) {
				*EstCent = 2;
				*Cent1 = (BCK[p-1]+BCK[p])/2;
				brk = 26;
				break;
			}
		}
		if (brk!=0) break;

		for (p=posN; p>=2; p--) {
			if ((p==posN && p>2 && near[p-1]==0 && near[p]==1 && (p==3 || near[p-2]==0) && BCK[p-2]>BCK[p-1]+ThrNar) || (p==2 && p<posN && near[p]==1 && near[p+1]==0 && (p==posN-1 || near[p+2]==0) && BCK[p]>BCK[p+1]+ThrNar)) {
				*EstCent = 2;
				*Cent1 = (BCK[p-1]+BCK[p])/2;
				brk = 10;
				break;
			}
		}
		if (brk!=0) break;

		for (p=posN; p>=2; p--) {
			if (((p>4 && p==posN && BCK[p-4]>BCK[p-3]+ThrNar) || (p==4 && p<posN && BCK[p]<BCK[p+1]-ThrNar)) && near[p-2]==1 && near[p-1]==0 && near[p]==1 && BCK[p-2]>BCK[p-1]+ThrNar) {
				*EstCent = 2;
				*Cent1 = (BCK[p-3]+BCK[p-2])/2;
				brk = 13;
				break;
			}
			if (((p>4 && p==posN && BCK[p-4]<BCK[p-3]+ThrNar) || (p==4 && p<posN && BCK[p]>BCK[p+1]-ThrNar)) && near[p-2]==1 && near[p-1]==0 && near[p]==1 && BCK[p-2]>BCK[p-1]+ThrNar) {
				*EstCent = 2;
				*Cent1 = (BCK[p-1]+BCK[p])/2;
				brk = 16;
				break;
			}
		}
		if (brk!=0) break;
	} while (0);

	if (*EstCent==0) {
		CompIntvl(BCK, BCK_Comp, RBCTR, Cent0, posN);
		PkAveN(BCK_Comp, Pk, Cent1, 1, posN);
		i = *Cent1;

		if (posN>=7) {
			*Cent1=Cent0;
		} else {
			CalcThrWid(*Pk, Thr, &Thrsh);

			difMin = 9999;
			difSum = posCnt = posMin = 0;
			for (pos=1; pos<=posN; pos++) {
				difpos = BCK_Comp[pos]-*Cent1;
				if (difpos<Thrsh && difpos>-Thrsh) {
					difSum += difpos;
					posCnt++;
				}
			}
			if (posCnt>=1) {
				*Cent1 += difSum/posCnt;
			}
			for (pos=1; pos<=posN; pos++) {
				difpos = BCK_Comp[pos]-*Cent1;
				if (difpos<Thrsh && difpos>-Thrsh && abs(difpos)<difMin) {
					difMin = difpos;
					posMin = pos;
				}
			}
		}
	} else {
		CompIntvl(BCK, BCK_Comp, RBCTR, *Cent1, posN);
	}
	Thr[13] = brk;
}

static void proc1(uint32_t* BCK_Comp, uint32_t Pk, uint32_t Cent1, uint32_t* Cent2, int* EstCent, int posN, int* Thr)
{
	int32_t  dif[posMax+1], itg[posMax+1], fitg[idxMax], slp[idxMax] = {0}, slpmin[grpMax];
	int32_t  Thrsh, itpl, slpAve=0;
	int8_t  pol[posMax+1], ok[posMax+1];
	int16_t  fpos[idxMax], fok[idxMax];
	int  gpend[grpMax], gpnum[grpMax];
	int  pos, i, idx, idxN, grp, grpN;
	int  dst, brk=0;
	static int  fs_Hz[8]={0, 32000, 44100, 48000, 88200, 96000, 176400, 192000};
	int  grp0=0, grp1=0, grpB=7, idx0, idx1, slpNew;
	int  cnt0, pbgn=0, p, difmn=0, posmn=0, difcent=0;
	int  maxgok, eqgok, gokgp[grpMax], gok[grpMax], gok0[grpMax];
	int  maxdst, eqdst, dstgp[grpMax], gdst[grpMax], ratio4=4;
	int  maxsum, eqsum, sumgp[grpMax], gsum[grpMax];
	static int  okWeit[5]={0,1,4,6,8};
	int  max2sum, outstd=0, grpT, dev;
	int  ParModSlpP, ParModSlpM, ParAveSlp, ParChgSlp, ParBigPk;
	static int  ThrDif, ThrRat, init=1, ofst, boost, knee;
	static int  ThrModSlpP, ThrModSlpM, ThrAveSlp, ThrChgSlp;
	static int  gokWeit, dstWeit, numWeit, ThrBigPk=1522;

	if (init!=0) {
		init = 0;
		if ((Thr[0]&0xf)==2) dev = 1790; // nBCK=2
		else                 dev = 3579; // nBCK=1
		gokWeit = (Thr[0]>>12) & 0xf;
		dstWeit = (Thr[0]>>8) & 0xf;
		numWeit = (Thr[0]>>4) & 0xf;
		ParBigPk = (Thr[6]>>12) & 0xf;
		ThrDif = (Thr[6]>>8) & 0xf;
		ThrRat = Thr[6] & 0xff;
		ThrBigPk = (80+ParBigPk)*dev/100;
		ofst = (Thr[7]>>8) & 0xff;
		boost = (Thr[7]>>4) & 0xf;
		knee = Thr[7] & 0xf;
		okWeit[4] = (Thr[8]>>8) & 0xff;
		okWeit[3] = (Thr[8]>>4) & 0xf;
		okWeit[2] = Thr[8] & 0xf;
		ParModSlpP = (Thr[9]>>8) & 0xff;
		ParModSlpM = Thr[9] & 0xff;
		ThrModSlpP = (ParModSlpP * fs_Hz[Thr[12]])>>12;
		ThrModSlpM = (ParModSlpM * fs_Hz[Thr[12]])>>12;
		ParAveSlp = (Thr[10]>>8) & 0xff;
		ParChgSlp = Thr[10] & 0xff;
		ThrAveSlp = (ParAveSlp * fs_Hz[Thr[12]])>>12;
		ThrChgSlp = (ParChgSlp * fs_Hz[Thr[12]])>>12;
	}

	*Cent2 = Cent1;
	dif[0] = itg[0] = 0;
	pol[0] = 0;
	if (*EstCent==0) {
		CalcThrWid(Pk, Thr, &Thrsh);
	} else {
		Thrsh = Thr[2];
	}

	for (pos=1; pos<=posN; pos++) {
		dif[pos] = BCK_Comp[pos]-Cent1;
		if (dif[pos]>Thrsh) pol[pos] = 1;
		else if (dif[pos]>-Thrsh) pol[pos] = 0;
		else pol[pos] = -1;
		itg[pos] = itg[pos-1]+dif[pos];
	}
	if (*EstCent==0) {
		for (cnt0=0, pos=1; pos<=posN; pos++) {
			if (pol[pos]==0) cnt0++;
			if (cnt0>=2 && (pos==posN || pol[pos]!=0)) {
				if (pol[pos]!=0) pbgn = pos-cnt0;
				else             pbgn = pos-cnt0+1;
				for (difmn=9999, p=pbgn; p<pbgn+cnt0; p++) {
					if (difmn>abs(dif[p])) {
						difmn = abs(dif[p]);
						posmn = p;
						difcent = dif[p];
					}
				}
				for (p=pbgn; p<pbgn+cnt0; p++) {
					if (p!=posmn) {
						if (dif[p]-difcent>Thr[2]) pol[p] = 1;
						if (dif[p]-difcent<-Thr[2]) pol[p] = -1;
					}
				}
			}
			if (pol[pos]!=0) cnt0 = 0;
		}
	}

	pol2ok(pol, ok, EstCent, posN);
	for (idx=0, pos=0; pos<=posN; pos++) {
		fpos[idx] = pos;
		fitg[idx] = itg[pos];
		fok[idx] = ok[pos];
		if (idx>0)
			slp[idx] = (fitg[idx]-fitg[idx-1])/(fpos[idx]-fpos[idx-1]);
		idx++;
	}
	idxN = idx;

	for (idx=1; idx<=idxN-2; idx++) {
		itpl = (fitg[idx+1]-fitg[idx-1])/(fpos[idx+1]-fpos[idx-1])*(fpos[idx]-fpos[idx-1])+fitg[idx-1];
		if (itpl>fitg[idx]+Thr[4]) {
			for (i=idx; i<=idxN-2; i++) {
				fpos[i] = fpos[i+1];
				fitg[i] = fitg[i+1];
				fok[i] = fok[i+1];
				slp[i] = slp[i+1];
			}
			slp[idx] = (fitg[idx]-fitg[idx-1])/(fpos[idx]-fpos[idx-1]);
			idxN--;
			if (idx>1) idx -= 2;
			else idx--;
		}
	}

	for (grp=0, slpmin[0]=slp[1], idx=2; idx<=idxN; idx++) {
		if (idx==idxN || slpmin[grp]-Thr[5]>slp[idx]) { // ThrSlp
			gpend[grp]=idx-1;
			if (grp==0) gpnum[grp] = idx-1;
			else gpnum[grp] = gpend[grp]-gpend[grp-1];
			if (idx==idxN) break;
			if (grp<grpMax-1)
				grp++;
			slpmin[grp] = slp[idx];
		} else if (slpmin[grp]>slp[idx]) {
			slpmin[grp] = slp[idx];
		}
	}
	grpN = grp+1;

	for (grp=0; grp<grpN; grp++) {
		gok[grp] = 0;
		gok0[grp] = 0;
		for (idx=gpend[grp]-gpnum[grp]; idx<=gpend[grp]; idx++) {
			gok[grp] += okWeit[fok[idx]];
			if (fok[idx]==0)
				gok0[grp]++;
		}
	}
	for (maxgok=0, eqgok=1, grp=0; grp<=grpN-1; grp++) {
		if (maxgok==gok[grp]) {
			eqgok++;
			gokgp[eqgok] = grp;
		}
		if (maxgok<gok[grp]) {
			maxgok = gok[grp];
			eqgok = 1;
			gokgp[1] = grp;
		}
	}

	for (grp=0; grp<grpN; grp++) {
		dst = fpos[gpend[grp]]-fpos[gpend[grp]-gpnum[grp]];
		if (dst>knee)
			gdst[grp] = dst*boost-ofst;
		else
			gdst[grp] = dst;
	}
	for (maxdst=0, eqdst=1, grp=0; grp<=grpN-1; grp++) {
		if (maxdst==gdst[grp]) {
			eqdst++;
			dstgp[eqdst] = grp;
		}
		if (maxdst<gdst[grp]) {
			maxdst = gdst[grp];
			eqdst = 1;
			dstgp[1] = grp;
		}
	}

	for (maxsum=0, eqsum=1, grp=0; grp<=grpN-1; grp++) {
		gsum[grp] = gok[grp]*gokWeit+gdst[grp]*dstWeit+gpnum[grp]*numWeit;
		if (maxsum==gsum[grp]) {
			eqsum++;
			sumgp[eqsum] = grp;
		}
		if (maxsum<gsum[grp]) {
			maxsum = gsum[grp];
			eqsum = 1;
			sumgp[1] = grp;
		}
	}

	if (eqsum==1) {
		for (max2sum=0, grp=0; grp<grpN; grp++) {
			if (max2sum<gsum[grp] && grp!=sumgp[1]) {
				max2sum = gsum[grp];
			}
		}
		if (maxsum>max2sum+ThrDif && maxsum>(max2sum*ThrRat)>>3) {
			outstd = 1;
		}
	}
	if (*EstCent>0)
		outstd = 1;

	if (eqgok==1 && eqdst==1 && gokgp[1]==dstgp[1]) {
		grpB = gokgp[1];
		brk = 5;
	} else if (eqgok==1 && eqdst!=1) {
		for (i=1; i<=eqdst; i++) {
			if (gokgp[1]==dstgp[i]) {
				grpB = gokgp[1];
				brk = 5;
				break;
			}
		}
	} else if (eqgok!=1 && eqdst==1) {
		for (i=1; i<=eqgok; i++) {
			if (dstgp[1]==gokgp[i]) {
				grpB = dstgp[1];
				brk = 5;
				break;
			}
		}
	}
	if (brk!=5) {
		if (eqsum==1) {
			grpB = sumgp[1];
			brk = 10;
		} else {
			for (maxdst=0, i=1; i<=eqsum; i++) {
				grp = sumgp[i];
				if (maxdst<=gdst[grp]) {
					maxdst = gdst[grp];
					grpB = grp;
				}
			}
			brk = 15;
		}
	}
	CalcAve(gpend, gpnum, slp, grpB, &slpAve);

	grpT = posN>>1;
	if (Thr[12]!=0 && grpN<=grpT && (slpAve>(ThrModSlpP*ratio4)>>2 || -slpAve>(ThrModSlpM*ratio4)>>2) && outstd==0) {
		if (grpB==0 && slpmin[0]>0) {
			grp0 = 0; grp1 = 1;
			brk += 20;
		} else if (grpB==grpN-1 && slpmin[grpB]<0) {
			grp0 = grpB-1; grp1 = grpB;
			brk += 20;
		} else if (0<grpB && grpB<grpN-1) {
			if (slpmin[grpB]<0) {
				grp0 = grpB-1; grp1 = grpB;
			} else {
				grp0 = grpB; grp1 = grpB+1;
			}
			brk += 20;
		}
	} else if (Thr[12]!=0 && grpN>grpT && outstd==0) {
		if (grpB==0) {
			grp0 = 0; grp1 = 1;
			brk += 40;
		} else if (grpB==grpN-1) {
			grp0 = grpB-1; grp1 = grpB;
			brk += 40;
		} else if (gok0[grpB-1]==0 && gok0[grpB+1]!=0) {
			grp0 = grpB-1; grp1 = grpB;
			brk += 40;
		} else if (gok0[grpB-1]!=0 && gok0[grpB+1]==0) {
			grp0 = grpB; grp1 = grpB+1;
			brk += 40;
		} else if (gsum[grpB-1]>gsum[grpB+1]) {
			grp0 = grpB-1; grp1 = grpB;
			brk += 60;
		} else if (gsum[grpB-1]<gsum[grpB+1]) {
			grp0 = grpB; grp1 = grpB+1;
			brk += 60;
		} else if (gdst[grpB-1]>gdst[grpB+1]) {
			grp0 = grpB-1; grp1 = grpB;
			brk += 80;
		} else if (gdst[grpB-1]<gdst[grpB+1]) {
			grp0 = grpB; grp1 = grpB+1;
			brk += 80;
		} else if (slpmin[grpB]<0) {
			grp0 = grpB-1; grp1 = grpB;
			brk += 80;
		} else {
			grp0 = grpB; grp1 = grpB+1;
			brk += 80;
		}
	}
	if (grp1>0) {
		idx0 = gpend[grp0]-gpnum[grp0];
		idx1 = gpend[grp1];
		slpNew = (fitg[idx1]-fitg[idx0])/(fpos[idx1]-fpos[idx0]);
		if (abs(slpNew) < ThrAveSlp) {
			slpAve = slpNew;
		} else {
			slpAve = 0;
			brk += 2;
		}
	}
	if (*EstCent>0 || abs(slpAve)<ThrChgSlp || (int)Pk<ThrBigPk)
		*Cent2 += slpAve;

	Thr[15] = brk;
	Thr[16] = grpN;
}

static void SWPLL_Freq(SW_PLL_PARA* Para_p, SW_PLL_STAT* Stat_p, uint32_t BCK_FreqR, uint32_t RBCTR_diff, uint32_t* BCK_200, uint8_t* Frq_Reset)
{
	static uint32_t  BCK_Freq[posMax+1], BCK_Comp[posMax+1], RBCTR_df[posMax+1];
	static uint32_t  Pk, Cent0, Cent1, Cent2, dev, BCK_last=0, RstCntr=0;
	static int  posIn=1, posN=1, posBgn=1, EstCent=0, Init=1, Thr[17];
	static int  RBCR_1st, RBCR_2nd, RBCTR_1est, RBCTR_2nd;
	static int  prev=0, LPFz=0, LPFcf=409; // LPFcf=0.1
	uint32_t  RBCR_acc, RBCTR_acc, temp;

	if (Init == 1) {
		Init = 0;
		Thr[0] = Para_p->Frq_Para[0];
		if ((Thr[0]&0xf) == 2) dev = 1790;
		else                   dev = 3579;
		Thr[1] = (dev * Para_p->Frq_Para[1]) >> 12;
		Thr[2] = (dev * Para_p->Frq_Para[2]) >> 12;
		Thr[3] = (dev << 12) / (4096 + Para_p->Frq_Para[3]);
		Thr[4] = (Thr[2] * Para_p->Frq_Para[4]) >> 12;
		Thr[5] = (Thr[2] * Para_p->Frq_Para[5]) >> 12;
		Thr[6] = Para_p->Frq_Para[6];
		Thr[7] = Para_p->Frq_Para[7];
		Thr[8] = Para_p->Frq_Para[8];
		Thr[9] = Para_p->Frq_Para[9];
		Thr[10] = Para_p->Frq_Para[10];
		Thr[11] = Para_p->Frq_Para[3];
		*Frq_Reset = 1;
	}

	if (*Frq_Reset != 0) {
		*Frq_Reset = 0;
		posIn = 1;
		posN = 1;
		posBgn = 1;
		EstCent = 0;
		BCK_Freq[0] = BCK_Freq[1] = BCK_FreqR;
		RBCTR_df[0] = RBCTR_df[1] = RBCTR_diff;
		PkAve1(BCK_FreqR, &Pk, &Cent0, 1);
		*BCK_200 = BCK_FreqR;
		BCK_last = BCK_FreqR;
		RstCntr=0;
		RBCR_1st = Para_p->RBCR;
		return;
	}
	Thr[12] = Stat_p->Detect_fs;
	Thr[13] = Thr[15] = Thr[16] = 0;

	++posIn;
	if (posIn < posMax) {
		BCK_Freq[posIn] = BCK_FreqR;
		RBCTR_df[posIn] = RBCTR_diff;
	} else {
		*BCK_200 = BCK_last;
		return;
	}

	PkAve1(BCK_FreqR, &Pk, &Cent0, 0);
	if (Pk > dev + Thr[1]) {
		EstCent = 0;
		posBgn = posIn;
		posN = 1;
		RstCntr = 6;
		PkAve1(BCK_FreqR, &Pk, &Cent0, 1);
		*BCK_200 = BCK_last;
		RBCR_1st = Para_p->RBCR;
		return;
	}

	++posN;
	if (posN == 2) {
		RBCR_2nd = Para_p->RBCR;
		RBCTR_2nd = Para_p->RBCTR;
	}

	if (posN == 7) {
		temp = RBCR_2nd - RBCR_1st;
		if (Cent0 > 40000) {
			temp = ((uint64_t)temp << 31) / Cent0;
			RBCTR_1est = RBCTR_2nd - temp;
		} else {
			RBCTR_1est = 0x80000000;
		}
	}
	if (posN >= 7 && RBCTR_1est != (int)0x80000000) {
		RBCR_acc = Para_p->RBCR - RBCR_1st;
		RBCTR_acc = Para_p->RBCTR - RBCTR_1est;
		if (RBCTR_acc < 40000)
			temp = 40000;
		else
			temp = RBCTR_acc;
		temp = ((uint64_t)RBCR_acc << 31) / temp;

		if (posN == 8)
			LPFz = (temp + prev) >> 1;
		if (posN >= 9)
			LPFz += (((((int)temp + prev) >> 1) - LPFz) * LPFcf) >> 12;
		prev = temp;
		if (posN >= 8)
			Cent0 = LPFz;
	}

	if (RstCntr>0) {
		RstCntr--;
		*BCK_200 = BCK_last;
		return;
	}

	if (EstCent < 4 || posN <= 7) {
		proc0(BCK_Freq, BCK_Comp, RBCTR_df, &Pk, Cent0, &Cent1, &EstCent, posBgn, posN, Thr);
	} else {
		CompIntvl(BCK_Freq+posBgn-1, BCK_Comp, RBCTR_df+posBgn-1, Cent1, posN);
	}

	if (posN >= 7) {
		proc1(BCK_Comp, Pk, Cent1, &Cent2, &EstCent, posN, Thr);
		*BCK_200 = Cent2;
	} else {
		*BCK_200 = Cent1;
	}
	BCK_last = *BCK_200;
}


static void SWPLL_Phase(SW_PLL_PARA* Para_p, SW_PLL_STAT* Stat_p, int32_t* Acc_In, uint8_t* Ph_Reset)
{
	uint32_t  BCK_DSM[8] = {0, 699, 466, 466, 233, 233, 117, 117};  // 32bit
	uint16_t  Ratio_fs[8] = {0, 8, 11, 12, 22, 24, 44, 48};
	uint8_t  fs;
	int32_t  R14;
	int8_t   S14;
	int32_t  T14;
	int32_t  U14;
	int32_t  V14;
	int32_t  W14;
	int32_t  X14;
	int8_t   Y14;
	int32_t  Z14;
	int32_t  AA14;
	int8_t   AB14;
	int8_t   AC14;
	int32_t  AD14;
	int8_t   AE14;
	int32_t  AF14;
	int8_t   AG14;
	int8_t   AH14;
	int8_t   AI14;
	int32_t  AJ14;
	int32_t  AK14;
	int32_t  AL14;
	int8_t   AM14;
	int8_t   AN14;
	int32_t  AO14;
	int32_t  AP14;
	int32_t  AQ14;
	int32_t  AR14;
	int8_t   AS14;
	int8_t   AT14;
	int32_t  AU14;
	int32_t  AV14;
	int32_t  AW14;
	int8_t   AX14;

	static int32_t  R13;
	static int32_t  T13;
	static int32_t  V13;
	static int32_t  W13;
	static int32_t  Z13;
	static int32_t  AA13;
	static int8_t   AB13;
	static int8_t   AC13;
	static int8_t   AH13;
	static int8_t   AI13;
	static int32_t  AJ13;
	static int32_t  AL13;
	static int8_t   AM13;
	static int32_t  AR13;
	static int8_t   AT13;
	static int8_t   AX13;

	static uint16_t  V4, V5, V6, V7, V8, V9;
	static uint16_t  Y3, Y4, Y5, Y6, AB2, AB3, AB4, AB5;
	static int32_t  Y7;

	fs = Stat_p->Detect_fs;
	R14 = Stat_p->Phase_Err;

	if (*Ph_Reset != 0) {  // Reset after each Transition
		V4 = Para_p->Ph_Para[0];
		V5 = Para_p->Ph_Para[1];
		V6 = Para_p->Ph_Para[2];
		V7 = Para_p->Ph_Para[3];
		V8 = Para_p->Ph_Para[4];
		V9 = Para_p->Ph_Para[5];
		Y3 = Para_p->Ph_Para[6];
		Y4 = Para_p->Ph_Para[7];
		Y5 = Para_p->Ph_Para[8];
		Y6 = Para_p->Ph_Para[9];
		AB2 = Para_p->Ph_Para[10];
		AB3 = Para_p->Ph_Para[11];
		AB4 = Para_p->Ph_Para[12] * Ratio_fs[fs];
		AB5 = Para_p->Ph_Para[13] * Ratio_fs[fs];
		Y7 = (BCK_DSM[fs] << 8) / V6;
		R13 = 0;
		T13 = 0;
		V13 = 0;
		W13 = 0;
		Z13 = 0;
		AA13 = 0;
		AB13 = 0;
		AC13 = 0;
		AH13 = 0;
		AI13 = 0;
		AJ13 = 0;
		AL13 = 0;
		AM13 = 0;
		AR13 = 0;
		AT13 = 0;
		AX13 = 0;
		*Ph_Reset = 0;
	}

	S14 = (T13-R13>Y3 && T13-R14<Y4) ? 1 : 0;
	T14 = (S14!=0) ? (R14+T13) >> 1 : R13;
	U14 = T14-T13;
	V14 = (T14>V13+1) ? T14-1 : ((T14<V13) ? T14 : V13);
	W14 = (T14>W13+Y5) ? T14-Y5 : ((T14<W13) ? T14 : W13);
	X14 = (AT13==0) ? V14-V13 : W14-W13;
	Y14 = (abs((Y6==0) ? U14 : X14)>=2 && AC13==0) ? 1 : 2;
	Z14 = (AB13!=0 || AX13!=0) ? ((Y14==2) ? ((Y6==0) ? U14 : X14) : 0) : ((Y14==2) ? Z13+((Y6==0) ? U14 : X14) : Z13);
	AA14 = (AB13!=0 || AX13!=0) ? ((Y14==2) ? 1 : 0) : ((Y14==2) ? ((AA13<32767) ? AA13+1 : AA13) : 0);
	AB14 = (Y14==1 || (Y14==2 && abs(Z14)>=V4 && AA14>=V5) || (AH13>0 && abs(T14)<AB4)) ? 1 : 0;
	AC14 = (Y14==1) ? 1 : (((Y14==2 && abs(Z14)>=V4 && AA14>=V5) || AX13!=0) ? 0 : AC13);
	AD14 = (AA14==0) ? 0 : BCK_DSM[fs]*abs(Z14)/AA14;
	AE14 = ((T14>=0 && Z14>0) || (T14<0 && Z14<0)) ? 1 : 0;
	AF14 = max(AJ13, (int32_t)V7);
	AG14 = (Y14==2 && abs(Z14)>=V4 && AA14>=V5) ? ((((AE14!=0 && AD14>=V7) || (AE14==0 && AD14>AF14+(AF14 >> 3)+1))) ? 1 : ((AM13==0 || AX13!=0) ? 2 : 0)) : 0;
	AH14 = (AG14==2 && AE14!=0 && abs(T14)>AB4 && AH13<V9) ? AH13+1 : ((abs(T14)<AB4) ? 0 : AH13);
	AI14 = (AG14==2 && AE14!=0 && abs(T14)>AB4) ? ((T14<0) ? -1 : 1) : AI13;
	AJ14 = (AG14==2) ? max(AJ13,((AH14>2) ? V8*(1<<(AH14-3)) : 1)) : (((T14<0 && AI14==1) || (T14>0 && AI14==-1)) ? 1 : AJ13);
	AK14 = (AG14==1) ? Y7*Z14/AA14 : 0;
	AL14 = (AG14==1 && abs(AK14)>abs((AX13!=0) ? 0 : AL13)) ? AK14 : ((AX13!=0) ? 0 : AL13);
	AM14 = (AG14==1) ? 1 : ((AX13!=0) ? 0 : AM13);
	AN14 = (AG14==2) ? ((AE14!=0) ? ((abs(T14)>AB4) ? 1 : 2) : ((AD14>AJ14+1) ? 3 : 0)) : 0;
	AO14 = (AD14+AJ14)*256*((AN14==1) ? ((Z14>0) ? 1 : -1) : 0);
	AP14 = AD14*256*((AN14==2) ? ((Z14>0) ? 1 : -1) : 0);
	AQ14 = (AD14-AJ14)*256*((AN14==3) ? ((Z14>0) ? 1 : -1) : 0);
	AR14 = (AM14!=0) ? ((AG14==1 && abs(AK14)>abs((AX13!=0) ? 0 : AL13)) ? 1 : AR13+1) : 0;
	AS14 = (AM14!=0) ? (((X14>0 && AL14<0) || (X14<0 && AL14>0) || AR14>V6) ? 1 : 2) : 0;
	AT14 = (AS14==1) ? ((AR14>V6) ? 0 : 1) : AT13;
	AU14 = (AS14==1) ? abs(AL14)*V6 : 0;
	AV14 = (AS14==2) ? AL14 : ((AS14==1) ? ((abs(T14)>AB5) ? ((T14>=0) ? ((AL14>0) ? ((AU14>AB2) ? AB2 : AU14) : ((AU14>AB3) ? AB3 : AU14)) : 0) : 0) : 0);
	AW14 = (AS14==1) ? ((abs(T14)>AB5) ? ((T14>=0) ? 0 : ((AL14<0) ? ((AU14>AB2) ? -AB2 : -AU14) : ((AU14>AB3) ? -AB3 : -AU14))) : 0) : 0;
	AX14 = (AS14==1) ? 1 : 0;

	*Acc_In += AO14+AP14+AQ14+AV14+AW14;

	if (S14 != 0)
		Stat_p->Alert |= PHASE_SPIKE;

// Save Previous Value
	R13 = R14;
	T13 = T14;
	V13 = V14;
	W13 = W14;
	Z13 = Z14;
	AA13 = AA14;
	AB13 = AB14;
	AC13 = AC14;
	AH13 = AH14;
	AI13 = AI14;
	AJ13 = AJ14;
	AL13 = AL14;
	AM13 = AM14;
	AR13 = AR14;
	AT13 = AT14;
	AX13 = AX14;
}

static void SW_PLL_Core(SW_PLL_PARA* Para_p, SW_PLL_STAT* Stat_p)
{
	const uint8_t  Pre_Div[8] = {0, 3, 2, 2, 2, 2, 2, 2};
	const uint16_t Main_Div[8] = {0, 262, 241, 262, 241, 262, 241, 262};
	const uint8_t  TCR2_Div[8] = {0, 3, 3, 3, 1, 1, 0, 0};
	const uint8_t  Comp_fs[8] = {0, 12, 8, 8, 4, 4, 2, 2};

	static uint8_t  State = 0;
	static uint8_t  prev_fs = 0;
	static uint32_t R_Counter = 0;
	static uint32_t T_Counter = 0;
	static int32_t  PLL2_Freq = 9437 * 256; // 48kHz
	static uint32_t prev_RBCR = 0x80000000;
	static uint32_t prev_RBCTR = 0x80000000;
	static uint32_t prev_TBCR = 0;
	static uint32_t prev_TBCTR = 0;
	static uint8_t  T1_State = 0;
	static uint8_t  Out_fs = 3; // 48kHz
	static uint8_t  Timer_Reset = 0;
	static uint32_t Timer_Offset = 0;
	static int32_t  Phase_Offset = 0;
	static int32_t  prev_Phase = 0;

	static uint8_t  Stable_State = 0;
	static uint8_t  Ph_Reset = 0;
	static uint8_t  Frq_Reset = 0;

	uint8_t  prev_State;
	uint32_t TBCR_diff;
	uint32_t Interval;
	uint32_t RBCR_diff;
	uint32_t RBCTR_diff;
	uint32_t BCK_200;
	uint32_t BCK_FreqT;
	uint32_t BCK_FreqR;
	uint8_t  Loop_Skip;
	uint32_t State_Timer;
	uint8_t  Timeout;
	uint8_t  tmp_fs;
	uint8_t  FB_fs;
	uint8_t  Active_State = 0;

	int32_t  Acc_In = 0;
	int32_t  temp;
	int32_t  Fwd_DSM = 0;
	uint32_t compare = 0;

// [1] init
	Stat_p->Alert = 0;
	Stat_p->Detect_fs = 0;
	Stat_p->Phase_Err = 0;

// [2] Counter diff
	TBCR_diff = Para_p->TBCR - prev_TBCR;
	Interval = Para_p->TBCTR - prev_TBCTR;
	if (Interval < 40000)
		temp = 40000;
	else
		temp = Interval;
	BCK_FreqT = ((uint64_t)TBCR_diff << 32) / temp;
	prev_TBCR = Para_p->TBCR;
	prev_TBCTR = Para_p->TBCTR;

	RBCR_diff = Para_p->RBCR - prev_RBCR;
	RBCTR_diff = Para_p->RBCTR - prev_RBCTR;
	if (RBCTR_diff < 40000)
		temp = 40000;
	else
		temp = RBCTR_diff;
	BCK_FreqR = ((uint64_t)RBCR_diff << 31) / temp;
	prev_RBCR = Para_p->RBCR;
	prev_RBCTR = Para_p->RBCTR;

	if (Interval < 400000) {
		Loop_Skip = 1;
		Stat_p->Alert |= INTVL_UNDER;
	}
	else if(Interval > 8000000){
		Loop_Skip = 1;
		Stat_p->Alert |= INTVL_OVER;
	}
	else{
		Loop_Skip = 0;
	}

	if (Timer_Reset != 0)
		Timer_Offset = Para_p->TBCTR;
	State_Timer = Para_p->TBCTR - Timer_Offset;

	if (State == Invalid)
		compare = Para_p->Trans_Time[2];
	if (State == Change)
		compare = Para_p->Trans_Time[3];
	if (State == Transition) {
		if (T1_State == 0)
			compare = Para_p->Trans_Time[0];
		else
			compare = Para_p->Trans_Time[1];
	}
	if (State_Timer >= compare)
		Timeout = 1;
	else
		Timeout = 0;

	temp = Para_p->RBCR - (Para_p->TBCR << 1);
	if (Stable_State == 0)
		Phase_Offset = temp;
	Stat_p->Phase_Err = (temp - Phase_Offset) >> 1;

	temp = Stat_p->Phase_Err - prev_Phase;
	prev_Phase = Stat_p->Phase_Err;
	if ((uint32_t)abs(temp) > Para_p->Trans_Time[4])
		Stat_p->Alert |= PHASE_JUMP;

// [3] fs Detect
	Stat_p->Detect_fs = 0;

	for (tmp_fs = 7; tmp_fs > 0; tmp_fs--) {
		if (BCK_FreqR > Para_p->Valid_Lower[tmp_fs]) {
			if (BCK_FreqR < Para_p->Valid_Upper[tmp_fs]) {
				if (tmp_fs != prev_fs) {
					R_Counter = 1;
					prev_fs = tmp_fs;
				}
				else if(++R_Counter >= Para_p->Valid_Count){
					R_Counter = Para_p->Valid_Count;
					Stat_p->Detect_fs = tmp_fs;
				}
				break;
			}
			prev_fs = 0;
			Stat_p->Alert |= REF_INVAL;
			break;
		}
	}
	if (tmp_fs == 0) {
		prev_fs = 0;
		Stat_p->Alert |= REF_INVAL;
	}

//[3]-2 freq t
	do {
		FB_fs = 0;
		if (Stat_p->Detect_fs != 0) {
			if (BCK_FreqT > Para_p->Valid_Lower[Stat_p->Detect_fs]) {
				if (BCK_FreqT < Para_p->Valid_Upper[Stat_p->Detect_fs]) {
					if (++T_Counter >= Para_p->Valid_Count) {
						T_Counter = Para_p->Valid_Count;
						FB_fs = Stat_p->Detect_fs;
					}
					break;
				}
			}
		}
		T_Counter = 0;
		Stat_p->Alert |= FB_INVAL;
	}while(0);

// [4]State Machine :
	prev_State = State;

	switch(prev_State) {
	case Fixed:
		if (Para_p->Fixed_fs == 0)
			State = Invalid;
		break;
	case Invalid:
		if (Para_p->Fixed_fs != 0)
			State = Fixed;
		else if(Stat_p->Detect_fs != 0 && (Para_p->Mute_Ack != 0 || Timeout !=0))
			State = Change;
		break;
	case Change:
		if (Para_p->Fixed_fs != 0)
			State = Fixed;
		else if (Stat_p->Detect_fs == 0)
			State = Invalid;
		else if (FB_fs != 0)
			State = Transition;
		break;
	case Transition:
		if (Para_p->Fixed_fs != 0)
			State = Fixed;
		else if (Stat_p->Detect_fs == 0 || FB_fs == 0)
			State = Invalid;
		else {
			if (T1_State != 0 && Timeout != 0)
				State = Stable;
			if (T1_State == 0 && Timeout != 0)
				T1_State = 1;
		}
		break;
	case Stable:
		if (Para_p->Fixed_fs != 0)
			State = Fixed;
		else if (Stat_p->Detect_fs == 0 || FB_fs == 0)
			State = Invalid;
		break;
	default:
		pr_err("invalid prev_State\n");
	}

	switch(State) {
	case Fixed:
		Timer_Reset = 0;
		Stat_p->Mute_Req = 0;
		Active_State = 0;
		Stable_State = 0;
		break;
	case Invalid:
		if (prev_State != Invalid)
			Timer_Reset = 1;
		else
			Timer_Reset = 0;
		Stat_p->Mute_Req = 1;
		Active_State = 0;
		Stable_State = 0;
		Frq_Reset = 1;
		break;
	case Change:
		if (prev_State != Change) {
			Timer_Reset = 1;
		}
		else {
			Timer_Reset = 0;
			if (Timeout != 0 )
				Stat_p->Alert |= FB_TIMEOUT;
		}
		Stat_p->Mute_Req = 1;
		Out_fs = Stat_p->Detect_fs;
		Active_State = 0;
		Stable_State = 0;
		break;
	case Transition:
		if (prev_State != Transition) {
			Timer_Reset = 1;
			T1_State = 0;
		}
		else {
			Timer_Reset = 0;
		}
		if ((Para_p->Cond_Flag & MUTE_REL) != 0)
			Stat_p->Mute_Req = 1;
		else
			Stat_p->Mute_Req = 0;
		Active_State = 1;
		Stable_State = 0;
		Ph_Reset = 1;
		break;
	case Stable:
		Stat_p->Mute_Req = 0;
		Active_State = 1;
		Stable_State = 1;
		break;
	default:
		pr_err("invalid state\n");
	}

// [5]Feedforward Process
	if (State == Change || State == Transition || State == Invalid) {
		SWPLL_Freq(Para_p, Stat_p, BCK_FreqR, RBCTR_diff, &BCK_200, &Frq_Reset);

		temp = BCK_200 * Comp_fs[Out_fs];
		temp = (uint64_t)temp * 25 / 384;
		temp = (temp + 1) >> 1;
		Fwd_DSM = temp - (Main_Div[Out_fs] << 16);

// for Monitor from here. Can be removed
		temp = (Para_p->Frq_Para[18] << 16) | Para_p->Frq_Para[17];
		temp = temp * Comp_fs[Out_fs];
		temp = (uint64_t)temp * 25 / 384;
		temp = (temp + 1) >> 1;
		temp = temp - (Main_Div[Out_fs] << 16);
		Para_p->Frq_Para[17] = temp & 0xffff;
// for Monitor to here
	}

// [6]Feedback Loop
	if (Active_State && !Loop_Skip) {
		if (Stable_State != 0)
			SWPLL_Phase(Para_p, Stat_p, &Acc_In, &Ph_Reset);
	}

// [7]Output Process
	if (State == Change || State == Transition)
		PLL2_Freq = Fwd_DSM << 8;
	else
		PLL2_Freq += Acc_In;

	if (PLL2_Freq > 8388352) {  // 32767 * 2^8
		PLL2_Freq = 8388352;
		Stat_p->Alert |= DSM_OVER;
	} else if (PLL2_Freq < -8388608) {  // -32768 * 2^8
		PLL2_Freq = -8388608;
		Stat_p->Alert |= DSM_UNDER;
	}
	Stat_p->PLL_DSM = (uint32_t)PLL2_Freq >> 8 & 0xFFFF;
	Stat_p->PLL_MAIN_DIV = Main_Div[Out_fs];
	Stat_p->PLL_PRE_DIV = Pre_Div[Out_fs];
	Stat_p->TCR2_DIV = TCR2_Div[Out_fs];

	Stat_p->State = State;
	Stat_p->PLL2_Freq = PLL2_Freq;
}

static u64 sw_pll_read_spdif_cbits(int is_unlock)
{
	u64 new_cbits = SPDIF_CBITS_INVALID;
	int rv = 0;

	if (sw_pll_priv->spdif_cbits_read_func)
		new_cbits = sw_pll_priv->spdif_cbits_read_func(is_unlock, &rv);
	if (rv < 0) {
		pr_debug("%s: read error:%d new_cbits:%llu\n", __func__, rv, new_cbits);
		new_cbits = SPDIF_CBITS_INVALID;
	}

	return new_cbits;
}

static int sw_pll_read_spdif_lock(int* dpll_ret_lock, int* srpc_ret_lock, int* spdif_open_stat, uint32_t* spdif_irq_stat)
{
	int dpll_new_lock = -1;
	int srpc_new_lock = -1;
	uint32_t spdif_new_irq_stat = 0;
	int open_stat = -1;
	int rv = 0;

	if (sw_pll_priv->spdif_lock_read_func)
		rv = sw_pll_priv->spdif_lock_read_func(&dpll_new_lock, &srpc_new_lock, &open_stat, &spdif_new_irq_stat);
	if (rv < 0) {
		pr_debug("%s: read error:%d dpll_new_lock:%d srpc_new_lock:%d\n", __func__, rv, dpll_new_lock, srpc_new_lock);
		dpll_new_lock = -1;
		srpc_new_lock = -1;
		spdif_new_irq_stat = 0;
		open_stat = -1;
	}

	*dpll_ret_lock = dpll_new_lock;
	*srpc_ret_lock = srpc_new_lock;
	*spdif_irq_stat = spdif_new_irq_stat;
	*spdif_open_stat = open_stat;

	return rv;
}

static void sw_pll_event_notify(SW_PLL_STAT Stat, SW_PLL_PARA Para)
{
	static int old_fs,old_alert,old_state,old_mute,old_fixed_fs;
	static int unlock_check_counter = 3;
	uint32_t irq_event = 0;
	uint32_t workque_event = 0;
	uint32_t tmp_event = 0;
	uint32_t spdif_irq_event = 0;
	int rv;
	char is_lock = 0;
	char is_unlock = 0;
	char is_spdif_reset = 0;
	static int stable_dpll_lock_counter;

	//Check the change point sw_pll
	if (old_fs != Stat.Detect_fs)
		irq_event |= INT_SW_PLL_FS;
	if (old_state != Stat.State)
		irq_event |= INT_SW_PLL_STATE;
	if (old_alert != Stat.Alert)
		irq_event |= INT_SW_PLL_ALERT;
	if (old_mute != Stat.Mute_Req) {
		irq_event |= INT_SW_PLL_MUTE;
		if (Stat.Mute_Req == 0)
			sw_pll_set_mute_ack(0);
	}
	if (old_fixed_fs != Para.Fixed_fs)
		if (Para.Fixed_fs == 0)
			sw_pll_set_mute_ack(1);

	//Check the change point spdif
	spdif_irq_event = atomic_read(&sw_pll_priv->spdif_event);
	atomic_set(&sw_pll_priv->spdif_event, 0);
	if (spdif_irq_event)
		irq_event |= INT_SW_PLL_SPDIF;

	//Save old value
	old_fs = Stat.Detect_fs;
	old_state = Stat.State;
	old_alert = Stat.Alert;
	old_mute = Stat.Mute_Req;
	sw_pll_priv->prev_fixed_fs = old_fixed_fs;
	old_fixed_fs = Para.Fixed_fs;

	//Unlock if it does not become Locked by 2 times
	if (unlock_check_counter < 3)
		unlock_check_counter++;
	if ((Para.Fixed_fs == 0) && (sw_pll_priv->prev_fixed_fs != 0))
		unlock_check_counter = 0;
	if ((irq_event & INT_SW_PLL_FS) && (Stat.Detect_fs != 0))
		unlock_check_counter = 3;
	if (unlock_check_counter == 2)
		irq_event |= INT_SW_PLL_UNLOCK_2TIMES;

	/* for cbits read */
	if (Stat.Detect_fs && (sw_pll_priv->opt_i2s_mode == SW_PLL_MODE_OPT))
		irq_event |= INT_SW_PLL_READ_CBITS;

	/* for read spdif status */
	if (sw_pll_priv->opt_i2s_mode == SW_PLL_MODE_OPT)
		irq_event |= INT_SW_PLL_READ_SPDIF_STAT;

	/* for spdif reset*/
	if (sw_pll_priv->opt_i2s_mode == SW_PLL_MODE_OPT) {
		if ((Stat.State == Stable) && sw_pll_priv->spdif_srpc_lock != 1) {
			stable_dpll_lock_counter++;
			if (stable_dpll_lock_counter > 100) {
				irq_event |= INT_SW_PLL_SPDIF_RESET;
				stable_dpll_lock_counter = 0;
			}
		} else {
			stable_dpll_lock_counter = 0;
		}
	}

	if (irq_event) {
		if (irq_event & INT_SW_PLL_FS)
			workque_event |= INT_SW_PLL_FS;
		if (irq_event & INT_SW_PLL_STATE)
			workque_event |= INT_SW_PLL_STATE;
		if (irq_event & INT_SW_PLL_ALERT)
			workque_event |= INT_SW_PLL_ALERT;
		if (irq_event & INT_SW_PLL_MUTE)
			workque_event |= INT_SW_PLL_MUTE;
		if (irq_event & INT_SW_PLL_READ_CBITS)
			workque_event |= INT_SW_PLL_READ_CBITS;
		if (irq_event & INT_SW_PLL_READ_SPDIF_STAT)
			workque_event |= INT_SW_PLL_READ_SPDIF_STAT;

		if (irq_event & INT_SW_PLL_SPDIF_RESET) {
			is_spdif_reset = 0x10;
			workque_event |= INT_SW_PLL_SPDIF_RESET;
		}

		if ((irq_event & INT_SW_PLL_FS) && (Stat.Detect_fs != 0)) {
			is_lock = 0x1;
			workque_event |= INT_SW_PLL_LOCK;
		}
		if ((irq_event & INT_SW_PLL_FS)  && (Stat.Detect_fs == 0) && (Para.Fixed_fs == 0)  && (sw_pll_priv->prev_fixed_fs == 0)) {
			pr_debug("%s unlock. detect_fs=0\n", __func__);
			workque_event |= INT_SW_PLL_UNLOCK;
			is_unlock = 0x2;
		}
		if (!is_unlock && (spdif_irq_event & INT_LOSS_LOCK)) {
			pr_debug("%s unlock. from spdif loss lock\n", __func__);
			workque_event |= INT_SW_PLL_UNLOCK;
			is_unlock = 0x4;
		}
		//Unlock if it does not become Locked by 2 times
		if (!is_unlock && (irq_event & INT_SW_PLL_UNLOCK_2TIMES)) {
			pr_debug("%s unlock. does not become Locked by 2 times\n", __func__);
			workque_event |= INT_SW_PLL_UNLOCK;
			is_unlock = 0x8;
		}
		if (spdif_irq_event & INT_BIT_ERR)
			workque_event |= INT_SW_PLL_SPDIF_BITERR;

		//logging
		sw_pll_priv->unlock_log = is_unlock | is_lock | is_spdif_reset;
		sw_pll_priv->spdif_event_log = spdif_irq_event;
	}

	if (workque_event) {
		mutex_lock(&sw_pll_priv->sw_pll_workque_lock);
		tmp_event = atomic_read(&sw_pll_priv->irq_event);
		atomic_set(&sw_pll_priv->irq_event, tmp_event | workque_event);
		sw_pll_priv->send_data = Stat;
		mutex_unlock(&sw_pll_priv->sw_pll_workque_lock);

		rv = queue_work(sw_pll_priv->wq, &sw_pll_priv->work);
		if (!rv)
			pr_debug("%s:%d queue_work can't queue\n", __func__, __LINE__);
	}

}

static void sw_pll_kthread_main(void)
{
	static int counter;
	SW_PLL_STAT Stat;
	uint32_t tcr2;
	unsigned long flags;
	int fixed_on,fixed_off;
	uint8_t log_unlock;
	uint64_t spdif_cbits;
	int spdif_dpll_lock = -1;
	int spdif_srpc_lock = -1;
	int spdif_open_stat = -1;
	uint32_t spdif_irq_stat = 0;
	uint8_t opt_i2s_mode = 0;
	uint32_t spdif_event_log = 0;

	spin_lock_irqsave(&sw_pll_priv->sw_pll_lock, flags);
	Para.RBCR = ioread32(sw_pll_priv->sai6_base + FSL_SAI_RBCTN);
	Para.TBCR = ioread32(sw_pll_priv->sai6_base + FSL_SAI_TBCTN);
	spin_unlock_irqrestore(&sw_pll_priv->sw_pll_lock, flags);
	Para.RBCTR = ioread32(sw_pll_priv->sai6_base + FSL_SAI_RTCAP);
	Para.TBCTR = ioread32(sw_pll_priv->sai6_base + FSL_SAI_TTCAP);

	mutex_lock(&sw_pll_priv->fixed_lock);
	fixed_on = atomic_read(&sw_pll_priv->fixed_on);
	fixed_off = atomic_read(&sw_pll_priv->fixed_off);
	if (fixed_on) {
		Para.Fixed_fs = 1;
		atomic_set(&sw_pll_priv->fixed_on, 0);
	}
	else if (!fixed_on && fixed_off) {
		Para.Fixed_fs = 0;
		atomic_set(&sw_pll_priv->fixed_off, 0);
	}
	mutex_unlock(&sw_pll_priv->fixed_lock);

	SW_PLL_Core(&Para, &Stat);

	if (Para.Fixed_fs == 0) {
		//DSM
		sw_pll_get_dsm_lock();
		iowrite32(Stat.PLL_DSM,sw_pll_priv->ccm_base + 0x1C);
		sw_pll_get_dsm_unlock();
		//TRC2
		tcr2 = ioread32(sw_pll_priv->sai6_base + 0x10) & 0xffffff00;
		tcr2 = tcr2 | Stat.TCR2_DIV;
		iowrite32(tcr2,sw_pll_priv->sai6_base + 0x10);
	}

	//set pll infomation for transport
	mutex_lock(&sw_pll_priv->transport_lock);
	sw_pll_priv->transport_data.PLL_DSM = Stat.PLL_DSM;
	sw_pll_priv->transport_data.Detect_fs = Stat.Detect_fs;
	sw_pll_priv->transport_data.State = Stat.State;
	sw_pll_priv->transport_data.Fixed_fs = Para.Fixed_fs;
	mutex_unlock(&sw_pll_priv->transport_lock);

	//for logging
	log_unlock = sw_pll_priv->unlock_log;
	sw_pll_priv->unlock_log = 0;
	spdif_cbits = sw_pll_priv->spdif_cbits;
	opt_i2s_mode = sw_pll_priv->opt_i2s_mode;
	spdif_dpll_lock = sw_pll_priv->spdif_dpll_lock;
	spdif_srpc_lock = sw_pll_priv->spdif_srpc_lock;
	spdif_open_stat = sw_pll_priv->spdif_open_stat;
	spdif_irq_stat = sw_pll_priv->spdif_irq_stat;
	spdif_event_log = sw_pll_priv->spdif_event_log;

	//for notify
	sw_pll_event_notify(Stat, Para);

	//for logging
	counter++;
	if (sw_pll_priv->is_log_trigger && (counter > 200)) {
		pr_info("fs:%d alert%x state:%d PLL2_Freq:%d Phase_Err:%d log:%d fixed_fs:%d mute:%d ack:%d cbits:%llx dlock:%d slock:%d sie:%x\n",
			Stat.Detect_fs, Stat.Alert, Stat.State, Stat.PLL2_Freq, Stat.Phase_Err, sw_pll_priv->log_counter, Para.Fixed_fs, Stat.Mute_Req, Para.Mute_Ack, spdif_cbits, spdif_dpll_lock, spdif_srpc_lock, spdif_irq_stat);
		counter = 0;
	}
	if (sw_pll_priv->is_log_trigger && (sw_pll_priv->log_counter < sw_pll_priv->log_size)) {
#ifdef LONG_TIME_LOG
		sw_pll_priv->logdata[sw_pll_priv->log_counter].RBCR = Para.RBCR;
		sw_pll_priv->logdata[sw_pll_priv->log_counter].RBCTR = Para.RBCTR;
#else
		sw_pll_priv->logdata[sw_pll_priv->log_counter].RBCR = Para.RBCR;
		sw_pll_priv->logdata[sw_pll_priv->log_counter].TBCR = Para.TBCR;
		sw_pll_priv->logdata[sw_pll_priv->log_counter].RBCTR = Para.RBCTR;
		sw_pll_priv->logdata[sw_pll_priv->log_counter].TBCTR = Para.TBCTR;
		sw_pll_priv->logdata[sw_pll_priv->log_counter].State = Stat.State;
		sw_pll_priv->logdata[sw_pll_priv->log_counter].PLL2_Freq = Stat.PLL2_Freq;
		sw_pll_priv->logdata[sw_pll_priv->log_counter].Phase_Err = Stat.Phase_Err;
		sw_pll_priv->logdata[sw_pll_priv->log_counter].Detect_fs = Stat.Detect_fs;
		sw_pll_priv->logdata[sw_pll_priv->log_counter].Alert = Stat.Alert;
		sw_pll_priv->logdata[sw_pll_priv->log_counter].Mute_Req = Stat.Mute_Req;
		sw_pll_priv->logdata[sw_pll_priv->log_counter].Mute_Ack = Para.Mute_Ack;
		sw_pll_priv->logdata[sw_pll_priv->log_counter].Fixed_fs = Para.Fixed_fs;
		sw_pll_priv->logdata[sw_pll_priv->log_counter].Unlock = log_unlock;
		sw_pll_priv->logdata[sw_pll_priv->log_counter].spdif_cbits = spdif_cbits;
		sw_pll_priv->logdata[sw_pll_priv->log_counter].spdif_dpll_lock = spdif_dpll_lock;
		sw_pll_priv->logdata[sw_pll_priv->log_counter].spdif_srpc_lock = spdif_srpc_lock;
		sw_pll_priv->logdata[sw_pll_priv->log_counter].opt_i2s_mode = opt_i2s_mode;
		sw_pll_priv->logdata[sw_pll_priv->log_counter].spdif_open_stat = spdif_open_stat;
		sw_pll_priv->logdata[sw_pll_priv->log_counter].spdif_irq_stat = spdif_irq_stat;
		sw_pll_priv->logdata[sw_pll_priv->log_counter].spdif_event_log = spdif_event_log;
#endif
		sw_pll_priv->log_counter++;
	}

	set_current_state(TASK_INTERRUPTIBLE);
	schedule_timeout(msecs_to_jiffies(8));
}

static void sw_pll_logging_interval(void)
{
	int jiffies_error = 0;
	u64 jiffies_new = 0;
	static u64 jiffies_old;
	static int counter;
	static int diff[101] = {0};
	int i = 0;

	jiffies_new = jiffies_to_msecs(jiffies);
	jiffies_error = jiffies_new - jiffies_old;
	jiffies_old = jiffies_new;

	if (jiffies_error < 0)
		jiffies_error = 0;
	if (jiffies_error > 100)
		jiffies_error = 100;

	diff[(unsigned char)jiffies_error]++;
	counter++;

	if (counter >= 5000) {
		pr_info("####start####\n");
		for (i = 0; i <= 100 ; i++) {
			if (diff[i] != 0) {
				pr_info("kthread:%d[ms] %d count\n", i, diff[i]);
			}
		}
		for (i = 0; i <= 100; i++) {
			diff[i] = 0;
		}
		counter = 0;
		pr_info("####end####\n");
	}

}

static int sw_pll_kthread(void *arg)
{
	while (!kthread_should_stop()) {
		 sw_pll_kthread_main();
		if (LOG_LEVEL_INTERVAL <= sw_pll_priv->is_log_trigger)
			sw_pll_logging_interval();
	}
	return 0;
}

static char sw_pll_is_only_cnew(const char *event, const char *header)
{
	return !strncmp(event, header, 17);
}

static void sw_pll_detected_wq(struct work_struct *w)
{
	struct platform_device *pdev = sw_pll_priv->pdev;
	char event[80] = "EVENT=SW_PLL_ISR";
	char event_status[80];
	char *envp[] = {
		event,
		event_status,
		NULL,
	};
	u32 irq_event;
	u32 spdif_event = 0;
	char is_unlock = 0;
	char is_lock = 0;
	u64 new_cbits = SPDIF_CBITS_INVALID;
	int spdif_dpll_lock = -1;
	int spdif_srpc_lock = -1;
	int spdif_open_stat = -1;
	uint32_t spdif_irq_stat = 0;
	int rv;
	SW_PLL_STAT wq_stat_data;

	mutex_lock(&sw_pll_priv->sw_pll_workque_lock);
	irq_event = atomic_read(&sw_pll_priv->irq_event);
	atomic_set(&sw_pll_priv->irq_event, 0);
	wq_stat_data = sw_pll_priv->send_data;
	mutex_unlock(&sw_pll_priv->sw_pll_workque_lock);

	if (irq_event & INT_SW_PLL_FS)
		strncat(event, ",FS", 3);
	if (irq_event & INT_SW_PLL_STATE)
		strncat(event, ",STATE", 6);
	if (irq_event & INT_SW_PLL_ALERT)
		strncat(event, ",ALERT", 6);
	if (irq_event & INT_SW_PLL_MUTE)
		strncat(event, ",MUTE", 5);

	if (irq_event & INT_SW_PLL_LOCK) {
		is_lock = 0x1;
		strncat(event, ",LOCK", 5);
	}
	if (irq_event & INT_SW_PLL_UNLOCK) {
		pr_debug("%s unlock. detect_fs=0\n", __func__);
		strncat(event, ",ULOCK", 6);
		is_unlock = 0x1;
	}
	if (irq_event & INT_SW_PLL_SPDIF_BITERR) {
		strncat(event, ",BITERR", 7);
		spdif_event = 0x1;
	}

	if (irq_event & INT_SW_PLL_READ_CBITS || is_unlock) {
		new_cbits = sw_pll_read_spdif_cbits(is_unlock);
		if (0 <= new_cbits && !(new_cbits & SPDIF_CBITS_INVALID)) {
			dev_dbg(&pdev->dev, "%s:read cbits:%llx\n", __func__, new_cbits);
			sw_pll_priv->spdif_cbits = new_cbits;
			strncat(event, ",CNEW", 5);
		}
		if (is_unlock)
			sw_pll_priv->spdif_cbits = SPDIF_CBITS_INVALID;
	}

	//check spdif lock state
	if (irq_event & INT_SW_PLL_READ_SPDIF_STAT) {
		rv = sw_pll_read_spdif_lock(&spdif_dpll_lock, &spdif_srpc_lock, &spdif_open_stat, &spdif_irq_stat);
		if (rv < 0) {
			spdif_dpll_lock = -1;
			spdif_srpc_lock = -1;
		}
	}

	//call spdif softreset
	if (irq_event & INT_SW_PLL_SPDIF_RESET) {
		dev_info(&pdev->dev, "call softreset in sw pll\n");
		if (sw_pll_priv->spdif_softreset_func)
			sw_pll_priv->spdif_softreset_func();
	}

	//set sw pll priv
	sw_pll_priv->spdif_dpll_lock = spdif_dpll_lock;
	sw_pll_priv->spdif_srpc_lock = spdif_srpc_lock;
	sw_pll_priv->spdif_open_stat = spdif_open_stat;
	sw_pll_priv->spdif_irq_stat = spdif_irq_stat;

	if (!sw_pll_is_only_cnew(event, "EVENT=SW_PLL_ISR")) {
		snprintf(event_status, sizeof(event_status),
			 "STATUS=FS=%d,STATE=%d,ALERT=%x,MUTE=%d,SPDIF=%x,CNEW=%llx",
			 wq_stat_data.Detect_fs, wq_stat_data.State,
			 wq_stat_data.Alert, wq_stat_data.Mute_Req, spdif_event, sw_pll_priv->spdif_cbits);
		pr_debug("send uevent %s\n", event);
		pr_debug("status:%s\n", event_status);

		blocking_notifier_call_chain(&sw_pll_notify_list, 0, &wq_stat_data);
		/* notify userspace */
		kobject_uevent_env(&pdev->dev.kobj, KOBJ_CHANGE, envp);
	}
}

static ssize_t enable_store(struct kobject *kobj, struct kobj_attribute *attr, const char *buf, size_t count)
{
	sscanf(buf, "%d", &sw_pll_priv->is_enable);
	if (sw_pll_priv->is_enable && !sw_pll_priv->kthread_tsk) {
		sw_pll_priv->kthread_tsk = kthread_run(sw_pll_kthread, sw_pll_priv, "sw_pll_kthread");
	}
	else if (!sw_pll_priv->is_enable && sw_pll_priv->kthread_tsk) {
		kthread_stop(sw_pll_priv->kthread_tsk);
		sw_pll_priv->kthread_tsk = NULL;
	}

	return count;
}

static ssize_t enable_show(struct kobject *kobj, struct kobj_attribute *attr, char *buf)
{
	return snprintf(buf, PAGE_SIZE, "%d\n", sw_pll_priv->is_enable);
}


static ssize_t log_trriger_store(struct kobject *kobj, struct kobj_attribute *attr, const char *buf, size_t count)
{
	int tmp;

	sscanf(buf, "%d", &tmp);

	if(!sw_pll_priv->logdata && tmp) {
		sw_pll_priv->logdata = kzalloc(sizeof(SW_PLL_LOGGING) * LOG_MAX_SIZE, GFP_KERNEL);
		sw_pll_priv->log_size = LOG_MAX_SIZE;
		if (!sw_pll_priv->logdata){
			pr_err("%s can't malloc memory!!\n", __func__);
			sw_pll_priv->log_size = 0;
		}
	}
	sw_pll_priv->is_log_trigger = tmp;

	return count;
}

static ssize_t log_trriger_show(struct kobject *kobj, struct kobj_attribute *attr, char *buf)
{
	return snprintf(buf, PAGE_SIZE, "%d\n", sw_pll_priv->is_log_trigger);
}

static ssize_t data_show(struct kobject *kobj, struct kobj_attribute *attr, char *buf)
{
	int i;
	char tmp_buf[LOG_SIZE_PER_LINE] = {};
	char total_buf[LOG_SIZE_TOTAL] = {};

	if (sw_pll_priv->log_read_counter == sw_pll_priv->log_counter)
		return 0;

	for (i = sw_pll_priv->log_read_counter; i < (sw_pll_priv->log_read_counter + sw_pll_priv->log_read_size); i++) {
#ifdef LONG_TIME_LOG
		snprintf(tmp_buf, sizeof(tmp_buf), "%x,%x\n", sw_pll_priv->logdata[i].RBCR, sw_pll_priv->logdata[i].RBCTR);
#else
		snprintf(tmp_buf, sizeof(tmp_buf), "%x,%x,%x,%x,%x,%d,%d,%d,%x,%d,%d,%d,%d,%llx,%d,%d,%d,%d,%x,%x\n",
			sw_pll_priv->logdata[i].RBCR, sw_pll_priv->logdata[i].TBCR, sw_pll_priv->logdata[i].RBCTR,
			sw_pll_priv->logdata[i].TBCTR, sw_pll_priv->logdata[i].PLL2_Freq, sw_pll_priv->logdata[i].Phase_Err,
			sw_pll_priv->logdata[i].State, sw_pll_priv->logdata[i].Detect_fs, sw_pll_priv->logdata[i].Alert,
			sw_pll_priv->logdata[i].Mute_Req, sw_pll_priv->logdata[i].Mute_Ack, sw_pll_priv->logdata[i].Fixed_fs,
			sw_pll_priv->logdata[i].Unlock, sw_pll_priv->logdata[i].spdif_cbits, sw_pll_priv->logdata[i].spdif_dpll_lock,
			sw_pll_priv->logdata[i].spdif_srpc_lock, sw_pll_priv->logdata[i].opt_i2s_mode, sw_pll_priv->logdata[i].spdif_open_stat,
			sw_pll_priv->logdata[i].spdif_irq_stat, sw_pll_priv->logdata[i].spdif_event_log);
#endif
		strncat(total_buf, tmp_buf, sizeof(tmp_buf));
	}
	if (sw_pll_priv->log_read_counter + sw_pll_priv->log_read_size <= sw_pll_priv->log_counter)
		sw_pll_priv->log_read_counter += sw_pll_priv->log_read_size;
	else
		sw_pll_priv->log_read_counter = sw_pll_priv->log_counter;

	return snprintf(buf, PAGE_SIZE, "%s", total_buf);
}

static ssize_t set_param_store(struct kobject *kobj, struct kobj_attribute *attr, const char *buf, size_t count)
{
	sscanf(buf, "%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%hu,%hu,%hu,%hu,%hu,%hu,%hu,%hu,%hu,%hu,%hu,%hu,%hu,%hu,%u,%u,%u,%u,%u,%hhu,%hu,%hu,%hu,%hu,%hu,%hu,%hu,%hu,%hu,%hu,%hu,%u,%hhu",
		&Para.Valid_Upper[0], &Para.Valid_Upper[1], &Para.Valid_Upper[2], &Para.Valid_Upper[3], &Para.Valid_Upper[4], &Para.Valid_Upper[5], &Para.Valid_Upper[6], &Para.Valid_Upper[7], \
		&Para.Valid_Lower[0], &Para.Valid_Lower[1], &Para.Valid_Lower[2], &Para.Valid_Lower[3], &Para.Valid_Lower[4], &Para.Valid_Lower[5], &Para.Valid_Lower[6], &Para.Valid_Lower[7], \
		&Para.Ph_Para[0], &Para.Ph_Para[1], &Para.Ph_Para[2], &Para.Ph_Para[3], &Para.Ph_Para[4], &Para.Ph_Para[5], &Para.Ph_Para[6], &Para.Ph_Para[7], &Para.Ph_Para[8], &Para.Ph_Para[9], &Para.Ph_Para[10], &Para.Ph_Para[11], &Para.Ph_Para[12], &Para.Ph_Para[13], \
		&Para.Trans_Time[0], &Para.Trans_Time[1], &Para.Trans_Time[2], &Para.Trans_Time[3], &Para.Trans_Time[4], \
		&Para.Valid_Count, \
		&Para.Frq_Para[0], &Para.Frq_Para[1], &Para.Frq_Para[2], &Para.Frq_Para[3], &Para.Frq_Para[4], &Para.Frq_Para[5], &Para.Frq_Para[6], &Para.Frq_Para[7], &Para.Frq_Para[8], &Para.Frq_Para[9], &Para.Frq_Para[10], \
		&Para.Cond_Flag, &Para.Fixed_fs \
	);
	return count;
}

static ssize_t set_param_show(struct kobject *kobj, struct kobj_attribute *attr, char *buf)
{
	return snprintf(buf, PAGE_SIZE, "%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u\n",
			Para.Valid_Upper[0], Para.Valid_Upper[1], Para.Valid_Upper[2], Para.Valid_Upper[3], Para.Valid_Upper[4], Para.Valid_Upper[5], Para.Valid_Upper[6], Para.Valid_Upper[7],
			Para.Valid_Lower[0], Para.Valid_Lower[1], Para.Valid_Lower[2], Para.Valid_Lower[3], Para.Valid_Lower[4], Para.Valid_Lower[5], Para.Valid_Lower[6], Para.Valid_Lower[7],
			Para.Ph_Para[0], Para.Ph_Para[1], Para.Ph_Para[2], Para.Ph_Para[3], Para.Ph_Para[4], Para.Ph_Para[5], Para.Ph_Para[6], Para.Ph_Para[7], Para.Ph_Para[8], Para.Ph_Para[9], Para.Ph_Para[10], Para.Ph_Para[11], Para.Ph_Para[12], Para.Ph_Para[13],
			Para.Trans_Time[0], Para.Trans_Time[1], Para.Trans_Time[2], Para.Trans_Time[3], Para.Trans_Time[4],
			Para.Valid_Count,
			Para.Frq_Para[0], Para.Frq_Para[1], Para.Frq_Para[2], Para.Frq_Para[3], Para.Frq_Para[4], Para.Frq_Para[5], Para.Frq_Para[6], Para.Frq_Para[7], Para.Frq_Para[8], Para.Frq_Para[9], Para.Frq_Para[10],
			Para.Cond_Flag, Para.Fixed_fs);
}

static ssize_t pll_info_show(struct kobject *kobj, struct kobj_attribute *attr, char *buf)
{
	SW_PLL_INFO send_transport_data;

	mutex_lock(&sw_pll_priv->transport_lock);
	send_transport_data = sw_pll_priv->transport_data;
	mutex_unlock(&sw_pll_priv->transport_lock);

	return snprintf(buf, PAGE_SIZE, "%d,%d,%d,%d\n",
		send_transport_data.PLL_DSM, send_transport_data.Detect_fs,
		send_transport_data.State, send_transport_data.Fixed_fs);
}


static struct kobj_attribute log_trigger_attribute = __ATTR(log_trigger, 0644, log_trriger_show, log_trriger_store);
static struct kobj_attribute data_attribute = __ATTR(data, 0444, data_show, NULL);
static struct kobj_attribute enable_attribute = __ATTR(enable, 0644, enable_show, enable_store);
static struct kobj_attribute set_param_attribute = __ATTR(set_param, 0644, set_param_show, set_param_store);
static struct kobj_attribute pll_info_attribute = __ATTR(pll_info, 0444, pll_info_show, NULL);

static struct attribute *attrs[] = {
	&log_trigger_attribute.attr,
	&data_attribute.attr,
	&enable_attribute.attr,
	&set_param_attribute.attr,
	&pll_info_attribute.attr,
	NULL,
};

static struct attribute_group attr_group = {
	.attrs = attrs,
};

static int sony_sw_pll_probe(struct platform_device *pdev)
{
	int ret;

	sw_pll_priv = devm_kzalloc(&pdev->dev, sizeof(*sw_pll_priv), GFP_KERNEL);
	if (!sw_pll_priv)
		return -ENOMEM;

	sw_pll_priv->pdev = pdev;
	sw_pll_priv->is_log_trigger = 0;
	sw_pll_priv->is_enable = 0;
	sw_pll_priv->kthread_tsk = NULL;
	atomic_set(&sw_pll_priv->irq_event, 0);
	atomic_set(&sw_pll_priv->spdif_event, 0);
	atomic_set(&sw_pll_priv->fixed_on, 0);
	atomic_set(&sw_pll_priv->fixed_off, 0);
	sw_pll_priv->prev_fixed_fs = 0;
	sw_pll_priv->spdif_cbits_read_func = NULL;
	sw_pll_priv->spdif_lock_read_func = NULL;
	sw_pll_priv->spdif_softreset_func = NULL;
	sw_pll_priv->unlock_log = 0;
	sw_pll_priv->spdif_cbits = SPDIF_CBITS_INVALID;
	sw_pll_priv->opt_i2s_mode = 0;
	sw_pll_priv->log_size = 0;
	sw_pll_priv->log_read_size = LOG_READ_SIZE;
	sw_pll_priv->logdata = NULL;
	sw_pll_priv->log_counter = 0;
	sw_pll_priv->log_read_counter = 0;

	sw_pll_priv->spdif_event_log = 0;
	sw_pll_priv->sai6_base = ioremap_nocache(SAI6_BASE_ADDR,0x100);
	sw_pll_priv->ccm_base = ioremap_nocache(ANATOP_BASE_ADDR,0x1c);
	spin_lock_init(&sw_pll_priv->sw_pll_lock);
	mutex_init(&sw_pll_priv->dsm_lock);
	mutex_init(&sw_pll_priv->fixed_lock);
	mutex_init(&sw_pll_priv->transport_lock);
	mutex_init(&sw_pll_priv->sw_pll_workque_lock);

	sw_pll_priv->sw_pll_kobj = kobject_create_and_add("sw_pll", kernel_kobj);
	if (!sw_pll_priv->sw_pll_kobj)
		return -ENOMEM;

	ret = sysfs_create_group(sw_pll_priv->sw_pll_kobj, &attr_group);
	if (ret)
		kobject_put(sw_pll_priv->sw_pll_kobj);

	sw_pll_priv->wq = alloc_workqueue("sw_pll_wq", WQ_MEM_RECLAIM, 0);
	if (!sw_pll_priv->wq)
		return -ENOMEM;

	INIT_WORK(&sw_pll_priv->work, sw_pll_detected_wq);

//tmp value for 200Mhz
	Para.Valid_Upper[0] = 0;
	Para.Valid_Upper[1] = 44028844;
	Para.Valid_Upper[2] = 60677250;
	Para.Valid_Upper[3] = 66043265;
	Para.Valid_Upper[4] = 121354500;
	Para.Valid_Upper[5] = 132086531;
	Para.Valid_Upper[6] = 242709000;
	Para.Valid_Upper[7] = 264173062;
	Para.Valid_Lower[0] = 0;
	Para.Valid_Lower[1] = 43932087;
	Para.Valid_Lower[2] = 60543907;
	Para.Valid_Lower[3] = 65898130;
	Para.Valid_Lower[4] = 121087814;
	Para.Valid_Lower[5] = 131796260;
	Para.Valid_Lower[6] = 242175627;
	Para.Valid_Lower[7] = 263592520;
	Para.Ph_Para[0] = 10;
	Para.Ph_Para[1] = 8;
	Para.Ph_Para[2] = 41;
	Para.Ph_Para[3] = 16;
	Para.Ph_Para[4] = 4;
	Para.Ph_Para[5] = 7;
	Para.Ph_Para[6] = 20;
	Para.Ph_Para[7] = 20;
	Para.Ph_Para[8] = 8;
	Para.Ph_Para[9] = 0;
	Para.Ph_Para[10] = 512;
	Para.Ph_Para[11] = 3072;
	Para.Ph_Para[12] = 8;
	Para.Ph_Para[13] = 4;
	Para.Trans_Time[0] = 16000000;
	Para.Trans_Time[1] = 100000000;
	Para.Trans_Time[2] = 200000000;
	Para.Trans_Time[3] = 60000000;
	Para.Trans_Time[4] = 10240;
	Para.Valid_Count = 2;
	Para.Frq_Para[0] = 4370;
	Para.Frq_Para[1] = 1638;
	Para.Frq_Para[2] = 288;
	Para.Frq_Para[3] = 1228;
	Para.Frq_Para[4] = 1228;
	Para.Frq_Para[5] = 1228;
	Para.Frq_Para[6] = 21522;
	Para.Frq_Para[7] = 52;
	Para.Frq_Para[8] = 2660;
	Para.Frq_Para[9] = 4892;
	Para.Frq_Para[10] = 7172;
	Para.Cond_Flag = 0;
	Para.Fixed_fs = 1;

	//start in boot
	sw_pll_priv->kthread_tsk = kthread_run(sw_pll_kthread, sw_pll_priv, "sw_pll_kthread");
	sw_pll_priv->is_enable = true;
	return 0;
}

static int sony_sw_pll_remove(struct platform_device *pdev)
{
	if (sw_pll_priv->kthread_tsk != NULL) {
		kthread_stop(sw_pll_priv->kthread_tsk);
		sw_pll_priv->kthread_tsk = NULL;
	}
	if (sw_pll_priv->logdata != NULL) {
		kfree(sw_pll_priv->logdata);
		sw_pll_priv->logdata = NULL;
	}
	iounmap(sw_pll_priv->sai6_base);
	iounmap(sw_pll_priv->ccm_base);

	return 0;
}

static int sw_pll_suspend(struct device *dev)
{
	if (sw_pll_priv->kthread_tsk != NULL) {
		kthread_stop(sw_pll_priv->kthread_tsk);
		sw_pll_priv->kthread_tsk = NULL;
	}
	return 0;
}

static int sw_pll_resume(struct device *dev)
{
	if (sw_pll_priv->is_enable && !sw_pll_priv->kthread_tsk) {
		sw_pll_priv->kthread_tsk = kthread_run(sw_pll_kthread, sw_pll_priv, "sw_pll_kthread");
	}
	return 0;
}

static const struct dev_pm_ops sw_pll_dev_pmops = {
	.suspend = sw_pll_suspend,
	.resume  = sw_pll_resume,
};

static const struct of_device_id sony_sw_pll_dt_ids[] = {
	{
	.compatible = "sony,sw_pll",
	},
	{} };
MODULE_DEVICE_TABLE(of, sony_sw_pll_dt_ids);

static struct platform_driver sony_sw_pll_driver = {
	.probe = sony_sw_pll_probe,
	.remove = sony_sw_pll_remove,
	.driver = {
		.name = DRV_NAME,
		.owner = THIS_MODULE,
		.of_match_table = sony_sw_pll_dt_ids,
		.pm    = &sw_pll_dev_pmops,
	},
};

module_platform_driver(sony_sw_pll_driver);
MODULE_DESCRIPTION("Sony Software PLL Driver");
MODULE_AUTHOR("Sony Home Entertainment & Sound Products Inc.");
MODULE_LICENSE("GPL");
MODULE_VERSION("v1.00");
