// SPDX-License-Identifier: GPL-2.0
/*
 * twincodec.c
 *
 * twin codec pipe driver
 *
 * Copyright 2021 Sony Corporation
 */
#include <linux/clk.h>
#include <linux/delay.h>
#include <linux/device.h>
#include <linux/module.h>
#include <linux/pm_runtime.h>
#include <sound/pcm_params.h>
#include <sound/soc.h>
#include <sound/twincodec.h>

/* Define this label when enable Trace log */
/* #define TWINCODEC_TRACE */

/* Define this label when enable Debug log */
/* #define TWINCODEC_DEBUG */

/* Define print macros */
#ifdef TWINCODEC_TRACE
#define TRACE(fmt, args...) pr_info("TRACE: %s(%d): " \
				fmt, __func__, __LINE__, ##args)
#else
#define TRACE(fmt, args...) {}
#endif
#ifdef TWINCODEC_DEBUG
#define DEBUG(fmt, args...) pr_info("DEBUG: %s(%d): " \
				fmt, __func__, __LINE__, ##args)
#else
#define DEBUG(fmt, args...) pr_debug("DEBUG: %s(%d): " \
				fmt, __func__, __LINE__, ##args)
#endif

/* Max support codec-dai count */
#define MAX_DAI_COUNT	(4)

struct twin_priv {
	struct snd_soc_codec_driver codec_driver;
	struct snd_soc_codec *codec;

	char *dai_name[MAX_DAI_COUNT];
	struct snd_soc_dai *dai[MAX_DAI_COUNT];
	int dai_mute[MAX_DAI_COUNT];
	unsigned int dai_count;
	bool probe_child_codec;
	bool sync_runtime_pm;
	unsigned int mute_busy;
};

static const struct snd_soc_dapm_widget twin_dapm_widgets[] = {
	SND_SOC_DAPM_DAC("DAC1", "Playback", SND_SOC_NOPM, 0, 0),
	SND_SOC_DAPM_DAC("DAC2", "Playback", SND_SOC_NOPM, 0, 0),
	SND_SOC_DAPM_DAC("DAC3", "Playback", SND_SOC_NOPM, 0, 0),
	SND_SOC_DAPM_DAC("DAC4", "Playback", SND_SOC_NOPM, 0, 0),
	SND_SOC_DAPM_DAC("DAC5", "Playback", SND_SOC_NOPM, 0, 0),
	SND_SOC_DAPM_DAC("DAC6", "Playback", SND_SOC_NOPM, 0, 0),
	SND_SOC_DAPM_DAC("DAC7", "Playback", SND_SOC_NOPM, 0, 0),
	SND_SOC_DAPM_DAC("DAC8", "Playback", SND_SOC_NOPM, 0, 0),
	SND_SOC_DAPM_DAC("DAC9", "Playback", SND_SOC_NOPM, 0, 0),
	SND_SOC_DAPM_DAC("DAC10", "Playback", SND_SOC_NOPM, 0, 0),
	SND_SOC_DAPM_DAC("DAC11", "Playback", SND_SOC_NOPM, 0, 0),
	SND_SOC_DAPM_DAC("DAC12", "Playback", SND_SOC_NOPM, 0, 0),

	SND_SOC_DAPM_OUTPUT("AOUT1L"),
	SND_SOC_DAPM_OUTPUT("AOUT1R"),
	SND_SOC_DAPM_OUTPUT("AOUT2L"),
	SND_SOC_DAPM_OUTPUT("AOUT2R"),
	SND_SOC_DAPM_OUTPUT("AOUT3L"),
	SND_SOC_DAPM_OUTPUT("AOUT3R"),
	SND_SOC_DAPM_OUTPUT("AOUT4L"),
	SND_SOC_DAPM_OUTPUT("AOUT4R"),
	SND_SOC_DAPM_OUTPUT("AOUT5L"),
	SND_SOC_DAPM_OUTPUT("AOUT5R"),
	SND_SOC_DAPM_OUTPUT("AOUT6L"),
	SND_SOC_DAPM_OUTPUT("AOUT6R"),
	SND_SOC_DAPM_OUTPUT("AOUT7L"),
	SND_SOC_DAPM_OUTPUT("AOUT7R"),
	SND_SOC_DAPM_OUTPUT("AOUT8L"),
	SND_SOC_DAPM_OUTPUT("AOUT8R"),
	SND_SOC_DAPM_OUTPUT("AOUT9L"),
	SND_SOC_DAPM_OUTPUT("AOUT9R"),
	SND_SOC_DAPM_OUTPUT("AOUT10L"),
	SND_SOC_DAPM_OUTPUT("AOUT10R"),
	SND_SOC_DAPM_OUTPUT("AOUT11L"),
	SND_SOC_DAPM_OUTPUT("AOUT11R"),
	SND_SOC_DAPM_OUTPUT("AOUT12L"),
	SND_SOC_DAPM_OUTPUT("AOUT12R"),
};

static const struct snd_soc_dapm_route twin_dapm_routes[] = {
	/* Playback */
	{ "AOUT1L", NULL, "DAC1" },
	{ "AOUT1R", NULL, "DAC1" },
	{ "AOUT2L", NULL, "DAC2" },
	{ "AOUT2R", NULL, "DAC2" },
	{ "AOUT3L", NULL, "DAC3" },
	{ "AOUT3R", NULL, "DAC3" },
	{ "AOUT4L", NULL, "DAC4" },
	{ "AOUT4R", NULL, "DAC4" },
	{ "AOUT5L", NULL, "DAC5" },
	{ "AOUT5R", NULL, "DAC5" },
	{ "AOUT6L", NULL, "DAC6" },
	{ "AOUT6R", NULL, "DAC6" },
	{ "AOUT7L", NULL, "DAC7" },
	{ "AOUT7R", NULL, "DAC7" },
	{ "AOUT8L", NULL, "DAC8" },
	{ "AOUT8R", NULL, "DAC8" },
	{ "AOUT9L", NULL, "DAC9" },
	{ "AOUT9R", NULL, "DAC9" },
	{ "AOUT10L", NULL, "DAC10" },
	{ "AOUT10R", NULL, "DAC10" },
	{ "AOUT11L", NULL, "DAC11" },
	{ "AOUT11R", NULL, "DAC11" },
	{ "AOUT12L", NULL, "DAC12" },
	{ "AOUT12R", NULL, "DAC12" },
};

/* soc_mute control extension */

static BLOCKING_NOTIFIER_HEAD(twincodec_notify_list);

int register_twincodec_notifier(struct notifier_block *nb)
{
	return blocking_notifier_chain_register(&twincodec_notify_list, nb);
}
EXPORT_SYMBOL_GPL(register_twincodec_notifier);

int unregister_twincodec_notifier(struct notifier_block *nb)
{
	return blocking_notifier_chain_unregister(&twincodec_notify_list, nb);
}
EXPORT_SYMBOL_GPL(unregister_twincodec_notifier);

/* Format check functions */

static int check_fmt(struct snd_soc_dai *dai, unsigned int format)
{
	struct snd_soc_pcm_stream *playback;
	unsigned int fmt;
	u64 formats;
	int chk = 0;

	TRACE("enter\n");
	if (!dai)
		return -1;
	if (!dai->driver)
		return -1;
	playback = &dai->driver->playback;

	fmt = format & SND_SOC_DAIFMT_MASTER_MASK;
	DEBUG("master: 0x%x\n", fmt);
	if (fmt == SND_SOC_DAIFMT_CBS_CFS
	 || fmt == SND_SOC_DAIFMT_CBM_CFM
	 || fmt == SND_SOC_DAIFMT_CBM_CFS)
		chk |= 1;

	fmt = format & SND_SOC_DAIFMT_FORMAT_MASK;
	formats = playback->formats;
	DEBUG("fmt: %u formats: 0x%llx\n", fmt, formats);
	if ((fmt == SND_SOC_DAIFMT_I2S
	 && (formats & (SNDRV_PCM_FMTBIT_S16_LE |
			SNDRV_PCM_FMTBIT_S24_LE |
			SNDRV_PCM_FMTBIT_S32_LE)))
	 || (fmt == SND_SOC_DAIFMT_PDM
	 && (formats & (SNDRV_PCM_FMTBIT_DSD_U8 |
			SNDRV_PCM_FMTBIT_DSD_U16_LE |
			SNDRV_PCM_FMTBIT_DSD_U16_BE |
			SNDRV_PCM_FMTBIT_DSD_U32_LE |
			SNDRV_PCM_FMTBIT_DSD_U32_BE))))
		chk |= 2;

	TRACE("done(chk=%d)\n", chk);
	return (chk != 3);
}

static int check_param(struct snd_soc_dai *dai,
				struct snd_pcm_hw_params *params)
{
	struct snd_soc_pcm_stream *playback;
	snd_pcm_format_t format = params_format(params);
	unsigned int channels = params_channels(params);
	int rate = params_rate(params);
	u64 formats, rates;
	int chk = 0;

	TRACE("enter\n");
	if (!dai)
		return -1;
	if (!dai->driver)
		return -1;
	playback = &dai->driver->playback;

	formats = playback->formats;
	DEBUG("format: %u formats: 0x%llx\n", format, formats);
	if ((format == SNDRV_PCM_FORMAT_S16_LE
	 && (formats & SNDRV_PCM_FMTBIT_S16_LE))
	 || (format == SNDRV_PCM_FORMAT_S24_LE
	 && (formats & SNDRV_PCM_FMTBIT_S24_LE))
	 || (format == SNDRV_PCM_FORMAT_S32_LE
	 && (formats & SNDRV_PCM_FMTBIT_S32_LE))
	 || (format == SNDRV_PCM_FORMAT_DSD_U8
	 && (formats & SNDRV_PCM_FMTBIT_DSD_U8))
	 || (format == SNDRV_PCM_FORMAT_DSD_U16_LE
	 && (formats & SNDRV_PCM_FMTBIT_DSD_U16_LE))
	 || (format == SNDRV_PCM_FORMAT_DSD_U16_BE
	 && (formats & SNDRV_PCM_FMTBIT_DSD_U16_BE))
	 || (format == SNDRV_PCM_FORMAT_DSD_U32_LE
	 && (formats & SNDRV_PCM_FMTBIT_DSD_U32_LE))
	 || (format == SNDRV_PCM_FORMAT_DSD_U32_BE
	 && (formats & SNDRV_PCM_FMTBIT_DSD_U32_BE)))
		chk |= 1;

	if (channels >= playback->channels_min
	 && channels <= playback->channels_max)
		chk |= 2;

	rates = playback->rates;
	DEBUG("rate: %d rates: 0x%llx\n", rate, rates);
	if ((rates & SNDRV_PCM_RATE_KNOT)
	 || (rate == 32000 && (rates & SNDRV_PCM_RATE_32000))
	 || (rate == 44100 && (rates & SNDRV_PCM_RATE_44100))
	 || (rate == 48000 && (rates & SNDRV_PCM_RATE_48000))
	 || (rate == 88200 && (rates & SNDRV_PCM_RATE_88200))
	 || (rate == 96000 && (rates & SNDRV_PCM_RATE_96000))
	 || (rate == 176400 && (rates & SNDRV_PCM_RATE_176400))
	 || (rate == 192000 && (rates & SNDRV_PCM_RATE_192000)))
		chk |= 4;

	TRACE("done(chk=%d)\n", chk);
	return (chk != 7);
}

/* Redirect functions */

static int twin_set_sysclk(struct snd_soc_dai *dai,
				int clk_id, unsigned int freq, int dir)
{
	struct snd_soc_codec *codec = dai->codec;
	struct twin_priv *priv = snd_soc_codec_get_drvdata(codec);
	int idx, ret;

	TRACE("enter\n");
	for (idx = 0; idx < priv->dai_count; idx++) {
		if (priv->dai[idx]) {
			ret = snd_soc_dai_set_sysclk(priv->dai[idx],
							clk_id, freq, dir);
			if (ret)
				return ret;
		}
	}
	TRACE("done\n");
	return 0;
}

static int twin_set_pll(struct snd_soc_dai *dai, int pll_id, int source,
			unsigned int freq_in, unsigned int freq_out)
{
	struct snd_soc_codec *codec = dai->codec;
	struct twin_priv *priv = snd_soc_codec_get_drvdata(codec);
	int idx, ret;

	TRACE("enter\n");
	for (idx = 0; idx < priv->dai_count; idx++) {
		if (priv->dai[idx]) {
			ret = snd_soc_dai_set_pll(priv->dai[idx],
					pll_id, source, freq_in, freq_out);
			if (ret)
				return ret;
		}
	}
	TRACE("done\n");
	return 0;
}

static int twin_set_clkdiv(struct snd_soc_dai *dai, int div_id, int div)
{
	struct snd_soc_codec *codec = dai->codec;
	struct twin_priv *priv = snd_soc_codec_get_drvdata(codec);
	int idx, ret;

	TRACE("enter\n");
	for (idx = 0; idx < priv->dai_count; idx++) {
		if (priv->dai[idx]) {
			ret = snd_soc_dai_set_clkdiv(priv->dai[idx],
							div_id, div);
			if (ret)
				return ret;
		}
	}
	TRACE("done\n");
	return 0;
}

static int twin_set_bclk_ratio(struct snd_soc_dai *dai, unsigned int ratio)
{
	struct snd_soc_codec *codec = dai->codec;
	struct twin_priv *priv = snd_soc_codec_get_drvdata(codec);
	int idx, ret;

	TRACE("enter\n");
	for (idx = 0; idx < priv->dai_count; idx++) {
		if (priv->dai[idx]) {
			ret = snd_soc_dai_set_bclk_ratio(priv->dai[idx],
								ratio);
			if (ret)
				return ret;
		}
	}
	TRACE("done\n");
	return 0;
}

static int twin_set_fmt(struct snd_soc_dai *dai,
			       unsigned int format)
{
	struct snd_soc_codec *codec = dai->codec;
	struct twin_priv *priv = snd_soc_codec_get_drvdata(codec);
	int idx, ret;

	TRACE("enter\n");
	for (idx = 0; idx < priv->dai_count; idx++) {
		if (check_fmt(priv->dai[idx], format))
			priv->dai_mute[idx] |= 0x01;
		else
			priv->dai_mute[idx] &= ~0x01;

		if (priv->dai[idx]
		 && !(priv->dai_mute[idx] & 0x01)) {
			ret = snd_soc_dai_set_fmt(priv->dai[idx], format);
			if (ret)
				priv->dai_mute[idx] |= 0x01;
			else
				priv->dai_mute[idx] &= ~0x01;
		}
	}
	TRACE("done\n");
	return 0;
}

static int twin_digital_mute(struct snd_soc_dai *dai, int mute)
{
	struct snd_soc_codec *codec = dai->codec;
	struct twin_priv *priv = snd_soc_codec_get_drvdata(codec);
	int idx, ret;

	TRACE("enter\n");
	for (idx = 0; idx < priv->dai_count; idx++) {
		if (priv->dai[idx] && !priv->dai_mute[idx]) {
			ret = snd_soc_dai_digital_mute(priv->dai[idx], mute,
						SNDRV_PCM_STREAM_PLAYBACK);
			if (ret)
				return ret;
		}
	}
	if (mute && priv->mute_busy)
		usleep_range(priv->mute_busy, priv->mute_busy * 101 / 100);
	TRACE("done\n");
	return 0;
}

static int twin_startup(struct snd_pcm_substream *substream,
			struct snd_soc_dai *dai)
{
	struct snd_soc_codec *codec = dai->codec;
	struct twin_priv *priv = snd_soc_codec_get_drvdata(codec);
	int idx, ret;

	TRACE("enter\n");
	for (idx = 0; idx < priv->dai_count; idx++) {
		if (priv->dai[idx]
		 && priv->dai[idx]->driver->ops
		 && priv->dai[idx]->driver->ops->startup) {
			ret = priv->dai[idx]->driver->ops->startup(substream,
							priv->dai[idx]);
			if (ret)
				return ret;
		}
	}
	TRACE("done\n");
	return 0;
}

static void twin_shutdown(struct snd_pcm_substream *substream,
			 struct snd_soc_dai *dai)
{
	struct snd_soc_codec *codec = dai->codec;
	struct twin_priv *priv = snd_soc_codec_get_drvdata(codec);
	int idx;

	TRACE("enter\n");
	for (idx = 0; idx < priv->dai_count; idx++) {
		if (priv->dai[idx]
		&& priv->dai[idx]->driver->ops
		&& priv->dai[idx]->driver->ops->shutdown)
			priv->dai[idx]->driver->ops->shutdown(substream,
							priv->dai[idx]);
	}
	TRACE("done\n");
}

static int twin_hw_params(struct snd_pcm_substream *substream,
			  struct snd_pcm_hw_params *params,
			  struct snd_soc_dai *dai)
{
	struct snd_soc_codec *codec = dai->codec;
	struct twin_priv *priv = snd_soc_codec_get_drvdata(codec);
	int idx, ret;

	TRACE("enter\n");
	for (idx = 0; idx < priv->dai_count; idx++) {
		if (check_param(priv->dai[idx], params))
			priv->dai_mute[idx] |= 0x02;
		else
			priv->dai_mute[idx] &= ~0x02;

		if (priv->dai[idx]
		 && !(priv->dai_mute[idx] & 0x02)) {
			ret = soc_dai_hw_params(substream, params,
						priv->dai[idx]);
			if (ret)
				priv->dai_mute[idx] |= 0x02;
			else
				priv->dai_mute[idx] &= ~0x02;
		}
	}
	TRACE("done\n");
	return 0;
}

static int twin_hw_free(struct snd_pcm_substream *substream,
			struct snd_soc_dai *dai)
{
	struct snd_soc_codec *codec = dai->codec;
	struct twin_priv *priv = snd_soc_codec_get_drvdata(codec);
	int idx, ret;

	TRACE("enter\n");
	for (idx = 0; idx < priv->dai_count; idx++) {
		if (priv->dai[idx] && !priv->dai_mute[idx]
		 && priv->dai[idx]->driver->ops
		 && priv->dai[idx]->driver->ops->hw_free) {
			ret = priv->dai[idx]->driver->ops->hw_free(substream,
							priv->dai[idx]);
			if (ret)
				return ret;
		}
	}
	TRACE("done\n");
	return 0;
}

static int twin_prepare(struct snd_pcm_substream *substream,
			struct snd_soc_dai *dai)
{
	struct snd_soc_codec *codec = dai->codec;
	struct twin_priv *priv = snd_soc_codec_get_drvdata(codec);
	int idx, ret;

	TRACE("enter\n");
	for (idx = 0; idx < priv->dai_count; idx++) {
		if (priv->dai[idx] && !priv->dai_mute[idx]
		 && priv->dai[idx]->driver->ops
		 && priv->dai[idx]->driver->ops->prepare) {
			ret = priv->dai[idx]->driver->ops->prepare(substream,
							priv->dai[idx]);
			if (ret)
				return ret;
		}
	}
	TRACE("done\n");
	return 0;
}

static int twin_trigger(struct snd_pcm_substream  *substream,
			int cmd, struct snd_soc_dai *dai)
{
	struct snd_soc_codec *codec = dai->codec;
	struct twin_priv *priv = snd_soc_codec_get_drvdata(codec);
	int idx, ret;

	TRACE("enter\n");
	if (cmd == SNDRV_PCM_TRIGGER_STOP)
		blocking_notifier_call_chain(&twincodec_notify_list, 1, NULL);

	for (idx = 0; idx < priv->dai_count; idx++) {
		if (priv->dai[idx] && !priv->dai_mute[idx]
		 && priv->dai[idx]->driver->ops
		 && priv->dai[idx]->driver->ops->trigger) {
			ret = priv->dai[idx]->driver->ops->trigger(substream,
							cmd, priv->dai[idx]);
			if (ret)
				return ret;
		}
	}
	TRACE("done\n");
	return 0;
}

static int twin_codec_probe(struct snd_soc_codec *codec)
{
	struct twin_priv *priv = snd_soc_codec_get_drvdata(codec);
	struct snd_soc_dai_link_component compo;
	int idx;

	TRACE("enter\n");
	priv->codec = codec;

	for (idx = 0; idx < priv->dai_count; idx++) {
		compo.dai_name = priv->dai_name[idx];
		compo.of_node = NULL;
		compo.name = NULL;
		priv->dai[idx] = snd_soc_find_dai(&compo);
		DEBUG("dai%d:%s(0x%p)\n", idx, priv->dai_name[idx],
						priv->dai[idx]);
		if (priv->probe_child_codec) {
			if (priv->dai[idx]
			 && priv->dai[idx]->codec
			 && priv->dai[idx]->codec->driver
			 && priv->dai[idx]->codec->driver->probe)
				priv->dai[idx]->codec->driver->probe(
						priv->dai[idx]->codec);
		}
	}

	/* Register controls for child codecs */
	if (priv->dai[0]
	 && priv->dai[0]->codec
	 && priv->dai[0]->codec->driver) {
		priv->codec_driver.component_driver.controls =
			priv->dai[0]->codec->driver->component_driver.controls;
		priv->codec_driver.component_driver.num_controls =
			priv->dai[0]->codec->driver->component_driver.num_controls;
	}

	TRACE("done\n");
	return 0;
}

static int twin_codec_remove(struct snd_soc_codec *codec)
{
	struct twin_priv *priv = snd_soc_codec_get_drvdata(codec);
	int idx;

	TRACE("enter\n");
	if (priv->probe_child_codec) {
		for (idx = 0; idx < priv->dai_count; idx++) {
			if (priv->dai[idx]
			 && priv->dai[idx]->codec
			 && priv->dai[idx]->codec->driver
			 && priv->dai[idx]->codec->driver->remove)
				priv->dai[idx]->codec->driver->remove(
						priv->dai[idx]->codec);
		}
	}
	TRACE("done\n");
	return 0;
}

static int twin_codec_suspend(struct snd_soc_codec *codec)
{
	struct twin_priv *priv = snd_soc_codec_get_drvdata(codec);
	int idx;

	TRACE("enter\n");
	if (priv->probe_child_codec) {
		for (idx = 0; idx < priv->dai_count; idx++) {
			if (priv->dai[idx]
			 && priv->dai[idx]->codec
			 && priv->dai[idx]->codec->driver
			 && priv->dai[idx]->codec->driver->suspend)
				priv->dai[idx]->codec->driver->suspend(
						priv->dai[idx]->codec);
		}
	}
	TRACE("done\n");
	return 0;
}

static int twin_codec_resume(struct snd_soc_codec *codec)
{
	struct twin_priv *priv = snd_soc_codec_get_drvdata(codec);
	int idx;

	TRACE("enter\n");
	if (priv->probe_child_codec) {
		for (idx = 0; idx < priv->dai_count; idx++) {
			if (priv->dai[idx]
			 && priv->dai[idx]->codec
			 && priv->dai[idx]->codec->driver
			 && priv->dai[idx]->codec->driver->resume)
				priv->dai[idx]->codec->driver->resume(
						priv->dai[idx]->codec);
		}
	}
	TRACE("done\n");
	return 0;
}

static const struct snd_soc_dai_ops twin_dai_ops = {
	.set_sysclk	= twin_set_sysclk,
	.set_pll	= twin_set_pll,
	.set_clkdiv	= twin_set_clkdiv,
	.set_bclk_ratio	= twin_set_bclk_ratio,
	.set_fmt	= twin_set_fmt,
	.digital_mute	= twin_digital_mute,
	.startup	= twin_startup,
	.shutdown	= twin_shutdown,
	.hw_params	= twin_hw_params,
	.hw_free	= twin_hw_free,
	.prepare	= twin_prepare,
	.trigger	= twin_trigger,
};

static struct snd_soc_dai_driver twin_dai_driver = {
	.name = "twincodec-dai",
	.playback = {
		.stream_name = "Playback",
		.channels_min = 1,
		.channels_max = 20,
		.rates = SNDRV_PCM_RATE_KNOT,
		.formats = SNDRV_PCM_FMTBIT_S16_LE |
			   SNDRV_PCM_FMTBIT_S24_LE |
			   SNDRV_PCM_FMTBIT_S32_LE |
			   SNDRV_PCM_FMTBIT_DSD_U8 |
			   SNDRV_PCM_FMTBIT_DSD_U16_LE |
			   SNDRV_PCM_FMTBIT_DSD_U16_BE |
			   SNDRV_PCM_FMTBIT_DSD_U32_LE |
			   SNDRV_PCM_FMTBIT_DSD_U32_BE,
	},
	.ops = &twin_dai_ops,
};

static const struct snd_soc_codec_driver twin_codec_driver = {
	.probe = twin_codec_probe,
	.remove = twin_codec_remove,
	.suspend = twin_codec_suspend,
	.resume = twin_codec_resume,
	.component_driver = {
		.dapm_widgets		= twin_dapm_widgets,
		.num_dapm_widgets	= ARRAY_SIZE(twin_dapm_widgets),
		.dapm_routes		= twin_dapm_routes,
		.num_dapm_routes	= ARRAY_SIZE(twin_dapm_routes)
	},
};

/* Platform driver functions */

static int twin_probe(struct platform_device *pdev)
{
	struct twin_priv *priv;
	struct device *dev = &pdev->dev;
	int idx, ret;
	char *nam;
	unsigned int val;

	TRACE("enter\n");
	priv = devm_kzalloc(dev, sizeof(struct twin_priv), GFP_KERNEL);
	if (priv == NULL) {
		dev_err(dev, "failed memory allocate\n");
		return -ENOMEM;
	}
	dev_set_drvdata(dev, priv);

	for (idx = 0; idx < MAX_DAI_COUNT; idx++) {
		ret = of_property_read_string_index(dev->of_node,
							"codec-dais",
							idx,
							(const char **)&nam);
		if (ret)
			break;
		priv->dai_name[idx] = (char *)nam;
		priv->dai_mute[idx] = 0x00;
		priv->dai[idx] = NULL;
		DEBUG("dai%d: %s\n", idx, nam);
	}
	priv->dai_count = idx;
	DEBUG("dai count: %d\n", idx);

	if (!of_property_read_u32(dev->of_node, "mute-busy", &val))
		priv->mute_busy = val;
	priv->probe_child_codec = of_property_read_bool(dev->of_node,
							"probe-child-codec");
	priv->sync_runtime_pm = of_property_read_bool(dev->of_node,
							"sync_runtime_pm");

	memcpy(&priv->codec_driver, &twin_codec_driver,
			sizeof(struct snd_soc_codec_driver));

	ret = snd_soc_register_codec(dev, &priv->codec_driver,
						&twin_dai_driver, 1);
	if (ret) {
		dev_err(dev, "failed to register codec(%d)\n", ret);
		return ret;
	}

#ifdef CONFIG_PM
	pm_runtime_enable(dev);
#endif
	TRACE("done\n");
	return 0;
}

static int twin_remove(struct platform_device *pdev)
{
	TRACE("enter\n");
	snd_soc_unregister_codec(&pdev->dev);
	TRACE("done\n");
	return 0;
}

#ifdef CONFIG_PM
static int twin_rt_resume(struct device *dev)
{
	struct twin_priv *priv = dev_get_drvdata(dev);
	int idx;

	TRACE("enter\n");
	if (priv->sync_runtime_pm) {
		for (idx = 0; idx < priv->dai_count; idx++) {
			if (priv->dai[idx]
			 && priv->dai[idx]->dev
			 && priv->dai[idx]->dev->driver
			 && priv->dai[idx]->dev->driver->pm)
				pm_runtime_get(priv->dai[idx]->dev);
		}
	}
	TRACE("done\n");
	return 0;
}

static int twin_rt_suspend(struct device *dev)
{
	struct twin_priv *priv = dev_get_drvdata(dev);
	int idx;

	TRACE("enter\n");
	if (priv->sync_runtime_pm) {
		for (idx = 0; idx < priv->dai_count; idx++) {
			if (priv->dai[idx]
			 && priv->dai[idx]->dev
			 && priv->dai[idx]->dev->driver
			 && priv->dai[idx]->dev->driver->pm)
				pm_runtime_put(priv->dai[idx]->dev);
		}
	}
	TRACE("done\n");
	return 0;
}

const struct dev_pm_ops twin_pm_ops = {
	SET_RUNTIME_PM_OPS(twin_rt_suspend, twin_rt_resume, NULL)
};
#endif

static const struct of_device_id twin_of_match[] = {
	{ .compatible = "sony,twincodec", },
	{ }
};
MODULE_DEVICE_TABLE(of, twin_of_match);

static struct platform_driver twincodec_driver = {
	.probe		= twin_probe,
	.remove		= twin_remove,
	.driver		= {
		.name	= "twincodec",
		.of_match_table = twin_of_match,
#ifdef CONFIG_PM
		.pm		= &twin_pm_ops,
#endif
	},
};

module_platform_driver(twincodec_driver);
MODULE_DESCRIPTION("twin-codec pipe driver");
MODULE_AUTHOR("Sony Corporation");
MODULE_LICENSE("GPL v2");
