/*
 * Copyright 2017 NXP
 * Copyright 2019 Sony Home Entertainment & Sound Products Inc.
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

#include <linux/module.h>
#include <sound/soc.h>
#include <sound/pcm_params.h>
#include <linux/of_device.h>
#include <linux/of_gpio.h>

#include "fsl_sai.h"


struct sony_zone_data {
	int mute_gpio;
};


static const u32 pcm5102_rates[] = {
	8000, 11025, 16000, 22050,
	32000, 44100, 48000, 88200,
	96000, 176400, 192000, 352800,
	384000, 
};

static const u32 pcm5102_channels[] = {
	1, 2,
};


#define FIXED_MCLK	24576000	/* 24MHz. This also used MCLK for opt-in */

static int sony_pcm5102_hw_params(struct snd_pcm_substream *substream,
                struct snd_pcm_hw_params *params)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_dai *cpu_dai = rtd->cpu_dai;
	struct snd_soc_card *card = rtd->card;
	struct device *dev = card->dev;
	unsigned int channels = params_channels(params);
	unsigned int fmt = SND_SOC_DAIFMT_NB_NF | SND_SOC_DAIFMT_CBS_CFS;
	unsigned long mclk_freq;
	int ret;

	fmt |= SND_SOC_DAIFMT_I2S;

	ret = snd_soc_dai_set_fmt(cpu_dai, fmt);
	if (ret) {              
		dev_err(dev, "failed to set cpu dai fmt: %d\n", ret);
		return ret;
	}

	ret = snd_soc_dai_set_tdm_slot(cpu_dai,
					BIT(channels) - 1, BIT(channels) - 1,
					2, params_physical_width(params));
	if (ret) {
		dev_err(dev, "failed to set cpu dai tdm slot: %d\n", ret);
		return ret;
	}

	mclk_freq = FIXED_MCLK;
	ret = snd_soc_dai_set_sysclk(cpu_dai, FSL_SAI_CLK_MAST1, mclk_freq,
					SND_SOC_CLOCK_OUT);
	if (ret < 0)
		dev_err(dev, "failed to set cpui dai mclk1 rate (%lu): %d\n",
			mclk_freq, ret);

	return ret;
}


static int sony_pcm5102_startup(struct snd_pcm_substream *substream)
{
	struct snd_pcm_runtime *runtime = substream->runtime;
	static struct snd_pcm_hw_constraint_list constraint_rates;
	static struct snd_pcm_hw_constraint_list constraint_channels;
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_card *card = rtd->card;
	struct sony_zone_data *data = snd_soc_card_get_drvdata(card);
	int ret;

	constraint_channels.list = pcm5102_channels;
	constraint_channels.count = ARRAY_SIZE(pcm5102_channels);
	constraint_rates.list = pcm5102_rates;
	constraint_rates.count = ARRAY_SIZE(pcm5102_rates);

	ret = snd_pcm_hw_constraint_list(runtime, 0, SNDRV_PCM_HW_PARAM_CHANNELS,
						&constraint_channels);
	if (ret)
		return ret;

	ret = snd_pcm_hw_constraint_list(runtime, 0, SNDRV_PCM_HW_PARAM_RATE,
						&constraint_rates);
	if (ret)
		return ret;

	gpio_set_value_cansleep(data->mute_gpio, 1);

	return 0;
}

static void sony_pcm5102_shutdown(struct snd_pcm_substream *substream)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_card *card = rtd->card;
	struct sony_zone_data *data = snd_soc_card_get_drvdata(card);

	gpio_set_value_cansleep(data->mute_gpio, 0);

}


static struct snd_soc_ops sony_pcm5102_ops = {
	.hw_params = sony_pcm5102_hw_params,
	.startup = sony_pcm5102_startup,
	.shutdown = sony_pcm5102_shutdown,
};

static struct snd_soc_dai_link sony_pcm5102_dai = {
	.name = "pcm5102--out",
	.stream_name = "pcm5102-out",
	.codec_dai_name = "snd-soc-dummy-dai",
	.codec_name = "snd-soc-dummy",
	.ops = &sony_pcm5102_ops,
	.playback_only = 1,
};

static struct snd_soc_card sony_zone_out_card = {
	.dai_link = &sony_pcm5102_dai,
	.num_links = 1,
};

static int sony_zone_out_probe(struct platform_device *pdev)
{
	int ret;
	struct device_node *cpu_np/*, *codec_np*/;
	struct platform_device *cpu_pdev;
	struct sony_zone_data *priv;


	pr_debug("%s()\n", __func__);

	priv = devm_kzalloc(&pdev->dev, sizeof(*priv), GFP_KERNEL);
	if (!priv)
		return -ENOMEM;

	cpu_np = of_parse_phandle(pdev->dev.of_node, "audio-cpu", 0);
	if (!cpu_np) {
		dev_err(&pdev->dev, "audio dai phandle missing or invalid\n");
		ret = -EINVAL;
		goto fail;
	}

	cpu_pdev = of_find_device_by_node(cpu_np);
	if (!cpu_pdev) {
		dev_err(&pdev->dev, "failed to find SAI platform device\n");
		ret = -EINVAL;
		goto fail;
	}

	sony_pcm5102_dai.cpu_dai_name = dev_name(&cpu_pdev->dev);
	sony_pcm5102_dai.platform_of_node = cpu_np;


	sony_zone_out_card.dev = &pdev->dev;
	sony_zone_out_card.owner = THIS_MODULE;
	ret = snd_soc_of_parse_card_name(&sony_zone_out_card, "model");
	if (ret)
		goto fail;

	priv->mute_gpio = of_get_named_gpio(pdev->dev.of_node, "mute-gpio", 0);
	if (gpio_is_valid(priv->mute_gpio)) {
		ret = devm_gpio_request_one(&pdev->dev, priv->mute_gpio,
				GPIOF_OUT_INIT_LOW, "zone_mute");
		if (ret) {
			dev_err(&pdev->dev, "failed to get mute gpio\n");
			goto fail;
		}
	} else {
		dev_err(&pdev->dev, "invalid mute gpio\n");
		goto fail;
	}

	snd_soc_card_set_drvdata(&sony_zone_out_card, priv);
	ret = snd_soc_register_card(&sony_zone_out_card);
	if (ret) {
		dev_err(&pdev->dev, "failed to find SAI platform device\n");
		goto fail;
	}

	return 0;	

fail:
	if (cpu_np)
		of_node_put(cpu_np);

	return ret; 
}

static int sony_zone_out_remove(struct platform_device *pdev)
{
	struct snd_soc_card *card = platform_get_drvdata(pdev);

	return snd_soc_unregister_card(card);
}

#ifdef CONFIG_OF
static const struct of_device_id sony_zone_out_dt_match[] = {
	{.compatible = "fsl,sony-zone-out",},
	{}
};
#endif

static struct platform_driver sony_zone_out = {
	.driver = {
		.name = "sony-zone-out",
		.owner = THIS_MODULE,
		#ifdef CONFIG_OF
		.of_match_table = sony_zone_out_dt_match,
		#endif
	},
	.probe = sony_zone_out_probe,
	.remove = sony_zone_out_remove
};

module_platform_driver(sony_zone_out);

/* Module information */
MODULE_DESCRIPTION("sony zone out driver");
MODULE_LICENSE("GPL");
MODULE_ALIAS("sony zone out card");

