/*
 * Copyright (C) 2014 Gateworks Corporation
 * Copyright (C) 2011-2012 Freescale Semiconductor, Inc.
 *
 * Author: Tim Harvey <tharvey@gateworks.com>
 *
 * SPDX-License-Identifier:     GPL-2.0+
 */

#include <common.h>
#include <asm/io.h>
#include <asm/arch/imx-regs.h>
#include <asm/arch/sys_proto.h>
#include <asm/spl.h>
#include <spl.h>
#include <asm/mach-imx/hab.h>
#include <asm/mach-imx/boot_mode.h>
#include <g_dnl.h>
#include <u-boot/sha256.h>
#include <image.h>
#include <asm/sections.h>

DECLARE_GLOBAL_DATA_PTR;

#if defined(CONFIG_MX6)
/* determine boot device from SRC_SBMR1 (BOOT_CFG[4:1]) or SRC_GPR9 register */
u32 spl_boot_device(void)
{
	unsigned int bmode = readl(&src_base->sbmr2);
	u32 reg = imx6_src_get_boot_mode();

	/*
	 * Check for BMODE if serial downloader is enabled
	 * BOOT_MODE - see IMX6DQRM Table 8-1
	 */
	if (((bmode >> 24) & 0x03) == 0x01) /* Serial Downloader */
		return BOOT_DEVICE_BOARD;

	/*
	 * The above method does not detect that the boot ROM used
	 * serial downloader in case the boot ROM decided to use the
	 * serial downloader as a fall back (primary boot source failed).
	 *
	 * Infer that the boot ROM used the USB serial downloader by
	 * checking whether the USB PHY is currently active... This
	 * assumes that SPL did not (yet) initialize the USB PHY...
	 */
	if (is_usbotg_phy_active())
		return BOOT_DEVICE_BOARD;

	/* BOOT_CFG1[7:4] - see IMX6DQRM Table 8-8 */
	switch ((reg & IMX6_BMODE_MASK) >> IMX6_BMODE_SHIFT) {
	 /* EIM: See 8.5.1, Table 8-9 */
	case IMX6_BMODE_EMI:
		/* BOOT_CFG1[3]: NOR/OneNAND Selection */
		switch ((reg & IMX6_BMODE_EMI_MASK) >> IMX6_BMODE_EMI_SHIFT) {
		case IMX6_BMODE_ONENAND:
			return BOOT_DEVICE_ONENAND;
		case IMX6_BMODE_NOR:
			return BOOT_DEVICE_NOR;
		break;
		}
	/* Reserved: Used to force Serial Downloader */
	case IMX6_BMODE_RESERVED:
		return BOOT_DEVICE_BOARD;
	/* SATA: See 8.5.4, Table 8-20 */
#if !defined(CONFIG_MX6UL) && !defined(CONFIG_MX6ULL)
	case IMX6_BMODE_SATA:
		return BOOT_DEVICE_SATA;
#endif
	/* Serial ROM: See 8.5.5.1, Table 8-22 */
	case IMX6_BMODE_SERIAL_ROM:
		/* BOOT_CFG4[2:0] */
		switch ((reg & IMX6_BMODE_SERIAL_ROM_MASK) >>
			IMX6_BMODE_SERIAL_ROM_SHIFT) {
		case IMX6_BMODE_ECSPI1:
		case IMX6_BMODE_ECSPI2:
		case IMX6_BMODE_ECSPI3:
		case IMX6_BMODE_ECSPI4:
		case IMX6_BMODE_ECSPI5:
			return BOOT_DEVICE_SPI;
		case IMX6_BMODE_I2C1:
		case IMX6_BMODE_I2C2:
		case IMX6_BMODE_I2C3:
			return BOOT_DEVICE_I2C;
		}
		break;
	/* SD/eSD: 8.5.3, Table 8-15  */
	case IMX6_BMODE_SD:
	case IMX6_BMODE_ESD:
		return BOOT_DEVICE_MMC1;
	/* MMC/eMMC: 8.5.3 */
	case IMX6_BMODE_MMC:
	case IMX6_BMODE_EMMC:
		return BOOT_DEVICE_MMC1;
	/* NAND Flash: 8.5.2, Table 8-10 */
	case IMX6_BMODE_NAND_MIN ... IMX6_BMODE_NAND_MAX:
		return BOOT_DEVICE_NAND;
	}
	return BOOT_DEVICE_NONE;
}

#elif defined(CONFIG_MX7) || defined(CONFIG_IMX8M) || defined(CONFIG_IMX8)
/* Translate iMX7/iMX8M boot device to the SPL boot device enumeration */
u32 spl_boot_device(void)
{
	enum boot_device boot_device_spl = get_boot_device();

	switch (boot_device_spl) {
#if defined(CONFIG_MX7)
	case SD1_BOOT:
	case MMC1_BOOT:
	case SD2_BOOT:
	case MMC2_BOOT:
	case SD3_BOOT:
	case MMC3_BOOT:
		return BOOT_DEVICE_MMC1;
#elif defined(CONFIG_IMX8)
	case MMC1_BOOT:
                return BOOT_DEVICE_MMC1;
	case SD2_BOOT:
		return BOOT_DEVICE_MMC2_2;
        case SD3_BOOT:
                return BOOT_DEVICE_MMC1;
	case FLEXSPI_BOOT:
		return BOOT_DEVICE_SPI;
#elif defined(CONFIG_IMX8M)
	case SD1_BOOT:
	case MMC1_BOOT:
		return BOOT_DEVICE_MMC1;
#if defined(CONFIG_IMX8MM)
	case SD2_BOOT:
	case MMC2_BOOT:
		return BOOT_DEVICE_MMC1;
	case SD3_BOOT:
	case MMC3_BOOT:
		return BOOT_DEVICE_MMC2;
#else
	case SD2_BOOT:
	case MMC2_BOOT:
		return BOOT_DEVICE_MMC2;
#endif
#endif
	case NAND_BOOT:
		return BOOT_DEVICE_NAND;
	case SPI_NOR_BOOT:
		return BOOT_DEVICE_SPI;
	case QSPI_BOOT:
		return BOOT_DEVICE_NOR;
	case USB_BOOT:
		return BOOT_DEVICE_BOARD;
	default:
		return BOOT_DEVICE_NONE;
	}
}
#endif /* CONFIG_MX6 || CONFIG_MX7 || CONFIG_IMX8M || CONFIG_IMX8 */

#ifdef CONFIG_SPL_USB_GADGET_SUPPORT
int g_dnl_bind_fixup(struct usb_device_descriptor *dev, const char *name)
{
	put_unaligned(CONFIG_USB_GADGET_PRODUCT_NUM + 0xfff, &dev->idProduct);

	return 0;
}
#endif

#if defined(CONFIG_SPL_MMC_SUPPORT)
/* called from spl_mmc to see type of boot mode for storage (RAW or FAT) */
u32 spl_boot_mode(const u32 boot_device)
{
#if defined(CONFIG_MX7) || defined(CONFIG_IMX8M) || defined(CONFIG_IMX8)
	switch (get_boot_device()) {
	/* for MMC return either RAW or FAT mode */
	case SD1_BOOT:
	case SD2_BOOT:
	case SD3_BOOT:
#if defined(CONFIG_SPL_FAT_SUPPORT)
		return MMCSD_MODE_FS;
#else
		return MMCSD_MODE_RAW;
#endif
		break;
	case MMC1_BOOT:
	case MMC2_BOOT:
	case MMC3_BOOT:
#if defined(CONFIG_SPL_FAT_SUPPORT)
		return MMCSD_MODE_FS;
#elif defined(CONFIG_SUPPORT_EMMC_BOOT)
		return MMCSD_MODE_EMMCBOOT;
#else
		return MMCSD_MODE_RAW;
#endif
		break;
	default:
		puts("spl: ERROR:  unsupported device\n");
		hang();
	}

#else /* defined(CONFIG_MX7) || defined(CONFIG_IMX8M) || defined(CONFIG_IMX8) */
	switch (spl_boot_device()) {
	/* for MMC return either RAW or FAT mode */
	case BOOT_DEVICE_MMC1:
	case BOOT_DEVICE_MMC2:
#if defined(CONFIG_SPL_FAT_SUPPORT)
		return MMCSD_MODE_FS;
#elif defined(CONFIG_SUPPORT_EMMC_BOOT)
		return MMCSD_MODE_EMMCBOOT;
#else
		return MMCSD_MODE_RAW;
#endif
		break;
	default:
		puts("spl: ERROR:  unsupported device\n");
		hang();
	}
#endif
}
#endif

#if defined(CONFIG_SECURE_BOOT)

/*
 * +------------+  0x0 (DDR_UIMAGE_START) -
 * |   Header   |                          |
 * +------------+  0x40                    |
 * |            |                          |
 * |            |                          |
 * |            |                          |
 * |            |                          |
 * | Image Data |                          |
 * .            |                          |
 * .            |                           > Stuff to be authenticated ----+
 * .            |                          |                                |
 * |            |                          |                                |
 * |            |                          |                                |
 * +------------+                          |                                |
 * |            |                          |                                |
 * | Fill Data  |                          |                                |
 * |            |                          |                                |
 * +------------+ Align to ALIGN_SIZE      |                                |
 * |    IVT     |                          |                                |
 * +------------+ + IVT_SIZE              -                                 |
 * |            |                                                           |
 * |  CSF DATA  | <---------------------------------------------------------+
 * |            |
 * +------------+
 * |            |
 * | Fill Data  |
 * |            |
 * +------------+ + CSF_PAD_SIZE
 */

__weak void __noreturn jump_to_image_no_args(struct spl_image_info *spl_image)
{
	typedef void __noreturn (*image_entry_noargs_t)(void);
	uint32_t offset;

	image_entry_noargs_t image_entry =
		(image_entry_noargs_t)(unsigned long)spl_image->entry_point;

	debug("image entry point: 0x%lX\n", spl_image->entry_point);

	if (spl_image->flags & SPL_FIT_FOUND) {
		image_entry();
	} else {
		/* HAB looks for the CSF at the end of the authenticated data therefore,
		 * we need to subtract the size of the CSF from the actual filesize */
		offset = spl_image->size - CONFIG_CSF_SIZE;
		if (!imx_hab_authenticate_image(spl_image->load_addr,
						offset + IVT_SIZE + CSF_PAD_SIZE,
						offset)) {
			image_entry();
		} else {
			puts("spl: ERROR:  image authentication unsuccessful\n");
			hang();
		}
	}
}

ulong board_spl_fit_size_align(ulong size)
{
	/* HAB authenticate_image requests the IVT offset is aligned to 0x1000 */
#define ALIGN_SIZE		0x1000

	size = ALIGN(size, ALIGN_SIZE);
	size += 2 * CONFIG_CSF_SIZE;

	if (size > CONFIG_SYS_BOOTM_LEN)
		panic("spl: ERROR: image too big\n");

	return size;
}

void board_spl_fit_post_load(ulong load_addr, size_t length)
{
	uint32_t offset = length - CONFIG_CSF_SIZE;
	if (imx_hab_authenticate_image(load_addr,
					offset + IVT_SIZE + CSF_PAD_SIZE,
					offset)) {
		puts("spl: ERROR:  image authentication unsuccessful\n");
		hang();
	}
}

void* board_spl_fit_buffer_addr(ulong fit_size, int bl_len)
{
	int align_len = ARCH_DMA_MINALIGN - 1;

	/* Some devices like SDP, NOR, NAND, SPI are using bl_len =1, so their fit address
	 * is different with SD/MMC, this cause mismatch with signed address. Thus, adjust
	 * the bl_len to align with SD/MMC.
	 */
	if (bl_len < 512)
		bl_len = 512;

	return  (void *)((CONFIG_SYS_TEXT_BASE - fit_size - bl_len -
			align_len) & ~align_len);
}
#endif

#if defined(CONFIG_MX6) && defined(CONFIG_SPL_OS_BOOT)
int dram_init_banksize(void)
{
	gd->bd->bi_dram[0].start = CONFIG_SYS_SDRAM_BASE;
	gd->bd->bi_dram[0].size = imx_ddr_size();

	return 0;
}
#endif

#if IS_ENABLED(CONFIG_SPL_LOAD_FIT)
static int spl_verify_fit_hash(const void *fit)
{
	unsigned long size;
	u8 value[SHA256_SUM_LEN];
	int value_len;
	ulong fit_hash;

#if CONFIG_IS_ENABLED(OF_CONTROL)
	if (gd->fdt_blob && !fdt_check_header(gd->fdt_blob)) {
		fit_hash = roundup((unsigned long)&_end +
				     fdt_totalsize(gd->fdt_blob), 4) + 0x18000;
	}
#else
	fit_hash = (unsigned long)&_end + 0x18000;
#endif

	size = fdt_totalsize(fit);

	if (calculate_hash(fit, size, "sha256", value, &value_len)) {
		printf("Unsupported hash algorithm\n");
		return -1;
	}

	if (value_len != SHA256_SUM_LEN) {
		printf("Bad hash value len\n");
		return -1;
	} else if (memcmp(value, (const void *)fit_hash, value_len) != 0) {
		printf("Bad hash value\n");
		return -1;
	}

	return 0;
}

/*
 * read the address where the IVT header must sit
 * from IVT image header, loaded from SPL into
 * an malloced buffer and copy the IVT header
 * to this address
 */
void *spl_load_simple_fit_fix_load(const void *fit)
{
	struct ivt *ivt;
	unsigned long new;
	unsigned long offset;
	unsigned long size;
	u8 *tmp = (u8 *)fit;

	if (IS_ENABLED(CONFIG_SECURE_BOOT)) {
		if (IS_ENABLED(CONFIG_IMX_SPL_FIT_FDT_SIGNATURE)) {
			u32 offset = ALIGN(fdt_totalsize(fit), 0x1000);

			if (imx_hab_authenticate_image((uintptr_t)fit,
						       offset + 2 * CSF_PAD_SIZE,
						       offset + CSF_PAD_SIZE)) {
#ifdef CONFIG_ANDROID_SUPPORT
				printf("spl: ERROR:  FIT FDT authentication unsuccessful\n");
				return NULL;
#else
				panic("spl: ERROR:  FIT FDT authentication unsuccessful\n");
#endif
			}
		} else {
			int ret = spl_verify_fit_hash(fit);

			if (ret && imx_hab_is_enabled())
				panic("spl: ERROR:  FIT hash verify unsuccessful\n");

			debug("spl_verify_fit_hash %d\n", ret);
		}
	}

	offset = ALIGN(fdt_totalsize(fit), 0x1000);
	size = ALIGN(fdt_totalsize(fit), 4);
	size = board_spl_fit_size_align(size);
	tmp += offset;
	ivt = (struct ivt *)tmp;
	if (ivt->hdr.magic != IVT_HEADER_MAGIC) {
		debug("no IVT header found\n");
		return (void *)fit;
	}
	debug("%s: ivt: %p offset: %lx size: %lx\n", __func__, ivt, offset, size);
	debug("%s: ivt self: %x\n", __func__, ivt->self);
	new = ivt->self;
	new -= offset;
	debug("%s: new %lx\n", __func__, new);
	memcpy((void *)new, fit, size);

	return (void *)new;
}
#endif
