/*
 * Copyright 2008 - 2009 Windriver, <www.windriver.com>
 * Author: Tom Rix <Tom.Rix@windriver.com>
 *
 * (C) Copyright 2014 Linaro, Ltd.
 * Rob Herring <robh@kernel.org>
 * Copyright 2019 Sony Home Entertainment & Sound Products Inc.
 *
 * SPDX-License-Identifier:	GPL-2.0+
 */
#include <common.h>
#include <command.h>
#include <console.h>
#include <g_dnl.h>
#include <usb.h>
#include <div64.h>

static unsigned long long tick_to_ms(uint64_t tick)
{	unsigned long tb = get_tbclk();

	tick *= 1000;
	do_div(tick, tb);
	return tick;
}

static int do_fastboot(cmd_tbl_t *cmdtp, int flag, int argc, char *const argv[])
{
	int controller_index;
	char *usb_controller;
	char *timeout;
	int ret;
	uint64_t tick_base;
	uint64_t tick_elapsed;
	int timeout_ms = 0;

	if (argc < 2)
		return CMD_RET_USAGE;

	usb_controller = argv[1];
	controller_index = simple_strtoul(usb_controller, NULL, 0);
#ifdef CONFIG_FASTBOOT_USB_DEV
	controller_index = CONFIG_FASTBOOT_USB_DEV;
#endif
	if (argc == 3) {
		timeout = argv[2];
		timeout_ms = simple_strtoul(timeout, NULL, 0);
	}

	ret = board_usb_init(controller_index, USB_INIT_DEVICE);
	if (ret) {
		pr_err("USB init failed: %d", ret);
		return CMD_RET_FAILURE;
	}

	g_dnl_clear_detach();
	ret = g_dnl_register("usb_dnl_fastboot");
	if (ret)
		return ret;

	if (!g_dnl_board_usb_cable_connected()) {
		puts("\rUSB cable not detected.\n" \
		     "Command exit.\n");
		ret = CMD_RET_FAILURE;
		goto exit;
	}

	tick_base = get_ticks();
	while (1) {
		tick_elapsed = get_ticks() - tick_base;
		if (g_dnl_detach())
			break;
		if (ctrlc())
			break;
		if (timeout_ms && !g_dnl_requested()
		        && (tick_to_ms(tick_elapsed) > timeout_ms)) {
			puts("fastboot timeout\n");
			break;
		}
		usb_gadget_handle_interrupts(controller_index);
	}

	ret = CMD_RET_SUCCESS;

exit:
	g_dnl_unregister();
	g_dnl_clear_detach();
	board_usb_cleanup(controller_index, USB_INIT_DEVICE);

	return ret;
}

U_BOOT_CMD(
	fastboot, 3, 1, do_fastboot,
	"use USB Fastboot protocol",
	"<USB_controller> [timeout_ms]\n"
	"    - run as a fastboot usb device"
);
