/* SPDX-License-Identifier: GPL-2.0 */
/*
 * sircs.c - sircs driver
 *
 * Copyright 2019 Sony Home Entertainment & Sound Products Inc.
 * Copyright 2021 Sony Home Entertainment & Sound Products Inc.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/virtio.h>
#include <linux/rpmsg.h>
#include <linux/err.h>
#include <linux/cdev.h>
#include <linux/platform_device.h>
#include <linux/input.h>
#include <linux/cm4.h>

#define _SIRCS_C_
#include "sircs_code_table.h"

#define DRV_NAME                "sircs"
#define ERRLOG(fmt, ...)        pr_err(DRV_NAME ": Error: " fmt, ## __VA_ARGS__)
#define WRNLOG(fmt, ...)        pr_warn(DRV_NAME ": Warning: " fmt, ## __VA_ARGS__)
#define MSGLOG(fmt, ...)        pr_info(DRV_NAME ": " fmt, ## __VA_ARGS__)
#define DBGLOG(fmt, ...)        pr_debug(DRV_NAME ": %s(%d): " fmt, __func__, __LINE__, ## __VA_ARGS__)


/*******************************************************************************
 * Definition
*******************************************************************************/
#define IR_DRV_NAME             "irdrv"
#define SIRCS_DRIVER_NAME       "sircs_key"
#define SIRCS_DRIVER_VERSION    (11)
#define SIRCS_DRIVER_PRODUCT    (11)
#define SIRCS_DRIVER_VENDOR     (11)

#define CM4COM_TYPE_REQ         (0x01)

/*******************************************************************************
 * Variables
*******************************************************************************/
static struct input_dev *sircs_input_dev = NULL;

/*******************************************************************************
 * Functions
*******************************************************************************/
void sircs_cm4_callback(u8 type, u8 length, u8 seq, u8 *data)
{
    int i = 0;

    if (data == NULL) {
        return;
    }

    if (sircs_input_dev == NULL) {
        return;
    }

    //MSGLOG("recv msg type=%d length=%d seq=%d\n", type, length, seq);

    if ((type == CM4COM_TYPE_REQ) && (length == 8) && (seq == 0)) {
        u32 scandata = (data[3] << 24) | (data[2] << 16) | (data[1] << 8) | data[0];
        u8 onoff = data[4];
        if ((onoff == 0) || (onoff == 1)) {
            for (i = 0; i < SIRCS_CODE_NUM; i++) {
                if (sircs_code_table[i].sircs_code == scandata) {
                    MSGLOG("key %s %s [0x%x]\n", (onoff == 1)? "down": "up  ", sircs_code_table[i].key_name, scandata);
                    input_event(sircs_input_dev, EV_KEY, sircs_code_table[i].key_code, onoff);
                    input_sync(sircs_input_dev);
                    break;
                }
            }
        }
    }
}

static int sircs_create_input(void)
{
    int i = 0;
    int result = 0;

    /* Allocate input device */
    sircs_input_dev = input_allocate_device();
    if (sircs_input_dev == NULL) {
        ERRLOG("allocate failed\n");
        return -ENOMEM;
    }

    /* Register input device */
    sircs_input_dev->name       = SIRCS_DRIVER_NAME;
    sircs_input_dev->phys       = SIRCS_DRIVER_NAME;
    sircs_input_dev->id.bustype = BUS_HOST;
    sircs_input_dev->id.vendor  = SIRCS_DRIVER_VENDOR;
    sircs_input_dev->id.product = SIRCS_DRIVER_PRODUCT;
    sircs_input_dev->id.version = SIRCS_DRIVER_VERSION;

    set_bit(EV_KEY, sircs_input_dev->evbit);
    for (i = 0; i < SIRCS_CODE_NUM; i++) {
        set_bit(sircs_code_table[i].key_code, sircs_input_dev->keybit);
#ifndef CONFIG_HA_SILENT_LOG
        MSGLOG("regist key:%d\n", sircs_code_table[i].key_code);
#else
        DBGLOG("regist key:%d\n", sircs_code_table[i].key_code);
#endif
    }

    result = input_register_device(sircs_input_dev);
    if (result) {
        input_free_device(sircs_input_dev);
        ERRLOG("Register input device failed(%d)\n", result);
        return result;
    }

    return 0;
}

static int  sircs_core_probe(struct platform_device *pdev)
{
    int result = 0;

    DBGLOG("probe\n");

    result = rpmsg_cm4_register_callback(CM4_MESSAGE_ID_SIRCS, sircs_cm4_callback);
    if (result != 0) {
        ERRLOG("regist callback failed(%d)\n", result);
        return -EPROBE_DEFER;
    }

    result = sircs_create_input();
    if (result != 0) {
        ERRLOG("create input failed(%d)\n", result);
    }

    return 0;
}

static int  sircs_core_remove(struct platform_device *pdev)
{
    DBGLOG("remove\n");

    input_unregister_device(sircs_input_dev);

    return 0;
}

static const struct of_device_id sircs_dt_ids[] = {
    { .compatible = "sony,sircs", },
    { },
};
MODULE_DEVICE_TABLE(of, sircs_dt_ids);

static struct platform_driver sircs_driver = {
    .driver = {
        .name = IR_DRV_NAME,
        .owner = THIS_MODULE,
        .of_match_table = sircs_dt_ids,
    },
    .probe = sircs_core_probe,
    .remove = sircs_core_remove,
};

static int __init sircs_core_init(void)
{
    int result = 0;

    DBGLOG("init\n");

    result = platform_driver_register(&sircs_driver);
    if (result != 0) {
        ERRLOG("regist driver failed(%d)\n", result);
        return result;
    }

    return 0;
}

static void __exit sircs_core_exit(void)
{
    DBGLOG("exit\n");

    platform_driver_unregister(&sircs_driver);
}

module_init(sircs_core_init);
module_exit(sircs_core_exit);


MODULE_AUTHOR("Sony Home Entertainment & Sound Products Inc.");
MODULE_DESCRIPTION("sircs");
MODULE_LICENSE("GPL");

