/* SPDX-License-Identifier: GPL-2.0 */
/*
 *
 * The RICOH5B Tuner driver
 *
 *
 * Copyright 2015, 2016, 2021, 2022 Sony Corporation
 *
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/fs.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <linux/list.h>
#include <linux/i2c.h>
#include <linux/irq.h>
#include <linux/jiffies.h>
#include <linux/uaccess.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/io.h>
#include <linux/platform_device.h>
#include <linux/gpio.h>
#include <linux/miscdevice.h>
#include <linux/spinlock.h>
#include <linux/cdev.h>
#include <linux/of_gpio.h>
#include <linux/time.h>
#include <linux/poll.h>

#include <linux/gpio/consumer.h>
#include <linux/regulator/consumer.h>
#include <linux/of_irq.h>

#include <media/v4l2-device.h>
#include <media/v4l2-ioctl.h>
#include <media/v4l2-ctrls.h>
#include <media/v4l2-event.h>

#include "ricoh5b_cmd.h"
#include "ricoh5b_main.h"
#include "ricoh5b_fm_cmd.h"

#ifdef CONFIG_REGMON_DEBUG
#include <misc/regmon.h>
#endif

// if debug for the flow, define bellow
// #define  RICOH5B_MAIN_DEBUG  RICOH5B_CONF_DEBUG

// if debug for register access, define bellow
// #define  RICOH5B_MAIN_REG_DEBUG  RICOH5B_CONF_DEBUG

#define LOG_ERR(fmt, args...)		pr_err("%s: " fmt, __func__, ##args)
#define LOG_INFO(fmt, args...)		pr_info("%s: " fmt, __func__, ##args)
#define LOG_DEBUG(fmt, args...)		pr_debug("%s: " fmt, __func__, ##args)

#ifdef RICOH5B_MAIN_DEBUG
#define LOG_DBG_ERR(fmt, args...)	pr_err("ricoh5b: " fmt, ##args)
#define LOG_DBG(fmt, args...)		pr_info("ricoh5b: " fmt, ##args)
#define LOG_DBG_FUNC(fmt, args...)	pr_info("%s: " fmt, __func__, ##args)
#define LOG_DBG_WARN(fmt, args...)	pr_warn("ricoh5b: " fmt, ##args)
  #if (RICOH5B_MAIN_DEBUG == RICOH5B_CONF_DEBUG_VERBOSE)
  #define LOG_DBG_V(fmt, args...)	pr_info("ricoh5b: " fmt, ##args)
  #define LOG_DEBUG_V(fmt, args...)	pr_debug("%s: " fmt, __func__, ##args)
  #else
  #define LOG_DBG_V(fmt, args...)	{}
  #define LOG_DEBUG_V(fmt, args...)	{}
  #endif 
#else  /* RICOH5B_MAIN_DEBUG */
#define LOG_DBG_ERR(fmt, args...)	{}
#define LOG_DBG(fmt, args...)		{}
#define LOG_DBG_FUNC(fmt, args...)	{}
#define LOG_DBG_WARN(fmt, args...)	{}
#define LOG_DBG_V(fmt, args...)		{}
#define LOG_DEBUG_V(fmt, args...)	{}
#endif /* RICOH5B_MAIN_DEBUG */


/* Retry wait time 6ms, chip was in standby  */
#define CHIP_RETRY_WAIT_TIME    6000  /* us */

/* Chip Power ON wait time 500ms */
#define CHIP_POWER_ON_WAIT_TIME_MS    500  /* ms */

/*
 * Block Power ON wait time
 * X'tal 32kHz -> 100ms
 * X'tal 12MHz -> 30ms
 */
#define BLOCK_POWER_ON_WAIT_TIME_MS    100  /* ms */

/* muting wait time 20ms */
#define MUTING_WAIT_TIME_MS    20    /* ms */

#define MTK_I2C_READ_WRITE_LEN(read_len, write_len) \
		(((read_len) << 8) | ((write_len) << 0))

/* repeat state check */
#define RDS_FLAG                 1
#define REC_AUDIO_MODE_FLAG      2
#define SIGNAL_STRENGTH_FLAG     3
#define RDS_RECEIVE_STAT_FLAG    4
#define RICOH5B_STATE_POLL_TIME (500 / (1000 / HZ)) // ms

#define RDS_RCV_STAT_TIME_OUT   (2000 / (1000 / HZ)) // ms

/* ricoh5b init parameter */
#define RICOH5B_MUTE_ON              1
#define RICOH5B_MUTE_OFF             0
#define RICOH5B_MODE_STEREO          1
#define RICOH5B_MODE_MONO            0
#define RICOH5B_FIRST_FREQ           80000
#define RICOH5B_FIRST_SENSE_MODE     0x00
#define RICOH5B_FIRST_SENSE_HIGH     19
#define RICOH5B_FIRST_SENSE_LOW      19
#define RICOH5B_RDS_MODE_ON          1
#define RICOH5B_RDS_MODE_OFF         0
#define RICOH5B_SIGNAL_STRENGTH_MASK 0x7F

/* ricoh5b rds use parameter */
#define RICOH5B_RDS_NOERR            0x80
#define RICOH5B_RDS_UNC              0x40
#define RICOH5B_RDS_BLOCK            0x38
#define RICOH5B_RDS_BLOCK_B          0x01
#define RICOH5B_RDS_BLOCK_C          0x03
#define ACCESSARY_CHECK_DELAY_TIME (1 / (1000 / HZ)) // 1ms
#define RICOH5B_RDS_BLOCK_B_PS       0x00
#define RICOH5B_RDS_BUF          8
#define RICOH5B_RDS_BLOCK_MASK       0xF0
#define RICOH5B_RDS_SEGMENT_MASK     0x03
#define RICOH5B_RDS_SEGMENT_ADDR_1   0x00
#define RICOH5B_RDS_SEGMENT_ADDR_2   0x01
#define RICOH5B_RDS_SEGMENT_ADDR_3   0x02
#define RICOH5B_RDS_SEGMENT_ADDR_4   0x03
#define RICOH5B_RDS_LIMIT_MIN        0x20
#define RICOH5B_RDS_LIMIT_MAX        0xFE
#define RICOH5B_LAST_DATA            7
#define FIRST_POSITION               0
#define SECOND_POSITION              2
#define THIRD_POSITION               4
#define FOURTH_POSITION              6
#define RDS_WRITE_ENABLE             1
#define RDS_WRITE_DISABLE            0
#define RDS_BLOCK_SHIFT              3
#define RICOH5B_RDS_RETRY            5

/* de-emphasis mode */
#define DEEMPHASIS_LOW               0x00
#define DEEMPHASIS_DEMP_HIGH         0x40
#define DEEMPHASIS_DEMP_CE_HIGH      0x01

#define RICOH5B_RSSI_THRESHOLD       19      /* dBu */
#define RICOH5B_I2C_RETRY_COUNT      5
#define SIGNAL_STRENGTH_RETRY        6

/* poll status */
#define POLL_CHECK_FLAG_ENABLE       1
#define POLL_CHECK_FLAG_DISABLE      0

/* Power On Sequence Step */
enum P_ON_STEP {
	P_ON_STEP_REG_RESET,
	P_ON_STEP_MODULE_ON,
	P_ON_STEP_XTAL_SETTING,
	P_ON_STEP_SYNTHE_SETTING,
	P_ON_STEP_AGC_SETTING,
	P_ON_STEP_DSP_INIT_SETTING,
	P_ON_STEP_CHIP_ON,
	P_ON_STEP_WAIT_CHIP_ON,
	P_ON_STEP_BLOCK_ON,
	P_ON_STEP_WAIT_BLOCK_ON,
	P_ON_STEP_CK_GEN_SETTING,
	P_ON_STEP_RDSI_SETTING,
	P_ON_STEP_AUDIO_SETTING,
	P_ON_STEP_RTRIM_SETTING,
	P_ON_STEP_END
};

/* Mute On Sequence Step */
enum MUTE_ON_STEP {
	MUTE_ON_STEP_SET_WAIT_COUNT,
	MUTE_ON_STEP_WAIT_WAIT_COUNT,
	MUTE_ON_STEP_MUTE_ON,
	MUTE_ON_STEP_WAIT_MUTE_ON,
	MUTE_ON_STEP_END
};

/* Mute Off Sequence Step */
enum MUTE_OFF_STEP {
	MUTE_OFF_STEP_MUTE_OFF,
	MUTE_OFF_STEP_WAIT_MUTE_OFF,
	MUTE_OFF_STEP_SET_WAIT_COUNT,
	MUTE_OFF_STEP_END
};

/* adjust AGCDET offset Sequence */
enum AGCDET_OFF_STEP {
	AGCDET_STEP_SET_AGCDET_OFFSET,
	AGCDET_STEP_GET_AD6,
	AGCDET_STEP_CALC_BEST_DET,
	AGCDET_STEP_JUDGE_DET,
	AGCDET_STEP_END
};

/* align AGCG Sequence */
enum AGCG_OFF_STEP {
	AGCG_STEP_SET_PGM,
	AGCG_STEP_WAIT_PGM,
	AGCG_STEP_CALC_BEST_IML,
	AGCG_STEP_IML_ADJUST,
	AGCG_STEP_GET_IMLI,
	AGCG_STEP_JUDGE_IML,
	AGCG_STEP_SET_AVECOUNT_128,
	AGCG_STEP_IQ_CORRECT,
	AGCG_STEP_GET_PGM,
	AGCG_STEP_GET_PG,
	AGCG_STEP_PG_3_RETRIEVE,
	AGCG_STEP_PG_1_RETRIEVE,
	AGCG_STEP_PG_2_RETRIEVE,
	AGCG_STEP_PG_2_IQ_CORRECT,
	AGCG_STEP_PG_2_GET_PG,
	AGCG_STEP_PG_2_JUDGE_PG,
	AGCG_STEP_SET_AVECOUNT_16,
	AGCG_STEP_SET_END
};

/* CTUNE Sequence */
enum CTUNE_STEP {
	CTUNE_STEP_START_CTUNE,
	CTUNE_STEP_GET_CTUNE,
	CTUNE_STEP_END
};

/* lock status */
enum LOCK_STS {
	LOCK_STS_IDLE,
	LOCK_STS_ADD,
	LOCK_STS_DEL
};

struct ricoh5b_dev {
	wait_queue_head_t read_wq;
	struct mutex read_mutex;
	struct mutex station_mutex;
	struct mutex rds_mode_mutex;
	struct mutex repeat_check;
	struct i2c_client *client;
	struct video_device *video_device;
	struct v4l2_device v4l2_dev;
	struct workqueue_struct *workqueue;
	struct delayed_work ricoh5b_rds_work;
	struct mutex rds_rcv_stat_mutex;
	struct workqueue_struct *rds_rcv_stat_queue;
	struct delayed_work rds_rcv_stat_work;
	enum LOCK_STS rds_rcv_stat_lock_sts;
	int rds_rcv_stat;
	int audio_mode;
	int mute_state;
	int tuned_freq;
	int sense_mode;
	int threshold_val;
	int rds_mode;
	int deemp;
	int deemp_ce;
	int station_check;
	int rec_audio_mode;
	int rds_receive_stat;
	unsigned char sense_high;
	unsigned char sense_low;
	struct gpio_desc *ce_gpio;
	struct gpio_desc *iic_gpio;
	struct regulator *vdd_regulator;
	bool vdd_en;
	bool iic_irq_en;
	bool event_disable;
};

static struct workqueue_struct *workqueue_check;
static struct delayed_work ricoh5b_check_work;
static char tuner_open;
static char res_signal_strength;
static char change_signal_strength;
static char change_audmode;
static char change_rds_data;
static char change_rds_receive_stat;
static int  gain_value  = RICOH5B_VOLUME_0DB;
static int  rbds_enable = RICOH5B_REG_RDS_ON_VALUE;

static int DeEmphasis = DEEMPHASIS_50US;
module_param_named(deemphasis, DeEmphasis, int, 0000);
MODULE_PARM_DESC(deemphasis, "deemphasis");

static int GainValue = GAIN_0DB;
module_param_named(gainvalue, GainValue, int, 0000);
MODULE_PARM_DESC(gainvalue, "gainvalue");

static int RdsMode = RDS_MODE_RDS;
module_param_named(rdsmode, RdsMode, int, 0000);
MODULE_PARM_DESC(rdsmode, "rdsmode");

static struct ricoh5b_dev *ricoh5b_device_info;
static struct i2c_client *ricoh5b_i2c_client;

/* operation_routines */
static ssize_t ricoh5b_read(
	struct file *file,
	char __user *buf,
	size_t len,
	loff_t *loff
);

static int ricoh5b_open(
	struct file *file
);
static int ricoh5b_close(
	struct file *file
);
static unsigned int ricoh5b_poll(
	struct file *file,
	struct poll_table_struct *s_wait
);
static int ricoh5b_querycap(
	struct file            *file,
	void                   *private,
	struct v4l2_capability *capability
);
static int ricoh5b_s_tuner(
	struct file             *file,
	void                    *private,
	const struct v4l2_tuner *tuner
);
static int ricoh5b_g_tuner(
	struct file       *file,
	void              *private,
	struct v4l2_tuner *tuner
);
static int ricoh5b_s_frequency(
	struct file                 *file,
	void                        *private,
	const struct v4l2_frequency *frequency
);
static int ricoh5b_g_frequency(
	struct file           *file,
	void                  *private,
	struct v4l2_frequency *frequency
);
static int ricoh5b_s_ctrl(
	struct file         *file,
	void                *private,
	struct v4l2_control *control
);
static int ricoh5b_g_ctrl(
	struct file         *file,
	void                *private,
	struct v4l2_control *control
);

static int ricoh5b_sense_mode_set(
	struct ricoh5b_dev *ricoh5b_dev,
	int value
);

static int ricoh5b_threshold_set(
	struct ricoh5b_dev *ricoh5b_dev,
	int value
);

static int ricoh5b_station_check(struct ricoh5b_dev *ricoh5b_dev);
static int ricoh5b_rssi_check(int sens, int *value);
static int ricoh5b_fo_sat_check(void);
static int ricoh5b_rec_audio_mode(void);
static void ricoh5b_next_rds_eint(struct ricoh5b_dev *ricoh5b_dev);
static int ricoh5b_read_reg(int address, int *value);

static void ricoh5b_enable_regulator(struct ricoh5b_dev *ricoh5b_dev);
static void ricoh5b_disable_regulator(struct ricoh5b_dev *ricoh5b_dev);

static int ricoh5b_init_rds_rcv_stat(void);
static void ricoh5b_destroy_rds_rcv_stat(void);
static void ricoh5b_cancel_rds_rcv_stat(void);
static void ricoh5b_set_rds_rcv_stat(void);
static void ricoh5b_timeout_rds_rcv_stat(struct work_struct *work);

static const struct v4l2_file_operations ricoh5b_file_operations = {
	.owner           = THIS_MODULE,
	.open            = ricoh5b_open,
	.release         = ricoh5b_close,
	.read            = ricoh5b_read,
	.unlocked_ioctl  = video_ioctl2,
	.poll            = ricoh5b_poll,
};

static const struct v4l2_ioctl_ops ricoh5b_ioctl_operations = {
	.vidioc_querycap       = ricoh5b_querycap,
	.vidioc_s_tuner        = ricoh5b_s_tuner,
	.vidioc_g_tuner        = ricoh5b_g_tuner,
	.vidioc_s_frequency    = ricoh5b_s_frequency,
	.vidioc_g_frequency    = ricoh5b_g_frequency,
	.vidioc_s_ctrl         = ricoh5b_s_ctrl,
	.vidioc_g_ctrl         = ricoh5b_g_ctrl,
};

/*---------------*/
/* command table */
/*---------------*/

/* register reset */
static const struct ricoh5b_reg ricoh5b_cmd_reg_reset_tbl[] = {
	{ RICOH5B_REG_ADR_SRST,    RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_SRST_ALL  },
	{ RICOH5B_REG_ADR_MUTE,    RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_MUTE_ON   }
};

/* module power ON */
static const struct ricoh5b_reg ricoh5b_cmd_module_power_on_tbl[] = {
	{ RICOH5B_REG_ADR_BLOCK2_PE,  RICOH5B_REG_BIT_BLOCK2_PE,  RICOH5B_REG_DAT_PELO_ON | RICOH5B_REG_DAT_PEFMMIX_ON |
	 RICOH5B_REG_DAT_PEFMLNA_ON    },
	{ RICOH5B_REG_ADR_VCO,        RICOH5B_REG_NON_MASK,       RICOH5B_REG_DAT_VCO                                    },
	{ RICOH5B_REG_ADR_IMFMEN_EN,  RICOH5B_REG_NON_MASK,       RICOH5B_REG_DAT_IMFMEN_ON | RICOH5B_REG_DAT_FMEN_ON    },
	{ RICOH5B_REG_ADR_SPOFS_SYN,  RICOH5B_REG_NON_MASK,       RICOH5B_REG_DAT_SPOFS_SYN                              }
};

/* X'tal setting */
static const struct ricoh5b_reg ricoh5b_cmd_xtal_setting_tbl[] = {
#if (RICOH5B_SELECT_X_TAL == RICOH5B_X_TAL_32KHZ)
	{ RICOH5B_REG_ADR_XTAL,        RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_XTAL_32K                                            },
	{ RICOH5B_REG_ADR_PELPFVGA,    RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_PELPFVGA_ON | RICOH5B_REG_DAT_MIX_ISEL_I_SOURCE     },
	{ RICOH5B_REG_ADR_CLK_WIN,     RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_CLK_WIN_32K                                         },
	{ RICOH5B_REG_ADR_CK_VCO,      RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_CK_VCO_32K                                          },
	{ RICOH5B_REG_ADR_VTJHA_CK,    RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_VTJHA_CK_0_96 | RICOH5B_REG_DAT_VTJLA_CK_0_24 |
	 RICOH5B_REG_DAT_VTJLM_CK_0_28 },
	{ RICOH5B_REG_ADR_CKBRD,       RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_CKBRD | RICOH5B_REG_DAT_CKBRS                       },
	{ RICOH5B_REG_ADR_PADEN_CK,    RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_PADEN_CK_ENABLE | RICOH5B_REG_DAT_AMPEN_CK_ENABLE   },
	{ RICOH5B_REG_ADR_SYN_VTJH,    RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_SYN_VTJH_0_92 | RICOH5B_REG_DAT_SYN_VTJL_0_20       },
	{ RICOH5B_REG_ADR_LOC_T2,      RICOH5B_REG_BIT_LOC_T2,  RICOH5B_REG_DAT_LOC_T2_220U                                       },
	{ RICOH5B_REG_ADR_RSW_CK,      RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_SW0_CK_32K | RICOH5B_REG_DAT_RSW_CK_5000K           },
	{ RICOH5B_REG_ADR_CIO_SYN,     RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_CIO_SYN_2_25 | RICOH5B_REG_DAT_SYN_RDF_35K          },
#else
	{ RICOH5B_REG_ADR_XTAL,        RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_XTAL_12M                                            },
	{ RICOH5B_REG_ADR_PELPFVGA,    RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_XFS_12M | RICOH5B_REG_DAT_PELPFVGA_ON |
	 RICOH5B_REG_DAT_MIX_ISEL_I_SOURCE },
	{ RICOH5B_REG_ADR_CLK_WIN,     RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_CLK_WIN_12M                                         },
	{ RICOH5B_REG_ADR_CK_VCO,      RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_CK_VCO_12M                                          },
	{ RICOH5B_REG_ADR_PADEN_CK,    RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_PADEN_CK_DISABLE | RICOH5B_REG_DAT_AMPEN_CK_DISABLE },
	{ RICOH5B_REG_ADR_LOC_T2,      RICOH5B_REG_BIT_LOC_T2,  RICOH5B_REG_DAT_LOC_T2_100U                                       },
	{ RICOH5B_REG_ADR_RSW_CK,      RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_SW0_CK_12M | RICOH5B_REG_DAT_RSW_CK_760K            },
	{ RICOH5B_REG_ADR_CIO_SYN,     RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_CIO_SYN_18_0 | RICOH5B_REG_DAT_SYN_RDF_100K         },
#endif
	{ RICOH5B_REG_ADR_RSYS,        RICOH5B_REG_BIT_RSYS,  RICOH5B_REG_DAT_RSYS_FM400K_AMAUTO                                  },
	{ RICOH5B_REG_ADR_ISEL2,       RICOH5B_REG_NON_MASK,  RICOH5B_REG_BIT_XBOOST_BOOST | RICOH5B_REG_DAT_ISEL2_NORMAL         },
	{ RICOH5B_REG_ADR_LOSTART_EN,  RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_LOSTART_EN_ENABLE | RICOH5B_REG_DAT_LOSTART         }
};

/* synthe setting */
static const struct ricoh5b_reg ricoh5b_cmd_synthe_setting_tbl[] = {
	{ RICOH5B_REG_ADR_ADCMOD,      RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_ADCMOD_EXCHANGE_IQ           },
	{ RICOH5B_REG_ADR_AD1_RSTB,    RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_AD1_RSTB_NORMAL              },
	{ RICOH5B_REG_ADR_BW,          RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_BW_DEFAULT                   }
};

/* AGC setting */
static const struct ricoh5b_reg ricoh5b_cmd_agc_setting_tbl[] = {
	{ RICOH5B_REG_ADR_FMIN,          RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_FMIN_24DB | RICOH5B_REG_DAT_PGM_18DB       },
	{ RICOH5B_REG_ADR_AGC_AUTO,      RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_AGC_AUTO_OFF                               },
	{ RICOH5B_REG_ADR_TABLE_MODE,    RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_TABLE_MODE                                 },
	{ RICOH5B_REG_ADR_BLOCK1_PE,     RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_BLOCK1_AGC_SETTING                         },
#if RICOH5B_USE_TUNER
	{ RICOH5B_REG_ADR_AM_VIFO_UPPER,   RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_AM_VIFO_UPPER                            },
	{ RICOH5B_REG_ADR_AM_VIFO_LOWER,   RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_AM_VIFO_LOWER                            },
	{ RICOH5B_REG_ADR_AM_VUD_UPPER,    RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_AM_VUD_UPPER                             },
	{ RICOH5B_REG_ADR_AM_VUD_LOWER,    RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_AM_VUD_LOWER                             },
	{ RICOH5B_REG_ADR_AM_PIF_SCALE,    RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_AM_PIF_SCALE                             },
	{ RICOH5B_REG_ADR_AM_THR_IFR,      RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_AM_THR_IFR | RICOH5B_REG_DAT_AM_DEC_IFR  },
	{ RICOH5B_REG_ADR_WIN_CNT_VAL,     RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_WIN_CNT_VAL                              },
	{ RICOH5B_REG_ADR_AM_VIFO_INIT,    RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_AM_VIFO_INIT                             },
	{ RICOH5B_REG_ADR_AM_VUD_INIT,     RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_AM_VUD_INIT                              },
	{ RICOH5B_REG_ADR_AM_PIF_INIT,     RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_AM_PIF_INIT                              }
#endif
};

/* DSP initial setting */
static const struct ricoh5b_reg ricoh5b_cmd_dsp_init_setting_tbl[] = {
#if RICOH5B_USE_TUNER
	{ RICOH5B_REG_ADR_AM_DAGC,        RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_AM_DAGC                            },
	{ RICOH5B_REG_ADR_AM_TH_DET,      RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_AM_TH_DET                          },
#endif
	{ RICOH5B_REG_ADR_FM_RSSI_CONST,  RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_FM_RSSI_CONST                      },
	{ RICOH5B_REG_ADR_SMT,            RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_SMT_OFF                            },
	{ RICOH5B_REG_ADR_HCC,            RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_HCC_OFF | RICOH5B_REG_DAT_SNC_OFF  },
	{ RICOH5B_REG_ADR_VBFSEL,         RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_VBFSEL_DEFAULT                     },
	{ RICOH5B_REG_ADR_DET_DC_ADJ,     RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_DET_DC_ADJ_PLUS_0                  },
	{ RICOH5B_REG_ADR_ITRIM,          RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_ITRIM_6MA                          },
	{ RICOH5B_REG_ADR_COM_SEL,        RICOH5B_REG_BIT_COM_SEL,   RICOH5B_REG_DAT_COM_SEL_300M                       },
	{ RICOH5B_REG_ADR_AD6_I_SW,       RICOH5B_REG_BIT_AD6_I_SW,  RICOH5B_REG_DAT_AD6_I_SW                           }
};

/* Clock Generator setting */
static const struct ricoh5b_reg ricoh5b_cmd_clk_setting_tbl[] = {
#if (RICOH5B_SELECT_X_TAL == RICOH5B_X_TAL_32KHZ)
	{ RICOH5B_REG_ADR_PADEN_CK,    RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_PADEN_CK_ENABLE
	 | RICOH5B_REG_DAT_AMPEN_CK_ENABLE | RICOH5B_REG_DAT_CAPOUT_CK_ENABLE | RICOH5B_REG_DAT_CAPADJ_CK_0_25          },
	{ RICOH5B_REG_ADR_RSW_CK,      RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_SW0_CK_32K | RICOH5B_REG_DAT_RSW_CK_400K  },
#endif
	{ RICOH5B_REG_ADR_UL_MASK,     RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_UL_MASK_OFF                               }
};

/* RDSI output setting */
static const struct ricoh5b_reg ricoh5b_cmd_rdsi_output_tbl[] = {
	{ RICOH5B_REG_ADR_RDS_OUT_EN_H,  RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_RDS_OUT_EN_H  },
	{ RICOH5B_REG_ADR_RDS_OUT_EN_L,  RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_RDS_OUT_EN_L  }
};

/* DSP alignment start */
static const struct ricoh5b_reg ricoh5b_cmd_dsp_align_start_tbl[] = {
	{ RICOH5B_REG_ADR_BW,          RICOH5B_REG_BIT_BW,           RICOH5B_REG_DAT_BW_FM_75K                                    },
#if RICOH5B_USE_TUNER
	{ RICOH5B_REG_ADR_STATION_EN,  RICOH5B_REG_BIT_STATION_EN,   RICOH5B_REG_DAT_STATION_EN_ENABLE                            },
#endif //RICOH5B_USE_TUNER
	{ RICOH5B_REG_ADR_FO_SAT_SEL,  RICOH5B_REG_NON_MASK,         RICOH5B_REG_DAT_FO_SAT_10K | RICOH5B_REG_DAT_DCO_AVESEL_IDLE },
	{ RICOH5B_REG_ADR_FC_SW,       RICOH5B_REG_BIT_FC_SW,        RICOH5B_REG_DAT_FC_SW_500K                                   },
	{ RICOH5B_REG_ADR_IMFMSW,      RICOH5B_REG_BIT_IMFMSW | RICOH5B_REG_BIT_IMFMA,    RICOH5B_REG_DAT_IMFMSW_ENABLE |
	 RICOH5B_REG_DAT_IMFMA_MIN },
	{ RICOH5B_REG_ADR_AVECOUNT,    RICOH5B_REG_BIT_AVECOUNT,     RICOH5B_REG_DAT_AVECOUNT_16                                  },
	{ RICOH5B_REG_ADR_FMIN,        RICOH5B_REG_NON_MASK,         RICOH5B_REG_DAT_FMIN_MINUS_36DB | RICOH5B_REG_DAT_PGM_0DB    },
	{ RICOH5B_REG_ADR_XDOWN,       RICOH5B_REG_NON_MASK,         RICOH5B_REG_DAT_XDOWN_MUCH | RICOH5B_REG_DAT_DET_AFM_FM      }
};

/* DSP alignment end */
static const struct ricoh5b_reg ricoh5b_cmd_dsp_align_end_tbl[] = {
	{ RICOH5B_REG_ADR_IMFMSW,      RICOH5B_REG_BIT_IMFMSW | RICOH5B_REG_BIT_IMFMA,    RICOH5B_REG_DAT_IMFMSW_DISABLE |
	 RICOH5B_REG_DAT_IMFMA_MIN },
	{ RICOH5B_REG_ADR_BW,          RICOH5B_REG_BIT_BW,                  RICOH5B_REG_DAT_BW_FM_130K                    },
	{ RICOH5B_REG_ADR_FO_SAT_SEL,  RICOH5B_REG_NON_MASK,                RICOH5B_REG_DAT_FO_SAT_25K                    },
	{ RICOH5B_REG_ADR_FC_SW,       RICOH5B_REG_BIT_FC_SW,               RICOH5B_REG_DAT_FC_SW_1M                      },
	{ RICOH5B_REG_ADR_IMF,         RICOH5B_REG_NON_MASK,                RICOH5B_REG_DAT_IMF_OFF                       },
	{ RICOH5B_REG_ADR_PGM,         RICOH5B_REG_BIT_PGM,                 RICOH5B_REG_DAT_PGM_0DB                       }
};

/* reset RFAGC */
static const struct ricoh5b_reg ricoh5b_cmd_reset_rfagc_tbl[] = {
	{ RICOH5B_REG_ADR_RST_RFAGC,  RICOH5B_REG_BIT_RST_RFAGC,  RICOH5B_REG_DAT_RST_RFAGC_ENABLE   },
	{ RICOH5B_REG_ADR_RST_RFAGC,  RICOH5B_REG_BIT_RST_RFAGC,  RICOH5B_REG_DAT_RST_RFAGC_DISABLE  }
};

/* Image Signal Output Level OFF */
static const struct ricoh5b_reg ricoh5b_cmd_im_output_off_tbl[] = {
	{ RICOH5B_REG_ADR_IMFMA,  RICOH5B_REG_BIT_IMFMA,  RICOH5B_REG_DAT_IMFMA_MIN  }
};

/* Analog audio output */
static const struct ricoh5b_reg ricoh5b_cmd_audio_output_analog_tbl[] = {
	{ RICOH5B_REG_ADR_IISD_SETTING,    RICOH5B_REG_NON_MASK,           RICOH5B_REG_DAT_IISD_SETTING_ANALOG    },
	{ RICOH5B_REG_ADR_AUDIO_SETTING,   RICOH5B_REG_BIT_AUDIO_SETTING,  RICOH5B_REG_DAT_AUDIO_SETTING_ANALOG   },
	{ RICOH5B_REG_ADR_IIS_EN,          RICOH5B_REG_BIT_IIS_EN,         RICOH5B_REG_DAT_IIS_EN_DAC_OUT         },
};

/* IIS 32fs audio output */
static const struct ricoh5b_reg ricoh5b_cmd_audio_output_iis_32fs_tbl[] = {
	{ RICOH5B_REG_ADR_IISD_SETTING,    RICOH5B_REG_NON_MASK,            RICOH5B_REG_DAT_IISD_SETTING_DIGITAL                 },
	{ RICOH5B_REG_ADR_AUDIO_SETTING,   RICOH5B_REG_BIT_AUDIO_SETTING,   RICOH5B_REG_DAT_AUDIO_SETTING_DIGITAL                },
	{ RICOH5B_REG_ADR_IIS_EN,          RICOH5B_REG_BIT_IIS_SEL  |  RICOH5B_REG_BIT_IIS_32_64  |  RICOH5B_REG_BIT_IIS_EN,
	 RICOH5B_REG_DAT_IIS_SEL_IIS_FORMAT  | RICOH5B_REG_DAT_IIS_32FS  |  RICOH5B_REG_DAT_IIS_EN_IIS_OUT }
};

/* IIS 64fs audio output */
static const struct ricoh5b_reg ricoh5b_cmd_audio_output_iis_64fs_tbl[] = {
	{ RICOH5B_REG_ADR_IISD_SETTING,    RICOH5B_REG_NON_MASK,           RICOH5B_REG_DAT_IISD_SETTING_DIGITAL                 },
	{ RICOH5B_REG_ADR_AUDIO_SETTING,   RICOH5B_REG_BIT_AUDIO_SETTING,  RICOH5B_REG_DAT_AUDIO_SETTING_DIGITAL                },
	{ RICOH5B_REG_ADR_IIS_EN,          RICOH5B_REG_BIT_IIS_SEL  |  RICOH5B_REG_BIT_IIS_32_64  |  RICOH5B_REG_BIT_IIS_EN,
	 RICOH5B_REG_DAT_IIS_SEL_IIS_FORMAT  | RICOH5B_REG_DAT_IIS_64FS  |  RICOH5B_REG_DAT_IIS_EN_IIS_OUT }
};

/* chip power off */
static const struct ricoh5b_reg ricoh5b_cmd_chip_power_off_tbl[] = {
	{ RICOH5B_REG_ADR_BLOCK1_PE,  RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_BLOCK1_POWER_OFF  },
	{ RICOH5B_REG_ADR_PE,         RICOH5B_REG_BIT_PE,    RICOH5B_REG_DAT_PE_OFF            }
};

#define RICOH5B_CMD_REG_RESET_TBL_SIZE              (ARRAY_SIZE(ricoh5b_cmd_reg_reset_tbl))
#define RICOH5B_CMD_MODULE_POWER_ON_TBL_SIZE        (ARRAY_SIZE(ricoh5b_cmd_module_power_on_tbl))
#define RICOH5B_CMD_XTAL_SETTING_TBL_SIZE           (ARRAY_SIZE(ricoh5b_cmd_xtal_setting_tbl))
#define RICOH5B_CMD_SYNTHE_SETTING_TBL_SIZE         (ARRAY_SIZE(ricoh5b_cmd_synthe_setting_tbl))
#define RICOH5B_CMD_AGC_SETTING_TBL_SIZE            (ARRAY_SIZE(ricoh5b_cmd_agc_setting_tbl))
#define RICOH5B_CMD_DSP_INIT_SETTING_TBL_SIZE       (ARRAY_SIZE(ricoh5b_cmd_dsp_init_setting_tbl))
#define RICOH5B_CMD_CLK_SETTING_TBL_SIZE            (ARRAY_SIZE(ricoh5b_cmd_clk_setting_tbl))
#define RICOH5B_CMD_RDSI_OUTPUT_TBL_SIZE            (ARRAY_SIZE(ricoh5b_cmd_rdsi_output_tbl))
#define RICOH5B_CMD_DSP_ALIGN_START_TBL_SIZE        (ARRAY_SIZE(ricoh5b_cmd_dsp_align_start_tbl))
#define RICOH5B_CMD_DSP_ALIGN_END_TBL_SIZE          (ARRAY_SIZE(ricoh5b_cmd_dsp_align_end_tbl))
#define RICOH5B_CMD_RESET_RFAGC_TBL_SIZE            (ARRAY_SIZE(ricoh5b_cmd_reset_rfagc_tbl))
#define RICOH5B_CMD_IM_OUTPUT_OFF_TBL_SIZE          (ARRAY_SIZE(ricoh5b_cmd_im_output_off_tbl))
#define RICOH5B_CMD_AUDIO_OUTPUT_ANALOG_TBL_SIZE    (ARRAY_SIZE(ricoh5b_cmd_audio_output_analog_tbl))
#define RICOH5B_CMD_AUDIO_OUTPUT_IIS_32FS_TBL_SIZE  (ARRAY_SIZE(ricoh5b_cmd_audio_output_iis_32fs_tbl))
#define RICOH5B_CMD_AUDIO_OUTPUT_IIS_64FS_TBL_SIZE  (ARRAY_SIZE(ricoh5b_cmd_audio_output_iis_64fs_tbl))
#define RICOH5B_CMD_CHIP_POWER_OFF_TBL_SIZE         (ARRAY_SIZE(ricoh5b_cmd_chip_power_off_tbl))

/*-------*/
/* other */
/*-------*/

/* align AGCG table */
static unsigned char  align_agcg_tbl[] = {
	RICOH5B_REG_DAT_PGM_0DB,
	RICOH5B_REG_DAT_PGM_6DB,
	RICOH5B_REG_DAT_PGM_12DB,
	RICOH5B_REG_DAT_PGM_18DB,
	RICOH5B_REG_DAT_PGM_24DB
};
#define ALIGN_AGCG_TBL_SIZE      (ARRAY_SIZE(align_agcg_tbl))

/******************************************************************************/
/* global variables definition which are refered only this file               */
/******************************************************************************/
static struct ricoh5b_pg *p_fine_pg;
static unsigned char      p_fine_pg_step;
static unsigned char      det_offset;

/********************/
/*@ regmon_routines */
/********************/

#ifdef CONFIG_REGMON_DEBUG

static int ricoh5b_regmon_write_reg(
	void          *private_data,
	unsigned int   address,
	unsigned int   value
);

static int ricoh5b_regmon_read_reg(
	void          *private_data,
	unsigned int   address,
	unsigned int  *value
);

static regmon_reg_info_t ricoh5b_reg_info[] = {
	{ "0x00", 0x00 },
	{ "0x01", 0x01 },
	{ "0x02", 0x02 },
	{ "0x03", 0x03 },
	{ "0x04", 0x04 },
	{ "0x05", 0x05 },
	{ "0x06", 0x06 },
	{ "0x07", 0x07 },
	{ "0x08", 0x08 },
	{ "0x09", 0x09 },
	{ "0x0A", 0x0A },
	{ "0x0B", 0x0B },
	{ "0x0C", 0x0C },
	{ "0x0D", 0x0D },
	{ "0x0E", 0x0E },
	{ "0x0F", 0x0F },
	{ "0x10", 0x10 },
	{ "0x11", 0x11 },
	{ "0x12", 0x12 },
	{ "0x13", 0x13 },
	{ "0x14", 0x14 },
	{ "0x15", 0x15 },
	{ "0x16", 0x16 },
	{ "0x17", 0x17 },
	{ "0x18", 0x18 },
	{ "0x19", 0x19 },
	{ "0x1A", 0x1A },
	{ "0x1B", 0x1B },
	{ "0x1C", 0x1C },
	{ "0x1D", 0x1D },
	{ "0x1E", 0x1E },
	{ "0x1F", 0x1F },
	{ "0x20", 0x20 },
	{ "0x21", 0x21 },
	{ "0x22", 0x22 },
	{ "0x23", 0x23 },
	{ "0x24", 0x24 },
	{ "0x25", 0x25 },
	{ "0x26", 0x26 },
	{ "0x27", 0x27 },
	{ "0x28", 0x28 },
	{ "0x29", 0x29 },
	{ "0x2A", 0x2A },
	{ "0x2B", 0x2B },
	{ "0x2C", 0x2C },
	{ "0x2D", 0x2D },
	{ "0x2E", 0x2E },
	{ "0x2F", 0x2F },
	{ "0x30", 0x30 },
	{ "0x31", 0x31 },
	{ "0x32", 0x32 },
	{ "0x33", 0x33 },
	{ "0x34", 0x34 },
	{ "0x35", 0x35 },
	{ "0x36", 0x36 },
	{ "0x37", 0x37 },
	{ "0x38", 0x38 },
	{ "0x39", 0x39 },
	{ "0x3A", 0x3A },
	{ "0x3B", 0x3B },
	{ "0x3C", 0x3C },
	{ "0x3D", 0x3D },
	{ "0x3E", 0x3E },
	{ "0x3F", 0x3F },
	{ "0x40", 0x40 },
	{ "0x41", 0x41 },
	{ "0x42", 0x42 },
	{ "0x43", 0x43 },
	{ "0x44", 0x44 },
	{ "0x45", 0x45 },
	{ "0x46", 0x46 },
	{ "0x47", 0x47 },
	{ "0x48", 0x48 },
	{ "0x49", 0x49 },
	{ "0x4A", 0x4A },
	{ "0x4B", 0x4B },
	{ "0x4C", 0x4C },
	{ "0x4D", 0x4D },
	{ "0x4E", 0x4E },
	{ "0x4F", 0x4F },
	{ "0x50", 0x50 },
	{ "0x51", 0x51 },
	{ "0x52", 0x52 },
	{ "0x53", 0x53 },
	{ "0x54", 0x54 },
	{ "0x55", 0x55 },
	{ "0x56", 0x56 },
	{ "0x57", 0x57 },
	{ "0x58", 0x58 },
	{ "0x59", 0x59 },
	{ "0x5A", 0x5A },
	{ "0x5B", 0x5B },
	{ "0x5C", 0x5C },
	{ "0x5D", 0x5D },
	{ "0x5E", 0x5E },
	{ "0x5F", 0x5F },
	{ "0x60", 0x60 },
	{ "0x61", 0x61 },
	{ "0x62", 0x62 },
	{ "0x63", 0x63 },
	{ "0x64", 0x64 },
	{ "0x65", 0x65 },
	{ "0x66", 0x66 },
	{ "0x67", 0x67 },
	{ "0x68", 0x68 },
	{ "0x69", 0x69 },
	{ "0x6A", 0x6A },
	{ "0x6B", 0x6B },
	{ "0x6C", 0x6C },
	{ "0x6D", 0x6D },
	{ "0x6E", 0x6E },
	{ "0x6F", 0x6F },
	{ "0x70", 0x70 },
	{ "0x71", 0x71 },
	{ "0x72", 0x72 },
	{ "0x73", 0x73 },
	{ "0x74", 0x74 },
	{ "0x75", 0x75 },
	{ "0x76", 0x76 },
	{ "0x77", 0x77 },
	{ "0x78", 0x78 },
	{ "0x79", 0x79 },
	{ "0x7A", 0x7A },
	{ "0x7B", 0x7B },
	{ "0x7C", 0x7C },
	{ "0x7D", 0x7D },
	{ "0x7E", 0x7E },
	{ "0x7F", 0x7F },
	{ "0x80", 0x80 },
	{ "0x81", 0x81 },
	{ "0x82", 0x82 },
	{ "0x83", 0x83 },
	{ "0x84", 0x84 },
	{ "0x85", 0x85 },
	{ "0x86", 0x86 },
	{ "0x87", 0x87 },
	{ "0x88", 0x88 },
	{ "0x89", 0x89 },
	{ "0x8A", 0x8A },
	{ "0x8B", 0x8B },
	{ "0x8C", 0x8C },
	{ "0x8D", 0x8D },
	{ "0x8E", 0x8E },
	{ "0x8F", 0x8F },
	{ "0x90", 0x90 },
	{ "0x91", 0x91 },
	{ "0x92", 0x92 },
	{ "0x93", 0x93 },
	{ "0x94", 0x94 },
	{ "0x95", 0x95 },
	{ "0x96", 0x96 },
	{ "0x97", 0x97 },
	{ "0x98", 0x98 },
	{ "0x99", 0x99 },
	{ "0x9A", 0x9A },
	{ "0x9B", 0x9B },
	{ "0x9C", 0x9C },
	{ "0x9D", 0x9D },
	{ "0x9E", 0x9E },
	{ "0x9F", 0x9F },
	{ "0xA0", 0xA0 },
	{ "0xA1", 0xA1 },
	{ "0xA2", 0xA2 },
	{ "0xA3", 0xA3 },
	{ "0xA4", 0xA4 },
	{ "0xA5", 0xA5 },
	{ "0xA6", 0xA6 },
	{ "0xA7", 0xA7 },
	{ "0xA8", 0xA8 },
	{ "0xA9", 0xA9 },
	{ "0xAA", 0xAA },
	{ "0xAB", 0xAB },
	{ "0xAC", 0xAC },
	{ "0xAD", 0xAD },
	{ "0xAE", 0xAE },
	{ "0xAF", 0xAF },
	{ "0xB0", 0xB0 },
	{ "0xB1", 0xB1 },
	{ "0xB2", 0xB2 },
	{ "0xB3", 0xB3 },
	{ "0xB4", 0xB4 },
	{ "0xB5", 0xB5 },
	{ "0xB6", 0xB6 },
	{ "0xB7", 0xB7 },
	{ "0xB8", 0xB8 },
	{ "0xB9", 0xB9 },
	{ "0xBA", 0xBA },
	{ "0xBB", 0xBB },
	{ "0xBC", 0xBC },
	{ "0xBD", 0xBD },
	{ "0xBE", 0xBE },
	{ "0xBF", 0xBF },
	{ "0xC0", 0xC0 },
	{ "0xC1", 0xC1 },
	{ "0xC2", 0xC2 },
	{ "0xC3", 0xC3 },
	{ "0xC4", 0xC4 },
	{ "0xC5", 0xC5 },
	{ "0xC6", 0xC6 },
	{ "0xC7", 0xC7 },
	{ "0xC8", 0xC8 },
	{ "0xC9", 0xC9 },
	{ "0xCA", 0xCA },
	{ "0xCB", 0xCB },
	{ "0xCC", 0xCC },
	{ "0xCD", 0xCD },
	{ "0xCE", 0xCE },
	{ "0xCF", 0xCF },
	{ "0xD0", 0xD0 },
	{ "0xD1", 0xD1 },
	{ "0xD2", 0xD2 },
	{ "0xD3", 0xD3 },
	{ "0xD4", 0xD4 },
	{ "0xD5", 0xD5 },
	{ "0xD6", 0xD6 },
	{ "0xD7", 0xD7 },
	{ "0xD8", 0xD8 },
	{ "0xD9", 0xD9 },
	{ "0xDA", 0xDA },
	{ "0xDB", 0xDB },
	{ "0xDC", 0xDC },
	{ "0xDD", 0xDD },
	{ "0xDE", 0xDE },
	{ "0xDF", 0xDF },
	{ "0xE0", 0xE0 },
	{ "0xE1", 0xE1 },
	{ "0xE2", 0xE2 },
	{ "0xE3", 0xE3 },
	{ "0xE4", 0xE4 },
	{ "0xE5", 0xE5 },
	{ "0xE6", 0xE6 },
	{ "0xE7", 0xE7 },
	{ "0xE8", 0xE8 },
	{ "0xE9", 0xE9 },
	{ "0xEA", 0xEA },
	{ "0xEB", 0xEB },
	{ "0xEC", 0xEC },
	{ "0xED", 0xED },
	{ "0xEE", 0xEE },
	{ "0xEF", 0xEF },
	{ "0xF0", 0xF0 },
	{ "0xF1", 0xF1 },
	{ "0xF2", 0xF2 },
	{ "0xF3", 0xF3 },
	{ "0xF4", 0xF4 },
	{ "0xF5", 0xF5 },
	{ "0xF6", 0xF6 },
	{ "0xF7", 0xF7 },
	{ "0xF8", 0xF8 },
	{ "0xF9", 0xF9 },
	{ "0xFA", 0xFA },
	{ "0xFB", 0xFB },
	{ "0xFC", 0xFC },
	{ "0xFD", 0xFD },
	{ "0xFE", 0xFE },
	{ "0xFF", 0xFF },
};

static regmon_customer_info_t ricoh5b_customer_info = {
	.name           = "ricoh5b_tuner",
	.reg_info       = ricoh5b_reg_info,
	.reg_info_count = sizeof(ricoh5b_reg_info) / sizeof(regmon_reg_info_t),
	.write_reg      = ricoh5b_regmon_write_reg,
	.read_reg       = ricoh5b_regmon_read_reg,
	.private_data   = NULL,
};

static int ricoh5b_regmon_add(void *data)
{
	ricoh5b_customer_info.private_data = data;
	regmon_add(&ricoh5b_customer_info);
	return 0;
}

static int ricoh5b_regmon_del(void)
{
	regmon_del(&ricoh5b_customer_info);
	return 0;
}

static int ricoh5b_regmon_write_reg(void *private_data,
	unsigned int address, unsigned int value)
{
	int ret = 0;
	unsigned char buf_w[WRITE_BUF_SIZE] = {0, };
	unsigned char add_w = 0;

	add_w = (unsigned char)value;
	memcpy(buf_w, &add_w, 1);

	ret = ricoh5b_cmd_write_byte((unsigned char)address, buf_w, 2);

	return ret;

}

static int ricoh5b_regmon_read_reg(void *private_data, unsigned int address,
	unsigned int *value)
{
	int ret = 0;
	unsigned char buf_r[READ_BUF_SIZE] = {0, };

	ret = ricoh5b_cmd_read_byte((unsigned char)address, buf_r);

	*value = buf_r[0];

	return ret;
}

#endif

/******************************************************************************/
/*!
 * @brief get AGCDET offset
 */
/******************************************************************************/
unsigned char ricoh5b_common_cmd_get_det_offset(void)
{
	return det_offset;
}

/******************************************************************************/
/*!
 * @brief wait timer API (use only under 20000us)
 */
/******************************************************************************/
void ricoh5b_cmd_set_sequence_timer(unsigned long timeout_us)
{
	usleep_range(timeout_us, timeout_us * 2);
}

/******************************************************************************/
/*!
 * @brief immunity measure Enable/Disable status get
 */
/******************************************************************************/
enum RICOH5B_IMMUNITY ricoh5b_cmd_get_immunity_measure(void)
{
	#if 0	/* At 2014/09/12, not use below */
		if (p_get_immunity_measure
			!= (enum RICOH5B_IMMUNITY (*)(void))RICOH5B_NULL)
			return p_get_immunity_measure();
		else
			return RICOH5B_IMMUNITY_UNKNOWN;
	#else
		return RICOH5B_IMMUNITY_UNKNOWN;
	#endif
}

/******************************************************************************/
/*!
 * @brief Image Rejection Phase/Gain correction address
 */
/******************************************************************************/
static unsigned char get_pg_correction_address(unsigned char pgm)
{
	if (pgm <= RICOH5B_REG_DAT_PGM_4DB)
		return RICOH5B_REG_ADR_RGCOR_0_L;
	else if (pgm <= RICOH5B_REG_DAT_PGM_8DB)
		return RICOH5B_REG_ADR_RGCOR_1_L;
	else if (pgm <= RICOH5B_REG_DAT_PGM_14DB)
		return RICOH5B_REG_ADR_RGCOR_2_L;
	else if (pgm <= RICOH5B_REG_DAT_PGM_20DB)
		return RICOH5B_REG_ADR_RGCOR_3_L;
	else
		return RICOH5B_REG_ADR_RGCOR_4_L;
}

/******************************************************************************/
/*!
 * @brief set Phase/Gain table command sequence
 */
/******************************************************************************/
static int set_pg_tbl_cmd_sequence(void)
{
	unsigned char data[RICOH5B_REG_LENGTH_PG];
	unsigned char address;
	unsigned short gain;
	unsigned short phase;
	int ret = 0;

	if (p_fine_pg_step < RICOH5B_AGCG_NUM) {
		/* set Phase/Gain correction */
		/* PRQA S 506 */
		if ((p_fine_pg->gain[p_fine_pg_step] >= 0x400)
		 && (p_fine_pg->gain[p_fine_pg_step] <= 0xC00)) {
			gain  = p_fine_pg->gain[p_fine_pg_step];
			phase = p_fine_pg->phase[p_fine_pg_step];
		} else {
			gain  = RICOH5B_REG_DAT_RGCOR_DEFAULT;
			phase = RICOH5B_REG_DAT_RPHCOR_DEFAULT;
		}

		address =
		    get_pg_correction_address(align_agcg_tbl[p_fine_pg_step]);
		data[0] = RICOH5B_GET_LOWER_BYTE(gain);
		data[1] = RICOH5B_GET_UPPER_BYTE(gain);
		data[2] = RICOH5B_GET_LOWER_BYTE(phase);
		data[3] = RICOH5B_GET_UPPER_BYTE(phase);

		ret = ricoh5b_cmd_write_byte(
		    address, data, RICOH5B_REG_LENGTH_PG);
		p_fine_pg_step++;
	}

	return ret;
}

/******************************************************************************/
/*!
 * @brief set Phase/Gain table command
 */
/******************************************************************************/
int ricoh5b_send_set_pg_tbl(struct ricoh5b_pg *p_pg)
{
	int ret = 0;

	if (p_pg != (struct ricoh5b_pg *)RICOH5B_NULL) {
		p_fine_pg = p_pg;
		ret = set_pg_tbl_cmd_sequence();
	}
	return ret;
}

/******************************************************************************/
/*!
 * @brief set De-Emphasis command
 */
/******************************************************************************/
static int ricoh5b_send_set_de_emphasis(struct ricoh5b_dev *ricoh5b_dev)
{
	int ret = 0;

	ret = ricoh5b_cmd_write_bit(RICOH5B_REG_ADR_DEMP,
		ricoh5b_dev->deemp, RICOH5B_REG_BIT_DEMP);
	if (ret < 0) {
		LOG_ERR("RICOH5B_REG_BIT_DEMP err = %d\n", ret);
		goto send_err;
	}

	ret = ricoh5b_cmd_write_bit(RICOH5B_REG_ADR_DEMP_CE,
		ricoh5b_dev->deemp_ce, RICOH5B_REG_BIT_DEMP_CE);
	if (ret < 0) {
		LOG_ERR("RICOH5B_REG_BIT_DEMP_CE err = %d\n", ret);
		goto send_err;
	}

	return 0;

send_err:
	return -1;
}

/******************************************************************************/
/*!
 * @brief Write Packet Create Function
 */
/******************************************************************************/
static void generate_write_packet(unsigned char *buf, unsigned char reg_addr,
	unsigned char *data, unsigned char count)
{
	int i;

	if ((count <= DATA_WRITE_LENGTH_MAX)
		&& (data != (unsigned char *)RICOH5B_NULL)) {
		for (i = count-1; i >= 0; i--)
			buf[WRITE_INDEX_DATA+i] = data[i];
		buf[WRITE_INDEX_ADDRESS] = reg_addr;
	}
}

/******************************************************************************/
/*!
 * @brief Data Read Function
 */
/******************************************************************************/
int ricoh5b_cmd_read_byte(unsigned char reg_addr, unsigned char *buf)
{
	int ret = 0;
	int retry = RICOH5B_I2C_RETRY_COUNT;
	struct i2c_msg msgs[2];
	int msgs_num = ARRAY_SIZE(msgs);

	memset(&msgs[0], 0, sizeof(msgs));
	msgs[0].addr   = ricoh5b_i2c_client->addr;
	msgs[0].flags  = 0;
	msgs[0].len    = 1;
	msgs[0].buf    = &reg_addr;
	msgs[1].addr   = ricoh5b_i2c_client->addr;
	msgs[1].flags  = I2C_M_RD;
	msgs[1].len    = 1;
	msgs[1].buf    = buf;

	do {
		retry--;

		buf[0] = 0;

		ret = i2c_transfer(
		    ricoh5b_i2c_client->adapter, &msgs[0], msgs_num);
		if (ret == msgs_num) {
			#ifdef RICOH5B_MAIN_REG_DEBUG
			LOG_DBG(" read: adr[%02x] %02x\n",
					reg_addr, buf[0]);
			#endif
			ret = msgs[1].len;
			return ret;
		}

		LOG_ERR("I2C Read error"
			" ret=%d addr=0x%x reg_addr=0x%x retry=%d\n",
			ret,
			ricoh5b_i2c_client->addr,
			reg_addr,
			retry);

		ricoh5b_cmd_set_sequence_timer(
			CHIP_RETRY_WAIT_TIME); /* Retry, chip was in standby */
	} while (retry);

	return ret > 0 ? -EIO : ret;
}

/******************************************************************************/
/*! @brief Multi Byte data Write API
 *
 *  @param  reg_addr  [in]  Register Address
 *  @param  data      [in]  First Address of Data buffer
 *  @param  count     [in]  (Address + Data) Length
 *
 */
/******************************************************************************/
int ricoh5b_cmd_write_byte(unsigned char reg_addr, unsigned char *buf,
	size_t count)
{
	int ret = 0, retry = RICOH5B_I2C_RETRY_COUNT;
	struct i2c_msg msgs[1];
	int msgs_num = ARRAY_SIZE(msgs);
	#ifdef RICOH5B_MAIN_REG_DEBUG
	//unsigned char buf_read[READ_BUF_SIZE] = {0, };
	#endif

	generate_write_packet(buf, reg_addr, buf, count - 1);
	#ifdef RICOH5B_MAIN_REG_DEBUG
	{
		int i;
		int len = 0;
		char pr_buf[8+READ_BUF_SIZE*3];

		len += snprintf(&pr_buf[len], sizeof(pr_buf)-len,
			"adr[%02x] ", buf[0]);
		for (i = 1; i < count; i++) {
			len += snprintf(&pr_buf[len], sizeof(pr_buf)-len,
				"%02x ", buf[i]);
		}
		LOG_DBG("write: %s\n", pr_buf);
	}
	#endif

	memset(&msgs[0], 0, sizeof(msgs));
	msgs[0].addr   = ricoh5b_i2c_client->addr;
	msgs[0].flags  = 0;
	msgs[0].len    = count;
	msgs[0].buf    = buf;

	/* Write data */
	do {
		retry--;

		ret = i2c_transfer(ricoh5b_i2c_client->adapter, msgs, msgs_num);
		if (ret == msgs_num) {
			#ifdef RICOH5B_MAIN_REG_DEBUG
			// ricoh5b_cmd_read_byte(reg_addr, buf_read);
			#endif
			return count;
		}
		ricoh5b_cmd_set_sequence_timer(
			CHIP_RETRY_WAIT_TIME); /* Retry, chip was in standby */
	} while (retry);

	return ret > 0 ? -EIO : ret;
}

/******************************************************************************/
/*! @brief Mask 1Byte data Write API
 *
 *  @param  reg_addr  [in]  Register Address
 *  @param  mask      [in]  Bit Mask
 *  @param  data      [in]  Data
 *
 */
/******************************************************************************/
int ricoh5b_cmd_write_bit(unsigned char reg_addr, unsigned char data,
	unsigned char mask)
{
	int ret = 0;
	unsigned char buf[WRITE_BUF_SIZE] = {0, };

	ret = ricoh5b_cmd_read_byte(reg_addr,
				buf); /* Read current register value */
	if (ret < 0) {
		LOG_ERR("ricoh5b_cmd_read_byte returned %d\n", ret);
		ret = -EIO;
		return ret;
	}

	buf[0] &= (unsigned char)(~mask);	/* clear target bit	*/
	buf[0] |= (data & mask);		/* add target bit */

	ret = ricoh5b_cmd_write_byte(reg_addr,
				buf,
				RICOH5B_REG_ADDRESS_LENGTH
				 + DATA_1BYTE_LENGTH); /* Write data */
	if (ret != RICOH5B_REG_ADDRESS_LENGTH + DATA_1BYTE_LENGTH) {
		LOG_ERR("i2c_master_send returned %d\n", ret);
		ret = -EIO;
	}

	return ret;
}

/******************************************************************************/
/*! @brief table set data sequencial Write API
 *
 *  @param  tbl    [in]  Write Data Table
 *  @param  size   [in]  Table Size
 *
 */
/******************************************************************************/
int ricoh5b_cmd_write_sequencial(struct ricoh5b_reg *tbl, unsigned char count)
{
	int ret = 0;
	unsigned char w_tbl_index = 0;/* Sequencial Write table index */
	unsigned char buf[WRITE_BUF_SIZE] = {0, };

	LOG_DBG_V("count=%d\n", count);
	for (w_tbl_index = 0; w_tbl_index < count; w_tbl_index++) {
		/* write */
		if (tbl[w_tbl_index].mask
			== RICOH5B_REG_NON_MASK) { /* write byte */
			memcpy(buf,
				&(tbl[w_tbl_index].data), DATA_1BYTE_LENGTH);
			ret = ricoh5b_cmd_write_byte(
				tbl[w_tbl_index].address,
				buf,
				RICOH5B_REG_ADDRESS_LENGTH
				 + DATA_1BYTE_LENGTH);
			if (ret != RICOH5B_REG_ADDRESS_LENGTH
				+ DATA_1BYTE_LENGTH) {
				LOG_ERR("ricoh5b_cmd_write_byte returned %d\n",
					ret);
				ret = (ret >= 0) ? -EIO : ret;
				return ret;
			}
		} else {	/* write bit */
			ret = ricoh5b_cmd_write_bit(
				tbl[w_tbl_index].address,
				tbl[w_tbl_index].data,
				tbl[w_tbl_index].mask);
			if (ret != RICOH5B_REG_ADDRESS_LENGTH
				+ DATA_1BYTE_LENGTH) {
				LOG_ERR("ricoh5b_cmd_write_bit returned %d\n",
					ret);
				ret = (ret >= 0) ? -EIO : ret;
				return ret;
			}
		}
	}

	return ret;
}

/******************************************************************************/
/*! @brief audio output setting command
 *
 *  @param  kind  [in]  Analog      -> 0 :RICOH5B_AUDIO_OUTPUT_ANALOG<br>
 *             IIS 32fs mode  -> 1 :RICOH5B_AUDIO_OUTPUT_IIS_32FS<br>
 *             IIS 64fs mode  -> 2 :RICOH5B_AUDIO_OUTPUT_IIS_64FS<br>
 */
/******************************************************************************/
static int ricoh5b_send_audio_output_setting(unsigned char kind)
{
	int ret = 0;

	switch (kind) {
	case RICOH5B_AUDIO_OUTPUT_ANALOG:
		ret = ricoh5b_cmd_write_sequencial(
			(struct ricoh5b_reg *)
			&ricoh5b_cmd_audio_output_analog_tbl[0],
			RICOH5B_CMD_AUDIO_OUTPUT_ANALOG_TBL_SIZE);
		break;
	case RICOH5B_AUDIO_OUTPUT_IIS_32FS:
		ret = ricoh5b_cmd_write_sequencial(
			(struct ricoh5b_reg *)
			&ricoh5b_cmd_audio_output_iis_32fs_tbl[0],
			RICOH5B_CMD_AUDIO_OUTPUT_IIS_32FS_TBL_SIZE);
		break;
	case RICOH5B_AUDIO_OUTPUT_IIS_64FS:
		ret = ricoh5b_cmd_write_sequencial(
			(struct ricoh5b_reg *)
			&ricoh5b_cmd_audio_output_iis_64fs_tbl[0],
			RICOH5B_CMD_AUDIO_OUTPUT_IIS_64FS_TBL_SIZE);
		break;
	default:
		break;
	}

	return ret;
}

/******************************************************************************/
/*!
 * @brief Tuner Initialize sequence
 *
 */
/******************************************************************************/
static int ricoh5b_initialize_setting(void)
{
	int step;
	int ret = 0;

	for (step = 0; step < P_ON_STEP_END; step++) {
		LOG_DBG("IOCTL ricoh5b Initialize Step == %d\n", step);
		switch (step) {
		case P_ON_STEP_REG_RESET:
			ret = ricoh5b_cmd_write_sequencial(
				(struct ricoh5b_reg *)
				&ricoh5b_cmd_reg_reset_tbl[0],
				RICOH5B_CMD_REG_RESET_TBL_SIZE);
			if (ret < 0) {
				LOG_ERR("RICOH5B_CMD_REG_RESET_TBL err = %d\n",
					ret);
				goto send_err;
			}
			break;
		case P_ON_STEP_MODULE_ON:
			ret = ricoh5b_cmd_write_sequencial(
				(struct ricoh5b_reg *)
				&ricoh5b_cmd_module_power_on_tbl[0],
				RICOH5B_CMD_MODULE_POWER_ON_TBL_SIZE);
			if (ret < 0) {
				LOG_ERR("RICOH5B_CMD_MODULE_POWER_ON_TBL"
					" err = %d\n", ret);
				goto send_err;
			}
			break;
		case P_ON_STEP_XTAL_SETTING:
			ret = ricoh5b_cmd_write_sequencial(
				(struct ricoh5b_reg *)
				&ricoh5b_cmd_xtal_setting_tbl[0],
				RICOH5B_CMD_XTAL_SETTING_TBL_SIZE);
			if (ret < 0) {
				LOG_ERR("RICOH5B_CMD_XTAL_SETTING_TBL"
					" err = %d\n", ret);
				goto send_err;
			}
			break;
		case P_ON_STEP_SYNTHE_SETTING:
			ret = ricoh5b_cmd_write_sequencial(
				(struct ricoh5b_reg *)
				&ricoh5b_cmd_synthe_setting_tbl[0],
				RICOH5B_CMD_SYNTHE_SETTING_TBL_SIZE);
			if (ret < 0) {
				LOG_ERR("RICOH5B_CMD_SYNTHE_SETTING_TBL"
					" err = %d\n", ret);
				goto send_err;
			}
			break;
		case P_ON_STEP_AGC_SETTING:
			ret = ricoh5b_cmd_write_sequencial(
				(struct ricoh5b_reg *)
				&ricoh5b_cmd_agc_setting_tbl[0],
				RICOH5B_CMD_AGC_SETTING_TBL_SIZE);
			if (ret < 0) {
				LOG_ERR("RICOH5B_CMD_AGC_SETTING_TBL"
					" err = %d\n", ret);
				goto send_err;
			}
			break;
		case P_ON_STEP_DSP_INIT_SETTING:
			ret = ricoh5b_cmd_write_sequencial(
				(struct ricoh5b_reg *)
				&ricoh5b_cmd_dsp_init_setting_tbl[0],
				RICOH5B_CMD_DSP_INIT_SETTING_TBL_SIZE);
			if (ret < 0) {
				LOG_ERR("RICOH5B_CMD_DSP_INIT_SETTING_TBL"
					" err = %d\n", ret);
				goto send_err;
			}
			break;
		case P_ON_STEP_CHIP_ON:
			ret = ricoh5b_cmd_write_bit(RICOH5B_REG_ADR_PE,
				RICOH5B_REG_DAT_PE_ON, RICOH5B_REG_BIT_PE);
			if (ret < 0) {
				LOG_ERR("P_ON_STEP_CHIP_ON err = %d\n", ret);
				goto send_err;
			}
			break;
		case P_ON_STEP_WAIT_CHIP_ON:
			msleep(CHIP_POWER_ON_WAIT_TIME_MS);
			break;
		case P_ON_STEP_BLOCK_ON:
			ret = ricoh5b_cmd_write_bit(RICOH5B_REG_ADR_BLOCK1_PE,
				RICOH5B_REG_DAT_BLOCK1_PE_ALL_ON,
				RICOH5B_REG_NON_MASK);
			if (ret < 0) {
				LOG_ERR("P_ON_STEP_BLOCK_ON err = %d\n", ret);
				goto send_err;
			}
			break;
		case P_ON_STEP_WAIT_BLOCK_ON:
			msleep(BLOCK_POWER_ON_WAIT_TIME_MS);
			break;
		case P_ON_STEP_CK_GEN_SETTING:
			ret = ricoh5b_cmd_write_sequencial(
				(struct ricoh5b_reg *)
				&ricoh5b_cmd_clk_setting_tbl[0],
				RICOH5B_CMD_CLK_SETTING_TBL_SIZE);
			if (ret < 0) {
				LOG_ERR("RICOH5B_CMD_CLK_SETTING_TBL"
					" err = %d\n", ret);
				goto send_err;
			}
			break;
		case P_ON_STEP_RDSI_SETTING:
			ret = ricoh5b_cmd_write_sequencial(
				(struct ricoh5b_reg *)
				&ricoh5b_cmd_rdsi_output_tbl[0],
				RICOH5B_CMD_RDSI_OUTPUT_TBL_SIZE);
			if (ret < 0) {
				LOG_ERR("RICOH5B_CMD_RDSI_OUTPUT_TBL"
					" err = %d\n", ret);
				goto send_err;
			}
			break;
		case P_ON_STEP_AUDIO_SETTING:
			ret = ricoh5b_send_audio_output_setting(
				RICOH5B_SELECT_AUDIO_OUTPUT);
			if (ret < 0) {
				LOG_ERR("P_ON_STEP_AUDIO_SETTING err = %d\n",
					ret);
				goto send_err;
			}
			break;
		case P_ON_STEP_RTRIM_SETTING:
			ret = ricoh5b_cmd_write_bit(
				RICOH5B_REG_ADR_RTRIM_SEL,
				RICOH5B_REG_DAT_RTRIM_SEL_MANUAL,
				RICOH5B_REG_NON_MASK);
			if (ret < 0) {
				LOG_ERR("P_ON_STEP_RTRIM_SETTING err = %d\n",
					ret);
				goto send_err;
			}
			break;
		default:
			break;
		}
	}

	return 0;

send_err:
	return ret;
}

/******************************************************************************/
/*!
 * @brief Tuner audio output mute on sequence
 */
/******************************************************************************/
static int ricoh5b_audio_output_mute_on(void)
{
	int step;
	int ret = 0;

	for (step = 0; step < MUTE_ON_STEP_END; step++) {
		LOG_DBG("IOCTL ricoh5b Mute ON Step == %d\n", step);
		switch (step) {
		case MUTE_ON_STEP_SET_WAIT_COUNT:
			ret = ricoh5b_cmd_write_bit(
				RICOH5B_REG_ADR_MUTE_WAIT_CNT,
				RICOH5B_REG_DAT_MUTE_WAIT_CNT_MUTE_ON,
				RICOH5B_REG_NON_MASK);
			if (ret < 0) {
				LOG_ERR("MUTE_ON_STEP_SET_WAIT_COUNT"
					" err = %d\n", ret);
				goto send_err;
			}
			break;
		case MUTE_ON_STEP_WAIT_WAIT_COUNT:
			msleep(MUTING_WAIT_TIME_MS);
			break;
		case MUTE_ON_STEP_MUTE_ON:
			ret = ricoh5b_cmd_write_bit(RICOH5B_REG_ADR_MUTE,
				RICOH5B_REG_DAT_MUTE_ON,
				RICOH5B_REG_BIT_MUTE);
			if (ret < 0) {
				LOG_ERR("MUTE_ON_STEP_MUTE_ON err = %d\n",
					ret);
				goto send_err;
			}
			break;
		case MUTE_ON_STEP_WAIT_MUTE_ON:
			msleep(MUTING_WAIT_TIME_MS);
			if (ret < 0) {
				LOG_ERR("MUTE_ON_STEP_WAIT_MUTE_ON err = %d\n",
					ret);
				goto send_err;
			}
			break;
		default:
			break;
		}
	}

	return 0;
send_err:
	return -1;
}

/******************************************************************************/
/*!
 * @brief Tuner audio output mute off sequence
 */
/******************************************************************************/
int ricoh5b_audio_output_mute_off(void)
{
	int step;
	int ret = 0;

	for (step = 0; step < MUTE_OFF_STEP_END; step++) {
		LOG_DBG("IOCTL ricoh5b Mute OFF Step == %d\n", step);
		switch (step) {
		case MUTE_OFF_STEP_MUTE_OFF:
			ret = ricoh5b_cmd_write_bit(RICOH5B_REG_ADR_MUTE,
				RICOH5B_REG_DAT_MUTE_OFF,
				RICOH5B_REG_BIT_MUTE);
			if (ret < 0) {
				LOG_ERR("MUTE_OFF_STEP_MUTE_OFF err = %d\n",
					ret);
				goto send_err;
			}
			break;
		case MUTE_OFF_STEP_WAIT_MUTE_OFF:
			msleep(MUTING_WAIT_TIME_MS);
			break;
		case MUTE_OFF_STEP_SET_WAIT_COUNT:
			ret = ricoh5b_cmd_write_bit(
				RICOH5B_REG_ADR_MUTE_WAIT_CNT,
				RICOH5B_REG_DAT_MUTE_WAIT_CNT_MUTE_OFF,
				RICOH5B_REG_NON_MASK);
			if (ret < 0) {
				LOG_ERR("MUTE_OFF_STEP_SET_WAIT_COUNT"
					" err = %d\n", ret);
				goto send_err;
			}
			break;
		default:
			break;
		}
	}
	return 0;

send_err:
	return -1;
}

/******************************************************************************/
/*! @brief DITHER (modulater DC-offset) setting command
 *
 *  @param  band    [in]  FM/AM/IDLE
 *          audio_type  [in]  Analog      -> 0 :RICOH5B_AUDIO_OUTPUT_ANALOG<br>
 *              IIS 32fs mode  -> 1 :RICOH5B_AUDIO_OUTPUT_IIS_32FS<br>
 *              IIS 64fs mode  -> 2 :RICOH5B_AUDIO_OUTPUT_IIS_64FS<br>
 */
/******************************************************************************/
int ricoh5b_send_dither_setting(enum RICOH5B_BAND_STATUS band,
	unsigned char audio_type)
{
	int ret = 0;

	if (audio_type == RICOH5B_AUDIO_OUTPUT_ANALOG) {
		/* Analog audio */
		if (band == RICOH5B_BAND_FM) {
			ret = ricoh5b_cmd_write_bit(RICOH5B_REG_ADR_DITHER,
				RICOH5B_REG_DAT_DITHER_AUDIO_ANALOG_FM,
				RICOH5B_REG_NON_MASK);
		} else if (band == RICOH5B_BAND_AM) {
			ret = ricoh5b_cmd_write_bit(RICOH5B_REG_ADR_DITHER,
				RICOH5B_REG_DAT_DITHER_AUDIO_ANALOG_AM,
				RICOH5B_REG_NON_MASK);
		} else {
			/* do nothing */
		}
	} else if ((audio_type == RICOH5B_AUDIO_OUTPUT_IIS_32FS)
		|| (audio_type == RICOH5B_AUDIO_OUTPUT_IIS_64FS)) {
		/* Digital audio */
		ret = ricoh5b_cmd_write_bit(RICOH5B_REG_ADR_DITHER,
			RICOH5B_REG_DAT_DITHER_AUDIO_IIS,
			RICOH5B_REG_NON_MASK);
	} else {
		/* do nothing */
	}
	return ret;
}

static int ricoh5b_power_stop(void)
{
	int ret = 0;

	ret = ricoh5b_cmd_write_bit(RICOH5B_REG_ADR_MUTE_WAIT_CNT,
		RICOH5B_REG_DAT_MUTE_WAIT_CNT_MUTE_ON,
		RICOH5B_REG_NON_MASK);
	if (ret < 0) {
		LOG_ERR("RICOH5B_REG_ADR_MUTE_WAIT_CNT err = %d\n", ret);
		goto send_err;
	}
	msleep(20);
	ret = ricoh5b_cmd_write_bit(RICOH5B_REG_ADR_MUTE,
		RICOH5B_REG_DAT_MUTE_ON, RICOH5B_REG_BIT_MUTE);
	if (ret < 0) {
		LOG_ERR("RICOH5B_REG_ADR_MUTE err = %d\n", ret);
		goto send_err;
	}
	msleep(20);

	ret = ricoh5b_cmd_write_sequencial((struct ricoh5b_reg *)
		&ricoh5b_cmd_chip_power_off_tbl[0],
		RICOH5B_CMD_CHIP_POWER_OFF_TBL_SIZE);
	if (ret < 0) {
		LOG_ERR("RICOH5B_CMD_CHIP_POWER_OFF_TBL err = %d\n", ret);
		goto send_err;
	}

	return 0;

send_err:
	return ret;
}

/***********************/
/*@ operation_routines */
/***********************/
static void ricoh5b_repeat_check(struct work_struct *work)
{
	struct ricoh5b_dev *ricoh5b_dev;
	int audmode = 0;
	int signal_strength = 0;
	int ret = 0;

	if (tuner_open == 0)
		return;

	ricoh5b_dev = ricoh5b_device_info;

	if (ricoh5b_dev->event_disable)
		goto repeat_check_skip;

	mutex_lock(&ricoh5b_dev->repeat_check);

	ret = ricoh5b_rec_audio_mode();
	if (ret < 0)
		LOG_ERR("Stereo read Err\n");
	else
		audmode = ret;

	ret = ricoh5b_station_check(ricoh5b_dev);
	if (ret < 0)
		signal_strength = 0;
	else
		signal_strength = 1;

	if (ricoh5b_dev->rds_rcv_stat != ricoh5b_dev->rds_receive_stat) {
		change_rds_receive_stat = POLL_CHECK_FLAG_ENABLE;
		ricoh5b_dev->rds_receive_stat = ricoh5b_dev->rds_rcv_stat;
		LOG_DEBUG("rds_receive_stat = %d\n",
				ricoh5b_dev->rds_receive_stat);
	}

	if (audmode != ricoh5b_dev->rec_audio_mode) {
		change_audmode = POLL_CHECK_FLAG_ENABLE;
		ricoh5b_dev->rec_audio_mode = audmode;
	}

	if (signal_strength != res_signal_strength) {
		change_signal_strength = POLL_CHECK_FLAG_ENABLE;
		res_signal_strength = signal_strength;
	}

	if (change_audmode == POLL_CHECK_FLAG_ENABLE
		|| change_signal_strength == POLL_CHECK_FLAG_ENABLE
		|| change_rds_receive_stat == POLL_CHECK_FLAG_ENABLE) {
		wake_up_interruptible(&ricoh5b_dev->read_wq);
	}

	mutex_unlock(&ricoh5b_dev->repeat_check);

repeat_check_skip:
	if (tuner_open == 1) {
		queue_delayed_work(
			workqueue_check,
			&ricoh5b_check_work,
			RICOH5B_STATE_POLL_TIME
		);
	}

}

static int ricoh5b_rec_audio_mode(void)
{
	int ret = 0;
	unsigned char buf_stereo[READ_BUF_SIZE] = {0, };

	ret = ricoh5b_cmd_read_byte(RICOH5B_REG_ADR_STEREO, buf_stereo);
	if (ret < 0) {
		LOG_ERR("ricoh5b_stereo_check Error\n");
		return -1;
	}

	if ((buf_stereo[0] & RICOH5B_REG_BIT_FO_SAT)
		== RICOH5B_REG_DAT_FO_SAT_INSIDE) {
		/* Notify the current stereo indicator */
		ret = buf_stereo[0] & RICOH5B_REG_BIT_STEREO;
	} else {
		/*
		 * Notify the stereo indicator as mono.
		 * prevent tuner_mode from flicking during invalid frequency.
		 */
		ret = RICOH5B_REG_DAT_STEREO_OFF;
	}

	return ret;
}

static int ricoh5b_fo_sat_check(void)
{
	unsigned char buf_status[READ_BUF_SIZE] = {0, };
	int ret = 0;

	ret = ricoh5b_cmd_read_byte(RICOH5B_REG_ADR_FO_SAT, buf_status);
	if (ret < 0) {
		LOG_ERR("read err = %d\n", ret);
		goto read_err;
	}

	if ((buf_status[0] & RICOH5B_REG_BIT_FO_SAT)
		== RICOH5B_REG_DAT_FO_SAT_INSIDE) {
		return 0;
	}

read_err:
	return -1;
}

static int ricoh5b_rssi_check(int sens, int *value)
{
	unsigned char buf_status[READ_BUF_SIZE] = {0, };
	int ret = 0;
	int rssi_rd = -1;

	ret = ricoh5b_cmd_read_byte(RICOH5B_REG_ADR_RSSI_RD, buf_status);
	if (ret < 0) {
		LOG_ERR("read err = %d\n", ret);
		goto read_err;
	}
	rssi_rd = buf_status[0] & RICOH5B_SIGNAL_STRENGTH_MASK;

	if (rssi_rd < sens)
		return -1;

	*value = rssi_rd;

	return 0;

read_err:
	return -1;
}

static int ricoh5b_station_check(struct ricoh5b_dev *ricoh5b_dev)
{
	int ret = 0;
	int retry = 0;
	int rssi_raw = 0;
	unsigned char val = 0;

	msleep(80);

	for (retry = 0; retry < SIGNAL_STRENGTH_RETRY; retry++) {

		msleep(20);

		ret = ricoh5b_fo_sat_check();
		if (ret) {
			if (ricoh5b_dev->station_check != 0)
				LOG_DBG_ERR("fo_sat Error\n");
			goto check_ng;
		}

		if (ricoh5b_dev->sense_mode == SENSE_MODE_HIGH)
			val = ricoh5b_dev->sense_high;
		else
			val = ricoh5b_dev->sense_low;
		ret = ricoh5b_rssi_check(val, &rssi_raw);
		if (ret) {
			if (ricoh5b_dev->station_check != 0)
				LOG_DBG_ERR("rssi err\n");
			goto check_ng;
		}
	}

	mutex_lock(&ricoh5b_dev->station_mutex);
	ricoh5b_dev->station_check = 1;
	mutex_unlock(&ricoh5b_dev->station_mutex);

	LOG_DEBUG_V("station check OK rssi_raw=%d\n", rssi_raw);

	return 0;

check_ng:
	mutex_lock(&ricoh5b_dev->station_mutex);
	ricoh5b_dev->station_check = 0;
	mutex_unlock(&ricoh5b_dev->station_mutex);
	return -1;
}

static unsigned char buf_rds_data[RICOH5B_RDS_BUF];
static unsigned char buf_rds_check[RICOH5B_RDS_BUF];
static unsigned char buf_rds_resdata[RICOH5B_RDS_BUF];

static void ricoh5b_enable_rds_irq(struct ricoh5b_dev *ricoh5b_dev)
{
	if (!ricoh5b_dev->iic_irq_en) {
		enable_irq(ricoh5b_dev->client->irq);
		LOG_DBG_FUNC("enable_irq\n");
	}
	ricoh5b_dev->iic_irq_en = true;
}

static void ricoh5b_disable_rds_irq(struct ricoh5b_dev *ricoh5b_dev)
{
	if (ricoh5b_dev->iic_irq_en) {
		disable_irq(ricoh5b_dev->client->irq);
		LOG_DBG_FUNC("disable_irq\n");
	}
	ricoh5b_dev->iic_irq_en = false;
}

static void ricoh5b_next_rds_eint(struct ricoh5b_dev *ricoh5b_dev)
{
	mutex_lock(&ricoh5b_dev->rds_mode_mutex);
	if (ricoh5b_dev->rds_mode == RICOH5B_RDS_MODE_ON)
		ricoh5b_enable_rds_irq(ricoh5b_dev);
	else
		ricoh5b_disable_rds_irq(ricoh5b_dev);
	mutex_unlock(&ricoh5b_dev->rds_mode_mutex);
}

static void ricoh5b_get_rds(struct work_struct *work)
{
	struct ricoh5b_dev *ricoh5b_dev;
	unsigned char buf_status[READ_BUF_SIZE] = {0, };
	unsigned char buf_data_l[READ_BUF_SIZE] = {0, };
	unsigned char buf_data_h[READ_BUF_SIZE] = {0, };
	unsigned char buf_offset_l[READ_BUF_SIZE] = {0, };
	unsigned char buf_offset_h[READ_BUF_SIZE] = {0, };
	unsigned char rds_noerr = 0;
	unsigned char rds_unc = 0;
	unsigned char rds_block = 0;
	unsigned char blocktype = 0;
	unsigned char segment_addr = 0;
	static int write_pos;
	int i = 0;
	int gpio_val = 0;
	static int next_data_write = RDS_WRITE_DISABLE;

	ricoh5b_dev = ricoh5b_device_info;

	gpio_val = gpiod_get_value(ricoh5b_dev->iic_gpio);
	if (gpio_val == 1) {	/* Active */

		ricoh5b_cmd_read_byte(RICOH5B_REG_ADR_RDS_STATUS, buf_status);
		ricoh5b_cmd_read_byte(RICOH5B_REG_ADR_RDS_DATA_L, buf_data_l);
		ricoh5b_cmd_read_byte(RICOH5B_REG_ADR_RDS_DATA_H, buf_data_h);
		ricoh5b_cmd_read_byte(RICOH5B_REG_ADR_RDS_OFFSET_L, buf_offset_l);
		ricoh5b_cmd_read_byte(RICOH5B_REG_ADR_RDS_OFFSET_H, buf_offset_h);
		mutex_lock(&ricoh5b_dev->station_mutex);
		if (ricoh5b_dev->station_check == 0) {
			if (next_data_write != RDS_WRITE_DISABLE)
				LOG_DBG("signal level drop !!!\n");
			next_data_write = RDS_WRITE_DISABLE;
			memset(&buf_rds_data, 0, sizeof(buf_rds_data));
			memset(&buf_rds_check, 0, sizeof(buf_rds_check));
			ricoh5b_next_rds_eint(ricoh5b_dev);
			mutex_unlock(&ricoh5b_dev->station_mutex);
			return;
		}
		mutex_unlock(&ricoh5b_dev->station_mutex);

		rds_noerr = buf_status[0] & RICOH5B_RDS_NOERR;
		rds_unc   = buf_status[0] & RICOH5B_RDS_UNC;
		rds_block = (buf_status[0] & RICOH5B_RDS_BLOCK) >> RDS_BLOCK_SHIFT;
		blocktype = buf_data_h[0] & RICOH5B_RDS_BLOCK_MASK;
		segment_addr = (buf_data_l[0] & RICOH5B_RDS_SEGMENT_MASK);

		LOG_DEBUG_V("rds_noerr = 0x%x\n", rds_noerr);
		LOG_DEBUG_V("rds_unc = 0x%x\n", rds_unc);
		LOG_DEBUG_V("rds_block = 0x%x\n", rds_block);
		LOG_DEBUG_V("blocktype = 0x%x\n", blocktype);
		LOG_DEBUG_V("buf_data_l[0] = 0x%x\n", buf_data_l[0]);
		LOG_DEBUG_V("buf_data_h[0] = 0x%x\n", buf_data_h[0]);
		LOG_DEBUG_V("segment addr = 0x%x\n", segment_addr);

		if (rds_noerr == RICOH5B_RDS_NOERR || rds_unc == RICOH5B_RDS_UNC) {
			ricoh5b_set_rds_rcv_stat();
			if (rds_block == RICOH5B_RDS_BLOCK_B) {
				if (blocktype == RICOH5B_RDS_BLOCK_B_PS) {
					if (segment_addr == RICOH5B_RDS_SEGMENT_ADDR_1
						&& write_pos == FIRST_POSITION) {
						next_data_write = RDS_WRITE_ENABLE;
					} else if (segment_addr == RICOH5B_RDS_SEGMENT_ADDR_2
						&& write_pos == SECOND_POSITION) {
						next_data_write = RDS_WRITE_ENABLE;
					} else if (segment_addr == RICOH5B_RDS_SEGMENT_ADDR_3
						&& write_pos == THIRD_POSITION) {
						next_data_write = RDS_WRITE_ENABLE;
					} else if (segment_addr == RICOH5B_RDS_SEGMENT_ADDR_4
						&& write_pos == FOURTH_POSITION) {
						next_data_write = RDS_WRITE_ENABLE;
					}
				} else {
					ricoh5b_next_rds_eint(ricoh5b_dev);
					return;
				}
			}

			if (rds_block == RICOH5B_RDS_BLOCK_C) {
				if (next_data_write) {
					next_data_write = RDS_WRITE_DISABLE;
					if ((((buf_data_h[0] >= RICOH5B_RDS_LIMIT_MIN)
						&& (buf_data_h[0] <= RICOH5B_RDS_LIMIT_MAX))
						&& ((buf_data_l[0] >= RICOH5B_RDS_LIMIT_MIN)
						&& (buf_data_l[0] <= RICOH5B_RDS_LIMIT_MAX)))) {
						buf_rds_data[write_pos] = buf_data_h[0];
						write_pos++;
						buf_rds_data[write_pos] = buf_data_l[0];
						write_pos++;
					} else {
						LOG_ERR("write data error h = 0x%x l = 0x%x\n",
							buf_data_h[0], buf_data_l[0]);
						write_pos = FIRST_POSITION;
						memset(&buf_rds_data, 0, sizeof(buf_rds_data));
						ricoh5b_next_rds_eint(ricoh5b_dev);
						return;
					}

					if (write_pos == RICOH5B_RDS_BUF) {
						write_pos = FIRST_POSITION;
						for (i = 0; i < RICOH5B_RDS_BUF; i++) {
							if (buf_rds_data[i] == buf_rds_check[i]) {
								if (i == RICOH5B_LAST_DATA) {
									mutex_lock(&ricoh5b_dev->repeat_check);
									memcpy(buf_rds_resdata, &buf_rds_check[0], sizeof(buf_rds_resdata));
									LOG_DBG("ps_name: %02x %02x %02x %02x %02x %02x %02x %02x\n",
									buf_rds_resdata[0], buf_rds_resdata[1], buf_rds_resdata[2],
									buf_rds_resdata[3], buf_rds_resdata[4], buf_rds_resdata[5],
									buf_rds_resdata[6], buf_rds_resdata[7]);
									change_rds_data = POLL_CHECK_FLAG_ENABLE;
									mutex_unlock(&ricoh5b_dev->repeat_check);
									memset(&buf_rds_check, 0, sizeof(buf_rds_check));
									wake_up_interruptible(&ricoh5b_dev->read_wq);
								}
							} else {
								memcpy(buf_rds_check, &buf_rds_data[0], sizeof(buf_rds_check));
								memset(&buf_rds_data, 0, sizeof(buf_rds_data));
								break;
							}
						}
					}
				}
			}
		} else {	/* Inactive */
			LOG_DBG("get rds data error\n");
			write_pos = FIRST_POSITION;
			next_data_write = RDS_WRITE_DISABLE;
			memset(&buf_rds_data, 0, sizeof(buf_rds_data));
		}
	}

	ricoh5b_next_rds_eint(ricoh5b_dev);
}

static irqreturn_t ricoh5b_get_rds_irq(int irq_num, void *data)
{
	struct ricoh5b_dev *ricoh5b_dev;

	ricoh5b_dev = data;

	queue_delayed_work(
		ricoh5b_dev->workqueue,
		&ricoh5b_dev->ricoh5b_rds_work,
		ACCESSARY_CHECK_DELAY_TIME
	);

	return IRQ_HANDLED;
}

static int ricoh5b_rds_set_status(struct ricoh5b_dev *ricoh5b_dev, int val)
{
	int ret = 0;

	mutex_lock(&ricoh5b_dev->rds_mode_mutex);
	if (val == RICOH5B_RDS_MODE_ON) {

		if (ricoh5b_dev->event_disable == true) {
			LOG_DEBUG("RDS on setting prohibited\n");
		} else {
			ret = ricoh5b_cmd_write_bit(RICOH5B_REG_ADR_RDS_SETTING_1,
				rbds_enable, RICOH5B_REG_RDS_ENABLE);
			if (ret < 0) {
				LOG_ERR("RICOH5B_REG_ADR_RDS_SETTING_1 err = %d\n",
					ret);
				goto send_err;
			}
			ret = ricoh5b_cmd_write_bit(RICOH5B_REG_ADR_RDS_SETTING_2,
				RICOH5B_REG_DAT_RDS_SETTING_2,
				RICOH5B_REG_RDS_ERR_VAL | RICOH5B_REG_RDS_SYNC_VAL);
			if (ret < 0) {
				LOG_ERR("RICOH5B_REG_ADR_RDS_SETTING_2 err = %d\n",
					ret);
				goto send_err;
			}

			ricoh5b_dev->rds_mode = RICOH5B_RDS_MODE_ON;
			ricoh5b_enable_rds_irq(ricoh5b_dev);
			LOG_DEBUG("RDS on\n");
		}

	} else if (val == RICOH5B_RDS_MODE_OFF) {
		unsigned char buf_status[READ_BUF_SIZE] = {0, };
		unsigned char buf_data_l[READ_BUF_SIZE] = {0, };
		unsigned char buf_data_h[READ_BUF_SIZE] = {0, };
		unsigned char buf_offset_l[READ_BUF_SIZE] = {0, };
		unsigned char buf_offset_h[READ_BUF_SIZE] = {0, };
		int gpio_val = 0;
		int i = 0;

		ret = ricoh5b_cmd_write_bit(RICOH5B_REG_ADR_RDS_SETTING_1,
			RICOH5B_REG_RDS_OFF_VALUE, RICOH5B_REG_RDS_ENABLE);
		if (ret < 0) {
			LOG_ERR("RICOH5B_REG_ADR_RDS_SETTING_1 err = %d\n",
				ret);
			goto send_err;
		}
		ret = ricoh5b_cmd_write_bit(RICOH5B_REG_ADR_RDS_SETTING_2,
			RICOH5B_REG_DAT_RDS_SETTING_2,
			RICOH5B_REG_RDS_ERR_VAL | RICOH5B_REG_RDS_SYNC_VAL);
		if (ret < 0) {
			LOG_ERR("RICOH5B_REG_ADR_RDS_SETTING_2 err = %d\n",
				ret);
			goto send_err;
		}
		ricoh5b_dev->rds_mode = RICOH5B_RDS_MODE_OFF;
		ricoh5b_disable_rds_irq(ricoh5b_dev);

		for (i = 0; i < RICOH5B_RDS_RETRY; i++) {
			gpio_val = gpiod_get_value(ricoh5b_dev->iic_gpio);
			if (gpio_val == 0)	/* Inactive */
				break;

			ricoh5b_cmd_read_byte(RICOH5B_REG_ADR_RDS_STATUS,
				buf_status);
			ricoh5b_cmd_read_byte(RICOH5B_REG_ADR_RDS_DATA_L,
				buf_data_l);
			ricoh5b_cmd_read_byte(RICOH5B_REG_ADR_RDS_DATA_H,
				buf_data_h);
			ricoh5b_cmd_read_byte(RICOH5B_REG_ADR_RDS_OFFSET_L,
				buf_offset_l);
			ricoh5b_cmd_read_byte(RICOH5B_REG_ADR_RDS_OFFSET_H,
				buf_offset_h);

			msleep(20);

		}

		ricoh5b_cancel_rds_rcv_stat();

		memset(&buf_rds_data, 0, sizeof(buf_rds_data));
		memset(&buf_rds_check, 0, sizeof(buf_rds_check));
		memset(&buf_rds_resdata, 0, sizeof(buf_rds_resdata));
		change_rds_data = POLL_CHECK_FLAG_DISABLE;
		LOG_DEBUG("RDS off\n");
	}

	mutex_unlock(&ricoh5b_dev->rds_mode_mutex);

	return 0;
send_err:
	mutex_unlock(&ricoh5b_dev->rds_mode_mutex);
	return -1;
}

static ssize_t ricoh5b_read(
	struct file *file,
	char __user *buf,
	size_t len, loff_t *loff)
{
	struct ricoh5b_dev *ricoh5b_dev;
	char res_buf[9] = {0};
	int ret = 0;
	ssize_t res_size = 0;

	ricoh5b_dev = ricoh5b_device_info;

	mutex_lock(&ricoh5b_dev->repeat_check);
	res_size = sizeof(res_buf);

	if (change_signal_strength == POLL_CHECK_FLAG_ENABLE) {
		res_buf[0] = SIGNAL_STRENGTH_FLAG;
		res_buf[1] = res_signal_strength;
	} else if (change_audmode == POLL_CHECK_FLAG_ENABLE) {
		res_buf[0] = REC_AUDIO_MODE_FLAG;
		res_buf[1] = ricoh5b_dev->rec_audio_mode;
	} else if (change_rds_receive_stat == POLL_CHECK_FLAG_ENABLE) {
		res_buf[0] = RDS_RECEIVE_STAT_FLAG;
		res_buf[1] = ricoh5b_dev->rds_receive_stat;
	} else if (change_rds_data == POLL_CHECK_FLAG_ENABLE) {
		res_buf[0] = RDS_FLAG;
		memcpy(&res_buf[1], &buf_rds_resdata[0], RICOH5B_RDS_BUF);
	}

	ret = copy_to_user((void *)buf, res_buf, res_size);
	if (ret != 0) {
		LOG_ERR("copy_to_user Error ret = %d\n", ret);
		mutex_unlock(&ricoh5b_dev->repeat_check);
		return -1;
	}

	if (res_buf[0] == SIGNAL_STRENGTH_FLAG) {
		change_signal_strength = POLL_CHECK_FLAG_DISABLE;
	} else if (res_buf[0] == REC_AUDIO_MODE_FLAG) {
		change_audmode = POLL_CHECK_FLAG_DISABLE;
	} else if (res_buf[0] == RDS_RECEIVE_STAT_FLAG) {
		change_rds_receive_stat = POLL_CHECK_FLAG_DISABLE;
	} else if (res_buf[0] == RDS_FLAG) {
		change_rds_data = POLL_CHECK_FLAG_DISABLE;
		memset(&buf_rds_resdata, 0, sizeof(buf_rds_resdata));
	}

	mutex_unlock(&ricoh5b_dev->repeat_check);

	return res_size;
}

static int ricoh5b_open(struct file *file)
{
	struct ricoh5b_dev *ricoh5b_dev;
	int ret = 0;

	LOG_DBG_FUNC("Start\n");

	ricoh5b_dev = ricoh5b_device_info;

	/* Tuner VDD On */
	ricoh5b_enable_regulator(ricoh5b_dev);
	msleep(20);

	/* Tuner Power On */
	gpiod_set_value(ricoh5b_dev->ce_gpio, 1);
	msleep(20);

	mutex_lock(&ricoh5b_dev->read_mutex);
	ret = ricoh5b_initialize_setting();
	if (ret < 0) {
		LOG_ERR("ricoh5b_initialize_setting err = %d\n", ret);
		goto send_err;
	}

	ret = ricoh5b_audio_output_mute_on();
	if (ret < 0) {
		LOG_ERR("ricoh5b_audio_output_mute_on err = %d\n", ret);
		goto send_err;
	} else {
		ricoh5b_dev->mute_state = RICOH5B_MUTE_ON;
	}

	ret = ricoh5b_send_audio_output_setting(RICOH5B_SELECT_AUDIO_OUTPUT);
	if (ret < 0) {
		LOG_ERR("ricoh5b_send_audio_output_setting err = %d\n", ret);
		goto send_err;
	}

	ret = ricoh5b_fm_send_set_band(ricoh5b_dev->tuned_freq / 10,
		gain_value);
	if (ret < 0) {
		LOG_ERR("ricoh5b_fm_send_set_band err = %d\n", ret);
		goto send_err;
	}

	/* deemphasis set */
	ret = ricoh5b_send_set_de_emphasis(ricoh5b_dev);
	if (ret < 0) {
		LOG_ERR("ricoh5b_send_set_de_emphasis err = %d\n", ret);
		goto send_err;
	}

	mutex_unlock(&ricoh5b_dev->read_mutex);

	memset(&buf_rds_data, 0, sizeof(buf_rds_data));
	memset(&buf_rds_check, 0, sizeof(buf_rds_check));
	memset(&buf_rds_resdata, 0, sizeof(buf_rds_resdata));

	ricoh5b_cancel_rds_rcv_stat();

	tuner_open = 1;
	change_rds_data = POLL_CHECK_FLAG_DISABLE;

	queue_delayed_work(
		workqueue_check,
		&ricoh5b_check_work,
		RICOH5B_STATE_POLL_TIME
	);

	LOG_DBG_FUNC("End\n");

	return 0;
send_err:
	/* Power off */
	gpiod_set_value(ricoh5b_dev->ce_gpio, 0);
	/* Tuner VDD Off */
	ricoh5b_disable_regulator(ricoh5b_dev);

	mutex_unlock(&ricoh5b_dev->read_mutex);
	return -EIO;
}

static int ricoh5b_close(struct file *file)
{
	struct ricoh5b_dev *ricoh5b_dev;
	int ret = 0;

	LOG_DBG_FUNC("Start\n");

	ricoh5b_dev = ricoh5b_device_info;

	tuner_open = 0;

	cancel_delayed_work(&ricoh5b_check_work);
	flush_workqueue(workqueue_check);

	ret = ricoh5b_rds_set_status(ricoh5b_dev, 0);
	if (ret < 0)
		LOG_ERR("rds stop Error\n");

	cancel_delayed_work(&ricoh5b_dev->ricoh5b_rds_work);
	flush_workqueue(ricoh5b_dev->workqueue);

	mutex_lock(&ricoh5b_dev->read_mutex);

	ret = ricoh5b_power_stop();
	if (ret < 0)
		LOG_ERR("ricoh5b_power_stop err = %d\n", ret);
	mutex_unlock(&ricoh5b_dev->read_mutex);

	/* Power off */
	gpiod_set_value(ricoh5b_dev->ce_gpio, 0);

	/* Tuner VDD Off */
	ricoh5b_disable_regulator(ricoh5b_dev);

	LOG_DBG_FUNC("End\n");
	return 0;
}

static unsigned int ricoh5b_poll(
	struct file *file,
	struct poll_table_struct *s_wait)
{
	struct ricoh5b_dev *ricoh5b_dev;
	int ret = 0;

	ricoh5b_dev = ricoh5b_device_info;

	poll_wait(file, &ricoh5b_dev->read_wq, s_wait);

	mutex_lock(&ricoh5b_dev->repeat_check);

	if (change_rds_data == POLL_CHECK_FLAG_ENABLE ||
		change_signal_strength == POLL_CHECK_FLAG_ENABLE ||
		change_audmode == POLL_CHECK_FLAG_ENABLE ||
		change_rds_receive_stat == POLL_CHECK_FLAG_ENABLE) {
		ret |= POLLIN;
	} else {
		ret = 0;
	}
	mutex_unlock(&ricoh5b_dev->repeat_check);

	return ret;
}

static int ricoh5b_querycap(
	struct file *file,
	void *dummy,
	struct v4l2_capability *capability)
{
	struct ricoh5b_dev *ricoh5b_dev;

	ricoh5b_dev = ricoh5b_device_info;

	strlcpy(capability->driver, "ricoh5b_tuner",
		sizeof(capability->driver));
	strlcpy(capability->card, "ricoh5b", sizeof(capability->card));
	snprintf(capability->bus_info, sizeof(capability->bus_info),
		"I2C:%s", dev_name(&ricoh5b_dev->client->dev));
	capability->version	= KERNEL_VERSION(1, 0, 0);
	capability->device_caps = V4L2_CAP_TUNER | V4L2_CAP_RADIO;
	capability->capabilities =
			capability->device_caps | V4L2_CAP_DEVICE_CAPS;

	return 0;
}

static int ricoh5b_s_tuner(
	struct file *file,
	void *dummy,
	const struct v4l2_tuner *tuner)
{
	int ret = 0;
	unsigned char val = 0;
	__u32 set_mode = 0;
	struct ricoh5b_dev *ricoh5b_dev;

	ricoh5b_dev = ricoh5b_device_info;

	if (tuner->audmode == V4L2_TUNER_MODE_STEREO) {
		val = RICOH5B_REG_DAT_MONO_OFF;
		set_mode = V4L2_TUNER_MODE_STEREO;
	} else if (tuner->audmode == V4L2_TUNER_MODE_MONO) {
		val = RICOH5B_REG_DAT_MONO_ON;
		set_mode = V4L2_TUNER_MODE_MONO;
	}

	ret = ricoh5b_cmd_write_bit(RICOH5B_REG_ADR_MONO, val,
		RICOH5B_REG_BIT_MONO);

	if (ret != RICOH5B_REG_ADDRESS_LENGTH + DATA_1BYTE_LENGTH) {
		LOG_ERR("ricoh5b_cmd_write_bit Error = %d\n", ret);
		return ret;
	}

	ricoh5b_dev->audio_mode = set_mode;

	return 0;
}

static int ricoh5b_g_tuner(
	struct file *file,
	void *dummy,
	struct v4l2_tuner *tuner)
{
	struct ricoh5b_dev *ricoh5b_dev;
	int ret = 0;

	LOG_DBG_FUNC("Start\n");

	ricoh5b_dev = ricoh5b_device_info;

	strlcpy(tuner->name, "FM", sizeof(tuner->name));

	tuner->type = V4L2_TUNER_RADIO;
	tuner->capability = V4L2_TUNER_CAP_LOW | V4L2_TUNER_CAP_STEREO;

	if (ricoh5b_dev->audio_mode == V4L2_TUNER_MODE_STEREO)
		tuner->audmode = V4L2_TUNER_MODE_STEREO;
	else if (ricoh5b_dev->audio_mode == V4L2_TUNER_MODE_MONO)
		tuner->audmode = V4L2_TUNER_MODE_MONO;

	ret = ricoh5b_station_check(ricoh5b_dev);
	mutex_lock(&ricoh5b_dev->repeat_check);
	if (ret < 0) {
		tuner->signal = 0;
		res_signal_strength = 0;
	} else {
		tuner->signal = 1;
		res_signal_strength = 1;
	}
	mutex_unlock(&ricoh5b_dev->repeat_check);

	/* not support */
	tuner->afc = 0;

	LOG_DBG_FUNC("End\n");

	return 0;
}

static int ricoh5b_s_frequency(
	struct file *file,
	void *dummy,
	const struct v4l2_frequency *frequency)
{
	struct ricoh5b_dev *ricoh5b_dev;
	int ret = 0;
	unsigned short freq_set_val = 0;
	unsigned int  freq			= 0;

	LOG_DBG_FUNC("Start\n");

	ricoh5b_dev = ricoh5b_device_info;

	freq = frequency->frequency >> 4;

	mutex_lock(&ricoh5b_dev->station_mutex);
	ricoh5b_dev->station_check = 0;
	mutex_unlock(&ricoh5b_dev->station_mutex);
	if (ricoh5b_dev->tuned_freq != freq) {
		LOG_DBG("tuned_freq %d\n", freq / 100);
		ricoh5b_cancel_rds_rcv_stat();
	}

	freq_set_val = (unsigned short)(freq / 10);

	ret = ricoh5b_fm_send_set_freq(freq_set_val, 0);
	if (ret < 0) {
		LOG_ERR("ricoh5b_fm_send_set_freq err ret = %d\n", ret);
		return -1;
	}

	ricoh5b_dev->tuned_freq = freq;

	LOG_DBG_FUNC("End\n");
	return 0;
}

static int ricoh5b_g_frequency(
	struct file *file,
	void *dummy,
	struct v4l2_frequency *frequency)
{
	struct ricoh5b_dev *ricoh5b_dev;

	ricoh5b_dev = ricoh5b_device_info;

	if (frequency->tuner != 0) {
		LOG_ERR("invalid tuner %d\n", frequency->tuner);
		return -EINVAL;
	}

	frequency->type = V4L2_TUNER_RADIO;
	frequency->frequency = (unsigned int)ricoh5b_dev->tuned_freq * 16;

	return 0;
}

static int ricoh5b_sense_mode_set(struct ricoh5b_dev *ricoh5b_dev, int value)
{
	int ret = 0;

	if (value > SENSE_MODE_HIGH)
		value = SENSE_MODE_LOW;
	ret = ricoh5b_threshold_set(ricoh5b_dev, value);

	if (ret < 0) {
		LOG_ERR("write bit Error ret = %d\n", ret);
		return -1;
	}

	ricoh5b_dev->sense_mode = value;

	return 0;
}

/* threshold value set */
static int ricoh5b_threshold_set(struct ricoh5b_dev *ricoh5b_dev, int value)
{
	int ret = 0;
	unsigned char val = 0;

	if (value == SENSE_MODE_HIGH)
		val = ricoh5b_dev->sense_high;
	else if (value == SENSE_MODE_LOW)
		val = ricoh5b_dev->sense_low;

	ret = ricoh5b_cmd_write_bit(RICOH5B_REG_ADR_RSSI_THR, val,
		RICOH5B_REG_BIT_RSSI_THR);

	if (ret != RICOH5B_REG_ADDRESS_LENGTH + DATA_1BYTE_LENGTH) {
		LOG_ERR("write bit Error ret = %d\n", ret);
		return -1;
	}

	return 0;
}

static int ricoh5b_gainvalue_set(int gain)
{
	int ret = 0;
	unsigned char volume_data[RICOH5B_VOLUME_DATA_SIZE];
	unsigned char fm_volume = 0;

	switch (gain) {
	case GAIN_0DB:
		gain_value = RICOH5B_VOLUME_0DB;
		fm_volume = RICOH5B_REG_DAT_HPVOL_FM_0DB;
		break;
	case GAIN_MINUS3DB:
		gain_value = RICOH5B_VOLUME_MINUS_3DB;
		fm_volume = RICOH5B_REG_DAT_HPVOL_FM_MINUS_3DB;
		break;
	case GAIN_MINUS4DB:
		gain_value = RICOH5B_VOLUME_MINUS_4DB;
		fm_volume = RICOH5B_REG_DAT_HPVOL_FM_MINUS_4DB;
		break;
	case GAIN_MINUS5DB:
		gain_value = RICOH5B_VOLUME_MINUS_5DB;
		fm_volume = RICOH5B_REG_DAT_HPVOL_FM_MINUS_5DB;
		break;
	default:
		gain_value = RICOH5B_VOLUME_0DB;
		fm_volume = RICOH5B_REG_DAT_HPVOL_FM_0DB;
		break;
	}

	volume_data[0] = fm_volume;
	volume_data[1] = fm_volume;
	ret = ricoh5b_cmd_write_byte(RICOH5B_REG_ADR_HPLVOL, &volume_data[0],
		RICOH5B_REG_ADDRESS_LENGTH + RICOH5B_REG_LENGTH_VOL);
	if (ret < 0) {
		LOG_ERR("BAND_STEP_VOLUME err = %d\n", ret);
		goto send_err;
	}

	return 0;

send_err:
	return -1;
}

static void ricoh5b_rdsmode_set(int mode)
{
	if (mode == RDS_MODE_RDS)
		rbds_enable = RICOH5B_REG_RDS_ON_VALUE;
	else if (mode == RDS_MODE_RBDS)
		rbds_enable = RICOH5B_REG_RDS_ON_VALUE
				| RICOH5B_REG_DAT_RDS_SETTING_1_RBDS;
	else
		rbds_enable = RICOH5B_REG_RDS_ON_VALUE;

}

static int ricoh5b_emphasis_set(struct ricoh5b_dev *ricoh5b_dev, int emphasis)
{
	int ret = 0;

	switch (emphasis) {
	case DEEMPHASIS_50US:
		ricoh5b_dev->deemp    = DEEMPHASIS_DEMP_HIGH;
		ricoh5b_dev->deemp_ce = DEEMPHASIS_LOW;
		break;
	case DEEMPHASIS_60US:
		ricoh5b_dev->deemp    = DEEMPHASIS_DEMP_HIGH;
		ricoh5b_dev->deemp_ce = DEEMPHASIS_DEMP_CE_HIGH;
		break;
	case DEEMPHASIS_75US:
		ricoh5b_dev->deemp    = DEEMPHASIS_LOW;
		ricoh5b_dev->deemp_ce = DEEMPHASIS_LOW;
		break;
	case DEEMPHASIS_100US:
		ricoh5b_dev->deemp    = DEEMPHASIS_LOW;
		ricoh5b_dev->deemp_ce = DEEMPHASIS_DEMP_CE_HIGH;
		break;
	default:
		ricoh5b_dev->deemp    = DEEMPHASIS_DEMP_HIGH;
		ricoh5b_dev->deemp_ce = DEEMPHASIS_LOW;
		break;
	}

	/* deemphasis set */
	ret = ricoh5b_send_set_de_emphasis(ricoh5b_dev);
	if (ret < 0) {
		LOG_ERR("ricoh5b_send_set_de_emphasis err = %d\n", ret);
		goto send_err;
	}

	return 0;

send_err:
	return -1;
}

static void ricoh5b_event_disable(struct ricoh5b_dev *ricoh5b_dev, int value)
{
	int ret;

	if (value) {
		ricoh5b_dev->event_disable = true;
		ret = ricoh5b_rds_set_status(ricoh5b_dev, RICOH5B_RDS_MODE_OFF);
		if (ret < 0)
			LOG_ERR("ricoh5b_rds_set_status Error\n");

		mutex_lock(&ricoh5b_dev->repeat_check);
		change_signal_strength = POLL_CHECK_FLAG_DISABLE;
		change_audmode = POLL_CHECK_FLAG_DISABLE;
		change_rds_receive_stat = POLL_CHECK_FLAG_DISABLE;
		change_rds_data = POLL_CHECK_FLAG_DISABLE;
		mutex_unlock(&ricoh5b_dev->repeat_check);
		LOG_DEBUG("event_disable = true\n");
	} else {
		ricoh5b_dev->event_disable = false;
		LOG_DEBUG("event_disable = false\n");
	}
}

static int ricoh5b_s_ctrl(
	struct file *file,
	void *dummy,
	struct v4l2_control *control)
{
	struct ricoh5b_dev *ricoh5b_dev;
	int ret = 0;

	ricoh5b_dev = ricoh5b_device_info;

	switch (control->id) {
	case V4L2_CID_AUDIO_MUTE:
		if (control->value) {
			/* mute on */
			ret = ricoh5b_audio_output_mute_on();
			if (ret < 0)
				LOG_ERR("ricoh5b_audio_output_mute_on Error\n");
			else
				ricoh5b_dev->mute_state = RICOH5B_MUTE_ON;
		} else {
			/* mute off */
			ret = ricoh5b_audio_output_mute_off();
			if (ret < 0)
				LOG_ERR("ricoh5b_audio_output_mute_off"
					" Error\n");
			else
				ricoh5b_dev->mute_state = RICOH5B_MUTE_OFF;
		}
		break;
	case V4L2_CID_SENSE_MODE:
		ret = ricoh5b_sense_mode_set(ricoh5b_dev, control->value);
		if (ret < 0)
			LOG_ERR("ricoh5b_sense_mode_set Error\n");
		break;
	case V4L2_CID_SENSE_THRESHOLD_HIGH:
		ricoh5b_dev->sense_high = (unsigned char)(control->value&0xFF);
		ret = ricoh5b_threshold_set(ricoh5b_dev,
			ricoh5b_dev->sense_mode);
		if (ret < 0)
			LOG_ERR("ricoh5b_threshold_set Error\n");
		break;
	case V4L2_CID_SENSE_THRESHOLD_LOW:
		ricoh5b_dev->sense_low = (unsigned char)(control->value&0xFF);
		ret = ricoh5b_threshold_set(ricoh5b_dev,
			ricoh5b_dev->sense_mode);
		if (ret < 0)
			LOG_ERR("ricoh5b_threshold_set Error\n");
		break;
	case V4L2_CID_RDS_ENABLE:
		ret = ricoh5b_rds_set_status(ricoh5b_dev, control->value);
		if (ret < 0)
			LOG_ERR("ricoh5b_rds_set_status Error\n");
		break;
	case V4L2_CID_GAIN_SETTING:
		ret = ricoh5b_gainvalue_set(control->value);
		if (ret < 0)
			LOG_ERR("ricoh5b_gainvalue_set Error\n");
		break;
	case V4L2_CID_RDS_MODE:
		ricoh5b_rdsmode_set(control->value);
		break;
	case V4L2_CID_DEEMPHASIS:
		ret = ricoh5b_emphasis_set(ricoh5b_dev, control->value);
		if (ret < 0)
			LOG_ERR("ricoh5b_emphasis_set Error\n");
		break;
	case V4L2_CID_READ_REGISTER:
	{
		int ret = 0;
		int value = 0;

		ret = ricoh5b_read_reg(control->value, &value);
		if (ret < 0) {
			LOG_ERR("ricoh5b_read_reg Error\n");
			return -1;
		}
		control->value = value;
		break;
	}
	case V4L2_CID_EVENT_DISABLE:
		ricoh5b_event_disable(ricoh5b_dev, control->value);
		break;
	default:
		return(-EINVAL);
	}

	return ret;
}

static int ricoh5b_g_ctrl(
	struct file *file,
	void *dummy,
	struct v4l2_control *control)
{
	struct ricoh5b_dev *ricoh5b_dev;

	ricoh5b_dev = ricoh5b_device_info;

	switch (control->id) {
	case V4L2_CID_AUDIO_MUTE:
		control->value = ricoh5b_dev->mute_state;
		break;
	case V4L2_CID_SENSE_MODE:
		control->value = ricoh5b_dev->sense_mode;
		break;
	case V4L2_CID_SENSE_THRESHOLD_HIGH:
		control->value = ricoh5b_dev->sense_high;
		break;
	case V4L2_CID_SENSE_THRESHOLD_LOW:
		control->value = ricoh5b_dev->sense_low;
		break;
	case V4L2_CID_AUDIO_MODE:
	{
		int ret = 0;

		mutex_lock(&ricoh5b_dev->repeat_check);
		ret = ricoh5b_rec_audio_mode();
		if (ret < 0) {
			LOG_ERR("rec_audio_mode err ret = %d\n", ret);
			mutex_unlock(&ricoh5b_dev->repeat_check);
			return -1;
		}
		ricoh5b_dev->rec_audio_mode = ret;
		control->value = ricoh5b_dev->rec_audio_mode;
		mutex_unlock(&ricoh5b_dev->repeat_check);
		break;
	}
	case V4L2_CID_RAW_SIGNAL_STRENGTH:
	{
		int rssi_raw = 0;
		int ret = 0;

		ret = ricoh5b_rssi_check(0, &rssi_raw);
		if (ret < 0) {
			LOG_ERR("ricoh5b_rssi_check Error\n");
			control->value = 0;
			return -1;
		}
		control->value = rssi_raw;
		break;
	}
	case V4L2_CID_RDS_RECEIVE_STATUS:
	{
		mutex_lock(&ricoh5b_dev->repeat_check);
		control->value = ricoh5b_dev->rds_receive_stat;
		change_rds_receive_stat = POLL_CHECK_FLAG_DISABLE;
		mutex_unlock(&ricoh5b_dev->repeat_check);
		break;
	}
	case V4L2_CID_EVENT_DISABLE:
		control->value = (ricoh5b_dev->event_disable) ?
			EVENT_DISABLE : EVENT_ENABLE;
		break;
	default:
		return(-EINVAL);
	}

	return 0;
}

static int ricoh5b_read_reg(int address, int *value)
{
	int ret = 0;
	unsigned char buf_r[READ_BUF_SIZE] = {0, };

	ret = ricoh5b_cmd_read_byte((unsigned char)address, buf_r);

	*value = buf_r[0];

	return ret;
}

static void ricoh5b_enable_regulator(struct ricoh5b_dev *ricoh5b_dev)
{
	int ret;

	if (ricoh5b_dev->vdd_regulator == NULL)
		return;

	if (!ricoh5b_dev->vdd_en) {
		LOG_DBG("vdd_regulator enable\n");
		ret = regulator_enable(ricoh5b_dev->vdd_regulator);
		if (ret < 0)
			LOG_ERR("Failed to enable vdd_regulator (%d)\n", ret);
		ricoh5b_dev->vdd_en = true;
	} else {
		LOG_DBG_WARN("vdd_regulator is already enabled\n");
	}
}

static void ricoh5b_disable_regulator(struct ricoh5b_dev *ricoh5b_dev)
{
	int ret;

	if (ricoh5b_dev->vdd_regulator == NULL)
		return;

	if (ricoh5b_dev->vdd_en) {
		LOG_DBG("vdd_regulator disable\n");
		ret = regulator_disable(ricoh5b_dev->vdd_regulator);
		if (ret < 0)
			LOG_ERR("Failed to disable vdd_regulator (%d)\n", ret);
		ricoh5b_dev->vdd_en = false;
	} else {
		LOG_DBG_WARN("vdd_regulator is already disabled\n");
	}
}

static int ricoh5b_probe(struct i2c_client *client,
		   const struct i2c_device_id *id)
{
	int ret = 0;
	struct ricoh5b_dev *ricoh5b_dev = NULL;
	struct device *dev = &client->dev;
	static atomic_t instance = ATOMIC_INIT(0);
	unsigned char data[READ_BUF_SIZE] = {0, };

	LOG_DBG_FUNC("Start\n");

	ricoh5b_dev = kzalloc(sizeof(*ricoh5b_dev), GFP_KERNEL);
	if (ricoh5b_dev == NULL) {
		LOG_ERR("Failed to allocate memory for module data\n");
		ret = -ENOMEM;
		goto err_exit;
	}

	ricoh5b_dev->vdd_en = false;
	ricoh5b_dev->vdd_regulator = devm_regulator_get(dev, "tuner_vdd");
	if (IS_ERR(ricoh5b_dev->vdd_regulator)) {
		ricoh5b_dev->vdd_regulator = NULL;
		LOG_ERR("Failed to request tuner_vdd-supply\n");
		goto err_exit;
	}
	/* Tuner VDD Off */
	ricoh5b_disable_regulator(ricoh5b_dev);

	/* Tuner IIC : input */
	ricoh5b_dev->iic_gpio = devm_gpiod_get(dev, "tuner_iic", GPIOD_IN);
	if (IS_ERR(ricoh5b_dev->iic_gpio)) {
		ret = PTR_ERR(ricoh5b_dev->iic_gpio);
		LOG_ERR("Failed to request tuner_iic-gpios\n");
		goto err_exit;
	}
	msleep(20);

	/* Tuner CE : output LOW */
	ricoh5b_dev->ce_gpio  = devm_gpiod_get(dev, "tuner_ce", GPIOD_OUT_LOW);
	if (IS_ERR(ricoh5b_dev->ce_gpio)) {
		ret = PTR_ERR(ricoh5b_dev->ce_gpio);
		LOG_ERR("Failed to request tuner_ce-gpios\n");
		goto err_exit;
	}
	msleep(20);

	ricoh5b_dev->client = client;
	ricoh5b_i2c_client  = ricoh5b_dev->client;
	ricoh5b_device_info = ricoh5b_dev;
	i2c_set_clientdata(client, ricoh5b_dev);

	/* Tuner VDD On */
	ricoh5b_enable_regulator(ricoh5b_dev);
	msleep(20);
	/* Tuner Power On */
	gpiod_set_value(ricoh5b_dev->ce_gpio, 1);
	msleep(20);
	/* Check the existence of the device */
	ret = ricoh5b_cmd_read_byte(RICOH5B_REG_ADR_VERSION, data);
	if (ret > 0)
		LOG_INFO("version %02X\n", data[0]);
	else
		LOG_ERR("no device error %d\n", ret);
	/* Power off */
	gpiod_set_value(ricoh5b_dev->ce_gpio, 0);
	/* Tuner VDD Off */
	ricoh5b_disable_regulator(ricoh5b_dev);
	if (ret <= 0)
		goto err_no_device;

	/* init mutex and queues */
	init_waitqueue_head(&ricoh5b_dev->read_wq);
	mutex_init(&ricoh5b_dev->read_mutex);
	mutex_init(&ricoh5b_dev->station_mutex);
	mutex_init(&ricoh5b_dev->rds_mode_mutex);
	mutex_init(&ricoh5b_dev->repeat_check);

	ricoh5b_dev->workqueue = create_singlethread_workqueue("ricoh5b_rds");
	if (!ricoh5b_dev->workqueue) {
		LOG_ERR("create_singlethread_workqueue() rds error\n");
		goto err_workqueue;
	}

	workqueue_check = create_singlethread_workqueue("ricoh5b_check");
	if (!workqueue_check) {
		LOG_ERR("create_singlethread_workqueue()"
			" workqueue_check error\n");
		goto err_workqueue;
	}

	INIT_DELAYED_WORK(&ricoh5b_dev->ricoh5b_rds_work, ricoh5b_get_rds);
	INIT_DELAYED_WORK(&ricoh5b_check_work, ricoh5b_repeat_check);

	if (ricoh5b_init_rds_rcv_stat() < 0)
		goto err_workqueue;

	ricoh5b_dev->video_device = video_device_alloc();

	if (ricoh5b_dev->video_device == NULL) {
		LOG_ERR("ricoh5b_dev->video_device NULL\n");
		goto err_video_device_register;
	}

	strncpy(ricoh5b_dev->video_device->name,
		"ricoh5b_Tuner", sizeof(ricoh5b_dev->video_device->name));

	ricoh5b_dev->video_device->release	 = video_device_release_empty;
	ricoh5b_dev->video_device->fops		 = &ricoh5b_file_operations;
	ricoh5b_dev->video_device->ioctl_ops = &ricoh5b_ioctl_operations;

	v4l2_device_set_name(&ricoh5b_dev->v4l2_dev,
		ricoh5b_dev->video_device->name, &instance);
	ricoh5b_dev->video_device->v4l2_dev	 = &ricoh5b_dev->v4l2_dev;
	ret = v4l2_device_register(dev, &ricoh5b_dev->v4l2_dev);
	if (ret) {
		LOG_ERR("v4l2_device_register error\n");
		goto err_v4l2_device_register;
	}

	/* set private data to video structure */
	video_set_drvdata(ricoh5b_dev->video_device, ricoh5b_dev);

	ret = video_register_device(ricoh5b_dev->video_device,
		VFL_TYPE_RADIO, -1);
	if (ret < 0) {
		LOG_ERR("video_register_device() error\n");
		goto err_video_register;
	}

	/* init status value set */
	ricoh5b_dev->tuned_freq = RICOH5B_FIRST_FREQ;
	ricoh5b_dev->audio_mode = RICOH5B_MODE_STEREO;
	ricoh5b_dev->mute_state = RICOH5B_MUTE_ON;
	ricoh5b_dev->sense_mode = RICOH5B_FIRST_SENSE_MODE;
	ricoh5b_dev->sense_high = RICOH5B_FIRST_SENSE_HIGH;
	ricoh5b_dev->sense_low	= RICOH5B_FIRST_SENSE_LOW;
	ricoh5b_dev->rds_mode	= RICOH5B_RDS_MODE_OFF;
	ricoh5b_dev->rec_audio_mode = RICOH5B_MODE_MONO;
	ricoh5b_dev->rds_receive_stat = RDS_NOT_RECEIVED;
	ricoh5b_dev->station_check = 0;
	ricoh5b_dev->event_disable = true;

	if (DeEmphasis == DEEMPHASIS_50US) {
		ricoh5b_dev->deemp	  = DEEMPHASIS_DEMP_HIGH;
		ricoh5b_dev->deemp_ce = DEEMPHASIS_LOW;
	} else if (DeEmphasis == DEEMPHASIS_60US) {
		ricoh5b_dev->deemp	  = DEEMPHASIS_DEMP_HIGH;
		ricoh5b_dev->deemp_ce = DEEMPHASIS_DEMP_CE_HIGH;
	} else if (DeEmphasis == DEEMPHASIS_75US) {
		ricoh5b_dev->deemp	  = DEEMPHASIS_LOW;
		ricoh5b_dev->deemp_ce = DEEMPHASIS_LOW;
	} else if (DeEmphasis == DEEMPHASIS_100US) {
		ricoh5b_dev->deemp	  = DEEMPHASIS_LOW;
		ricoh5b_dev->deemp_ce = DEEMPHASIS_DEMP_CE_HIGH;
	} else {
		ricoh5b_dev->deemp	  = DEEMPHASIS_DEMP_HIGH;
		ricoh5b_dev->deemp_ce = DEEMPHASIS_LOW;
	}

	if (GainValue == GAIN_0DB)
		gain_value = RICOH5B_VOLUME_0DB;
	else if (GainValue == GAIN_MINUS3DB)
		gain_value = RICOH5B_VOLUME_MINUS_3DB;
	else if (GainValue == GAIN_MINUS4DB)
		gain_value = RICOH5B_VOLUME_MINUS_4DB;
	else if (GainValue == GAIN_MINUS5DB)
		gain_value = RICOH5B_VOLUME_MINUS_5DB;
	else
		gain_value = RICOH5B_VOLUME_0DB;

	if (RdsMode == RDS_MODE_RDS)
		rbds_enable = RICOH5B_REG_RDS_ON_VALUE;
	else if (RdsMode == RDS_MODE_RBDS)
		rbds_enable = RICOH5B_REG_RDS_ON_VALUE
				| RICOH5B_REG_DAT_RDS_SETTING_1_RBDS;
	else
		rbds_enable = RICOH5B_REG_RDS_ON_VALUE;

	/* pin interrupt setting */
	client->irq = gpiod_to_irq(ricoh5b_dev->iic_gpio);
	if (client->irq < 0) {
		LOG_ERR("gpiod_to_irq() error (%d)\n", ret);
		goto err_irq;
	}
	ret = devm_request_irq(dev, client->irq,
			ricoh5b_get_rds_irq,
			IRQF_TRIGGER_FALLING | IRQF_SHARED,
			dev_name(dev), ricoh5b_dev);
	if (ret) {
		LOG_ERR("devm_request_irq() error (%d)\n", ret);
		goto err_irq;
	}
	ricoh5b_dev->iic_irq_en = false;
	disable_irq(client->irq);
	LOG_DBG_FUNC("disable_irq\n");

	LOG_INFO("Probe success\n");
	LOG_DBG_FUNC("End\n");

#ifdef CONFIG_REGMON_DEBUG
	ricoh5b_regmon_add(ricoh5b_dev);
#endif

	return 0;

err_irq:
	devm_free_irq(dev, client->irq, ricoh5b_dev);
err_video_register:
	video_device_release(ricoh5b_dev->video_device);
err_v4l2_device_register:
	v4l2_device_unregister(&ricoh5b_dev->v4l2_dev);
err_workqueue:
err_video_device_register:
	mutex_destroy(&ricoh5b_dev->read_mutex);
	mutex_destroy(&ricoh5b_dev->station_mutex);
	mutex_destroy(&ricoh5b_dev->rds_mode_mutex);
	mutex_destroy(&ricoh5b_dev->repeat_check);
err_no_device:
	ricoh5b_device_info = NULL;
	ricoh5b_i2c_client = NULL;
	kfree(ricoh5b_dev);
err_exit:

	return ret;
}

static int ricoh5b_remove(struct i2c_client *client)
{
	struct ricoh5b_dev *ricoh5b_dev;
	int ret = 0;

	LOG_DBG_FUNC("Start\n");
	LOG_INFO("Remove\n");

	ricoh5b_dev = i2c_get_clientdata(client);

	devm_free_irq(&client->dev, client->irq, ricoh5b_dev);

	cancel_delayed_work(&ricoh5b_check_work);
	flush_workqueue(workqueue_check);
	destroy_workqueue(workqueue_check);

	cancel_delayed_work(&ricoh5b_dev->ricoh5b_rds_work);
	flush_workqueue(ricoh5b_dev->workqueue);
	destroy_workqueue(ricoh5b_dev->workqueue);

	ricoh5b_destroy_rds_rcv_stat();

	if (tuner_open) {
		/* ricoh5b off Sequence */
		ret = ricoh5b_power_stop();
		if (ret)
			LOG_ERR("ricoh5b_power_stop : ret= %d\n", ret);
	}

	/* Power off */
	gpiod_set_value(ricoh5b_dev->ce_gpio, 0);

	/* Tuner VDD Off */
	ricoh5b_disable_regulator(ricoh5b_dev);

	msleep(20);

	video_unregister_device(ricoh5b_dev->video_device);
	video_device_release(ricoh5b_dev->video_device);
	v4l2_device_unregister(&ricoh5b_dev->v4l2_dev);
	mutex_destroy(&ricoh5b_dev->read_mutex);
	mutex_destroy(&ricoh5b_dev->station_mutex);
	mutex_destroy(&ricoh5b_dev->rds_mode_mutex);
	mutex_destroy(&ricoh5b_dev->repeat_check);

	kfree(ricoh5b_dev);
	ricoh5b_dev = NULL;

	#ifdef CONFIG_REGMON_DEBUG
	ricoh5b_regmon_del();
	#endif

	LOG_DBG_FUNC("End\n");

	return ret;
}

#ifdef CONFIG_PM_SLEEP
static int ricoh5b_suspend(struct device *dev)
{
	int ret = 0;
	struct i2c_client *client = to_i2c_client(dev);
	struct ricoh5b_dev *ricoh5b_dev;

	LOG_DBG_FUNC("Start\n");

	ricoh5b_i2c_client	= client;
	ricoh5b_dev = ricoh5b_device_info;

	if (tuner_open) {
		ricoh5b_disable_rds_irq(ricoh5b_dev);

		cancel_delayed_work(&ricoh5b_check_work);
		flush_workqueue(workqueue_check);

		cancel_delayed_work(&ricoh5b_dev->ricoh5b_rds_work);
		flush_workqueue(ricoh5b_dev->workqueue);

		ricoh5b_cancel_rds_rcv_stat();

		/* ricoh5b off Sequence */
		ret = ricoh5b_power_stop();
		if (ret < 0)
			LOG_ERR("RICOH5B_REG_ADR_RDS_SETTING_2 err = %d\n",
				ret);
	}

	/* Power off */
	gpiod_set_value(ricoh5b_dev->ce_gpio, 0);

	/* Tuner VDD Off */
	ricoh5b_disable_regulator(ricoh5b_dev);

	LOG_DBG_FUNC("End\n");

	return ret;
}

static int ricoh5b_resume(struct device *dev)
{
	int ret = 0;
	struct i2c_client *client = to_i2c_client(dev);
	struct ricoh5b_dev *ricoh5b_dev;

	LOG_DBG_FUNC("Start\n");

	if (tuner_open) {
		ricoh5b_i2c_client = client;
		ricoh5b_dev = ricoh5b_device_info;

		/* Tuner VDD On */
		ricoh5b_enable_regulator(ricoh5b_dev);
		msleep(20);

		/* Tuner Power On */
		gpiod_set_value(ricoh5b_dev->ce_gpio, 1);
		msleep(20);

		mutex_lock(&ricoh5b_dev->read_mutex);
		ret = ricoh5b_initialize_setting();
		if (ret < 0) {
			LOG_ERR("ricoh5b_initialize_setting err = %d\n", ret);
			goto send_err;
		}

		ret = ricoh5b_audio_output_mute_on();
		if (ret < 0) {
			LOG_ERR("ricoh5b_audio_output_mute_on err = %d\n", ret);
			goto send_err;
		}

		ret = ricoh5b_send_audio_output_setting(
			RICOH5B_SELECT_AUDIO_OUTPUT);
		if (ret < 0) {
			LOG_ERR("ricoh5b_send_audio_output_setting err = %d\n",
				ret);
			goto send_err;
		}

		ret = ricoh5b_fm_send_set_band(ricoh5b_dev->tuned_freq / 10,
						gain_value);
		if (ret < 0) {
			LOG_ERR("ricoh5b_fm_send_set_band err = %d\n", ret);
			goto send_err;
		}

		if (ricoh5b_dev->audio_mode == V4L2_TUNER_MODE_STEREO) {
			ret = ricoh5b_cmd_write_bit(RICOH5B_REG_ADR_MONO,
				RICOH5B_REG_DAT_MONO_OFF,
				RICOH5B_REG_BIT_MONO);
		} else if (ricoh5b_dev->audio_mode == V4L2_TUNER_MODE_MONO) {
			ret = ricoh5b_cmd_write_bit(RICOH5B_REG_ADR_MONO,
				RICOH5B_REG_DAT_MONO_ON,
				RICOH5B_REG_BIT_MONO);
		}

		if (ret < 0) {
			LOG_ERR("RICOH5B_REG_ADR_MONO err = %d\n", ret);
			goto send_err;
		}

		ret = ricoh5b_threshold_set(ricoh5b_dev,
					ricoh5b_dev->sense_mode);
		if (ret < 0) {
			LOG_ERR("ricoh5b_threshold_set err = %d\n", ret);
			goto send_err;
		}

		if (ricoh5b_dev->rds_mode == RICOH5B_RDS_MODE_ON) {
			ret = ricoh5b_rds_set_status(ricoh5b_dev,
				RICOH5B_RDS_MODE_ON);
		} else if (ricoh5b_dev->rds_mode == RICOH5B_RDS_MODE_OFF) {
			ret = ricoh5b_rds_set_status(ricoh5b_dev,
				RICOH5B_RDS_MODE_OFF);
		}
		if (ret < 0) {
			LOG_ERR("ricoh5b RDS set err = %d\n", ret);
			goto send_err;
		}

		ret = ricoh5b_send_set_de_emphasis(ricoh5b_dev);
		if (ret < 0) {
			LOG_ERR("ricoh5b_send_set_de_emphasis err = %d\n", ret);
			goto send_err;
		}

		if (ricoh5b_dev->mute_state == RICOH5B_MUTE_OFF) {
			ret = ricoh5b_audio_output_mute_off();
			if (ret < 0) {
				LOG_ERR("ricoh5b_audio_output_mute_off"
					" err = %d\n", ret);
				goto send_err;
			} else {
				ricoh5b_dev->mute_state = RICOH5B_MUTE_OFF;
			}
		}

		mutex_unlock(&ricoh5b_dev->read_mutex);
	}

	LOG_DBG_FUNC("End\n");

	return 0;

send_err:
	mutex_unlock(&ricoh5b_dev->read_mutex);
	return ret;
}

static SIMPLE_DEV_PM_OPS(ricoh5b_pm_ops,
			 ricoh5b_suspend, ricoh5b_resume);
#endif	/* CONFIG_PM_SLEEP */

static void ricoh5b_shutdown(struct i2c_client *client)
{
	struct ricoh5b_dev *ricoh5b_dev;
	int ret = 0;

	LOG_DBG_FUNC("Start\n");

	ricoh5b_dev = i2c_get_clientdata(client);

	devm_free_irq(&client->dev, client->irq, ricoh5b_dev);

	cancel_delayed_work(&ricoh5b_check_work);
	flush_workqueue(workqueue_check);
	destroy_workqueue(workqueue_check);

	cancel_delayed_work(&ricoh5b_dev->ricoh5b_rds_work);
	flush_workqueue(ricoh5b_dev->workqueue);
	destroy_workqueue(ricoh5b_dev->workqueue);

	ricoh5b_destroy_rds_rcv_stat();

	if (tuner_open) {
		/* ricoh5b off Sequence */
		ret = ricoh5b_power_stop();
		if (ret)
			LOG_ERR("ricoh5b_power_stop : ret= %d\n", ret);
	}

	/* Power off */
	gpiod_set_value(ricoh5b_dev->ce_gpio, 0);

	/* Tuner VDD Off */
	ricoh5b_disable_regulator(ricoh5b_dev);

	LOG_DBG_FUNC("End\n");

}

static int ricoh5b_init_rds_rcv_stat(void)
{
	int ret = 0;
	struct ricoh5b_dev *ricoh5b_dev;

	ricoh5b_dev = ricoh5b_device_info;
	ricoh5b_dev->rds_rcv_stat_queue
		= create_singlethread_workqueue("ricoh5b_rds_rcv_stat");
	if (!ricoh5b_dev->rds_rcv_stat_queue) {
		LOG_ERR("create_singlethread_workqueue()"
			" rds_rcv_stat_queue error\n");
		ret = -1;
	} else {
		INIT_DELAYED_WORK(&ricoh5b_dev->rds_rcv_stat_work,
						ricoh5b_timeout_rds_rcv_stat);
		mutex_init(&ricoh5b_dev->rds_rcv_stat_mutex);
		ricoh5b_dev->rds_rcv_stat_lock_sts = LOCK_STS_IDLE;
		ricoh5b_dev->rds_rcv_stat = RDS_NOT_RECEIVED;
	}
	return ret;
}

static void ricoh5b_destroy_rds_rcv_stat(void)
{
	struct ricoh5b_dev *ricoh5b_dev;

	LOG_DBG_FUNC("start\n");
	ricoh5b_dev = ricoh5b_device_info;
	mutex_lock(&ricoh5b_dev->rds_rcv_stat_mutex);
	ricoh5b_dev->rds_rcv_stat_lock_sts = LOCK_STS_DEL;
	mutex_unlock(&ricoh5b_dev->rds_rcv_stat_mutex);

	cancel_delayed_work_sync(&ricoh5b_dev->rds_rcv_stat_work);
	destroy_workqueue(ricoh5b_dev->rds_rcv_stat_queue);

	mutex_lock(&ricoh5b_dev->rds_rcv_stat_mutex);
	ricoh5b_dev->rds_rcv_stat_lock_sts = LOCK_STS_IDLE;
	mutex_unlock(&ricoh5b_dev->rds_rcv_stat_mutex);
	LOG_DBG_FUNC("end\n");
}

static void ricoh5b_cancel_rds_rcv_stat(void)
{
	struct ricoh5b_dev *ricoh5b_dev;

	LOG_DBG_FUNC("start\n");
	ricoh5b_dev = ricoh5b_device_info;
	mutex_lock(&ricoh5b_dev->rds_rcv_stat_mutex);
	if (ricoh5b_dev->rds_rcv_stat_lock_sts != LOCK_STS_IDLE) {
		LOG_DBG_FUNC("mutex_unlock skip\n");
		mutex_unlock(&ricoh5b_dev->rds_rcv_stat_mutex);
		return;
	}
	ricoh5b_dev->rds_rcv_stat_lock_sts = LOCK_STS_DEL;
	mutex_unlock(&ricoh5b_dev->rds_rcv_stat_mutex);

	cancel_delayed_work_sync(&ricoh5b_dev->rds_rcv_stat_work);

	mutex_lock(&ricoh5b_dev->rds_rcv_stat_mutex);
	if (ricoh5b_dev->rds_rcv_stat != RDS_NOT_RECEIVED)
		LOG_DBG("rds_rcv_stat = %d\n", RDS_NOT_RECEIVED);
	ricoh5b_dev->rds_rcv_stat = RDS_NOT_RECEIVED;
	ricoh5b_dev->rds_rcv_stat_lock_sts = LOCK_STS_IDLE;
	mutex_unlock(&ricoh5b_dev->rds_rcv_stat_mutex);
	LOG_DBG_FUNC("end\n");
}

static void ricoh5b_set_rds_rcv_stat(void)
{
	struct ricoh5b_dev *ricoh5b_dev;

	ricoh5b_dev = ricoh5b_device_info;
	mutex_lock(&ricoh5b_dev->rds_rcv_stat_mutex);
	if (ricoh5b_dev->rds_rcv_stat_lock_sts != LOCK_STS_IDLE) {
		LOG_DBG_FUNC("mutex_unlock skip\n");
		mutex_unlock(&ricoh5b_dev->rds_rcv_stat_mutex);
		return;
	}
	ricoh5b_dev->rds_rcv_stat_lock_sts = LOCK_STS_ADD;
	mutex_unlock(&ricoh5b_dev->rds_rcv_stat_mutex);

	cancel_delayed_work_sync(&ricoh5b_dev->rds_rcv_stat_work);

	mutex_lock(&ricoh5b_dev->rds_rcv_stat_mutex);
	if (ricoh5b_dev->rds_rcv_stat == RDS_NOT_RECEIVED)
		LOG_DBG("rds_rcv_stat = %d\n", RDS_RECEIVED);
	ricoh5b_dev->rds_rcv_stat = RDS_RECEIVED;
	queue_delayed_work(
		ricoh5b_dev->rds_rcv_stat_queue,
		&ricoh5b_dev->rds_rcv_stat_work,
		RDS_RCV_STAT_TIME_OUT
	);
	ricoh5b_dev->rds_rcv_stat_lock_sts = LOCK_STS_IDLE;
	mutex_unlock(&ricoh5b_dev->rds_rcv_stat_mutex);
}

static void ricoh5b_timeout_rds_rcv_stat(struct work_struct *work)
{
	struct ricoh5b_dev *ricoh5b_dev;

	LOG_DBG_FUNC("start\n");
	ricoh5b_dev = ricoh5b_device_info;
	mutex_lock(&ricoh5b_dev->rds_rcv_stat_mutex);
	if (ricoh5b_dev->rds_rcv_stat_lock_sts != LOCK_STS_IDLE) {
		LOG_DBG_FUNC("mutex_unlock skip\n");
		mutex_unlock(&ricoh5b_dev->rds_rcv_stat_mutex);
		return;
	}
	if (ricoh5b_dev->rds_rcv_stat != RDS_NOT_RECEIVED)
		LOG_DBG("rds_rcv_stat = %d\n", RDS_NOT_RECEIVED);
	ricoh5b_dev->rds_rcv_stat = RDS_NOT_RECEIVED;
	mutex_unlock(&ricoh5b_dev->rds_rcv_stat_mutex);
	LOG_DBG_FUNC("end\n");
}

static const struct i2c_device_id ricoh5b_id[] = {
	{"ricoh5b", 0},
	{}
};

MODULE_DEVICE_TABLE(i2c, ricoh5b_id);

static struct i2c_driver ricoh5b_driver = {
	.id_table = ricoh5b_id,
	.probe	  = ricoh5b_probe,
	.remove	  = ricoh5b_remove,
	.shutdown = ricoh5b_shutdown,
	.driver	  = {
		.owner = THIS_MODULE,
		.name = "ricoh5b",
#ifdef CONFIG_PM_SLEEP
		.pm = &ricoh5b_pm_ops,
#endif	/* CONFIG_PM_SLEEP */
	},
};

/*
 * module load/unload record keeping
 */

static int __init ricoh5b_dev_init(void)
{
	LOG_DBG("Loading ricoh5b driver\n");

	return i2c_add_driver(&ricoh5b_driver);
}

module_init(ricoh5b_dev_init);

static void __exit ricoh5b_dev_exit(void)
{
	LOG_DBG("Unloading ricoh5b driver -S\n");
	i2c_del_driver(&ricoh5b_driver);
}

module_exit(ricoh5b_dev_exit);

MODULE_VERSION("1.0.4");
MODULE_AUTHOR("Sony Corporation");
MODULE_DESCRIPTION("RICOH Tuner ricoh5b driver");
MODULE_LICENSE("GPL");
