/* SPDX-License-Identifier: GPL-2.0 */
/*
 * Driver include for the RICOH5B Tuner chip.
 *
 * Copyright 2015, 2016, 2021 Sony Corporation
 *
 */

#ifndef _RICOH5B_MAIN_H_
#define _RICOH5B_MAIN_H_

#include<media/drv-intf/ricoh5b.h>
#include "st_cfg.h"

/* Device Slave Address */
#define RICOH5B_SERIAL_SLAVE_ADDRESS    0x64

/* Message Packet Number */
#define RICOH5B_MSG_READ_PACKET_NUM     2 /* write -> read */
#define RICOH5B_MSG_WRITE_PACKET_NUM    1 /* write */

/* Write Data Index */
#define WRITE_INDEX_ADDRESS             0
#define WRITE_INDEX_DATA                1

/* Transmit Data Length */
#define RICOH5B_REG_ADDRESS_LENGTH      1
#define RICOH5B_DATA_WRITE_LENGTH_MAX   4
#define RICOH5B_DATA_READ_LENGTH_MAX    4

#define REG_ADDRESS_LENGTH              RICOH5B_REG_ADDRESS_LENGTH
#define DATA_1BYTE_LENGTH               1
#define DATA_WRITE_LENGTH_MAX           RICOH5B_DATA_WRITE_LENGTH_MAX
#define DATA_READ_LENGTH_MAX            RICOH5B_DATA_READ_LENGTH_MAX

/* Read Buffer Size */
#define READ_BUF_SIZE                   DATA_READ_LENGTH_MAX

/* Write Buffer Size
 * Register Address + Data
 */
#define WRITE_BUF_SIZE	\
	(REG_ADDRESS_LENGTH + DATA_WRITE_LENGTH_MAX)

/* Write Volume Buffer Size */
#define RICOH5B_VOLUME_DATA_SIZE   3

/***** Registers *****/
/* Definition of Non Mask for Write Register */
#define RICOH5B_REG_NON_MASK                 0xFF

/* address 00h */
#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_BAND                 0x00
#define RICOH5B_REG_BIT_BAND                 0x10
#define RICOH5B_REG_DAT_BAND_FM              0x00
#define RICOH5B_REG_DAT_BAND_AM              0x10
#endif //RICOH5B_USE_TUNER

#define RICOH5B_REG_ADR_MONO                 0x00
#define RICOH5B_REG_BIT_MONO                 0x04
#define RICOH5B_REG_DAT_MONO_ON              0x04
#define RICOH5B_REG_DAT_MONO_OFF             0x00

#define RICOH5B_REG_ADR_MUTE                 0x00
#define RICOH5B_REG_BIT_MUTE                 0x02
#define RICOH5B_REG_DAT_MUTE_ON              0x02
#define RICOH5B_REG_DAT_MUTE_OFF             0x00

#define RICOH5B_REG_ADR_PE                   0x00
#define RICOH5B_REG_BIT_PE                   0x01
#define RICOH5B_REG_DAT_PE_ON                0x01
#define RICOH5B_REG_DAT_PE_OFF               0x00
#define RICOH5B_REG_LENGTH_PE                   1

/* address 01h */
#define RICOH5B_REG_ADR_RST_RFAGC            0x01
#define RICOH5B_REG_BIT_RST_RFAGC            0x40
#define RICOH5B_REG_DAT_RST_RFAGC_ENABLE     0x40
#define RICOH5B_REG_DAT_RST_RFAGC_DISABLE    0x00

#define RICOH5B_REG_ADR_LOSCI                0x01
#define RICOH5B_REG_BIT_LOSCI                0x10
#define RICOH5B_REG_DAT_LOSCI_UPPER          0x10
#define RICOH5B_REG_DAT_LOSCI_LOWER          0x00

#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_STATION_EN           0x01
#define RICOH5B_REG_BIT_STATION_EN           0x08
#define RICOH5B_REG_DAT_STATION_EN_ENABLE    0x08
#define RICOH5B_REG_DAT_STATION_EN_DISABLE   0x00
#endif //RICOH5B_USE_TUNER

#define RICOH5B_REG_ADR_XSEL                 0x01
#define RICOH5B_REG_BIT_XSEL                 0x01

/* address 02h */
#define RICOH5B_REG_ADR_PSY_L                0x02
#define RICOH5B_REG_LENGTH_PSY                  2

/* address 03h */
#define RICOH5B_REG_ADR_PSY_H                0x03

/* address 04h */
#define RICOH5B_REG_ADR_PSY_F                0x04
#define RICOH5B_REG_BIT_PSY_F                0x0F
#define RICOH5B_REG_DAT_PSY_F_0K             0x00
#define RICOH5B_REG_DAT_PSY_F_PLUS_10K       0x01
#define RICOH5B_REG_DAT_PSY_F_PLUS_20K       0x02
#define RICOH5B_REG_DAT_PSY_F_PLUS_30K       0x03
#define RICOH5B_REG_DAT_PSY_F_PLUS_40K       0x04
#define RICOH5B_REG_DAT_PSY_F_PLUS_50K       0x05
#define RICOH5B_REG_DAT_PSY_F_MINUS_10K      0x09
#define RICOH5B_REG_DAT_PSY_F_MINUS_20K      0x0A
#define RICOH5B_REG_DAT_PSY_F_MINUS_30K      0x0B
#define RICOH5B_REG_DAT_PSY_F_MINUS_40K      0x0C

#define RICOH5B_REG_BIT_PSY_F_SIGN           0x08
#define RICOH5B_REG_DAT_PSY_F_PLUS           0x00
#if RICOH5B_USE_TUNER
#define RICOH5B_REG_DAT_PSY_F_MINUS          0x08
#else  //RICOH5B_USE_TUNER
#define RICOH5B_REG_DAT_PSY_F_MINUS          0x10
#endif //RICOH5B_USE_TUNER
#define RICOH5B_REG_DAT_PSY_F_CONVERT_MINUS(X) \
	(X | RICOH5B_REG_DAT_PSY_F_MINUS)  /* X(0-5)[kHz] -X[kHz] change */

/* address 05h */
#define RICOH5B_REG_ADR_FO_SAT_POL           0x05
#define RICOH5B_REG_BIT_FO_SAT_POL           0x80
#define RICOH5B_REG_DAT_FO_SAT_POL_MINUS     0x80
#define RICOH5B_REG_DAT_FO_SAT_POL_PLUS      0x00

#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_STATION_RDY          0x05
#define RICOH5B_REG_BIT_STATION_RDY          0x40
#define RICOH5B_REG_DAT_STATION_RDY_OK       0x40
#define RICOH5B_REG_DAT_STATION_RDY_NG       0x00
#endif //RICOH5B_USE_TUNER

#define RICOH5B_REG_ADR_BMT_ON               0x05
#define RICOH5B_REG_BIT_BMT_ON               0x10

#define RICOH5B_REG_ADR_FO_SAT               0x05
#define RICOH5B_REG_BIT_FO_SAT               0x04
#define RICOH5B_REG_DAT_FO_SAT_OUTSIDE       0x04
#define RICOH5B_REG_DAT_FO_SAT_INSIDE        0x00
#define RICOH5B_REG_LENGTH_FO_SAT               1

#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_STATION              0x05
#define RICOH5B_REG_BIT_STATION              0x02
#define RICOH5B_REG_DAT_STATION_DETECTED     0x02
#define RICOH5B_REG_DAT_STATION_UNDETECT     0x00
#define RICOH5B_REG_LENGTH_STATION              1
#endif //RICOH5B_USE_TUNER

#define RICOH5B_REG_ADR_STEREO               0x05
#define RICOH5B_REG_BIT_STEREO               0x01
#define RICOH5B_REG_DAT_STEREO_ON            0x01
#define RICOH5B_REG_DAT_STEREO_OFF           0x00
#define RICOH5B_REG_LENGTH_STEREO               1

/* address 06h */
#define RICOH5B_REG_ADR_LOCK57K              0x06
#define RICOH5B_REG_BIT_LOCK57K              0x80

#define RICOH5B_REG_ADR_RSSI_RD              0x06
#define RICOH5B_REG_BIT_RSSI_RD              0x7F
#define RICOH5B_REG_DAT_RSSI_RD_MAX          0x7F
#define RICOH5B_REG_LENGTH_RSSI_RD              1

/* address 07h */
#define RICOH5B_REG_ADR_RDS_STATUS           0x07
#define RICOH5B_REG_LENGTH_RDS_DATA             3
				/* RDS data register (reg 07h, 08h, 09h) */

#define RICOH5B_REG_BIT_RDS_NOERR            0x80
#define RICOH5B_REG_DAT_RDS_ERROR            0x00
#define RICOH5B_REG_DAT_RDS_NO_ERROR         0x80

#define RICOH5B_REG_BIT_RDS_UNC              0x40
#define RICOH5B_REG_DAT_RDS_ERROR_CORRECTED    0x00
#define RICOH5B_REG_DAT_RDS_ERROR_UNCORRECTED  0x40

#define RICOH5B_REG_BIT_RDS_BLOCK            0x38
#define RICOH5B_REG_DAT_RDS_BLOCK_A          0x00
#define RICOH5B_REG_DAT_RDS_BLOCK_B          0x08
#define RICOH5B_REG_DAT_RDS_BLOCK_C          0x10
#define RICOH5B_REG_DAT_RDS_BLOCK_D          0x18
#define RICOH5B_REG_DAT_RDS_BLOCK_E          0x20
#define RICOH5B_REG_DAT_RDS_BLOCK_CD         0x28

#define RICOH5B_REG_BIT_RDS_INT              0x04
#define RICOH5B_REG_DAT_RDS_INT_VALID        0x00
#define RICOH5B_REG_DAT_RDS_INT_INVALID      0x04

#define RICOH5B_REG_BIT_RDS_OWF              0x02
#define RICOH5B_REG_DAT_RDS_OWF_CONTINUOUS   0x00
#define RICOH5B_REG_DAT_RDS_OWF_DISCONTINUOUS  0x02

#define RICOH5B_REG_BIT_RDS_SYNC             0x01
#define RICOH5B_REG_DAT_RDS_ASYNC            0x00
#define RICOH5B_REG_DAT_RDS_SYNC             0x01

/* address 08h */
#define RICOH5B_REG_ADR_RDS_DATA_L           0x08

/* address 09h */
#define RICOH5B_REG_ADR_RDS_DATA_H           0x09

/* address 0Ah */
#define RICOH5B_REG_ADR_RDS_OFFSET_L         0x0A

/* address 0Bh */
#define RICOH5B_REG_ADR_RDS_OFFSET_H         0x0B

/* address 0Dh */
#define RICOH5B_REG_ADR_DSP_RST              0x0D
#define RICOH5B_REG_BIT_DSP_RST              0x80
#define RICOH5B_REG_DAT_DSP_RST_ENABLE       0x80
#define RICOH5B_REG_DAT_DSP_RST_DISABLE      0x00

#define RICOH5B_REG_ADR_DEMP                 0x0D
#define RICOH5B_REG_BIT_DEMP                 0x40
#define RICOH5B_REG_DAT_DEMP_75US            0x00
#define RICOH5B_REG_DAT_DEMP_50US            0x40

#define RICOH5B_REG_ADR_IIS_LRP              0x0D
#define RICOH5B_REG_BIT_IIS_LRP              0x20
#define RICOH5B_REG_DAT_IIS_LRP_NORMAL       0x00
#define RICOH5B_REG_DAT_IIS_LRP_REVERSE      0x20

#define RICOH5B_REG_ADR_IIS_SEL              0x0D
#define RICOH5B_REG_BIT_IIS_SEL              0x18
#define RICOH5B_REG_DAT_IIS_SEL_OUTPUT_OFF   0x00
#define RICOH5B_REG_DAT_IIS_SEL_IIS_FORMAT   0x08
#define RICOH5B_REG_DAT_IIS_SEL_LEFT_ADJUST  0x10
#define RICOH5B_REG_DAT_IIS_SEL_RIGHT_ADJUST 0x18

#define RICOH5B_REG_ADR_IIS_32_64            0x0D
#define RICOH5B_REG_BIT_IIS_32_64            0x04
#define RICOH5B_REG_DAT_IIS_32FS             0x00
#define RICOH5B_REG_DAT_IIS_64FS             0x04

#define RICOH5B_REG_ADR_IIS_EN               0x0D
#define RICOH5B_REG_BIT_IIS_EN               0x03
#define RICOH5B_REG_DAT_IIS_EN_DAC_OUT       0x00
#define RICOH5B_REG_DAT_IIS_EN_IIS_OUT       0x01
#define RICOH5B_REG_DAT_IIS_EN_SPDIF_IN      0x02
#define RICOH5B_REG_DAT_IIS_EN_AUDIO_OFF     0x03

/* address 0Eh */
#define RICOH5B_REG_ADR_FM_OV                0x0E
#define RICOH5B_REG_BIT_FM_OV                0x80
#define RICOH5B_REG_DAT_FM_OV_NORMAL         0x00
#define RICOH5B_REG_DAT_FM_OV_OVER           0x80

#define RICOH5B_REG_ADR_BW                   0x0E
#define RICOH5B_REG_BIT_BW                   0x60
#define RICOH5B_REG_DAT_BW_FM_75K            0x00
#define RICOH5B_REG_DAT_BW_FM_105K           0x20
#define RICOH5B_REG_DAT_BW_FM_130K           0x40
#define RICOH5B_REG_DAT_BW_FM_160K           0x60
#define RICOH5B_REG_DAT_BW_AM_2_4K           0x00
#define RICOH5B_REG_DAT_BW_AM_3_0K           0x40
#define RICOH5B_REG_DAT_BW_AM_4_0K           0x60

/****************************************/
/* FM_OV(0Eh:[7]) BW (0Eh[6:5]) setting */
/****************************************/
enum RICOH5B_FM_BW {
	RICOH5B_FM_BW_CF75K   = RICOH5B_REG_DAT_FM_OV_NORMAL
				| RICOH5B_REG_DAT_BW_FM_75K,
	RICOH5B_FM_BW_OM75K   = RICOH5B_REG_DAT_FM_OV_OVER
				| RICOH5B_REG_DAT_BW_FM_75K,
	RICOH5B_FM_BW_CF105K  = RICOH5B_REG_DAT_FM_OV_NORMAL
				| RICOH5B_REG_DAT_BW_FM_105K,
	RICOH5B_FM_BW_OM105K  = RICOH5B_REG_DAT_FM_OV_OVER
				| RICOH5B_REG_DAT_BW_FM_105K,
	RICOH5B_FM_BW_CF130K  = RICOH5B_REG_DAT_FM_OV_NORMAL
				| RICOH5B_REG_DAT_BW_FM_130K,
	RICOH5B_FM_BW_OM130K  = RICOH5B_REG_DAT_FM_OV_OVER
				| RICOH5B_REG_DAT_BW_FM_130K,
	RICOH5B_FM_BW_CF160K  = RICOH5B_REG_DAT_FM_OV_NORMAL
				| RICOH5B_REG_DAT_BW_FM_160K,
	RICOH5B_FM_BW_OM160K  = RICOH5B_REG_DAT_FM_OV_OVER
				| RICOH5B_REG_DAT_BW_FM_160K
};

#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_LPFS_SEL             0x0E
#define RICOH5B_REG_BIT_LPFS_SEL             0x0C
#define RICOH5B_REG_DAT_LPFS_2_0K            0x00
#define RICOH5B_REG_DAT_LPFS_1_5K            0x04
#define RICOH5B_REG_DAT_LPFS_1_0K            0x08
#define RICOH5B_REG_DAT_LPFS_THROUGH         0x0C
#endif //RICOH5B_USE_TUNER

#define RICOH5B_REG_ADR_HPFS_SEL             0x0E
#define RICOH5B_REG_BIT_HPFS_SEL             0x03
#define RICOH5B_REG_DAT_HPFS_FM_THROUGH      0x00
#define RICOH5B_REG_DAT_HPFS_FM_50           0x01
#define RICOH5B_REG_DAT_HPFS_FM_100          0x02
#define RICOH5B_REG_DAT_HPFS_FM_150          0x03
#define RICOH5B_REG_DAT_HPFS_AM_50           0x00
#define RICOH5B_REG_DAT_HPFS_AM_100          0x01
#define RICOH5B_REG_DAT_HPFS_AM_150          0x02

#define RICOH5B_REG_DAT_BW_DEFAULT           RICOH5B_REG_DAT_BW_FM_105K

/* address 10h */
#define RICOH5B_REG_ADR_CIO_SYN              0x10
#define RICOH5B_REG_DAT_CIO_SYN_2_25         0x40
#define RICOH5B_REG_DAT_CIO_SYN_4_5          0x50
#define RICOH5B_REG_DAT_CIO_SYN_9_0          0x60
#define RICOH5B_REG_DAT_CIO_SYN_18_0         0x70

#define RICOH5B_REG_ADR_SYN_RDF              0x10
#define RICOH5B_REG_DAT_SYN_RDF_2K           0x08
#define RICOH5B_REG_DAT_SYN_RDF_35K          0x09
#define RICOH5B_REG_DAT_SYN_RDF_100K         0x0A

/* address 11h */
#define RICOH5B_REG_ADR_SMT                  0x11
#define RICOH5B_REG_DAT_SMT_OFF              0x00
#define RICOH5B_REG_DAT_SMT_8DBU             0x01

/* address 12h */
#define RICOH5B_REG_ADR_HCC                  0x12
#define RICOH5B_REG_BIT_HCC                  0xF0
#define RICOH5B_REG_DAT_HCC_OFF              0x00
#define RICOH5B_REG_DAT_HCC_18DBU            0x40
#define RICOH5B_REG_DAT_HCC_24DBU            0x70
#define RICOH5B_REG_DAT_HCC_28DBU            0x90
#define RICOH5B_REG_DAT_HCC_32DBU            0xB0

#define RICOH5B_REG_ADR_SNC                  0x12
#define RICOH5B_REG_BIT_SNC                  0x0F
#define RICOH5B_REG_DAT_SNC_OFF              0x00
#define RICOH5B_REG_DAT_SNC_38DBU            0x04
#define RICOH5B_REG_DAT_SNC_41DBU            0x05
#define RICOH5B_REG_DAT_SNC_44DBU            0x06
#define RICOH5B_REG_DAT_SNC_47DBU            0x07

/* address 13h */
#define RICOH5B_REG_ADR_RDS_SETTING_1        0x13
#define RICOH5B_REG_RDS_ENABLE               0x97
#define RICOH5B_REG_RDS_READ_3OR5            0x80
#define RICOH5B_REG_RDS_2OR5                 0x10
#define RICOH5B_REG_RBDS_EN                  0x04
#define RICOH5B_REG_RDS_DIS_EC               0x02
#define RICOH5B_REG_RDS_SYNC_GO              0x01
#define RICOH5B_REG_RDS_ON_VALUE             0xD3
#define RICOH5B_REG_RDS_OFF_VALUE            0x52
#define RICOH5B_REG_DAT_RDS_SETTING_1_DISABLE 0x10
#define RICOH5B_REG_DAT_RDS_SETTING_1_RDS    0x11
#define RICOH5B_REG_DAT_RDS_SETTING_1_RBDS   0x15
#define RICOH5B_REG_LENGTH_RDS_SETTING          2
				/* RDS setting register (reg 13h, 14h) */

/* address 14h */
#define RICOH5B_REG_ADR_RDS_SETTING_2        0x14
#define RICOH5B_REG_RDS_ERR_VAL              0xF0
#define RICOH5B_REG_RDS_SYNC_VAL             0x0F
#define RICOH5B_REG_DAT_RDS_SETTING_2        0x50

/* address 15h */
#define RICOH5B_REG_ADR_FO_SAT_SEL           0x15
#define RICOH5B_REG_BIT_FO_SAT_SEL           0xC0
#define RICOH5B_REG_DAT_FO_SAT_10K           0x00
#define RICOH5B_REG_DAT_FO_SAT_15K           0x40
#define RICOH5B_REG_DAT_FO_SAT_20K           0x80
#define RICOH5B_REG_DAT_FO_SAT_25K           0xC0

#define RICOH5B_REG_ADR_DCO_AVESEL           0x15
#define RICOH5B_REG_BIT_DCO_AVESEL           0x07
#define RICOH5B_REG_DAT_DCO_AVESEL_IDLE      0x00
#define RICOH5B_REG_DAT_DCO_AVESEL_AM_SEEK   0x02
#define RICOH5B_REG_DAT_DCO_AVESEL_FM_SEEK   0x03

/* address 18h */
#define RICOH5B_REG_ADR_RSYS                 0x18
#define RICOH5B_REG_BIT_RSYS                 0x30
#define RICOH5B_REG_DAT_RSYS_FM800K_AMAUTO   0x00
#define RICOH5B_REG_DAT_RSYS_FM400K_AMAUTO   0x10
#define RICOH5B_REG_DAT_RSYS_FM800K_AM800K   0x20
#define RICOH5B_REG_DAT_RSYS_FM400K_AM400K   0x30

#define RICOH5B_REG_ADR_XTAL                 0x18
#define RICOH5B_REG_DAT_XTAL_32K             0x00
#define RICOH5B_REG_DAT_XTAL_12M             0x02

/* address 20h */
#define RICOH5B_REG_ADR_SD_K_L               0x20

/* address 21h */
#define RICOH5B_REG_ADR_SD_K_H               0x21
#define RICOH5B_REG_DAT_SD_K_H_DEFAULT       0xFF

/* address 22h */
#define RICOH5B_REG_ADR_SD_TH_LO             0x22
#define RICOH5B_REG_DAT_SD_TH_LO             0x05

/* address 23h */
#define RICOH5B_REG_ADR_SD_TH_HI             0x23
#define RICOH5B_REG_DAT_SD_TH_HI             0x05

/* address 25h */
#define RICOH5B_REG_ADR_VTJHA_CK             0x25
#define RICOH5B_REG_BIT_VTJHA_CK             0xC0
#define RICOH5B_REG_DAT_VTJHA_CK_0_96        0x40
#define RICOH5B_REG_DAT_VTJHA_CK_0_92        0x00

#define RICOH5B_REG_ADR_VTJLA_CK             0x25
#define RICOH5B_REG_BIT_VTJLA_CK             0x30
#define RICOH5B_REG_DAT_VTJLA_CK_0_28        0x30
#define RICOH5B_REG_DAT_VTJLA_CK_0_24        0x20

#define RICOH5B_REG_ADR_VTJHM_CK             0x25
#define RICOH5B_REG_BIT_VTJHM_CK             0x0C
#define RICOH5B_REG_DAT_VTJHM_CK_0_92        0x00

#define RICOH5B_REG_ADR_VTJLM_CK             0x25
#define RICOH5B_REG_BIT_VTJLM_CK             0x03
#define RICOH5B_REG_DAT_VTJLM_CK_0_28        0x03

/* address 26h */
#define RICOH5B_REG_ADR_SW0_CK               0x26
#define RICOH5B_REG_DAT_SW0_CK_32K           0x30
#define RICOH5B_REG_DAT_SW0_CK_12M           0x20

#define RICOH5B_REG_ADR_RSW_CK               0x26
#define RICOH5B_REG_DAT_RSW_CK_400K          0x03
#define RICOH5B_REG_DAT_RSW_CK_760K          0x05
#define RICOH5B_REG_DAT_RSW_CK_5000K         0x07

/* address 27h */
#define RICOH5B_REG_ADR_CK_VCO               0x27
#define RICOH5B_REG_DAT_CK_VCO_32K           0xFF
#define RICOH5B_REG_DAT_CK_VCO_12M           0x77

/* address 28h */
#define RICOH5B_REG_ADR_CKBRD                0x28
#define RICOH5B_REG_BIT_CKBRD                0x80
#define RICOH5B_REG_DAT_CKBRD                0x80

#define RICOH5B_REG_ADR_CKBRS                0x28
#define RICOH5B_REG_BIT_CKBRS                0x40
#define RICOH5B_REG_DAT_CKBRS                0x40

/* address 2Dh */
#define RICOH5B_REG_ADR_SYN_VTJH             0x2D
#define RICOH5B_REG_BIT_SYN_VTJH             0xC0
#define RICOH5B_REG_DAT_SYN_VTJH_0_92        0x00

#define RICOH5B_REG_ADR_SYN_VTJL             0x2D
#define RICOH5B_REG_BIT_SYN_VTJL             0x30
#define RICOH5B_REG_DAT_SYN_VTJL_0_20        0x10

/* address 2Eh */
#define RICOH5B_REG_ADR_LOC_T2               0x2E
#define RICOH5B_REG_BIT_LOC_T2               0x30
#define RICOH5B_REG_DAT_LOC_T2_100U          0x00
#define RICOH5B_REG_DAT_LOC_T2_220U          0x10
#define RICOH5B_REG_DAT_LOC_T2_480U          0x20
#define RICOH5B_REG_DAT_LOC_T2_980U          0x30

/* address 30h */
#define RICOH5B_REG_ADR_BLOCK1_PE            0x30
#define RICOH5B_REG_DAT_BLOCK1_PE_ALL_ON     0xFF
#define RICOH5B_REG_DAT_BLOCK1_POWER_OFF     0xEF
#define RICOH5B_REG_DAT_BLOCK1_AGC_SETTING   0xE7  /* PESY/PECLK->OFF */

#define RICOH5B_REG_ADR_PEDA                 0x30
#define RICOH5B_REG_BIT_PEDA                 0x02
#define RICOH5B_REG_DAT_PEDA_ON              0x02
#define RICOH5B_REG_DAT_PEDA_OFF             0x00

/* address 32h */
#define RICOH5B_REG_ADR_DCO_OFF              0x32
#define RICOH5B_REG_BIT_DCO_OFF              0x40
#define RICOH5B_REG_DAT_DCO_OFF              0x40
#define RICOH5B_REG_DAT_DCO_ON               0x00

#define RICOH5B_REG_ADR_INTO_SEL             0x32
#define RICOH5B_REG_BIT_INTO_SEL             0x04

#define RICOH5B_REG_ADR_TUNE_SEL             0x32
#define RICOH5B_REG_BIT_TUNE_SEL             0x02

#define RICOH5B_REG_ADR_IISD_SEL             0x32
#define RICOH5B_REG_BIT_IISD_SEL             0x01

/* address 35h */
#define RICOH5B_REG_ADR_T12_OEB              0x35
#define RICOH5B_REG_BIT_T12_OEB              0x80
#define RICOH5B_REG_DAT_T12_OEB_LOW_NOISE    0x80
#define RICOH5B_REG_DAT_T12_OEB_NORMAL       0x00

#define RICOH5B_REG_ADR_T12_PEN              0x35
#define RICOH5B_REG_BIT_T12_PEN              0x40
#define RICOH5B_REG_DAT_T12_PEN_ON           0x40
#define RICOH5B_REG_DAT_T12_PEN_OFF          0x00

#define RICOH5B_REG_ADR_T12_DACEN            0x35
#define RICOH5B_REG_BIT_T12_DACEN            0x20
#define RICOH5B_REG_DAT_T12_DACEN_ENABLE     0x20
#define RICOH5B_REG_DAT_T12_DACEN_DISABLE    0x00

#define RICOH5B_REG_ADR_IISD_OEB             0x35
#define RICOH5B_REG_BIT_IISD_OEB             0x08
#define RICOH5B_REG_DAT_IISD_OEB_LOW_NOISE   0x08
#define RICOH5B_REG_DAT_IISD_OEB_NORMAL      0x00

#define RICOH5B_REG_ADR_IISD_PEN             0x35
#define RICOH5B_REG_BIT_IISD_PEN             0x04
#define RICOH5B_REG_DAT_IISD_PEN_ON          0x04
#define RICOH5B_REG_DAT_IISD_PEN_OFF         0x00

/*****************/
/* audio setting */
/*****************/
#define RICOH5B_REG_ADR_AUDIO_SETTING        0x35
#define RICOH5B_REG_BIT_AUDIO_SETTING        0xEC
#define RICOH5B_REG_DAT_AUDIO_SETTING_ANALOG 0xAC
#define RICOH5B_REG_DAT_AUDIO_SETTING_DIGITAL 0x88

#define RICOH5B_REG_ADR_DEMP_CE              0x35
#define RICOH5B_REG_BIT_DEMP_CE              0x01
#define RICOH5B_REG_DAT_DEMP_CE_100_OR_60US  0x01
#define RICOH5B_REG_DAT_DEMP_CE_75_OR_50US   0x00

/* address 37h */
#define RICOH5B_REG_ADR_UL_MASK              0x37
#define RICOH5B_REG_BIT_UL_MASK              0x20
#define RICOH5B_REG_DAT_UL_MASK_OFF          0x20
#define RICOH5B_REG_DAT_UL_MASK_ON           0x00

/* address 39h */
#define RICOH5B_REG_ADR_CLK_WIN              0x39
#define RICOH5B_REG_DAT_CLK_WIN_32K          0x80
#define RICOH5B_REG_DAT_CLK_WIN_12M          0x04

/* address 3Dh */
#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_IISD_BUFEN           0x3D
#else  //RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_IISD_BUFEN           0xF5
#endif //RICOH5B_USE_TUNER
#define RICOH5B_REG_BIT_IISD_BUFEN           0x40
#define RICOH5B_REG_DAT_IISD_BUFEN_ENABLE    0x40
#define RICOH5B_REG_DAT_IISD_BUFEN_DISABLE   0x00

#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_BUF_SEL              0x3D
#else  //RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_BUF_SEL              0xF5
#endif //RICOH5B_USE_TUNER
#define RICOH5B_REG_BIT_BUF_SEL              0x20
#define RICOH5B_REG_DAT_BUF_SEL_BUF2         0x20
#define RICOH5B_REG_DAT_BUF_SEL_BUF1         0x00

#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_BI                   0x3D
#else  //RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_BI                   0xF5
#endif //RICOH5B_USE_TUNER
#define RICOH5B_REG_BIT_BI                   0x18
#define RICOH5B_REG_DAT_BI_2MA               0x00
#define RICOH5B_REG_DAT_BI_1_5MA             0x08
#define RICOH5B_REG_DAT_BI_1_3MA             0x10
#define RICOH5B_REG_DAT_BI_1MA               0x18

#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_DU                   0x3D
#else  //RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_DU                   0xF5
#endif //RICOH5B_USE_TUNER
#define RICOH5B_REG_BIT_DU                   0x07
#define RICOH5B_REG_DAT_DU_BYPASS            0x00
#define RICOH5B_REG_DAT_DU_0_5K              0x01
#define RICOH5B_REG_DAT_DU_1K                0x02
#define RICOH5B_REG_DAT_DU_1_5K              0x03
#define RICOH5B_REG_DAT_DU_2K                0x04

/****************/
/* IISD setting */
/****************/
#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_IISD_SETTING         0x3D
#else  //RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_IISD_SETTING         0xF5
#endif //RICOH5B_USE_TUNER
#define RICOH5B_REG_DAT_IISD_SETTING_ANALOG \
		RICOH5B_REG_DAT_IISD_BUFEN_DISABLE
#define RICOH5B_REG_DAT_IISD_SETTING_DIGITAL \
		(RICOH5B_REG_DAT_IISD_BUFEN_ENABLE   \
		| RICOH5B_REG_DAT_BI_1MA | RICOH5B_REG_DAT_DU_0_5K)

/* address 3Eh */
#define RICOH5B_REG_ADR_FMIN                 0x3E
#define RICOH5B_REG_BIT_FMIN                 0xF0
#define RICOH5B_REG_DAT_FMIN_MINUS_36DB      0x00
#define RICOH5B_REG_DAT_FMIN_MINUS_6DB       0x50
#define RICOH5B_REG_DAT_FMIN_24DB            0xA0

#define RICOH5B_REG_ADR_PGM                  0x3E
#define RICOH5B_REG_BIT_PGM                  0x0F
#define RICOH5B_REG_DAT_PGM_0DB              0x00
#define RICOH5B_REG_DAT_PGM_2DB              0x01
#define RICOH5B_REG_DAT_PGM_4DB              0x02
#define RICOH5B_REG_DAT_PGM_6DB              0x03
#define RICOH5B_REG_DAT_PGM_8DB              0x04
#define RICOH5B_REG_DAT_PGM_10DB             0x05
#define RICOH5B_REG_DAT_PGM_12DB             0x06
#define RICOH5B_REG_DAT_PGM_14DB             0x07
#define RICOH5B_REG_DAT_PGM_16DB             0x08
#define RICOH5B_REG_DAT_PGM_18DB             0x09
#define RICOH5B_REG_DAT_PGM_20DB             0x0A
#define RICOH5B_REG_DAT_PGM_22DB             0x0B
#define RICOH5B_REG_DAT_PGM_24DB             0x0C
#define RICOH5B_REG_LENGTH_PGM                  1

/* address 3Fh */
#define RICOH5B_REG_ADR_AGC_AUTO             0x3F
#define RICOH5B_REG_BIT_AGC_AUTO             0x80
#define RICOH5B_REG_DAT_AGC_AUTO_ON          0x80
#define RICOH5B_REG_DAT_AGC_AUTO_OFF         0x00

#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_AMIM                 0x3F
#define RICOH5B_REG_BIT_AMIM                 0x1F
#define RICOH5B_REG_DAT_AMIM_MINUS_42DB      0x00
#define RICOH5B_REG_DAT_AMIM_MINUS_39DB      0x01
#define RICOH5B_REG_DAT_AMIM_MINUS_36DB      0x02
#define RICOH5B_REG_DAT_AMIM_MINUS_33DB      0x03
#define RICOH5B_REG_DAT_AMIM_MINUS_30DB      0x04
#define RICOH5B_REG_DAT_AMIM_MINUS_27DB      0x05
#define RICOH5B_REG_DAT_AMIM_MINUS_24DB      0x06
#define RICOH5B_REG_DAT_AMIM_MINUS_21DB      0x07
#define RICOH5B_REG_DAT_AMIM_MINUS_18DB      0x08
#define RICOH5B_REG_DAT_AMIM_MINUS_15DB      0x09
#define RICOH5B_REG_DAT_AMIM_MINUS_12DB      0x0A
#define RICOH5B_REG_DAT_AMIM_MINUS_9DB       0x0B
#define RICOH5B_REG_DAT_AMIM_MINUS_6DB       0x0C
#define RICOH5B_REG_DAT_AMIM_MINUS_3DB       0x0D
#define RICOH5B_REG_DAT_AMIM_0DB             0x0E
#define RICOH5B_REG_DAT_AMIM_3DB             0x0F
#define RICOH5B_REG_DAT_AMIM_6DB             0x10
#define RICOH5B_REG_DAT_AMIM_9DB             0x11
#define RICOH5B_REG_DAT_AMIM_12DB            0x12
#define RICOH5B_REG_DAT_AMIM_15DB            0x13
#define RICOH5B_REG_DAT_AMIM_18DB            0x14
#define RICOH5B_REG_DAT_AMIM_21DB            0x15
#define RICOH5B_REG_DAT_AMIM_24DB            0x16
#define RICOH5B_REG_DAT_AMIM_27DB            0x17
#define RICOH5B_REG_DAT_AMIM_30DB            0x18
#define RICOH5B_REG_DAT_AMIM_33DB            0x19
#define RICOH5B_REG_DAT_AMIM_36DB            0x1A
#endif //RICOH5B_USE_TUNER

/* address 40h */
#define RICOH5B_REG_ADR_FM_VIFO_UPPER        0x40
#define RICOH5B_REG_DAT_FM_VIFO_UPPER        0x05
#define RICOH5B_REG_DAT_FM_VIFO_UPPER_IMMUNITY_OUT  0x1A

/* address 41h */
#define RICOH5B_REG_ADR_FM_VIFO_LOWER        0x41
#define RICOH5B_REG_DAT_FM_VIFO_LOWER        0x97
#define RICOH5B_REG_DAT_FM_VIFO_LOWER_IMMUNITY_OUT  0x97

/* address 42h */
#define RICOH5B_REG_ADR_FM_VUD_UPPER         0x42
#define RICOH5B_REG_DAT_FM_VUD_UPPER         0x08
#define RICOH5B_REG_DAT_FM_VUD_UPPER_IMMUNITY_OUT   0x1D

/* address 43h */
#define RICOH5B_REG_ADR_FM_VUD_LOWER         0x43
#define RICOH5B_REG_DAT_FM_VUD_LOWER         0x83
#define RICOH5B_REG_DAT_FM_VUD_LOWER_IMMUNITY_OUT   0x83

/* address 46h */
#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_AM_VIFO_UPPER        0x46
#define RICOH5B_REG_DAT_AM_VIFO_UPPER        0x18
#endif //RICOH5B_USE_TUNER

/* address 47h */
#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_AM_VIFO_LOWER        0x47
#define RICOH5B_REG_DAT_AM_VIFO_LOWER        0x13
#endif //RICOH5B_USE_TUNER

/* address 48h */
#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_AM_VUD_UPPER         0x48
#define RICOH5B_REG_DAT_AM_VUD_UPPER         0x14
#endif //RICOH5B_USE_TUNER

/* address 49h */
#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_AM_VUD_LOWER         0x49
#define RICOH5B_REG_DAT_AM_VUD_LOWER         0x12
#endif //RICOH5B_USE_TUNER

/* address 4Ah */
#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_AM_PIF_SCALE         0x4A
#define RICOH5B_REG_DAT_AM_PIF_SCALE         0x0F
#endif //RICOH5B_USE_TUNER

/* address 4Bh */
#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_AM_THR_IFR           0x4B
#define RICOH5B_REG_BIT_AM_THR_IFR           0xF0
#define RICOH5B_REG_DAT_AM_THR_IFR           0x60

#define RICOH5B_REG_ADR_AM_DEC_IFR           0x4B
#define RICOH5B_REG_BIT_AM_DEC_IFR           0x0F
#define RICOH5B_REG_DAT_AM_DEC_IFR           0x03
#endif //RICOH5B_USE_TUNER

/* address 4Dh */
#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_WIN_CNT_VAL          0x4D
#define RICOH5B_REG_DAT_WIN_CNT_VAL          0xFF
#endif //RICOH5B_USE_TUNER

/* address 53h */
#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_AM_GDM_M             0x53
#endif //RICOH5B_USE_TUNER

/* address 54h */
#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_AM_DAGC              0x54
#define RICOH5B_REG_DAT_AM_DAGC              0xB0
#define RICOH5B_REG_LENGTH_AM_DAGC              2
						/* 2byte of Reg.53h - 54h */

#define RICOH5B_REG_ADR_AM_DAGC_AUTO         0x54
#define RICOH5B_REG_BIT_AM_DAGC_AUTO         0x80
#define RICOH5B_REG_DAT_AM_DAGC_AUTO_MODE    0x80
#define RICOH5B_REG_DAT_AM_DAGC_MANUAL_MODE  0x00

#define RICOH5B_REG_ADR_AM_GDE_M             0x54
#define RICOH5B_REG_BIT_AM_GDE_M             0x0F
#endif //RICOH5B_USE_TUNER

/* address 55h */
#define RICOH5B_REG_ADR_FM_RSSI_CONST        0x55
#define RICOH5B_REG_DAT_FM_RSSI_CONST        0x35

/* address 57h */
#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_RSSI_THR             0x57
#else
#define RICOH5B_REG_ADR_RSSI_THR             0x3F
#endif //RICOH5B_USE_TUNER
#define RICOH5B_REG_BIT_RSSI_THR             0x7F

/* address 58h */
#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_DITHER               0x58
#else
#define RICOH5B_REG_ADR_DITHER               0x97
#endif //RICOH5B_USE_TUNER
#define RICOH5B_REG_DAT_DITHER_AUDIO_IIS     0x00
#define RICOH5B_REG_DAT_DITHER_AUDIO_ANALOG_FM  0x28
#define RICOH5B_REG_DAT_DITHER_AUDIO_ANALOG_AM  0x7F

/* address 59h */
#define RICOH5B_REG_ADR_FREQ_ROTATION_L                 0x59
#define RICOH5B_REG_DAT_FREQ_ROTATION_L_FM_IF200K       0x55
#define RICOH5B_REG_DAT_FREQ_ROTATION_L_FM_IF300K       0x00
#define RICOH5B_REG_DAT_FREQ_ROTATION_L_FM_IF400K       0xAB
#define RICOH5B_REG_DAT_FREQ_ROTATION_L_AM              0xAB
#define RICOH5B_REG_DAT_FREQ_ROTATION_L_AM_SEEK         0xAB
#define RICOH5B_REG_DAT_FREQ_ROTATION_L_AM_1            0x6B
#define RICOH5B_REG_DAT_FREQ_ROTATION_L_AM_2            0xEB
#define RICOH5B_REG_DAT_FREQ_ROTATION_L_AM_3            0x77
#define RICOH5B_REG_DAT_FREQ_ROTATION_L_AM_4            0xDE
#define RICOH5B_REG_DAT_FREQ_ROTATION_L_AM_5            0x33
#define RICOH5B_REG_DAT_FREQ_ROTATION_L_AM_PREV         0x2A
#define RICOH5B_REG_DAT_FREQ_ROTATION_L_AM_NEXT         0x2A
#define RICOH5B_REG_DAT_FREQ_ROTATION_L_AM_MINUS_1K     0xDA
#define RICOH5B_REG_DAT_FREQ_ROTATION_L_AM_PLUS_1K      0x80
#define RICOH5B_REG_DAT_FREQ_ROTATION_L_AM_IF300K       0x00
#define RICOH5B_REG_DAT_FREQ_ROTATION_L_AM_IF300K_PREV  0x7F
#define RICOH5B_REG_DAT_FREQ_ROTATION_L_AM_IF300K_NEXT  0x81
#define RICOH5B_REG_DAT_FREQ_ROTATION_L_AM_IF300K_MINUS_1K  0x2F
#define RICOH5B_REG_DAT_FREQ_ROTATION_L_AM_IF300K_PLUS_1K   0xD5
#define RICOH5B_REG_LENGTH_FREQ_ROTATION          2

/* address 5Ah */
#define RICOH5B_REG_ADR_FREQ_ROTATION_H                 0x5A
#define RICOH5B_REG_BIT_FREQ_ROTATION_H                 0x7F
#define RICOH5B_REG_DAT_FREQ_ROTATION_H_FM_IF200K       0x21
#define RICOH5B_REG_DAT_FREQ_ROTATION_H_FM_IF300K       0x32
#define RICOH5B_REG_DAT_FREQ_ROTATION_H_FM_IF400K       0x42
#define RICOH5B_REG_DAT_FREQ_ROTATION_H_AM              0x42
#define RICOH5B_REG_DAT_FREQ_ROTATION_H_AM_SEEK         0x42
#define RICOH5B_REG_DAT_FREQ_ROTATION_H_AM_1            0x42
#define RICOH5B_REG_DAT_FREQ_ROTATION_H_AM_2            0x42
#define RICOH5B_REG_DAT_FREQ_ROTATION_H_AM_3            0x42
#define RICOH5B_REG_DAT_FREQ_ROTATION_H_AM_4            0x42
#define RICOH5B_REG_DAT_FREQ_ROTATION_H_AM_5            0x32
#define RICOH5B_REG_DAT_FREQ_ROTATION_H_AM_PREV         0x44
#define RICOH5B_REG_DAT_FREQ_ROTATION_H_AM_NEXT         0x41
#define RICOH5B_REG_DAT_FREQ_ROTATION_H_AM_MINUS_1K     0x42
#define RICOH5B_REG_DAT_FREQ_ROTATION_H_AM_PLUS_1K      0x42
#define RICOH5B_REG_DAT_FREQ_ROTATION_H_AM_IF300K       0x32
#define RICOH5B_REG_DAT_FREQ_ROTATION_H_AM_IF300K_PREV  0x33
#define RICOH5B_REG_DAT_FREQ_ROTATION_H_AM_IF300K_NEXT  0x30
#define RICOH5B_REG_DAT_FREQ_ROTATION_H_AM_IF300K_MINUS_1K  0x32
#define RICOH5B_REG_DAT_FREQ_ROTATION_H_AM_IF300K_PLUS_1K   0x31

#define RICOH5B_REG_ADR_DOWNCONV             0x5A
#define RICOH5B_REG_BIT_DOWNCONV             0x80
#define RICOH5B_REG_DAT_DOWNCONV_ON          0x80
#define RICOH5B_REG_DAT_DOWNCONV_OFF         0x00

/* address 5Ch */
#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_AM_TH_DET            0x5C
#define RICOH5B_REG_DAT_AM_TH_DET            0xC6
#endif //RICOH5B_USE_TUNER

/* address 61h */
#define RICOH5B_REG_ADR_PG                   0x61
#define RICOH5B_REG_BIT_PG                   0xF0

#define RICOH5B_REG_ADR_FMI                  0x61
#define RICOH5B_REG_BIT_FMI                  0x0F

/* address 69h */
#define RICOH5B_REG_ADR_AD6_DATA             0x69
#define RICOH5B_REG_BIT_AD6_DATA             0x3F
#define RICOH5B_REG_LENGTH_AD6_DATA             1

/* address 6Bh */
#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_AM_DAGC_GAIN         0x6B
#define RICOH5B_REG_LENGTH_AM_DAGC_GAIN         1

#define RICOH5B_REG_ADR_AM_GDM               0x6B
#define RICOH5B_REG_BIT_AM_GDM               0xF0

#define RICOH5B_REG_ADR_AM_GDE               0x6B
#define RICOH5B_REG_BIT_AM_GDE               0x0F
#endif //RICOH5B_USE_TUNER

/* address 7Fh */
#define RICOH5B_REG_ADR_IF_ERR               0x7F
#define RICOH5B_REG_LENGTH_IF_ERR               1

/* address 82h */
#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_R_AMTUNE_L           0x82
#define RICOH5B_REG_BIT_R_AMTUNE_L           0xFF
#define RICOH5B_REG_LENGTH_R_AMTUNE             2
#endif //RICOH5B_USE_TUNER

/* address 83h */
#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_R_AMTUNE_H           0x83
#define RICOH5B_REG_BIT_R_AMTUNE_H           0x0F
#endif //RICOH5B_USE_TUNER

/* address 84h, 85h */
#define RICOH5B_REG_ADR_HPLVOL               0x84
#define RICOH5B_REG_ADR_HPRVOL               0x85
#define RICOH5B_REG_LENGTH_VOL                  2

#define RICOH5B_REG_DAT_HPVOL_FM_0DB         0x39
#define RICOH5B_REG_DAT_HPVOL_FM_MINUS_3DB   0x28
#define RICOH5B_REG_DAT_HPVOL_FM_MINUS_4DB   0x24
#define RICOH5B_REG_DAT_HPVOL_FM_MINUS_5DB   0x20
#define RICOH5B_REG_DAT_HPVOL_AM             0x10
#define RICOH5B_REG_DAT_HPVOL_OFF            0x00

/* address 90h */
#define RICOH5B_REG_ADR_BLOCK2_PE            0x90
#define RICOH5B_REG_BIT_BLOCK2_PE            0x1F
#define RICOH5B_REG_DAT_BLOCK2_PE_ALL_ON     0xFF

#define RICOH5B_REG_BIT_PELO                 0x10
#define RICOH5B_REG_DAT_PELO_ON              0x10
#define RICOH5B_REG_DAT_PELO_OFF             0x00

#if RICOH5B_USE_TUNER
#define RICOH5B_REG_BIT_PEAMMIX              0x08
#define RICOH5B_REG_DAT_PEAMMIX_ON           0x08
#define RICOH5B_REG_DAT_PEAMMIX_OFF          0x00

#define RICOH5B_REG_BIT_PEAMLNA              0x04
#define RICOH5B_REG_DAT_PEAMLNA_ON           0x04
#define RICOH5B_REG_DAT_PEAMLNA_OFF          0x00
#endif //RICOH5B_USE_TUNER

#define RICOH5B_REG_BIT_PEFMMIX              0x02
#define RICOH5B_REG_DAT_PEFMMIX_ON           0x02
#define RICOH5B_REG_DAT_PEFMMIX_OFF          0x00

#define RICOH5B_REG_BIT_PEFMLNA              0x01
#define RICOH5B_REG_DAT_PEFMLNA_ON           0x01
#define RICOH5B_REG_DAT_PEFMLNA_OFF          0x00

/* address 92h */
#define RICOH5B_REG_ADR_LOSTART_EN           0x92
#define RICOH5B_REG_BIT_LOSTART_EN           0x04
#define RICOH5B_REG_DAT_LOSTART_EN_ENABLE    0x04
#define RICOH5B_REG_DAT_LOSTART_EN_DISABLE   0x00

#define RICOH5B_REG_ADR_LOSTART              0x92
#define RICOH5B_REG_BIT_LOSTART              0x03
#define RICOH5B_REG_DAT_LOSTART              0x03

/* address 93h */
#define RICOH5B_REG_ADR_VCO                  0x93
#define RICOH5B_REG_DAT_VCO                  0xEE

/* address 94h */
#define RICOH5B_REG_ADR_IFSEL                0x94
#define RICOH5B_REG_DAT_SYN_DIV_1_1          0x20
#define RICOH5B_REG_DAT_IFSEL_200K           0x00
#define RICOH5B_REG_DAT_IFSEL_300K           0x01
#define RICOH5B_REG_DAT_IFSEL_400K           0x02
#define RICOH5B_REG_LENGTH_IFSEL                1

/* address 96h */
#define RICOH5B_REG_ADR_IMFMEN_EN            0x96
#define RICOH5B_REG_DAT_IMFMEN_ON            0x80
#define RICOH5B_REG_DAT_IMFMEN_OFF           0x00

#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_IMAMEN_EN            0x96
#define RICOH5B_REG_DAT_IMAMEN_ON            0x40
#define RICOH5B_REG_DAT_IMAMEN_OFF           0x00
#endif //RICOH5B_USE_TUNER

#define RICOH5B_REG_ADR_FMEN_EN              0x96
#define RICOH5B_REG_DAT_FMEN_ON              0x20
#define RICOH5B_REG_DAT_FMEN_OFF             0x00

#define RICOH5B_REG_ADR_96_TEST              0x96
#define RICOH5B_REG_DAT_96_TEST              0x18

/* address 98h */
#define RICOH5B_REG_ADR_IMF                  0x98
#define RICOH5B_REG_BIT_IMF                  0x70
#define RICOH5B_REG_DAT_IMF_OFF              0x00
#define RICOH5B_REG_DAT_IMF_AM               0x40
#define RICOH5B_REG_DAT_IMF_76_8MHZ          0x40
#define RICOH5B_REG_DAT_IMF_87_77MHZ         0x50
#define RICOH5B_REG_DAT_IMF_102_4MHZ         0x60

/* address 9Ah */
#define RICOH5B_REG_ADR_SPOFS_SYN            0x9A
#define RICOH5B_REG_DAT_SPOFS_SYN            0x02

/* address 9Dh */
#define RICOH5B_REG_ADR_ADCMOD               0x9D
#define RICOH5B_REG_DAT_ADCMOD_EXCHANGE_IQ   0x04
#define RICOH5B_REG_DAT_ADCMOD_NOT_EXCHANGE_IQ  0x00

/* address A0h */
#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_AD1_RSTB             0xA0
#else  //RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_AD1_RSTB             0xD7
#endif //RICOH5B_USE_TUNER
#define RICOH5B_REG_DAT_AD1_RSTB_NORMAL      0x01

/* address A1h */
#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_RTRIM_SEL            0xA1
#else
#define RICOH5B_REG_ADR_RTRIM_SEL            0xD6
#endif //RICOH5B_USE_TUNER
#define RICOH5B_REG_BIT_RTRIM_SEL            0x10
#define RICOH5B_REG_DAT_RTRIM_SEL_MANUAL     0x10
#define RICOH5B_REG_DAT_RTRIM_SEL_NORMAL     0x00

/* address A2h */
#define RICOH5B_REG_ADR_JAM_POWER            0xA2
#define RICOH5B_REG_DAT_JAM_POWER            0x00

/* address A8h */
#define RICOH5B_REG_ADR_FGCOR_0_L            0xA8
#define RICOH5B_REG_LENGTH_PG                   4

/* address A9h */
#define RICOH5B_REG_ADR_FGCOR_0_H            0xA9

/* address ACh */
#define RICOH5B_REG_ADR_FGCOR_1_L            0xAC

/* address ADh */
#define RICOH5B_REG_ADR_FGCOR_1_H            0xAD

/* address B0h */
#define RICOH5B_REG_ADR_FGCOR_2_L            0xB0

/* address B1h */
#define RICOH5B_REG_ADR_FGCOR_2_H            0xB1

/* address B4h */
#define RICOH5B_REG_ADR_FGCOR_3_L            0xB4

/* address B5h */
#define RICOH5B_REG_ADR_FGCOR_3_H            0xB5

/* address B8h */
#define RICOH5B_REG_ADR_FGCOR_4_L            0xB8

/* address B9h */
#define RICOH5B_REG_ADR_FGCOR_4_H            0xB9

/* address BCh */
#define RICOH5B_REG_ADR_RGCOR_0_L            0xBC
#define RICOH5B_REG_DAT_RGCOR_DEFAULT        0x0800

/* address BDh */
#define RICOH5B_REG_ADR_RGCOR_0_H            0xBD

/* address BEh */
#define RICOH5B_REG_ADR_RPHCOR_0_L           0xBE
#define RICOH5B_REG_DAT_RPHCOR_DEFAULT       0x0800

/* address BFh */
#define RICOH5B_REG_ADR_RPHCOR_0_H           0xBF

/* address C0h */
#define RICOH5B_REG_ADR_RGCOR_1_L            0xC0

/* address C1h */
#define RICOH5B_REG_ADR_RGCOR_1_H            0xC1

/* address C4h */
#define RICOH5B_REG_ADR_RGCOR_2_L            0xC4

/* address C5h */
#define RICOH5B_REG_ADR_RGCOR_2_H            0xC5

/* address C8h */
#define RICOH5B_REG_ADR_RGCOR_3_L            0xC8

/* address C9h */
#define RICOH5B_REG_ADR_RGCOR_3_H            0xC9

/* address CCh */
#define RICOH5B_REG_ADR_RGCOR_4_L            0xCC

/* address CDh */
#define RICOH5B_REG_ADR_RGCOR_4_H            0xCD

/* address D0h */
#define RICOH5B_REG_ADR_MUTE_WAIT_CNT        0xD0
#define RICOH5B_REG_DAT_MUTE_WAIT_CNT_MUTE_ON   0x00
#define RICOH5B_REG_DAT_MUTE_WAIT_CNT_MUTE_OFF  0x60

/* address D1h */
#define RICOH5B_REG_ADR_DCCAL_INV            0xD1
#define RICOH5B_REG_BIT_DCCAL_INV            0x01
#define RICOH5B_REG_BIT_DCCAL_INV_ADD        0x01
#define RICOH5B_REG_BIT_DCCAL_INV_SUB        0x00

/* address D2h */
#define RICOH5B_REG_ADR_FM_VIFO_INIT         0xD2
#define RICOH5B_REG_DAT_FM_VIFO_INIT         0xDB
#define RICOH5B_REG_DAT_FM_VIFO_INIT_IMMUNITY_OUT  0xDA

/* address D3h */
#define RICOH5B_REG_ADR_FM_VUD_INIT          0xD3
#define RICOH5B_REG_DAT_FM_VUD_INIT          0xEA
#define RICOH5B_REG_DAT_FM_VUD_INIT_IMMUNITY_OUT  0xFA

/* address D5h */
#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_AM_VIFO_INIT         0xD5
#define RICOH5B_REG_DAT_AM_VIFO_INIT         0xDC
#endif //RICOH5B_USE_TUNER

/* address D6h */
#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_AM_VUD_INIT          0xD6
#define RICOH5B_REG_DAT_AM_VUD_INIT          0xA8
#endif //RICOH5B_USE_TUNER

/* address D7h */
#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_AM_PIF_INIT          0xD7
#define RICOH5B_REG_DAT_AM_PIF_INIT          0x1F
#endif //RICOH5B_USE_TUNER

/* address DAh */
#define RICOH5B_REG_ADR_TABLE_MODE           0xDA
#define RICOH5B_REG_BIT_TABLE_MODE           0x0C
#define RICOH5B_REG_DAT_TABLE_MODE           0x00

/* address E0h */
#define RICOH5B_REG_ADR_AVECOUNT             0xE0
#define RICOH5B_REG_BIT_AVECOUNT             0xC0
#define RICOH5B_REG_DAT_AVECOUNT_128         0x00
#define RICOH5B_REG_DAT_AVECOUNT_16          0x40

#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_CTUNE1710            0xE0
#define RICOH5B_REG_BIT_CTUNE1710            0x08
#define RICOH5B_REG_DAT_CTUNE1710_ON         0x08
#define RICOH5B_REG_DAT_CTUNE1710_OFF        0x00

#define RICOH5B_REG_ADR_GOCTUNE              0xE0
#define RICOH5B_REG_BIT_GOCTUNE              0x04
#define RICOH5B_REG_DAT_GOCTUNE_START        0x04
#define RICOH5B_REG_DAT_GOCTUNE_STOP         0x00
#define RICOH5B_REG_LENGTH_GOCTUNE              1
#endif //RICOH5B_USE_TUNER

#define RICOH5B_REG_ADR_IMLADJUST            0xE0
#define RICOH5B_REG_BIT_IMLADJUST            0x02
#define RICOH5B_REG_DAT_IMLADJUST_START      0x02
#define RICOH5B_REG_DAT_IMLADJUST_STOP       0x00

#define RICOH5B_REG_ADR_IQCORRECT            0xE0
#define RICOH5B_REG_BIT_IQCORRECT            0x01
#define RICOH5B_REG_DAT_IQCORRECT_START      0x01
#define RICOH5B_REG_DAT_IQCORRECT_STOP       0x00

/* address E1h */
#define RICOH5B_REG_ADR_IMLI_L               0xE1
#define RICOH5B_REG_LENGTH_IMLI                 2

/* address E2h */
#define RICOH5B_REG_ADR_IMLI_H               0xE2

/* address E3h */
#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_FCTUNE               0xE3
#define RICOH5B_REG_DAT_FCTUNE_DEFAULT       0x7F
#define RICOH5B_REG_LENGTH_FCTUNE               1
#endif //RICOH5B_USE_TUNER

/* address E4h */
#define RICOH5B_REG_ADR_DAGC_SFT             0xE4

#define RICOH5B_REG_BIT_DAGC_SFT_EN          0x80
#define RICOH5B_REG_DAT_DAGC_SFT_EN_ON       0x80
#define RICOH5B_REG_DAT_DAGC_SFT_EN_OFF      0x00

#define RICOH5B_REG_BIT_DAGC_SFT_MASK        0x07
#define RICOH5B_REG_DAT_DAGC_SFT_MASK_3BIT   0x03
#define RICOH5B_REG_DAT_DAGC_SFT_MASK_1BIT   0x01
#define RICOH5B_REG_DAT_DAGC_SFT_MASK_NOT    0x00

/* address E5h */
#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_AM_ADJ               0xE5
#define RICOH5B_REG_BIT_AM_ADJ               0xC0
#define RICOH5B_REG_DAT_AM_ADJ_AMP_THROUGH   0x40
#define RICOH5B_REG_DAT_AM_ADJ_NORMAL        0x00
#endif //RICOH5B_USE_TUNER

#define RICOH5B_REG_ADR_FMLNAS               0xE5
#define RICOH5B_REG_BIT_FMLNAS               0x10
#define RICOH5B_REG_DAT_FMLNAS_DIFFERENTIAL  0x10
#define RICOH5B_REG_DAT_FMLNAS_SINGLE        0x00

#define RICOH5B_REG_ADR_FMLNAI               0xE5
#define RICOH5B_REG_BIT_FMLNAI               0x01
#define RICOH5B_REG_DAT_FMLNAI_LOW_POWER     0x01
#define RICOH5B_REG_DAT_FMLNAI_NORMAL        0x00

/* address E6h */
#define RICOH5B_REG_ADR_AD6_I_SW             0xE6
#define RICOH5B_REG_BIT_AD6_I_SW             0x80
#define RICOH5B_REG_DAT_AD6_I_SW             0x80

#define RICOH5B_REG_ADR_AD6_GAIN_O           0xE6
#define RICOH5B_REG_BIT_AD6_GAIN_O           0x70
#define RICOH5B_REG_DAT_AD6_GAIN_O_0_5       0x00
#define RICOH5B_REG_DAT_AD6_GAIN_O_1         0x20

#define RICOH5B_REG_ADR_AD6_GAIN_I           0xE6
#define RICOH5B_REG_BIT_AD6_GAIN_I           0x0C

#define RICOH5B_REG_ADR_AD6_HOLDMODE         0xE6
#define RICOH5B_REG_BIT_AD6_HOLDMODE         0x03
#define RICOH5B_REG_DAT_AD6_HOLDMODE         0x03

/* address E7h */
#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_CTUNE_EN             0xE7
#define RICOH5B_REG_BIT_CTUNE_EN             0x20
#define RICOH5B_REG_DAT_CTUNE_EN_ON          0x20
#define RICOH5B_REG_DAT_CTUNE_EN_OFF         0x00

#define RICOH5B_REG_ADR_CTUNE_HI             0xE7
#define RICOH5B_REG_BIT_CTUNE_HI             0x10
#define RICOH5B_REG_DAT_CTUNE_HI_1600KHZ     0x10
#define RICOH5B_REG_DAT_CTUNE_HI_480KHZ      0x00
#endif //RICOH5B_USE_TUNER

/* address E8h */
#define RICOH5B_REG_ADR_XFS                  0xE8
#define RICOH5B_REG_BIT_XFS                  0xC0
#define RICOH5B_REG_DAT_XFS_32K              0x00
#define RICOH5B_REG_DAT_XFS_12M              0x80

#define RICOH5B_REG_ADR_PELPFVGA             0xE8
#define RICOH5B_REG_DAT_PELPFVGA_ON          0x08

#define RICOH5B_REG_ADR_FC_SW                0xE8
#define RICOH5B_REG_BIT_FC_SW                0x10
#define RICOH5B_REG_DAT_FC_SW_1M             0x00
#define RICOH5B_REG_DAT_FC_SW_500K           0x10

#define RICOH5B_REG_ADR_MIX_ISEL             0xE8
#define RICOH5B_REG_BIT_MIX_ISEL             0x01
#define RICOH5B_REG_DAT_MIX_ISEL_I_SOURCE    0x01
#define RICOH5B_REG_DAT_MIX_ISEL_INTERNAL    0x00

/* address E9h */
#define RICOH5B_REG_ADR_IMFMSW               0xE9
#define RICOH5B_REG_BIT_IMFMSW               0x10
#define RICOH5B_REG_DAT_IMFMSW_ENABLE        0x10
#define RICOH5B_REG_DAT_IMFMSW_DISABLE       0x00

#define RICOH5B_REG_ADR_IMFMA                0xE9
#define RICOH5B_REG_BIT_IMFMA                0x0F
#define RICOH5B_REG_DAT_IMFMA_MAX            0x0F
#define RICOH5B_REG_DAT_IMFMA_MIN            0x00

/* address EAh */
#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_IMAMSW               0xEA
#define RICOH5B_REG_BIT_IMAMSW               0x10
#define RICOH5B_REG_DAT_IMAMSW_ENABLE        0x10
#define RICOH5B_REG_DAT_IMAMSW_DISABLE       0x00
#endif //RICOH5B_USE_TUNER

/* address EBh */
#define RICOH5B_REG_ADR_VBFSEL               0xEB
#define RICOH5B_REG_BIT_VBFSEL               0x07
#define RICOH5B_REG_DAT_VBFSEL_1_1V          0x01
#define RICOH5B_REG_DAT_VBFSEL_1_0V          0x02

#define RICOH5B_REG_DAT_VBFSEL_DEFAULT       RICOH5B_REG_DAT_VBFSEL_1_0V

/* address ECh */
#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_AMRFSW               0xEC
#define RICOH5B_REG_BIT_AMRFSW               0x20
#define RICOH5B_REG_DAT_AMRFSW_ENABLE        0x20
#define RICOH5B_REG_DAT_AMRFSW_DISABLE       0x00
#endif //RICOH5B_USE_TUNER

/* address EEh */
#define RICOH5B_REG_ADR_ITRIM                0xEE
#define RICOH5B_REG_DAT_ITRIM_6MA            0x40

/* address EFh */
#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_AMYSEL               0xEF
#define RICOH5B_REG_BIT_AMYSEL               0x0F
#define RICOH5B_REG_DAT_AMYSEL_MAX           0x0F
#endif //RICOH5B_USE_TUNER

/* address F7h */
#define RICOH5B_REG_ADR_RDS_OUT_EN_H         0xF7
#define RICOH5B_REG_DAT_RDS_OUT_EN_H         0x55

/* address F8h */
#define RICOH5B_REG_ADR_RDS_OUT_EN_L         0xF8
#define RICOH5B_REG_DAT_RDS_OUT_EN_L         0xAA

/* address F9h */
#define RICOH5B_REG_ADR_XBOOST               0xF9
#define RICOH5B_REG_BIT_XBOOST               0x20
#define RICOH5B_REG_BIT_XBOOST_BOOST         0x20
#define RICOH5B_REG_BIT_XBOOST_NORMAL        0x00

#define RICOH5B_REG_ADR_ISEL2                0xF9
#define RICOH5B_REG_BIT_ISEL2                0x10
#define RICOH5B_REG_DAT_ISEL2_LOW_POWER      0x10
#define RICOH5B_REG_DAT_ISEL2_NORMAL         0x00

/* address FAh */
#define RICOH5B_REG_ADR_XDOWN                0xFA
#define RICOH5B_REG_BIT_XDOWN                0x60
#define RICOH5B_REG_DAT_XDOWN_LITTLE         0x60
#define RICOH5B_REG_DAT_XDOWN_MUCH           0x00

#define RICOH5B_REG_ADR_DET_AFM              0xFA
#define RICOH5B_REG_BIT_DET_AFM              0x10
#define RICOH5B_REG_DAT_DET_AFM_AM           0x00
#define RICOH5B_REG_DAT_DET_AFM_FM           0x10

#define RICOH5B_REG_ADR_DET_DC_ADJ           0xFA
#define RICOH5B_REG_BIT_DET_DC_ADJ           0x0F
#define RICOH5B_REG_DAT_DET_DC_ADJ_PLUS_7    0x07
#define RICOH5B_REG_DAT_DET_DC_ADJ_PLUS_6    0x06
#define RICOH5B_REG_DAT_DET_DC_ADJ_PLUS_5    0x05
#define RICOH5B_REG_DAT_DET_DC_ADJ_PLUS_4    0x04
#define RICOH5B_REG_DAT_DET_DC_ADJ_PLUS_3    0x03
#define RICOH5B_REG_DAT_DET_DC_ADJ_PLUS_2    0x02
#define RICOH5B_REG_DAT_DET_DC_ADJ_PLUS_1    0x01
#define RICOH5B_REG_DAT_DET_DC_ADJ_PLUS_0    0x00
#define RICOH5B_REG_DAT_DET_DC_ADJ_MINUS_0   0x0F
#define RICOH5B_REG_DAT_DET_DC_ADJ_MINUS_1   0x0E
#define RICOH5B_REG_DAT_DET_DC_ADJ_MINUS_2   0x0D
#define RICOH5B_REG_DAT_DET_DC_ADJ_MINUS_3   0x0C
#define RICOH5B_REG_DAT_DET_DC_ADJ_MINUS_4   0x0B
#define RICOH5B_REG_DAT_DET_DC_ADJ_MINUS_5   0x0A
#define RICOH5B_REG_DAT_DET_DC_ADJ_MINUS_6   0x09
#define RICOH5B_REG_DAT_DET_DC_ADJ_MINUS_7   0x08

/* address FBh */
#if RICOH5B_USE_TUNER
#define RICOH5B_REG_ADR_COM_SEL              0xFB
#else
#define RICOH5B_REG_ADR_COM_SEL              0xF4
#endif //RICOH5B_USE_TUNER
#define RICOH5B_REG_BIT_COM_SEL              0xC0
#define RICOH5B_REG_DAT_COM_SEL_300M         0xC0
#define RICOH5B_REG_DAT_COM_SEL_400M         0x00

/* address FDh */
#define RICOH5B_REG_ADR_PADEN_CK             0xFD
#define RICOH5B_REG_DAT_PADEN_CK_ENABLE      0x10
#define RICOH5B_REG_DAT_PADEN_CK_DISABLE     0x00
#define RICOH5B_REG_DAT_AMPEN_CK_ENABLE      0x08
#define RICOH5B_REG_DAT_AMPEN_CK_DISABLE     0x00
#define RICOH5B_REG_DAT_CAPOUT_CK_ENABLE     0x04
#define RICOH5B_REG_DAT_CAPADJ_CK_0_25       0x02

/* address FEh */
#define RICOH5B_REG_ADR_SRST                 0xFE
#define RICOH5B_REG_DAT_SRST_DSP             0x0A
#define RICOH5B_REG_DAT_SRST_REGISTER        0xA0
#define RICOH5B_REG_DAT_SRST_ALL             0xAA

/* address FFh */
#define RICOH5B_REG_ADR_VERSION              0xFF
#define RICOH5B_REG_LENGTH_VERSION              1

#define RICOH5B_NULL          (void *)0

#endif /* _RICOH5B_MAIN_H_ */
