// SPDX-License-Identifier: GPL-2.0

/* ac_cut.c: AC-CUT Detect Driver for HA
 * Copyright 2021, 2022 Sony Corporation
 */
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/delay.h>
#include <linux/gpio.h>
#include <linux/interrupt.h>
#include <linux/of_irq.h>
#include <linux/of_gpio.h>
#include <linux/of_device.h>
#include <linux/platform_device.h>
#include <linux/reboot.h>
#include <linux/cm4.h>

#define DRV_NAME		"ac_cut"

#define PINCTR_IOMUXC_BASE	0x30330000
#define PINCTR_IOMUXC_REG_SIZE	0xFFFF

#define PINCTR_SAI1_TXD0	(0x3033018C - PINCTR_IOMUXC_BASE)
#define PINCTR_SAI1_TXD1	(0x30330190 - PINCTR_IOMUXC_BASE)
#define PINCTR_SAI1_TXD2	(0x30330194 - PINCTR_IOMUXC_BASE)
#define PINCTR_SAI1_TXD3	(0x30330198 - PINCTR_IOMUXC_BASE)
#define PINCTR_SAI1_RXD7	(0x30330180 - PINCTR_IOMUXC_BASE)
#define PINCTR_SAI1_TXD5	(0x303301A0 - PINCTR_IOMUXC_BASE)
#define PINCTR_SAI1_TXD6	(0x303301A4 - PINCTR_IOMUXC_BASE)
#define PINCTR_SAI1_TXD7	(0x303301A8 - PINCTR_IOMUXC_BASE)
#define PINCTR_SAI1_TXFS	(0x30330184 - PINCTR_IOMUXC_BASE)
#define PINCTR_SAI1_TXC		(0x30330188 - PINCTR_IOMUXC_BASE)
#define PINCTR_SAI2_MCLK	(0x303301C8 - PINCTR_IOMUXC_BASE)
#define PINCTR_SAI5_MCLK	(0x30330158 - PINCTR_IOMUXC_BASE)

#define ERRLOG(fmt, ...)	pr_err(DRV_NAME ": Error: " fmt, \
					## __VA_ARGS__)
#define WRNLOG(fmt, ...)	pr_warn(DRV_NAME ": Warning: " fmt, \
					## __VA_ARGS__)
#define MSGLOG(fmt, ...)	pr_info(DRV_NAME ": " fmt, ## __VA_ARGS__)
#define DBGLOG(fmt, ...)	pr_debug(DRV_NAME ": " fmt, ## __VA_ARGS__)

enum {
	PCONT_ON	= 0,
	PCONT1,
	PCONT_AMP,
	PCONT_VCORE,
	PCONT_18V,
	PCONT_33V,
	HW_SYSRSTB,
	PCONT_TS,
	PCONT_IR,
	PCONT_MIC,
	HW_MUTE,
	ZONE_MUTE,
	USB_A_SW,
	PCONT_RY,
	LCD_RST,
	PCONT_LCD,
	PCONT_ETH,
	PCONT_LCD_BRT,
	GPIOS_COUNT		// total GPIO pins
};

static char const *gpio_name[] = {
	"pcont_on-gpio",
	"pcont1-gpio",
	"pcont_amp-gpio",
	"pcont_vcore-gpio",
	"pcont_18v-gpio",
	"pcont_33v-gpio",
	"hw_sysrstb-gpio",
	"pcont_ts-gpio",
	"pcont_ir-gpio",
	"pcont_mic-gpio",
	"hw_mute-gpio",
	"zone_soc_mute-gpio",
	"usb_a_sw-gpio",
	"pcont_ry-gpio",
	"lcd_rst-gpio",
	"pcont_lcd-gpio",
	"pcont_eth-gpio",
	"pcont_lcd_bright-gpio",
};

struct pinctrl_detail {
	void __iomem		*address;
	int			alt;
};

struct gpio_param {
	int			pin;
	enum of_gpio_flags	flags;
};

/* private data */
struct priv {
	struct delayed_work	dwork;		/* Workqueue */
	struct gpio_desc	*gpio_accut;	/* ACCUT-GPIO */
	int			irq;		/* ACCUT-IRQ */
	struct gpio_param	*pcont_gpios;
	int			*sai_gpios;	/* SAI GPIOs */
	int			sai_count;	/* SAI GPIO count */
	void __iomem		*iomuxc_base;	/* iomuxc base address */
	unsigned int		det_delay;	/* Detection Delay[ms] */
	int			sai_pins_count;
	struct pinctrl_detail	*pins_data;
};

enum {
	ACCUT_SEQUENCE_TYPE_1 = 1,
};

/*----------------------------------------------------------------------*/
/*	gpio control function						*/
/*----------------------------------------------------------------------*/
static void pcont_gpio_set(struct priv *p_priv, int index, int value)
{
	int	pin;
	enum	of_gpio_flags flags;

	pin = p_priv->pcont_gpios[index].pin;
	if (pin >= 0) {
		flags = p_priv->pcont_gpios[index].flags;
		value = (flags & OF_GPIO_ACTIVE_LOW) ? !value : value;
		DBGLOG("set %s=%d\n", gpio_name[index], value);
		gpio_set_value(pin, value);
	}
}

static void pcont_gpio_set_cansleep(struct priv *p_priv, int index, int value)
{
	int	pin;
	enum	of_gpio_flags flags;

	pin = p_priv->pcont_gpios[index].pin;
	if (pin >= 0) {
		flags = p_priv->pcont_gpios[index].flags;
		value = (flags & OF_GPIO_ACTIVE_LOW) ? !value : value;
		DBGLOG("set %s(s)=%d\n", gpio_name[index], value);
		gpio_set_value_cansleep(pin, value);
	}
}

/*----------------------------------------------------------------------*/
/*	ACCUT IRQ entry							*/
/*----------------------------------------------------------------------*/
static irqreturn_t ac_cut_irq(int irq_num, void *data)
{
	struct priv	*p_priv = data;
	int i, result, pin;

	MSGLOG("IRQ enter\n");
	pcont_gpio_set(p_priv, HW_MUTE, 1);
	pcont_gpio_set(p_priv, ZONE_MUTE, 1);
	mdelay(15);

        if (p_priv->sai_pins_count) {
		for (i = 0; i < p_priv->sai_pins_count; i++)
			iowrite32(p_priv->pins_data[i].alt,
					p_priv->pins_data[i].address);
	} else {
		iowrite32(0x05, p_priv->iomuxc_base + PINCTR_SAI1_TXD0);
		iowrite32(0x05, p_priv->iomuxc_base + PINCTR_SAI1_TXD1);
		iowrite32(0x05, p_priv->iomuxc_base + PINCTR_SAI1_TXD2);
		iowrite32(0x05, p_priv->iomuxc_base + PINCTR_SAI1_TXD3);
		iowrite32(0x05, p_priv->iomuxc_base + PINCTR_SAI1_RXD7);
		iowrite32(0x05, p_priv->iomuxc_base + PINCTR_SAI1_TXD5);
		iowrite32(0x05, p_priv->iomuxc_base + PINCTR_SAI1_TXD6);
		iowrite32(0x05, p_priv->iomuxc_base + PINCTR_SAI1_TXD7);
		iowrite32(0x05, p_priv->iomuxc_base + PINCTR_SAI1_TXFS);
		iowrite32(0x05, p_priv->iomuxc_base + PINCTR_SAI1_TXC);
		iowrite32(0x05, p_priv->iomuxc_base + PINCTR_SAI2_MCLK);
		iowrite32(0x05, p_priv->iomuxc_base + PINCTR_SAI5_MCLK);
	}

	if (p_priv->sai_gpios) {
		DBGLOG("set gpio to nonactive\n");
		for (i = 0; i < p_priv->sai_count; i++) {
			pin = p_priv->sai_gpios[i];
			result = gpio_direction_output(pin, 0);
			if (result)
				ERRLOG("set gpio direction(%d=%d)\n",
					pin, result);
		}
	}

	if (p_priv->pcont_gpios[PCONT_AMP].pin >= 0
	 && !gpio_cansleep(p_priv->pcont_gpios[PCONT_AMP].pin)) {
		pcont_gpio_set(p_priv, PCONT_AMP, 0);
		mdelay(15);
	}
	pcont_gpio_set(p_priv, PCONT_ON, 0);
	pcont_gpio_set(p_priv, PCONT_RY, 0);

	schedule_delayed_work(&p_priv->dwork, 0);
	DBGLOG("IRQ done\n");
	return IRQ_HANDLED;
}

/*----------------------------------------------------------------------*/
/*	Interrupt process						*/
/*----------------------------------------------------------------------*/
static void ac_cut_proc(struct work_struct *p_work)
{
	struct delayed_work *p_data;
	struct priv	*p_priv;

	DBGLOG("proc enter\n");
	p_data = container_of(p_work, struct delayed_work, work);
	p_priv = container_of(p_data, struct priv, dwork);

	if (p_priv->pcont_gpios[PCONT_AMP].pin >= 0
	 && gpio_cansleep(p_priv->pcont_gpios[PCONT_AMP].pin))
		pcont_gpio_set_cansleep(p_priv, PCONT_AMP, 0);
	pcont_gpio_set_cansleep(p_priv, PCONT1, 0);
	pcont_gpio_set_cansleep(p_priv, PCONT_MIC, 0);
	pcont_gpio_set_cansleep(p_priv, PCONT_TS, 0);
	pcont_gpio_set_cansleep(p_priv, PCONT_IR, 0);
	pcont_gpio_set_cansleep(p_priv, USB_A_SW, 0);
	pcont_gpio_set_cansleep(p_priv, LCD_RST, 1);
	pcont_gpio_set_cansleep(p_priv, PCONT_LCD, 0);
	pcont_gpio_set_cansleep(p_priv, PCONT_LCD_BRT, 0);
	pcont_gpio_set_cansleep(p_priv, PCONT_ETH, 0);

	rpmsg_cm4_accut_notifier();

	MSGLOG("Restart after 3s\n");
	usleep_range(3000000, 3001000);
	emergency_restart();
}

/*----------------------------------------------------------------------*/
/*	ACCUT IRQ entry							*/
/*----------------------------------------------------------------------*/
static irqreturn_t ac_cut_irq_thread(int irq_num, void *data)
{
	struct priv	*p_priv = data;
	int i, result, pin;

	DBGLOG("IRQ thread enter\n");
	usleep_range(p_priv->det_delay * 1000,
		     p_priv->det_delay * 1000 + 10);

	/* Detect accut again to remove noise */
	if (gpiod_get_value(p_priv->gpio_accut) == 0)
		return IRQ_HANDLED;

	MSGLOG("Start force shutdown sequence\n");
	pcont_gpio_set(p_priv, HW_MUTE, 1);
	pcont_gpio_set(p_priv, ZONE_MUTE, 1);
	mdelay(15);

        if (p_priv->sai_pins_count) {
		for (i = 0; i < p_priv->sai_pins_count; i++)
			iowrite32(p_priv->pins_data[i].alt,
					p_priv->pins_data[i].address);
	} else {
		iowrite32(0x05, p_priv->iomuxc_base + PINCTR_SAI1_TXD0);
		iowrite32(0x05, p_priv->iomuxc_base + PINCTR_SAI1_TXD1);
		iowrite32(0x05, p_priv->iomuxc_base + PINCTR_SAI1_TXD2);
		iowrite32(0x05, p_priv->iomuxc_base + PINCTR_SAI1_TXD3);
		iowrite32(0x05, p_priv->iomuxc_base + PINCTR_SAI1_RXD7);
		iowrite32(0x05, p_priv->iomuxc_base + PINCTR_SAI1_TXD5);
		iowrite32(0x05, p_priv->iomuxc_base + PINCTR_SAI1_TXD6);
		iowrite32(0x05, p_priv->iomuxc_base + PINCTR_SAI1_TXD7);
		iowrite32(0x05, p_priv->iomuxc_base + PINCTR_SAI1_TXFS);
		iowrite32(0x05, p_priv->iomuxc_base + PINCTR_SAI1_TXC);
		iowrite32(0x05, p_priv->iomuxc_base + PINCTR_SAI2_MCLK);
		iowrite32(0x05, p_priv->iomuxc_base + PINCTR_SAI5_MCLK);
	}

	if (p_priv->sai_gpios) {
		DBGLOG("set gpio to nonactive\n");
		for (i = 0; i < p_priv->sai_count; i++) {
			pin = p_priv->sai_gpios[i];
			result = gpio_direction_output(pin, 0);
			if (result)
				ERRLOG("set gpio direction(%d=%d)\n",
					pin, result);
		}
	}

	if (p_priv->pcont_gpios[PCONT_AMP].pin >= 0) {
		if (gpio_cansleep(p_priv->pcont_gpios[PCONT_AMP].pin))
			pcont_gpio_set_cansleep(p_priv, PCONT_AMP, 0);
		else
			pcont_gpio_set(p_priv, PCONT_AMP, 0);
		mdelay(15);
	}
	pcont_gpio_set_cansleep(p_priv, PCONT1, 0);
	pcont_gpio_set_cansleep(p_priv, PCONT_MIC, 0);
	pcont_gpio_set_cansleep(p_priv, PCONT_TS, 0);
	pcont_gpio_set_cansleep(p_priv, PCONT_IR, 0);
	pcont_gpio_set_cansleep(p_priv, USB_A_SW, 0);
	pcont_gpio_set_cansleep(p_priv, LCD_RST, 1);
	pcont_gpio_set_cansleep(p_priv, PCONT_LCD, 0);
	pcont_gpio_set_cansleep(p_priv, PCONT_LCD_BRT, 0);
	pcont_gpio_set_cansleep(p_priv, PCONT_ETH, 0);
	pcont_gpio_set(p_priv, PCONT_ON, 0);
	pcont_gpio_set(p_priv, PCONT_RY, 0);

	rpmsg_cm4_accut_notifier();

	MSGLOG("Restart after 3s\n");
	usleep_range(3000000, 3001000);
	emergency_restart();

	DBGLOG("IRQ thread done\n");
	return IRQ_HANDLED;
}

/*----------------------------------------------------------------------*/
/*	AC-Cut Sequence-1 (thread)					*/
/*----------------------------------------------------------------------*/
static irqreturn_t ac_cut_seq1_thread(int irq_num, void *data)
{
	struct priv	*p_priv = data;
	int i, result, pin;

	MSGLOG("Start force shutdown sequence [type-1]\n");

	if (p_priv->det_delay) {
		usleep_range(p_priv->det_delay * 1000,
			     p_priv->det_delay * 1000 + 10);

		/* Detect accut again to remove noise */
		if (gpiod_get_value(p_priv->gpio_accut) == 0)
			return IRQ_HANDLED;
	}

	pcont_gpio_set(p_priv, HW_MUTE, 1);
	pcont_gpio_set(p_priv, ZONE_MUTE, 1);

        if (p_priv->sai_pins_count) {
		for (i = 0; i < p_priv->sai_pins_count; i++)
			iowrite32(p_priv->pins_data[i].alt,
					p_priv->pins_data[i].address);
	} else {
		iowrite32(0x05, p_priv->iomuxc_base + PINCTR_SAI1_TXD0);
		iowrite32(0x05, p_priv->iomuxc_base + PINCTR_SAI1_TXD1);
		iowrite32(0x05, p_priv->iomuxc_base + PINCTR_SAI1_TXD2);
		iowrite32(0x05, p_priv->iomuxc_base + PINCTR_SAI1_TXD3);
		iowrite32(0x05, p_priv->iomuxc_base + PINCTR_SAI1_RXD7);
		iowrite32(0x05, p_priv->iomuxc_base + PINCTR_SAI1_TXD5);
		iowrite32(0x05, p_priv->iomuxc_base + PINCTR_SAI1_TXD6);
		iowrite32(0x05, p_priv->iomuxc_base + PINCTR_SAI1_TXD7);
		iowrite32(0x05, p_priv->iomuxc_base + PINCTR_SAI1_TXFS);
		iowrite32(0x05, p_priv->iomuxc_base + PINCTR_SAI1_TXC);
		iowrite32(0x05, p_priv->iomuxc_base + PINCTR_SAI2_MCLK);
		iowrite32(0x05, p_priv->iomuxc_base + PINCTR_SAI5_MCLK);
	}

	if (p_priv->sai_gpios) {
		DBGLOG("set gpio to nonactive\n");
		for (i = 0; i < p_priv->sai_count; i++) {
			pin = p_priv->sai_gpios[i];
			result = gpio_direction_output(pin, 0);
			if (result)
				ERRLOG("set gpio direction(%d=%d)\n",
					pin, result);
		}
	}

	pcont_gpio_set_cansleep(p_priv, HW_SYSRSTB, 0);
	mdelay(5);
	pcont_gpio_set_cansleep(p_priv, PCONT_33V, 0);
	mdelay(5);
	pcont_gpio_set_cansleep(p_priv, PCONT_18V, 0);
	mdelay(5);
	if (p_priv->pcont_gpios[PCONT_AMP].pin >= 0) {
		if (gpio_cansleep(p_priv->pcont_gpios[PCONT_AMP].pin))
			pcont_gpio_set_cansleep(p_priv, PCONT_AMP, 0);
		else
			pcont_gpio_set(p_priv, PCONT_AMP, 0);
	}
	mdelay(15);

	pcont_gpio_set_cansleep(p_priv, PCONT_MIC, 0);
	pcont_gpio_set_cansleep(p_priv, PCONT_TS, 0);
	pcont_gpio_set_cansleep(p_priv, PCONT_IR, 0);
	pcont_gpio_set_cansleep(p_priv, USB_A_SW, 0);
	pcont_gpio_set_cansleep(p_priv, LCD_RST, 1);
	pcont_gpio_set_cansleep(p_priv, PCONT_LCD, 0);
	pcont_gpio_set_cansleep(p_priv, PCONT_LCD_BRT, 0);
	pcont_gpio_set_cansleep(p_priv, PCONT_ETH, 0);
	pcont_gpio_set_cansleep(p_priv, PCONT1, 0);
	pcont_gpio_set_cansleep(p_priv, PCONT_VCORE, 0);
	pcont_gpio_set(p_priv, PCONT_ON, 0);
	pcont_gpio_set(p_priv, PCONT_RY, 0);

	rpmsg_cm4_accut_notifier();

	MSGLOG("Restart after 3s\n");
	usleep_range(3000000, 3001000);
	emergency_restart();

	DBGLOG("End ac_cut sequence1(thread)\n");
	return IRQ_HANDLED;
}

/*----------------------------------------------------------------------*/
/*	platform device Probe						*/
/*----------------------------------------------------------------------*/
static int ac_cut_probe(struct platform_device *p_dev)
{
	struct device		*dev = &p_dev->dev;
	struct priv		*p_priv;
	int			result;
	int			pin, i, cnt;
	int			offset, alt;
	unsigned int		dly;
	unsigned int		type = 0;
	enum of_gpio_flags	flags;

	/* Allocate private data memory */
	p_priv = devm_kzalloc(dev, sizeof(struct priv), GFP_KERNEL);
	if (IS_ERR(p_priv)) {
		result = PTR_ERR(p_priv);
		ERRLOG("private data allocate(%d)\n", result);
		return result;
	}

	/* Get gpio pins from devicetree */
	p_priv->gpio_accut = devm_gpiod_get(dev, "accut", GPIOD_IN);
	if (IS_ERR(p_priv->gpio_accut)) {
		result = PTR_ERR(p_priv->gpio_accut);
		ERRLOG("accut-gpios is not detect(%d)\n", result);
		return result;
	}

	p_priv->pcont_gpios = devm_kzalloc(dev,
				(size_t)GPIOS_COUNT *
				 sizeof(struct gpio_param),
				 GFP_KERNEL);
	if (IS_ERR(p_priv->pcont_gpios)) {
		result = PTR_ERR(p_priv->pcont_gpios);
		ERRLOG("failed to allocate pcont_gpios(%d)\n", result);
		return result;
	}
	for (i = 0; i < GPIOS_COUNT; i++) {
		pin = of_get_named_gpio_flags(dev->of_node, gpio_name[i],
					      0, &flags);
		if (pin < 0) {
			if (pin == -EPROBE_DEFER) {
				ERRLOG("%s is busy(%d)\n", gpio_name[i], pin);
				return -EPROBE_DEFER;
			}
			WRNLOG("%s is not detect(%d)\n", gpio_name[i], pin);
			p_priv->pcont_gpios[i].pin = -1;
		} else {
			DBGLOG("detect %s(%d)\n", gpio_name[i], pin);
			p_priv->pcont_gpios[i].pin = pin;
			p_priv->pcont_gpios[i].flags = flags;
		}
	}

	p_priv->iomuxc_base = ioremap_nocache(PINCTR_IOMUXC_BASE,
						PINCTR_IOMUXC_REG_SIZE);
	cnt = of_gpio_named_count(dev->of_node, "sai-gpios");
	if (cnt > 0) {
		DBGLOG("detect sai-gpios(%d)\n", cnt);
		p_priv->sai_count = cnt;
		p_priv->sai_gpios = devm_kzalloc(dev,
						(size_t)cnt * sizeof(int),
						GFP_KERNEL);
		if (IS_ERR(p_priv->sai_gpios)) {
			ERRLOG("shutdown-gpios buffer allocate\n");
			return -ENOMEM;
		}
		for (i = 0; i < cnt; i++)
			p_priv->sai_gpios[i] = of_get_named_gpio(
							dev->of_node,
							"sai-gpios",
							i);
	} else {
		WRNLOG("sai-gpios is not detect(%d)\n", cnt);
		p_priv->sai_gpios = NULL;
	}

	cnt = of_property_count_u32_elems(dev->of_node, "sai-pins");
	if (cnt > 0) {
		p_priv->sai_pins_count = cnt / 5;
		p_priv->pins_data = devm_kzalloc(dev,
					(size_t)p_priv->sai_pins_count *
					 sizeof(struct pinctrl_detail),
					 GFP_KERNEL);
		for (i = 0; i < p_priv->sai_pins_count; i++) {
			if (!of_property_read_u32_index(dev->of_node,
						"sai-pins", i * 5, &offset) &&
			    !of_property_read_u32_index(dev->of_node,
						"sai-pins", i * 5 + 3, &alt)) {
				p_priv->pins_data[i].address =
						p_priv->iomuxc_base + offset;
				p_priv->pins_data[i].alt = alt;
			}
		}
	} else {
		p_priv->sai_pins_count = 0;
		p_priv->pins_data = NULL;
	}

	/* Get detect delay value */
	p_priv->det_delay = 0;
	result = device_property_read_u32(dev, "detection-delay", &dly);
	if (!result)
		p_priv->det_delay = dly;
	MSGLOG("detection-delay=%u[ms]\n", p_priv->det_delay);

	result = device_property_read_u32(dev, "sequence_type", &type);
	MSGLOG("sequence type=%d(%d)\n", type, result);

	/* Create delayed Workqueue */
	INIT_DELAYED_WORK(&p_priv->dwork, ac_cut_proc);

	/* Set irq handler */
	p_priv->irq = gpiod_to_irq(p_priv->gpio_accut);
	DBGLOG("accut_irq=%d\n", p_priv->irq);
	if (type == ACCUT_SEQUENCE_TYPE_1) {
		result = devm_request_threaded_irq(dev, p_priv->irq,
			NULL, ac_cut_seq1_thread,
			IRQF_SHARED | IRQF_TRIGGER_FALLING | IRQF_ONESHOT,
			dev_name(dev), p_priv);
	} else {
		if (!p_priv->det_delay)
			result = devm_request_irq(dev, p_priv->irq,
				ac_cut_irq,
				IRQF_SHARED | IRQF_TRIGGER_FALLING,
				dev_name(dev), p_priv);
		else
			result = devm_request_threaded_irq(dev, p_priv->irq,
				NULL, ac_cut_irq_thread,
				IRQF_SHARED | IRQF_TRIGGER_FALLING |
				IRQF_ONESHOT,
				dev_name(dev), p_priv);
	}

	if (result) {
		ERRLOG("request irq handler(%d)\n", result);
		return result;
	}

	/* Set provate data */
	dev_set_drvdata(dev, p_priv);

	DBGLOG("probe done\n");
	return 0;
}

/*----------------------------------------------------------------------*/
/*	platform device  Remove						*/
/*----------------------------------------------------------------------*/
static int ac_cut_remove(struct platform_device *p_dev)
{
	struct device	*dev = &p_dev->dev;
	struct priv	*p_priv = dev_get_drvdata(dev);

	DBGLOG("remove enter\n");
	devm_free_irq(dev, p_priv->irq, p_priv);
	DBGLOG("remove done\n");
	return 0;
}

/* -------------------------------------------------------------------- */
/*	Configration platform device driver				*/
/* -------------------------------------------------------------------- */
static const struct of_device_id ac_cut_dt_ids[] = {
	{ .compatible = "sony,ac_cut", },
	{}
};
MODULE_DEVICE_TABLE(of, ac_cut_dt_ids);

static struct platform_driver ac_cut_driver = {
	.probe     = ac_cut_probe,
	.remove    = ac_cut_remove,
	.driver    = {
		.name  = DRV_NAME,
		.owner = THIS_MODULE,
		.of_match_table = ac_cut_dt_ids,
	},
};

module_platform_driver(ac_cut_driver);
MODULE_DESCRIPTION("AC-CUT Detect Driver for HA");
MODULE_AUTHOR("Sony Home Entertainment & Sound Products Inc.");
MODULE_LICENSE("GPL");
MODULE_VERSION("v1.00");

