/* SPDX-License-Identifier: GPL-2.0 */
/*
 * leds_cm4.c - led driver
 *
 * Copyright 2021 Sony Corporation
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/virtio.h>
#include <linux/rpmsg.h>
#include <linux/err.h>
#include <linux/cdev.h>
#include <linux/platform_device.h>
#include <linux/cm4.h>

#define DRV_NAME		"leds_cm4"
#define ERRLOG(fmt, ...)	pr_err(DRV_NAME ": Error: " fmt, ## __VA_ARGS__)
#define WRNLOG(fmt, ...)	pr_warn(DRV_NAME ": Warning: " fmt, ## __VA_ARGS__)
#define MSGLOG(fmt, ...)	pr_info(DRV_NAME ": " fmt, ## __VA_ARGS__)
#define DBGLOG(fmt, ...)	pr_debug(DRV_NAME ": %s(%d): " fmt, __func__, __LINE__, ## __VA_ARGS__)

/*******************************************************************************
 * Definition
*******************************************************************************/
#define POWER_LED_ID		(1)
#define POWER_LED_CHG_RESP	(0x81)
#define POWER_LED_STS_RESP	(0x82)

u8	leds_cm4_mng_info_seq;

/*******************************************************************************
 * Variables
*******************************************************************************/

/*******************************************************************************
 * Functions
*******************************************************************************/
int leds_core_cm4_send(u8 type, u8 length, u8 *data)
{
	int ret;

	if ((data == NULL) || (length < 4)) {
		ERRLOG("param error data=%p length=%d\n", data, length);
		return -EINVAL;
	}

	ret = rpmsg_cm4_send(CM4_MESSAGE_ID_LED, type, length,
						leds_cm4_mng_info_seq, data);
	if (ret != 0) {
		ERRLOG("rpmsg_cm4_send failed(%d)\n", ret);
		return -EINVAL;
	}
	leds_cm4_mng_info_seq++;
	return 0;
}

static ssize_t leds_core_status_show(struct device *dev,
				struct device_attribute *attr,
				char *buf)
{
	ssize_t rb = 0;
	u8 data[4] = {0x02, 0x00, POWER_LED_ID, 0x00};

	leds_core_cm4_send(CM4_MESSAGE_TYPE_REQ, 4, data);

	return rb;
}

static ssize_t leds_core_power_store(struct device *dev,
				struct device_attribute *attr,
				const char *buf, size_t size)
{
	u8 power;
	u8 data[4] = {0x01, 0x00, POWER_LED_ID, 0x00};

	if (kstrtou8(buf, 0, &power))
		return -EINVAL;

	if (power > 1)
		return -EINVAL;

	data[3] = power;
	leds_core_cm4_send(CM4_MESSAGE_TYPE_REQ, 4, data);

	return size;
}

static DEVICE_ATTR(status, 0600, leds_core_status_show, NULL);
static DEVICE_ATTR(lpower, 0600, NULL, leds_core_power_store);

static struct attribute *attributes[] = {
	&dev_attr_status.attr,
	&dev_attr_lpower.attr,
	NULL,
};

static const struct attribute_group attr_group = {
	.attrs	= (struct attribute **)attributes,
};

void leds_core_callback(u8 type, u8 length, u8 seq, u8 *data)
{
	DBGLOG("callback\n");

	if ((data == NULL) || (length <= 0)) {
		ERRLOG("param error data=%p length=%d\n", data, length);
		return;
	}

	switch (data[0]) {
	case POWER_LED_CHG_RESP:
		MSGLOG("change	status ret=%d sts=%d\n", data[1], data[3]);
		break;
	case POWER_LED_STS_RESP:
		MSGLOG("current status ret=%d sts=%d\n", data[1], data[3]);
		break;
	default:
		ERRLOG("invalid command id =0x%02x\n", data[0]);
		break;
	}

}

static int leds_core_probe(struct platform_device *pdev)
{
	int ret = 0;

	DBGLOG("probe\n");

	ret = rpmsg_cm4_register_callback(CM4_MESSAGE_ID_LED,
							leds_core_callback);
	if (ret != 0) {
		ERRLOG("regist callback failed(%d)\n", ret);
		return -EPROBE_DEFER;
	}

	ret = sysfs_create_group(&pdev->dev.kobj, &attr_group);
	if (ret) {
		ERRLOG("sysfs create failed(%d)\n", ret);
		return ret;
	}

	return 0;
}

static int leds_core_remove(struct platform_device *pdev)
{
	DBGLOG("remove\n");

	sysfs_remove_group(&pdev->dev.kobj, &attr_group);

	return 0;
}

static const struct of_device_id leds_core_dt_ids[] = {
	{ .compatible = "sony,leds_cm4", },
	{ },
};
MODULE_DEVICE_TABLE(of, leds_core_dt_ids);

static struct platform_driver leds_core_driver = {
	.driver = {
		.name = DRV_NAME,
		.owner = THIS_MODULE,
		.of_match_table = leds_core_dt_ids,
	},
	.probe = leds_core_probe,
	.remove = leds_core_remove,
};

static int __init leds_core_init(void)
{
	int result = 0;

	DBGLOG("init\n");

	result = platform_driver_register(&leds_core_driver);
	if (result != 0) {
		ERRLOG("register driver failed(%d)\n", result);
		return result;
	}

	leds_cm4_mng_info_seq = 0;

	return 0;
}

static void __exit leds_core_exit(void)
{
	DBGLOG("exit\n");

	platform_driver_unregister(&leds_core_driver);
}

module_init(leds_core_init);
module_exit(leds_core_exit);


MODULE_AUTHOR("Sony Corporation");
MODULE_DESCRIPTION("leds_cm4");
MODULE_LICENSE("GPL");

