// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright 2021 Sony Corporation
 */

#include <linux/err.h>
#include <linux/gpio/consumer.h>
#include <linux/module.h>
#include <linux/mux/driver.h>
#include <linux/of_platform.h>
#include <linux/platform_device.h>
#include <linux/property.h>
#include <linux/delay.h>

#define MUTE_INTERVAL (200) /* ms */

#undef TRACE_PRINT_ON
#define TRACE_TAG "####### "

/* trace print macro */
#ifdef TRACE_PRINT_ON
	#define print_trace(fmt, args...) pr_info(TRACE_TAG "" fmt, ##args)
#else
	#define print_trace(fmt, args...) {}
#endif

struct njm41050_priv {
	struct gpio_descs *gpios;
	int state;
	struct workqueue_struct *wq;
	struct delayed_work work;
};

static void delayed_work_func(struct work_struct *work)
{
	struct delayed_work *dwork = to_delayed_work(work);
	struct njm41050_priv *priv = container_of(
		dwork, struct njm41050_priv, work);
	int val[priv->gpios->ndescs];
	int i;

	print_trace("%s()\n", __func__);

	for (i = 0; i < priv->gpios->ndescs; i++)
		val[i] = (priv->state >> i) & 1;
	gpiod_set_array_value_cansleep(priv->gpios->ndescs,
				       priv->gpios->desc,
				       val);
}

static int njm41050_set(struct mux_control *mux, int state)
{
	struct njm41050_priv *priv = mux_chip_priv(mux->chip);
	int val_muting[priv->gpios->ndescs];
	int i;

	print_trace("%s()\n", __func__);

	if (priv->state == state)
		return 0;

	cancel_delayed_work_sync(&priv->work);
	priv->state = state;
	for (i = 0; i < priv->gpios->ndescs; i++)
		val_muting[i] = 0;
	gpiod_set_array_value_cansleep(priv->gpios->ndescs,
				       priv->gpios->desc,
				       val_muting);

	queue_delayed_work(priv->wq, &priv->work,
		msecs_to_jiffies(MUTE_INTERVAL));

	return 0;
}

static const struct mux_control_ops njm41050_ops = {
	.set = njm41050_set,
};

static const struct of_device_id njm41050_dt_ids[] = {
	{ .compatible = "njr,njm41050", },
	{ /* sentinel */ }
};
MODULE_DEVICE_TABLE(of, njm41050_dt_ids);

static int njm41050_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct mux_chip *mux_chip;
	struct njm41050_priv *priv;
	int pins;
	s32 idle_state;
	int ret;

	print_trace("%s()\n", __func__);

	pins = gpiod_count(dev, "mux");
	if (pins < 0)
		return pins;

	mux_chip = devm_mux_chip_alloc(dev, 1, sizeof(*priv));
	if (IS_ERR(mux_chip))
		return PTR_ERR(mux_chip);

	priv = mux_chip_priv(mux_chip);
	priv->state = 0;
	priv->wq = create_singlethread_workqueue("njm41050_wq");
	if (!priv->wq)
		return -ENOMEM;
	INIT_DELAYED_WORK(&priv->work, delayed_work_func);
	mux_chip->ops = &njm41050_ops;

	priv->gpios = devm_gpiod_get_array(dev, "mux", GPIOD_OUT_LOW);
	if (IS_ERR(priv->gpios)) {
		ret = PTR_ERR(priv->gpios);
		if (ret != -EPROBE_DEFER)
			dev_err(dev, "failed to get gpios\n");
		goto out1;
	}
	WARN_ON(pins != priv->gpios->ndescs);
	mux_chip->mux->states = 1 << pins;

	ret = device_property_read_u32(dev, "idle-state", (u32 *)&idle_state);
	if (ret >= 0 && idle_state != MUX_IDLE_AS_IS) {
		if (idle_state < 0 || idle_state >= mux_chip->mux->states) {
			dev_err(dev, "invalid idle-state %u\n", idle_state);
			ret = -EINVAL;
			goto out1;
		}

		mux_chip->mux->idle_state = idle_state;
	}

	ret = devm_mux_chip_register(dev, mux_chip);
	if (ret < 0)
		goto out1;

	dev_info(dev, "%u-way mux-controller registered\n",
		 mux_chip->mux->states);

	return 0;
out1:
	destroy_workqueue(priv->wq);
	return ret;
}

static struct platform_driver njm41050_driver = {
	.driver = {
		.name = "gpio-mux",
		.of_match_table	= of_match_ptr(njm41050_dt_ids),
	},
	.probe = njm41050_probe,
};
module_platform_driver(njm41050_driver);

MODULE_DESCRIPTION("NJM41050 driver");
MODULE_AUTHOR("Sony Corporation");
MODULE_LICENSE("GPL v2");
MODULE_VERSION("v1.00");
