/*
 * drivers/net/phy/realtek.c
 *
 * Driver for Realtek PHYs
 *
 * Author: Johnson Leung <r58129@freescale.com>
 *
 * Copyright (c) 2004 Freescale Semiconductor, Inc.
 *
 * This program is free software; you can redistribute  it and/or modify it
 * under  the terms of  the GNU General  Public License as published by the
 * Free Software Foundation;  either version 2 of the  License, or (at your
 * option) any later version.
 *
 */
#include <linux/phy.h>
#include <linux/module.h>
#include <linux/delay.h>

#define RTL821x_PHYSR		0x11
#define RTL821x_PHYSR_DUPLEX	0x2000
#define RTL821x_PHYSR_SPEED	0xc000
#define RTL821x_INER		0x12
#define RTL821x_INER_INIT	0x6400
#define RTL821x_INSR		0x13
#define RTL8211E_INER_LINK_STATUS 0x400

#define RTL8211F_INER_LINK_STATUS 0x0010
#define RTL8211F_INSR		0x1d
#define RTL8211F_PAGE_SELECT	0x1f
#define RTL8211F_TX_DELAY	0x100

#define RTL8211FD_VX_RX_DELAY	0x008

MODULE_DESCRIPTION("Realtek PHY driver");
MODULE_AUTHOR("Johnson Leung");
MODULE_LICENSE("GPL");

struct rtl8211fd_vx_context {
	u16 bmcr;
	u16 advertise;
	u16 control1000;
};

struct rtl8211fd_vx_priv {
	struct rtl8211fd_vx_context save_context;
	bool context_saved;
};

static u16 rtl8211fd_vx_register_patch1[] = {
	31, 0x0B82, 16, 0x0010, 27, 0x802B, 28, 0x8700,
	27, 0xB82E, 28, 0x0001, 16, 0x0090, 27, 0xA016,
	28, 0x0000, 27, 0xA012, 28, 0x0000, 27, 0xA014,
	28, 0x1800, 28, 0x8010, 28, 0x1800, 28, 0x801c,
	28, 0x1800, 28, 0x8022, 28, 0x1800, 28, 0x805d,
	28, 0x1800, 28, 0x8073, 28, 0x1800, 28, 0x8077,
	28, 0x1800, 28, 0x80de, 28, 0x1800, 28, 0x80ef,
	28, 0xd710, 28, 0x6060, 28, 0x1800, 28, 0x0d1a,
	28, 0x8602, 28, 0xd710, 28, 0x2e71, 28, 0x0d1e,
	28, 0xd71e, 28, 0x7f8c, 28, 0x1800, 28, 0x0bba,
	28, 0x1000, 28, 0x0722, 28, 0x1000, 28, 0x07a1,
	28, 0x1800, 28, 0x00a6, 28, 0xcc00, 28, 0x1000,
	28, 0x07c5, 28, 0xd502, 28, 0x8307, 28, 0xd501,
	28, 0xc30f, 28, 0xc2b0, 28, 0xc117, 28, 0xd500,
	28, 0xc20c, 28, 0x1000, 28, 0x0659, 28, 0xd502,
	28, 0xa080, 28, 0xa308, 28, 0xd500, 28, 0xc430,
	28, 0x9af0, 28, 0x0c0f, 28, 0x0102, 28, 0xc808,
	28, 0xd024, 28, 0xd1db, 28, 0xd06c, 28, 0xd1c4,
	28, 0xd702, 28, 0x9410, 28, 0xc030, 28, 0x401c,
	28, 0xa3b0, 28, 0x8208, 28, 0x40ea, 28, 0x1000,
	28, 0x0722, 28, 0x1000, 28, 0x07a1, 28, 0x1800,
	28, 0x804b, 28, 0x1000, 28, 0x07a7, 28, 0xd702,
	28, 0x35a8, 28, 0x0025, 28, 0x6114, 28, 0x1000,
	28, 0x0579, 28, 0x1000, 28, 0x0662, 28, 0x3188,
	28, 0x00b2, 28, 0xd702, 28, 0x36a3, 28, 0x0078,
	28, 0x37a5, 28, 0x0025, 28, 0x5d00, 28, 0x1800,
	28, 0x03ae, 28, 0xd700, 28, 0x616c, 28, 0xd603,
	28, 0xd60c, 28, 0xd610, 28, 0xd61c, 28, 0xd624,
	28, 0xd628, 28, 0xd630, 28, 0xd638, 28, 0x1800,
	28, 0x072a, 28, 0xd603, 28, 0xd60c, 28, 0xd610,
	28, 0xd61c, 28, 0xd620, 28, 0xd62c, 28, 0xd630,
	28, 0xd638, 28, 0x1800, 28, 0x072a, 28, 0x8102,
	28, 0xa110, 28, 0x1800, 28, 0x047e, 28, 0x263c,
	28, 0x807a, 28, 0xac20, 28, 0x9f10, 28, 0x2019,
	28, 0x0a6c, 28, 0xd710, 28, 0x2d69, 28, 0x0a6c,
	28, 0x1000, 28, 0x80d4, 28, 0xa240, 28, 0x8910,
	28, 0xce02, 28, 0x0c70, 28, 0x0f10, 28, 0xaf01,
	28, 0x8f01, 28, 0x1000, 28, 0x09ed, 28, 0x8e02,
	28, 0xa808, 28, 0xbf10, 28, 0x1000, 28, 0x09ff,
	28, 0xd710, 28, 0x60bc, 28, 0xd71e, 28, 0x7fa4,
	28, 0x1800, 28, 0x0ba4, 28, 0xce04, 28, 0x0c70,
	28, 0x0f20, 28, 0xaf01, 28, 0x8f01, 28, 0x1000,
	28, 0x09ed, 28, 0x8e04, 28, 0xd71e, 28, 0x6064,
	28, 0x1800, 28, 0x0ba4, 28, 0xa520, 28, 0x0c0f,
	28, 0x0504, 28, 0xd710, 28, 0x60b9, 28, 0xd71e,
	28, 0x7fa4, 28, 0x1800, 28, 0x0ba4, 28, 0xa13e,
	28, 0xc240, 28, 0x84c0, 28, 0xb104, 28, 0xa704,
	28, 0xa701, 28, 0xa510, 28, 0xd710, 28, 0x4018,
	28, 0x9910, 28, 0x9f10, 28, 0x8510, 28, 0xd710,
	28, 0xc001, 28, 0x4000, 28, 0x5a46, 28, 0xa101,
	28, 0xa43f, 28, 0xa660, 28, 0x9104, 28, 0xc001,
	28, 0x4000, 28, 0xd710, 28, 0x60ea, 28, 0x7fc6,
	28, 0xd71e, 28, 0x2425, 28, 0x0a43, 28, 0x1800,
	28, 0x80d0, 28, 0xa420, 28, 0xa680, 28, 0xd71e,
	28, 0x2425, 28, 0x0a43, 28, 0xd710, 28, 0x7f86,
	28, 0x1000, 28, 0x0a1c, 28, 0x1800, 28, 0x808c,
	28, 0xd711, 28, 0x6081, 28, 0xaa11, 28, 0x1800,
	28, 0x80db, 28, 0xaa22, 28, 0xa904, 28, 0xa901,
	28, 0xd710, 28, 0x0800, 28, 0x8520, 28, 0xd71f,
	28, 0x3ce1, 28, 0x80e4, 28, 0x1800, 28, 0x0c72,
	28, 0xa208, 28, 0xc001, 28, 0xd710, 28, 0x60a0,
	28, 0xd71e, 28, 0x7fac, 28, 0x1800, 28, 0x0bba,
	28, 0x8208, 28, 0x1800, 28, 0x0bc1, 28, 0x1000,
	28, 0x0722, 28, 0x1000, 28, 0x07a1, 28, 0x1800,
	28, 0x03d9, 27, 0xA026, 28, 0x03d7, 27, 0xA024,
	28, 0x0c71, 27, 0xA022, 28, 0x0a69, 27, 0xA020,
	28, 0x047d, 27, 0xA006, 28, 0x0722, 27, 0xA004,
	28, 0x0375, 27, 0xA002, 28, 0x00a2, 27, 0xA000,
	28, 0x0d18, 27, 0xA008, 28, 0xff00, 31, 0x0B82,
	16, 0x0010, 27, 0x8465, 28, 0xaf84, 28, 0x7daf,
	28, 0x84ed, 28, 0xaf84, 28, 0xf0af, 28, 0x84f3,
	28, 0xaf85, 28, 0x02af, 28, 0x8509, 28, 0xaf85,
	28, 0x14af, 28, 0x8526, 28, 0x0211, 28, 0x8fbf,
	28, 0x5335, 28, 0x0252, 28, 0x00ef, 28, 0x21bf,
	28, 0x533b, 28, 0x0252, 28, 0x001e, 28, 0x21bf,
	28, 0x5338, 28, 0x0252, 28, 0x001e, 28, 0x21ad,
	28, 0x3003, 28, 0xaf10, 28, 0x8402, 28, 0x84a4,
	28, 0xaf10, 28, 0x84f8, 28, 0xfaef, 28, 0x69fb,
	28, 0xcfbf, 28, 0x87f7, 28, 0xd819, 28, 0xd907,
	28, 0xbf54, 28, 0xf102, 28, 0x51bc, 28, 0x0719,
	28, 0xd819, 28, 0xd907, 28, 0xbf54, 28, 0xfa02,
	28, 0x51bc, 28, 0x0719, 28, 0xd819, 28, 0xd907,
	28, 0xbf53, 28, 0x4d02, 28, 0x51bc, 28, 0x0719,
	28, 0xd000, 28, 0xd907, 28, 0xbf55, 28, 0x2d02,
	28, 0x51bc, 28, 0x0719, 28, 0xd819, 28, 0xd9bf,
	28, 0x54ee, 28, 0x0251, 28, 0xbcc7, 28, 0xffef,
	28, 0x96fe, 28, 0xfc04, 28, 0xaf0f, 28, 0xccaf,
	28, 0x0f1c, 28, 0x0251, 28, 0xbc7c, 28, 0x0001,
	28, 0xbf53, 28, 0xda02, 28, 0x51bc, 28, 0xaf2f,
	28, 0xc40c, 28, 0x5469, 28, 0x00af, 28, 0x3447,
	28, 0xbf54, 28, 0x19d1, 28, 0x0202, 28, 0x51bc,
	28, 0xaf36, 28, 0x3fd1, 28, 0x10bf, 28, 0x565d,
	28, 0x0251, 28, 0xbc02, 28, 0x5985, 28, 0xef67,
	28, 0xd100, 28, 0xaf35, 28, 0x1a00, 27, 0xb818,
	28, 0x1081, 27, 0xb81a, 28, 0x0FC9, 27, 0xb81c,
	28, 0x0F1A, 27, 0xb81e, 28, 0x2FC1, 27, 0xb846,
	28, 0x3445, 27, 0xb848, 28, 0x3639, 27, 0xb84a,
	28, 0x3508, 27, 0xb84c, 28, 0xffff, 27, 0xb832,
	28, 0x0079, 27, 0x801e, 28, 0x0004, 31, 0x0D08,
	21, 0x0819, 31, 0x0D41, 17, 0x1524, 18, 0x0724,
	31, 0xd05, 17, 0xc02, 31, 0xd05, 17, 0xe02,
	31, 0xbc4, 23, 0xc022, 31, 0xd05, 17, 0xf02,
	31, 0xbc5, 16, 0xb803, 31, 0xbc4, 22, 0x8100,
	31, 0xbc4, 23, 0x8022, 31, 0xbc4, 22, 0x8100,
	31, 0x0A43, 27, 0x0000, 28, 0x0000, 31, 0x0B82,
	23, 0x0000, 27, 0x802B, 28, 0x0000, 31, 0x0B82,
	16, 0x0000,
};

static u16 rtl8211fd_vx_register_patch2[] = {
	31, 0x0a46, 21, 0x0302, 27, 0x807D, 28, 0xD06B,
	27, 0x807E, 28, 0x6ABF, 27, 0x807F, 28, 0xCF9B,
	27, 0x809A, 28, 0xB294, 27, 0x809B, 28, 0x970B,
	27, 0x809C, 28, 0xDB18, 27, 0x809D, 28, 0x04A0,
	27, 0x80D7, 28, 0x5033, 27, 0x80AA, 28, 0xF06B,
	27, 0x80AB, 28, 0x6ABF, 27, 0x80AC, 28, 0xCF4F,
	27, 0x80C7, 28, 0xB294, 27, 0x80C8, 28, 0x970B,
	27, 0x80C9, 28, 0xDB18, 27, 0x80CA, 28, 0x04A0,
	27, 0x80DB, 28, 0x5033, 27, 0x8011, 28, 0x9737,
	27, 0x804F, 28, 0x2422, 27, 0x8050, 28, 0x44E7,
	27, 0x8053, 28, 0x3300, 27, 0x805A, 28, 0x3300,
	27, 0x8061, 28, 0x3300, 27, 0x81AB, 28, 0x9069,
	31, 0x0a86, 17, 0xffff, 31, 0x0a80, 22, 0x3f46,
	31, 0x0a81, 22, 0x69a7, 31, 0x0bc0, 20, 0x3461,
	31, 0x0a92, 16, 0x0fff, 31, 0xa43, 4, 0x1e1,
	27, 0x87f7, 28, 0x700, 28, 0x1f8, 28, 0x7aff,
	27, 0x87fd, 28, 0xa00, 27, 0x87fe, 28, 0x08d3,
	31, 0xa87, 20, 0x0, 31, 0xa92, 16, 0x0001,
	31, 0xa86, 16, 0xc3c0, 31, 0xa92, 17, 0xc000,
	31, 0xa93, 21, 0x350b, 31, 0xa90, 17, 0x81a,
	31, 0xa5d, 16, 0x0, 27, 0x8165, 28, 0xb702,
	31, 0x0a46, 21, 0x0300, 27, 0x801E,
};

static int rtl821x_ack_interrupt(struct phy_device *phydev)
{
	int err;

	err = phy_read(phydev, RTL821x_INSR);

	return (err < 0) ? err : 0;
}

static int rtl8211f_ack_interrupt(struct phy_device *phydev)
{
	int err;

	phy_write(phydev, RTL8211F_PAGE_SELECT, 0xa43);
	err = phy_read(phydev, RTL8211F_INSR);
	/* restore to default page 0 */
	phy_write(phydev, RTL8211F_PAGE_SELECT, 0x0);

	return (err < 0) ? err : 0;
}

static int rtl8211b_config_intr(struct phy_device *phydev)
{
	int err;

	if (phydev->interrupts == PHY_INTERRUPT_ENABLED)
		err = phy_write(phydev, RTL821x_INER,
				RTL821x_INER_INIT);
	else
		err = phy_write(phydev, RTL821x_INER, 0);

	return err;
}

static int rtl8211e_config_intr(struct phy_device *phydev)
{
	int err;

	if (phydev->interrupts == PHY_INTERRUPT_ENABLED)
		err = phy_write(phydev, RTL821x_INER,
				RTL8211E_INER_LINK_STATUS);
	else
		err = phy_write(phydev, RTL821x_INER, 0);

	return err;
}

static int rtl8211f_config_intr(struct phy_device *phydev)
{
	int err;

	if (phydev->interrupts == PHY_INTERRUPT_ENABLED)
		err = phy_write(phydev, RTL821x_INER,
				RTL8211F_INER_LINK_STATUS);
	else
		err = phy_write(phydev, RTL821x_INER, 0);

	return err;
}

static int rtl8211f_config_init(struct phy_device *phydev)
{
	int ret;
	u16 reg;

	ret = genphy_config_init(phydev);
	if (ret < 0)
		return ret;

	phy_write(phydev, RTL8211F_PAGE_SELECT, 0xd08);
	reg = phy_read(phydev, 0x11);

	/* enable TX-delay for rgmii-id and rgmii-txid, otherwise disable it */
	if (phydev->interface == PHY_INTERFACE_MODE_RGMII_ID ||
	    phydev->interface == PHY_INTERFACE_MODE_RGMII_TXID)
		reg |= RTL8211F_TX_DELAY;
	else
		reg &= ~RTL8211F_TX_DELAY;

	phy_write(phydev, 0x11, reg);
	/* restore to default page 0 */
	phy_write(phydev, RTL8211F_PAGE_SELECT, 0x0);

	return 0;
}

/* save relevant PHY registers to private copy */
static void rtl8211fd_vx_context_save(struct phy_device *phydev,
					struct rtl8211fd_vx_context *context)
{
	context->bmcr = phy_read(phydev, MII_BMCR);
	context->advertise = phy_read(phydev, MII_ADVERTISE);
	context->control1000 = phy_read(phydev, MII_CTRL1000);
}

/* restore relevant PHY registers from private copy */
static void rtl8211fd_vx_context_restore(struct phy_device *phydev,
					   const struct rtl8211fd_vx_context *context)
{
	phy_write(phydev, MII_BMCR, context->bmcr);
	phy_write(phydev, MII_ADVERTISE, context->advertise);
	phy_write(phydev, MII_CTRL1000, context->control1000);
}

static int rtl8211fd_vx_suspend(struct phy_device *phydev)
{
	struct rtl8211fd_vx_priv *priv = phydev->priv;

	mutex_lock(&phydev->lock);

	if (phydev->attached_dev) {
		rtl8211fd_vx_context_save(phydev, &priv->save_context);
		priv->context_saved = true;
	}

	mutex_unlock(&phydev->lock);

	return 0;
}

static int rtl8211fd_vx_resume(struct phy_device *phydev)
{
	struct rtl8211fd_vx_priv *priv = phydev->priv;

	if (priv->context_saved) {
		priv->context_saved = false;
		rtl8211fd_vx_context_restore(phydev, &priv->save_context);
	}

	return 0;
}

static int rtl8211fd_vx_probe(struct phy_device *phydev)
{
	struct device *dev = &phydev->mdio.dev;
	struct rtl8211fd_vx_priv *priv;

	priv = devm_kzalloc(dev, sizeof(*priv), GFP_KERNEL);
	if (!priv)
		return -ENOMEM;

	phydev->priv = priv;

	return 0;
}

static int rtl8211fd_vx_config_init(struct phy_device *phydev)
{
	int ret, i;
	u16 reg, adr;

	ret = genphy_config_init(phydev);
	if (ret < 0)
		return ret;

	do { /* REGISTER PATCH */
		phy_write(phydev, 31, 0x0B82);
		phy_write(phydev, 16, 0x0010);
		phy_write(phydev, 31, 0x0B80);
		/* Check the PHY is ready to patch, time out if not ready after 400ms. */
		for (i = 0; i < 400; i++) {
			reg = phy_read(phydev, 16);
			if (reg & (1 << 6)) /* r 16 6 6 0x1 */
				break;
			msleep(1);
		}
		if (i >= 400) {
			pr_warn("%s: Step2 time out\n", __func__);
			break;
		}
		for (i = 0; i < ARRAY_SIZE(rtl8211fd_vx_register_patch1); i += 2) {
			adr = rtl8211fd_vx_register_patch1[i];
			reg = rtl8211fd_vx_register_patch1[i + 1];
			phy_write(phydev, adr, reg);
		}
		reg = phy_read(phydev, 16);
		if (reg & (1 << 6)) { /* r 16 6 6 0x0 // Check the PHY patched */
			pr_warn("%s: PHY patch failed\n", __func__);
			break;
		}
		for (i = 0; i < ARRAY_SIZE(rtl8211fd_vx_register_patch2); i += 2) {
			adr = rtl8211fd_vx_register_patch2[i];
			reg = rtl8211fd_vx_register_patch2[i + 1];
			phy_write(phydev, adr, reg);
		}
		reg = phy_read(phydev, 28);
		if (reg != 0x0004) { /* R 28 15 0 0x0004 // Check the value is 0x0004 */
			pr_warn("%s: Check the value failed\n", __func__);
			break;
		}
		pr_info("%s: PHY patched\n", __func__);
	} while(0);

	phy_write(phydev, RTL8211F_PAGE_SELECT, 0xd08);
	reg = phy_read(phydev, 0x11);

	/* enable TX-delay for rgmii-id and rgmii-txid, otherwise disable it */
	if (phydev->interface == PHY_INTERFACE_MODE_RGMII_ID ||
	    phydev->interface == PHY_INTERFACE_MODE_RGMII_TXID)
		reg |= RTL8211F_TX_DELAY;
	else
		reg &= ~RTL8211F_TX_DELAY;

	phy_write(phydev, 0x11, reg);
	reg = phy_read(phydev, 0x15);

	/* enable RX-delay for rgmii-id and rgmii-rxid, otherwise disable it */
	if (phydev->interface == PHY_INTERFACE_MODE_RGMII_ID ||
	    phydev->interface == PHY_INTERFACE_MODE_RGMII_RXID)
		reg |= RTL8211FD_VX_RX_DELAY;
	else
		reg &= ~RTL8211FD_VX_RX_DELAY;

	phy_write(phydev, 0x15, reg);
	/* restore to default page 0 */
	phy_write(phydev, RTL8211F_PAGE_SELECT, 0x0);

	return 0;
}

static struct phy_driver realtek_drvs[] = {
	{
		.phy_id         = 0x00008201,
		.name           = "RTL8201CP Ethernet",
		.phy_id_mask    = 0x0000ffff,
		.features       = PHY_BASIC_FEATURES,
		.flags          = PHY_HAS_INTERRUPT,
		.config_aneg    = &genphy_config_aneg,
		.read_status    = &genphy_read_status,
	}, {
		.phy_id		= 0x001cc912,
		.name		= "RTL8211B Gigabit Ethernet",
		.phy_id_mask	= 0x001fffff,
		.features	= PHY_GBIT_FEATURES,
		.flags		= PHY_HAS_INTERRUPT,
		.config_aneg	= &genphy_config_aneg,
		.read_status	= &genphy_read_status,
		.ack_interrupt	= &rtl821x_ack_interrupt,
		.config_intr	= &rtl8211b_config_intr,
	}, {
		.phy_id		= 0x001cc914,
		.name		= "RTL8211DN Gigabit Ethernet",
		.phy_id_mask	= 0x001fffff,
		.features	= PHY_GBIT_FEATURES,
		.flags		= PHY_HAS_INTERRUPT,
		.config_aneg	= genphy_config_aneg,
		.read_status	= genphy_read_status,
		.ack_interrupt	= rtl821x_ack_interrupt,
		.config_intr	= rtl8211e_config_intr,
		.suspend	= genphy_suspend,
		.resume		= genphy_resume,
	}, {
		.phy_id		= 0x001cc915,
		.name		= "RTL8211E Gigabit Ethernet",
		.phy_id_mask	= 0x001fffff,
		.features	= PHY_GBIT_FEATURES,
		.flags		= PHY_HAS_INTERRUPT,
		.config_aneg	= &genphy_config_aneg,
		.read_status	= &genphy_read_status,
		.ack_interrupt	= &rtl821x_ack_interrupt,
		.config_intr	= &rtl8211e_config_intr,
		.suspend	= genphy_suspend,
		.resume		= genphy_resume,
	}, {
		.phy_id		= 0x001cc916,
		.name		= "RTL8211F Gigabit Ethernet",
		.phy_id_mask	= 0x001fffff,
		.features	= PHY_GBIT_FEATURES,
		.flags		= PHY_HAS_INTERRUPT,
		.config_aneg	= &genphy_config_aneg,
		.config_init	= &rtl8211f_config_init,
		.read_status	= &genphy_read_status,
		.ack_interrupt	= &rtl8211f_ack_interrupt,
		.config_intr	= &rtl8211f_config_intr,
		.suspend	= genphy_suspend,
		.resume		= genphy_resume,
	}, {
		.phy_id		= 0x001cc859,
		.name		= "RTL8211FD-VX Gigabit Ethernet",
		.phy_id_mask	= 0x001ffff0,
		.features	= PHY_GBIT_FEATURES,
		.flags		= PHY_HAS_INTERRUPT,
		.probe		= &rtl8211fd_vx_probe,
		.config_aneg	= &genphy_config_aneg,
		.config_init	= &rtl8211fd_vx_config_init,
		.read_status	= &genphy_read_status,
		.ack_interrupt	= &rtl8211f_ack_interrupt,
		.config_intr	= &rtl8211f_config_intr,
		.suspend	= &rtl8211fd_vx_suspend,
		.resume		= &rtl8211fd_vx_resume,
	},
};

module_phy_driver(realtek_drvs);

static struct mdio_device_id __maybe_unused realtek_tbl[] = {
	{ 0x001cc912, 0x001fffff },
	{ 0x001cc914, 0x001fffff },
	{ 0x001cc915, 0x001fffff },
	{ 0x001cc916, 0x001fffff },
	{ 0x001cc859, 0x001ffff0 },
	{ }
};

MODULE_DEVICE_TABLE(mdio, realtek_tbl);
