/*
 * Freescale S/PDIF ALSA SoC Digital Audio Interface (DAI) driver
 *
 * Copyright (C) 2013-2016 Freescale Semiconductor, Inc.
 * Copyright 2020,2021 Sony Home Entertainment & Sound Products Inc.
 *
 * Based on stmp3xxx_spdif_dai.c
 * Vladimir Barinov <vbarinov@embeddedalley.com>
 * Copyright 2008 SigmaTel, Inc
 * Copyright 2008 Embedded Alley Solutions, Inc
 *
 * This file is licensed under the terms of the GNU General Public License
 * version 2.  This program  is licensed "as is" without any warranty of any
 * kind, whether express or implied.
 */

#include <linux/bitrev.h>
#include <linux/clk.h>
#include <linux/clk-provider.h>
#include <linux/module.h>
#include <linux/of_address.h>
#include <linux/of_device.h>
#include <linux/of_irq.h>
#include <linux/of_gpio.h>
#include <linux/gpio/consumer.h>
#include <linux/regmap.h>
#include <linux/pm_runtime.h>
#include <linux/busfreq-imx.h>
#include <linux/workqueue.h>
#include <linux/gpio.h>

#include <sound/asoundef.h>
#include <sound/dmaengine_pcm.h>
#include <sound/soc.h>

#include "fsl_spdif.h"
#include "imx-pcm.h"
#include "fsl_dma_workaround.h"

#include <misc/sw_pll.h>

#define FSL_SPDIF_TXFIFO_WML	0x8
#define FSL_SPDIF_RXFIFO_WML	0x8

#define INTR_FOR_PLAYBACK	(INT_TXFIFO_RESYNC)
#ifdef DETECT_RX_FIFO_OV
#define INTR_ADDITIONAL_RXOV	(INT_RXFIFO_UNOV | INT_RXFIFO_FUL)
#define INTR_IGNORE_UEVENT	(INTR_ADDITIONAL_RXOV | INT_URX_FUL | INT_QRX_FUL)
#define INTR_FOR_CAPTURE	(INT_SYM_ERR | INT_BIT_ERR | INT_URX_FUL |\
				INT_URX_OV | INT_QRX_FUL | INT_QRX_OV |\
				INT_UQ_SYNC | INT_UQ_ERR | INT_RXFIFO_RESYNC |\
				INT_LOSS_LOCK | INT_DPLL_LOCKED | INTR_ADDITIONAL_RXOV)
#else
#define INTR_FOR_CAPTURE	(INT_LOSS_LOCK | INT_DPLL_LOCKED)
#endif

#define SIE_INTR_FOR(tx)	(tx ? INTR_FOR_PLAYBACK : INTR_FOR_CAPTURE)

/* Index list for the values that has if (DPLL Locked) condition */
static u8 srpc_dpll_locked[] = { 0x0, 0x1, 0x2, 0x3, 0x4, 0xa, 0xb };
#define SRPC_NODPLL_START1	0x5
#define SRPC_NODPLL_START2	0xc

#define DEFAULT_RXCLK_SRC	1

struct fsl_spdif_soc_data {
	bool imx;
	bool constrain_period_size;
	bool dma_workaround;
	u32 tx_burst;
	u32 rx_burst;
	u32 interrupts;
	u64 tx_formats;
	u64 rx_rates;
};

/*
 * SPDIF control structure
 * Defines channel status, subcode and Q sub
 */
struct spdif_mixer_control {
	/* spinlock to access control data */
	spinlock_t ctl_lock;

	/* IEC958 channel tx status bit */
	unsigned char ch_status[4];

	/* User bits */
	unsigned char subcode[2 * SPDIF_UBITS_SIZE];

	/* Q subcode part of user bits */
	unsigned char qsub[2 * SPDIF_QSUB_SIZE];

	/* Buffer offset for U/Q */
	u32 upos;
	u32 qpos;

	/* Ready buffer index of the two buffers */
	u32 ready_buf;
};

/**
 * fsl_spdif_priv: Freescale SPDIF private data
 *
 * @fsl_spdif_control: SPDIF control data
 * @cpu_dai_drv: cpu dai driver
 * @pdev: platform device pointer
 * @regmap: regmap handler
 * @dpll_locked: dpll lock flag
 * @txrate: the best rates for playback
 * @txclk_df: STC_TXCLK_DF dividers value for playback
 * @sysclk_df: STC_SYSCLK_DF dividers value for playback
 * @txclk_src: STC_TXCLK_SRC values for playback
 * @rxclk_src: SRPC_CLKSRC_SEL values for capture
 * @txclk: tx clock sources for playback
 * @rxclk: rx clock sources for capture
 * @coreclk: core clock for register access via DMA
 * @sysclk: system clock for rx clock rate measurement
 * @spbaclk: SPBA clock (optional, depending on SoC design)
 * @dma_params_tx: DMA parameters for transmit channel
 * @dma_params_rx: DMA parameters for receive channel
 */
struct fsl_spdif_priv {
	struct spdif_mixer_control fsl_spdif_control;
	struct snd_soc_dai_driver cpu_dai_drv;
	struct platform_device *pdev;
	struct regmap *regmap;
	bool dpll_locked;
	u32 txrate[SPDIF_TXRATE_MAX];
	u8 txclk_df[SPDIF_TXRATE_MAX];
	u16 sysclk_df[SPDIF_TXRATE_MAX];
	u8 txclk_src[SPDIF_TXRATE_MAX];
	u8 rxclk_src;
	struct clk *txclk[STC_TXCLK_SRC_MAX];
	struct clk *rxclk;
	struct clk *coreclk;
	struct clk *sysclk;
	struct clk *spbaclk;
	const struct fsl_spdif_soc_data *soc;
	struct fsl_dma_workaround_info *dma_info;
	struct snd_dmaengine_dai_dma_data dma_params_tx;
	struct snd_dmaengine_dai_dma_data dma_params_rx;
	/* regcache for SRPC */
	u32 regcache_srpc;
	struct clk *pll8k_clk;
	struct clk *pll11k_clk;
	struct kobject *spdif_isr_kobject;
	struct workqueue_struct *wq;
	struct work_struct work;
	atomic_t irq_event;
#ifdef DETECT_RX_FIFO_OV
	bool rxfifo_stable;
	u32 rxfifo_unov_count;
#endif
	u64 cbits;
	atomic_t work_que_enable;
	struct kobject *spdif_kobj;
	int spdif_switch_s0;
	int spdif_switch_s1;
	int spdif_switch_amixer;
	char opened;
	struct mutex softreset_lock;
};

static struct fsl_spdif_priv *static_spdif_priv;

static struct fsl_spdif_soc_data fsl_spdif_vf610 = {
	.imx = false,
	.dma_workaround = false,
	.tx_burst = FSL_SPDIF_TXFIFO_WML,
	.rx_burst = FSL_SPDIF_RXFIFO_WML,
	.interrupts = 1,
	.tx_formats = FSL_SPDIF_FORMATS_PLAYBACK,
	.rx_rates = FSL_SPDIF_RATES_CAPTURE,
	.constrain_period_size = false,
};

static struct fsl_spdif_soc_data fsl_spdif_imx35 = {
	.imx = true,
	.dma_workaround = false,
	.tx_burst = FSL_SPDIF_TXFIFO_WML,
	.rx_burst = FSL_SPDIF_RXFIFO_WML,
	.interrupts = 1,
	.tx_formats = FSL_SPDIF_FORMATS_PLAYBACK,
	.rx_rates = FSL_SPDIF_RATES_CAPTURE,
	.constrain_period_size = false,
};

/*
 * In imx8qxp rev 1, the DMA request signal is not reverted. For SPDIF
 * DMA request is low valid, but EDMA assert is high valid, so we
 * need to use GPT to transfer the DMA request signal
 */
static struct fsl_spdif_soc_data fsl_spdif_imx8qxp_v1 = {
	.imx = true,
	.dma_workaround = true,
	.tx_burst = 2,
	.rx_burst = 2,
	.interrupts = 2,
	.tx_formats = SNDRV_PCM_FMTBIT_S24_LE,
	.rx_rates = FSL_SPDIF_RATES_CAPTURE,
	.constrain_period_size = true,
};

static struct fsl_spdif_soc_data fsl_spdif_imx8qm = {
	.imx = true,
	.dma_workaround = false,
	.tx_burst = 2,
	.rx_burst = 2,
	.interrupts = 2,
	.tx_formats = SNDRV_PCM_FMTBIT_S24_LE,
	.rx_rates = (FSL_SPDIF_RATES_CAPTURE | SNDRV_PCM_RATE_192000),
	.constrain_period_size = true,
};

static struct fsl_spdif_soc_data fsl_spdif_imx8mm = {
	.imx = true,
	.dma_workaround = false,
	.tx_burst = FSL_SPDIF_TXFIFO_WML,
	.rx_burst = FSL_SPDIF_RXFIFO_WML,
	.interrupts = 1,
	.tx_formats = FSL_SPDIF_FORMATS_PLAYBACK,
	.rx_rates = (FSL_SPDIF_RATES_CAPTURE | SNDRV_PCM_RATE_88200 |
			 SNDRV_PCM_RATE_176400 | SNDRV_PCM_RATE_192000 ),
	.constrain_period_size = false,
};

/* DPLL locked and lock loss interrupt handler */
static void spdif_irq_dpll_lock(struct fsl_spdif_priv *spdif_priv)
{
	struct regmap *regmap = spdif_priv->regmap;
/*	struct platform_device *pdev = spdif_priv->pdev; */
	u32 locked;

	regmap_read(regmap, REG_SPDIF_SRPC, &locked);
	locked &= SRPC_DPLL_LOCKED;

/*	dev_dbg(&pdev->dev, "isr: Rx dpll %s \n",
			locked ? "locked" : "loss lock"); */

	spdif_priv->dpll_locked = locked ? true : false;
}

/* Receiver found illegal symbol interrupt handler */
static void spdif_irq_sym_error(struct fsl_spdif_priv *spdif_priv)
{
	struct regmap *regmap = spdif_priv->regmap;
/*	struct platform_device *pdev = spdif_priv->pdev; */

/*	dev_dbg(&pdev->dev, "isr: receiver found illegal symbol\n"); */

	/* Clear illegal symbol if DPLL unlocked since no audio stream */
	if (!spdif_priv->dpll_locked)
		regmap_update_bits(regmap, REG_SPDIF_SIE, INT_SYM_ERR, 0);
}

/* U/Q Channel receive register full */
static void spdif_irq_uqrx_full(struct fsl_spdif_priv *spdif_priv, char name)
{
	struct spdif_mixer_control *ctrl = &spdif_priv->fsl_spdif_control;
	struct regmap *regmap = spdif_priv->regmap;
	struct platform_device *pdev = spdif_priv->pdev;
	u32 *pos, size, val, reg;

	switch (name) {
	case 'U':
		pos = &ctrl->upos;
		size = SPDIF_UBITS_SIZE;
		reg = REG_SPDIF_SRU;
		break;
	case 'Q':
		pos = &ctrl->qpos;
		size = SPDIF_QSUB_SIZE;
		reg = REG_SPDIF_SRQ;
		break;
	default:
		dev_err(&pdev->dev, "unsupported channel name\n");
		return;
	}

/*	dev_dbg(&pdev->dev, "isr: %c Channel receive register full\n", name); */

	if (*pos >= size * 2) {
		*pos = 0;
	} else if (unlikely((*pos % size) + 3 > size)) {
		dev_err(&pdev->dev, "User bit receive buffer overflow\n");
		return;
	}

	regmap_read(regmap, reg, &val);
	ctrl->subcode[*pos++] = val >> 16;
	ctrl->subcode[*pos++] = val >> 8;
	ctrl->subcode[*pos++] = val;
}

/* U/Q Channel sync found */
static void spdif_irq_uq_sync(struct fsl_spdif_priv *spdif_priv)
{
	struct spdif_mixer_control *ctrl = &spdif_priv->fsl_spdif_control;
/*	struct platform_device *pdev = spdif_priv->pdev; */

/*	dev_dbg(&pdev->dev, "isr: U/Q Channel sync found\n"); */

	/* U/Q buffer reset */
	if (ctrl->qpos == 0)
		return;

	/* Set ready to this buffer */
	ctrl->ready_buf = (ctrl->qpos - 1) / SPDIF_QSUB_SIZE + 1;
}

/* U/Q Channel framing error */
static void spdif_irq_uq_err(struct fsl_spdif_priv *spdif_priv)
{
	struct spdif_mixer_control *ctrl = &spdif_priv->fsl_spdif_control;
	struct regmap *regmap = spdif_priv->regmap;
/*	struct platform_device *pdev = spdif_priv->pdev; */
	u32 val;

/*	dev_dbg(&pdev->dev, "isr: U/Q Channel framing error\n"); */

	/* Read U/Q data to clear the irq and do buffer reset */
	regmap_read(regmap, REG_SPDIF_SRU, &val);
	regmap_read(regmap, REG_SPDIF_SRQ, &val);

	/* Drop this U/Q buffer */
	ctrl->ready_buf = 0;
	ctrl->upos = 0;
	ctrl->qpos = 0;
}

#ifdef DETECT_RX_FIFO_OV
static void spdif_irq_rxfifo_unov(struct fsl_spdif_priv *spdif_priv)
{
	struct regmap *regmap = spdif_priv->regmap;
	struct platform_device *pdev = spdif_priv->pdev;

	if (spdif_priv->rxfifo_stable) {
		regmap_update_bits(regmap, REG_SPDIF_SIE, INT_RXFIFO_UNOV, 0);
		spdif_priv->rxfifo_unov_count++;
		spdif_priv->rxfifo_stable = false;
		dev_err(&pdev->dev,
			"isr: Rx FIFO under/overrun during capture(%u)\n",
			spdif_priv->rxfifo_unov_count);
	}
}

static void spdif_irq_rxfifo_ful(struct fsl_spdif_priv *spdif_priv, u32 sis)
{
	struct regmap *regmap = spdif_priv->regmap;

	if (!(sis & INT_RXFIFO_UNOV)) {
		regmap_update_bits(regmap, REG_SPDIF_SIE, INT_RXFIFO_FUL, 0);
		spdif_priv->rxfifo_stable = true;
	}
}
#endif

/* Get spdif interrupt status and clear the interrupt */
static u32 spdif_intr_status_clear(struct fsl_spdif_priv *spdif_priv)
{
	struct regmap *regmap = spdif_priv->regmap;
	u32 val, val2;

	regmap_read(regmap, REG_SPDIF_SIS, &val);
	regmap_read(regmap, REG_SPDIF_SIE, &val2);

	regmap_write(regmap, REG_SPDIF_SIC, val & val2);

	return val;
}

static int spdif_softreset(struct fsl_spdif_priv *spdif_priv);

int fsl_spdif_softreset_static(void)
{
	struct fsl_spdif_priv *spdif_priv = static_spdif_priv;
	struct platform_device *pdev = spdif_priv->pdev;
	struct regmap *regmap = spdif_priv->regmap;
	u32 origin_scr, origin_sie, origin_srpc;
	int ret;
	u32 work_que_enable;

	mutex_lock(&spdif_priv->softreset_lock);

	regmap_read(regmap, REG_SPDIF_SCR, &origin_scr);
	regmap_read(regmap, REG_SPDIF_SRPC, &origin_srpc);
	regmap_read(regmap, REG_SPDIF_SIE, &origin_sie);
	/* Disable all the interrupts before softreset */
	regmap_update_bits(regmap, REG_SPDIF_SIE, 0xffffff, 0);
	ret = spdif_softreset(spdif_priv);
	if (ret)
		dev_err(&pdev->dev, "soft reset fail\n");
	spdif_intr_status_clear(spdif_priv);
	work_que_enable = atomic_read(&spdif_priv->work_que_enable);
	if (work_que_enable == 0) {
		mutex_unlock(&spdif_priv->softreset_lock);
		dev_dbg(&pdev->dev, "spdif workque don't enable after softreset\n");
		return ret;
	}
	regmap_write(regmap, REG_SPDIF_SCR, origin_scr);
	regmap_write(regmap, REG_SPDIF_SRPC, origin_srpc);
	regmap_write(regmap, REG_SPDIF_SIE, origin_sie);

	mutex_unlock(&spdif_priv->softreset_lock);

	return ret;
}

static void spdif_detected_wq(struct work_struct *w)
{
	struct fsl_spdif_priv *spdif_priv = container_of(w, struct fsl_spdif_priv, work);
	struct platform_device *pdev = spdif_priv->pdev;
	u32 irq_event;
	char event[80] = "EVENT=SPDIF_ISR";
	u32 work_que_enable;

	work_que_enable = atomic_read(&spdif_priv->work_que_enable);
	if (work_que_enable == 0) {
		dev_dbg(&pdev->dev, "spdif workque don't enable\n");
		return;
	}

	irq_event = atomic_read(&spdif_priv->irq_event);
	atomic_set(&spdif_priv->irq_event, 0);

	if (irq_event & INT_DPLL_LOCKED)
		strncat(event, ",LOCK", 5);
	if (irq_event & INT_CNEW)
		strncat(event, ",CNEW", 5);
	if (irq_event & INT_BIT_ERR)
		strncat(event, ",BITERR", 7);
	if (irq_event & INT_LOSS_LOCK)
		strncat(event, ",ULOCK", 6);

	dev_dbg(&pdev->dev, "spdif uevent event=%x %s\n", irq_event, event);

	if (irq_event & INT_LOSS_LOCK) {
		//call softreset for continue dpll issue
		fsl_spdif_softreset_static();
	}
	/* notify userspace */
#ifdef CONFIG_SONY_SW_PLL
	sw_pll_set_spdif_event(irq_event);
#endif
}

static irqreturn_t spdif_isr(int irq, void *devid)
{
	struct fsl_spdif_priv *spdif_priv = (struct fsl_spdif_priv *)devid;
	struct platform_device *pdev = spdif_priv->pdev;
	u32 sis;
	u32 sie;
	unsigned int count = 0;
#ifdef DETECT_RX_FIFO_OV
	u32 sie;
#endif
	bool need_uevent = false;
	u32 tmp_event;
	u32 irq_event = 0;
	u32 work_que_enable;

	work_que_enable = atomic_read(&spdif_priv->work_que_enable);

#ifdef DETECT_RX_FIFO_OV
	regmap_read(spdif_priv->regmap, REG_SPDIF_SIE, &sie);
#endif

	do {

	regmap_read(spdif_priv->regmap, REG_SPDIF_SIE, &sie);
	sis = spdif_intr_status_clear(spdif_priv);
	++count;
	dev_dbg(&pdev->dev, "isr: sis = %08x, masked = %08x (%d)\n", sis, sis & sie, count);
	sis &= sie;

	if (sis & INT_DPLL_LOCKED) {
		spdif_irq_dpll_lock(spdif_priv);
		need_uevent = true;
		irq_event |= INT_DPLL_LOCKED;
	}

	if (sis & INT_TXFIFO_UNOV)
/*		dev_dbg(&pdev->dev, "isr: Tx FIFO under/overrun\n"); */ ;

	if (sis & INT_TXFIFO_RESYNC)
/*		dev_dbg(&pdev->dev, "isr: Tx FIFO resync\n"); */ ;

	if (sis & INT_CNEW)
/*		dev_dbg(&pdev->dev, "isr: cstatus new\n"); */ ;

	if (sis & INT_VAL_NOGOOD)
/*		dev_dbg(&pdev->dev, "isr: validity flag no good\n"); */ ;

	if (sis & INT_SYM_ERR)
		spdif_irq_sym_error(spdif_priv);

	if (sis & INT_BIT_ERR)
/*		dev_dbg(&pdev->dev, "isr: receiver found parity bit error\n"); */ ;

	if (sis & INT_URX_FUL)
		spdif_irq_uqrx_full(spdif_priv, 'U');

	if (sis & INT_URX_OV)
/*		dev_dbg(&pdev->dev, "isr: U Channel receive register overrun\n"); */ ;

	if (sis & INT_QRX_FUL)
		spdif_irq_uqrx_full(spdif_priv, 'Q');

	if (sis & INT_QRX_OV)
/*		dev_dbg(&pdev->dev, "isr: Q Channel receive register overrun\n"); */ ;

	if (sis & INT_UQ_SYNC)
		spdif_irq_uq_sync(spdif_priv);

	if (sis & INT_UQ_ERR)
		spdif_irq_uq_err(spdif_priv);

#ifdef DETECT_RX_FIFO_OV
	if (sis & sie & INT_RXFIFO_UNOV) {
		spdif_irq_rxfifo_unov(spdif_priv);
		dev_dbg(&pdev->dev, "isr: Rx FIFO under/overrun\n");
	}
#else
	if (sis & INT_RXFIFO_UNOV)
/*		dev_dbg(&pdev->dev, "isr: Rx FIFO under/overrun\n"); */ ;
#endif

	if (sis & INT_RXFIFO_RESYNC)
/*		dev_dbg(&pdev->dev, "isr: Rx FIFO resync\n"); */ ;

	if (sis & INT_LOSS_LOCK) {
		spdif_irq_dpll_lock(spdif_priv);
		need_uevent = true;
		irq_event |= INT_LOSS_LOCK;
	}

	/* FIXME: Write Tx FIFO to clear TxEm */
	if (sis & INT_TX_EM)
/*		dev_dbg(&pdev->dev, "isr: Tx FIFO empty\n"); */ ;

	/* FIXME: Read Rx FIFO to clear RxFIFOFul */
#ifdef DETECT_RX_FIFO_OV
	if (sis & sie & INT_RXFIFO_FUL) {
		spdif_irq_rxfifo_ful(spdif_priv, sis);
		dev_dbg(&pdev->dev, "isr: Rx FIFO full\n");
	}
#else
	if (sis & INT_RXFIFO_FUL)
/*		dev_dbg(&pdev->dev, "isr: Rx FIFO full\n"); */ ;
#endif

#ifdef DETECT_RX_FIFO_OV
	if (!(sis & sie & ~INTR_IGNORE_UEVENT))
		need_uevent = false;
#endif

	if (need_uevent && work_que_enable) {
		tmp_event = atomic_read(&spdif_priv->irq_event);
		atomic_set(&spdif_priv->irq_event, tmp_event | irq_event);
		queue_work(spdif_priv->wq, &spdif_priv->work);
	}

	need_uevent = false;
	irq_event = 0;
	regmap_read(spdif_priv->regmap, REG_SPDIF_SIE, &sie);
	regmap_read(spdif_priv->regmap, REG_SPDIF_SIS, &sis);
	} while (count < 10 && (sis & sie & ~(INT_URX_FUL | INT_QRX_FUL | INT_TX_EM | INT_RXFIFO_FUL)));

	return IRQ_HANDLED;
}

static int spdif_softreset(struct fsl_spdif_priv *spdif_priv)
{
	struct regmap *regmap = spdif_priv->regmap;
	u32 val, cycle = 1000;

	regcache_cache_bypass(regmap, true);

	regmap_write(regmap, REG_SPDIF_SCR, SCR_SOFT_RESET);

	/*
	 * RESET bit would be cleared after finishing its reset procedure,
	 * which typically lasts 8 cycles. 1000 cycles will keep it safe.
	 */
	do {
		regmap_read(regmap, REG_SPDIF_SCR, &val);
	} while ((val & SCR_SOFT_RESET) && cycle--);

	regcache_cache_bypass(regmap, false);
	regcache_mark_dirty(regmap);
	regcache_sync(regmap);

	if (cycle)
		return 0;
	else
		return -EBUSY;
}

static void spdif_set_cstatus(struct spdif_mixer_control *ctrl,
				u8 mask, u8 cstatus)
{
	ctrl->ch_status[3] &= ~mask;
	ctrl->ch_status[3] |= cstatus & mask;
}

static void spdif_write_channel_status(struct fsl_spdif_priv *spdif_priv)
{
	struct spdif_mixer_control *ctrl = &spdif_priv->fsl_spdif_control;
	struct regmap *regmap = spdif_priv->regmap;
	struct platform_device *pdev = spdif_priv->pdev;
	u32 ch_status;

	ch_status = (bitrev8(ctrl->ch_status[0]) << 16) |
		    (bitrev8(ctrl->ch_status[1]) << 8) |
		    bitrev8(ctrl->ch_status[2]);
	regmap_write(regmap, REG_SPDIF_STCSCH, ch_status);

	dev_dbg(&pdev->dev, "STCSCH: 0x%06x\n", ch_status);

	ch_status = bitrev8(ctrl->ch_status[3]) << 16;
	regmap_write(regmap, REG_SPDIF_STCSCL, ch_status);

	dev_dbg(&pdev->dev, "STCSCL: 0x%06x\n", ch_status);
}

/* Set SPDIF PhaseConfig register for rx clock */
static int spdif_set_rx_clksrc(struct fsl_spdif_priv *spdif_priv,
				enum spdif_gainsel gainsel, int dpll_locked)
{
	struct regmap *regmap = spdif_priv->regmap;
	u8 clksrc = spdif_priv->rxclk_src;

	if (clksrc >= SRPC_CLKSRC_MAX || gainsel >= GAINSEL_MULTI_MAX)
		return -EINVAL;

	regmap_update_bits(regmap, REG_SPDIF_SRPC,
			SRPC_CLKSRC_SEL_MASK | SRPC_GAINSEL_MASK,
			SRPC_CLKSRC_SEL_SET(clksrc) | SRPC_GAINSEL_SET(gainsel));

	return 0;
}

static int spdif_set_sample_rate(struct snd_pcm_substream *substream,
				int sample_rate)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct fsl_spdif_priv *spdif_priv = snd_soc_dai_get_drvdata(rtd->cpu_dai);
	struct spdif_mixer_control *ctrl = &spdif_priv->fsl_spdif_control;
	struct regmap *regmap = spdif_priv->regmap;
	struct platform_device *pdev = spdif_priv->pdev;
	unsigned long csfs = 0;
	u32 stc, mask, rate;
	u8 clk, txclk_df;
	u16 sysclk_df;

	switch (sample_rate) {
	case 32000:
		rate = SPDIF_TXRATE_32000;
		csfs = IEC958_AES3_CON_FS_32000;
		break;
	case 44100:
		rate = SPDIF_TXRATE_44100;
		csfs = IEC958_AES3_CON_FS_44100;
		break;
	case 48000:
		rate = SPDIF_TXRATE_48000;
		csfs = IEC958_AES3_CON_FS_48000;
		break;
	case 96000:
		rate = SPDIF_TXRATE_96000;
		csfs = IEC958_AES3_CON_FS_96000;
		break;
	case 192000:
		rate = SPDIF_TXRATE_192000;
		csfs = IEC958_AES3_CON_FS_192000;
		break;
	default:
		dev_err(&pdev->dev, "unsupported sample rate %d\n", sample_rate);
		return -EINVAL;
	}

	clk = spdif_priv->txclk_src[rate];
	if (clk >= STC_TXCLK_SRC_MAX) {
		dev_err(&pdev->dev, "tx clock source is out of range\n");
		return -EINVAL;
	}

	txclk_df = spdif_priv->txclk_df[rate];
	if (txclk_df == 0) {
		dev_err(&pdev->dev, "the txclk_df can't be zero\n");
		return -EINVAL;
	}

	sysclk_df = spdif_priv->sysclk_df[rate];

	dev_dbg(&pdev->dev, "expected clock rate = %d\n",
			(64 * sample_rate * txclk_df * sysclk_df));
	dev_dbg(&pdev->dev, "actual clock rate = %ld\n",
			clk_get_rate(spdif_priv->txclk[clk]));

	/* set fs field in consumer channel status */
	spdif_set_cstatus(ctrl, IEC958_AES3_CON_FS, csfs);

	/* select clock source and divisor */
	stc = STC_TXCLK_ALL_EN | STC_TXCLK_SRC_SET(clk) |
	      STC_TXCLK_DF(txclk_df) | STC_SYSCLK_DF(sysclk_df);
	mask = STC_TXCLK_ALL_EN_MASK | STC_TXCLK_SRC_MASK |
	       STC_TXCLK_DF_MASK | STC_SYSCLK_DF_MASK;
	regmap_update_bits(regmap, REG_SPDIF_STC, mask, stc);

	dev_dbg(&pdev->dev, "set sample rate to %dHz for %dHz playback\n",
			spdif_priv->txrate[rate], sample_rate);

	return 0;
}

static int fsl_spdif_startup(struct snd_pcm_substream *substream,
			     struct snd_soc_dai *cpu_dai)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct fsl_spdif_priv *spdif_priv = snd_soc_dai_get_drvdata(rtd->cpu_dai);
	struct platform_device *pdev = spdif_priv->pdev;
	struct regmap *regmap = spdif_priv->regmap;
	u32 scr, mask;
	int ret;

	/* Reset module and interrupts only for first initialization */
	if (!cpu_dai->active) {
		mutex_lock(&spdif_priv->softreset_lock);
		ret = spdif_softreset(spdif_priv);
		if (ret) {
			mutex_unlock(&spdif_priv->softreset_lock);
			dev_err(&pdev->dev, "failed to soft reset\n");
			goto err;
		}
		spdif_intr_status_clear(spdif_priv);

		/* Disable all the interrupts */
		regmap_update_bits(regmap, REG_SPDIF_SIE, 0xffffff, 0);
		mutex_unlock(&spdif_priv->softreset_lock);
	}

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
		scr = SCR_TXFIFO_AUTOSYNC | SCR_TXFIFO_CTRL_NORMAL |
			SCR_TXSEL_NORMAL | SCR_USRC_SEL_CHIP |
			SCR_TXFIFO_FSEL_IF8;
		mask = SCR_TXFIFO_AUTOSYNC_MASK | SCR_TXFIFO_CTRL_MASK |
			SCR_TXSEL_MASK | SCR_USRC_SEL_MASK |
			SCR_TXFIFO_FSEL_MASK;
	} else {
		scr = SCR_RXFIFO_FSEL_IF8 | SCR_RXFIFO_AUTOSYNC;
		mask = SCR_RXFIFO_FSEL_MASK | SCR_RXFIFO_AUTOSYNC_MASK|
			SCR_RXFIFO_CTL_MASK | SCR_RXFIFO_OFF_MASK;
	}
	regmap_update_bits(regmap, REG_SPDIF_SCR, mask, scr);

	/* Power up SPDIF module */
	regmap_update_bits(regmap, REG_SPDIF_SCR, SCR_LOW_POWER, 0);

	if (spdif_priv->soc->constrain_period_size) {
		if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK)
			snd_pcm_hw_constraint_step(substream->runtime, 0,
				SNDRV_PCM_HW_PARAM_PERIOD_SIZE,
				spdif_priv->dma_params_tx.maxburst);
		else
			snd_pcm_hw_constraint_step(substream->runtime, 0,
				SNDRV_PCM_HW_PARAM_PERIOD_SIZE,
				spdif_priv->dma_params_rx.maxburst);
	}

	if (spdif_priv->wq)
		atomic_set(&spdif_priv->work_que_enable, 1);

	spdif_priv->opened |= 0x1;

	return 0;

err:
	return ret;
}

static void fsl_spdif_shutdown(struct snd_pcm_substream *substream,
				struct snd_soc_dai *cpu_dai)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct fsl_spdif_priv *spdif_priv = snd_soc_dai_get_drvdata(rtd->cpu_dai);
	struct regmap *regmap = spdif_priv->regmap;
	u32 scr, mask;

	if (spdif_priv->wq) {
		regmap_update_bits(regmap, REG_SPDIF_SIE, 0xffffff, 0);
		atomic_set(&spdif_priv->work_que_enable, 0);
		flush_workqueue(spdif_priv->wq);
	}

	spdif_priv->dpll_locked = false;

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
		scr = 0;
		mask = SCR_TXFIFO_AUTOSYNC_MASK | SCR_TXFIFO_CTRL_MASK |
			SCR_TXSEL_MASK | SCR_USRC_SEL_MASK |
			SCR_TXFIFO_FSEL_MASK;
	} else {
		scr = SCR_RXFIFO_OFF | SCR_RXFIFO_CTL_ZERO;
		mask = SCR_RXFIFO_FSEL_MASK | SCR_RXFIFO_AUTOSYNC_MASK|
			SCR_RXFIFO_CTL_MASK | SCR_RXFIFO_OFF_MASK;
	}
	regmap_update_bits(regmap, REG_SPDIF_SCR, mask, scr);

	/* Power down SPDIF module only if tx&rx are both inactive */
	if (!cpu_dai->active) {
		spdif_intr_status_clear(spdif_priv);
		regmap_update_bits(regmap, REG_SPDIF_SCR,
				SCR_LOW_POWER, SCR_LOW_POWER);
	}

	spdif_priv->opened = 0;

}

static int fsl_spdif_hw_params(struct snd_pcm_substream *substream,
				struct snd_pcm_hw_params *params,
				struct snd_soc_dai *dai)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct fsl_spdif_priv *spdif_priv = snd_soc_dai_get_drvdata(rtd->cpu_dai);
	struct spdif_mixer_control *ctrl = &spdif_priv->fsl_spdif_control;
	struct platform_device *pdev = spdif_priv->pdev;
	u32 sample_rate = params_rate(params);
	int ret = 0;

	if (spdif_priv->soc->dma_workaround)
		configure_gpt_dma(substream, spdif_priv->dma_info);

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
		ret  = spdif_set_sample_rate(substream, sample_rate);
		if (ret) {
			dev_err(&pdev->dev, "%s: set sample rate failed: %d\n",
					__func__, sample_rate);
			return ret;
		}
		spdif_set_cstatus(ctrl, IEC958_AES3_CON_CLOCK,
				  IEC958_AES3_CON_CLOCK_1000PPM);
		spdif_write_channel_status(spdif_priv);
	} else {
		/* Setup rx clock source */
		ret = spdif_set_rx_clksrc(spdif_priv, SPDIF_DEFAULT_GAINSEL, 1);
	}

	spdif_priv->opened |= 0x2;

	return ret;
}

static int fsl_spdif_trigger(struct snd_pcm_substream *substream,
				int cmd, struct snd_soc_dai *dai)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct fsl_spdif_priv *spdif_priv = snd_soc_dai_get_drvdata(rtd->cpu_dai);
	struct regmap *regmap = spdif_priv->regmap;
	bool tx = substream->stream == SNDRV_PCM_STREAM_PLAYBACK;
	u32 intr = SIE_INTR_FOR(tx);
	u32 dmaen = SCR_DMA_xX_EN(tx);
#ifdef DETECT_RX_FIFO_OV
	struct spdif_mixer_control *ctrl = &spdif_priv->fsl_spdif_control;
	unsigned long flags;
#endif

	switch (cmd) {
	case SNDRV_PCM_TRIGGER_START:
	case SNDRV_PCM_TRIGGER_RESUME:
	case SNDRV_PCM_TRIGGER_PAUSE_RELEASE:
#ifdef DETECT_RX_FIFO_OV
		spdif_priv->rxfifo_stable = false;
#endif
		regmap_update_bits(regmap, REG_SPDIF_SIE, intr, intr);
		regmap_update_bits(regmap, REG_SPDIF_SCR, dmaen, dmaen);
		break;
	case SNDRV_PCM_TRIGGER_STOP:
	case SNDRV_PCM_TRIGGER_SUSPEND:
	case SNDRV_PCM_TRIGGER_PAUSE_PUSH:
#ifdef DETECT_RX_FIFO_OV
		spin_lock_irqsave(&ctrl->ctl_lock, flags);
#endif
		regmap_update_bits(regmap, REG_SPDIF_SCR, dmaen, 0);
		regmap_update_bits(regmap, REG_SPDIF_SIE, intr, 0);
#ifdef DETECT_RX_FIFO_OV
		spin_unlock_irqrestore(&ctrl->ctl_lock, flags);
#endif
		break;
	default:
		return -EINVAL;
	}

	if ((cmd == SNDRV_PCM_TRIGGER_START) || (cmd == SNDRV_PCM_TRIGGER_RESUME) || (cmd == SNDRV_PCM_TRIGGER_PAUSE_RELEASE))
		spdif_priv->opened |= 0x4;
	if ((cmd == SNDRV_PCM_TRIGGER_STOP) || (cmd == SNDRV_PCM_TRIGGER_SUSPEND) || (cmd == SNDRV_PCM_TRIGGER_PAUSE_PUSH))
		spdif_priv->opened |= 0x8;

	return 0;
}

static int fsl_spdif_hw_free(struct snd_pcm_substream *substream,
			     struct snd_soc_dai *dai)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct fsl_spdif_priv *spdif_priv = snd_soc_dai_get_drvdata(rtd->cpu_dai);

	if (spdif_priv->soc->dma_workaround)
		clear_gpt_dma(substream, spdif_priv->dma_info);

	return 0;
}

static u32 fsl_spdif_txclk_caldiv(struct fsl_spdif_priv *spdif_priv,
				struct clk *clk, u64 savesub,
				enum spdif_txrate index, bool round)
{
	const u32 rate[] = { 32000, 44100, 48000, 96000, 192000 };
	bool is_sysclk = clk_is_match(clk, spdif_priv->sysclk);
	u64 rate_actual, sub;
	u32 arate;
	u16 sysclk_dfmin, sysclk_dfmax, sysclk_df;
	u8 txclk_df;

	/* The sysclk has an extra divisor [2, 512] */
	sysclk_dfmin = is_sysclk ? 2 : 1;
	sysclk_dfmax = is_sysclk ? 512 : 1;

	for (sysclk_df = sysclk_dfmin; sysclk_df <= sysclk_dfmax; sysclk_df++) {
		for (txclk_df = 1; txclk_df <= 128; txclk_df++) {
			rate_actual = clk_get_rate(clk);

			arate = rate_actual / 64;
			arate /= txclk_df * sysclk_df;

			if (arate == rate[index]) {
				/* We are lucky */
				savesub = 0;
				spdif_priv->txclk_df[index] = txclk_df;
				spdif_priv->sysclk_df[index] = sysclk_df;
				spdif_priv->txrate[index] = arate;
				goto out;
			} else if (arate / rate[index] == 1) {
				/* A little bigger than expect */
				sub = (u64)(arate - rate[index]) * 100000;
				do_div(sub, rate[index]);
				if (sub >= savesub)
					continue;
				savesub = sub;
				spdif_priv->txclk_df[index] = txclk_df;
				spdif_priv->sysclk_df[index] = sysclk_df;
				spdif_priv->txrate[index] = arate;
			} else if (rate[index] / arate == 1) {
				/* A little smaller than expect */
				sub = (u64)(rate[index] - arate) * 100000;
				do_div(sub, rate[index]);
				if (sub >= savesub)
					continue;
				savesub = sub;
				spdif_priv->txclk_df[index] = txclk_df;
				spdif_priv->sysclk_df[index] = sysclk_df;
				spdif_priv->txrate[index] = arate;
			}
		}
	}

out:
	return savesub;
}

static int fsl_spdif_probe_txclk(struct fsl_spdif_priv *spdif_priv,
				enum spdif_txrate index)
{
	const u32 rate[] = { 32000, 44100, 48000, 96000, 192000 };
	struct platform_device *pdev = spdif_priv->pdev;
	struct device *dev = &pdev->dev;
	u64 savesub = 100000, ret;
	struct clk *clk;
	int i;

	for (i = 0; i < STC_TXCLK_SRC_MAX; i++) {
		clk = spdif_priv->txclk[i];
		if (IS_ERR(clk)) {
			dev_err(dev, "no rxtx%d clock in devicetree\n", i);
			return PTR_ERR(clk);
		}
		if (!clk_get_rate(clk))
			continue;

		ret = fsl_spdif_txclk_caldiv(spdif_priv, clk, savesub, index,
					     i == STC_TXCLK_SPDIF_ROOT);
		if (savesub == ret)
			continue;

		savesub = ret;
		spdif_priv->txclk_src[index] = i;

		/* To quick catch a divisor, we allow a 0.1% deviation */
		if (savesub < 100)
			break;
	}

	dev_dbg(&pdev->dev, "use rxtx%d as tx clock source for %dHz sample rate\n",
			spdif_priv->txclk_src[index], rate[index]);
	dev_dbg(&pdev->dev, "use txclk df %d for %dHz sample rate\n",
			spdif_priv->txclk_df[index], rate[index]);
	if (clk_is_match(spdif_priv->txclk[spdif_priv->txclk_src[index]], spdif_priv->sysclk))
		dev_dbg(&pdev->dev, "use sysclk df %d for %dHz sample rate\n",
				spdif_priv->sysclk_df[index], rate[index]);
	dev_dbg(&pdev->dev, "the best rate for %dHz sample rate is %dHz\n",
			rate[index], spdif_priv->txrate[index]);

	return 0;
}

static int fsl_spdif_set_dai_sysclk(struct snd_soc_dai *cpu_dai,
		int clk_id, unsigned int freq, int dir)
{
	struct fsl_spdif_priv *data = snd_soc_dai_get_drvdata(cpu_dai);
	struct platform_device *pdev = data->pdev;
	struct device *dev = &pdev->dev;
	struct clk *clk, *p, *pll = 0, *npll = 0;
	u64 ratio = freq;
	int ret, i;

	if (dir != SND_SOC_CLOCK_OUT || freq == 0 || clk_id != STC_TXCLK_SPDIF_ROOT)
		return 0;

	if (data->pll8k_clk == NULL || data->pll11k_clk == NULL)
		return 0;

	clk = data->txclk[clk_id];
	if (IS_ERR_OR_NULL(clk)) {
		dev_err(dev, "no rxtx%d clock in devicetree\n", clk_id);
		return PTR_ERR(clk);
	}

	p = clk;
	while (p && data->pll8k_clk && data->pll11k_clk) {
		struct clk *pp = clk_get_parent(p);

		if (clk_is_match(pp, data->pll8k_clk) ||
		    clk_is_match(pp, data->pll11k_clk)) {
			pll = pp;
			break;
		}
		p = pp;
	}

	if (pll) {
		npll = (do_div(ratio, 8000) ? data->pll11k_clk : data->pll8k_clk);
		if (!clk_is_match(pll, npll)) {
			ret = clk_set_parent(p, npll);
			if (ret < 0)
				dev_warn(cpu_dai->dev,
					"failed to set parent %s: %d\n",
					__clk_get_name(npll), ret);
		}
	}

	ret = clk_set_rate(clk, freq);
	if (ret < 0) {
		dev_err(cpu_dai->dev, "failed to set clock rate (%u): %d\n",
			freq, ret);
		return ret;
	}

	for (i = 0; i < SPDIF_TXRATE_MAX; i++) {
		ret = fsl_spdif_probe_txclk(data, i);
		if (ret)
			return ret;
	}

	return 0;
}

static const struct snd_soc_dai_ops fsl_spdif_dai_ops = {
	.startup = fsl_spdif_startup,
	.set_sysclk = fsl_spdif_set_dai_sysclk,
	.hw_params = fsl_spdif_hw_params,
	.trigger = fsl_spdif_trigger,
	.shutdown = fsl_spdif_shutdown,
	.hw_free = fsl_spdif_hw_free,
};

/*
 * FSL SPDIF IEC958 controller(mixer) functions
 *
 *	Channel status get/put control
 *	User bit value get/put control
 *	Valid bit value get control
 *	DPLL lock status get control
 *	User bit sync mode selection control
 */

static int fsl_spdif_info(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_info *uinfo)
{
	uinfo->type = SNDRV_CTL_ELEM_TYPE_IEC958;
	uinfo->count = 1;

	return 0;
}

static int fsl_spdif_pb_get(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *uvalue)
{
	struct snd_soc_dai *cpu_dai = snd_kcontrol_chip(kcontrol);
	struct fsl_spdif_priv *spdif_priv = snd_soc_dai_get_drvdata(cpu_dai);
	struct spdif_mixer_control *ctrl = &spdif_priv->fsl_spdif_control;

	uvalue->value.iec958.status[0] = ctrl->ch_status[0];
	uvalue->value.iec958.status[1] = ctrl->ch_status[1];
	uvalue->value.iec958.status[2] = ctrl->ch_status[2];
	uvalue->value.iec958.status[3] = ctrl->ch_status[3];

	return 0;
}

static int fsl_spdif_pb_put(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *uvalue)
{
	struct snd_soc_dai *cpu_dai = snd_kcontrol_chip(kcontrol);
	struct fsl_spdif_priv *spdif_priv = snd_soc_dai_get_drvdata(cpu_dai);
	struct spdif_mixer_control *ctrl = &spdif_priv->fsl_spdif_control;

	ctrl->ch_status[0] = uvalue->value.iec958.status[0];
	ctrl->ch_status[1] = uvalue->value.iec958.status[1];
	ctrl->ch_status[2] = uvalue->value.iec958.status[2];
	ctrl->ch_status[3] = uvalue->value.iec958.status[3];

	spdif_write_channel_status(spdif_priv);

	return 0;
}

/* Get channel status from SPDIF_RX_CCHAN register */
static int fsl_spdif_capture_get(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_dai *cpu_dai = snd_kcontrol_chip(kcontrol);
	struct fsl_spdif_priv *spdif_priv = snd_soc_dai_get_drvdata(cpu_dai);
	struct regmap *regmap = spdif_priv->regmap;
	u32 cstatus, val;

	regmap_read(regmap, REG_SPDIF_SIS, &val);
	if (!(val & INT_CNEW))
		return -EAGAIN;

	regmap_read(regmap, REG_SPDIF_SRCSH, &cstatus);
	ucontrol->value.iec958.status[0] = (cstatus >> 16) & 0xFF;
	ucontrol->value.iec958.status[1] = (cstatus >> 8) & 0xFF;
	ucontrol->value.iec958.status[2] = cstatus & 0xFF;

	regmap_read(regmap, REG_SPDIF_SRCSL, &cstatus);
	ucontrol->value.iec958.status[3] = (cstatus >> 16) & 0xFF;
	ucontrol->value.iec958.status[4] = (cstatus >> 8) & 0xFF;
	ucontrol->value.iec958.status[5] = cstatus & 0xFF;

	/* Clear intr */
	regmap_write(regmap, REG_SPDIF_SIC, INT_CNEW);

	return 0;
}

/*
 * Get User bits (subcode) from chip value which readed out
 * in UChannel register.
 */
static int fsl_spdif_subcode_get(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_dai *cpu_dai = snd_kcontrol_chip(kcontrol);
	struct fsl_spdif_priv *spdif_priv = snd_soc_dai_get_drvdata(cpu_dai);
	struct spdif_mixer_control *ctrl = &spdif_priv->fsl_spdif_control;
	unsigned long flags;
	int ret = -EAGAIN;

	spin_lock_irqsave(&ctrl->ctl_lock, flags);
	if (ctrl->ready_buf) {
		int idx = (ctrl->ready_buf - 1) * SPDIF_UBITS_SIZE;
		memcpy(&ucontrol->value.iec958.subcode[0],
				&ctrl->subcode[idx], SPDIF_UBITS_SIZE);
		ret = 0;
	}
	spin_unlock_irqrestore(&ctrl->ctl_lock, flags);

	return ret;
}

/* Q-subcode information. The byte size is SPDIF_UBITS_SIZE/8 */
static int fsl_spdif_qinfo(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_info *uinfo)
{
	uinfo->type = SNDRV_CTL_ELEM_TYPE_BYTES;
	uinfo->count = SPDIF_QSUB_SIZE;

	return 0;
}

/* Get Q subcode from chip value which readed out in QChannel register */
static int fsl_spdif_qget(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_dai *cpu_dai = snd_kcontrol_chip(kcontrol);
	struct fsl_spdif_priv *spdif_priv = snd_soc_dai_get_drvdata(cpu_dai);
	struct spdif_mixer_control *ctrl = &spdif_priv->fsl_spdif_control;
	unsigned long flags;
	int ret = -EAGAIN;

	spin_lock_irqsave(&ctrl->ctl_lock, flags);
	if (ctrl->ready_buf) {
		int idx = (ctrl->ready_buf - 1) * SPDIF_QSUB_SIZE;
		memcpy(&ucontrol->value.bytes.data[0],
				&ctrl->qsub[idx], SPDIF_QSUB_SIZE);
		ret = 0;
	}
	spin_unlock_irqrestore(&ctrl->ctl_lock, flags);

	return ret;
}

/* Valid bit information */
/* Get valid good bit from interrupt status register */
static int fsl_spdif_rx_vbit_get(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_dai *cpu_dai = snd_kcontrol_chip(kcontrol);
	struct fsl_spdif_priv *spdif_priv = snd_soc_dai_get_drvdata(cpu_dai);
	struct regmap *regmap = spdif_priv->regmap;
	u32 val;

	regmap_read(regmap, REG_SPDIF_SIS, &val);
	ucontrol->value.integer.value[0] = (val & INT_VAL_NOGOOD) != 0;
	regmap_write(regmap, REG_SPDIF_SIC, INT_VAL_NOGOOD);

	return 0;
}

static int fsl_spdif_tx_vbit_get(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_dai *cpu_dai = snd_kcontrol_chip(kcontrol);
	struct fsl_spdif_priv *spdif_priv = snd_soc_dai_get_drvdata(cpu_dai);
	struct regmap *regmap = spdif_priv->regmap;
	u32 val;

	regmap_read(regmap, REG_SPDIF_SCR, &val);
	val = (val & SCR_VAL_MASK) >> SCR_VAL_OFFSET;
	val = 1 - val;
	ucontrol->value.integer.value[0] = val;

	return 0;
}

static int fsl_spdif_tx_vbit_put(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_dai *cpu_dai = snd_kcontrol_chip(kcontrol);
	struct fsl_spdif_priv *spdif_priv = snd_soc_dai_get_drvdata(cpu_dai);
	struct regmap *regmap = spdif_priv->regmap;
	u32 val = (1 - ucontrol->value.integer.value[0]) << SCR_VAL_OFFSET;

	regmap_update_bits(regmap, REG_SPDIF_SCR, SCR_VAL_MASK, val);

	return 0;
}

static int fsl_spdif_rx_rcm_get(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_dai *cpu_dai = snd_kcontrol_chip(kcontrol);
	struct fsl_spdif_priv *spdif_priv = snd_soc_dai_get_drvdata(cpu_dai);
	struct regmap *regmap = spdif_priv->regmap;
	u32 val;

	regmap_read(regmap, REG_SPDIF_SCR, &val);
	val = (val & SCR_RAW_CAPTURE_MODE) ? 1 : 0;
	ucontrol->value.integer.value[0] = val;

	return 0;
}

static int fsl_spdif_rx_rcm_put(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_dai *cpu_dai = snd_kcontrol_chip(kcontrol);
	struct fsl_spdif_priv *spdif_priv = snd_soc_dai_get_drvdata(cpu_dai);
	struct regmap *regmap = spdif_priv->regmap;
	u32 val = (ucontrol->value.integer.value[0] ? SCR_RAW_CAPTURE_MODE : 0);

	if (val)
		cpu_dai->driver->capture.formats |= SNDRV_PCM_FMTBIT_S32_LE;
	else
		cpu_dai->driver->capture.formats &= ~SNDRV_PCM_FMTBIT_S32_LE;

	regmap_update_bits(regmap, REG_SPDIF_SCR, SCR_RAW_CAPTURE_MODE, val);

	return 0;
}

/* DPLL lock information */
static int fsl_spdif_rxrate_info(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_info *uinfo)
{
	uinfo->type = SNDRV_CTL_ELEM_TYPE_INTEGER;
	uinfo->count = 1;
	uinfo->value.integer.min = 16000;
	uinfo->value.integer.max = 96000;

	return 0;
}

static u32 gainsel_multi[GAINSEL_MULTI_MAX] = {
	24, 16, 12, 8, 6, 4, 3,
};

/* Get RX data clock rate given the SPDIF bus_clk */
static int spdif_get_rxclk_rate(struct fsl_spdif_priv *spdif_priv,
				enum spdif_gainsel gainsel)
{
	struct regmap *regmap = spdif_priv->regmap;
	struct platform_device *pdev = spdif_priv->pdev;
	u64 tmpval64, busclk_freq = 0;
	u32 freqmeas, phaseconf;
	u8 clksrc;

	regmap_read(regmap, REG_SPDIF_SRFM, &freqmeas);
	regmap_read(regmap, REG_SPDIF_SRPC, &phaseconf);

	clksrc = (phaseconf >> SRPC_CLKSRC_SEL_OFFSET) & 0xf;

	/* Get bus clock from system */
	if (srpc_dpll_locked[clksrc] && (phaseconf & SRPC_DPLL_LOCKED))
		busclk_freq = clk_get_rate(spdif_priv->sysclk);

	/* FreqMeas_CLK = (BUS_CLK * FreqMeas) / 2 ^ 10 / GAINSEL / 128 */
	tmpval64 = (u64) busclk_freq * freqmeas;
	do_div(tmpval64, gainsel_multi[gainsel] * 1024);
	do_div(tmpval64, 128 * 1024);

	dev_dbg(&pdev->dev, "FreqMeas: %d\n", freqmeas);
	dev_dbg(&pdev->dev, "BusclkFreq: %lld\n", busclk_freq);
	dev_dbg(&pdev->dev, "RxRate: %lld\n", tmpval64);

	return (int)tmpval64;
}

/*
 * Get DPLL lock or not info from stable interrupt status register.
 * User application must use this control to get locked,
 * then can do next PCM operation
 */
static int fsl_spdif_rxrate_get(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_dai *cpu_dai = snd_kcontrol_chip(kcontrol);
	struct fsl_spdif_priv *spdif_priv = snd_soc_dai_get_drvdata(cpu_dai);
	int rate = 0;

	if (spdif_priv->dpll_locked)
		rate = spdif_get_rxclk_rate(spdif_priv, SPDIF_DEFAULT_GAINSEL);

	ucontrol->value.integer.value[0] = rate;

	return 0;
}

/*
 * User bit sync mode:
 * 1 CD User channel subcode
 * 0 Non-CD data
 */
static int fsl_spdif_usync_get(struct snd_kcontrol *kcontrol,
			       struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_dai *cpu_dai = snd_kcontrol_chip(kcontrol);
	struct fsl_spdif_priv *spdif_priv = snd_soc_dai_get_drvdata(cpu_dai);
	struct regmap *regmap = spdif_priv->regmap;
	u32 val;

	regmap_read(regmap, REG_SPDIF_SRCD, &val);
	ucontrol->value.integer.value[0] = (val & SRCD_CD_USER) != 0;

	return 0;
}

/*
 * User bit sync mode:
 * 1 CD User channel subcode
 * 0 Non-CD data
 */
static int fsl_spdif_usync_put(struct snd_kcontrol *kcontrol,
				struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_dai *cpu_dai = snd_kcontrol_chip(kcontrol);
	struct fsl_spdif_priv *spdif_priv = snd_soc_dai_get_drvdata(cpu_dai);
	struct regmap *regmap = spdif_priv->regmap;
	u32 val = ucontrol->value.integer.value[0] << SRCD_CD_USER_OFFSET;

	regmap_update_bits(regmap, REG_SPDIF_SRCD, SRCD_CD_USER, val);

	return 0;
}

static int fsl_spdif_info_cstate_rate(struct snd_kcontrol *kcontrol,
				      struct snd_ctl_elem_info *uinfo)
{
	uinfo->type = SNDRV_CTL_ELEM_TYPE_INTEGER64;
	uinfo->count = 2;
	uinfo->value.integer.min = 0;
	uinfo->value.integer.max = S64_MAX;

	return 0;
}

static int fsl_spdif_lockstate_get_static(int *lock_stat)
{
	struct fsl_spdif_priv *spdif_priv = static_spdif_priv;
	struct regmap *regmap = spdif_priv->regmap;
	u32 locked = 0;
	int rv = 0;

	if (!lock_stat)
		return -EINVAL;

	if (!spdif_priv->opened) {
		*lock_stat = -1;
		return 0;
	}

	rv = regmap_read(regmap, REG_SPDIF_SRPC, &locked);
	locked &= SRPC_DPLL_LOCKED;

	if (locked)
		*lock_stat = 1;
	else
		*lock_stat = 0;

	return rv;
}

#ifdef CONFIG_SONY_SW_PLL
static unsigned int rbit32(unsigned int x)
{
	x = ((x & 0x55555555) << 1) | ((x & 0xAAAAAAAA) >> 1);
	x = ((x & 0x33333333) << 2) | ((x & 0xCCCCCCCC) >> 2);
	x = ((x & 0x0F0F0F0F) << 4) | ((x & 0xF0F0F0F0) >> 4);
	x = ((x & 0x00FF00FF) << 8) | ((x & 0xFF00FF00) >> 8);
	return (x << 16) | (x >> 16);
}

static u64 fsl_spdif_cstate_get(int *error)
{
	struct fsl_spdif_priv *spdif_priv = static_spdif_priv;
	struct regmap *regmap = spdif_priv->regmap;
	u32 cstatus, val, tmp_srcsh, tmp_srcsl;
	u64 cbits;

	regmap_read(regmap, REG_SPDIF_SIS, &val);
	if (!(val & INT_CNEW)) {
		*error = -EAGAIN;
		return SPDIF_CBITS_INVALID;
	}

	regmap_read(regmap, REG_SPDIF_SRCSH, &cstatus);
	tmp_srcsh = rbit32(cstatus) >> 8;

	regmap_read(regmap, REG_SPDIF_SRCSL, &cstatus);
	tmp_srcsl = rbit32(cstatus) >> 8;

	cbits = ((u64)tmp_srcsl << 24) | tmp_srcsh;

	/* Clear intr */
	regmap_write(regmap, REG_SPDIF_SIC, INT_CNEW);

	return cbits;
}

u64 fsl_spdif_cstate_get_static(int unlock, int *error)
{
	struct fsl_spdif_priv *spdif_priv = static_spdif_priv;
	struct platform_device *pdev = spdif_priv->pdev;
	static u64 old_cbits = SPDIF_CBITS_INVALID;
	static int compare_count;
	const int compare_stable_count = 2;
	u64 new_cbits = SPDIF_CBITS_INVALID;
	int rv = 0;

	if (unlock) {
		old_cbits = SPDIF_CBITS_INVALID;
		spdif_priv->cbits = SPDIF_CBITS_INVALID;
		*error = -2;
		return SPDIF_CBITS_INVALID;
	}

	new_cbits = fsl_spdif_cstate_get(&rv);
	if (rv < 0) {
		dev_dbg(&pdev->dev, "%s:call read err:%d\n", __func__, rv);
		*error = rv;
		old_cbits = SPDIF_CBITS_INVALID;
		spdif_priv->cbits = SPDIF_CBITS_INVALID;
		return SPDIF_CBITS_INVALID;
	}
	if (new_cbits != old_cbits) {
		old_cbits = new_cbits;
		compare_count = 1;
		*error = -3;
		return SPDIF_CBITS_INVALID;
	} else if (compare_count < compare_stable_count) {
		compare_count++;
		if (compare_count == compare_stable_count) {
			*error = 0;
			spdif_priv->cbits = new_cbits;
			return new_cbits;
		}
		*error = -4;
		return SPDIF_CBITS_INVALID;
	} else {
		*error = 0;
		return SPDIF_CBITS_INVALID;
	}

	dev_err(&pdev->dev, "%s: Unreachable. new_cbits:%llx rv:%d\n", __func__, new_cbits, rv);
	*error = -5;
	return SPDIF_CBITS_INVALID;
}

static int fsl_spdif_lockstate_get_static_swpll(int *dpll_lock_stat, int *lock_stat, int* open_stat, uint32_t *spdif_irq_stat)
{
	struct fsl_spdif_priv *spdif_priv = static_spdif_priv;
	struct regmap *regmap = spdif_priv->regmap;
	u32 locked = 0;
	u32 spdif_sie = 0;
	int rv = 0;

	if (!lock_stat || !dpll_lock_stat)
		return -EINVAL;

	if (!spdif_priv->opened) {
		*lock_stat = -1;
		*dpll_lock_stat = -1;
		*open_stat = -1;
		return 0;
	}

	rv = regmap_read(regmap, REG_SPDIF_SRPC, &locked);
	if (rv < 0)
		pr_err("regmap read error REG_SPDIF_SRPC :%d\n", rv);

	locked &= SRPC_DPLL_LOCKED;

	if (locked)
		*lock_stat = 1;
	else
		*lock_stat = 0;

	rv = regmap_read(regmap, REG_SPDIF_SIE, &spdif_sie);
	if (rv < 0)
		pr_err("regmap read error REG_SPDIF_SIE :%d\n", rv);

	*dpll_lock_stat = spdif_priv->dpll_locked;
	*spdif_irq_stat = spdif_sie;
	*open_stat = spdif_priv->opened;

	return 0;
}
#endif

static int fsl_spdif_cstate_rate_get(struct snd_kcontrol *kcontrol,
				     struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_dai *cpu_dai = snd_kcontrol_chip(kcontrol);
	struct fsl_spdif_priv *spdif_priv = snd_soc_dai_get_drvdata(cpu_dai);
	struct platform_device *pdev = spdif_priv->pdev;
	int rate = 0;

	dev_dbg(&pdev->dev, "%s:call cbits:%llx\n", __func__, spdif_priv->cbits);
	ucontrol->value.integer64.value[0] = spdif_priv->cbits;
	ucontrol->value.integer64.value[1] = rate;
	if (spdif_priv->cbits & SPDIF_CBITS_INVALID)
	{
		dev_dbg(&pdev->dev, "%s:error cbits read :%llx\n", __func__, spdif_priv->cbits);
		return -EAGAIN;
	}

	return 0;
}

static const struct {
	const char *name;
	int code;
} spdif_amixer_src_info[] = {
	{"aux", SPDIF_SWITCH_AUX},
	{"tv", SPDIF_SWITCH_TV},
	{"coax", SPDIF_SWITCH_COAX},
};

static const struct {
	const char *name;
	int code;
	char S0;
	char S1;
} spdif_src_info[] = {
	{"aux", SPDIF_SWITCH_AUX, 0, 1},
	{"tv", SPDIF_SWITCH_TV, 1, 1},
	{"coax", SPDIF_SWITCH_COAX, 1, 0},
	{"hdmi", SPDIF_SWITCH_HDMI, 0, 0},
};

int fsl_spdif_set_spdif_switch(int source)
{
	struct fsl_spdif_priv *spdif_priv = static_spdif_priv;
	int input_source = -1;

	if (source == SPDIF_SWITCH_MODE_OPT)
		input_source = spdif_priv->spdif_switch_amixer;
	else if (source == SPDIF_SWITCH_MODE_HDMI)
		input_source = SPDIF_SWITCH_HDMI;
	else
		input_source = source;


	if (input_source != SPDIF_SWITCH_HDMI)
		spdif_priv->spdif_switch_amixer = input_source;


	pr_debug("%s:%d  %s:%d input_source:%d pin_s1:%d pin_s0:%d S1:%d S0:%d\n",
		__func__, __LINE__,
		spdif_src_info[input_source].name, spdif_src_info[input_source].code, input_source,
		spdif_priv->spdif_switch_s1, spdif_priv->spdif_switch_s0,
		spdif_src_info[input_source].S1, spdif_src_info[input_source].S0);

	if (spdif_priv->spdif_switch_s0 && spdif_priv->spdif_switch_s1) {
		gpio_set_value_cansleep(spdif_priv->spdif_switch_s0, spdif_src_info[input_source].S0);
		gpio_set_value_cansleep(spdif_priv->spdif_switch_s1, spdif_src_info[input_source].S1);
	}

	return 1;
}
EXPORT_SYMBOL_GPL(fsl_spdif_set_spdif_switch);

static int fsl_spdif_info_spdif_switch(struct snd_kcontrol *kcontrol,
				      struct snd_ctl_elem_info *uinfo)
{
	const int size = ARRAY_SIZE(spdif_amixer_src_info);

	uinfo->type = SNDRV_CTL_ELEM_TYPE_ENUMERATED;
	uinfo->count = 1;
	uinfo->value.enumerated.items = size;

	if (uinfo->value.enumerated.item >= size)
		uinfo->value.enumerated.item = size - 1;

	strlcpy(uinfo->value.enumerated.name,
		spdif_amixer_src_info[uinfo->value.enumerated.item].name,
		sizeof(uinfo->value.enumerated.name));

	return 0;
}

/* source get callback */
static int fsl_spdif_spdif_switch_get(struct snd_kcontrol *kcontrol,
				   struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_dai *cpu_dai = snd_kcontrol_chip(kcontrol);
	struct fsl_spdif_priv *spdif_priv = snd_soc_dai_get_drvdata(cpu_dai);

	ucontrol->value.enumerated.item[0] = spdif_priv->spdif_switch_amixer;

	return 0;
}

/* source put callback */
static int fsl_spdif_spdif_switch_put(struct snd_kcontrol *kcontrol,
				   struct snd_ctl_elem_value *ucontrol)
{
	unsigned int source;

	source = ucontrol->value.enumerated.item[0];
	if (source >= ARRAY_SIZE(spdif_amixer_src_info))
		return -EINVAL;

	fsl_spdif_set_spdif_switch(source);

	return 1;
}

/* FSL SPDIF IEC958 controller defines */
static struct snd_kcontrol_new fsl_spdif_ctrls[] = {
	/* Status cchanel controller */
	{
		.iface = SNDRV_CTL_ELEM_IFACE_MIXER,
		.name = SNDRV_CTL_NAME_IEC958("", PLAYBACK, DEFAULT),
		.access = SNDRV_CTL_ELEM_ACCESS_READ |
			SNDRV_CTL_ELEM_ACCESS_WRITE |
			SNDRV_CTL_ELEM_ACCESS_VOLATILE,
		.info = fsl_spdif_info,
		.get = fsl_spdif_pb_get,
		.put = fsl_spdif_pb_put,
	},
	{
		.iface = SNDRV_CTL_ELEM_IFACE_PCM,
		.name = SNDRV_CTL_NAME_IEC958("", CAPTURE, DEFAULT),
		.access = SNDRV_CTL_ELEM_ACCESS_READ |
			SNDRV_CTL_ELEM_ACCESS_VOLATILE,
		.info = fsl_spdif_info,
		.get = fsl_spdif_capture_get,
	},
	/* User bits controller */
	{
		.iface = SNDRV_CTL_ELEM_IFACE_PCM,
		.name = "IEC958 Subcode Capture Default",
		.access = SNDRV_CTL_ELEM_ACCESS_READ |
			SNDRV_CTL_ELEM_ACCESS_VOLATILE,
		.info = fsl_spdif_info,
		.get = fsl_spdif_subcode_get,
	},
	{
		.iface = SNDRV_CTL_ELEM_IFACE_PCM,
		.name = "IEC958 Q-subcode Capture Default",
		.access = SNDRV_CTL_ELEM_ACCESS_READ |
			SNDRV_CTL_ELEM_ACCESS_VOLATILE,
		.info = fsl_spdif_qinfo,
		.get = fsl_spdif_qget,
	},
	/* Valid bit error controller */
	{
		.iface = SNDRV_CTL_ELEM_IFACE_PCM,
		.name = "IEC958 Rx V-Bit Errors",
		.access = SNDRV_CTL_ELEM_ACCESS_READ |
			SNDRV_CTL_ELEM_ACCESS_VOLATILE,
		.info = snd_ctl_boolean_mono_info,
		.get = fsl_spdif_rx_vbit_get,
	},
	{
		.iface = SNDRV_CTL_ELEM_IFACE_PCM,
		.name = "IEC958 Tx V-Bit",
		.access = SNDRV_CTL_ELEM_ACCESS_READ |
			SNDRV_CTL_ELEM_ACCESS_WRITE |
			SNDRV_CTL_ELEM_ACCESS_VOLATILE,
		.info = snd_ctl_boolean_mono_info,
		.get = fsl_spdif_tx_vbit_get,
		.put = fsl_spdif_tx_vbit_put,
	},
	/* DPLL lock info get controller */
	{
		.iface = SNDRV_CTL_ELEM_IFACE_PCM,
		.name = "RX Sample Rate",
		.access = SNDRV_CTL_ELEM_ACCESS_READ |
			SNDRV_CTL_ELEM_ACCESS_VOLATILE,
		.info = fsl_spdif_rxrate_info,
		.get = fsl_spdif_rxrate_get,
	},
	/* User bit sync mode set/get controller */
	{
		.iface = SNDRV_CTL_ELEM_IFACE_PCM,
		.name = "IEC958 USyncMode CDText",
		.access = SNDRV_CTL_ELEM_ACCESS_READ |
			SNDRV_CTL_ELEM_ACCESS_WRITE |
			SNDRV_CTL_ELEM_ACCESS_VOLATILE,
		.info = snd_ctl_boolean_mono_info,
		.get = fsl_spdif_usync_get,
		.put = fsl_spdif_usync_put,
	},
	{
		.iface = SNDRV_CTL_ELEM_IFACE_PCM,
		.name = "IEC958 Rx Raw Capture Mode Bit",
		.access = SNDRV_CTL_ELEM_ACCESS_READ |
			SNDRV_CTL_ELEM_ACCESS_WRITE |
			SNDRV_CTL_ELEM_ACCESS_VOLATILE,
		.info = snd_ctl_boolean_mono_info,
		.get = fsl_spdif_rx_rcm_get,
		.put = fsl_spdif_rx_rcm_put,
	},
	{
		.iface = SNDRV_CTL_ELEM_IFACE_PCM,
		.name = "RX Cstate and Rate",
		.access = SNDRV_CTL_ELEM_ACCESS_READ |
			SNDRV_CTL_ELEM_ACCESS_VOLATILE,
		.info = fsl_spdif_info_cstate_rate,
		.get = fsl_spdif_cstate_rate_get,
	},
	{
		.iface = SNDRV_CTL_ELEM_IFACE_MIXER,
		.name = "spdif switch",
		.access = SNDRV_CTL_ELEM_ACCESS_READ |
			SNDRV_CTL_ELEM_ACCESS_WRITE |
			SNDRV_CTL_ELEM_ACCESS_VOLATILE,
		.info = fsl_spdif_info_spdif_switch,
		.get = fsl_spdif_spdif_switch_get,
		.put = fsl_spdif_spdif_switch_put,
	},
};

static int fsl_spdif_dai_probe(struct snd_soc_dai *dai)
{
	struct fsl_spdif_priv *spdif_private = snd_soc_dai_get_drvdata(dai);

	snd_soc_dai_init_dma_data(dai, &spdif_private->dma_params_tx,
				  &spdif_private->dma_params_rx);

	snd_soc_add_dai_controls(dai, fsl_spdif_ctrls, ARRAY_SIZE(fsl_spdif_ctrls));

	return 0;
}

static struct snd_soc_dai_driver fsl_spdif_dai = {
	.probe = &fsl_spdif_dai_probe,
	.playback = {
		.stream_name = "CPU-Playback",
		.channels_min = 2,
		.channels_max = 2,
		.rates = FSL_SPDIF_RATES_PLAYBACK,
		.formats = FSL_SPDIF_FORMATS_PLAYBACK,
	},
	.capture = {
		.stream_name = "CPU-Capture",
		.channels_min = 2,
		.channels_max = 2,
		.rates = FSL_SPDIF_RATES_CAPTURE,
		.formats = FSL_SPDIF_FORMATS_CAPTURE,
	},
	.ops = &fsl_spdif_dai_ops,
};

static const struct snd_soc_component_driver fsl_spdif_component = {
	.name		= "fsl-spdif",
};

/* FSL SPDIF REGMAP */
static const struct reg_default fsl_spdif_reg_defaults[] = {
	{REG_SPDIF_SCR,    0x00000400},
	{REG_SPDIF_SRCD,   0x00000000},
	{REG_SPDIF_SIE,	   0x00000000},
	{REG_SPDIF_STL,	   0x00000000},
	{REG_SPDIF_STR,	   0x00000000},
	{REG_SPDIF_STCSCH, 0x00000000},
	{REG_SPDIF_STCSCL, 0x00000000},
	{REG_SPDIF_STC,	   0x00020f00},
};

static bool fsl_spdif_readable_reg(struct device *dev, unsigned int reg)
{
	switch (reg) {
	case REG_SPDIF_SCR:
	case REG_SPDIF_SRCD:
	case REG_SPDIF_SRPC:
	case REG_SPDIF_SIE:
	case REG_SPDIF_SIS:
	case REG_SPDIF_SRL:
	case REG_SPDIF_SRR:
	case REG_SPDIF_SRCSH:
	case REG_SPDIF_SRCSL:
	case REG_SPDIF_SRU:
	case REG_SPDIF_SRQ:
	case REG_SPDIF_STCSCH:
	case REG_SPDIF_STCSCL:
	case REG_SPDIF_SRFM:
	case REG_SPDIF_STC:
		return true;
	default:
		return false;
	}
}

static bool fsl_spdif_volatile_reg(struct device *dev, unsigned int reg)
{
	switch (reg) {
	case REG_SPDIF_SRPC:
	case REG_SPDIF_SIS:
	case REG_SPDIF_SRL:
	case REG_SPDIF_SRR:
	case REG_SPDIF_SRCSH:
	case REG_SPDIF_SRCSL:
	case REG_SPDIF_SRU:
	case REG_SPDIF_SRQ:
	case REG_SPDIF_SRFM:
		return true;
	default:
		return false;
	}
}

static bool fsl_spdif_writeable_reg(struct device *dev, unsigned int reg)
{
	switch (reg) {
	case REG_SPDIF_SCR:
	case REG_SPDIF_SRCD:
	case REG_SPDIF_SRPC:
	case REG_SPDIF_SIE:
	case REG_SPDIF_SIC:
	case REG_SPDIF_STL:
	case REG_SPDIF_STR:
	case REG_SPDIF_STCSCH:
	case REG_SPDIF_STCSCL:
	case REG_SPDIF_STC:
		return true;
	default:
		return false;
	}
}

static const struct regmap_config fsl_spdif_regmap_config = {
	.reg_bits = 32,
	.reg_stride = 4,
	.val_bits = 32,

	.max_register = REG_SPDIF_STC,
	.reg_defaults = fsl_spdif_reg_defaults,
	.num_reg_defaults = ARRAY_SIZE(fsl_spdif_reg_defaults),
	.readable_reg = fsl_spdif_readable_reg,
	.volatile_reg = fsl_spdif_volatile_reg,
	.writeable_reg = fsl_spdif_writeable_reg,
	.cache_type = REGCACHE_FLAT,
};

static const struct of_device_id fsl_spdif_dt_ids[] = {
	{ .compatible = "fsl,imx8qxp-v1-spdif", .data = &fsl_spdif_imx8qxp_v1, },
	{ .compatible = "fsl,imx8mm-spdif", .data = &fsl_spdif_imx8mm, },
	{ .compatible = "fsl,imx8qm-spdif", .data = &fsl_spdif_imx8qm, },
	{ .compatible = "fsl,imx35-spdif", .data = &fsl_spdif_imx35, },
	{ .compatible = "fsl,vf610-spdif", .data = &fsl_spdif_vf610, },
	{}
};
MODULE_DEVICE_TABLE(of, fsl_spdif_dt_ids);

static ssize_t lock_state_show(struct kobject *kobj, struct kobj_attribute *attr, char *buf)
{
	int now_lock_state = -1;
	int rv = 0;

	rv = fsl_spdif_lockstate_get_static(&now_lock_state);
	if (rv < 0) {
		pr_warn("fsl_spdif_lockstate_get_static error:rv=%d\n", rv);
		now_lock_state = -1;
	}

	return snprintf(buf, PAGE_SIZE, "%d\n", now_lock_state);
}

static struct kobj_attribute lock_state_attribute = __ATTR(lock_state, 0444, lock_state_show, NULL);

static struct attribute *attrs[] = {
	&lock_state_attribute.attr,
	NULL,
};

static struct attribute_group attr_group = {
	.attrs = attrs,
};

static int fsl_spdif_probe(struct platform_device *pdev)
{
	struct device_node *np = pdev->dev.of_node;
	struct fsl_spdif_priv *spdif_priv;
	struct spdif_mixer_control *ctrl;
	struct resource *res;
	const struct of_device_id *of_id;
	void __iomem *regs;
	int irq, ret, i;
	u32 buffer_size;
	char tmp[16];
	int spdif_switch_s0, spdif_switch_s1;

	if (!np)
		return -ENODEV;

	spdif_priv = devm_kzalloc(&pdev->dev, sizeof(*spdif_priv), GFP_KERNEL);
	if (!spdif_priv)
		return -ENOMEM;

	spdif_priv->pdev = pdev;

	of_id = of_match_device(fsl_spdif_dt_ids, &pdev->dev);
	if (!of_id || !of_id->data)
		return -EINVAL;

	spdif_priv->soc = of_id->data;

	/* Initialize this copy of the CPU DAI driver structure */
	memcpy(&spdif_priv->cpu_dai_drv, &fsl_spdif_dai, sizeof(fsl_spdif_dai));
	spdif_priv->cpu_dai_drv.name = dev_name(&pdev->dev);
	spdif_priv->cpu_dai_drv.playback.formats =
				spdif_priv->soc->tx_formats;
	spdif_priv->cpu_dai_drv.capture.rates =
				spdif_priv->soc->rx_rates;

	/* Get the addresses and IRQ */
	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	regs = devm_ioremap_resource(&pdev->dev, res);
	if (IS_ERR(regs))
		return PTR_ERR(regs);

	spdif_priv->regmap = devm_regmap_init_mmio_clk(&pdev->dev,
			"core", regs, &fsl_spdif_regmap_config);
	if (IS_ERR(spdif_priv->regmap)) {
		dev_err(&pdev->dev, "regmap init failed\n");
		return PTR_ERR(spdif_priv->regmap);
	}

	irq = platform_get_irq(pdev, 0);
	if (irq < 0) {
		dev_err(&pdev->dev, "no irq for node %s\n", pdev->name);
		return irq;
	}

	spdif_priv->spdif_isr_kobject = kobject_create_and_add("spdif_isr",
							   &pdev->dev.kobj);
	if (!spdif_priv->spdif_isr_kobject)
		return -ENOMEM;

	spdif_priv->wq = alloc_workqueue("spdif_wq", WQ_MEM_RECLAIM, 0);
	if (!spdif_priv->wq)
		return -ENOMEM;

	atomic_set(&spdif_priv->irq_event, 0);
	INIT_WORK(&spdif_priv->work, spdif_detected_wq);

	ret = devm_request_irq(&pdev->dev, irq, spdif_isr, 0,
			       dev_name(&pdev->dev), spdif_priv);
	if (ret) {
		dev_err(&pdev->dev, "could not claim irq %u\n", irq);
		return ret;
	}
	if (spdif_priv->soc->interrupts > 1) {
		irq = platform_get_irq(pdev, 1);
		if (irq < 0) {
			dev_err(&pdev->dev, "no irq for node %s\n", pdev->name);
			return irq;
		}

		ret = devm_request_irq(&pdev->dev, irq, spdif_isr, 0,
			       dev_name(&pdev->dev), spdif_priv);
		if (ret) {
			dev_err(&pdev->dev, "could not claim irq %u\n", irq);
			return ret;
		}
	}

	for (i = 0; i < STC_TXCLK_SRC_MAX; i++) {
		sprintf(tmp, "rxtx%d", i);
		spdif_priv->txclk[i] = devm_clk_get(&pdev->dev, tmp);
		if (IS_ERR(spdif_priv->txclk[i])) {
			dev_err(&pdev->dev, "no rxtx%d clock in devicetree\n", i);
			return PTR_ERR(spdif_priv->txclk[i]);
		}
	}

	/* Get system clock for rx clock rate calculation */
	spdif_priv->sysclk = spdif_priv->txclk[5];
	if (IS_ERR(spdif_priv->sysclk)) {
		dev_err(&pdev->dev, "no sys clock (rxtx5) in devicetree\n");
		return PTR_ERR(spdif_priv->sysclk);
	}

	/* Get core clock for data register access via DMA */
	spdif_priv->coreclk = devm_clk_get(&pdev->dev, "core");
	if (IS_ERR(spdif_priv->coreclk)) {
		dev_err(&pdev->dev, "no core clock in devicetree\n");
		return PTR_ERR(spdif_priv->coreclk);
	}

	spdif_priv->spbaclk = devm_clk_get(&pdev->dev, "spba");
	if (IS_ERR(spdif_priv->spbaclk))
		dev_warn(&pdev->dev, "no spba clock in devicetree\n");

	/* Select clock source for rx/tx clock */
	spdif_priv->rxclk = spdif_priv->txclk[1];
	if (IS_ERR(spdif_priv->rxclk)) {
		dev_err(&pdev->dev, "no rxtx1 clock in devicetree\n");
		return PTR_ERR(spdif_priv->rxclk);
	}
	spdif_priv->rxclk_src = DEFAULT_RXCLK_SRC;

	spdif_priv->pll8k_clk = devm_clk_get(&pdev->dev, "pll8k");
	if (IS_ERR(spdif_priv->pll8k_clk))
		spdif_priv->pll8k_clk = NULL;

	spdif_priv->pll11k_clk = devm_clk_get(&pdev->dev, "pll11k");
	if (IS_ERR(spdif_priv->pll11k_clk))
		spdif_priv->pll11k_clk = NULL;

	for (i = 0; i < SPDIF_TXRATE_MAX; i++) {
		ret = fsl_spdif_probe_txclk(spdif_priv, i);
		if (ret)
			return ret;
	}

	/* Initial spinlock for control data */
	ctrl = &spdif_priv->fsl_spdif_control;
	spin_lock_init(&ctrl->ctl_lock);

	/* Init tx channel status default value */
	ctrl->ch_status[0] = IEC958_AES0_CON_NOT_COPYRIGHT |
			     IEC958_AES0_CON_EMPHASIS_5015;
	ctrl->ch_status[1] = IEC958_AES1_CON_DIGDIGCONV_ID;
	ctrl->ch_status[2] = 0x00;
	ctrl->ch_status[3] = IEC958_AES3_CON_FS_44100 |
			     IEC958_AES3_CON_CLOCK_1000PPM;

	spdif_priv->dpll_locked = false;

	spdif_priv->dma_params_tx.maxburst = spdif_priv->soc->tx_burst;
	spdif_priv->dma_params_rx.maxburst = spdif_priv->soc->rx_burst;
	spdif_priv->dma_params_tx.addr = res->start + REG_SPDIF_STL;
	spdif_priv->dma_params_rx.addr = res->start + REG_SPDIF_SRL;

#ifdef DETECT_RX_FIFO_OV
	spdif_priv->rxfifo_stable = false;
	spdif_priv->rxfifo_unov_count = 0;
#endif

	spdif_priv->cbits = SPDIF_CBITS_INVALID;
	atomic_set(&spdif_priv->work_que_enable, 0);

	spdif_priv->spdif_switch_amixer = 0;
	spdif_priv->opened = 0;
	mutex_init(&spdif_priv->softreset_lock);

	/*Clear the val bit for Tx*/
	regmap_update_bits(spdif_priv->regmap, REG_SPDIF_SCR,
					SCR_VAL_MASK, 1 << SCR_VAL_OFFSET);

	pm_runtime_enable(&pdev->dev);

	regcache_cache_only(spdif_priv->regmap, true);
	/* Register with ASoC */
	dev_set_drvdata(&pdev->dev, spdif_priv);

	ret = devm_snd_soc_register_component(&pdev->dev, &fsl_spdif_component,
					      &spdif_priv->cpu_dai_drv, 1);
	if (ret) {
		dev_err(&pdev->dev, "failed to register DAI: %d\n", ret);
		return ret;
	}

	if (of_property_read_u32(np, "fsl,dma-buffer-size", &buffer_size))
		buffer_size = IMX_SPDIF_DMABUF_SIZE;

	if (spdif_priv->soc->dma_workaround)
		spdif_priv->dma_info =
			fsl_dma_workaround_alloc_info("tcd_pool_spdif",
						      &pdev->dev,
						      "nxp,imx8qm-acm",
						      FSL_DMA_WORKAROUND_SPDIF);
	ret = imx_pcm_dma_init(pdev, buffer_size);
	if (ret)
		dev_err(&pdev->dev, "imx_pcm_dma_init failed: %d\n", ret);

	spdif_switch_s0 = of_get_named_gpio(pdev->dev.of_node, "spdif-switch-s0-gpio", 0);
	if (gpio_is_valid(spdif_switch_s0)) {
		gpio_direction_output(spdif_switch_s0, 0);
		spdif_priv->spdif_switch_s0 = spdif_switch_s0;
	} else {
		spdif_priv->spdif_switch_s0 = 0;
	}
	spdif_switch_s1 = of_get_named_gpio(pdev->dev.of_node, "spdif-switch-s1-gpio", 0);
	if (gpio_is_valid(spdif_switch_s1)) {
		gpio_direction_output(spdif_switch_s1, 0);
		spdif_priv->spdif_switch_s1 = spdif_switch_s1;
	} else {
		spdif_priv->spdif_switch_s1 = 0;
	}

	static_spdif_priv = spdif_priv;
#ifdef CONFIG_SONY_SW_PLL
	sw_pll_register_cbits_read_func(fsl_spdif_cstate_get_static);
	sw_pll_register_lock_read_func(fsl_spdif_lockstate_get_static_swpll);
	sw_pll_register_softreset_func(fsl_spdif_softreset_static);
#endif

	spdif_priv->spdif_kobj = kobject_create_and_add("spdif", kernel_kobj);
	if (!spdif_priv->spdif_kobj)
		return -ENOMEM;

	ret = sysfs_create_group(spdif_priv->spdif_kobj, &attr_group);
	if (ret)
		kobject_put(spdif_priv->spdif_kobj);

	return ret;
}

static int fsl_spdif_remove(struct platform_device *pdev)
{
	struct fsl_spdif_priv *spdif_priv = dev_get_drvdata(&pdev->dev);

	if (spdif_priv->soc->dma_workaround)
		fsl_dma_workaround_free_info(spdif_priv->dma_info, &pdev->dev);

	if (spdif_priv->wq) {
		flush_workqueue(spdif_priv->wq);
		destroy_workqueue(spdif_priv->wq);
		spdif_priv->wq = NULL;
	}
	return 0;
}

#ifdef CONFIG_PM
static int fsl_spdif_runtime_resume(struct device *dev)
{
	struct fsl_spdif_priv *spdif_priv = dev_get_drvdata(dev);
	int ret;
	int i;

	ret = clk_prepare_enable(spdif_priv->coreclk);
	if (ret) {
		dev_err(dev, "failed to enable core clock\n");
		return ret;
	}

	if (!IS_ERR(spdif_priv->spbaclk)) {
		ret = clk_prepare_enable(spdif_priv->spbaclk);
		if (ret) {
			dev_err(dev, "failed to enable spba clock\n");
			goto disable_core_clk;
		}
	}

	for (i = 0; i < STC_TXCLK_SRC_MAX; i++) {
		ret = clk_prepare_enable(spdif_priv->txclk[i]);
		if (ret)
			goto disable_spba_clk;
	}

	request_bus_freq(BUS_FREQ_HIGH);

	regcache_cache_only(spdif_priv->regmap, false);
	regcache_mark_dirty(spdif_priv->regmap);

	regmap_update_bits(spdif_priv->regmap, REG_SPDIF_SRPC,
			SRPC_CLKSRC_SEL_MASK | SRPC_GAINSEL_MASK,
			spdif_priv->regcache_srpc);

	ret = regcache_sync(spdif_priv->regmap);
	if (ret)
		goto disable_tx_clk;

	return 0;

disable_tx_clk:
disable_spba_clk:
	for (i--; i >= 0; i--)
		clk_disable_unprepare(spdif_priv->txclk[i]);
	if (!IS_ERR(spdif_priv->spbaclk))
		clk_disable_unprepare(spdif_priv->spbaclk);
disable_core_clk:
	clk_disable_unprepare(spdif_priv->coreclk);

	return ret;
}

static int fsl_spdif_runtime_suspend(struct device *dev)
{
	struct fsl_spdif_priv *spdif_priv = dev_get_drvdata(dev);
	int i;

	regmap_read(spdif_priv->regmap, REG_SPDIF_SRPC,
			&spdif_priv->regcache_srpc);
	regcache_cache_only(spdif_priv->regmap, true);
	release_bus_freq(BUS_FREQ_HIGH);

	for (i = 0; i < STC_TXCLK_SRC_MAX; i++)
		clk_disable_unprepare(spdif_priv->txclk[i]);

	if (!IS_ERR(spdif_priv->spbaclk))
		clk_disable_unprepare(spdif_priv->spbaclk);
	clk_disable_unprepare(spdif_priv->coreclk);

	return 0;
}
#endif

static const struct dev_pm_ops fsl_spdif_pm = {
	SET_SYSTEM_SLEEP_PM_OPS(pm_runtime_force_suspend, pm_runtime_force_resume)
	SET_RUNTIME_PM_OPS(fsl_spdif_runtime_suspend, fsl_spdif_runtime_resume,
			   NULL)
};

static struct platform_driver fsl_spdif_driver = {
	.driver = {
		.name = "fsl-spdif-dai",
		.of_match_table = fsl_spdif_dt_ids,
		.pm = &fsl_spdif_pm,
	},
	.probe = fsl_spdif_probe,
	.remove = fsl_spdif_remove,
};

module_platform_driver(fsl_spdif_driver);

MODULE_AUTHOR("Freescale Semiconductor, Inc.");
MODULE_DESCRIPTION("Freescale S/PDIF CPU DAI Driver");
MODULE_LICENSE("GPL v2");
MODULE_ALIAS("platform:fsl-spdif-dai");
