// SPDX-License-Identifier: GPL-2.0
/*
 * ist3602.c
 *
 * IST3602 Character LCD driver for Linux
 *
 * Copyright 2021, 2022 Sony Corporation
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/device.h>
#include <linux/gpio.h>
#include <linux/gpio/consumer.h>
#include <linux/miscdevice.h>
#include <linux/spi/spi.h>
#include <linux/delay.h>
#include <linux/types.h>
#include <linux/slab.h>
#include <linux/wait.h>
#include <linux/uaccess.h>
#include <linux/pm_runtime.h>
#include <linux/regulator/consumer.h>
#include <linux/ist3602.h>

/* Define this label when enable Debug log */
/* #define IST3602_DEBUG */

/* Define print macros */
#ifdef IST3602_DEBUG
#define DEBUG(fmt, args...) pr_info("%s(%d): " fmt, __func__, __LINE__, ##args)
#else
#define DEBUG(fmt, args...) pr_debug("%s(%d): " fmt, __func__, __LINE__, ##args)
#endif

#define LCD_MINOR		156

#define IST3602_DRV_DEVNAME	"ist3602_drv"
#define DRV_MAJOR	0
#define DRV_MINOR	0
#define DEV_NUM		1
#define VOL_LIMIT	(18000)		/* Regurator voltage limit[mV] */
#define BASE_DISP_CLOCK	(750000)	/* Base display clock[Hz] */
#define DUTY_1_9	(9)		/* 1/9 Display Duty */
#define DUTY_1_17	(17)		/* 1/17 Display Duty */
#define DUTY_1_25	(25)		/* 1/25 Display Duty */
#define DUTY_1_33	(33)		/* 1/33 Display Duty */
#define BUFFER_SIZE	(256)		/* Buffer Size */
#define TIMEOUT_COUNT	(1000)		/* Timeout[ms] */

/* IST3602 command code	0xBBDDD (BB=Busy[ms],DDD=Data) */
#define DISPLAY_CLEAR	0x02001		/* (2ms) */
#define RETURN_ZERO	0x02002		/* (2ms) */
#define SET_ENTRY_MODE	0x00004		/* (0ms) */
#define SET_DISPLAY	0x00008		/* (0ms) */
#define FUNCTIONSET_0	0x00020		/* (0ms) */
#define FUNCTIONSET_1	0x00021		/* (0ms) */
#define FUNCTIONSET_2	0x00022		/* (0ms) */
#define FUNCTIONSET_3	0x00023		/* (0ms) */
#define SET_DDRAM	0x00090		/* (0ms) */
/* Command on IST=0 */
#define SHIFT_CSR_DSP	0x00010		/* (0ms) */
#define SET_CGRAM	0x00091		/* (0ms) */
/* Command on IST=1 */
#define FOLLOWS_CTRL	0x00010		/* (0ms) */
#define SET_ICONRAM	0x00040		/* (0ms) */
#define POWER_CTRL1	0x00030		/* (0ms) */
#define POWER_CTRL2	0x00060		/* (0ms) */
#define V0_CTRL2	0x64070		/* (100ms) */
/* Command on IST=2 */
#define SET_DISP_MODE	0x00010		/* (0ms) */
#define SET_DISP_PTN	0x00060		/* (0ms) */
#define SEL_CGRAM	0x00040		/* (0ms) */
/* Command on IST=3 */
#define START_LINE	0x00082		/* (0ms) */
#define CONTRAST_SET	0x00081		/* (0ms) */
#define RGAIN_SET	0x000A0		/* (0ms) */
/* Command on TestMode */
#define IST_ENTRY	0x00088		/* (0ms) */
#define FRAME_RATE_ADJ	0x00020		/* (0ms) */
#define FRAME_RATE	0x000B2		/* (0ms) */
#define OST_CLK_SEL	0x00090		/* (0ms) */
#define OST_DIV_SEL	0x00098		/* (0ms) */
#define EXIT_ENTRY	0x000E3		/* (0ms) */

enum ecape_sequence_mode {
	ESC_MODE_IDLE,
	ESC_MODE_START,
	ESC_MODE_CHECK_THIRD_CHAR,
	ESC_MODE_CLEAR_DISPLAY,
	ESC_MODE_CHECK_FOURTH_CHAR,
	ESC_MODE_CURSOR_TO_X,
	ESC_MODE_CURSOR_TO_Y,
	ESC_MODE_CHANGE_WIDTH,
	ESC_MODE_CONTRAST,
	ESC_MODE_REGULATION_RATE,
	ESC_MODE_FRAME_RATE,
	ESC_MODE_ICON_ON,
	ESC_MODE_ICON_OFF,
	ESC_MODE_SET_CGRAM_1,
	ESC_MODE_SET_CGRAM_2,
	ESC_MODE_SET_CGRAM_3,
	ESC_MODE_SET_CGRAM_4,
	ESC_MODE_CHANGE_START_LINE,
};

struct ist3602_priv {
	struct spi_device	*spi;
	struct miscdevice	miscdev;
	struct gpio_desc	*dimmer_gpio;
	struct gpio_desc	*lcd_en_gpio;
	struct gpio_desc	*lcd_rst_gpio;
	struct regulator	*power;
	struct regulator	*backlight;
	int width;		/* Displayed colmuns */
	int lines;		/* Displayed lines */
	int fonts;		/* Font size */
	int contrast;		/* Contrast current value */
	int ratio;		/* Regulation Ratio current value */
	int contrast_std;	/* Contrast when Dimmer OFF */
	int ratio_std;		/* Regulation Ratio when Dimmer OFF */
	int contrast_dmr;	/* Contrast when Dimmer ON */
	int ratio_dmr;		/* Regulation Ratio when Dimmer ON */
	int frame_rate;		/* Frame rate value */
	int column_offset;	/* Column offset value */
	int bwidth;		/* Buffer size per line */
	int bias;		/* Voltage Follow Bias select */
	int booster;		/* Booster select */
	int dimmer_flag;	/* Dimmer ON/OFF flag */
	bool enable_dmr_gpio;	/* Enable to control Dimmer gpio */
	bool backlight_link;	/* Set backlight ON/OFF with display ON/OFF */
	bool backlight_state;	/* Backlight status */

	int column;		/* Column pos */
	int line;		/* Line pos */
	int flag;		/* Flag(b3=Display,b2=blink,b1=cursor) */
	int esc;		/* Escape sequence mode number */
	int val;		/* Set value in Escape sequence mode */
	int cgnum;		/* CGRAM number */
	int cglen;		/* CGRAM length */
	u8 icon[0x20];		/* ICON buffer */
	u8 cgbuf[8];		/* CGRAM buffer */

	int *buff;		/* SPI Ring buffer */
	int bufwrp;		/* Write Pointer */
	int bufrdp;		/* Read Pointer */
	int bufcnt;		/* Data counter */
	int spierr;		/* SPI errno */
	struct mutex		buff_lock;
	struct task_struct	*buff_task;
	wait_queue_head_t	buff_wait;
};

static const int code_swap_table[] = {
	0x1F, 0x04,		/* "II"(0x04) */
	0x5C, 0x05,		/* "\"(0x05) */
	0x7E, 0x06,		/* "~"(0x06) */
	0x80, 0xE9,		/* "~!"(0xE9) */
	0x81, 0x9F,		/* "~?"(0x9F) */
	0x82, 0x12,		/* "ss"(0x12) */
	0x83, 0xE6,		/* "en"(0xE6) */
	0x84, 0x07,		/* "o'"(0x07) */
	0x85, 0xE5,		/* "fl"(0xE5) */
	0x86, 0xE4,		/* "Ct"(0xE4) */
};

static const int ist3602_init[] = {
	/* b27-20:Repeat,b19-12b=Busy,b8=D/C,b7-0:Data */
	FUNCTIONSET_0,		/* Function set (IS=00) */
	DISPLAY_CLEAR,		/* Display Clear */
	RETURN_ZERO,		/* Return 0 pos */
	SET_ENTRY_MODE | 0x02,	/* Set entry mode(Incliment) */
	SET_DISPLAY | 0x00,	/* Display control(D=0,C=0,B=0) */
	FUNCTIONSET_1,		/* Function set (IS=01) */
	FOLLOWS_CTRL | 0x00,	/* Follows control(1/4 BIAS) */
	SET_ICONRAM | 0x00,	/* Set ICON RAM Address */
	0x1F00100,		/* ICON RAM FILL(0x00 x 32) */
	POWER_CTRL1 | 0x00,	/* Power control1(SLEEP mode) */
	POWER_CTRL2 | 0x0F,	/* ICON/Power Control(I/R/L/F) */
	V0_CTRL2 | 0x00,	/* V0 control Set(booster:5X) */
	FUNCTIONSET_3,		/* Function set (IS=11) */
	START_LINE,		/* Start line setting */
	0x000,			/* (0) */
	FUNCTIONSET_2,		/* Function set (IS=10) */
	SET_DISP_MODE | 0x00,	/* Set display mode(D=0,F=0) */
	SET_DISP_PTN | 0x00,	/* Set display pattern(NRM,NOALL) */
	SEL_CGRAM | 0x07,	/* Select CGRAM & COM/SEG direction */
	FUNCTIONSET_0,		/* Function set (IS=00) */
	SET_CGRAM,		/* Set CGRAM address */
	0x000,			/* (0) */
	/* Default Font */
	0x102,			/* 0x00:"   # " */
	0x105,			/* 0x00:"  # #" */
	0x0400104,		/* 0x00:"  #  " */
				/* 0x00:"  #  " */
				/* 0x00:"  #  " */
				/* 0x00:"  #  " */
				/* 0x00:"  #  " */
	0x100,
	0x0400104,		/* 0x01:"  #  " */
				/* 0x01:"  #  " */
				/* 0x01:"  #  " */
				/* 0x01:"  #  " */
				/* 0x01:"  #  " */
	0x114,			/* 0x01:"# #  " */
	0x108,			/* 0x01:" #   " */
	0x0200100,
				/* 0x02:"     " */
				/* 0x02:"     " */
	0x10A,			/* 0x02:" # # " */
	0x0100115,		/* 0x02:"# # #" */
				/* 0x02:"# # #" */
	0x10A,			/* 0x02:" # # " */
	0x0100100,		/* 0x02:"     " */

	0x11F,			/* 0x03:"#####" */
	0x111,			/* 0x03:"#   #" */
	0x112,			/* 0x03:"#  # " */
	0x114,			/* 0x03:"# #  " */
	0x118,			/* 0x03:"##   " */
	0x110,			/* 0x03:"#    " */
	0x0100100,		/* 0x03:"     " */

	/* Extra Font */
	0x11F,			/* 0x1f:"#####" */
	0x040010A,		/* 0x1f:" # # " */
				/* 0x1f:" # # " */
				/* 0x1f:" # # " */
				/* 0x1f:" # # " */
				/* 0x1f:" # # " */
	0x11F,			/* 0x1f:"#####" */
	0x0100100,
				/* 0x5C:"     " */
	0x110,			/* 0x5C:"#    " */
	0x108,			/* 0x5C:" #   " */
	0x104,			/* 0x5C:"  #  " */
	0x102,			/* 0x5C:"   # " */
	0x101,			/* 0x5C:"    #" */
	0x0100100,		/* 0x5C:"     " */

	0x10D,			/* 0x7E:" ## #" */
	0x116,			/* 0x7E:"# ## " */
	0x0500100,		/* 0x7E:"     " */
				/* 0x7E:"     " */
				/* 0x7E:"     " */
				/* 0x7E:"     " */
				/* 0x7E:"     " */

	0x111,			/* 0x84:"#   #" */
	0x10E,			/* 0x84:" ### " */
	0x0200111,		/* 0x84:"#   #" */
				/* 0x84:"#   #" */
				/* 0x84:"#   #" */
	0x10E,			/* 0x84:" ### " */
	0x111,			/* 0x84:"#   #" */
	0x100,
	SET_DDRAM,		/* Set DDRAM address */
	0x000,			/* (0) */
};

static const int ist3602_icon_clr[] = {
	FUNCTIONSET_1,		/* Function set (IS=01) */
	SET_ICONRAM | 0x00,	/* Set ICON RAM Address */
	0x1F00100,		/* ICON RAM FILL(0x00 x 32) */
	FUNCTIONSET_0,		/* Function set (IS=00) */
};

static const int ist3602_icon_all[] = {
	FUNCTIONSET_1,		/* Function set (IS=01) */
	SET_ICONRAM | 0x00,	/* Set ICON RAM Address */
	0x1F0011F,		/* ICON RAM FILL(0x1F x 32) */
	FUNCTIONSET_0,		/* Function set (IS=00) */
};

/* Device single-open policy control */
static atomic_t ist3602_available = ATOMIC_INIT(1);

static const int byteperline[] = { 0x60, 0x30, 0x20, 0x18 };

static int ist3602_thread(void *arg)
{
	struct ist3602_priv *priv = (struct ist3602_priv *)arg;
	int val, bsy, rep, cnt, ret;
	u16 dat;

	while (!kthread_should_stop()) {
		mutex_lock(&priv->buff_lock);
		if (priv->bufcnt > 0) {
			val = priv->buff[priv->bufrdp];
			priv->bufrdp = (priv->bufrdp + 1) % BUFFER_SIZE;
			priv->bufcnt--;
			mutex_unlock(&priv->buff_lock);

			bsy = ((val & 0x00FF000) >> 12) * 1000;
			rep = ((val & 0xFF00000) >> 20) + 1;
			dat = val & 0x1FF;
			if (rep > 1)
				DEBUG("send 0x%03x x %d\n", dat, rep);
			else
				DEBUG("send 0x%03x\n", dat);
			for (cnt = ret = 0; cnt < rep; cnt++) {
				ret |= spi_write(priv->spi, &dat, sizeof(dat));
				usleep_range(bsy, bsy + 10);
			}
			if (ret) {
				DEBUG("SPI failed(%d)\n", ret);
				mutex_lock(&priv->buff_lock);
				priv->spierr |= ret;
				mutex_unlock(&priv->buff_lock);
			}
		} else {
			mutex_unlock(&priv->buff_lock);
			DEBUG("task sleep\n");
			wait_event_interruptible(priv->buff_wait,
				(priv->bufcnt > 0 || kthread_should_stop()));
			DEBUG("task wakeup\n");
		}
	}

	return 0;
}

static int ist3602_send(struct ist3602_priv *priv, int val)
{
	int ret, tmo;

	for (tmo = 0; tmo < TIMEOUT_COUNT; tmo++) {
		mutex_lock(&priv->buff_lock);
		if (priv->bufcnt < BUFFER_SIZE)
			break;
		mutex_unlock(&priv->buff_lock);
		usleep_range(1000, 1010);
	}
	if (tmo < TIMEOUT_COUNT) {
		priv->buff[priv->bufwrp] = val;
		priv->bufwrp = (priv->bufwrp + 1) % BUFFER_SIZE;
		priv->bufcnt++;
		ret = priv->spierr;
		priv->spierr = 0;
		mutex_unlock(&priv->buff_lock);
	} else {
		ret = -EBUSY;
	}
	wake_up_interruptible(&priv->buff_wait);

	return ret;
}

static void ist3602_set_disp(struct ist3602_priv *priv, int syncslp)
{
	if (syncslp) {
		ist3602_send(priv, FUNCTIONSET_1);
		if (priv->flag & 0x04)
			ist3602_send(priv, POWER_CTRL1);
		else
			ist3602_send(priv, POWER_CTRL1 | 0x01);
		ist3602_send(priv, FUNCTIONSET_0);
	}
	ist3602_send(priv, SET_DISPLAY | priv->flag);
}

static void ist3602_set_pos(struct ist3602_priv *priv)
{
	if (priv->column >= priv->width)
		priv->column = priv->width - 1;
	if (priv->line >= priv->lines)
		priv->line = priv->lines - 1;
	ist3602_send(priv, SET_DDRAM);
	ist3602_send(priv, priv->line * priv->bwidth
			 + (priv->column + priv->column_offset) % priv->bwidth);
}

static void ist3602_set_framerate(struct ist3602_priv *priv)
{
	int row_freq = 0;
	int framerate_ctl = 0;

	/* Row frequency = Frame rate * Duty */
	/* Frame rate control = BASE_DISP_CLOCK / Row frequency - 1 */
	switch (priv->fonts * priv->lines) {
	case 1:
		row_freq = priv->frame_rate * DUTY_1_9;
		break;
	case 2:
		row_freq = priv->frame_rate * DUTY_1_17;
		break;
	case 3:
		row_freq = priv->frame_rate * DUTY_1_25;
		break;
	case 4:
		row_freq = priv->frame_rate * DUTY_1_33;
		break;
	default:
		dev_err(&priv->spi->dev, "Failed to set frame rate\n");
		return;
	}
	framerate_ctl = BASE_DISP_CLOCK / row_freq - 1;
	DEBUG("frame rate control = %d\n", framerate_ctl);

	ist3602_send(priv, FUNCTIONSET_3);
	ist3602_send(priv, IST_ENTRY);
	ist3602_send(priv, IST_ENTRY);
	ist3602_send(priv, IST_ENTRY);
	ist3602_send(priv, IST_ENTRY);
	ist3602_send(priv, FRAME_RATE_ADJ | 0x08);
	ist3602_send(priv, FRAME_RATE);
	ist3602_send(priv, framerate_ctl & 0x00FF);
	ist3602_send(priv, (framerate_ctl & 0xFF00) >> 8);
	ist3602_send(priv, OST_CLK_SEL | 0x03);
	ist3602_send(priv, OST_DIV_SEL | 0x01);
	ist3602_send(priv, EXIT_ENTRY);
	ist3602_send(priv, FUNCTIONSET_0);
	ist3602_set_pos(priv);
}

static void ist3602_set_mode(struct ist3602_priv *priv)
{
	/* Set Lines & Font size */
	ist3602_send(priv, FUNCTIONSET_2);
	ist3602_send(priv, SET_DISP_MODE
			| ((priv->lines - 1) & 3)
			| ((priv->fonts - 1) & 3) << 2);
	ist3602_send(priv, FUNCTIONSET_0);

	priv->bwidth = byteperline[(priv->lines - 1) & 3];

	/* Set Frame rate & Position */
	if (priv->width > priv->bwidth)
		priv->width = priv->bwidth;
	ist3602_set_framerate(priv);
}

static void ist3602_set_pattern(struct ist3602_priv *priv, int cgnum, u8 *ptn)
{
	int i;

	if (cgnum <= 3) {
		ist3602_send(priv, SET_CGRAM);
		ist3602_send(priv, cgnum * 8);
		for (i = 0; i < 8; i++)
			ist3602_send(priv, *(ptn++) | 0x100);
		ist3602_set_pos(priv);
	}
}

static void ist3602_set_icon(struct ist3602_priv *priv, int num, int on)
{
	int iadr, ibit, i;

	if (num == 999) {
		if (on) {
			memset(priv->icon, 0x1f, sizeof(priv->icon));
			for (i = 0; i < ARRAY_SIZE(ist3602_icon_all); i++)
				ist3602_send(priv, (int)ist3602_icon_all[i]);
		} else {
			memset(priv->icon, 0, sizeof(priv->icon));
			for (i = 0; i < ARRAY_SIZE(ist3602_icon_clr); i++)
				ist3602_send(priv, (int)ist3602_icon_clr[i]);
		}
		ist3602_set_pos(priv);
		return;
	}

	if (num >= 0xA0)
		return;

	iadr = num / 5;
	ibit = num % 5;
	if (on)
		priv->icon[iadr] |= (1 << ibit);
	else
		priv->icon[iadr] &= ~(1 << ibit);
	ist3602_send(priv, FUNCTIONSET_1);
	ist3602_send(priv, SET_ICONRAM | iadr);
	ist3602_send(priv, priv->icon[iadr] | 0x100);
	ist3602_send(priv, FUNCTIONSET_0);
	ist3602_set_pos(priv);
}

static void ist3602_set_regulator(struct ist3602_priv *priv)
{
	int v0;

	/* RGR = 2.0 + rate * 0.5              */
	/* V0 = RGR * [(393 + CT) / 648] * 2.1 */
	v0 = ((20 + priv->ratio * 5)
		* (((393 + priv->contrast) * 2100) / 648)) / 10;
	DEBUG("regurator voltage=%d[mV]\n", v0);

	if (v0 <= VOL_LIMIT) {
		ist3602_send(priv, FUNCTIONSET_3);
		ist3602_send(priv, CONTRAST_SET);
		ist3602_send(priv, priv->contrast);
		ist3602_send(priv, RGAIN_SET | priv->ratio);
		ist3602_send(priv, FUNCTIONSET_0);
	} else {
		dev_warn(&priv->spi->dev,
			"Warning: regurator-voltage over 18V\n");
	}
}

static void ist3602_set_bias(struct ist3602_priv *priv)
{
	ist3602_send(priv, FUNCTIONSET_1);
	ist3602_send(priv, FOLLOWS_CTRL | priv->bias);
	ist3602_send(priv, FUNCTIONSET_0);
}

static void ist3602_set_booster(struct ist3602_priv *priv)
{
	ist3602_send(priv, FUNCTIONSET_1);
	ist3602_send(priv, V0_CTRL2 | priv->booster);
	ist3602_send(priv, FUNCTIONSET_0);
}

static void ist3602_put_char(struct ist3602_priv *priv, int c)
{
	int i;

	for (i = 0; i < ARRAY_SIZE(code_swap_table); i++) {
		if (code_swap_table[i] == c) {
			c = code_swap_table[i ^ 1];
			break;
		}
	}
	ist3602_send(priv, c | 0x100);
	priv->column++;
	if (priv->column == priv->width) {
		priv->line = (priv->line + 1) % priv->lines;
		priv->column = 0;
		ist3602_set_pos(priv);
	} else if (priv->column == (priv->bwidth - priv->column_offset)) {
		ist3602_set_pos(priv);
	}
}

static void ist3602_cls_home(struct ist3602_priv *priv, int cls)
{
	priv->column = 0;
	priv->line = 0;

	/* Clear the display */
	if (cls)
		ist3602_send(priv, DISPLAY_CLEAR);

	/* Return 0 position */
	ist3602_send(priv, RETURN_ZERO);
	if (priv->column_offset)
		ist3602_set_pos(priv);
}

static int ist3602_set_backlight(struct ist3602_priv *priv, int on)
{
	int ret = 0;

	if (on && !priv->backlight_state) {
		ret = regulator_enable(priv->backlight);
		if (ret) {
			dev_err(&priv->spi->dev,
				"Failed to enable regulator (%d)\n", ret);
			return ret;
		}
		priv->backlight_state = true;
	} else if (!on && priv->backlight_state) {
		ret = regulator_disable(priv->backlight);
		if (ret) {
			dev_err(&priv->spi->dev,
				"Failed to disable regulator (%d)\n", ret);
			return ret;
		}
		priv->backlight_state = false;
	} else {
		dev_warn(&priv->spi->dev,
			 "Warning: Backlight is already %s\n",
			 (on ? "on" : "off"));
	}
	return ret;
}

static int ist3602_initialize(struct ist3602_priv *priv)
{
	int ret = 0;
	int i;

	DEBUG("enter\n");
	/* parameter initialize */
	priv->column = 0;
	priv->line = 0;
	priv->flag = 0x04;
	memset(priv->icon, 0, sizeof(priv->icon));

	/* Initial Register Set */
	for (i = 0; i < ARRAY_SIZE(ist3602_init); i++)
		ret |= ist3602_send(priv, (int)ist3602_init[i]);

	/* Set Dimmer ON/OFF */
	if (priv->dimmer_gpio) {
		if (priv->enable_dmr_gpio)
			gpiod_set_value_cansleep(priv->dimmer_gpio,
						 priv->dimmer_flag);
		else
			gpiod_set_value_cansleep(priv->dimmer_gpio, 0);
	}

	/* Set Contrast & Regulation Ratio */
	if (!priv->enable_dmr_gpio && priv->dimmer_flag) {
		priv->contrast = priv->contrast_dmr;
		priv->ratio = priv->ratio_dmr;
	} else {
		priv->contrast = priv->contrast_std;
		priv->ratio = priv->ratio_std;
	}
	ist3602_set_regulator(priv);

	/* Set Voltage follow Bias */
	ist3602_set_bias(priv);

	/* Set booster */
	ist3602_set_booster(priv);

	/* Set Lines, Font size, Frame rate & Position */
	ist3602_set_mode(priv);

	/* Display ON */
	ist3602_send(priv, SET_DISPLAY | priv->flag);

	if (ret) {
		dev_err(&priv->spi->dev,
			"failed to write initial data(%d)\n", ret);
	}

	DEBUG("done\n");
	return ret;
}

static struct ist3602_priv *to_ist3602_priv(struct file *file)
{
	return container_of(file->private_data, struct ist3602_priv, miscdev);
}

static void ist3602_write_char(struct ist3602_priv *priv, char c)
{
	int h;

	if (priv->esc == ESC_MODE_START) {
		switch (c) {
		case '[':
			priv->esc++;
			break;
		default:
			ist3602_put_char(priv, c);
			priv->esc = ESC_MODE_IDLE;
		}
	} else if (priv->esc == ESC_MODE_CHECK_THIRD_CHAR) {
		switch (c) {
		case '2':
			priv->esc = ESC_MODE_CLEAR_DISPLAY;
			break;
		case 'H':	/* Return 0 pos */
			ist3602_cls_home(priv, 0);
			priv->esc = ESC_MODE_IDLE;
			break;
		case 'L':	/* Go to fourth character check */
			priv->esc = ESC_MODE_CHECK_FOURTH_CHAR;
			break;
		default:
			priv->esc = ESC_MODE_IDLE;
		}
	} else if (priv->esc == ESC_MODE_CLEAR_DISPLAY) {
		switch (c) {
		case 'J':	/* clear the display */
			ist3602_cls_home(priv, 1);
			priv->esc = ESC_MODE_IDLE;
			break;
		default:
			priv->esc = ESC_MODE_IDLE;
		}
	} else if (priv->esc == ESC_MODE_CHECK_FOURTH_CHAR) {
		switch (c) {
		case 'D':	/* Display ON */
			priv->flag |= 0x04;
			ist3602_set_disp(priv, 1);
			if (priv->backlight_link) {
				usleep_range(5000, 5010);
				ist3602_set_backlight(priv, 1);
			}
			priv->esc = ESC_MODE_IDLE;
			break;
		case 'd':	/* Display OFF */
			priv->flag &= ~0x04;
			if (priv->backlight_link) {
				ist3602_set_backlight(priv, 0);
				usleep_range(20000, 20010);
			}
			ist3602_set_disp(priv, 1);
			priv->esc = ESC_MODE_IDLE;
			break;
		case 'C':	/* Cursor ON */
			priv->flag |= 0x02;
			ist3602_set_disp(priv, 0);
			priv->esc = ESC_MODE_IDLE;
			break;
		case 'c':	/* Cursor OFF */
			priv->flag &= ~0x02;
			ist3602_set_disp(priv, 0);
			priv->esc = ESC_MODE_IDLE;
			break;
		case 'B':	/* Cursor blink ON */
			priv->flag |= 0x01;
			ist3602_set_disp(priv, 0);
			priv->esc = ESC_MODE_IDLE;
			break;
		case 'b':	/* Cursor blink OFF */
			priv->flag &= ~0x01;
			ist3602_set_disp(priv, 0);
			priv->esc = ESC_MODE_IDLE;
			break;
		case '+':	/* Dimmer OFF */
			priv->dimmer_flag = 0;
			if (priv->dimmer_gpio && priv->enable_dmr_gpio) {
				gpiod_set_value_cansleep(priv->dimmer_gpio,
							 priv->dimmer_flag);
			} else if (!priv->enable_dmr_gpio) {
				priv->contrast = priv->contrast_std;
				priv->ratio = priv->ratio_std;
				ist3602_set_regulator(priv);
			}
			priv->esc = ESC_MODE_IDLE;
			break;
		case '-':	/* Dimmer ON */
			priv->dimmer_flag = 1;
			if (priv->dimmer_gpio && priv->enable_dmr_gpio) {
				gpiod_set_value_cansleep(priv->dimmer_gpio,
							 priv->dimmer_flag);
			} else if (!priv->enable_dmr_gpio) {
				priv->contrast = priv->contrast_dmr;
				priv->ratio = priv->ratio_dmr;
				ist3602_set_regulator(priv);
			}
			priv->esc = ESC_MODE_IDLE;
			break;
		case 'E':	/* Backlight ON */
			ist3602_set_backlight(priv, 1);
			priv->esc = ESC_MODE_IDLE;
			break;
		case 'e':	/* Backlight OFF */
			ist3602_set_backlight(priv, 0);
			priv->esc = ESC_MODE_IDLE;
			break;
		case 'F':	/* Large font */
			priv->fonts = 2;
			ist3602_set_mode(priv);
			priv->esc = ESC_MODE_IDLE;
			break;
		case 'f':	/* Small font */
			priv->fonts = 1;
			ist3602_set_mode(priv);
			priv->esc = ESC_MODE_IDLE;
			break;
		case 'N':	/* Two lines */
			priv->lines = 2;
			ist3602_set_mode(priv);
			priv->esc = ESC_MODE_IDLE;
			break;
		case 'n':	/* One line */
			priv->lines = 1;
			ist3602_set_mode(priv);
			priv->esc = ESC_MODE_IDLE;
			break;
		case 'l':	/* Shift cursor left */
			if (priv->column > 0) {
				priv->column--;
				if (priv->column ==
				    (priv->bwidth - priv->column_offset - 1))
					ist3602_set_pos(priv);
				else
					ist3602_send(priv,
							SHIFT_CSR_DSP | 0x00);
			}
			priv->esc = ESC_MODE_IDLE;
			break;
		case 'r':	/* shift cursor right */
			if (priv->column < (priv->width - 1)) {
				priv->column++;
				if (priv->column ==
				    (priv->bwidth - priv->column_offset))
					ist3602_set_pos(priv);
				else
					ist3602_send(priv,
							SHIFT_CSR_DSP | 0x04);
			}
			priv->esc = ESC_MODE_IDLE;
			break;
		case 'L':	/* shift display left */
			ist3602_send(priv, SHIFT_CSR_DSP | 0x08);
			priv->esc = ESC_MODE_IDLE;
			break;
		case 'R':	/* shift display right */
			ist3602_send(priv, SHIFT_CSR_DSP | 0x0C);
			priv->esc = ESC_MODE_IDLE;
			break;
		case 'k':	/* kill end of line */
			for (h = priv->column; h < priv->width; h++) {
				if (h == (priv->bwidth - priv->column_offset)) {
					ist3602_send(priv, SET_DDRAM);
					ist3602_send(priv,
						     priv->line * priv->bwidth);
				}
				ist3602_send(priv, ' ' | 0x100);
			}
			ist3602_set_pos(priv);
			priv->esc = ESC_MODE_IDLE;
			break;
		case 'I':	/* reinitialize display */
			if (priv->lcd_rst_gpio) {
				gpiod_set_value_cansleep(priv->lcd_rst_gpio, 1);
				usleep_range(5000, 5010);
				gpiod_set_value_cansleep(priv->lcd_rst_gpio, 0);
				usleep_range(50, 60);
			}
			mutex_lock(&priv->buff_lock);
			priv->bufwrp = priv->bufrdp = priv->bufcnt = 0;
			mutex_unlock(&priv->buff_lock);
			ist3602_initialize(priv);
			priv->esc = ESC_MODE_IDLE;
			break;
		case 'x':	/* Shift cursor to x */
			priv->esc = ESC_MODE_CURSOR_TO_X;
			priv->val = 0;
			break;
		case 'y':	/* Shift cursor to y */
			priv->esc = ESC_MODE_CURSOR_TO_Y;
			priv->val = 0;
			break;
		case 'w':	/* Change width */
			priv->esc = ESC_MODE_CHANGE_WIDTH;
			priv->val = 0;
			break;
		case 'a':	/* Contrast */
			priv->esc = ESC_MODE_CONTRAST;
			priv->val = 0;
			break;
		case 'z':	/* Regulation-rate */
			priv->esc = ESC_MODE_REGULATION_RATE;
			priv->val = 0;
			break;
		case 'A':	/* Frame rate */
			priv->esc = ESC_MODE_FRAME_RATE;
			priv->val = 0;
			break;
		case 'G':	/* CGRAM setting */
			priv->esc = ESC_MODE_SET_CGRAM_1;
			priv->val = 0;
			break;
		case 'M':	/* Icon ON */
			priv->esc = ESC_MODE_ICON_ON;
			priv->val = 0;
			break;
		case 'm':	/* Icon OFF */
			priv->esc = ESC_MODE_ICON_OFF;
			priv->val = 0;
			break;
		case 'Q':	/* Display icons */
			ist3602_send(priv, FUNCTIONSET_1);
			ist3602_send(priv, POWER_CTRL2 | 0x0F);
			ist3602_send(priv, FUNCTIONSET_0);
			priv->esc = ESC_MODE_IDLE;
			break;
		case 'q':	/* Hide icons */
			ist3602_send(priv, FUNCTIONSET_1);
			ist3602_send(priv, POWER_CTRL2 | 0x07);
			ist3602_send(priv, FUNCTIONSET_0);
			priv->esc = ESC_MODE_IDLE;
			break;
		case 's':	/* Change start line of character */
			priv->esc = ESC_MODE_CHANGE_START_LINE;
			break;
		default:
			priv->esc = ESC_MODE_IDLE;
		}
	} else if (priv->esc >= ESC_MODE_CURSOR_TO_X &&
		   priv->esc <= ESC_MODE_ICON_OFF) {
		switch (c) {
		case 'x':
			if (priv->esc == ESC_MODE_CURSOR_TO_Y) {
				priv->line = priv->val;
				priv->esc = ESC_MODE_CURSOR_TO_X;
				priv->val = 0;
			} else {
				priv->esc = ESC_MODE_IDLE;
			}
			break;
		case 'y':
			if (priv->esc == ESC_MODE_CURSOR_TO_X) {
				priv->column = priv->val;
				priv->esc = ESC_MODE_CURSOR_TO_Y;
				priv->val = 0;
			} else {
				priv->esc = ESC_MODE_IDLE;
			}
			break;
		case '0' ... '9':
			priv->val = (priv->val * 10) + (c - '0');
			break;
		case ';':
			if (priv->esc == ESC_MODE_CURSOR_TO_X) {
				priv->column = priv->val;
				ist3602_set_pos(priv);
			} else if (priv->esc == ESC_MODE_CURSOR_TO_Y) {
				priv->line = priv->val;
				ist3602_set_pos(priv);
			} else if (priv->esc == ESC_MODE_CHANGE_WIDTH) {
				priv->width = priv->val;
				ist3602_set_mode(priv);
			} else if (priv->esc == ESC_MODE_CONTRAST) {
				priv->contrast = (priv->val & 0xFF);
				ist3602_set_regulator(priv);
			} else if (priv->esc == ESC_MODE_REGULATION_RATE) {
				priv->ratio = (priv->val & 0x0F);
				ist3602_set_regulator(priv);
			} else if (priv->esc == ESC_MODE_FRAME_RATE) {
				priv->frame_rate = priv->val;
				ist3602_set_framerate(priv);
			} else if (priv->esc == ESC_MODE_ICON_ON) {
				ist3602_set_icon(priv, priv->val, 1);
			} else if (priv->esc == ESC_MODE_ICON_OFF) {
				ist3602_set_icon(priv, priv->val, 0);
			}
			priv->esc = ESC_MODE_IDLE;
			break;
		default:
			priv->esc = ESC_MODE_IDLE;
		}
	} else if (priv->esc >= ESC_MODE_SET_CGRAM_1 &&
		   priv->esc <= ESC_MODE_SET_CGRAM_3) {
		switch (c) {
		case '0' ... '9':
			priv->val = (priv->val * 16) + (c - '0');
			priv->esc++;
			break;
		case 'A' ... 'F':
			priv->val = (priv->val * 16) + (c - 'A' + 10);
			priv->esc++;
			break;
		case 'a' ... 'f':
			priv->val = (priv->val * 16) + (c - 'a' + 10);
			priv->esc++;
			break;
		case ';':
			if (priv->cglen >= 7)
				ist3602_set_pattern(priv, priv->cgnum,
								priv->cgbuf);
			priv->esc = ESC_MODE_IDLE;
			break;
		default:
			priv->esc = ESC_MODE_IDLE;
		}

		if (priv->esc == ESC_MODE_SET_CGRAM_2) {
			priv->cgnum = priv->val;
			priv->cglen = 0;
			priv->val = 0;
		} else if (priv->esc == ESC_MODE_SET_CGRAM_4) {
			if (priv->cglen < 8)
				priv->cgbuf[priv->cglen++] = priv->val;
			priv->esc = ESC_MODE_SET_CGRAM_2;
			priv->val = 0;
		}
	} else if (priv->esc == ESC_MODE_CHANGE_START_LINE) {
		switch (c) {
		case '0' ... '7':
			priv->val = (c - '0');
			ist3602_send(priv, FUNCTIONSET_3);
			ist3602_send(priv, START_LINE);
			ist3602_send(priv, priv->val);
			ist3602_send(priv, FUNCTIONSET_0);
			priv->esc = ESC_MODE_IDLE;
			break;
		default:
			priv->esc = ESC_MODE_IDLE;
		}
	} else {
		switch (c) {
		case 0x1B:	/*Escape sequence mode */
			priv->esc = ESC_MODE_START;
			break;
		case 0x08:	/*Back space */
			if (priv->column > 0) {
				priv->column--;
				if (priv->column ==
				    (priv->bwidth - priv->column_offset - 1))
					ist3602_set_pos(priv);
				else
					ist3602_send(priv,
						     SHIFT_CSR_DSP | 0x00);
				ist3602_send(priv, ' ' | 0x100);
				ist3602_send(priv, SHIFT_CSR_DSP | 0x00);
			}
			break;
		case 0x09:	/* print a space instead of the tab */
			ist3602_put_char(priv, ' ');
			break;
		case 0x0C:	/* clear the display */
			ist3602_cls_home(priv, 1);
			break;
		case 0x0A:	/* Line feed */
			priv->line = (priv->line + 1) % priv->lines;
			priv->column = 0;
			ist3602_set_pos(priv);
			break;
		case 0x0D:	/* go to the beginning of the same line */
			priv->column = 0;
			ist3602_set_pos(priv);
			break;
		default:
			ist3602_put_char(priv, c);
		}
	}
}

static ssize_t ist3602_write(struct file *file, const char __user *buf,
			     size_t count, loff_t *ppos)
{
	struct ist3602_priv *priv = to_ist3602_priv(file);
	const char __user *tmp = buf;
	char c;

	for (; count-- > 0; (*ppos)++, tmp++) {
		if (!in_interrupt() && (((count + 1) & 0x1f) == 0))
			/*
			 * let's be a little nice with other processes
			 * that need some CPU
			 */
			schedule();

		if (get_user(c, tmp))
			return -EFAULT;

		ist3602_write_char(priv, c);
	}

	return tmp - buf;
}

static int ist3602_open(struct inode *inode, struct file *file)
{
	int ret;

	ret = -EBUSY;
	if (!atomic_dec_and_test(&ist3602_available))
		goto fail;	/* open only once at a time */

	ret = -EPERM;
	if (file->f_mode & FMODE_READ)	/* device is write-only */
		goto fail;

	return nonseekable_open(inode, file);

fail:
	atomic_inc(&ist3602_available);
	return ret;
}

static int ist3602_release(struct inode *inode, struct file *file)
{
	atomic_inc(&ist3602_available);
	return 0;
}

static long ist3602_ioctl(struct file *file, unsigned int cmd,
			  unsigned long arg)
{
	struct ist3602_priv *priv = to_ist3602_priv(file);
	struct ist3602_params *params = (struct ist3602_params *)&(priv->width);

	switch (cmd) {
	case IST3602_GETPARAMS:
		if (copy_to_user((void __user *)arg, params,
		    sizeof(struct ist3602_params)))
			return -EFAULT;
		break;
	default:
		DEBUG("unsupported command %d\n", cmd);
		return -EFAULT;
	}
	return 0;
}

static const struct file_operations ist3602_fops = {
	.write   = ist3602_write,
	.open    = ist3602_open,
	.release = ist3602_release,
	.llseek  = no_llseek,
	.unlocked_ioctl = ist3602_ioctl,
#ifdef CONFIG_COMPAT
	.compat_ioctl = ist3602_ioctl,
#endif
};

static int ist3602_probe(struct spi_device *spi)
{
	struct ist3602_priv *priv;
	struct device *dev = &spi->dev;
	struct device_node *np = dev->of_node;
	static struct lock_class_key key;
	int ret = 0;
	int val;
	int i;
	const char *boot_msg;

	DEBUG("enter\n");
	priv = devm_kzalloc(dev, sizeof(*priv), GFP_KERNEL);
	if (!priv) {
		ret = -ENOMEM;
		dev_err(dev, "Failed to allocate memory (%d)\n", ret);
		return ret;
	}

	priv->buff = devm_kzalloc(dev, sizeof(int) * BUFFER_SIZE, GFP_KERNEL);
	if (!priv->buff) {
		ret = -ENOMEM;
		dev_err(dev, "Failed to allocate buffer memory (%d)\n", ret);
		return ret;
	}

	/* Default settings */
	priv->width = 16;
	priv->lines = 2;
	priv->fonts = 1;
	priv->contrast_std = 170;
	priv->ratio_std = 8;
	priv->contrast_dmr = 90;
	priv->ratio_dmr = 8;
	priv->column_offset = 0;
	priv->frame_rate = 70;
	priv->bias = 0;
	priv->booster = 0;
	priv->dimmer_flag = 0;
	priv->spi = spi;
	mutex_init(&priv->buff_lock);
	__init_waitqueue_head(&priv->buff_wait, "buff_wait", &key);
	spi_set_drvdata(spi, priv);

	priv->power = devm_regulator_get(dev, "lcd");
	if (IS_ERR(priv->power)) {
		ret = PTR_ERR(priv->power);
		dev_err(dev, "no power-supply (%d)\n", ret);
		return ret;
	}
	priv->backlight = devm_regulator_get(dev, "lcd_bright");
	if (IS_ERR(priv->backlight)) {
		ret = PTR_ERR(priv->backlight);
		dev_err(dev, "no backlight-supply (%d)\n", ret);
		return ret;
	}
	priv->lcd_en_gpio = devm_gpiod_get(dev, "lcd_en", GPIOD_OUT_LOW);
	if (IS_ERR(priv->lcd_en_gpio)) {
		ret = PTR_ERR(priv->lcd_en_gpio);
		if (ret != -ENOENT) {
			dev_err(dev, "Failed to get GPIO for enable LCD (%d)\n",
					ret);
			return ret;
		}
		priv->lcd_en_gpio = NULL;
	}
	priv->lcd_rst_gpio = devm_gpiod_get(dev, "lcd_rst", GPIOD_OUT_HIGH);
	if (IS_ERR(priv->lcd_rst_gpio)) {
		ret = PTR_ERR(priv->lcd_rst_gpio);
		if (ret != -ENOENT) {
			dev_err(dev, "Failed to get GPIO for LCD reset (%d)\n",
					ret);
			return ret;
		}
		priv->lcd_rst_gpio = NULL;
	}
	priv->dimmer_gpio = devm_gpiod_get(dev, "dimmer", GPIOD_OUT_LOW);
	if (IS_ERR(priv->dimmer_gpio)) {
		ret = PTR_ERR(priv->dimmer_gpio);
		if (ret != -ENOENT) {
			dev_err(dev, "Failed to get GPIO for dimmer (%d)\n",
					ret);
			return ret;
		}
		priv->dimmer_gpio = NULL;
	}
	ret = device_property_read_u32(dev, "display-height-chars", &val);
	if (!ret)
		priv->lines = val;
	ret = device_property_read_u32(dev, "display-width-chars", &val);
	if (!ret)
		priv->width = val;
	ret = device_property_read_u32(dev, "display-font-size", &val);
	if (!ret)
		priv->fonts = val;
	ret = device_property_read_u32(dev, "contrast", &val);
	if (!ret)
		priv->contrast_std = val;
	ret = device_property_read_u32(dev, "regulation-ratio", &val);
	if (!ret)
		priv->ratio_std = val;
	ret = device_property_read_u32(dev, "contrast-dimmer", &val);
	if (!ret)
		priv->contrast_dmr = val;
	ret = device_property_read_u32(dev, "regulation-ratio-dimmer", &val);
	if (!ret)
		priv->ratio_dmr = val;
	ret = device_property_read_u32(dev, "column-offset", &val);
	if (!ret)
		priv->column_offset = val;
	ret = device_property_read_u32(dev, "frame-rate", &val);
	if (!ret)
		priv->frame_rate = val;
	ret = device_property_read_u32(dev, "bias", &val);
	if (!ret)
		priv->bias = val & 0x03;
	ret = device_property_read_u32(dev, "booster", &val);
	if (!ret)
		priv->booster = val & 0x03;
	if (of_property_read_bool(np, "dimmer_on_boot"))
		priv->dimmer_flag = 1;
	priv->enable_dmr_gpio = of_property_read_bool(np, "enable_dimmer_gpio");
	priv->backlight_link = of_property_read_bool(np, "backlight_link");

	spi->mode = SPI_MODE_3;
	spi->bits_per_word = 9;
	ret = spi_setup(spi);
	if (ret) {
		dev_err(dev, "Failed to setup spi (%d)\n", ret);
		return ret;
	}

	priv->miscdev.minor	= LCD_MINOR,
	priv->miscdev.name	= "lcd",
	priv->miscdev.fops	= &ist3602_fops,
	ret = misc_register(&priv->miscdev);
	if (ret) {
		dev_err(dev, "Failed to register misc device (%d)\n", ret);
		return ret;
	}

	pm_runtime_enable(dev);
	if (of_property_read_bool(np, "boot_on")) {
		DEBUG("enable regulators\n");
		pm_runtime_forbid(dev);
		device_property_read_string(dev, "boot_message", &boot_msg);
		if (boot_msg) {
			for (i = 0; i < strlen(boot_msg); i++)
				ist3602_write_char(priv, boot_msg[i]);
		}
	}

	DEBUG("done\n");
	return ret;
}

static int ist3602_remove(struct spi_device *spi)
{
	struct ist3602_priv *priv = spi_get_drvdata(spi);

	DEBUG("enter\n");
	misc_deregister(&priv->miscdev);

	if (pm_runtime_active(&spi->dev)) {
		DEBUG("runtime still active\n");
		pm_runtime_allow(&spi->dev);
	}

	pm_runtime_disable(&spi->dev);

	DEBUG("done\n");
	return 0;
}

static int ist3602_power_control(struct device *dev, int on)
{
	struct ist3602_priv *priv = dev_get_drvdata(dev);
	int ret = 0;

	DEBUG("enter\n");
	if (on) {
		DEBUG("ON\n");
		if (priv->lcd_rst_gpio)
			gpiod_set_value_cansleep(priv->lcd_rst_gpio, 1);
		ret = ist3602_set_backlight(priv, 1);
		if (ret)
			return ret;
		usleep_range(5000, 5010);
		ret = regulator_enable(priv->power);
		if (ret) {
			dev_err(dev, "Failed to enable regulator (%d)\n", ret);
			return ret;
		}
		if (priv->lcd_en_gpio)
			gpiod_set_value_cansleep(priv->lcd_en_gpio, 1);
		usleep_range(10, 20);

		pinctrl_pm_select_default_state(dev);

		if (priv->lcd_rst_gpio) {
			gpiod_set_value_cansleep(priv->lcd_rst_gpio, 0);
			usleep_range(50, 60);
		}

		mutex_lock(&priv->buff_lock);
		priv->bufwrp = priv->bufrdp = priv->bufcnt = 0;
		mutex_unlock(&priv->buff_lock);
		priv->buff_task = kthread_run(ist3602_thread,
					(void *)priv, "ist3602_task");
		if (IS_ERR(priv->buff_task)) {
			dev_err(dev, "Failed thread run (%ld)\n",
						PTR_ERR(priv->buff_task));
			return PTR_ERR(priv->buff_task);
		}

		ret = ist3602_initialize(priv);
	} else {
		DEBUG("OFF\n");
		if (priv->buff_task) {
			kthread_stop(priv->buff_task);
			priv->buff_task = NULL;
		}

		if (priv->dimmer_gpio)
			gpiod_set_raw_value_cansleep(priv->dimmer_gpio, 0);

		if (priv->lcd_rst_gpio)
			gpiod_set_value_cansleep(priv->lcd_rst_gpio, 1);

		pinctrl_pm_select_sleep_state(dev);

		if (priv->lcd_en_gpio)
			gpiod_set_value_cansleep(priv->lcd_en_gpio, 0);
		ret = regulator_disable(priv->power);
		if (ret) {
			dev_err(dev, "Failed to disable regulator (%d)\n", ret);
			return ret;
		}
		ret = ist3602_set_backlight(priv, 0);
		if (ret)
			return ret;
	}
	DEBUG("done\n");
	return ret;
}

static int ist3602_suspend(struct device *dev)
{
	if (pm_runtime_active(dev))
		return ist3602_power_control(dev, 0);
	return 0;
}

static int ist3602_resume(struct device *dev)
{
	if (pm_runtime_active(dev))
		return ist3602_power_control(dev, 1);
	return 0;
}

static int ist3602_runtime_suspend(struct device *dev)
{
	return ist3602_power_control(dev, 0);
}

static int ist3602_runtime_resume(struct device *dev)
{
	return ist3602_power_control(dev, 1);
}

static const struct dev_pm_ops ist3602_pm_ops = {
	.suspend = ist3602_suspend,
	.resume = ist3602_resume,
	.runtime_suspend = ist3602_runtime_suspend,
	.runtime_resume = ist3602_runtime_resume,
};

static const struct of_device_id ist3602_of_match[] = {
	{ .compatible = "ist,ist3602" },
	{ /* sentinel */ }
};
MODULE_DEVICE_TABLE(of, ist3602_of_match);

static struct spi_driver ist3602_spi_driver = {
	.driver = {
		.name  = IST3602_DRV_DEVNAME,
		.owner = THIS_MODULE,
		.pm = &ist3602_pm_ops,
		.of_match_table = ist3602_of_match,
	},
	.probe  = ist3602_probe,
	.remove = ist3602_remove,
};
module_spi_driver(ist3602_spi_driver);

MODULE_AUTHOR("Sony Corporation");
MODULE_DESCRIPTION("IST3602 Character LCD driver");
MODULE_LICENSE("GPL");
