/****************************************************************************/
/* AUTHOR:  Nagase                                                          */
/* PURPOSE: EDID OWN SPEC                                                   */
/*                                                                          */
/* Copyright 2021 Sony Corporation                                          */
/****************************************************************************/
#ifndef _HD_EDID_OWN_SPEC_C_
#define _HD_EDID_OWN_SPEC_C_

#define HD_EDID__PRODUCT_ID    0x05D1
#define HD_EDID__WEEK_YEAR    0x1E31

/*==== Include =============================================================*/
#include "hd_edid_own_spec.h"
#include "hd_define.h"

/*==== Typedef =============================================================*/

/*==== Local Variables =====================================================*/

/* Default Edid Data */
const uint8_t s_HdEdidDefaultBasicDisplay[] = {		/* BASIC Display Parameter */
	0x80,0x50,0x2D,0x78,0x0A
};

const uint8_t s_HdEdidDefaultColor[] = {			/* Color Characteristics */
	0x0D,0xC9,0xA0,0x57,0x47,0x98,0x27,0x12,0x48,0x4C
};

const HD_EDID_DTD_TIMING_t s_HdEdidDefaultDtdTiming[] = {	/* Detailed Timing */
	{0x04,{0x01,0x1D,0x00,0x72,0x51,0xD0,0x1E,0x20,0x6E,0x28,0x55,0x00,0xC4,0x8E,0x21,0x00,0x00,0x1E}},	/* vic4, 1280x720p(16:9) */
	{0x03,{0x8C,0x0A,0xD0,0x8A,0x20,0xE0,0x2D,0x10,0x10,0x3E,0x96,0x00,0xC4,0x8E,0x21,0x00,0x00,0x18}},	/* vic3,  720x480p(16:9) */
	{0x12,{0x8C,0x0A,0xD0,0x90,0x20,0x40,0x31,0x20,0x0C,0x40,0x55,0x00,0xC4,0x8E,0x21,0x00,0x00,0x18}},	/* vic18, 720x576p(16:9) */
	{0x13,{0x01,0x1D,0x00,0xBC,0x52,0xD0,0x1E,0x20,0xB8,0x28,0x55,0x40,0xC4,0x8E,0x21,0x00,0x00,0x1E}}	/* vic19,1280x720p(16:9) */
};

const uint8_t s_HdEdidDefaultDtdName[] = {
	0x00,0x00,0x00,0xFC,0x00,0x53,0x4F,0x4E,0x59,0x20,0x41,0x56,0x53,0x59,0x53,0x54,0x45,0x4d		/* Monitor Name (SONY AVSYSTEM)*/
};

const uint8_t s_HdEdidDefaultDtdRange[] = {
	0x00,0x00,0x00,0xFD,0x00,0x31,0x3D,0x0F,0x2E,0x08,0x00,0x0A,0x20,0x20,0x20,0x20,0x20,0x20		/* Monitor Range Limit		*/
};

const uint8_t s_HdEdidDefaultDtdSirial[] = {
	0x00,0x00,0x00,0xFE,0x00,0x0A,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20
};

const uint8_t s_HdEdidDefaultDtdVga[] = {
	0xd6,0x09,0x80,0xa0,0x20,0xe0,0x2d,0x10,0x08,0x60,0x22,0x00,0x12,0x8e,0x21,0x08,0x08,0x18		/* VGA						*/
};

const uint8_t s_HdEdidDefaultDtv = HD_EDID_MASK_DTV_AUDIO | HD_EDID_MASK_DTV_YCBCR444 | HD_EDID_MASK_DTV_YCBCR422;
		/* audio, YCbCr444, YCbCr422 */

const HD_EDID_VIDEO_DATA_t s_HdEdidDefaultVideo[] = {
	{HD_VIC97,	0x0104},	/* 3840x2160p(60Hz) 16:9		, 6G */
	{HD_VIC96,	0x0104},	/* 3840x2160p(50Hz) 16:9		, 6G */
	{HD_VIC93,	0x0100},	/* 3840x2160p(24Hz) 16:9 */
	{HD_VIC94,	0x0100},	/* 3840x2160p(25Hz) 16:9 */
	{HD_VIC95,	0x0100},	/* 3840x2160p(30Hz) 16:9 */
	{HD_VIC98,	0x0100},	/* 4096x2160p(24Hz) 256:135 */
	{HD_VIC16,	0x0100},	/* 1920x1080p(60Hz) 16:9 */
	{HD_VIC31,	0x0100},	/* 1920x1080p(50Hz) 16:9 */
	{HD_VIC5,	0x0100},	/* 1920x1080i(60Hz) 16:9 */
	{HD_VIC4,	0x0100},	/* 1280x 720p(60Hz) 16:9 */
	{HD_VIC19,	0x0100},	/* 1280x 720p(50Hz) 16:9 */
	{HD_VIC20,	0x0100},	/* 1920x1080i(50Hz) 16:9 */
	{HD_VIC32,	0x0100},	/* 1920x1080p(24Hz) 16:9 */
	{HD_VIC34,	0x0100},	/* 1920x1080p(30Hz) 16:9 */
	{HD_VIC33,	0x0100},	/* 1920x1080p(25Hz) 16:9 */
	{HD_VIC60,	0x0100},	/* 1280x 720p(24Hz) 16:9 */
	{HD_VIC62,	0x0100},	/* 1280x 720p(30Hz) 16:9 */
	{HD_VIC3,	0x4100},	/*  720x 480p(60Hz) 16:9		, Native */
	{HD_VIC18,	0x8100},	/*  720x 576p(50Hz) 16:9		, Mandatory */
	{HD_VIC2,	0x8000},	/*  720x 480p(60Hz)  4:3		, Mandatory */
	{HD_VIC17,	0x8100},	/*  720x 576p(50Hz)  4:3		, Mandatory */
	{HD_VIC1,	0x8000},	/*  640x 480p(60Hz)  4:3		, Mandatory */
	{HD_VIC101,	0x0104},	/* 4096x2160p(50Hz) 256:135		, 6G*/
	{HD_VIC102,	0x0104},	/* 4096x2160p(60Hz) 256:135		, 6G*/
	{HD_VIC99,	0x0100},	/* 4096x2160p(25Hz) 256:135 */
	{HD_VIC100,	0x0100},	/* 4096x2160p(30Hz) 256:135 */
	{HD_VIC63,	0x0110},	/* 1920x1080p(120Hz) 			, 2ND_BLOCK */
	{HD_VIC64,	0x0110},	/* 1920x1080p(100Hz)			, 2ND_BLOCK */
	{HD_VIC117,	0x011a},	/* 3840x2160p(100Hz)			, 8K, Y420, 2ND_BLOCK */
	{HD_VIC118,	0x011a},	/* 3840x2160p(120Hz)			, 8K, Y420, 2ND_BLOCK */
	{HD_VIC194,	0x011a},	/* 7680x4320p(24Hz)				, 8K, Y420, 2ND_BLOCK */
	{HD_VIC195,	0x011a},	/* 7680x4320p(25Hz)				, 8K, Y420, 2ND_BLOCK */
	{HD_VIC196,	0x011a},	/* 7680x4320p(30Hz)				, 8K, Y420, 2ND_BLOCK */
	{HD_VIC198,	0x011a},	/* 7680x4320p(50Hz)				, 8K, Y420, 2ND_BLOCK */
	{HD_VIC199,	0x011a},	/* 7680x4320p(60Hz)				, 8K, Y420, 2ND_BLOCK */
	{HD_VIC218,	0x011a},	/* 4096x2160p(100Hz)			, 8K, Y420, 2ND_BLOCK */
	{HD_VIC219,	0x011a},	/* 4096x2160p(120Hz)			, 8K, Y420, 2ND_BLOCK */
};

const HD_EDID_AUDIO_DATA_t s_HdEdidDefaultAudio[] = {
	{HD_EDID_SAD_KIND__LPCM,		HD_EDID_SAD_FORMAT_PCM,				0x00,								0x07, 0x7f, 0x00, 0x07},	/* LPCM MAX=8ch/Fs192,176,96,88,48,44,32KHz/24,20,16bit */
	{HD_EDID_SAD_KIND__AC3,			HD_EDID_SAD_FORMAT_AC3,				0x00,								0x05, 0x07, 0x00, 0x50},	/* AC3 MAX=6ch/Fs48,44,32KHz/640bps */
	{HD_EDID_SAD_KIND__MPEG2AAC,	HD_EDID_SAD_FORMAT_AAC,				0x00,								0x05, 0x07, 0x00, 0x30},	/* AAC MAX=6ch/Fs48,44,32KHz/384bps */
	{HD_EDID_SAD_KIND__DTS,			HD_EDID_SAD_FORMAT_DTS,				0x00,								0x05, 0x1f, 0x00, 0xc0},	/* DTS MAX=6ch/Fs96,88,48,44,32KHz/1536bps */
	{HD_EDID_SAD_KIND__DDP,			HD_EDID_SAD_FORMAT_DDP,	 			0x00,								0x07, 0x04, 0x00, 0x03},	/* DD+ MAX=8ch/Fs48KHz */
	{HD_EDID_SAD_KIND__MAT,			HD_EDID_SAD_FORMAT_MAT,	 			0x00,								0x07, 0x54, 0x00, 0x03},	/* MAT(TrueHD) MAX8=ch/Fs192,176,96,88,48KHz */
	{HD_EDID_SAD_KIND__DTSX,		HD_EDID_SAD_FORMAT_DTSHD,			0x00,								0x07, 0x7e, 0x00, HD_EDID_SAD_FORMAT_SUB_DTSUHD|0x01},
	{HD_EDID_SAD_KIND__DTSHD,		HD_EDID_SAD_FORMAT_DTSHD,			0x00,								0x07, 0x7e, 0x00, 0x01},	/* DTS-HD MAX=8ch/Fs192,176,96,88,48,44KHz/master audio suport */
	{HD_EDID_SAD_KIND__DSD,			HD_EDID_SAD_FORMAT_DSD,				0x00,								0x05, 0x02, 0x00, 0x00},	/* DSD MAX=6ch/Fs44KHz */
	{HD_EDID_SAD_KIND__MPEG4AAC,	HD_EDID_SAD_FORMAT_EXTENSION_TYPE,	HD_EDID_SAD_FORMAT_MPEG_4AAC,		0x05, 0x04, 0x00, 0x34}		/* MPEG4_AAC MAX=6ch/1024_TL */
};

const HD_EDID_AUDIO_DATA_t s_HdEdidDefaultAudioBasic = {
	HD_EDID_SAD_KIND__LPCM, HD_EDID_SAD_FORMAT_PCM, 0x01, 0x00, 0x07, 0x00, 0x07		/* LPCM MAX=2ch/Fs48,44,32KHz/24,20,16bit */
};

const HD_EDID_SPD_DATA_t s_HdEdidDefaultSpd = {
	{0x5f,0x00,0x00}
};

const HD_EDID_SPD_DATA_t s_HdEdidDefaultSpdBasic = {
	{0x01,0x00,0x00}
};

const HD_EDID_VSDB_EXTENSION_t s_HdEdidDefaultVsdb = {
	0x01,		/* Support AI */
	0x07,		/* DeepColor 36bit, 30bit, Y444 */
	0x00,		/* reserve */
	0x3c,		/* Max TMDS Clock 300(297)MHz */
	0x0f,		/* CNC */
	0x01,		/* 3D Present */
	0x00,		/* Image Size */
	0x00,		/* Reserve */
	0x04,		/* HDMI VIC Len */
	0x00,		/* Reserve */
	{0x01, 0x02, 0x03, 0x04, 0x00, 0x00, 0x00}	/* HDMI VIC DATA */
};

const HD_EDID_SCDS_EXTENSION_t s_HdEdidDefaultScds = {
	0x01,		/* Version */
	0x78,		/* Max TMDS Clock 600MHz */
	0x01,		/* SCDC */
	0x00,		/* ReadReq Capable NonSupport */
	0x00,		/* CABLE_STATUS */
	0x00,		/* CCBPCI */
	0x00,		/* LTE Scramble */
	0x00,		/* Independent View */
	0x00,		/* Dual View */
	0x00,		/* 3D OSD Disparity */
	0x05,		/* Max FRL Rate */
	0x00,		/* UHD_VIC */
	0x03,		/* DeepColor Y420 42bit, 40bit */
	0x00,		/* M_Delta */
	0x00, 		/* CinemaVRR */
	0x00,		/* CNMVRR */
	0x00,		/* FVA */
	0x00,		/* ALLM */
	0x00,		/* FAPA_start_location */
	0x00,		/* VRR MAX_9_8 */
	0x00,		/* VRR_MIN */
	0x00,		/* VRR_MAX_7_0 */
	0x00,		/* DSC_1p2 */
	0x00,		/* DSC_Native_420 */
	0x00,		/* DSC_All_bpp */
	0x00,		/* DSC_16bpc */
	0x00,		/* DSC_12bpc */
	0x00,		/* DSC_10bpc */
	0x00,		/* DSC_MaxSlices */
	0x00		/* DSC_TotalChunkKBytes */
};

const HD_EDID_EXT_VCDB_DATA_t s_HdEdidDefaultExtVcdb = {
	0x01,0x01,0x03,0x03,0x03
};

const HD_EDID_EXT_CMR_DATA_t s_HdEdidDefaultExtCmr = {
	0x06,	/* BT2020 RGB, BT2020 YCC */
	0x03,	/* AdobeRGB, AdobeYCC601 */
	0x07,	/* sYCC601, xvYCC709, xvYCC601 */
	0x00,
	0x01	/* Metadata MD0 */
};

const HD_EDID_EXT_HDR_DATA_t s_HdEdidDefaultExtHdr = {
	0x0f, 0x01, {0xff, 0xff, 0x00}
};

const uint8_t s_HdEdidDefaultVsdbHdmiVic[] = {
	0x01, 0x02, 0x03, 0x04, 0x00, 0x00, 0x00	/* HDMI VIC DATA */
};

/* HDMI2.0 Y420 EDID */
const uint8_t s_HdEdidDefaultExtY420vdb[] = {
	HD_VIC96,		/* 3840x2160p/50Hz 16:9 */
	HD_VIC97,		/* 3840x2160p/60HZ 16:9 */
	HD_VIC101,		/* 4096x2160p/50Hz 16:9 */
	HD_VIC102,		/* 4096x2160p/60HZ 16:9 */
/* 	HD_VIC106,		 *//* 3840x2160p/50Hz 4:3 */
/* 	HD_VIC107 		 *//* 3840x2160p/60HZ 4:3 */
	HD_VIC117,		/* 3840x2160p/100HZ */
	HD_VIC118,		/* 3840x2160p/120HZ */
	HD_VIC194,		/* 7680x4320p/24HZ */
	HD_VIC195,		/* 7680x4320p/25HZ */
	HD_VIC196,		/* 7680x4320p/30HZ */
	HD_VIC198,		/* 7680x4320p/50HZ */
	HD_VIC199,		/* 7680x4320p/60HZ */
	HD_VIC218,		/* 4096x2160p/100HZ */
	HD_VIC219		/* 4096x2160p/120HZ */
};

/*==== Define ==============================================================*/
#define HD_EDID_DEFAULT_SVD_TBL_SIZE		(sizeof(s_HdEdidDefaultVideo) / sizeof(s_HdEdidDefaultVideo[0]))
#define HD_EDID_DEFAULT_Y420VDB_TBL_SIZE	(sizeof(s_HdEdidDefaultExtY420vdb) / sizeof(s_HdEdidDefaultExtY420vdb[0]))

#define HD_EDID_PRODUCTID_MASK	0x00ff
#define HD_EDID_PRODUCTID_SHIFT	8

#define HD_EDID_YEARWEEK_MASK	0x00ff
#define HD_EDID_YEARWEEK_SHIFT	8

/*==== Code ================================================================*/
/*--------------------------------------------------------------------------*/
/* @brief     : Get Own Manufacture ID                                      */
/*--------------------------------------------------------------------------*/
void hd_edid_GetOwnManufactureId(uint8_t *p_buff)
{
	if (p_buff == NULL) {
		return;
	}

	p_buff[0] = 0x4d;
	p_buff[1] = 0xd9;
}

/*--------------------------------------------------------------------------*/
/* @brief     : Get Own Product ID                                          */
/*--------------------------------------------------------------------------*/
void hd_edid_GetOwnProductId(uint8_t *p_buff)
{
	if (p_buff == NULL) {
		return;
	}

	p_buff[0] = HD_EDID__PRODUCT_ID & HD_EDID_PRODUCTID_MASK;
	p_buff[1] = (HD_EDID__PRODUCT_ID >> HD_EDID_PRODUCTID_SHIFT) & HD_EDID_PRODUCTID_MASK;
}

/*--------------------------------------------------------------------------*/
/* @brief     : Get Own Week & Year                                         */
/*--------------------------------------------------------------------------*/
void hd_edid_GetOwnWeekYear(uint8_t *p_week, uint8_t *p_year)
{
	if ((p_year == NULL) || (p_week == NULL)) {
		return;
	}

	*p_year = (HD_EDID__WEEK_YEAR >> HD_EDID_PRODUCTID_SHIFT) & HD_EDID_YEARWEEK_MASK;
	*p_week = HD_EDID__WEEK_YEAR & HD_EDID_YEARWEEK_MASK;
}

/*--------------------------------------------------------------------------*/
/* @brief     : Get Own EDID Version                                        */
/*--------------------------------------------------------------------------*/
void hd_edid_GetOwnEdidVersion(uint8_t *p_buff)
{
	if (p_buff == NULL) {
		return;
	}

	p_buff[0] = 0x01;
	p_buff[1] = 0x03;
}

/*--------------------------------------------------------------------------*/
/* @brief     : Get Own Basic Display                                       */
/*--------------------------------------------------------------------------*/
void hd_edid_GetOwnBasicDisplay(uint8_t *p_buff)
{
	uint8_t i;

	if (p_buff == NULL) {
		return;
	}

	for (i = 0; i < HD_EDID_LEN_BASIC_DISPLAY; i++) {
		p_buff[i] = s_HdEdidDefaultBasicDisplay[i];
	}
}

/*--------------------------------------------------------------------------*/
/* @brief     : Get Own Color                                               */
/*--------------------------------------------------------------------------*/
void hd_edid_GetOwnColor(uint8_t *p_buff)
{
	uint8_t i;

	if (p_buff == NULL) {
		return;
	}

	for (i = 0; i < HD_EDID_LEN_COLOR; i++) {
		p_buff[i] = s_HdEdidDefaultColor[i];
	}
}

/*--------------------------------------------------------------------------*/
/* @brief     : Get Own Established Timings                                 */
/*--------------------------------------------------------------------------*/
void hd_edid_GetOwnEtd(uint8_t *p_buff)
{
	if (p_buff == NULL) {
		return;
	}

	/* Established Timings 1-8  640x480@60Hz(IBM,VGA) */
	p_buff[0] = 0x20;
	p_buff[1] = 0x00;
	p_buff[2] = 0x00;
}

/*--------------------------------------------------------------------------*/
/* @brief     : Get Own DTD Timing                                          */
/*--------------------------------------------------------------------------*/
uint8_t hd_edid_GetOwnDtdTiming(HD_EDID_DTD_TIMING_t *p_buff)
{
	uint8_t i;

	if (p_buff == NULL) {
		return 0;
	}

	for (i = 0; i < (sizeof(s_HdEdidDefaultDtdTiming) / sizeof(s_HdEdidDefaultDtdTiming[0])); i++) {
		p_buff[i] = s_HdEdidDefaultDtdTiming[i];
	}
	return i;
}

/*--------------------------------------------------------------------------*/
/* @brief     : Get Own DTD Name                                            */
/*--------------------------------------------------------------------------*/
void hd_edid_GetOwnDtdName(uint8_t *p_buff)
{
	uint8_t i;

	if (p_buff == NULL) {
		return;
	}

	for (i = 0; i < HD_EDID_LEN_DTD; i++) {
		p_buff[i] = s_HdEdidDefaultDtdName[i];
	}
}

/*--------------------------------------------------------------------------*/
/* @brief     : Get Own DTD Range                                           */
/*--------------------------------------------------------------------------*/
void hd_edid_GetOwnDtdRange(uint8_t *p_buff)
{
	uint8_t i;

	if (p_buff == NULL) {
		return;
	}

	for (i = 0; i < HD_EDID_LEN_DTD; i++) {
		p_buff[i] = s_HdEdidDefaultDtdRange[i];
	}
}

/*--------------------------------------------------------------------------*/
/* @brief     : Get Own DTD Sirial                                          */
/*--------------------------------------------------------------------------*/
void hd_edid_GetOwnDtdSirial(uint8_t *p_buff)
{
	uint8_t i;

	if (p_buff == NULL) {
		return;
	}

	for (i = 0; i < HD_EDID_LEN_DTD; i++) {
		p_buff[i] = s_HdEdidDefaultDtdSirial[i];
	}
}

/*--------------------------------------------------------------------------*/
/* @brief     : Get Own DTD VGA                                             */
/*--------------------------------------------------------------------------*/
void hd_edid_GetOwnDtdVga(uint8_t *p_buff)
{
	uint8_t i;

	if (p_buff == NULL) {
		return;
	}

	for (i = 0; i < HD_EDID_LEN_DTD; i++) {
		p_buff[i] = s_HdEdidDefaultDtdVga[i];
	}
}

/*--------------------------------------------------------------------------*/
/* @brief     : Get Own DTV                                                 */
/*--------------------------------------------------------------------------*/
void hd_edid_GetOwnDtv(uint8_t *p_dtv)
{
	if (p_dtv == NULL) {
		return;
	}
	*p_dtv = s_HdEdidDefaultDtv;
}

/*--------------------------------------------------------------------------*/
/* @brief     : Get Own Video                                               */
/*--------------------------------------------------------------------------*/
uint8_t hd_edid_GetOwnVideo(HD_EDID_VIDEO_DATA_t *p_video_data)
{
	uint8_t i, loop_max, data_cnt = 0;

	if (p_video_data == NULL) {
		return data_cnt;
	}

	loop_max = sizeof(s_HdEdidDefaultVideo) / sizeof(s_HdEdidDefaultVideo[0]);
	for (i = 0; i < loop_max; i++) {
		p_video_data[data_cnt] = s_HdEdidDefaultVideo[i];
		data_cnt++;
	}
	return data_cnt;
}

/*--------------------------------------------------------------------------*/
/* @brief     : Get Own Audio                                               */
/*--------------------------------------------------------------------------*/
uint8_t hd_edid_GetOwnAudio(HD_EDID_AUDIO_DATA_t *p_sad)
{
	uint8_t i, data_cnt = 0;

	if (p_sad == NULL) {
		return data_cnt;
	}

	for (i = 0; i < (sizeof(s_HdEdidDefaultAudio) / sizeof(s_HdEdidDefaultAudio[0])); i++) {
		p_sad[data_cnt] = s_HdEdidDefaultAudio[i];
		data_cnt++;
	}
	return data_cnt;
}

/*--------------------------------------------------------------------------*/
/* @brief     : Get Own Audio Basic                                         */
/*--------------------------------------------------------------------------*/
void hd_edid_GetOwnAudioBasic(HD_EDID_AUDIO_DATA_t *p_sad)
{
	if (p_sad == NULL) {
		return;
	}

	*p_sad = s_HdEdidDefaultAudioBasic;

	return;
}

/*--------------------------------------------------------------------------*/
/* @brief     : Get Own SPD                                                 */
/*--------------------------------------------------------------------------*/
bool_t hd_edid_GetOwnSpd(HD_EDID_SPD_DATA_t *p_spd)
{
	if (p_spd == NULL) {
		return HD_FALSE;
	}

	*p_spd = s_HdEdidDefaultSpd;

	return HD_TRUE;
}

/*--------------------------------------------------------------------------*/
/* @brief     : Get Own SPD Basic                                           */
/*--------------------------------------------------------------------------*/
void hd_edid_GetOwnSpdBasic(HD_EDID_SPD_DATA_t *p_spd)
{
	if (p_spd == NULL) {
		return;
	}

	*p_spd = s_HdEdidDefaultSpdBasic;

	return;
}

/*--------------------------------------------------------------------------*/
/* @brief     : Get Own VSBD                                                */
/*--------------------------------------------------------------------------*/
bool_t hd_edid_GetOwnVsdb(HD_EDID_VSDB_EXTENSION_t *p_vsdb)
{
	if (p_vsdb == NULL) {
		return HD_FALSE;
	}

	*p_vsdb = s_HdEdidDefaultVsdb;

	return HD_TRUE;
}

/*--------------------------------------------------------------------------*/
/* @brief     : Get Own SCDS                                                */
/*--------------------------------------------------------------------------*/
uint8_t hd_edid_GetOwnScds(HD_EDID_SCDS_EXTENSION_t *p_scds)
{
	if (p_scds == NULL) {
		return 0;
	}

	*p_scds = s_HdEdidDefaultScds;

	return HD_EDID_LEN_DESC_SCDS_MAX;
}

/*--------------------------------------------------------------------------*/
/* @brief     : Get Own Ext VCDB                                            */
/*--------------------------------------------------------------------------*/
bool_t hd_edid_GetOwnExtVcdb(HD_EDID_EXT_VCDB_DATA_t *p_vcdb)
{
	if (p_vcdb == NULL) {
		return HD_FALSE;
	}

	*p_vcdb = s_HdEdidDefaultExtVcdb;

	return HD_TRUE;
}

/*--------------------------------------------------------------------------*/
/* @brief     : Get Own Ext Colorimetry                                     */
/*--------------------------------------------------------------------------*/
bool_t hd_edid_GetOwnExtCmr(HD_EDID_EXT_CMR_DATA_t *p_cmr)
{
	if (p_cmr == NULL) {
		return HD_FALSE;
	}

	*p_cmr = s_HdEdidDefaultExtCmr;

	return HD_TRUE;
}

/*--------------------------------------------------------------------------*/
/* @brief     : Get Own Ext HDR                                             */
/*--------------------------------------------------------------------------*/
bool_t hd_edid_GetOwnExtHdr(HD_EDID_EXT_HDR_DATA_t *p_hdr)
{
	if (p_hdr == NULL) {
		return HD_FALSE;
	}

	*p_hdr = s_HdEdidDefaultExtHdr;

	return HD_TRUE;
}

/* HDMI2.0 Y420 EDID */
/*--------------------------------------------------------------------------*/
/* @brief     : Get Own Ext Y420 VDB                                        */
/*--------------------------------------------------------------------------*/
uint8_t hd_edid_GetOwnExtY420vdb(uint8_t *p_vic)
{
	uint8_t i;

	if (p_vic == NULL) {
		return 0;
	}

	for (i = 0; i < HD_EDID_DEFAULT_Y420VDB_TBL_SIZE; i++) {
		p_vic[i] = s_HdEdidDefaultExtY420vdb[i];
	}

	return HD_EDID_DEFAULT_Y420VDB_TBL_SIZE;
}

/*--------------------------------------------------------------------------*/
/* @brief     : Is Support Video                                            */
/*--------------------------------------------------------------------------*/
bool_t hd_edid_IsSupportVideo(uint8_t vic, uint16_t *p_state)
{
	uint8_t i, loop_max;

	if (p_state == NULL) {
		return HD_FALSE;
	}

	loop_max = sizeof(s_HdEdidDefaultVideo) / sizeof(s_HdEdidDefaultVideo[0]);
	for (i = 0; i < loop_max; i++) {
		if (vic == s_HdEdidDefaultVideo[i].vic) {
			*p_state = s_HdEdidDefaultVideo[i].state;
			break;
		}
	}

	if (i < loop_max) {
		return HD_TRUE;
	} else {
		return HD_FALSE;
	}
}

/* HDMI2.0 Y420 EDID */
/*--------------------------------------------------------------------------*/
/* @brief     : Is Support Y420                                             */
/*--------------------------------------------------------------------------*/
bool_t hd_edid_IsSupportY420(uint8_t vic)
{
	bool_t  result = HD_FALSE;
	uint8_t i;

	for (i = 0; i < HD_EDID_DEFAULT_Y420VDB_TBL_SIZE; i++) {
		if (vic == s_HdEdidDefaultExtY420vdb[i]) {
			result = HD_TRUE;
		}
	}
	return result;
}

/* HDMI2.1 HF-SCDB EDID */
/*--------------------------------------------------------------------------*/
/* @brief     : Get Own Ext SCDB                                            */
/*--------------------------------------------------------------------------*/
uint8_t hd_edid_GetOwnExtScdb(HD_EDID_EXT_HFSCDB_DATA_t *p_hfscdb)
{
	if (p_hfscdb == NULL) {
		return HD_FALSE;
	}

	p_hfscdb->hfscdb_data[0] = 0x00;
	p_hfscdb->hfscdb_data[1] = 0x00;
	p_hfscdb->scds_data = s_HdEdidDefaultScds;

	return HD_EDID_LEN_DESC_SCDS_MAX;
}

/*--------------------------------------------------------------------------*/
/* @brief     : Get Support Video State                                     */
/*--------------------------------------------------------------------------*/
bool_t hd_edid_GetSupportVideoState(uint8_t vic, uint16_t *p_state)
{
	uint8_t i, loop_max;

	if (p_state == NULL) {
		return HD_FALSE;
	}

	loop_max = sizeof(s_HdEdidDefaultVideo) / sizeof(s_HdEdidDefaultVideo[0]);
	for (i = 0; i < loop_max; i++) {
		if (vic == s_HdEdidDefaultVideo[i].vic) {
			*p_state = s_HdEdidDefaultVideo[i].state;
			break;
		}
	}

	if (i < loop_max) {
		return HD_TRUE;
	} else {
		return HD_FALSE;
	}
}

/*--------------------------------------------------------------------------*/
/* @brief     : Get Support Audio                                           */
/*--------------------------------------------------------------------------*/
uint8_t hd_edid_GetSupportAudio(uint8_t format, uint8_t param, HD_EDID_AUDIO_DATA_t *p_sad)
{
	uint8_t	i, ret = HD_EDID_SAD_KIND__NOTSUPPORT;
	bool_t	found = HD_FALSE;

	if (p_sad == NULL) {
		return ret;
	}

	for (i = 0; i < (sizeof(s_HdEdidDefaultAudio) / sizeof(s_HdEdidDefaultAudio[0])); i++) {
		if (format == s_HdEdidDefaultAudio[i].format_code) {
			switch (format) {
			case HD_EDID_SAD_FORMAT_DTSHD:
				if ((param & (HD_EDID_MASK_SAD_DTSX_PARAM_V1 | HD_EDID_MASK_SAD_DTSX_PARAM_V2)) == 0) {
					/* Not DTS:X */
					if ((s_HdEdidDefaultAudio[i].param & (HD_EDID_MASK_SAD_DTSX_PARAM_V1 | HD_EDID_MASK_SAD_DTSX_PARAM_V2)) == 0) {
						/* Not DTS:X */
						found = HD_TRUE;
					}
				} else {
					/* DTS:X */
					if ((s_HdEdidDefaultAudio[i].param & (HD_EDID_MASK_SAD_DTSX_PARAM_V1 | HD_EDID_MASK_SAD_DTSX_PARAM_V2)) != 0) {
						/* DTS:X */
						found = HD_TRUE;
					}
				}
				break;
			case HD_EDID_SAD_FORMAT_EXTENSION_TYPE:
				switch ((param & HD_EDID_MASK_SAD_EXTENSION_TYPE_CODE) >> HD_EDID_SHFT_SAD_EXTENSION_TYPE_CODE) {
				case HD_EDID_SAD_FORMAT_MPEG_4AAC:
					if (s_HdEdidDefaultAudio[i].extension_code == HD_EDID_SAD_FORMAT_MPEG_4AAC) {
						found = HD_TRUE;
					}
					break;
				default:
					/* do nothing */
					break;
				}
				break;
			default:
				found = HD_TRUE;
				break;
			}
		}
		if (found != HD_FALSE) {
			*p_sad = s_HdEdidDefaultAudio[i];
			ret = s_HdEdidDefaultAudio[i].format_code;
			break;
		}
	}

	return ret;
}

/*--------------------------------------------------------------------------*/
/* @brief     : Is Support SPD                                              */
/*--------------------------------------------------------------------------*/
bool_t hd_edid_IsSupportSpd(HD_EDID_SPD_DATA_t *p_spd)
{
	if (p_spd == NULL) {
		return HD_FALSE;
	}

	*p_spd = s_HdEdidDefaultSpd;

	return HD_TRUE;
}

/*--------------------------------------------------------------------------*/
/* @brief     : Is Support Audio for eARC                                   */
/*--------------------------------------------------------------------------*/
bool_t hd_edid_IsSupportAudioForEarc(uint8_t format, uint8_t *p_sad)
{
	uint8_t i;

	if (p_sad == NULL) {
		return HD_FALSE;
	}

	for (i = 0; i < (sizeof(s_HdEdidDefaultAudio) / sizeof(s_HdEdidDefaultAudio[0])); i++) {
		if (format == s_HdEdidDefaultAudio[i].format_code) {
			break;
		}
	}

	if (i < (sizeof(s_HdEdidDefaultAudio) / sizeof(s_HdEdidDefaultAudio[0]))) {
		p_sad[0] = ((s_HdEdidDefaultAudio[i].format_code << HD_EDID_SHFT_SAD_FORMAT) | s_HdEdidDefaultAudio[i].max_channel);
		p_sad[1] = s_HdEdidDefaultAudio[i].freq;
		p_sad[2] = s_HdEdidDefaultAudio[i].param;
		return HD_TRUE;
	} else {
		return HD_FALSE;
	}
}

/*--------------------------------------------------------------------------*/
/* @brief     : Is Support VSDB HDMI VIC                                    */
/*--------------------------------------------------------------------------*/
bool_t hd_edid_IsSupportVsdbHdmiVic(uint8_t hdmi_vic)
{
	bool_t  result = HD_FALSE;
	uint8_t i;

	for (i = 0; i < HD_EDID_CNT_VSDB_HDMI_VIC; i++) {
		if (hdmi_vic == s_HdEdidDefaultVsdbHdmiVic[i]) {
			result = HD_TRUE;
		}
	}
	return result;
}

#endif	/* _HD_EDID_OWN_SPEC_C_ */
