/****************************************************************************/
/* AUTHOR:  Nagase                                                          */
/* PURPOSE: EDID RAM                                                        */
/*                                                                          */
/* Copyright 2021 Sony Corporation                                          */
/****************************************************************************/
#define _HD_EDID_RAM_C_

/*==== Include =============================================================*/
#include "hd_edid_ram.h"
#include "hd_edid_own_spec.h"
#include "hd_define.h"

/*==== Define ==============================================================*/
#define HD_EDID_CEA_BLOCK_WRITE_MAX		(2)

#define HD_EDID_BUFF_SIZE_SVD			60
#define HD_EDID_BUFF_SIZE_SAD			20

#define HD_EDID_LEN_Y420CMDB_VIC		HD_EDID_BUFF_SIZE_SVD

#define HD_MAX__MODULE_PORT_TX			2
#define NULL       ((void*)0)

/*==== Typedef =============================================================*/
/* DTD */
typedef struct {
	uint8_t					count;
	HD_EDID_DTD_TIMING_t	timing[HD_EDID_CNT_DTD_TIMING];
	uint8_t					name[HD_EDID_LEN_DTD];
	uint8_t					range[HD_EDID_LEN_DTD];
} HD_EDID_DTD_t;

typedef struct {
	uint16_t	pixel_clock;
	uint16_t	h_active;
	uint16_t	v_active;
	uint16_t	h_blank;
	uint16_t	v_blank;
	bool_t		interlace;
	bool_t		aspect16_9;
} HD_EDID_DTD_DATA_t;

/* DTV MONITOR */
typedef struct {
	uint8_t	data;
} HD_EDID_DTV_t;

/* SVD */
typedef struct {
	uint8_t					count;
	HD_EDID_VIDEO_DATA_t	data[HD_EDID_BUFF_SIZE_SVD];
} HD_EDID_VIDEO_t;

/* SAD */
typedef struct {
	uint8_t					count;
	HD_EDID_AUDIO_DATA_t	data[HD_EDID_BUFF_SIZE_SAD];
} HD_EDID_AUDIO_t;

/* SPD */
typedef struct {
	bool_t				valid;
	HD_EDID_SPD_DATA_t	data;
} HD_EDID_SPD_t;

/* VSDB */
typedef struct {
	bool_t						valid;
	uint8_t						count;
	uint16_t					phy_addr;
	HD_EDID_VSDB_EXTENSION_t	ext;
} HD_EDID_VSDB_t;

/* HF-VSDB */
typedef struct {
	bool_t						valid;
	uint8_t						count;
	HD_EDID_SCDS_EXTENSION_t	data;
} HD_EDID_HFVSDB_t;

/* Colorimetry */
typedef struct {
	bool_t					valid;
	HD_EDID_EXT_CMR_DATA_t	data;
} HD_EDID_EXT_CMR_t;

/* VCDB */
typedef struct {
	bool_t					valid;
	HD_EDID_EXT_VCDB_DATA_t	data;
} HD_EDID_EXT_VCDB_t;

/* VSVDB */
typedef struct {
	bool_t							valid;
	HD_EDID_EXT_VSVDB_DV_TYPE_t		type;
	HD_EDID_EXT_VSVDB_DV_DATA_t		data;
} HD_EDID_EXT_VSVDB_t;

typedef struct {
	bool_t							valid;
	HD_EDID_EXT_VSADB_DATA_t		data;
} HD_EDID_EXT_VSADB_t;

/* HDMI2.0 Y420 EDID */
/* Y420VDB */
typedef struct {
	bool_t		valid;
	uint8_t		count;		/* number of vic */
	uint8_t		vic[HD_EDID_LEN_Y420VDB_VIC];
} HD_EDID_EXT_Y420VDB_t;

/* Y420CMDB */
typedef struct {
	bool_t		valid;
	uint8_t		count;		/* number of mask */
	uint8_t		mask[HD_EDID_LEN_Y420CMDB_MASK];
	uint8_t		vic[HD_EDID_LEN_Y420CMDB_VIC];
} HD_EDID_EXT_Y420CMDB_t;

/* HDR */
typedef struct {
	bool_t						valid;
	uint8_t						length;
	HD_EDID_EXT_HDR_DATA_t		data;
} HD_EDID_EXT_HDR_t;

/* HF-SCDB */
typedef struct {
	bool_t						valid;
	uint8_t						count;
	HD_EDID_EXT_HFSCDB_DATA_t	data;
} HD_EDID_EXT_HFSCDB_t;

/* Extension Data Block */
typedef struct {
	HD_EDID_EXT_CMR_t		cmr;
	HD_EDID_EXT_VCDB_t		vcdb;
	HD_EDID_EXT_VSVDB_t		vsvdb;
	HD_EDID_EXT_VSADB_t		vsadb;
	HD_EDID_EXT_Y420VDB_t	y420vdb;
	HD_EDID_EXT_Y420CMDB_t	y420cmdb;
	HD_EDID_EXT_HDR_t		hdr;
	HD_EDID_EXT_HFSCDB_t	hfscdb;
} HD_EDID_EXT_t;

typedef struct {
	uint8_t				manufacture_id[HD_EDID_LEN_MANUFACTURE];
	uint8_t				product_code[HD_EDID_LEN_PRODUCT];
	uint8_t				week;
	uint8_t				year;
	uint8_t				edid_version[HD_EDID_LEN_EDID_VERSION];
	uint8_t				basic_display[HD_EDID_LEN_BASIC_DISPLAY];
	uint8_t				color[HD_EDID_LEN_COLOR];
	uint8_t				etd[HD_EDID_LEN_ETD];
	HD_EDID_DTD_t		dtd;
	bool_t				is_hdmi;
	HD_EDID_DTV_t		dtv;
	HD_EDID_VIDEO_t		video;
	HD_EDID_AUDIO_t		audio;
	HD_EDID_SPD_t		speaker;
	HD_EDID_VSDB_t		vsdb;
	HD_EDID_HFVSDB_t	hfvsdb;
	HD_EDID_EXT_t		ext;
} HD_EDID_ANALYZE_DATA_t;

typedef struct {
	uint8_t	count;
	uint8_t	timing[HD_EDID_CNT_DTD_TIMING][HD_EDID_LEN_DTD];
} HD_EDID_BUFFER_DTD_t;

/* HDMI2.0 HDMI_VIC copy to SVD */
typedef struct {
	uint8_t	hdmi_vic;
	uint8_t	cea_vic;
	bool_t	isHdmiVic;
} HD_EDID_VIC_CROSS_REF_t;

typedef struct {
	uint8_t					count;
	HD_EDID_VIDEO_DATA_t	data[HD_EDID_BUFF_SIZE_SVD];
} HD_EDID_BUFF_VIDEO_t;

typedef struct {
	HD_EDID_BUFF_VIDEO_t	cea_block[HD_EDID_CEA_BLOCK_WRITE_MAX];
} HD_EDID_BUFF_CEA_BLOCK_t;

/*==== Local Variables =====================================================*/
static uint8_t  s_HdEdidRam[HD_EDID_TABLE_SIZE];

/*---- Sink EDID Data */
static uint8_t *s_HdEdidRamSink[HD_MAX__MODULE_PORT_TX];

/*---- AnalyzeData */
static HD_EDID_ANALYZE_DATA_t s_HdEdidAnalyzeData[HD_MAX__MODULE_PORT_TX];
static HD_EDID_ANALYZE_DATA_t s_HdEdidDefault;

static HD_EDID_BUFF_CEA_BLOCK_t	s_HdEdidVideoWorkBuff;

static uint8_t	s_HdEdidTxSize[HD_MAX__MODULE_PORT_TX];

/*==== Prototype Declaration ===============================================*/
static void hd_edid_RamInitializeDataSink(uint8_t);
static void hd_edid_RamInitializeAnalyze(uint8_t);
static void hd_edid_RamInitializeVideoWorkBuff(void);
static void hd_edid_GetOwnEdid(void);

/* Analyze */
static void hd_edid_AnalyzeBlock0(uint8_t);
static void hd_edid_AnalyzeBlock1(uint8_t, uint8_t);

static void hd_edid_AnalyzeDtd(uint8_t *, HD_EDID_DTD_t *);
static void hd_edid_GetDtdTimingData(uint8_t *, HD_EDID_DTD_DATA_t *);
static bool_t hd_edid_IsSupportDtd(HD_EDID_DTD_DATA_t *, uint8_t *);
static void hd_edid_AnalyzeDtv(uint8_t, HD_EDID_DTV_t *);
static void hd_edid_AnalyzeVideoDataBlock(uint8_t *, HD_EDID_VIDEO_t *);
static void hd_edid_SetVideoData(HD_EDID_VIDEO_t *);
static void hd_edid_AnalyzeAudio(uint8_t *, HD_EDID_AUDIO_t *);
static void hd_edid_AnalyzeSpeaker(uint8_t *, HD_EDID_SPD_t *);
static void hd_edid_AnalyzeVsdb(uint8_t *, HD_EDID_VSDB_t *);
static void hd_edid_AnalyzeVsdbExt(uint8_t *, uint8_t, HD_EDID_VSDB_EXTENSION_t *);
static void hd_edid_AnalyzeVsdbExtVideo(uint8_t *, uint8_t, HD_EDID_VSDB_EXTENSION_t *);
static void hd_edid_AnalyzeHfVsdb(uint8_t *, HD_EDID_HFVSDB_t *);
static void hd_edid_AnalyzeScds(uint8_t *, HD_EDID_SCDS_EXTENSION_t *, uint8_t);

static void hd_edid_AnalyzeExt(uint8_t *, HD_EDID_EXT_t *);
/* HDMI2.0 Y420 EDID */
static void hd_edid_AnalyzeVcdb(uint8_t *, HD_EDID_EXT_VCDB_t *);

static void hd_edid_AnalyzeVsvdb(uint8_t *, HD_EDID_EXT_VSVDB_t *, uint8_t);
static void hd_edid_AnalyzeVsadb(uint8_t *, HD_EDID_EXT_VSADB_t *);

static void hd_edid_AnalyzeCmr(uint8_t *, HD_EDID_EXT_CMR_t *);
static void hd_edid_CopyVideoBufferToY420cmdb(HD_EDID_EXT_Y420CMDB_t *);
static void hd_edid_AnalyzeY420vdb(uint8_t *, HD_EDID_EXT_Y420VDB_t *, uint8_t);
static void hd_edid_AnalyzeY420cmdb(uint8_t *, HD_EDID_EXT_Y420CMDB_t *, HD_EDID_EXT_Y420VDB_t *, uint8_t);
static void hd_edid_Check420Support(HD_EDID_EXT_Y420CMDB_t *, HD_EDID_EXT_Y420VDB_t *);
/* HDMI2.0 Y420 EDID */
static void hd_edid_AnalyzeHdr(uint8_t *, HD_EDID_EXT_HDR_t *);
/* HDMI2.1 HF-SCDB EDID */
static void hd_edid_AnalyzeHfScdb(uint8_t *, HD_EDID_EXT_HFSCDB_t *);

static void hd_edid_GetVideoIdentificationCode(uint8_t *);

/*==== Code ================================================================*/
void hd_edid_GetSinkResolution(uint8_t out_idx, uint8_t *p_svd, uint8_t *p_dtd)
{
	uint8_t  i;

	if ((p_svd == 0) || (p_dtd == 0))
	{
		return;
	}

	/* DTD */
	for (i = 0; i < s_HdEdidAnalyzeData[out_idx].dtd.count; i++) {
		p_dtd[i] = s_HdEdidAnalyzeData[out_idx].dtd.timing[i].vic;
	}

	/* SVD */
	for (i = 0; i < s_HdEdidAnalyzeData[out_idx].video.count; i++) {
		p_svd[i] = s_HdEdidAnalyzeData[out_idx].video.data[i].vic;
	}

	return;
}

bool hd_edid_IsHdmi(uint8_t out_idx)
{
	return (bool)s_HdEdidAnalyzeData[out_idx].is_hdmi;
}


/*--------------------------------------------------------------------------*/
/* @brief     : EDID RAM Initialize                                         */
/*--------------------------------------------------------------------------*/
void hd_edid_RamInitialize(void)
{
	/* Get Own EDID Data */
	hd_edid_GetOwnEdid();
}

/*--------------------------------------------------------------------------*/
/* @brief     : EDID RAM Initialize Data                                    */
/*--------------------------------------------------------------------------*/
void hd_edid_RamInitializeData(void)
{
	uint16_t		i;

	for (i = 0; i < HD_EDID_TABLE_SIZE; i++) {
		s_HdEdidRam[i] = 0x00;
	}
	for (i = 0; i < HD_MAX__MODULE_PORT_TX; i++) {
		s_HdEdidTxSize[i] = 0;
	}
}

/*--------------------------------------------------------------------------*/
/* @brief     : EDID RAM Initialize Data Sink                               */
/*--------------------------------------------------------------------------*/
static void hd_edid_RamInitializeDataSink(uint8_t idx)
{
	/* param check */
	if (idx >= HD_MAX__MODULE_PORT_TX) {
		return;
	}

	s_HdEdidRamSink[idx] = 0x00;
}

/*--------------------------------------------------------------------------*/
/* @brief     : EDID RAM Initialize Analyze                                 */
/*--------------------------------------------------------------------------*/
static void hd_edid_RamInitializeAnalyze(uint8_t idx)
{
	uint8_t		*p_data;
	uint16_t	i;

	p_data = (uint8_t *)(&s_HdEdidAnalyzeData[idx]);

	for (i = 0; i < sizeof(HD_EDID_ANALYZE_DATA_t); i++) {
		p_data[i] = 0;
	}
}

/*--------------------------------------------------------------------------*/
/* @brief     : EDID RAM Initialize Video Work Buffer                       */
/*--------------------------------------------------------------------------*/
static void hd_edid_RamInitializeVideoWorkBuff(void)
{
	uint8_t		*p_data;
	uint16_t	i;

	p_data = (uint8_t *)(&s_HdEdidVideoWorkBuff);
	for (i = 0; i < sizeof(s_HdEdidVideoWorkBuff); i++) {
		p_data[i] = 0;
	}
}

/*--------------------------------------------------------------------------*/
/* @brief     : Get Own EDID                                                */
/*--------------------------------------------------------------------------*/
static void hd_edid_GetOwnEdid(void)
{
	hd_edid_GetOwnBasicDisplay(s_HdEdidDefault.basic_display);
	hd_edid_GetOwnColor(s_HdEdidDefault.color);

	s_HdEdidDefault.dtd.count = hd_edid_GetOwnDtdTiming(&s_HdEdidDefault.dtd.timing[0]);
	hd_edid_GetOwnDtdName(s_HdEdidDefault.dtd.name);
	hd_edid_GetOwnDtdRange(s_HdEdidDefault.dtd.range);
	s_HdEdidDefault.is_hdmi = HD_TRUE;
	hd_edid_GetOwnDtv(&s_HdEdidDefault.dtv.data);
	s_HdEdidDefault.video.count = hd_edid_GetOwnVideo(s_HdEdidDefault.video.data);
	s_HdEdidDefault.audio.count = hd_edid_GetOwnAudio(s_HdEdidDefault.audio.data);
	s_HdEdidDefault.speaker.valid = hd_edid_GetOwnSpd(&s_HdEdidDefault.speaker.data);
	s_HdEdidDefault.vsdb.phy_addr = 0x1000;
	s_HdEdidDefault.vsdb.valid = hd_edid_GetOwnVsdb(&s_HdEdidDefault.vsdb.ext);
	s_HdEdidDefault.hfvsdb.count = hd_edid_GetOwnScds(&s_HdEdidDefault.hfvsdb.data);
	s_HdEdidDefault.hfvsdb.valid = HD_TRUE;
	s_HdEdidDefault.ext.vcdb.valid = hd_edid_GetOwnExtVcdb(&s_HdEdidDefault.ext.vcdb.data);
	s_HdEdidDefault.ext.cmr.valid = hd_edid_GetOwnExtCmr(&s_HdEdidDefault.ext.cmr.data);
	/* HDMI2.0 Y420 EDID */
	s_HdEdidDefault.ext.y420vdb.count = hd_edid_GetOwnExtY420vdb(&s_HdEdidDefault.ext.y420vdb.vic[0]);
	s_HdEdidDefault.ext.y420vdb.valid = HD_TRUE;
	s_HdEdidDefault.ext.y420cmdb.valid = HD_FALSE;		/* don't use Y420CMDB due to not supporting 4K/60Hz/Y444 */

	s_HdEdidDefault.ext.hdr.valid = hd_edid_GetOwnExtHdr(&s_HdEdidDefault.ext.hdr.data);
	s_HdEdidDefault.ext.hdr.length = HD_EDID_LEN_HDR_LUMINANCE_NON_SUPPORT;

	/* HDMI2.1 HF-SCDB EDID */
	s_HdEdidDefault.ext.hfscdb.count = hd_edid_GetOwnExtScdb(&s_HdEdidDefault.ext.hfscdb.data);
	s_HdEdidDefault.ext.hfscdb.valid = HD_TRUE;
}

/* Analyze */
/*--------------------------------------------------------------------------*/
/* @brief     : Analyze EDID                                                */
/*--------------------------------------------------------------------------*/
void hd_edid_Analyze(HD_MODULE_PORT_TX_t tx)
{
	HD_EDID_ANALYZE_DATA_t	*p_data;
	uint8_t					ram_idx;
	uint16_t				i;
	uint8_t					block_size;

	ram_idx = tx;

	if (ram_idx >= HD_MAX__MODULE_PORT_TX) {
		return;
	}

	hd_edid_RamInitializeAnalyze(ram_idx);
	for (i = 0; i < HD_EDID_TABLE_SIZE; i++) {
		s_HdEdidRam[i] = s_HdEdidRamSink[ram_idx][i];
	}

	hd_edid_AnalyzeBlock0(ram_idx);
	block_size = hd_edid_GetTxEdidBlockSize(ram_idx);
	for (i = 1; i < block_size; i++) {
		/* SVD Sink Clear */
		hd_edid_RamInitializeVideoWorkBuff();
		hd_edid_AnalyzeBlock1(ram_idx, i);
		if (s_HdEdidAnalyzeData[ram_idx].ext.y420cmdb.valid == HD_TRUE) {
			/* 420CMDB */
			hd_edid_Check420Support(&s_HdEdidAnalyzeData[ram_idx].ext.y420cmdb, &s_HdEdidAnalyzeData[ram_idx].ext.y420vdb);
		}
	}
	p_data = &s_HdEdidAnalyzeData[ram_idx];
	hd_edid_SetVideoData(&p_data->video);
}

/*--------------------------------------------------------------------------*/
/* @brief     : Analyze Block 0                                             */
/*--------------------------------------------------------------------------*/
static void hd_edid_AnalyzeBlock0(uint8_t idx)
{
	HD_EDID_ANALYZE_DATA_t	*p_data;
	uint8_t					*p_src;
	uint8_t					i;

	/* parameter check */
	if (idx >= HD_MAX__MODULE_PORT_TX) {
		return;
	}

	p_data = &s_HdEdidAnalyzeData[idx];
	p_src  = &s_HdEdidRam[HD_EDID_POS_BLOCK_0_TOP];

	for (i = 0; i < HD_EDID_LEN_MANUFACTURE; i++) {
		p_data->manufacture_id[i] = p_src[HD_EDID_POS_MANUFACTURE + i];
	}

	for (i = 0; i < HD_EDID_LEN_PRODUCT; i++) {
		p_data->product_code[i] = p_src[HD_EDID_POS_PRODUCT + i];
	}

	p_data->week = p_src[HD_EDID_POS_WEEK];
	p_data->year = p_src[HD_EDID_POS_YEAR];

	for (i = 0; i < HD_EDID_LEN_EDID_VERSION; i++) {
		p_data->edid_version[i] = p_src[HD_EDID_POS_EDID_VERSION + i];
	}

	for (i = 0; i < HD_EDID_LEN_BASIC_DISPLAY; i++) {
		p_data->basic_display[i] = p_src[HD_EDID_POS_BASIC_DISPLAY + i];
	}

	for (i = 0; i < HD_EDID_LEN_COLOR; i++) {
		p_data->color[i] = p_src[HD_EDID_POS_COLOR + i];
	}

	for (i = 0; i < HD_EDID_LEN_ETD; i++) {
		p_data->etd[i] = p_src[HD_EDID_POS_ETD + i];
	}

	for (i = 0; i < HD_EDID_CNT_DTD; i++ ) {
		hd_edid_AnalyzeDtd(&p_src[HD_EDID_POS_DTD_1 + (HD_EDID_LEN_DTD * i)], &p_data->dtd);
	}

	p_data->is_hdmi = HD_FALSE;
}

/*--------------------------------------------------------------------------*/
/* @brief     : Analyze Block 1                                             */
/*--------------------------------------------------------------------------*/
static void hd_edid_AnalyzeBlock1(uint8_t idx, uint8_t block_num)
{
	HD_EDID_ANALYZE_DATA_t	*p_data;
	uint8_t					*p_src;
	uint8_t					r_addr, dtd_start;
	uint8_t					cea_block_data[HD_EDID_LEN_CEA_MAX + HD_EDID_LEN_CEA_HEADER];

	/* parameter check */
	if (idx >= HD_MAX__MODULE_PORT_TX) {
		return;
	}

	if (block_num == HD_EDID_BLOCK_1) {
		s_HdEdidAnalyzeData->video = s_HdEdidDefault.video;
	}

	if (!hd_edid_IsHeaderCorrectBlockN(s_HdEdidRam[block_num * HD_EDID_BLOCK_SIZE])) {
		return;
	}

	p_data = &s_HdEdidAnalyzeData[idx];
	p_src  = &s_HdEdidRam[block_num * HD_EDID_BLOCK_SIZE];

	hd_edid_AnalyzeDtv(p_src[HD_EDID_POS_DTV_MONITOR], &p_data->dtv);

	r_addr = HD_EDID_POS_START_DATA_BLOCK;
	dtd_start = p_src[HD_EDID_POS_OFFSET_DTD];
	while (r_addr < p_src[HD_EDID_POS_OFFSET_DTD]) {
		cea_block_data[0] = 0x00;

		r_addr = hd_edid_GetCeaBlockData(p_src, r_addr, dtd_start, cea_block_data);

		switch ((cea_block_data[0] & HD_EDID_MASK_CEA_TAG) >> HD_EDID_SHFT_CEA_TAG) {
		case HD_EDID_CEA_TAG_VIDEO:
			hd_edid_GetVideoIdentificationCode(cea_block_data);
			/* HDMI2.0 Y420 EDID */
			hd_edid_CopyVideoBufferToY420cmdb(&p_data->ext.y420cmdb);
		default:
			/* do nothing */
			break;
		}
	}

	r_addr = HD_EDID_POS_START_DATA_BLOCK;
	while (r_addr < p_src[HD_EDID_POS_OFFSET_DTD]) {
		cea_block_data[0] = 0x00;

		r_addr = hd_edid_GetCeaBlockData(p_src, r_addr, dtd_start, cea_block_data);

		switch ((cea_block_data[0] & HD_EDID_MASK_CEA_TAG) >> HD_EDID_SHFT_CEA_TAG) {
		case HD_EDID_CEA_TAG_VIDEO:
			hd_edid_AnalyzeVideoDataBlock(cea_block_data, &p_data->video);
			break;

		case HD_EDID_CEA_TAG_AUDIO:
			hd_edid_AnalyzeAudio(cea_block_data, &p_data->audio);
			break;

		case HD_EDID_CEA_TAG_SPEAKER:
			if (p_data->speaker.valid == HD_FALSE) {
				hd_edid_AnalyzeSpeaker(cea_block_data, &p_data->speaker);
			}
			break;

		case HD_EDID_CEA_TAG_VENDOR:
			if (p_data->vsdb.valid == HD_FALSE) {
				hd_edid_AnalyzeVsdb(cea_block_data, &p_data->vsdb);
				if (p_data->vsdb.valid != HD_FALSE) {
					p_data->is_hdmi = HD_TRUE;
				}
			}

			if ((p_data->vsdb.valid == HD_FALSE) || (p_data->hfvsdb.valid == HD_FALSE))
				hd_edid_AnalyzeHfVsdb(cea_block_data, &p_data->hfvsdb);	/* HF VSDB */
			
			break;

		case HD_EDID_CEA_TAG_EXT:
			/* HDMI2.0 Y420 EDID */
			hd_edid_AnalyzeExt(cea_block_data, &p_data->ext);
			break;

		default:
			/* do nothing */
			break;
		}
	}

	/* DTD */
	while ((r_addr < (HD_EDID_POS_CHECKSUM - HD_EDID_LEN_DTD)) && ((p_src[r_addr] != 0x00) && (p_src[r_addr + 1] != 0x00))) {
		hd_edid_AnalyzeDtd(&p_src[r_addr], &p_data->dtd);
		r_addr += HD_EDID_LEN_DTD;
	}
}

/*--------------------------------------------------------------------------*/
/* @brief     : Analyze DTD                                                 */
/*--------------------------------------------------------------------------*/
static void hd_edid_AnalyzeDtd(uint8_t *p_src, HD_EDID_DTD_t *p_dtd)
{
	HD_EDID_DTD_DATA_t	timing;
	uint16_t			desc_flag;
	uint8_t				tag, i;
	uint8_t				vic;

	desc_flag = p_src[HD_EDID_POS_DTD_DESC_FLAG];
	desc_flag = ((desc_flag << 8) & 0xff00) | p_src[HD_EDID_POS_DTD_DESC_FLAG + 1];
	if (desc_flag != HD_EDID_DTD_DESC_FLAG) {
		/* Timing Data */
		if (p_dtd->count < HD_EDID_CNT_DTD_TIMING) {
			hd_edid_GetDtdTimingData(p_src, &timing);
			if (hd_edid_IsSupportDtd(&timing, &vic) != HD_FALSE) {
				/* support */
				for (i = 0; i < HD_EDID_LEN_DTD; i++) {
					p_dtd->timing[p_dtd->count].vic     = vic;
					p_dtd->timing[p_dtd->count].data[i] = p_src[i];
				}
				p_dtd->count++;
			}
		}
	} else {
		tag = p_src[HD_EDID_POS_DTD_TAG];
		if (tag == HD_EDID_DTD_TAG_NAME) {
			if (p_dtd->name[HD_EDID_POS_DTD_TAG] == 0x00) {
				for (i = 0; i < HD_EDID_LEN_DTD; i++) {
					p_dtd->name[i] = p_src[i];
				}
			}
		} else if (tag == HD_EDID_DTD_TAG_RANGE) {
			if (p_dtd->range[HD_EDID_POS_DTD_TAG] == 0x00) {
				for (i = 0; i < HD_EDID_LEN_DTD; i++) {
					p_dtd->range[i] = p_src[i];
				}
			}
		} else {
			/* do nothing */
		}
	}
}

/*--------------------------------------------------------------------------*/
/* @brief     : Get DTD Timing Data                                         */
/*--------------------------------------------------------------------------*/
static void hd_edid_GetDtdTimingData(uint8_t *p_src, HD_EDID_DTD_DATA_t *p_data)
{
	uint16_t h_total, v_total, ratio;

	p_data->pixel_clock = (p_src[HD_EDID_POS_DTD_PIXEL_CLOCK_UP] << 8);
	p_data->pixel_clock |= p_src[HD_EDID_POS_DTD_PIXEL_CLOCK_LOW];

	p_data->h_active = ((p_src[HD_EDID_POS_DTD_H_ACTIVE_UP] & HD_EDID_MASK_DTD_H_ACTIVE_UP) >> HD_EDID_SHFT_DTD_H_ACTIVE_UP);
	p_data->h_active = ((p_data->h_active << 8) & 0x0f00);
	p_data->h_active |= p_src[HD_EDID_POS_DTD_H_ACTIVE_LOW];

	p_data->v_active = ((p_src[HD_EDID_POS_DTD_V_ACTIVE_UP] & HD_EDID_MASK_DTD_V_ACTIVE_UP) >> HD_EDID_SHFT_DTD_V_ACTIVE_UP);
	p_data->v_active = ((p_data->v_active << 8) & 0x0f00);
	p_data->v_active |= p_src[HD_EDID_POS_DTD_V_ACTIVE_LOW];

	p_data->h_blank = ((p_src[HD_EDID_POS_DTD_H_BLANK_UP] & HD_EDID_MASK_DTD_H_BLANK_UP) >> HD_EDID_SHFT_DTD_H_BLANK_UP);
	p_data->h_blank = ((p_data->h_blank << 8) & 0x0f00);
	p_data->h_blank |= p_src[HD_EDID_POS_DTD_H_BLANK_LOW];

	p_data->v_blank = ((p_src[HD_EDID_POS_DTD_V_BLANK_UP] & HD_EDID_MASK_DTD_V_BLANK_UP) >> HD_EDID_SHFT_DTD_V_BLANK_UP);
	p_data->v_blank = ((p_data->v_blank << 8) & 0x0f00);
	p_data->v_blank |= p_src[HD_EDID_POS_DTD_V_BLANK_LOW];

	h_total = ((p_src[HD_EDID_POS_DTD_H_TOTAL_UP] & HD_EDID_MASK_DTD_H_TOTAL_UP) >> HD_EDID_SHFT_DTD_H_TOTAL_UP);
	h_total = ((h_total << 8) & 0x0f00);
	h_total |= p_src[HD_EDID_POS_DTD_H_TOTAL_LOW];

	v_total = ((p_src[HD_EDID_POS_DTD_V_TOTAL_UP] & HD_EDID_MASK_DTD_V_TOTAL_UP) >> HD_EDID_SHFT_DTD_V_TOTAL_UP);
	v_total = ((v_total << 8) & 0x0f00);
	v_total |= p_src[HD_EDID_POS_DTD_V_TOTAL_LOW];

	if ((p_src[HD_EDID_POS_DTD_INTERLACE] & HD_EDID_MASK_DTD_INTERLACE) != HD_FALSE) {
		p_data->interlace = HD_TRUE;
		p_data->v_active *= 2;
	} else {
		p_data->interlace = HD_FALSE;
	}

	ratio = h_total * 1000 / v_total;
	if ((ratio > 1689) && (ratio < 1867)) {
		p_data->aspect16_9 = HD_TRUE;
	} else if ((ratio > 1267) && (ratio < 1400)) {
		p_data->aspect16_9 = HD_FALSE;
	} else {
		p_data->aspect16_9 = HD_TRUE;
	}
}

/*--------------------------------------------------------------------------*/
/* @brief     : Is Support DTD                                              */
/*--------------------------------------------------------------------------*/
static bool_t hd_edid_IsSupportDtd(HD_EDID_DTD_DATA_t *p_data, uint8_t *p_vic)
{
	uint8_t i;
	bool_t	ret_val = HD_FALSE;

	typedef struct {
		uint8_t				vic;
		HD_EDID_DTD_DATA_t	vic_info;
	} VIC_TABLE_t;

	typedef struct {
		uint16_t	h_active;
		uint16_t	v_active;
	} PIXEL_TABLE_t;

	const VIC_TABLE_t vic_table[] = {
		{HD_VIC1,	{2517,	640,	480,	160,	45,	HD_FALSE,	HD_TRUE}},
		{HD_VIC4,	{7425,	1280,	720,	370,	30,	HD_FALSE,	HD_TRUE}},
		{HD_VIC5,	{7425,	1920,	1080,	280,	45,	HD_TRUE,	HD_TRUE}},
		{HD_VIC16,	{14850,	1920,	1080,	280,	45,	HD_FALSE,	HD_TRUE}},
		{HD_VIC19,	{7425,	1280,	720,	700,	30,	HD_FALSE,	HD_TRUE}},
		{HD_VIC20,	{7425,	1920,	1080,	720,	45,	HD_TRUE,	HD_TRUE}},
		{HD_VIC31,	{14850,	1920,	1080,	720,	45,	HD_FALSE,	HD_TRUE}},
	};
	const PIXEL_TABLE_t pixel_table_480p = {720, 480};			/* VIC2 or VIC3 */
	const PIXEL_TABLE_t pixel_table_576p = {720, 576};			/* VIC17 or VIC18 */

	const VIC_TABLE_t vic_table_4k[] = {
		{HD_VIC93,	{29700,	3840,	2160,	1660,	90,	HD_FALSE,	HD_TRUE}},
		{HD_VIC94,	{29700,	3840,	2160,	1440,	90,	HD_FALSE,	HD_TRUE}},
		{HD_VIC95,	{29700,	3840,	2160,	560,	90,	HD_FALSE,	HD_TRUE}},
		{HD_VIC96,	{59400,	3840,	2160,	1440,	90,	HD_FALSE,	HD_TRUE}},
		{HD_VIC97,	{59400,	3840,	2160,	560,	90,	HD_FALSE,	HD_TRUE}},
	};

	for (i = 0; (ret_val == HD_FALSE) && (i < sizeof(vic_table_4k) / sizeof(vic_table_4k[0])); i++) {
		if ((vic_table_4k[i].vic_info.pixel_clock == p_data->pixel_clock)
		&&  (vic_table_4k[i].vic_info.h_active == p_data->h_active)
		&&  (vic_table_4k[i].vic_info.v_active == p_data->v_active)
		&&  (vic_table_4k[i].vic_info.h_blank == p_data->h_blank)
		&&  (vic_table_4k[i].vic_info.v_blank == p_data->v_blank)
		&&  (vic_table_4k[i].vic_info.interlace == p_data->interlace)
		&&  (vic_table_4k[i].vic_info.aspect16_9 == p_data->aspect16_9)) {
			*p_vic = vic_table_4k[i].vic;
			ret_val = HD_TRUE;
		}
	}

	for (i = 0; (ret_val == HD_FALSE) && (i < (sizeof(vic_table) / sizeof(vic_table[0]))); i++) {
		if ((vic_table[i].vic_info.h_active == p_data->h_active)
		&&  (vic_table[i].vic_info.v_active == p_data->v_active)
		&&  (vic_table[i].vic_info.h_blank == p_data->h_blank)
		&&  (vic_table[i].vic_info.interlace == p_data->interlace)) {
			*p_vic = vic_table[i].vic;
			ret_val = HD_TRUE;
		}
	}

	if (ret_val == HD_FALSE) {
		if ((pixel_table_480p.h_active == p_data->h_active) && (pixel_table_480p.v_active == p_data->v_active)) {
			if (p_data->aspect16_9 != HD_FALSE) {
				*p_vic = HD_VIC3;
			} else {
				*p_vic = HD_VIC2;
			}
			ret_val = HD_TRUE;
		} else if ((pixel_table_576p.h_active == p_data->h_active) && (pixel_table_576p.v_active == p_data->v_active)) {
			if (p_data->aspect16_9 != HD_FALSE) {
				*p_vic = HD_VIC18;
			} else {
				*p_vic = HD_VIC17;
			}
			ret_val = HD_TRUE;
		} else {
			/* do nothing */
		}
	}

	return ret_val;
}

/*--------------------------------------------------------------------------*/
/* @brief     : Analyze DTV                                                 */
/*--------------------------------------------------------------------------*/
static void hd_edid_AnalyzeDtv(uint8_t data, HD_EDID_DTV_t *p_dtv)
{
	p_dtv->data |= (data & (HD_EDID_FMT_DTV__UNDERSCAN | HD_EDID_FMT_DTV__AUDIO | HD_EDID_FMT_DTV__YCBCR444 | HD_EDID_FMT_DTV__YCBCR422));

}

/*--------------------------------------------------------------------------*/
/* @brief     : Get Video Identification Code                               */
/*--------------------------------------------------------------------------*/
static void hd_edid_GetVideoIdentificationCode(uint8_t *p_data)
{
	uint8_t i, desc_cnt;
	uint8_t vic, temp;

	desc_cnt = (p_data[HD_EDID_POS_CEA_HEADER] & HD_EDID_MASK_CEA_LEN) / HD_EDID_LEN_DESC_SVD;
	for (i = 0; i < desc_cnt; i++) {
		temp = p_data[HD_EDID_LEN_CEA_HEADER + (i * HD_EDID_LEN_DESC_SVD)];
		if (temp <= HD_VIC64) {
			vic = temp;
		} else if ((temp <= HD_VIC127) || (HD_VIC193 <= temp)) {
			vic = temp;
		} else {
			vic = (temp & HD_EDID_MASK_SVD_FORMAT);
		}

		if (s_HdEdidVideoWorkBuff.cea_block[0].count < HD_EDID_BUFF_SIZE_SVD) {
			s_HdEdidVideoWorkBuff.cea_block[0].data[s_HdEdidVideoWorkBuff.cea_block[0].count].vic = vic;
			s_HdEdidVideoWorkBuff.cea_block[0].count++;
		}
	}
}

/*--------------------------------------------------------------------------*/
/* @brief     : Analyze Video Data Block                                    */
/*--------------------------------------------------------------------------*/
static void hd_edid_AnalyzeVideoDataBlock(uint8_t *p_data, HD_EDID_VIDEO_t *p_video)
{
	uint8_t i, j, desc_cnt;
	uint8_t vic, temp;
	bool_t  is_native_code = HD_FALSE;

	desc_cnt = (p_data[HD_EDID_POS_CEA_HEADER] & HD_EDID_MASK_CEA_LEN) / HD_EDID_LEN_DESC_SVD;
	for (i = 0; i < desc_cnt; i++) {
		is_native_code = HD_FALSE;
		temp = p_data[HD_EDID_LEN_CEA_HEADER + (i * HD_EDID_LEN_DESC_SVD)];

		if (temp <= HD_VIC64) {
			vic = temp;
		} else if ((temp <= HD_VIC127) || (HD_VIC193 <= temp)) {
			vic = temp;
		} else {
			vic = (temp & HD_EDID_MASK_SVD_FORMAT);
			if (temp & HD_EDID_MASK_SVD_NATIVE) {
				is_native_code = HD_TRUE;
			}
		}

		for (j = 0; j < p_video->count; j++) {
			if (vic == p_video->data[j].vic) {
				p_video->data[j].state |= HD_EDID_FMT_SINK_SUPPORT;
				if (is_native_code == HD_TRUE) {
					p_video->data[j].state |= HD_EDID_FMT_SVD__NATIVE;
				}
				break;
			}
		}
	}
}

/*--------------------------------------------------------------------------*/
/* @brief     : Set Video Data                                              */
/*--------------------------------------------------------------------------*/
static void hd_edid_SetVideoData(HD_EDID_VIDEO_t *p_video)
{
	uint8_t i, j;

	for (i = 0, j = 0; i < p_video->count; i++) {
		if (p_video->data[i].state & HD_EDID_FMT_SINK_SUPPORT) {
			if (i != j) {
				p_video->data[j] = p_video->data[i];
			}
			j++;
		}
	}
	p_video->count = j;
}

/* HDMI2.0 Y420 EDID */
/*--------------------------------------------------------------------------*/
/* @brief     : Copy Video Buffer to Y420CMDB                               */
/*--------------------------------------------------------------------------*/
static void hd_edid_CopyVideoBufferToY420cmdb(HD_EDID_EXT_Y420CMDB_t *p_dest)
{
	uint8_t i;

	p_dest->count = 0;
	for (i = 0; i < HD_EDID_LEN_Y420CMDB_MASK; i++) {
		p_dest->mask[i] = 0x00;
	}
	if (s_HdEdidVideoWorkBuff.cea_block[0].count < HD_EDID_LEN_Y420CMDB_VIC) {
		for (i = 0; i < s_HdEdidVideoWorkBuff.cea_block[0].count; i++) {
			p_dest->vic[i] = s_HdEdidVideoWorkBuff.cea_block[0].data[i].vic;
		}
	}
}


/*--------------------------------------------------------------------------*/
/* @brief     : Analyze Audio                                               */
/*--------------------------------------------------------------------------*/
static void hd_edid_AnalyzeAudio(uint8_t *p_data, HD_EDID_AUDIO_t *p_audio)
{
	uint8_t					i, j, len;
	uint8_t					format, max_channel, param;
	HD_EDID_AUDIO_DATA_t	sad;
	bool_t					analyze_ng = HD_FALSE;

	len = (p_data[HD_EDID_POS_CEA_HEADER] & HD_EDID_MASK_CEA_LEN);

	for (i = 0; i < len; i += HD_EDID_LEN_DESC_SAD) {
		analyze_ng  = HD_FALSE;
		format      = ((p_data[HD_EDID_LEN_CEA_HEADER + i] & HD_EDID_MASK_SAD_FORMAT) >> HD_EDID_SHFT_SAD_FORMAT);
		max_channel = (p_data[HD_EDID_LEN_CEA_HEADER + i] & HD_EDID_MASK_SAD_CHANNEL);
		param       = (p_data[HD_EDID_LEN_CEA_HEADER + i + HD_EDID_POS_SAD_PARAM]);

		if (hd_edid_GetSupportAudio(format, param, &sad) != HD_EDID_SAD_KIND__NOTSUPPORT) {
			if (format == HD_EDID_SAD_FORMAT_DTSHD) {
				if (((param & HD_EDID_MASK_SAD_DTSX_PARAM_V1) != 0x00)
				||  ((param & HD_EDID_MASK_SAD_DTSX_PARAM_V2) != 0x00)) {
					format = HD_EDID_SAD_FORMAT_DTSUHD;
				}
			}
			for (j = 0; j < p_audio->count; j++) {
				/* extension type */
				if (format == HD_EDID_SAD_FORMAT_EXTENSION_TYPE) {
					if ((((param & HD_EDID_MASK_SAD_EXTENSION_TYPE_CODE) >> HD_EDID_SHFT_SAD_EXTENSION_TYPE_CODE) == 
					     ((p_audio->data[j].param & HD_EDID_MASK_SAD_EXTENSION_TYPE_CODE) >> HD_EDID_SHFT_SAD_EXTENSION_TYPE_CODE))
					&& (max_channel == p_audio->data[j].max_channel) && (param == p_audio->data[j].param)){
						break;
					}
				}
				if ((format == p_audio->data[j].format_code) && (max_channel == p_audio->data[j].max_channel) && (param == p_audio->data[j].param)) {
					break;
				}
			}

			if (j < p_audio->count) {
				continue;
			}

			/* Max Channel */
			if (sad.max_channel > max_channel) {
				sad.max_channel = max_channel;
			}

			/* Frequency */
			sad.freq &= (p_data[HD_EDID_LEN_CEA_HEADER + i + HD_EDID_POS_SAD_FREQ] & HD_EDID_MASK_SAD_FREQ);

			/* Parameter */
			/* param = (p_data[HD_EDID_LEN_CEA_HEADER + i + HD_EDID_POS_SAD_PARAM]); */
			switch (sad.kind) {
			case HD_EDID_SAD_KIND__LPCM:
			case HD_EDID_SAD_KIND__MAT:
			case HD_EDID_SAD_KIND__DTSHD:
			case HD_EDID_SAD_KIND__DTSX:
			case HD_EDID_SAD_KIND__DSD:
			case HD_EDID_SAD_KIND__MPEG4AAC:
				sad.param &= param;
				break;

			case HD_EDID_SAD_KIND__DDP:
				if ((sad.freq & HD_EDID_SAD_FREQ_48K) == 0) {
					analyze_ng = HD_TRUE;
				} else {
					sad.param &= param;
				}
				break;

			case HD_EDID_SAD_KIND__AC3:
			case HD_EDID_SAD_KIND__MPEG2AAC:
			case HD_EDID_SAD_KIND__DTS:
				if (sad.param > param) {
					sad.param = param;
				}
				break;
			default:
				/* do nothing */
				break;
			}

			if (analyze_ng == HD_FALSE) {
				p_audio->data[p_audio->count] = sad;
				p_audio->count++;
			}
			if (p_audio->count >= HD_EDID_BUFF_SIZE_SAD) {
				break;
			}
		}
	}
}

/*--------------------------------------------------------------------------*/
/* @brief     : Analyze Speaker                                             */
/*--------------------------------------------------------------------------*/
static void hd_edid_AnalyzeSpeaker(uint8_t *p_data, HD_EDID_SPD_t *p_speaker)
{
	uint8_t				i;
	HD_EDID_SPD_DATA_t	speaker;

	if (hd_edid_IsSupportSpd(&speaker) != HD_FALSE) {
		p_speaker->valid = HD_TRUE;
		for (i = 0; i < HD_EDID_LEN_DESC_SPD; i++) {
			speaker.param[i] &= p_data[HD_EDID_LEN_CEA_HEADER + i];
		}
		p_speaker->data = speaker;
	}
}

/*--------------------------------------------------------------------------*/
/* @brief     : Analyze VSDB                                                */
/*--------------------------------------------------------------------------*/
static void hd_edid_AnalyzeVsdb(uint8_t *p_data, HD_EDID_VSDB_t *p_vsdb)
{
	uint8_t len;

	len = (p_data[HD_EDID_POS_CEA_HEADER] & HD_EDID_MASK_CEA_LEN);
	/* fail safe */
	if (len < HD_EDID_LEN_MIN) {
		return;
	}
	if (!hd_edid_IsCorrectHdmiDisplay(&p_data[0])) {
		return;
	}

	p_vsdb->valid = HD_TRUE;

	/* Copy Physical Address */
	p_vsdb->phy_addr = p_data[HD_EDID_POS_PHY_ADDR];
	p_vsdb->phy_addr = ((p_vsdb->phy_addr << 8) & 0xff00) | p_data[HD_EDID_POS_PHY_ADDR + 1];

	if (len >= HD_EDID_POS_VSDB_EXTENSION) {
		hd_edid_AnalyzeVsdbExt(&p_data[HD_EDID_POS_VSDB_EXTENSION], (len - HD_EDID_POS_VSDB_EXTENSION + 1), &p_vsdb->ext);
	}
	p_vsdb->count = len;
}

/*--------------------------------------------------------------------------*/
/* @brief     : Analyze VSDB Ext                                            */
/*--------------------------------------------------------------------------*/
static void hd_edid_AnalyzeVsdbExt(uint8_t *p_data, uint8_t len, HD_EDID_VSDB_EXTENSION_t *p_vsdb_ext)
{
	uint8_t offset, tmp;
	uint8_t video_latency = 0, audio_latency = 0, i_video_latency = 0, i_audio_latency = 0;
	uint8_t i;

	offset = 0;

	/* Support AI, Deep Color */
	if (offset < len) {
		p_vsdb_ext->supports_ai = ((p_data[offset] & HD_EDID_MASK_VSDB_SUPPORTS_AI) >> HD_EDID_SHFT_VSDB_SUPPORTS_AI);
		p_vsdb_ext->deep_color  = ((p_data[offset] & HD_EDID_MASK_VSDB_DEEP_COLOR) >> HD_EDID_SHFT_VSDB_DEEP_COLOR);
		p_vsdb_ext->deep_color  &= s_HdEdidDefault.vsdb.ext.deep_color;
		offset++;
	}

	/* Max TMDS Clock */
	if (offset < len) {
		p_vsdb_ext->tmds_clock = p_data[offset];
		offset++;
	}

	/* Latency filed present, HDMI Video present, CNC */
	if (offset < len) {
		tmp = p_data[offset++];

		/* CNC */
		p_vsdb_ext->cnc = tmp & HD_EDID_MASK_VSDB_CNC;

		/* Latency */
		if ((tmp & HD_EDID_MASK_VSDB_LATENCY_PRESENT) != HD_FALSE) {
			if (len >= offset + HD_EDID_LEN_VSDB_LATENCY) {
				i_video_latency = video_latency = p_data[offset++];
				i_audio_latency = audio_latency = p_data[offset++];
			} else {
				offset = len;
			}
		}
		if ((tmp & HD_EDID_MASK_VSDB_I_LATENCY_PRESENT) != HD_FALSE) {
			if ((tmp & HD_EDID_MASK_VSDB_LATENCY_PRESENT) == HD_FALSE) {
				offset = len;
			} else {
				if (len >= (offset + HD_EDID_LEN_VSDB_I_LATENCY)) {
					i_video_latency = p_data[offset++];
					i_audio_latency = p_data[offset++];
				} else {
					offset = len;
				}
			}
		}

		for (i = 0; i < HD_MAX__MODULE_PORT_TX; i++) {
			if (p_vsdb_ext == &s_HdEdidAnalyzeData[i].vsdb.ext) {
				//hd_edid_SetLatency(i, video_latency, audio_latency, i_video_latency, i_audio_latency);
				break;
			}
		}

		if ((tmp & HD_EDID_MASK_VSDB_HDMI_VIDEO_PRESENT) != HD_FALSE) {
			hd_edid_AnalyzeVsdbExtVideo(&p_data[offset], len - offset, p_vsdb_ext);
		}
	}
}

/*--------------------------------------------------------------------------*/
/* @brief     : Analyze VSDB Ext Video                                      */
/*--------------------------------------------------------------------------*/
static void hd_edid_AnalyzeVsdbExtVideo(uint8_t *p_data, uint8_t len, HD_EDID_VSDB_EXTENSION_t *p_vsdb_ext)
{
	uint8_t offset, len_data, tmp;
	uint8_t i, count;

	offset    = 0;
	len_data  = 0;
	count     = 0;

	/* 3D Present, Image Size */
	if (offset < len) {
		if ((p_data[offset] & HD_EDID_MASK_VSDB_3D_PRESENT) != HD_FALSE) {
			p_vsdb_ext->_3d_present = 1;
		}
		p_vsdb_ext->image_size = ((p_data[offset] & HD_EDID_MASK_VSDB_IMAGE_SIZE) >> HD_EDID_SHFT_VSDB_IMAGE_SIZE);
		offset++;
	}

	/* HDMI Vic Len */
	if (offset < len) {
		tmp = p_data[offset++];

		/* HDMI VIC */
		len_data = ((tmp & HD_EDID_MASK_VSDB_HDMI_VIC_LEN) >> HD_EDID_SHFT_VSDB_HDMI_VIC_LEN);
		if ((offset + len_data) <= len) {
			p_vsdb_ext->hdmi_vic_len = len_data;
			for (i = 0; i < p_vsdb_ext->hdmi_vic_len; i++) {
				if (hd_edid_IsSupportVsdbHdmiVic(p_data[offset + i])) {
					p_vsdb_ext->hdmi_vic[count] = p_data[offset + i];
					count++;
				}
			}
			p_vsdb_ext->hdmi_vic_len = count;
		}
	}
}

/*--------------------------------------------------------------------------*/
/* @brief     : Analyze HF-VSDB                                             */
/*--------------------------------------------------------------------------*/
static void hd_edid_AnalyzeHfVsdb(uint8_t *p_data, HD_EDID_HFVSDB_t *p_hfvsdb)
{
	uint8_t len;

	len = (p_data[HD_EDID_POS_CEA_HEADER] & HD_EDID_MASK_CEA_LEN);
	/* fail safe */
	if (len < HD_EDID_LEN_MIN) {
		return;
	}
	if (hd_edid_IsCorrectScdsIeee(&p_data[0]) == HD_FALSE) {
		return;
	}
	if (p_data[HD_EDID_POS_SCDS_TMDS_CLOCK] <= s_HdEdidDefault.vsdb.ext.tmds_clock) {
		return;
	}
	p_hfvsdb->valid = HD_TRUE;

	hd_edid_AnalyzeScds(p_data, &p_hfvsdb->data, len);
	if (len > HD_EDID_LEN_DESC_SCDS_MAX) {
		len = HD_EDID_LEN_DESC_SCDS_MAX;
	}
	p_hfvsdb->count = len;
}

/*--------------------------------------------------------------------------*/
/* @brief     : Analyze SCDS                                                */
/*--------------------------------------------------------------------------*/
static void hd_edid_AnalyzeScds(uint8_t *p_data, HD_EDID_SCDS_EXTENSION_t *p_scds, uint8_t len)
{
	p_scds->version = p_data[HD_EDID_POS_SCDS_VERSION];

	p_scds->_3d_osd_disparity = (p_data[HD_EDID_POS_SCDS_3D_OSD_DISPARITY] & HD_EDID_MASK_SCDS_3D_OSD_DISPARITY);
	p_scds->_3d_osd_disparity &= s_HdEdidDefault.hfvsdb.data._3d_osd_disparity;

	p_scds->dual_view = ((p_data[HD_EDID_POS_SCDS_DUAL_VIEW] & HD_EDID_MASK_SCDS_DUAL_VIEW) >> HD_EDID_SHFT_SCDS_DUAL_VIEW);
	p_scds->dual_view &= s_HdEdidDefault.hfvsdb.data.dual_view;

	p_scds->independent_view = ((p_data[HD_EDID_POS_SCDS_INDEPENDENT_VIEW] & HD_EDID_MASK_SCDS_INDEPENDENT_VIEW) >> HD_EDID_SHFT_SCDS_INDEPENDENT_VIEW);
	p_scds->independent_view &= s_HdEdidDefault.hfvsdb.data.independent_view;

	p_scds->lte_scramble = ((p_data[HD_EDID_POS_SCDS_LTE_SCRAMBLE] & HD_EDID_MASK_SCDS_LTE_SCRAMBLE) >> HD_EDID_SHFT_SCDS_LTE_SCRAMBLE);
	p_scds->lte_scramble &= s_HdEdidDefault.hfvsdb.data.lte_scramble;

	p_scds->rr_capable = ((p_data[HD_EDID_POS_SCDS_RR_CAPABLE] & HD_EDID_MASK_SCDS_RR_CAPABLE) >> HD_EDID_SHFT_SCDS_RR_CAPABLE);
	p_scds->rr_capable &= s_HdEdidDefault.hfvsdb.data.rr_capable;

	p_scds->scdc_present = s_HdEdidDefault.hfvsdb.data.scdc_present;

	p_scds->tmds_character_rate = p_data[HD_EDID_POS_SCDS_TMDS_CLOCK];

	p_scds->deep_color_420 = p_data[HD_EDID_POS_SCDS_DEEP_COLOR_420] & HD_EDID_MASK_SCDS_DEEP_COLOR;
	p_scds->deep_color_420 &= s_HdEdidDefault.hfvsdb.data.deep_color_420;

	if (((p_data[HD_EDID_POS_SCDS_MAX_FRL_RATE] & HD_EDID_MASK_SCDS_MAX_FRL_RATE) >> HD_EDID_SHFT_SCDS_MAX_FRL_RATE) > s_HdEdidDefault.hfvsdb.data.max_frl_rate) {
		p_scds->max_frl_rate = s_HdEdidDefault.hfvsdb.data.max_frl_rate;
	} else {
		p_scds->max_frl_rate = ((p_data[HD_EDID_POS_SCDS_MAX_FRL_RATE] & HD_EDID_MASK_SCDS_MAX_FRL_RATE) >> HD_EDID_SHFT_SCDS_MAX_FRL_RATE);
	}

	p_scds->uhd_vic = ((p_data[HD_EDID_POS_SCDS_UHD_VIC] & HD_EDID_MASK_SCDS_UHD_VIC) >> HD_EDID_SHFT_SCDS_UHD_VIC);	/* for PF */

	p_scds->m_delta = ((p_data[HD_EDID_POS_SCDS_M_DELTA] & HD_EDID_MASK_SCDS_M_DELTA) >> HD_EDID_SHFT_SCDS_M_DELTA);

	p_scds->cinema_vrr = ((p_data[HD_EDID_POS_SCDS_CINEMA_VRR] & HD_EDID_MASK_SCDS_CINEMA_VRR) >> HD_EDID_SHFT_SCDS_CINEMA_VRR);

	p_scds->cnm_vrr = ((p_data[HD_EDID_POS_SCDS_CNM_VRR] & HD_EDID_MASK_SCDS_CNM_VRR) >> HD_EDID_SHFT_SCDS_CNM_VRR);

	p_scds->fva = ((p_data[HD_EDID_POS_SCDS_FVA] & HD_EDID_MASK_SCDS_FVA) >> HD_EDID_SHFT_SCDS_FVA);

	p_scds->allm = ((p_data[HD_EDID_POS_SCDS_ALLM] & HD_EDID_MASK_SCDS_ALLM) >> HD_EDID_SHFT_SCDS_ALLM);

	p_scds->fapa_start_location = ((p_data[HD_EDID_POS_SCDS_FAPA_START_LOCATION] & HD_EDID_MASK_SCDS_FAPA_START_LOCATION) >> HD_EDID_SHFT_SCDS_FAPA_START_LOCATION);

	/* HDMI2.1 */
	p_scds->cable_status = ((p_data[HD_EDID_POS_SCDS_CABLE_STATUS] & HD_EDID_MASK_SCDS_CABLE_STATUS) >> HD_EDID_SHFT_SCDS_CABLE_STATUS);

	p_scds->ccbpci = ((p_data[HD_EDID_POS_SCDS_CCBPCI] & HD_EDID_MASK_SCDS_CCBPCI) >> HD_EDID_SHFT_SCDS_CCBPCI);

	p_scds->uhd_vic = ((p_data[HD_EDID_POS_SCDS_UHD_VIC] & HD_EDID_MASK_SCDS_UHD_VIC) >> HD_EDID_SHFT_SCDS_UHD_VIC);

	if (len >= HD_EDID_POS_SCDS_PB_7) {
		p_scds->m_delta = ((p_data[HD_EDID_POS_SCDS_M_DELTA] & HD_EDID_MASK_SCDS_M_DELTA) >> HD_EDID_SHFT_SCDS_M_DELTA);

		p_scds->cinema_vrr = ((p_data[HD_EDID_POS_SCDS_CINEMA_VRR] & HD_EDID_MASK_SCDS_CINEMA_VRR) >> HD_EDID_SHFT_SCDS_CINEMA_VRR);

		p_scds->cnm_vrr = ((p_data[HD_EDID_POS_SCDS_CNM_VRR] & HD_EDID_MASK_SCDS_CNM_VRR) >> HD_EDID_SHFT_SCDS_CNM_VRR);

		p_scds->fva = ((p_data[HD_EDID_POS_SCDS_FVA] & HD_EDID_MASK_SCDS_FVA) >> HD_EDID_SHFT_SCDS_FVA);

		p_scds->allm = ((p_data[HD_EDID_POS_SCDS_ALLM] & HD_EDID_MASK_SCDS_ALLM) >> HD_EDID_SHFT_SCDS_ALLM);

		p_scds->fapa_start_location = ((p_data[HD_EDID_POS_SCDS_FAPA_START_LOCATION] & HD_EDID_MASK_SCDS_FAPA_START_LOCATION) >> HD_EDID_SHFT_SCDS_FAPA_START_LOCATION);

		p_scds->vrr_max_9_8 = ((p_data[HD_EDID_POS_SCDS_VRR_MAX_9_8] & HD_EDID_MASK_SCDS_VRR_MAX_9_8) >> HD_EDID_SHFT_SCDS_VRR_MAX_9_8);

		p_scds->vrr_min = ((p_data[HD_EDID_POS_SCDS_VRR_MIN] & HD_EDID_MASK_SCDS_VRR_MIN) >> HD_EDID_SHFT_SCDS_VRR_MIN);

		p_scds->vrr_max_7_0 = p_data[HD_EDID_POS_SCDS_VRR_MAX_7_0];
	}
	if (len >= HD_EDID_POS_SCDS_PB_8) {
		p_scds->dsc_1p2 = ((p_data[HD_EDID_POS_SCDS_DSC_1p2] & HD_EDID_MASK_SCDS_DSC_1p2) >> HD_EDID_SHFT_SCDS_DSC_1p2);

		p_scds->dsc_native_420 = ((p_data[HD_EDID_POS_SCDS_DSC_NATIVE_420] & HD_EDID_MASK_SCDS_DSC_NATIVE_420) >> HD_EDID_SHFT_SCDS_DSC_NATIVE_420);

		p_scds->dsc_all_bpp = ((p_data[HD_EDID_POS_SCDS_DSC_ALL_BPP] & HD_EDID_MASK_SCDS_DSC_ALL_BPP) >> HD_EDID_SHFT_SCDS_DSC_ALL_BPP);

		p_scds->dsc_16bpc = ((p_data[HD_EDID_POS_SCDS_DSC_16bpc] & HD_EDID_MASK_SCDS_DSC_16bpc) >> HD_EDID_SHFT_SCDS_DSC_16bpc);

		p_scds->dsc_12bpc = ((p_data[HD_EDID_POS_SCDS_DSC_12bpc] & HD_EDID_MASK_SCDS_DSC_12bpc) >> HD_EDID_SHFT_SCDS_DSC_12bpc);

		p_scds->dsc_10bpc = ((p_data[HD_EDID_POS_SCDS_DSC_10bpc] & HD_EDID_MASK_SCDS_DSC_10bpc) >> HD_EDID_SHFT_SCDS_DSC_10bpc);
	}
	if (len >= HD_EDID_POS_SCDS_PB_9) {
		p_scds->dsc_max_frl_rate = ((p_data[HD_EDID_POS_SCDS_DSC_MAX_FRL_RATE] & HD_EDID_MASK_SCDS_DSC_MAX_FRL_RATE) >> HD_EDID_SHFT_SCDS_DSC_MAX_FRL_RATE);

		p_scds->dsc_max_slices = ((p_data[HD_EDID_POS_SCDS_DSC_MAX_SLICES] & HD_EDID_MASK_SCDS_DSC_MAX_SLICES) >> HD_EDID_SHFT_SCDS_DSC_MAX_SLICES);
	}
	if (len >= HD_EDID_POS_SCDS_PB_10) {
		p_scds->dsc_total_chunkKbytes = ((p_data[HD_EDID_POS_SCDS_DSC_TOTAL_CHUNK_KBYTES] & HD_EDID_MASK_SCDS_DSC_TOTAL_CHUNK_KBYTES) >> HD_EDID_SHFT_SCDS_DSC_TOTAL_CHUNK_KBYTES);
	}
}

/* HDMI2.0 Y420 EDID */
/*--------------------------------------------------------------------------*/
/* @brief     : Analyze Ext                                                 */
/*--------------------------------------------------------------------------*/
static void hd_edid_AnalyzeExt(uint8_t *p_data, HD_EDID_EXT_t *p_ext)
{
	uint8_t len, ext_tag;

	len = p_data[HD_EDID_POS_CEA_HEADER] & HD_EDID_MASK_CEA_LEN;
	ext_tag = p_data[HD_EDID_POS_EXT_TAG];
	switch (ext_tag) {
	case HD_EDID_EXT_TAG_CODE_VCDB:
		if ((len >= HD_EDID_LEN_DESC_VCDB) && (p_ext->vcdb.valid == HD_FALSE)) {
			hd_edid_AnalyzeVcdb(&p_data[HD_EDID_POS_EXT_TAG], &p_ext->vcdb);
		}
		break;

	case HD_EDID_EXT_TAG_CODE_VSVDB:
		if (p_ext->vsvdb.valid == HD_FALSE) {
			hd_edid_AnalyzeVsvdb(&p_data[HD_EDID_POS_EXT_TAG], &p_ext->vsvdb, len);
		}
		break;
	case HD_EDID_EXT_TAG_CODE_VSADB:
		if (p_ext->vsadb.valid == HD_FALSE) {
			hd_edid_AnalyzeVsadb(&p_data[HD_EDID_POS_EXT_TAG], &p_ext->vsadb);
		}
		break;

	case HD_EDID_EXT_TAG_CODE_CMR:
		if ((len >= HD_EDID_LEN_DESC_CMR) && (p_ext->cmr.valid == HD_FALSE)) {
			hd_edid_AnalyzeCmr(&p_data[HD_EDID_POS_EXT_TAG], &p_ext->cmr);
		}
		break;

	/* HDMI2.0 Y420 EDID */
	case HD_EDID_EXT_TAG_CODE_Y420VDB:
		hd_edid_AnalyzeY420vdb(&p_data[HD_EDID_POS_EXT_TAG], &p_ext->y420vdb, len);
		break;
	case HD_EDID_EXT_TAG_CODE_Y420CMDB:
		hd_edid_AnalyzeY420cmdb(&p_data[HD_EDID_POS_EXT_TAG], &p_ext->y420cmdb, &p_ext->y420vdb, len);
		break;

	case HD_EDID_EXT_TAG_CODE_HDR:
		if (p_ext->hdr.valid == HD_FALSE) {
			p_ext->hdr.length = len;
			hd_edid_AnalyzeHdr(&p_data[HD_EDID_POS_EXT_TAG], &p_ext->hdr);
		}
		break;
	/* HDMI2.1 HF-SCDB EDID */
	case HD_EDID_EXT_TAG_CODE_HFSCDB:
		if ((p_ext->hfscdb.valid == HD_FALSE) && (s_HdEdidAnalyzeData[0].hfvsdb.valid == HD_FALSE)) {
			hd_edid_AnalyzeHfScdb(p_data, &p_ext->hfscdb);
		}
		break;
	default:
		/* do nothing */
		break;
	}
}

/*--------------------------------------------------------------------------*/
/* @brief     : Analyze VCDB                                                */
/*--------------------------------------------------------------------------*/
static void hd_edid_AnalyzeVcdb(uint8_t *p_data, HD_EDID_EXT_VCDB_t *p_vcdb)
{
	p_vcdb->valid = HD_TRUE;

	p_vcdb->data.qy = ((p_data[HD_EDID_POS_VCDB_QY] & HD_EDID_MASK_VCDB_QY) >> HD_EDID_SHFT_VCDB_QY);
	p_vcdb->data.qy &= s_HdEdidDefault.ext.vcdb.data.qy;

	p_vcdb->data.qs = ((p_data[HD_EDID_POS_VCDB_QS] & HD_EDID_MASK_VCDB_QS) >> HD_EDID_SHFT_VCDB_QS);
	p_vcdb->data.qs &= s_HdEdidDefault.ext.vcdb.data.qs;

	p_vcdb->data.s_pt = ((p_data[HD_EDID_POS_VCDB_SPT] & HD_EDID_MASK_VCDB_SPT) >> HD_EDID_SHFT_VCDB_SPT);
	p_vcdb->data.s_pt &= s_HdEdidDefault.ext.vcdb.data.s_pt;

	p_vcdb->data.s_it = ((p_data[HD_EDID_POS_VCDB_SIT] & HD_EDID_MASK_VCDB_SIT) >> HD_EDID_SHFT_VCDB_SIT);
	p_vcdb->data.s_it &= s_HdEdidDefault.ext.vcdb.data.s_it;

	p_vcdb->data.s_ce = (p_data[HD_EDID_POS_VCDB_SCE] & HD_EDID_MASK_VCDB_SCE);
	p_vcdb->data.s_ce &= s_HdEdidDefault.ext.vcdb.data.s_ce;
}

/*--------------------------------------------------------------------------*/
/* @brief     : Analyze VSVDB                                               */
/*--------------------------------------------------------------------------*/
static void hd_edid_AnalyzeVsvdb(uint8_t *p_data, HD_EDID_EXT_VSVDB_t *p_vsvdb, uint8_t length)
{
	uint8_t version;

	/* parameter check */
	if (length < HD_EDID_LEN_VSVDB_OUI) {
		return;
	}

	if ((p_data[HD_EDID_POS_VSVDB_OUI0] != HD_EDID_VAL_VSVDB_OUI_DOLBY_0)
	||  (p_data[HD_EDID_POS_VSVDB_OUI1] != HD_EDID_VAL_VSVDB_OUI_DOLBY_1)
	||  (p_data[HD_EDID_POS_VSVDB_OUI2] != HD_EDID_VAL_VSVDB_OUI_DOLBY_2)) {
		/* if not Dolby */
		return;
	}

	if (length < HD_EDID_LEN_VSVDB_DV_VERSION) {
		/* if no version info */
		return;
	}

	version = ((p_data[HD_EDID_POS_VSVDB_DV_VERSION] & HD_EDID_MASK_VSVDB_DV_VERSION) >> HD_EDID_SHFT_VSVDB_DV_VERSION);
	p_vsvdb->valid = HD_FALSE;

	switch (version) {
	case HD_EDID_VAL_VSVDB_DV_VERSION_0:
		if (length == HD_EDID_LEN_VSVDB_DV_V0) {
			p_vsvdb->valid = HD_TRUE;
			p_vsvdb->type = HD_EDID_VSVDB_DV_TYPE_0;

			p_vsvdb->data.vsvdb_dv_v0.version					= version;
			p_vsvdb->data.vsvdb_dv_v0.supports_global_dimming	= ((p_data[HD_EDID_POS_VSVDB_DV_V0_GLOBAL_DIMMING] & HD_EDID_MASK_VSVDB_DV_V0_GLOBAL_DIMMING) >> HD_EDID_SHFT_VSVDB_DV_V0_GLOBAL_DIMMING);
			p_vsvdb->data.vsvdb_dv_v0.supports_2160p60hz		= ((p_data[HD_EDID_POS_VSVDB_DV_V0_2160P60HZ] & HD_EDID_MASK_VSVDB_DV_V0_2160P60HZ) >> HD_EDID_SHFT_VSVDB_DV_V0_2160P60HZ);
			p_vsvdb->data.vsvdb_dv_v0.supports_yuv422_12bit		= ((p_data[HD_EDID_POS_VSVDB_DV_V0_YUV422_12BIT] & HD_EDID_MASK_VSVDB_DV_V0_YUV422_12BIT) >> HD_EDID_SHFT_VSVDB_DV_V0_YUV422_12BIT);
			p_vsvdb->data.vsvdb_dv_v0.rx3_0						= ((p_data[HD_EDID_POS_VSVDB_DV_V0_RX3_0] & HD_EDID_MASK_VSVDB_DV_V0_RX3_0) >> HD_EDID_SHFT_VSVDB_DV_V0_RX3_0);
			p_vsvdb->data.vsvdb_dv_v0.ry3_0						= ((p_data[HD_EDID_POS_VSVDB_DV_V0_RY3_0] & HD_EDID_MASK_VSVDB_DV_V0_RY3_0) >> HD_EDID_SHFT_VSVDB_DV_V0_RY3_0);
			p_vsvdb->data.vsvdb_dv_v0.rx11_4					= p_data[HD_EDID_POS_VSVDB_DV_V0_RX11_4];
			p_vsvdb->data.vsvdb_dv_v0.ry11_4					= p_data[HD_EDID_POS_VSVDB_DV_V0_RY11_4];
			p_vsvdb->data.vsvdb_dv_v0.gx3_0						= ((p_data[HD_EDID_POS_VSVDB_DV_V0_GX3_0] & HD_EDID_MASK_VSVDB_DV_V0_GX3_0) >> HD_EDID_SHFT_VSVDB_DV_V0_GX3_0);
			p_vsvdb->data.vsvdb_dv_v0.gy3_0						= ((p_data[HD_EDID_POS_VSVDB_DV_V0_GY3_0] & HD_EDID_MASK_VSVDB_DV_V0_GY3_0) >> HD_EDID_SHFT_VSVDB_DV_V0_GY3_0);
			p_vsvdb->data.vsvdb_dv_v0.gx11_4					= p_data[HD_EDID_POS_VSVDB_DV_V0_GX11_4];
			p_vsvdb->data.vsvdb_dv_v0.gy11_4					= p_data[HD_EDID_POS_VSVDB_DV_V0_GY11_4];
			p_vsvdb->data.vsvdb_dv_v0.bx3_0						= ((p_data[HD_EDID_POS_VSVDB_DV_V0_BX3_0] & HD_EDID_MASK_VSVDB_DV_V0_BX3_0) >> HD_EDID_SHFT_VSVDB_DV_V0_BX3_0);
			p_vsvdb->data.vsvdb_dv_v0.by3_0						= ((p_data[HD_EDID_POS_VSVDB_DV_V0_BY3_0] & HD_EDID_MASK_VSVDB_DV_V0_BY3_0) >> HD_EDID_SHFT_VSVDB_DV_V0_BY3_0);
			p_vsvdb->data.vsvdb_dv_v0.bx11_4					= p_data[HD_EDID_POS_VSVDB_DV_V0_BX11_4];
			p_vsvdb->data.vsvdb_dv_v0.by11_4					= p_data[HD_EDID_POS_VSVDB_DV_V0_BY11_4];
			p_vsvdb->data.vsvdb_dv_v0.wx3_0						= ((p_data[HD_EDID_POS_VSVDB_DV_V0_WX3_0] & HD_EDID_MASK_VSVDB_DV_V0_WX3_0) >> HD_EDID_SHFT_VSVDB_DV_V0_WX3_0);
			p_vsvdb->data.vsvdb_dv_v0.wy3_0						= ((p_data[HD_EDID_POS_VSVDB_DV_V0_WY3_0] & HD_EDID_MASK_VSVDB_DV_V0_WY3_0) >> HD_EDID_SHFT_VSVDB_DV_V0_WY3_0);
			p_vsvdb->data.vsvdb_dv_v0.wx11_4					= p_data[HD_EDID_POS_VSVDB_DV_V0_WX11_4];
			p_vsvdb->data.vsvdb_dv_v0.wy11_4					= p_data[HD_EDID_POS_VSVDB_DV_V0_WY11_4];
			p_vsvdb->data.vsvdb_dv_v0.target_min_pq3_0			= ((p_data[HD_EDID_POS_VSVDB_DV_V0_TGT_MIN_PQ3_0] & HD_EDID_MASK_VSVDB_DV_V0_TGT_MIN_PQ3_0) >> HD_EDID_SHFT_VSVDB_DV_V0_TGT_MIN_PQ3_0);
			p_vsvdb->data.vsvdb_dv_v0.target_max_pq3_0			= ((p_data[HD_EDID_POS_VSVDB_DV_V0_TGT_MAX_PQ3_0] & HD_EDID_MASK_VSVDB_DV_V0_TGT_MAX_PQ3_0) >> HD_EDID_SHFT_VSVDB_DV_V0_TGT_MAX_PQ3_0);
			p_vsvdb->data.vsvdb_dv_v0.target_min_pq11_4			= p_data[HD_EDID_POS_VSVDB_DV_V0_TGT_MIN_PQ11_4];
			p_vsvdb->data.vsvdb_dv_v0.target_max_pq11_4			= p_data[HD_EDID_POS_VSVDB_DV_V0_TGT_MAX_PQ11_4];
			p_vsvdb->data.vsvdb_dv_v0.dm_major_version			= ((p_data[HD_EDID_POS_VSVDB_DV_V0_DM_MAJOR_VER] & HD_EDID_MASK_VSVDB_DV_V0_DM_MAJOR_VER) >> HD_EDID_SHFT_VSVDB_DV_V0_DM_MAJOR_VER);
			p_vsvdb->data.vsvdb_dv_v0.dm_minor_version			= ((p_data[HD_EDID_POS_VSVDB_DV_V0_DM_MINOR_VER] & HD_EDID_MASK_VSVDB_DV_V0_DM_MINOR_VER) >> HD_EDID_SHFT_VSVDB_DV_V0_DM_MINOR_VER);
		}
		break;
	case HD_EDID_VAL_VSVDB_DV_VERSION_1:
		if (length == HD_EDID_LEN_VSVDB_DV_V1_15) {
			p_vsvdb->valid = HD_TRUE;
			p_vsvdb->type = HD_EDID_VSVDB_DV_TYPE_1_15;

			p_vsvdb->data.vsvdb_dv_15v1.version					= version;
			p_vsvdb->data.vsvdb_dv_15v1.dm_version				= ((p_data[HD_EDID_POS_VSVDB_DV_15V1_DM_VERSION] & HD_EDID_MASK_VSVDB_DV_15V1_DM_VERSION) >> HD_EDID_SHFT_VSVDB_DV_15V1_DM_VERSION);
			p_vsvdb->data.vsvdb_dv_15v1.supports_2160p60hz		= ((p_data[HD_EDID_POS_VSVDB_DV_15V1_2160P60HZ] & HD_EDID_MASK_VSVDB_DV_15V1_2160P60HZ) >> HD_EDID_SHFT_VSVDB_DV_15V1_2160P60HZ);
			p_vsvdb->data.vsvdb_dv_15v1.supports_yuv422_12bit	= ((p_data[HD_EDID_POS_VSVDB_DV_15V1_YUV422_12BIT] & HD_EDID_MASK_VSVDB_DV_15V1_YUV422_12BIT) >> HD_EDID_SHFT_VSVDB_DV_15V1_YUV422_12BIT);
			p_vsvdb->data.vsvdb_dv_15v1.target_max_luminance	= ((p_data[HD_EDID_POS_VSVDB_DV_15V1_TGT_MAX_LUM] & HD_EDID_MASK_VSVDB_DV_15V1_TGT_MAX_LUM) >> HD_EDID_SHFT_VSVDB_DV_15V1_TGT_MAX_LUM);
			p_vsvdb->data.vsvdb_dv_15v1.supports_global_dimming	= ((p_data[HD_EDID_POS_VSVDB_DV_15V1_GLOBAL_DIMMING] & HD_EDID_MASK_VSVDB_DV_15V1_GLOBAL_DIMMING) >> HD_EDID_SHFT_VSVDB_DV_15V1_GLOBAL_DIMMING);
			p_vsvdb->data.vsvdb_dv_15v1.target_min_luminance	= ((p_data[HD_EDID_POS_VSVDB_DV_15V1_TGT_MIN_LUM] & HD_EDID_MASK_VSVDB_DV_15V1_TGT_MIN_LUM) >> HD_EDID_SHFT_VSVDB_DV_15V1_TGT_MIN_LUM);
			p_vsvdb->data.vsvdb_dv_15v1.colorimetry				= ((p_data[HD_EDID_POS_VSVDB_DV_15V1_COLORIMETRY] & HD_EDID_MASK_VSVDB_DV_15V1_COLORIMETRY) >> HD_EDID_SHFT_VSVDB_DV_15V1_COLORIMETRY);
			p_vsvdb->data.vsvdb_dv_15v1.rx						= p_data[HD_EDID_POS_VSVDB_DV_15V1_RX];
			p_vsvdb->data.vsvdb_dv_15v1.ry						= p_data[HD_EDID_POS_VSVDB_DV_15V1_RY];
			p_vsvdb->data.vsvdb_dv_15v1.gx						= p_data[HD_EDID_POS_VSVDB_DV_15V1_GX];
			p_vsvdb->data.vsvdb_dv_15v1.gy						= p_data[HD_EDID_POS_VSVDB_DV_15V1_GY];
			p_vsvdb->data.vsvdb_dv_15v1.bx						= p_data[HD_EDID_POS_VSVDB_DV_15V1_BX];
			p_vsvdb->data.vsvdb_dv_15v1.by						= p_data[HD_EDID_POS_VSVDB_DV_15V1_BY];
		} else if (length == HD_EDID_LEN_VSVDB_DV_V1_12) {
			p_vsvdb->valid = HD_TRUE;
			p_vsvdb->type = HD_EDID_VSVDB_DV_TYPE_1_12;

			p_vsvdb->data.vsvdb_dv_12v1.version					= version;
			p_vsvdb->data.vsvdb_dv_12v1.dm_version				= ((p_data[HD_EDID_POS_VSVDB_DV_12V1_DM_VERSION] & HD_EDID_MASK_VSVDB_DV_12V1_DM_VERSION) >> HD_EDID_SHFT_VSVDB_DV_12V1_DM_VERSION);
			p_vsvdb->data.vsvdb_dv_12v1.supports_2160p60hz		= ((p_data[HD_EDID_POS_VSVDB_DV_12V1_2160P60HZ] & HD_EDID_MASK_VSVDB_DV_12V1_2160P60HZ) >> HD_EDID_SHFT_VSVDB_DV_12V1_2160P60HZ);
			p_vsvdb->data.vsvdb_dv_12v1.supports_yuv422_12bit	= ((p_data[HD_EDID_POS_VSVDB_DV_12V1_YUV422_12BIT] & HD_EDID_MASK_VSVDB_DV_12V1_YUV422_12BIT) >> HD_EDID_SHFT_VSVDB_DV_12V1_YUV422_12BIT);
			p_vsvdb->data.vsvdb_dv_12v1.target_max_luminance	= ((p_data[HD_EDID_POS_VSVDB_DV_12V1_TGT_MAX_LUM] & HD_EDID_MASK_VSVDB_DV_12V1_TGT_MAX_LUM) >> HD_EDID_SHFT_VSVDB_DV_12V1_TGT_MAX_LUM);
			p_vsvdb->data.vsvdb_dv_12v1.supports_global_dimming	= ((p_data[HD_EDID_POS_VSVDB_DV_12V1_GLOBAL_DIMMING] & HD_EDID_MASK_VSVDB_DV_12V1_GLOBAL_DIMMING) >> HD_EDID_SHFT_VSVDB_DV_12V1_GLOBAL_DIMMING);
			p_vsvdb->data.vsvdb_dv_12v1.target_min_luminance	= ((p_data[HD_EDID_POS_VSVDB_DV_12V1_TGT_MIN_LUM] & HD_EDID_MASK_VSVDB_DV_12V1_TGT_MIN_LUM) >> HD_EDID_SHFT_VSVDB_DV_12V1_TGT_MIN_LUM);
			p_vsvdb->data.vsvdb_dv_12v1.colorimetry				= ((p_data[HD_EDID_POS_VSVDB_DV_12V1_COLORIMETRY] & HD_EDID_MASK_VSVDB_DV_12V1_COLORIMETRY) >> HD_EDID_SHFT_VSVDB_DV_12V1_COLORIMETRY);
			p_vsvdb->data.vsvdb_dv_12v1.unique_bx2_0			= ((p_data[HD_EDID_POS_VSVDB_DV_12V1_UNIQUE_BX2_0] & HD_EDID_MASK_VSVDB_DV_12V1_UNIQUE_BX2_0) >> HD_EDID_SHFT_VSVDB_DV_12V1_UNIQUE_BX2_0);
			p_vsvdb->data.vsvdb_dv_12v1.unique_by2_0			= ((p_data[HD_EDID_POS_VSVDB_DV_12V1_UNIQUE_BY2_0] & HD_EDID_MASK_VSVDB_DV_12V1_UNIQUE_BY2_0) >> HD_EDID_SHFT_VSVDB_DV_12V1_UNIQUE_BY2_0);
			p_vsvdb->data.vsvdb_dv_12v1.low_latency1_0			= ((p_data[HD_EDID_POS_VSVDB_DV_12V1_LOW_LATENCY] & HD_EDID_MASK_VSVDB_DV_12V1_LOW_LATENCY) >> HD_EDID_SHFT_VSVDB_DV_12V1_LOW_LATENCY);
			p_vsvdb->data.vsvdb_dv_12v1.unique_gx6_0			= ((p_data[HD_EDID_POS_VSVDB_DV_12V1_UNIQUE_GX6_0] & HD_EDID_MASK_VSVDB_DV_12V1_UNIQUE_GX6_0) >> HD_EDID_SHFT_VSVDB_DV_12V1_UNIQUE_GX6_0);
			p_vsvdb->data.vsvdb_dv_12v1.unique_ry0				= ((p_data[HD_EDID_POS_VSVDB_DV_12V1_UNIQUE_RY0] & HD_EDID_MASK_VSVDB_DV_12V1_UNIQUE_RY0) >> HD_EDID_SHFT_VSVDB_DV_12V1_UNIQUE_RY0);
			p_vsvdb->data.vsvdb_dv_12v1.unique_gy6_0			= ((p_data[HD_EDID_POS_VSVDB_DV_12V1_UNIQUE_GY6_0] & HD_EDID_MASK_VSVDB_DV_12V1_UNIQUE_GY6_0) >> HD_EDID_SHFT_VSVDB_DV_12V1_UNIQUE_GY6_0);
			p_vsvdb->data.vsvdb_dv_12v1.unique_ry1				= ((p_data[HD_EDID_POS_VSVDB_DV_12V1_UNIQUE_RY1] & HD_EDID_MASK_VSVDB_DV_12V1_UNIQUE_RY1) >> HD_EDID_SHFT_VSVDB_DV_12V1_UNIQUE_RY1);
			p_vsvdb->data.vsvdb_dv_12v1.unique_rx4_0			= ((p_data[HD_EDID_POS_VSVDB_DV_12V1_UNIQUE_RX4_0] & HD_EDID_MASK_VSVDB_DV_12V1_UNIQUE_RX4_0) >> HD_EDID_SHFT_VSVDB_DV_12V1_UNIQUE_RX4_0);
			p_vsvdb->data.vsvdb_dv_12v1.unique_ry4_2			= ((p_data[HD_EDID_POS_VSVDB_DV_12V1_UNIQUE_RY4_2] & HD_EDID_MASK_VSVDB_DV_12V1_UNIQUE_RY4_2) >> HD_EDID_SHFT_VSVDB_DV_12V1_UNIQUE_RY4_2);
		} else {
			/* do nothing */
		}
		break;
	case HD_EDID_VAL_VSVDB_DV_VERSION_2:
		if (length == HD_EDID_LEN_VSVDB_DV_V2) {
			p_vsvdb->valid = HD_TRUE;
			p_vsvdb->type = HD_EDID_VSVDB_DV_TYPE_2;

			p_vsvdb->data.vsvdb_dv_v2.version						= version;
			p_vsvdb->data.vsvdb_dv_v2.dm_version					= ((p_data[HD_EDID_POS_VSVDB_DV_V2_DM_VERSION] & HD_EDID_MASK_VSVDB_DV_V2_DM_VERSION) >> HD_EDID_SHFT_VSVDB_DV_V2_DM_VERSION);
			p_vsvdb->data.vsvdb_dv_v2.supports_backlight_control	= ((p_data[HD_EDID_POS_VSVDB_DV_V2_BACKLIGHT_CTRL] & HD_EDID_MASK_VSVDB_DV_V2_BACKLIGHT_CTRL) >> HD_EDID_SHFT_VSVDB_DV_V2_BACKLIGHT_CTRL);
			p_vsvdb->data.vsvdb_dv_v2.supports_yuv422_12bit			= ((p_data[HD_EDID_POS_VSVDB_DV_V2_YUV422_12BIT] & HD_EDID_MASK_VSVDB_DV_V2_YUV422_12BIT) >> HD_EDID_SHFT_VSVDB_DV_V2_YUV422_12BIT);
			p_vsvdb->data.vsvdb_dv_v2.target_min_pq_v2_4_0			= ((p_data[HD_EDID_POS_VSVDB_DV_V2_TGT_MIN_PQV2_4_0] & HD_EDID_MASK_VSVDB_DV_V2_TGT_MIN_PQV2_4_0) >> HD_EDID_SHFT_VSVDB_DV_V2_TGT_MIN_PQV2_4_0);
			p_vsvdb->data.vsvdb_dv_v2.supports_global_dimming		= ((p_data[HD_EDID_POS_VSVDB_DV_V2_GLOBAL_DIMMING] & HD_EDID_MASK_VSVDB_DV_V2_GLOBAL_DIMMING) >> HD_EDID_SHFT_VSVDB_DV_V2_GLOBAL_DIMMING);
			p_vsvdb->data.vsvdb_dv_v2.backlt_min_luma1_0			= ((p_data[HD_EDID_POS_VSVDB_DV_V2_BACKLT_MIN_LUMA1_0] & HD_EDID_MASK_VSVDB_DV_V2_BACKLT_MIN_LUMA1_0) >> HD_EDID_SHFT_VSVDB_DV_V2_BACKLT_MIN_LUMA1_0);
			p_vsvdb->data.vsvdb_dv_v2.target_max_pq_v2_4_0			= ((p_data[HD_EDID_POS_VSVDB_DV_V2_TGT_MAX_PQV2_4_0] & HD_EDID_MASK_VSVDB_DV_V2_TGT_MAX_PQV2_4_0) >> HD_EDID_SHFT_VSVDB_DV_V2_TGT_MAX_PQV2_4_0);
			p_vsvdb->data.vsvdb_dv_v2.reserved						= ((p_data[HD_EDID_POS_VSVDB_DV_V2_RESERVED] & HD_EDID_MASK_VSVDB_DV_V2_RESERVED) >> HD_EDID_SHFT_VSVDB_DV_V2_RESERVED);
			p_vsvdb->data.vsvdb_dv_v2.interface1_0					= ((p_data[HD_EDID_POS_VSVDB_DV_V2_INTERFACE1_0] & HD_EDID_MASK_VSVDB_DV_V2_INTERFACE1_0) >> HD_EDID_SHFT_VSVDB_DV_V2_INTERFACE1_0);
			p_vsvdb->data.vsvdb_dv_v2.unique_gx6_0					= ((p_data[HD_EDID_POS_VSVDB_DV_V2_UNIQUE_GX6_0] & HD_EDID_MASK_VSVDB_DV_V2_UNIQUE_GX6_0) >> HD_EDID_SHFT_VSVDB_DV_V2_UNIQUE_GX6_0);
			p_vsvdb->data.vsvdb_dv_v2.supports_10b_12b_444_1		= ((p_data[HD_EDID_POS_VSVDB_DV_V2_10B12B_444_1] & HD_EDID_MASK_VSVDB_DV_V2_10B12B_444_1) >> HD_EDID_SHFT_VSVDB_DV_V2_10B12B_444_1);
			p_vsvdb->data.vsvdb_dv_v2.unique_gy6_0					= ((p_data[HD_EDID_POS_VSVDB_DV_V2_UNIQUE_GY6_0] & HD_EDID_MASK_VSVDB_DV_V2_UNIQUE_GY6_0) >> HD_EDID_SHFT_VSVDB_DV_V2_UNIQUE_GY6_0);
			p_vsvdb->data.vsvdb_dv_v2.supports_10b_12b_444_0		= ((p_data[HD_EDID_POS_VSVDB_DV_V2_10B12B_444_0] & HD_EDID_MASK_VSVDB_DV_V2_10B12B_444_0) >> HD_EDID_SHFT_VSVDB_DV_V2_10B12B_444_0);
			p_vsvdb->data.vsvdb_dv_v2.unique_rx4_0					= ((p_data[HD_EDID_POS_VSVDB_DV_V2_UNIQUE_RX4_0] & HD_EDID_MASK_VSVDB_DV_V2_UNIQUE_RX4_0) >> HD_EDID_SHFT_VSVDB_DV_V2_UNIQUE_RX4_0);
			p_vsvdb->data.vsvdb_dv_v2.unique_bx2_0					= ((p_data[HD_EDID_POS_VSVDB_DV_V2_UNIQUE_BX2_0] & HD_EDID_MASK_VSVDB_DV_V2_UNIQUE_BX2_0) >> HD_EDID_SHFT_VSVDB_DV_V2_UNIQUE_BX2_0);
			p_vsvdb->data.vsvdb_dv_v2.unique_ry4_0					= ((p_data[HD_EDID_POS_VSVDB_DV_V2_UNIQUE_RY4_0] & HD_EDID_MASK_VSVDB_DV_V2_UNIQUE_RY4_0) >> HD_EDID_SHFT_VSVDB_DV_V2_UNIQUE_RY4_0);
			p_vsvdb->data.vsvdb_dv_v2.unique_by2_0					= ((p_data[HD_EDID_POS_VSVDB_DV_V2_UNIQUE_BY2_0] & HD_EDID_MASK_VSVDB_DV_V2_UNIQUE_BY2_0) >> HD_EDID_SHFT_VSVDB_DV_V2_UNIQUE_BY2_0);
		}
		break;
	default:
		/* unknown version */
		break;
	}
}

/*--------------------------------------------------------------------------*/
/* @brief     : Analyze VSADB                                               */
/*--------------------------------------------------------------------------*/
static void hd_edid_AnalyzeVsadb(uint8_t *p_data, HD_EDID_EXT_VSADB_t *p_vsadb)
{
	if ((p_data[HD_EDID_POS_VSVDB_OUI0] != HD_EDID_VAL_VSVDB_OUI_DOLBY_0)
	||  (p_data[HD_EDID_POS_VSVDB_OUI1] != HD_EDID_VAL_VSVDB_OUI_DOLBY_1)
	||  (p_data[HD_EDID_POS_VSVDB_OUI2] != HD_EDID_VAL_VSVDB_OUI_DOLBY_2)) {
		/* if not Dolby */
		return;
	}

	p_vsadb->valid = HD_TRUE;
	p_vsadb->data.head_phone					= ((p_data[HD_EDID_POS_VSADB_HEAD_PHONE] & HD_EDID_MASK_VSADB_HEAD_PHONE) >> HD_EDID_SHFT_VSADB_HEAD_PHONE);
	p_vsadb->data.output_configuration_support	= ((p_data[HD_EDID_POS_VSADB_OUTPUT_CONF_SUPPORT] & HD_EDID_MASK_VSADB_OUTPUT_CONF_SUPPORT) >> HD_EDID_SHFT_VSADB_OUTPUT_CONF_SUPPORT);
	p_vsadb->data.f63							= ((p_data[HD_EDID_POS_VSADB_F63] & HD_EDID_MASK_VSADB_F63) >> HD_EDID_SHFT_VSADB_F63);
	p_vsadb->data.version						= ((p_data[HD_EDID_POS_VSADB_VERSION] & HD_EDID_MASK_VSADB_VERSION) >> HD_EDID_SHFT_VSADB_VERSION);
	p_vsadb->data.f77							= ((p_data[HD_EDID_POS_VSADB_F77] & HD_EDID_MASK_VSADB_F77) >> HD_EDID_SHFT_VSADB_F77);
	p_vsadb->data.sink_product_capabilities		= ((p_data[HD_EDID_POS_VSADB_SINK_PRODUCT_CAP] & HD_EDID_MASK_VSADB_SINK_PRODUCT_CAP) >> HD_EDID_SHFT_VSADB_SINK_PRODUCT_CAP);
}

/*--------------------------------------------------------------------------*/
/* @brief     : Analyze CMR                                                 */
/*--------------------------------------------------------------------------*/
static void hd_edid_AnalyzeCmr(uint8_t *p_data, HD_EDID_EXT_CMR_t *p_cmr)
{
	p_cmr->valid = HD_TRUE;

	p_cmr->data.bt2020 = ((p_data[HD_EDID_POS_CMR_BT2020] & HD_EDID_MASK_CMR_BT2020) >> HD_EDID_SHFT_CMR_BT2020);
	p_cmr->data.bt2020 &= s_HdEdidDefault.ext.cmr.data.bt2020;

	p_cmr->data.adobe = ((p_data[HD_EDID_POS_CMR_ADOBE] & HD_EDID_MASK_CMR_ADOBE) >> HD_EDID_SHFT_CMR_ADOBE);
	p_cmr->data.adobe &= s_HdEdidDefault.ext.cmr.data.adobe;

	p_cmr->data.ycc = (p_data[HD_EDID_POS_CMR_YCC] & HD_EDID_MASK_CMR_YCC);
	p_cmr->data.ycc &= s_HdEdidDefault.ext.cmr.data.ycc;

	p_cmr->data.md = (p_data[HD_EDID_POS_CMR_MD] & HD_EDID_MASK_CMR_MD);
	p_cmr->data.md &= s_HdEdidDefault.ext.cmr.data.md;
}

/* HDMI2.0 Y420 EDID */
/*--------------------------------------------------------------------------*/
/* @brief     : Analyze Y420 VDB                                            */
/*--------------------------------------------------------------------------*/
static void hd_edid_AnalyzeY420vdb(uint8_t *p_data, HD_EDID_EXT_Y420VDB_t *p_y420vdb, uint8_t length)
{
	uint8_t i, j, vic;

	/* parameter check */
	if (length < HD_EDID_LEN_Y420VDB_EXTAG) {
		return;
	}

	p_y420vdb->valid = HD_FALSE;

	if (*p_data != HD_FALSE) {		/* check valid */
		for (i = 0; (i < (length - HD_EDID_LEN_Y420VDB_EXTAG)) && (i < HD_EDID_LEN_Y420VDB_VIC); i++) {
			vic = p_data[HD_EDID_LEN_Y420VDB_EXTAG + i];
	
			if (hd_edid_IsSupportY420(vic) != HD_FALSE) {		/* PRQA S 3416 */
				for (j = 0; j < p_y420vdb->count; j++) {
					if (vic == p_y420vdb->vic[j]) {
						break;
					}
				}
				if (j < p_y420vdb->count) {
					continue;
				}
	
				p_y420vdb->vic[p_y420vdb->count] = vic;
				p_y420vdb->count++;
			}
		}
		p_y420vdb->valid = HD_TRUE;
	}

}

/*--------------------------------------------------------------------------*/
/* @brief     : Analyze Y420CMDB                                            */
/*--------------------------------------------------------------------------*/
static void hd_edid_AnalyzeY420cmdb(uint8_t *p_data, HD_EDID_EXT_Y420CMDB_t *p_y420cmdb, HD_EDID_EXT_Y420VDB_t *p_y420vdb, uint8_t length)
{
	uint8_t i;

	/* parameter check */
	if (length < HD_EDID_LEN_Y420CMDB_EXTAG) {
		return;
	}

	p_y420cmdb->valid = HD_FALSE;

	if (*p_data != HD_FALSE) {		/* check valid */
		/* set only mask info because SVD may not be analyzed at this point. */
		if (length == HD_EDID_LEN_Y420CMDB_EXTAG) {
			/* set all mask to 0xff to analyze with vic info later. */
			for (i = 0; i < HD_EDID_LEN_Y420CMDB_MASK; i++) {
				p_y420cmdb->mask[i] = 0xff;
			}
			p_y420cmdb->count = HD_EDID_LEN_Y420CMDB_MASK;
		} else if (length > HD_EDID_LEN_Y420CMDB_EXTAG) {
			for (i = 0; (i < (length - HD_EDID_LEN_Y420CMDB_EXTAG)) && (i < HD_EDID_LEN_Y420CMDB_MASK); i++) {
				p_y420cmdb->mask[i] |= p_data[HD_EDID_LEN_Y420CMDB_EXTAG + i];
			}
			if (p_y420cmdb->count < i) {
				p_y420cmdb->count = i;
			}
		} else {
			/* do nothing */
		}
		p_y420cmdb->valid = HD_TRUE;
	}
}

/*--------------------------------------------------------------------------*/
/* @brief     : Analyze HDR                                                 */
/*--------------------------------------------------------------------------*/
static void hd_edid_AnalyzeHdr(uint8_t *p_data, HD_EDID_EXT_HDR_t *p_hdr)
{
	uint8_t i;

	p_hdr->valid = HD_TRUE;

	p_hdr->data.support_eotf = p_data[HD_EDID_POS_HDR_SUPPORT_EOTF] & s_HdEdidDefault.ext.hdr.data.support_eotf;

	p_hdr->data.md = p_data[HD_EDID_POS_HDR_MD] & s_HdEdidDefault.ext.hdr.data.md;

	for (i = 0; i < HD_EDID_NUM_HDR_LUMINANCE_DATA; i++) {
		p_hdr->data.luminance[i] = 0x00;
	}

	for (i = 0; (i + HD_EDID_POS_HDR_LUMINANCE_DATA) <= p_hdr->length; i++) {
		if (p_data[i + HD_EDID_POS_HDR_LUMINANCE_DATA] == 0x00) {
			p_hdr->length = i + HD_EDID_POS_HDR_LUMINANCE_DATA;
			break;
		}
		p_hdr->data.luminance[i] = p_data[i + HD_EDID_POS_HDR_LUMINANCE_DATA];
	}

	if ((p_hdr->length == (HD_EDID_POS_EXT_TAG + HD_EDID_POS_HDR_LUMINANCE_DATA + HD_EDID_POS_HDR_LUMINANCE_DATA_MIN))
	&&  (p_hdr->data.luminance[HD_EDID_POS_HDR_LUMINANCE_DATA_MAX] <= p_hdr->data.luminance[HD_EDID_POS_HDR_LUMINANCE_DATA_MIN])) {
		p_hdr->length = HD_EDID_LEN_HDR_LUMINANCE_NON_SUPPORT;
	}
}

/*--------------------------------------------------------------------------*/
/* @brief     : Check Y420 Support                                          */
/*--------------------------------------------------------------------------*/
static void hd_edid_Check420Support(HD_EDID_EXT_Y420CMDB_t *p_y420cmdb, HD_EDID_EXT_Y420VDB_t *p_y420vdb)
{
	uint8_t i, j;
	uint8_t shift;
	uint8_t vic_index = 0;
	bool_t  vdb_vic = HD_FALSE;


	/* SupportCheck */
	for (i = 0; i < p_y420cmdb->count; i++) {
		for (shift = 0; shift < 8; shift++) {
			vdb_vic = HD_FALSE;
			if ((p_y420cmdb->mask[i] >> shift) & 0x01) {
				if (hd_edid_IsSupportY420(p_y420cmdb->vic[vic_index]) != HD_FALSE) {
					for (j = 0; j < p_y420vdb->count; j++) {
						if (p_y420vdb->vic[j] == p_y420cmdb->vic[vic_index]) {
							vdb_vic = HD_TRUE;
							break;
						}
					}

					if (vdb_vic != HD_TRUE) {
						p_y420vdb->vic[p_y420vdb->count] = p_y420cmdb->vic[vic_index];
						p_y420vdb->count++;
						p_y420vdb->valid = HD_TRUE;
					}
				}
			}
			vic_index++;
		}
	}
}

/* HDMI2.1 HF-SCDB EDID */
/*--------------------------------------------------------------------------*/
/* @brief     : Analyze HF-SCDB                                             */
/*--------------------------------------------------------------------------*/
static void hd_edid_AnalyzeHfScdb(uint8_t *p_data, HD_EDID_EXT_HFSCDB_t *p_hfscdb)
{
	uint8_t len;

	len = (p_data[HD_EDID_POS_CEA_HEADER] & HD_EDID_MASK_CEA_LEN);
	/* fail safe */
	if (len < HD_EDID_LEN_MIN) {
		return;
	}
	if (p_data[HD_EDID_POS_SCDS_TMDS_CLOCK] <= s_HdEdidDefault.vsdb.ext.tmds_clock) {
		return;
	}
	p_hfscdb->valid = HD_TRUE;

	/* 2byte,3byte Reserved */
	p_hfscdb->data.hfscdb_data[0] = p_data[2];
	p_hfscdb->data.hfscdb_data[1] = p_data[3];
	/* HF-SCDB */
	hd_edid_AnalyzeScds(p_data, &p_hfscdb->data.scds_data, len);
	if (len > HD_EDID_LEN_DESC_SCDS_MAX) {
		len = HD_EDID_LEN_DESC_SCDS_MAX;
	}
	p_hfscdb->count = len;
}

/* Copy */
/*--------------------------------------------------------------------------*/
/* @brief     : Set EDID RAM Block 0                                        */
/*--------------------------------------------------------------------------*/
bool_t hd_edid_SetEdidRamBlock0(HD_MODULE_PORT_TX_t tx, uint8_t *p_src)
{
	if (tx >= HD_MAX__MODULE_PORT_TX) {
		return HD_FALSE;
	}

	hd_edid_RamInitializeDataSink(tx);

	/* parameter check */
	if ((tx >= HD_MAX__MODULE_PORT_TX) || (p_src == 0)) {
		return HD_FALSE;
	}

	s_HdEdidRamSink[tx] = p_src;

	return HD_TRUE;
}

/*--------------------------------------------------------------------------*/
/* @brief     : Set Tx EDID Block Size                                      */
/*--------------------------------------------------------------------------*/
void hd_edid_SetTxEdidBlockSize(uint8_t tx, uint8_t size)
{
	if (tx < HD_MAX__MODULE_PORT_TX) {
		s_HdEdidTxSize[tx] = size;
	}
}

/*--------------------------------------------------------------------------*/
/* @brief     : Get Tx EDID Block Size                                      */
/*--------------------------------------------------------------------------*/
uint8_t hd_edid_GetTxEdidBlockSize(uint8_t tx)
{
	return s_HdEdidTxSize[tx];
}

