/*
 * Freescale i.MX drm driver
 *
 * Copyright (C) 2011 Sascha Hauer, Pengutronix
 * Copyright 2020 NXP
 * Copyright 2020 Sony Home Entertainment & Sound Products Inc.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */
#include <linux/component.h>
#include <linux/device.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/pm_runtime.h>
#include <drm/drmP.h>
#include <drm/drm_atomic.h>
#include <drm/drm_atomic_helper.h>
#include <drm/drm_fb_helper.h>
#include <drm/drm_crtc_helper.h>
#include <drm/drm_gem_cma_helper.h>
#include <drm/drm_fb_cma_helper.h>
#include <drm/drm_plane_helper.h>
#include <drm/drm_of.h>
#include <video/imx-ipu-v3.h>
#include <video/dpu.h>
#include <video/imx-dcss.h>
#include <video/imx-lcdif.h>
#include <linux/of_reserved_mem.h>

#include "imx-drm.h"
#include "ipuv3/ipuv3-plane.h"
#if IS_ENABLED(CONFIG_DRM_IMX_CUSTOMIZATION_MAX_RESOLUTION)
#include <drm/drm_sysfs.h>
#endif

#include "../bridge/adv7511/adv7511.h"

#if IS_ENABLED(CONFIG_DRM_FBDEV_EMULATION)
static int legacyfb_depth = 16;
module_param(legacyfb_depth, int, 0444);
#endif

static void imx_drm_driver_lastclose(struct drm_device *drm)
{
	struct imx_drm_device *imxdrm = drm->dev_private;

	drm_fbdev_cma_restore_mode(imxdrm->fbhelper);
}

DEFINE_DRM_GEM_CMA_FOPS(imx_drm_driver_fops);

void imx_drm_connector_destroy(struct drm_connector *connector)
{
	drm_connector_unregister(connector);
	drm_connector_cleanup(connector);
}
EXPORT_SYMBOL_GPL(imx_drm_connector_destroy);

void imx_drm_encoder_destroy(struct drm_encoder *encoder)
{
	drm_encoder_cleanup(encoder);
}
EXPORT_SYMBOL_GPL(imx_drm_encoder_destroy);

int imx_drm_encoder_parse_of(struct drm_device *drm,
	struct drm_encoder *encoder, struct device_node *np)
{
	uint32_t crtc_mask = drm_of_find_possible_crtcs(drm, np);

	/*
	 * If we failed to find the CRTC(s) which this encoder is
	 * supposed to be connected to, it's because the CRTC has
	 * not been registered yet.  Defer probing, and hope that
	 * the required CRTC is added later.
	 */
	if (crtc_mask == 0)
		return -EPROBE_DEFER;

	encoder->possible_crtcs = crtc_mask;

	/* FIXME: this is the mask of outputs which can clone this output. */
	encoder->possible_clones = ~0;

	return 0;
}
EXPORT_SYMBOL_GPL(imx_drm_encoder_parse_of);

static const struct drm_ioctl_desc imx_drm_ioctls[] = {
	/* none so far */
};

static struct drm_driver imx_drm_driver = {
	.driver_features	= DRIVER_MODESET | DRIVER_GEM | DRIVER_PRIME |
				  DRIVER_ATOMIC,
	.lastclose		= imx_drm_driver_lastclose,
	.gem_free_object_unlocked = drm_gem_cma_free_object,
	.gem_vm_ops		= &drm_gem_cma_vm_ops,
	.dumb_create		= drm_gem_cma_dumb_create,

	.prime_handle_to_fd	= drm_gem_prime_handle_to_fd,
	.prime_fd_to_handle	= drm_gem_prime_fd_to_handle,
	.gem_prime_import	= drm_gem_prime_import,
	.gem_prime_export	= drm_gem_prime_export,
	.gem_prime_get_sg_table	= drm_gem_cma_prime_get_sg_table,
	.gem_prime_import_sg_table = drm_gem_cma_prime_import_sg_table,
	.gem_prime_vmap		= drm_gem_cma_prime_vmap,
	.gem_prime_vunmap	= drm_gem_cma_prime_vunmap,
	.gem_prime_mmap		= drm_gem_cma_prime_mmap,
	.ioctls			= imx_drm_ioctls,
	.num_ioctls		= ARRAY_SIZE(imx_drm_ioctls),
	.fops			= &imx_drm_driver_fops,
	.name			= "imx-drm",
	.desc			= "i.MX DRM graphics",
	.date			= "20120507",
	.major			= 1,
	.minor			= 0,
	.patchlevel		= 0,
};

static int compare_of(struct device *dev, void *data)
{
	struct device_node *np = data;

	/* Special case for DI, dev->of_node may not be set yet */
	if (strcmp(dev->driver->name, "imx-ipuv3-crtc") == 0) {
		struct ipu_client_platformdata *pdata = dev->platform_data;

		return pdata->of_node == np;
	} else if (strcmp(dev->driver->name, "imx-dpu-crtc") == 0) {
		struct dpu_client_platformdata *pdata = dev->platform_data;

		return pdata->of_node == np;
	}  else if (strcmp(dev->driver->name, "imx-dcss-crtc") == 0) {
		struct dcss_client_platformdata *pdata = dev->platform_data;

		return pdata->of_node == np;
	} else if (strcmp(dev->driver->name, "imx-lcdif-crtc") == 0) {
		struct lcdif_client_platformdata *pdata = dev->platform_data;
#if IS_ENABLED(CONFIG_DRM_FBDEV_EMULATION)
		/* set legacyfb_depth to be 32 for lcdif, since
		 * default format of the connectors attached to
		 * lcdif is usually RGB888
		 */
		if (pdata->of_node == np)
			legacyfb_depth = 32;
#endif

		return pdata->of_node == np;
	}

	/* This is a special case for dpu bliteng. */
	if (strcmp(dev->driver->name, "imx-drm-dpu-bliteng") == 0) {
		struct dpu_client_platformdata *pdata = dev->platform_data;

		return pdata->of_node == np;
	}

	/* Special case for LDB, one device for two channels */
	if (of_node_cmp(np->name, "lvds-channel") == 0) {
		np = of_get_parent(np);
		of_node_put(np);
	}

	return dev->of_node == np;
}

static const char *const imx_drm_dpu_comp_parents[] = {
	"fsl,imx8qm-dpu",
	"fsl,imx8qxp-dpu",
};

static const char *const imx_drm_dcss_comp_parents[] = {
	"nxp,imx8mq-dcss",
};

static bool imx_drm_parent_is_compatible(struct device *dev,
					 const char *const comp_parents[],
					 int comp_parents_size)
{
	struct device_node *port, *parent;
	bool ret = false;
	int i;

	port = of_parse_phandle(dev->of_node, "ports", 0);
	if (!port)
		return ret;

	parent = of_get_parent(port);

	for (i = 0; i < comp_parents_size; i++) {
		if (of_device_is_compatible(parent, comp_parents[i])) {
			ret = true;
			break;
		}
	}

	of_node_put(parent);

	of_node_put(port);

	return ret;
}

#if IS_ENABLED(CONFIG_DRM_IMX_CUSTOMIZATION_MAX_RESOLUTION)
static int
imx_drm_get_customized_fb_resolution(struct device *dev,
				     struct drm_customized_fb_resolution *res)
{
	const char *mode_str = NULL;
	u32 property_val;
	int ret;

	of_property_read_u32(dev->of_node,
			     "custom-max-fb-size-width",
			     &property_val);
	if (property_val <= 4096)
		res->max_width = (int)property_val;
	else
		return -EINVAL;

	of_property_read_u32(dev->of_node,
			     "custom-max-fb-size-height",
			     &property_val);
	if (property_val <= 4096)
		res->max_height = (int)property_val;
	else
		return -EINVAL;

	of_property_read_u32(dev->of_node,
			     "custom-fb-size-bpp",
			     &property_val);
	if (property_val <= 4096)
		res->bpp = (int)property_val;
	else
		return -EINVAL;

	of_property_read_u32(dev->of_node,
			     "custom-fb-frequency",
			     &property_val);
	if (property_val <= 4096)
		res->refresh = (int)property_val;
	else
		return -EINVAL;

	ret = of_property_read_string(dev->of_node,
				      "custom-fb-mode",
				      &mode_str);
	if (!ret && mode_str) {
		if (strcmp(mode_str, "i") == 0)
			res->interlace = true;
	} else {
		return -EINVAL;
	}

	DRM_DEBUG_KMS("customized resolution from dts: '%dx%d-%d@%d%s'\n",
		      res->max_width,
		      res->max_height,
		      res->bpp,
		      res->refresh,
		      (res->interlace)?"i":"p");

	return 0;
}
#endif /* IS_ENABLED(CONFIG_DRM_IMX_CUSTOMIZATION_MAX_RESOLUTION) */

static inline bool has_dpu(struct device *dev)
{
	return imx_drm_parent_is_compatible(dev, imx_drm_dpu_comp_parents,
					ARRAY_SIZE(imx_drm_dpu_comp_parents));
}

static inline bool has_dcss(struct device *dev)
{
	return imx_drm_parent_is_compatible(dev, imx_drm_dcss_comp_parents,
					ARRAY_SIZE(imx_drm_dcss_comp_parents));
}

static void add_dpu_bliteng_components(struct device *dev,
				       struct component_match **matchptr)
{
	/*
	 * As there may be two dpu bliteng device,
	 * so need add something in compare data to distinguish.
	 * Use its parent dpu's of_node as the data here.
	 */
	struct device_node *port, *parent;
	/* assume max dpu number is 8 */
	struct device_node *dpu[8];
	int num_dpu = 0;
	int i, j;
	bool found = false;

	for (i = 0; ; i++) {
		port = of_parse_phandle(dev->of_node, "ports", i);
		if (!port)
			break;

		parent = of_get_parent(port);

		for (j = 0; j < num_dpu; j++) {
			if (dpu[j] == parent) {
				found = true;
				break;
			}
		}

		if (found) {
			found = false;
		} else {
			if (num_dpu >= ARRAY_SIZE(dpu)) {
				dev_err(dev, "The number of found dpu is greater than max [%ld].\n",
					ARRAY_SIZE(dpu));
				of_node_put(parent);
				of_node_put(port);
				break;
			}

			dpu[num_dpu] = parent;
			num_dpu++;

			component_match_add(dev, matchptr, compare_of, parent);
		}

		of_node_put(parent);
		of_node_put(port);
	}
}

static int imx_drm_bind(struct device *dev)
{
	struct drm_device *drm;
	struct imx_drm_device *imxdrm;
	int ret;

	DRM_DEBUG_KMS("Entering ...\n");

	if (has_dpu(dev))
		imx_drm_driver.driver_features |= DRIVER_RENDER;

	drm = drm_dev_alloc(&imx_drm_driver, dev);
	if (IS_ERR(drm))
		return PTR_ERR(drm);

	imxdrm = devm_kzalloc(dev, sizeof(*imxdrm), GFP_KERNEL);
	if (!imxdrm) {
		ret = -ENOMEM;
		goto err_unref;
	}

	imxdrm->drm = drm;
	drm->dev_private = imxdrm;

	imxdrm->wq = alloc_ordered_workqueue("imxdrm", 0);
	if (!imxdrm->wq) {
		ret = -ENOMEM;
		goto err_unref;
	}

	init_waitqueue_head(&imxdrm->commit.wait);

	/*
	 * enable drm irq mode.
	 * - with irq_enabled = true, we can use the vblank feature.
	 *
	 * P.S. note that we wouldn't use drm irq handler but
	 *      just specific driver own one instead because
	 *      drm framework supports only one irq handler and
	 *      drivers can well take care of their interrupts
	 */
	drm->irq_enabled = true;

	/*
	 * set max width and height as default value(4096x4096).
	 * this value would be used to check framebuffer size limitation
	 * at drm_mode_addfb().
	 */
	drm->mode_config.min_width = 1;
	drm->mode_config.min_height = 1;
	drm->mode_config.max_width = 4096;
	drm->mode_config.max_height = 4096;

	if (has_dpu(dev) || has_dcss(dev)) {
		drm->mode_config.allow_fb_modifiers = true;
		dev_dbg(dev, "allow fb modifiers\n");
	}

#if IS_ENABLED(CONFIG_DRM_IMX_CUSTOMIZATION_MAX_RESOLUTION)
	ret = imx_drm_get_customized_fb_resolution(dev, &drm->customized_fb_res);
	if (ret)
		DRM_WARN("Failed to get customized fb resolution\n");
#endif

	ret = of_reserved_mem_device_init(dev);
	if (ret && ret != -ENODEV) {
		dev_err(drm->dev, "Couldn't claim the device memory region\n");
		goto err_unref;
	}

	drm_mode_config_init(drm);

	ret = drm_vblank_init(drm, MAX_CRTC);
	if (ret)
		goto err_kms;

	dev_set_drvdata(dev, drm);

	DRM_DEBUG_KMS("=> component_bind_all\n");
	/* Now try and bind all our sub-components */
	ret = component_bind_all(dev, drm);
	if (ret)
		goto err_kms;
	DRM_DEBUG_KMS("=> done with component_bind_all\n");

	drm_mode_config_reset(drm);

	/*
	 * All components are now initialised, so setup the fb helper.
	 * The fb helper takes copies of key hardware information, so the
	 * crtcs/connectors/encoders must not change after this point.
	 */
#if IS_ENABLED(CONFIG_DRM_FBDEV_EMULATION)
	if (legacyfb_depth != 16 && legacyfb_depth != 32) {
		dev_warn(dev, "Invalid legacyfb_depth.  Defaulting to 16bpp\n");
		legacyfb_depth = 16;
	}

	if (legacyfb_depth == 16 && has_dcss(dev))
		legacyfb_depth = 32;

	DRM_DEBUG_KMS("Creating Emulated Frame Buffer Device...\n");
	imxdrm->fbhelper = drm_fbdev_cma_init(drm, legacyfb_depth, MAX_CRTC);
	if (IS_ERR(imxdrm->fbhelper)) {
		ret = PTR_ERR(imxdrm->fbhelper);
		imxdrm->fbhelper = NULL;
		goto err_unbind;
	}
	DRM_DEBUG_KMS("Done!\n");
#endif

	drm_kms_helper_poll_init(drm);

	ret = drm_dev_register(drm, 0);
	if (ret)
		goto err_fbhelper;

#if IS_ENABLED(CONFIG_DRM_IMX_CUSTOMIZATION_MAX_RESOLUTION)
	drm_sysfs_max_custom_resolution_create(drm);
#endif

	DRM_DEBUG_KMS("Registered DRM device!\n");

	return 0;

err_fbhelper:
	drm_kms_helper_poll_fini(drm);
#if IS_ENABLED(CONFIG_DRM_FBDEV_EMULATION)
	if (imxdrm->fbhelper)
		drm_fbdev_cma_fini(imxdrm->fbhelper);
err_unbind:
#endif
	component_unbind_all(drm->dev, drm);
err_kms:
	dev_set_drvdata(dev, NULL);
	drm_mode_config_cleanup(drm);
	destroy_workqueue(imxdrm->wq);
err_unref:
	drm_dev_unref(drm);

	return ret;
}

static void imx_drm_unbind(struct device *dev)
{
	struct drm_device *drm = dev_get_drvdata(dev);
	struct imx_drm_device *imxdrm = drm->dev_private;

#if IS_ENABLED(CONFIG_DRM_IMX_CUSTOMIZATION_MAX_RESOLUTION)
	drm_sysfs_max_custom_resolution_destroy();
#endif

	if (has_dpu(dev))
		imx_drm_driver.driver_features &= ~DRIVER_RENDER;

	flush_workqueue(imxdrm->wq);

	drm_dev_unregister(drm);

	drm_kms_helper_poll_fini(drm);

	if (imxdrm->fbhelper)
		drm_fbdev_cma_fini(imxdrm->fbhelper);

	drm_mode_config_cleanup(drm);

	component_unbind_all(drm->dev, drm);
	dev_set_drvdata(dev, NULL);

	destroy_workqueue(imxdrm->wq);

	drm_dev_unref(drm);
}

static const struct component_master_ops imx_drm_ops = {
	.bind = imx_drm_bind,
	.unbind = imx_drm_unbind,
};

#if IS_ENABLED(CONFIG_DRM_IMX_CHANGE_RESOLUTION)
static ssize_t
imx_drm_sysfs_switch_res_store(struct device *dev,
				 struct device_attribute *attr,
				 const char *buf, size_t count)
{
	struct drm_device *drm = dev_get_drvdata(dev);
	struct imx_drm_device *imxdrm = drm->dev_private;
	int ret = -EINVAL;

	if (strncmp(buf, "1", 1)) {
		DRM_WARN("sysfs entry 'switch_res' only accepts '1' as value\n");
		return ret;
	}

	/* Tear down current emulated FBDEV */
	drm_fbdev_cma_fini(imxdrm->fbhelper);

	/* Re-create a brand new one */
	imxdrm->fbhelper = drm_fbdev_cma_init(drm, 32, MAX_CRTC);
	if (IS_ERR(imxdrm->fbhelper)) {
		DRM_ERROR("Failed to create emulated FBDEV\n");
		ret = PTR_ERR(imxdrm->fbhelper);
		imxdrm->fbhelper = NULL;
		return ret;
	}

	return count;
}

static DEVICE_ATTR(switch_res, S_IWUSR,	NULL, imx_drm_sysfs_switch_res_store);

static ssize_t
imx_drm_sysfs_new_res_store(struct device *dev,
				 struct device_attribute *attr,
				 const char *buf, size_t count)
{
	struct drm_device *drm = dev_get_drvdata(dev);
	struct drm_cmdline_mode mode;

	if (drm->mode_config.mode_options)
		kfree (drm->mode_config.mode_options);

	drm->mode_config.mode_options = kstrdup(buf, GFP_KERNEL);
	if (!drm->mode_config.mode_options)
		return -ENOMEM;
	strim(drm->mode_config.mode_options);

	/* check validity for mode options format */
	if (!drm_mode_parse_command_line_for_connector(drm->mode_config.mode_options,
						       NULL,
						       &mode)) {
		DRM_WARN("invalid option format: [%s]\n", drm->mode_config.mode_options);
		DRM_WARN("option format should be <xres>x<yres>[-<bpp>][@<refresh>][i]\n");
		return -EINVAL;
	}

	DRM_DEBUG_KMS("option format [%s] is valid\n", drm->mode_config.mode_options);

	return count;
}

static ssize_t
imx_drm_sysfs_new_res_show(struct device *dev,
			     struct device_attribute *attr, char *buf)
{
	struct drm_device *drm = dev_get_drvdata(dev);

	if (drm->mode_config.mode_options)
		return scnprintf(buf, PAGE_SIZE, "%s\n", drm->mode_config.mode_options);
	else
		return 0;
}

static DEVICE_ATTR(new_res, S_IRUGO | S_IWUSR, \
	imx_drm_sysfs_new_res_show, \
	imx_drm_sysfs_new_res_store);

static struct attribute*
imx_drm_sysfs_switch_res_attrs[] = {
	&dev_attr_switch_res.attr,
	&dev_attr_new_res.attr,
	NULL,
};


static const struct attribute_group
imx_drm_sysfs_switch_res_grp = {
	.name = "switch_res",
	.attrs = imx_drm_sysfs_switch_res_attrs,
};
#endif /* IS_ENABLED(CONFIG_DRM_IMX_CHANGE_RESOLUTION) */

static int imx_drm_platform_probe(struct platform_device *pdev)
{
	struct component_match *match = NULL;
	int ret;

	if (has_dpu(&pdev->dev))
		add_dpu_bliteng_components(&pdev->dev, &match);

	ret = drm_of_component_probe_with_match(&pdev->dev, match, compare_of,
						&imx_drm_ops);

	if (!ret)
		ret = dma_set_coherent_mask(&pdev->dev, DMA_BIT_MASK(32));

#if IS_ENABLED(CONFIG_DRM_IMX_CHANGE_RESOLUTION)
	ret = sysfs_create_group(&pdev->dev.kobj, &imx_drm_sysfs_switch_res_grp);
	if (ret) {
		DRM_ERROR("Failed to create sysfs group\n");
	}
#endif

	pm_runtime_enable(&pdev->dev);
	dev_info(&pdev->dev, "enable pm_runtime\n");
	/* Do nothing at runtime_resume, because drm_device is not allocate */
	pm_runtime_forbid(&pdev->dev);

	return ret;
}

static int imx_drm_platform_remove(struct platform_device *pdev)
{
	component_master_del(&pdev->dev, &imx_drm_ops);
	if (pm_runtime_active(&pdev->dev)) {
		dev_err(&pdev->dev, "runtime still active\n");
		pm_runtime_allow(&pdev->dev);
	}

	pm_runtime_disable(&pdev->dev);

	return 0;
}

#ifdef CONFIG_PM_SLEEP
static int imx_drm_runtime_suspend(struct device *dev)
{
	struct drm_device *drm_dev = dev_get_drvdata(dev);
	struct imx_drm_device *imxdrm;

	pr_debug("%s: called\n", __func__);
	/* The drm_dev is NULL before .load hook is called */
	if (drm_dev == NULL)
		return 0;

	drm_kms_helper_poll_disable(drm_dev);

	imxdrm = drm_dev->dev_private;
	imxdrm->state = drm_atomic_helper_suspend(drm_dev);
	if (IS_ERR(imxdrm->state)) {
		pr_warn("%s: suspend has been failed\n", __func__);
		drm_kms_helper_poll_enable(drm_dev);
		return PTR_ERR(imxdrm->state);
	}

	adv7511_regulator_off();

	pr_debug("%s: done\n", __func__);

	return 0;
}

static int imx_drm_runtime_resume(struct device *dev)
{
	struct drm_device *drm_dev = dev_get_drvdata(dev);
	struct imx_drm_device *imx_drm;

	pr_debug("%s: called\n", __func__);
	if (drm_dev == NULL)
		return 0;

	adv7511_regulator_on();

	imx_drm = drm_dev->dev_private;
	drm_atomic_helper_resume(drm_dev, imx_drm->state);
	drm_kms_helper_poll_enable(drm_dev);
	pr_debug("%s: done\n", __func__);

	return 0;
}

static int imx_drm_suspend(struct device *dev)
{
	struct drm_device *drm_dev = dev_get_drvdata(dev);

	/* The drm_dev is NULL before .load hook is called */
	if (drm_dev == NULL)
		return 0;

	if (pm_runtime_active(dev)) {
		dev_info(dev, "%s Call imx_drm_runtime_suspend\n", __func__);
		return imx_drm_runtime_suspend(dev);
	}

	return 0;
}

static int imx_drm_resume(struct device *dev)
{
	struct drm_device *drm_dev = dev_get_drvdata(dev);

	if (drm_dev == NULL)
		return 0;

	if (pm_runtime_active(dev)) {
		dev_info(dev, "%s Call imx_drm_runtime_resume\n", __func__);
		return imx_drm_runtime_resume(dev);
	}

	return 0;
}

#endif

//static SIMPLE_DEV_PM_OPS(imx_drm_pm_ops, imx_drm_suspend, imx_drm_resume);

static const struct dev_pm_ops imx_drm_pm_ops = {
	.runtime_suspend = imx_drm_runtime_suspend,
	.runtime_resume = imx_drm_runtime_resume,
	.suspend = imx_drm_suspend,
	.resume = imx_drm_resume,
};

static const struct of_device_id imx_drm_dt_ids[] = {
	{ .compatible = "fsl,imx-display-subsystem", },
	{ /* sentinel */ },
};
MODULE_DEVICE_TABLE(of, imx_drm_dt_ids);

static struct platform_driver imx_drm_pdrv = {
	.probe		= imx_drm_platform_probe,
	.remove		= imx_drm_platform_remove,
	.driver		= {
		.name	= "imx-drm",
		.pm	= &imx_drm_pm_ops,
		.of_match_table = imx_drm_dt_ids,
	},
};
module_platform_driver(imx_drm_pdrv);

MODULE_AUTHOR("Sascha Hauer <s.hauer@pengutronix.de>");
MODULE_DESCRIPTION("i.MX drm driver core");
MODULE_LICENSE("GPL");
