/* SPDX-License-Identifier: GPL-2.0 */
/******************************************************************************/
/*! @file  ricoh5b_fm_cmd.c
 *  @brief Source File to provide FM Command
 *
 * Copyright 2015, 2016, 2021 Sony Corporation
 *
 *
 */
/****************************************************************************/

/******************************************************************************/
/* include file                                                               */
/******************************************************************************/
#include "ricoh5b_main.h"
#include "ricoh5b_cmd.h"
#include "ricoh5b_fm_cmd.h"

/******************************************************************************/
/* object generation control and special settings                             */
/******************************************************************************/
/******************************************************************************/
/* symbolic constant definition which are refered only this file              */
/******************************************************************************/

/* Debug log setting */
// #define  RICOH5B_FM_CMD_DEBUG  RICOH5B_CONF_DEBUG

#define LOG_ERR(fmt, args...)		pr_err("%s: " fmt, __func__, ##args)
#define LOG_INFO(fmt, args...)		pr_info("%s: " fmt, __func__, ##args)

#ifdef RICOH5B_FM_CMD_DEBUG
#define LOG_DBG(fmt, args...)		pr_info("ricoh5b: " fmt, ##args)
#else  /* RICOH5B_FM_CMD_DEBUG */
#define LOG_DBG(fmt, args...)		{}
#endif /* RICOH5B_FM_CMD_DEBUG */


/* fine Phase/Gain table size */
#define FM_FINE_PG_TBL_SIZE		5

/* AGCDET offset */
#define FM_DET_OFFSET_CORRECTION	2

/* frequency setting reset RFAGC wait time */
#define FM_TUNING_RFAGC_WAIT_TIME	15000	/* ms */

/* frequency setting reset RFAGC wait time */
#define FM_STATION_RFAGC_WAIT_TIME	50000	/* ms */

/*
 * FO_SAT wait time
 * seek ON:	 FO_SAT 42.667ms,  wait 45ms
 * seek OFF: FO_SAT 341.333ms, wait 20ms
 */
#define FM_SEEK_ON_FO_SAT_WAIT_TIME	45000	/* ms */
#define FM_SEEK_OFF_FO_SAT_WAIT_TIME	20000	/* ms */

/* Digital Mixer setting wait */
#define FM_DMIXER_WAIT_TIME		45000	/* ms */

#define FM_IF_ERR_THRESHOLD		20

#define FM_RSSI_THRESHOLD		19	/* dBu */

#define FM_RSSI_MONO_THRESHOLD		28	/* dBu */

#define FM_CHECK_STATION_NUM		6

#define BIRDY_BEAT_MEASURE_FLAG		1
/******************************************************************************/
/* enumerated/structured type definition which are refered only this file     */
/******************************************************************************/

#ifdef BIRDY_BEAT_MEASURE_FLAG
/* FM birdy beat */
enum BIRDY_BEAT_MEASURE {
	BIRDY_BEAT_MEASURE_NONE,
	BIRDY_BEAT_MEASURE_100K,	/* 100kHz step measure */
	BIRDY_BEAT_MEASURE_50K_1,	/* 50kHz step measure  */
	BIRDY_BEAT_MEASURE_50K_2,	/* 50kHz step measure  */
	BIRDY_BEAT_MEASURE_SP		/* 102.75MHz measure   */
};
#endif
/*
 *--------------------------------------------------------
 * sequence
 *--------------------------------------------------------
 */

/* FM BAND SEQUENCE STEP */
enum BAND_STEP {
	BAND_STEP_SETTING_1,
	BAND_STEP_SETTING_2,
	BAND_STEP_SET_AGCDET_OFFSET_AND_ATT,
	BAND_STEP_SETTING_3,
	BAND_STEP_VOLUME,
	BAND_STEP_SET_FREQ,
	BAND_STEP_SET_DITHER,
	BAND_STEP_END
};

/* FREQUENCY SETTING SEQUENCE STEP */
enum FREQ_STEP {
	FREQ_STEP_SET_DMIXER,
	FREQ_STEP_SET_IMMUNITY_MEASURE,
#ifdef BIRDY_BEAT_MEASURE_FLAG
	FREQ_STEP_SET_BIRDY_BEAT_MEASURE,
#endif
	FREQ_STEP_SET_FREQ,
	FREQ_STEP_SET_FREQ_FINE_TUNING,
	FREQ_STEP_RESET_DSP,
	FREQ_STEP_SET_PG_TBL,
	FREQ_STEP_START_RFAGC_RESET,
	FREQ_STEP_WAIT_RFAGC_RESET,
	FREQ_STEP_STOP_RFAGC_RESET,
	FREQ_STEP_END
};

/* DSP ALIGNMENT SEQUENCE */
enum DSP_ALIGN_STEP {
	DSP_ALIGN_STEP_SET_BAND,
	DSP_ALIGN_STEP_SETTING_1,
	DSP_ALIGN_STEP_SET_IMF,
	DSP_ALIGN_STEP_SETTING_2,
	DSP_ALIGN_STEP_SET_IF,
	DSP_ALIGN_STEP_ALIGN_AGCG,
	DSP_ALIGN_STEP_END
};

enum STATION_STEP {
	STATION_STEP_SEEK_START_SETTING,
	STATION_STEP_RESET_RFAGC,
	STATION_STEP_WAIT_RFAGC_RESET,
	STATION_STEP_WAIT_FO_SAT,
	STATION_STEP_GET_FO_SAT,
	STATION_STEP_JUDGE_FO_SAT,
	STATION_STEP_GET_RSSI,
	STATION_STEP_JUDGE_RSSI,
	STATION_STEP_SET_IF400K,
	STATION_STEP_WAIT_DMIXER,
	STATION_STEP_SEEK_END_SETTING,
	STATION_STEP_END
};

/* Audio Info Get Sequence */
enum AUDIO_STEP {
	AUDIO_STEP_GET_STEREO,
	AUDIO_STEP_END
};

/*---------------*/
/* command table */
/*---------------*/
static struct ricoh5b_reg ricoh5b_fm_cmd_freq_rotation_if300k_tbl[] = {
	{ RICOH5B_REG_ADR_IFSEL,            RICOH5B_REG_NON_MASK,
	 RICOH5B_REG_DAT_SYN_DIV_1_1 | RICOH5B_REG_DAT_IFSEL_300K },
	{ RICOH5B_REG_ADR_FREQ_ROTATION_L,  RICOH5B_REG_NON_MASK,
	 RICOH5B_REG_DAT_FREQ_ROTATION_L_FM_IF300K },
	{ RICOH5B_REG_ADR_FREQ_ROTATION_H,  RICOH5B_REG_BIT_FREQ_ROTATION_H,
	 RICOH5B_REG_DAT_FREQ_ROTATION_H_FM_IF300K }
};

static struct ricoh5b_reg ricoh5b_fm_cmd_freq_rotation_if400k_tbl[] = {
	{ RICOH5B_REG_ADR_IFSEL,            RICOH5B_REG_NON_MASK,
	 RICOH5B_REG_DAT_SYN_DIV_1_1 |
	 RICOH5B_REG_DAT_IFSEL_400K },
	{ RICOH5B_REG_ADR_FREQ_ROTATION_L,  RICOH5B_REG_NON_MASK,
	 RICOH5B_REG_DAT_FREQ_ROTATION_L_FM_IF400K },
	{ RICOH5B_REG_ADR_FREQ_ROTATION_H,  RICOH5B_REG_BIT_FREQ_ROTATION_H,
	 RICOH5B_REG_DAT_FREQ_ROTATION_H_FM_IF400K }
};

static struct ricoh5b_reg ricoh5b_fm_cmd_band_step1_tbl[] = {
	{ RICOH5B_REG_ADR_AGC_AUTO,    RICOH5B_REG_BIT_AGC_AUTO,
	 RICOH5B_REG_DAT_AGC_AUTO_ON },
	{ RICOH5B_REG_ADR_BLOCK2_PE,   RICOH5B_REG_BIT_BLOCK2_PE,
	 RICOH5B_REG_DAT_PELO_ON |
	 RICOH5B_REG_DAT_PEFMMIX_ON | RICOH5B_REG_DAT_PEFMLNA_ON  },
	#if RICOH5B_USE_TUNER
	{ RICOH5B_REG_ADR_AMRFSW,      RICOH5B_REG_BIT_AMRFSW,
	 RICOH5B_REG_DAT_AMRFSW_DISABLE },
	{ RICOH5B_REG_ADR_BAND,        RICOH5B_REG_BIT_BAND,
	 RICOH5B_REG_DAT_BAND_FM },
	#endif //RICOH5B_USE_TUNER
	{ RICOH5B_REG_ADR_DCO_OFF,     RICOH5B_REG_BIT_DCO_OFF,
	 RICOH5B_REG_DAT_DCO_ON }
};

static struct ricoh5b_reg ricoh5b_fm_cmd_band_step2_tbl[] = {
	{ RICOH5B_REG_ADR_FMEN_EN,  RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_FMEN_ON | RICOH5B_REG_DAT_96_TEST          },
	{ RICOH5B_REG_ADR_FMLNAS,   RICOH5B_REG_NON_MASK,  RICOH5B_REG_DAT_FMLNAS_DIFFERENTIAL |
	 RICOH5B_REG_DAT_FMLNAI_LOW_POWER  }
};

static struct ricoh5b_reg ricoh5b_fm_cmd_band_step3_tbl[] = {
	{ RICOH5B_REG_ADR_FM_OV,         RICOH5B_REG_BIT_FM_OV | RICOH5B_REG_BIT_BW,  RICOH5B_FM_BW_OM105K            },
	{ RICOH5B_REG_ADR_JAM_POWER,     RICOH5B_REG_NON_MASK,            RICOH5B_REG_DAT_JAM_POWER                   },
	{ RICOH5B_REG_ADR_AD6_HOLDMODE,  RICOH5B_REG_BIT_AD6_GAIN_O | RICOH5B_REG_BIT_AD6_GAIN_I |
	 RICOH5B_REG_BIT_AD6_HOLDMODE,
	RICOH5B_REG_DAT_AD6_GAIN_O_1 | RICOH5B_REG_DAT_AD6_HOLDMODE },
	{ RICOH5B_REG_ADR_HPFS_SEL,      RICOH5B_REG_BIT_HPFS_SEL,        RICOH5B_REG_DAT_HPFS_FM_THROUGH             },
	{ RICOH5B_REG_ADR_COM_SEL,       RICOH5B_REG_BIT_COM_SEL,         RICOH5B_REG_DAT_COM_SEL_300M                }
};

static struct ricoh5b_reg ricoh5b_fm_cmd_immunity_exempt_tbl[] = {
	{ RICOH5B_REG_ADR_LOSCI,          RICOH5B_REG_BIT_LOSCI,     RICOH5B_REG_DAT_LOSCI_UPPER                      },
	{ RICOH5B_REG_ADR_DAGC_SFT,       RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_DAGC_SFT_EN_ON |
	RICOH5B_REG_DAT_DAGC_SFT_MASK_1BIT  },
	{ RICOH5B_REG_ADR_DOWNCONV,       RICOH5B_REG_BIT_DOWNCONV,  RICOH5B_REG_DAT_DOWNCONV_OFF                     },
	{ RICOH5B_REG_ADR_DCCAL_INV,      RICOH5B_REG_BIT_DCCAL_INV, RICOH5B_REG_BIT_DCCAL_INV_SUB                    },
	{ RICOH5B_REG_ADR_FC_SW,          RICOH5B_REG_BIT_FC_SW,     RICOH5B_REG_DAT_FC_SW_1M                         },
	{ RICOH5B_REG_ADR_SNC,            RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_HCC_OFF | RICOH5B_REG_DAT_SNC_OFF },
	{ RICOH5B_REG_ADR_FM_VIFO_INIT,   RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_FM_VIFO_INIT                     },
	{ RICOH5B_REG_ADR_FM_VUD_INIT,    RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_FM_VUD_INIT                      },
	{ RICOH5B_REG_ADR_FM_VIFO_UPPER,  RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_FM_VIFO_UPPER                    },
	{ RICOH5B_REG_ADR_FM_VIFO_LOWER,  RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_FM_VIFO_LOWER                    },
	{ RICOH5B_REG_ADR_FM_VUD_UPPER,   RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_FM_VUD_UPPER                     },
	{ RICOH5B_REG_ADR_FM_VUD_LOWER,   RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_FM_VUD_LOWER                     }
};

static struct ricoh5b_reg ricoh5b_fm_cmd_immunity_87_6M_tbl[] = {
	{ RICOH5B_REG_ADR_LOSCI,          RICOH5B_REG_BIT_LOSCI,     RICOH5B_REG_DAT_LOSCI_UPPER                      },
	{ RICOH5B_REG_ADR_DAGC_SFT,       RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_DAGC_SFT_EN_ON |
	RICOH5B_REG_DAT_DAGC_SFT_MASK_1BIT  },
	{ RICOH5B_REG_ADR_DOWNCONV,       RICOH5B_REG_BIT_DOWNCONV,  RICOH5B_REG_DAT_DOWNCONV_OFF                     },
	{ RICOH5B_REG_ADR_DCCAL_INV,      RICOH5B_REG_BIT_DCCAL_INV, RICOH5B_REG_BIT_DCCAL_INV_SUB                    },
	{ RICOH5B_REG_ADR_FC_SW,          RICOH5B_REG_BIT_FC_SW,     RICOH5B_REG_DAT_FC_SW_500K                       },
	{ RICOH5B_REG_ADR_SNC,            RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_HCC_OFF | RICOH5B_REG_DAT_SNC_47DBU },
	{ RICOH5B_REG_ADR_FM_VIFO_INIT,   RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_FM_VIFO_INIT_IMMUNITY_OUT        },
	{ RICOH5B_REG_ADR_FM_VUD_INIT,    RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_FM_VUD_INIT_IMMUNITY_OUT         },
	{ RICOH5B_REG_ADR_FM_VIFO_UPPER,  RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_FM_VIFO_UPPER_IMMUNITY_OUT       },
	{ RICOH5B_REG_ADR_FM_VIFO_LOWER,  RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_FM_VIFO_LOWER_IMMUNITY_OUT       },
	{ RICOH5B_REG_ADR_FM_VUD_UPPER,   RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_FM_VUD_UPPER_IMMUNITY_OUT        },
	{ RICOH5B_REG_ADR_FM_VUD_LOWER,   RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_FM_VUD_LOWER_IMMUNITY_OUT        }
};

static struct ricoh5b_reg ricoh5b_fm_cmd_immunity_98M_tbl[] = {
	{ RICOH5B_REG_ADR_LOSCI,          RICOH5B_REG_BIT_LOSCI,     RICOH5B_REG_DAT_LOSCI_UPPER                      },
	{ RICOH5B_REG_ADR_DAGC_SFT,       RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_DAGC_SFT_EN_ON |
	RICOH5B_REG_DAT_DAGC_SFT_MASK_1BIT  },
	{ RICOH5B_REG_ADR_DOWNCONV,       RICOH5B_REG_BIT_DOWNCONV,  RICOH5B_REG_DAT_DOWNCONV_OFF                     },
	{ RICOH5B_REG_ADR_DCCAL_INV,      RICOH5B_REG_BIT_DCCAL_INV, RICOH5B_REG_BIT_DCCAL_INV_SUB                    },
	{ RICOH5B_REG_ADR_FC_SW,          RICOH5B_REG_BIT_FC_SW,     RICOH5B_REG_DAT_FC_SW_1M                         },
	{ RICOH5B_REG_ADR_SNC,            RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_HCC_OFF | RICOH5B_REG_DAT_SNC_44DBU },
	{ RICOH5B_REG_ADR_FM_VIFO_INIT,   RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_FM_VIFO_INIT                     },
	{ RICOH5B_REG_ADR_FM_VUD_INIT,    RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_FM_VUD_INIT                      },
	{ RICOH5B_REG_ADR_FM_VIFO_UPPER,  RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_FM_VIFO_UPPER                    },
	{ RICOH5B_REG_ADR_FM_VIFO_LOWER,  RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_FM_VIFO_LOWER                    },
	{ RICOH5B_REG_ADR_FM_VUD_UPPER,   RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_FM_VUD_UPPER                     },
	{ RICOH5B_REG_ADR_FM_VUD_LOWER,   RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_FM_VUD_LOWER                     }
};

static struct ricoh5b_reg ricoh5b_fm_cmd_immunity_107_9M_tbl[] = {
	{ RICOH5B_REG_ADR_LOSCI,          RICOH5B_REG_BIT_LOSCI,     RICOH5B_REG_DAT_LOSCI_LOWER                      },
	{ RICOH5B_REG_ADR_DAGC_SFT,       RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_DAGC_SFT_EN_ON |
	RICOH5B_REG_DAT_DAGC_SFT_MASK_1BIT  },
	{ RICOH5B_REG_ADR_DOWNCONV,       RICOH5B_REG_BIT_DOWNCONV,  RICOH5B_REG_DAT_DOWNCONV_ON                      },
	{ RICOH5B_REG_ADR_DCCAL_INV,      RICOH5B_REG_BIT_DCCAL_INV, RICOH5B_REG_BIT_DCCAL_INV_ADD                    },
	{ RICOH5B_REG_ADR_FC_SW,          RICOH5B_REG_BIT_FC_SW,     RICOH5B_REG_DAT_FC_SW_500K                       },
	{ RICOH5B_REG_ADR_SNC,            RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_HCC_OFF | RICOH5B_REG_DAT_SNC_44DBU },
	{ RICOH5B_REG_ADR_FM_VIFO_INIT,   RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_FM_VIFO_INIT_IMMUNITY_OUT        },
	{ RICOH5B_REG_ADR_FM_VUD_INIT,    RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_FM_VUD_INIT_IMMUNITY_OUT         },
	{ RICOH5B_REG_ADR_FM_VIFO_UPPER,  RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_FM_VIFO_UPPER_IMMUNITY_OUT       },
	{ RICOH5B_REG_ADR_FM_VIFO_LOWER,  RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_FM_VIFO_LOWER_IMMUNITY_OUT       },
	{ RICOH5B_REG_ADR_FM_VUD_UPPER,   RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_FM_VUD_UPPER_IMMUNITY_OUT        },
	{ RICOH5B_REG_ADR_FM_VUD_LOWER,   RICOH5B_REG_NON_MASK,      RICOH5B_REG_DAT_FM_VUD_LOWER_IMMUNITY_OUT        }
};

#ifdef BIRDY_BEAT_MEASURE_FLAG
static struct ricoh5b_reg ricoh5b_fm_cmd_birdy_beat_tbl[] = {
	{ RICOH5B_REG_ADR_LOSCI,          RICOH5B_REG_BIT_LOSCI,      RICOH5B_REG_DAT_LOSCI_LOWER                     },
	{ RICOH5B_REG_ADR_DOWNCONV,       RICOH5B_REG_BIT_DOWNCONV,   RICOH5B_REG_DAT_DOWNCONV_ON                     },
	{ RICOH5B_REG_ADR_DCCAL_INV,      RICOH5B_REG_BIT_DCCAL_INV,  RICOH5B_REG_BIT_DCCAL_INV_ADD                   }
};
#endif

#define RICOH5B_FM_SEND_DSP_ALIGNBODY_STEP1_TBL_SIZE	ARRAY_SIZE(ricoh5b_fm_cmd_dsp_alignbody_step1_tbl)
#define RICOH5B_FM_SEND_DSP_ALIGNBODY_STEP2_TBL_SIZE	ARRAY_SIZE(ricoh5b_fm_cmd_dsp_alignbody_step2_tbl)
#define RICOH5B_FM_SEND_FREQ_ROTATION_IF300K_TBL_SIZE	ARRAY_SIZE(ricoh5b_fm_cmd_freq_rotation_if300k_tbl)
#define RICOH5B_FM_SEND_FREQ_ROTATION_IF400K_TBL_SIZE	ARRAY_SIZE(ricoh5b_fm_cmd_freq_rotation_if400k_tbl)
#define RICOH5B_FM_SEND_BAND_STEP1_TBL_SIZE		ARRAY_SIZE(ricoh5b_fm_cmd_band_step1_tbl)
#define RICOH5B_FM_SEND_BAND_STEP2_TBL_SIZE		ARRAY_SIZE(ricoh5b_fm_cmd_band_step2_tbl)
#define RICOH5B_FM_SEND_BAND_STEP3_TBL_SIZE		ARRAY_SIZE(ricoh5b_fm_cmd_band_step3_tbl)
#define RICOH5B_FM_SEND_IMMUNITY_EXEMPT_TBL_SIZE	ARRAY_SIZE(ricoh5b_fm_cmd_immunity_exempt_tbl)
#define RICOH5B_FM_SEND_IMMUNITY_87_6M_TBL_SIZE		ARRAY_SIZE(ricoh5b_fm_cmd_immunity_87_6M_tbl)
#define RICOH5B_FM_SEND_IMMUNITY_98M_TBL_SIZE		ARRAY_SIZE(ricoh5b_fm_cmd_immunity_98M_tbl)
#define RICOH5B_FM_SEND_IMMUNITY_107_9M_TBL_SIZE	ARRAY_SIZE(ricoh5b_fm_cmd_immunity_107_9M_tbl)

#ifdef BIRDY_BEAT_MEASURE_FLAG
#define RICOH5B_FM_SEND_BIRDY_BEAT_TBL_SIZE		ARRAY_SIZE(ricoh5b_fm_cmd_birdy_beat_tbl)
#endif

static const unsigned short fm_dmixer_setting_tbl[]	= { 7650, 7655, 10210, 10215, 10610, 10615 };
static const unsigned short fm_align_freq_tbl[]		= { 7680, 8777, 10240 };
static const unsigned char  fm_align_imf_tbl[]		= { 0x40, 0x50, 0x60 };
static const unsigned short fm_align_freq_tbl_if400k[]	= { 10240 };
static const unsigned char  fm_align_imf_tbl_if400k[]	= { 0x60 };
static const unsigned short fm_pg_freq_tbl[]		= { 7960, 8500, 9140, 9880 };

#ifdef BIRDY_BEAT_MEASURE_FLAG
static const unsigned short	fm_birdy_beat_tbl[]
	= { 8120,  8150,  8600,  8710,  8750,
	    8760,  8970,  9070,  9090,  9220,
	    9270,  9310,  9380,  9440,  9580,
	    9600,  9620,  9770,  9780,  9850,
	    9910, 10190, 10200, 10250, 10330,
	   10390, 10440, 10470, 10550, 10580,
	   10660, 10680, 10710
};
#endif

#define FM_DMIXER_SETTING_TBL_SIZE	ARRAY_SIZE(fm_dmixer_setting_tbl)
#define FM_ALIGN_FREQ_TBL_SIZE		ARRAY_SIZE(fm_align_freq_tbl)
#define FM_ALIGN_IMF_TBL_SIZE		ARRAY_SIZE(fm_align_imf_tbl)
#define FM_ALIGN_FREQ_TBL_IF400K_SIZE	ARRAY_SIZE(fm_align_freq_tbl_if400k)
#define FM_ALIGN_IMF_TBL_IF400K_SIZE	ARRAY_SIZE(fm_align_imf_tbl_if400k)
#define FM_PG_FREQ_TBL_SIZE		ARRAY_SIZE(fm_pg_freq_tbl)

#ifdef BIRDY_BEAT_MEASURE_FLAG
#define FM_BIRDY_BEAT_TBL_SIZE		ARRAY_SIZE(fm_birdy_beat_tbl)
#endif

/******************************************************************************/
/* global variables definition which are refered only this file		      */
/******************************************************************************/

static unsigned char		fm_volume;
static unsigned short		fm_tuning_freq;
		/* tuning frequency (10kHz step) */
static unsigned short		fm_tuning_freq_psy;
		/* frequency setting	 (100kHz step) */
static unsigned char		fm_tuning_freq_psy_f;
		/* frequency fine tuning (10kHz step) */
static enum RICOH5B_IF_TYPE		fm_tuning_if;
		/* Tuning IF Frequency */
static enum RICOH5B_SEEK_STATUS	fm_seek_status;
		/* seek ON/OFF */
static enum RICOH5B_INIT_STATUS	fm_align_status;
		/* DSP alignment status */
static struct ricoh5b_pg	fm_fine_pg_tbl[FM_FINE_PG_TBL_SIZE];
		/* fine Phase/Gain table */
#ifdef BIRDY_BEAT_MEASURE_FLAG
static enum BIRDY_BEAT_MEASURE	fm_birdy_beat_measure;
		/* birdy beat */
#endif

/******************************************************************************/
/* function prototypes							      */
/******************************************************************************/

#ifdef BIRDY_BEAT_MEASURE_FLAG
static void		fm_select_birdy_beat_measure(void);
#endif
static void		fm_select_if(void);
static int		fm_send_digital_mixer_setting(void);
static int		fm_send_set_immunity_measure(void);
static void		fm_calculate_freq_setting(void);
static int		fm_send_set_pg_tbl(void);
static unsigned char	fm_get_det_offset(void);
static int		fm_set_band_cmd_sequence(void);
static int		fm_set_freq_cmd_sequence(void);

/******************************************************************************/
/*									      */
/*LOCAL FUNCTIONS							      */
/*									      */
/******************************************************************************/

#ifdef BIRDY_BEAT_MEASURE_FLAG
/******************************************************************************/
/*! @brief birdy beat measure
 */
/******************************************************************************/
static void fm_select_birdy_beat_measure(void)
{
	unsigned char i;
	unsigned short freq_tmp;

	fm_birdy_beat_measure = BIRDY_BEAT_MEASURE_NONE;

	if (fm_tuning_freq == FM_BIRDY_BEAT_102_75MHZ) {
		fm_birdy_beat_measure = BIRDY_BEAT_MEASURE_SP;
	} else {
		if (fm_seek_status == RICOH5B_SEEK_ON)
			return;

		/* 100kHz step */
		for (i = 0; i < FM_BIRDY_BEAT_TBL_SIZE; i++) {
			if (fm_tuning_freq == fm_birdy_beat_tbl[i]) {
				fm_birdy_beat_measure = BIRDY_BEAT_MEASURE_100K;
				return;
			}
		}

		/* 50kHz step */
		if (fm_tuning_freq >= FM_BIRDY_BEAT_50KHZ_LOWER) {
			freq_tmp = fm_tuning_freq - FM_BIRDY_BEAT_50KHZ_LOWER;
			freq_tmp %= 40;

			if (freq_tmp == 0) {
				/* 87.55MHz + n * 400KHz */
				fm_birdy_beat_measure =
					BIRDY_BEAT_MEASURE_50K_1;
			} else if (freq_tmp == 10) {
				/* 87.65MHz + n * 400KHz */
				fm_birdy_beat_measure =
					BIRDY_BEAT_MEASURE_50K_2;
			}
		}
	}
}
#endif

/******************************************************************************/
/*! @brief frequency setting
 *		   IF300kHz: Default
 *		   IF400kHz: beat
 *		   IF400kHz: immunity
 */
/******************************************************************************/
static void fm_select_if(void)
{
	unsigned char i;

	fm_tuning_if = RICOH5B_IF_300K;

	if (ricoh5b_cmd_get_immunity_measure() ==
		RICOH5B_IMMUNITY_ENABLE) {	/* PRQA S 3416 */
		if ((fm_tuning_freq == FM_IMMUNITY_87_6MHZ)
			|| (fm_tuning_freq == FM_IMMUNITY_98MHZ)
			|| (fm_tuning_freq == FM_IMMUNITY_107_9MHZ)) {
			fm_tuning_if = RICOH5B_IF_400K;
			return;
		}
	} else if (ricoh5b_cmd_get_immunity_measure() ==
		RICOH5B_IMMUNITY_ENABLE_KR) {	/* PRQA S 3416 */
		if ((fm_tuning_freq == FM_IMMUNITY_88_1MHZ)
			|| (fm_tuning_freq == FM_IMMUNITY_98MHZ)
			|| (fm_tuning_freq == FM_IMMUNITY_107_9MHZ)) {
			fm_tuning_if = RICOH5B_IF_400K;
			return;
		}
	}

#ifdef BIRDY_BEAT_MEASURE_FLAG
	if (fm_birdy_beat_measure != BIRDY_BEAT_MEASURE_NONE) {
		fm_tuning_if = RICOH5B_IF_400K;
		return;
	}
#endif

	if (fm_seek_status == RICOH5B_SEEK_OFF) {
		for (i = 0; i < FM_DMIXER_SETTING_TBL_SIZE; i++) {
			if (fm_dmixer_setting_tbl[i] == fm_tuning_freq) {
				fm_tuning_if = RICOH5B_IF_400K;
				break;
			}
		}
	}
}

/******************************************************************************/
/*! @brief Digital Mixer
 */
/******************************************************************************/
static int fm_send_digital_mixer_setting(void)
{
	int ret = 0;

	switch (fm_tuning_if) {
	case RICOH5B_IF_300K:
		ret = ricoh5b_cmd_write_sequencial(
			&ricoh5b_fm_cmd_freq_rotation_if300k_tbl[0],
			RICOH5B_FM_SEND_FREQ_ROTATION_IF300K_TBL_SIZE);
		break;
	case RICOH5B_IF_400K:
		ret = ricoh5b_cmd_write_sequencial(
			&ricoh5b_fm_cmd_freq_rotation_if400k_tbl[0],
			RICOH5B_FM_SEND_FREQ_ROTATION_IF400K_TBL_SIZE);
		break;
	default:
		break;
	}

	return ret;
}

/******************************************************************************/
/*! @brief immunity test measure
 */
/******************************************************************************/
static int fm_send_set_immunity_measure(void)
{
	int ret = 0;

	if (ricoh5b_cmd_get_immunity_measure() ==
		RICOH5B_IMMUNITY_ENABLE) {	/* PRQA S 3416 */
		switch (fm_tuning_freq) {
		case FM_IMMUNITY_87_6MHZ:
			ret = ricoh5b_cmd_write_sequencial(
				&ricoh5b_fm_cmd_immunity_87_6M_tbl[0],
				RICOH5B_FM_SEND_IMMUNITY_87_6M_TBL_SIZE);
			break;
		case FM_IMMUNITY_98MHZ:
			ret = ricoh5b_cmd_write_sequencial(
				&ricoh5b_fm_cmd_immunity_98M_tbl[0],
				RICOH5B_FM_SEND_IMMUNITY_98M_TBL_SIZE);
			break;
		case FM_IMMUNITY_107_9MHZ:
			ret = ricoh5b_cmd_write_sequencial(
				&ricoh5b_fm_cmd_immunity_107_9M_tbl[0],
				RICOH5B_FM_SEND_IMMUNITY_107_9M_TBL_SIZE);
			break;
		default:
			ret = ricoh5b_cmd_write_sequencial(
				&ricoh5b_fm_cmd_immunity_exempt_tbl[0],
				RICOH5B_FM_SEND_IMMUNITY_EXEMPT_TBL_SIZE);
			break;
		}
	} else if (ricoh5b_cmd_get_immunity_measure() ==
		RICOH5B_IMMUNITY_ENABLE_KR) {	 /* PRQA S 3416 */
		switch (fm_tuning_freq) {
		case FM_IMMUNITY_88_1MHZ:
			ret = ricoh5b_cmd_write_sequencial(
				&ricoh5b_fm_cmd_immunity_87_6M_tbl[0],
				RICOH5B_FM_SEND_IMMUNITY_87_6M_TBL_SIZE);
			break;
		case FM_IMMUNITY_98MHZ:
			ret = ricoh5b_cmd_write_sequencial(
				&ricoh5b_fm_cmd_immunity_98M_tbl[0],
				RICOH5B_FM_SEND_IMMUNITY_98M_TBL_SIZE);
			break;
		case FM_IMMUNITY_107_9MHZ:
			ret = ricoh5b_cmd_write_sequencial(
				&ricoh5b_fm_cmd_immunity_107_9M_tbl[0],
				RICOH5B_FM_SEND_IMMUNITY_107_9M_TBL_SIZE);
			break;
		default:
			ret = ricoh5b_cmd_write_sequencial(
				&ricoh5b_fm_cmd_immunity_exempt_tbl[0],
				RICOH5B_FM_SEND_IMMUNITY_EXEMPT_TBL_SIZE);
			break;
		}
	} else {
		ret = ricoh5b_cmd_write_sequencial(
			&ricoh5b_fm_cmd_immunity_exempt_tbl[0],
			RICOH5B_FM_SEND_IMMUNITY_EXEMPT_TBL_SIZE);
	}
	return ret;
}

/******************************************************************************/
/*! @brief frequency calculate setting
 */
/******************************************************************************/
static void fm_calculate_freq_setting(void)
{
	LOG_DBG("fm_tuning_freq=%d\n", fm_tuning_freq);
	fm_tuning_freq_psy	 = fm_tuning_freq / 10;	/* 100kHZ step */

	/* 10kHz step fine tuning
	 * under 50kHz or over 50kHz
	 */
	fm_tuning_freq_psy_f =
		(unsigned char)(fm_tuning_freq % 10);  /* 10kHZ step */

	if (fm_tuning_freq_psy_f != 0) {
		if (fm_tuning_freq_psy_f <= FM_50KHZ) {
			fm_tuning_freq_psy_f =
				RICOH5B_REG_DAT_PSY_F_CONVERT_MINUS(
					fm_tuning_freq_psy_f);
		} else {
			fm_tuning_freq_psy_f = 10 - fm_tuning_freq_psy_f;
			fm_tuning_freq_psy++;
		}
	}

	/* Reference Frequency 400kHz double */
	fm_tuning_freq_psy <<= 1;

#ifdef BIRDY_BEAT_MEASURE_FLAG
	/* birdy beat measure */
	switch (fm_birdy_beat_measure) {
	case BIRDY_BEAT_MEASURE_100K:
		fm_tuning_freq_psy_f = RICOH5B_REG_DAT_PSY_F_PLUS_50K;
		fm_tuning_freq_psy++;				 /* +50kHz */
		break;
	case BIRDY_BEAT_MEASURE_50K_1:
		/* LO -350kHz */
		fm_tuning_freq_psy += 2;			/* +100kHz */
		break;
	case BIRDY_BEAT_MEASURE_50K_2:
		if ((fm_tuning_freq == FM_BIRDY_BEAT_89_65MHZ) ||
			(fm_tuning_freq == FM_BIRDY_BEAT_102_05MHZ)) {
			/* LO +450kHz */
			fm_tuning_freq_psy_f = RICOH5B_REG_DAT_PSY_F_PLUS_50K;
			fm_tuning_freq_psy += 2;		/* +100kHz */
		} else {
			/* LO +350kHz */
			/* do nothing */
		}
		break;
	case BIRDY_BEAT_MEASURE_SP:
		fm_tuning_freq_psy_f = RICOH5B_REG_DAT_PSY_F_PLUS_50K;
		break;
	default:
		break;
	}
#endif

}

/******************************************************************************/
/*! @brief set Phase/Gain correction command
 */
/******************************************************************************/
static int fm_send_set_pg_tbl(void)
{
	unsigned char i;
	int ret = 0;

	for (i = 0; i < FM_PG_FREQ_TBL_SIZE; i++) {
		if (fm_tuning_freq < fm_pg_freq_tbl[i])
			break;
	}
	ret = ricoh5b_send_set_pg_tbl(&fm_fine_pg_tbl[i]);

	return ret;
}

/******************************************************************************/
/*! @brief AGCDET offset setting
 */
/******************************************************************************/
static unsigned char fm_get_det_offset(void)
{
	unsigned char offset;

	offset = ricoh5b_common_cmd_get_det_offset();
	offset &= RICOH5B_REG_BIT_DET_DC_ADJ;

	if ((offset & 0x08) == 0x00) {
		if (offset >= FM_DET_OFFSET_CORRECTION)
			offset -= FM_DET_OFFSET_CORRECTION;
		else
			offset -= (FM_DET_OFFSET_CORRECTION + 1);
							/* PRQA S 3372 */
		offset &= RICOH5B_REG_BIT_DET_DC_ADJ;
	} else {
		offset = ((unsigned char)(offset - FM_DET_OFFSET_CORRECTION)) &
			RICOH5B_REG_BIT_DET_DC_ADJ;
		if (offset <= RICOH5B_REG_DAT_DET_DC_ADJ_PLUS_7)
			offset = RICOH5B_REG_DAT_DET_DC_ADJ_MINUS_7;
	}

	return offset;
}

/******************************************************************************/
/*! @brief set FM band and frequency command sequence
 */
/******************************************************************************/
static int fm_set_band_cmd_sequence(void)
{
	unsigned char volume_data[RICOH5B_VOLUME_DATA_SIZE];
	int step;
	int ret = 0;

	for (step = 0; step < BAND_STEP_END; step++) {
		switch (step) {
		case BAND_STEP_SETTING_1:
			ret = ricoh5b_cmd_write_sequencial(
				&ricoh5b_fm_cmd_band_step1_tbl[0],
				RICOH5B_FM_SEND_BAND_STEP1_TBL_SIZE);
			if (ret < 0) {
				LOG_ERR("FM_SEND_BAND_STEP1_TBL err=%d\n",
					ret);
				goto send_err;
			}
			break;
		case BAND_STEP_SETTING_2:
			ret = ricoh5b_cmd_write_sequencial(
				&ricoh5b_fm_cmd_band_step2_tbl[0],
				RICOH5B_FM_SEND_BAND_STEP2_TBL_SIZE);
			if (ret < 0) {
				LOG_ERR("FM_SEND_BAND_STEP2_TBL err=%d\n",
					ret);
				goto send_err;
			}
			break;
		case BAND_STEP_SET_AGCDET_OFFSET_AND_ATT:
			ret = ricoh5b_cmd_write_bit(
				RICOH5B_REG_ADR_DET_DC_ADJ, /* PRQA S 3441 */
				RICOH5B_REG_DAT_DET_AFM_FM +
					fm_get_det_offset(),
				RICOH5B_REG_BIT_DET_AFM |
					RICOH5B_REG_BIT_DET_DC_ADJ);
			if (ret < 0) {
				LOG_ERR("BAND_STEP_SET_AGCDET err=%d\n",
					ret);
				goto send_err;
			}
			break;
		case BAND_STEP_SETTING_3:
			ret = ricoh5b_cmd_write_sequencial(
				&ricoh5b_fm_cmd_band_step3_tbl[0],
				RICOH5B_FM_SEND_BAND_STEP3_TBL_SIZE);
			if (ret < 0) {
				LOG_ERR("FM_SEND_BAND_STEP3_TBL err=%d\n",
					ret);
				goto send_err;
			}
			break;
		case BAND_STEP_VOLUME:
			volume_data[0] = fm_volume;
			volume_data[1] = fm_volume;
			ret = ricoh5b_cmd_write_byte(
				RICOH5B_REG_ADR_HPLVOL, &volume_data[0],
				RICOH5B_REG_ADDRESS_LENGTH +
					RICOH5B_REG_LENGTH_VOL);
			if (ret < 0) {
				LOG_ERR("BAND_STEP_VOLUME err = %d\n",
					ret);
				goto send_err;
			}
			break;
		case BAND_STEP_SET_FREQ:
			ret = ricoh5b_fm_send_set_freq(fm_tuning_freq,
				RICOH5B_SEEK_OFF);
			if (ret < 0) {
				LOG_ERR("BAND_STEP_VOLUME err = %d\n",
					ret);
				goto send_err;
			}
			break;
		case BAND_STEP_SET_DITHER:
			ret = ricoh5b_send_dither_setting(RICOH5B_BAND_FM,
				RICOH5B_SELECT_AUDIO_OUTPUT);
			if (ret < 0) {
				LOG_ERR("BAND_STEP_SET_DITHER err = %d\n",
					ret);
				goto send_err;
			}
			break;
		default:
			break;
		}
	}
	return 0;

send_err:
	return ret;
}

/******************************************************************************/
/*! @brief set frequency command sequence
 */
/******************************************************************************/
static int fm_set_freq_cmd_sequence(void)
{
	unsigned char freq_data;
	int step;
	int ret = 0;

	for (step = 0; step < FREQ_STEP_END; step++) {
		switch (step) {
		case FREQ_STEP_SET_DMIXER:
#ifdef BIRDY_BEAT_MEASURE_FLAG
			fm_select_birdy_beat_measure();
#endif
			fm_select_if();
			ret = fm_send_digital_mixer_setting();
			if (ret < 0) {
				LOG_ERR("FREQ_STEP_SET_DMIXER err = %d\n",
					ret);
				goto send_err;
			}
			break;
		case FREQ_STEP_SET_IMMUNITY_MEASURE:
			ret = fm_send_set_immunity_measure();
			if (ret < 0) {
				LOG_ERR("FREQ_STEP_SET_IMMUNITY err=%d\n",
					ret);
				goto send_err;
			}
			break;
#ifdef BIRDY_BEAT_MEASURE_FLAG
		case FREQ_STEP_SET_BIRDY_BEAT_MEASURE:
			if ((fm_birdy_beat_measure == BIRDY_BEAT_MEASURE_SP)
				|| (fm_birdy_beat_measure ==
					BIRDY_BEAT_MEASURE_50K_1)) {
				ret = ricoh5b_cmd_write_sequencial(
					&ricoh5b_fm_cmd_birdy_beat_tbl[0],
					RICOH5B_FM_SEND_BIRDY_BEAT_TBL_SIZE);
				if (ret < 0) {
					LOG_ERR(
					  "FM_SEND_BIRDY_BEAT_TBL err=%d\n",
					  ret);
					goto send_err;
				}
			}
			break;
#endif
		case FREQ_STEP_SET_FREQ:
			fm_calculate_freq_setting();
			LOG_DBG("fm_tuning_freq=%d\n", fm_tuning_freq_psy);
			freq_data = RICOH5B_GET_LOWER_BYTE(fm_tuning_freq_psy);
			ret = ricoh5b_cmd_write_bit(RICOH5B_REG_ADR_PSY_L,
				freq_data, RICOH5B_REG_NON_MASK);
			if (ret < 0) {
				LOG_ERR("REG_ADR_PSY_L err = %d\n", ret);
				goto send_err;
			}

			freq_data = RICOH5B_GET_UPPER_BYTE(fm_tuning_freq_psy);
			ret = ricoh5b_cmd_write_bit(RICOH5B_REG_ADR_PSY_H,
				freq_data, RICOH5B_REG_NON_MASK);
			if (ret < 0) {
				LOG_ERR("REG_ADR_PSY_H err = %d\n", ret);
				goto send_err;
			}
			break;
		case FREQ_STEP_SET_FREQ_FINE_TUNING:
			ret = ricoh5b_cmd_write_bit(RICOH5B_REG_ADR_PSY_F,
				fm_tuning_freq_psy_f, RICOH5B_REG_NON_MASK);
			if (ret < 0) {
				LOG_ERR("REG_ADR_PSY_F err = %d\n", ret);
				goto send_err;
			}
			break;
		case FREQ_STEP_RESET_DSP:
			ret = ricoh5b_cmd_write_bit(RICOH5B_REG_ADR_SRST,
				RICOH5B_REG_DAT_SRST_DSP, RICOH5B_REG_NON_MASK);
			if (ret < 0) {
				LOG_ERR("REG_ADR_SRST err = %d\n", ret);
				goto send_err;
			}

			if (fm_align_status == RICOH5B_INIT_DONE)
				;
			else
				step = FREQ_STEP_START_RFAGC_RESET - 1;
			break;
		case FREQ_STEP_SET_PG_TBL:
			ret = fm_send_set_pg_tbl();
			if (ret < 0) {
				LOG_ERR("FREQ_STEP_SET_PG_TBL err = %d\n",
					ret);
				goto send_err;
			}
			break;
		case FREQ_STEP_START_RFAGC_RESET:
			ret = ricoh5b_cmd_write_bit(RICOH5B_REG_ADR_RST_RFAGC,
				RICOH5B_REG_DAT_RST_RFAGC_ENABLE,
				RICOH5B_REG_BIT_RST_RFAGC);
			if (ret < 0) {
				LOG_ERR("FREQ_STEP_START_RFAGC_RESET err=%d\n",
					ret);
				goto send_err;
			}
			break;
		case FREQ_STEP_WAIT_RFAGC_RESET:
			ricoh5b_cmd_set_sequence_timer(
				FM_TUNING_RFAGC_WAIT_TIME);
			break;
		case FREQ_STEP_STOP_RFAGC_RESET:
			ret = ricoh5b_cmd_write_bit(RICOH5B_REG_ADR_RST_RFAGC,
				RICOH5B_REG_DAT_RST_RFAGC_DISABLE,
				RICOH5B_REG_BIT_RST_RFAGC);
			if (ret < 0) {
				LOG_ERR("FREQ_STEP_STOP_RFAGC_RESET err=%d\n",
					ret);
				goto send_err;
			}
			break;
		default:
			break;
		}
	}
	return 0;

send_err:
	return ret;
}

/******************************************************************************/
/*									      */
/*GLOBAL FUNCTIONS							      */
/*									      */
/******************************************************************************/

/******************************************************************************/
/*! @brief set FM band and frequency command
 */
/******************************************************************************/
int ricoh5b_fm_send_set_band(unsigned short freq, enum RICOH5B_VOLUME gain)
{
	int ret = 0;

	switch (gain) {
	case RICOH5B_VOLUME_0DB:
		fm_volume = RICOH5B_REG_DAT_HPVOL_FM_0DB;
		break;
	case RICOH5B_VOLUME_MINUS_3DB:
		fm_volume = RICOH5B_REG_DAT_HPVOL_FM_MINUS_3DB;
		break;
	case RICOH5B_VOLUME_MINUS_4DB:
		fm_volume = RICOH5B_REG_DAT_HPVOL_FM_MINUS_4DB;
		break;
	case RICOH5B_VOLUME_MINUS_5DB:
		fm_volume = RICOH5B_REG_DAT_HPVOL_FM_MINUS_5DB;
		break;
	default:
		fm_volume = RICOH5B_REG_DAT_HPVOL_FM_0DB;
		break;
	}

	fm_tuning_freq = freq;
	ret = fm_set_band_cmd_sequence();

	return ret;
}

/******************************************************************************/
/*! @brief set FM frequency command
 */
/******************************************************************************/
int ricoh5b_fm_send_set_freq(unsigned short freq, enum RICOH5B_SEEK_STATUS seek)
{
	int ret = 0;

	fm_tuning_freq = freq;
	fm_seek_status = seek;
	ret = fm_set_freq_cmd_sequence();

	return ret;
}

unsigned short fm_calculate_freq_val_set(unsigned short freq_val)
{
	fm_tuning_freq = freq_val;

	fm_calculate_freq_setting();

	return fm_tuning_freq_psy;
}
/* end of ricoh5b_fm_cmd.c */
