/*
 * Copyright (C) 2018 MediaTek Inc.
 * Copyright 2022 Sony Corporation
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 */

#include <linux/interrupt.h>
#include <linux/module.h>
#include <linux/of.h>
#include <linux/of_irq.h>
#include <linux/platform_device.h>
#include <linux/slab.h>
#include <linux/workqueue.h>
#include <linux/gpio.h>
#include <hifi4dsp_load/hifi4dsp_load.h>

#define HIFI4DSP_RESET

#define DRV_NAME		"mtk-dsp_wdt"

struct mtk_dsp_wdt_dev {
	void __iomem *dsp_wdt_base;
	unsigned int dsp_wdt_irq_id;
};

static struct workqueue_struct *dsp_wdt_queue;
static struct work_struct dsp_wdt_work;

#ifdef HIFI4DSP_RESET
static atomic_t is_from_suspend;
#endif

static irqreturn_t mtk_dsp_wdt_isr(int irq, void *dev_id)
{
	/*
	u32 val;
	
	val = gpio_get_value(509);
	if(val != 1)
		return IRQ_HANDLED;
	*/
#ifdef HIFI4DSP_RESET
	if (atomic_read(&is_from_suspend))
		return IRQ_HANDLED;
#endif
	queue_work(dsp_wdt_queue, &dsp_wdt_work);

	return IRQ_HANDLED;
}

void dsp_wdt_work_handler(struct work_struct *unused)
{
	u32 tmp;
	char *log_buf = NULL;
	u32 log_size = 0;
	int ret = 0;
	int i;
	char tmp_buf[512];
	u32 tmp_log_start, tmp_log_size;

	/* disable watchdog */
	spi_read_register(0x1D010000, &tmp, SPI_SPEED_HIGH);
	tmp &= ~(1 << 0);
	tmp |= (0x22000000);
	spi_write_register(0x1D010000, tmp, SPI_SPEED_HIGH);

	spi_read_register(0x1D010000, &tmp, SPI_SPEED_HIGH);

	pr_notice("%s, watchdog mode:0x%x\n", __func__, tmp);

	ret = hifi4dsp_get_log_buf_size(&log_size);
	if (ret) {
		pr_info("Dump DSP log fail, ret=%d\n", ret);
		goto _end;
	}

	log_buf = kzalloc(log_size, GFP_KERNEL);
	if (!log_buf) {
		ret = -ENOMEM;
		goto _end;
	}

	ret = hifi4dsp_get_log_buf(log_buf, log_size);
	if (ret) {
		pr_info("Read log_buf fail, ret=%d\n", ret);
		goto _end;
	}

	pr_info("log_buf size: %d\n", log_size);

	pr_info("==========================\n");
	pr_info("=== dump dsp log start ===\n");
	pr_info("==========================\n");
	for (i = 0, tmp_log_start = 0; i < log_size; i++) {
		if (log_buf[i] == '\n' || i == log_size - 1) {
			tmp_log_size = i - tmp_log_start + 1;
			strncpy(tmp_buf, log_buf + tmp_log_start, tmp_log_size);
			tmp_buf[tmp_log_size] = '\0';
			pr_info("%s", tmp_buf);
			tmp_log_start = i + 1;
		}

	}
	pr_info("==========================\n");
	pr_info("=== dump dsp log end  ====\n");
	pr_info("==========================\n");
_end:
	kfree(log_buf);
	WARN_ON(1);
}

static int mtk_dsp_wdt_probe(struct platform_device *pdev)
{
	int err;
	struct mtk_dsp_wdt_dev *mtk_dsp_wdt;

	mtk_dsp_wdt = devm_kzalloc(&pdev->dev,
				sizeof(*mtk_dsp_wdt), GFP_KERNEL);
	if (!mtk_dsp_wdt)
		return -ENOMEM;

	mtk_dsp_wdt->dsp_wdt_irq_id =
				irq_of_parse_and_map(pdev->dev.of_node, 0);
	if (mtk_dsp_wdt->dsp_wdt_irq_id == 0U) {
		pr_notice("dsp wdt get IRQ ID failed\n");
		return -ENODEV;
	}
	
	/*pr_info(" mtk_dsp_wdt->dsp_wdt_irq_id : %d \n",mtk_dsp_wdt->dsp_wdt_irq_id);*/
	
	err = request_irq(mtk_dsp_wdt->dsp_wdt_irq_id, mtk_dsp_wdt_isr,
		IRQF_TRIGGER_NONE, DRV_NAME, mtk_dsp_wdt);
	
	/*
	err = request_irq(mtk_dsp_wdt->dsp_wdt_irq_id, mtk_dsp_wdt_isr,
		IRQF_TRIGGER_NONE | IRQF_SHARED, DRV_NAME, mtk_dsp_wdt);
		*/
	if (err != 0) {
		pr_notice("%s: failed to request irq (%d)\n", __func__, err);
		return err;
	}

	dsp_wdt_queue = create_singlethread_workqueue("dsp_wdt_kworker");
	INIT_WORK(&dsp_wdt_work, dsp_wdt_work_handler);
	pr_info("%s: completed \n", __func__);
	return 0;
}

static const struct of_device_id mtk_dsp_wdt_dt_ids[] = {
	{ .compatible = "mediatek,mt8570-dsp_wdt" },
	{ /* sentinel */ }
};

MODULE_DEVICE_TABLE(of, mtk_dsp_wdt_dt_ids);

static int mtk_dsp_wdt_pm_resume(struct device *device)
{
#ifdef HIFI4DSP_RESET
	pr_notice("mtk_dsp_wdt_pm_resume is resume!\n");
	atomic_set(&is_from_suspend, 0);
#endif
	return 0;
}

static int mtk_dsp_wdt_pm_suspend(struct device *device)
{
#ifdef HIFI4DSP_RESET
	pr_notice("mtk_dsp_wdt_pm_resume is suspend, set is_from_suspend to 1!\n");
	atomic_set(&is_from_suspend, 1);
	cancel_work_sync(&dsp_wdt_work);
#endif
	return 0;
}

struct dev_pm_ops const mtk_dsp_wdt_pm_ops = {
	.suspend = mtk_dsp_wdt_pm_suspend,
	.resume = mtk_dsp_wdt_pm_resume,
};

static struct platform_driver mtk_dsp_wdt_driver = {
	.probe		= mtk_dsp_wdt_probe,
	.driver		= {
		.name		= DRV_NAME,
		.of_match_table	= mtk_dsp_wdt_dt_ids,
		.pm = &mtk_dsp_wdt_pm_ops,
	},
};

module_platform_driver(mtk_dsp_wdt_driver);

