/* SPDX-License-Identifier: GPL-2.0 */
/*
 * rs232c_cm4.c - rs232c driver
 *
 * Copyright 2021, 2022 Sony Corporation
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/virtio.h>
#include <linux/rpmsg.h>
#include <linux/err.h>
#include <linux/cdev.h>
#include <linux/platform_device.h>
#include <linux/cm4.h>
#include <linux/kthread.h>
#include <misc/rs232c_cm4.h>

#define DRV_NAME		"rs232c_cm4"
#define ERRLOG(fmt, ...)	pr_err(DRV_NAME ": Error: " fmt, ## __VA_ARGS__)
#define WRNLOG(fmt, ...)	pr_warn(DRV_NAME ": Warning: " fmt, ## __VA_ARGS__)
#define MSGLOG(fmt, ...)	pr_info(DRV_NAME ": " fmt, ## __VA_ARGS__)
#define DBGLOG(fmt, ...)	pr_debug(DRV_NAME ": %s(%d): " fmt, __func__, __LINE__, ## __VA_ARGS__)


/*******************************************************************************
 * Definition
*******************************************************************************/
#define RS232C_SEND_DATA	(0x01)
#define RS232C_BAUD_REQ		(0x02)
#define RS232C_ENABLE_REQ	(0x21)
#define RS232C_FORMAT_REQ	(0x31)
#define RS232C_TESTCMD_REQ	(0x71)
#define RS232C_HOST_SND_RESP	(0x81)
#define RS232C_BAUDRATE_RESP	(0x82)
#define RS232C_HOST_RCV_NTF	(0x91)

#define RS232C_DRV_POWER_NORMAL		(0)
#define RS232C_DRV_POWER_SUSPEND	(1)

struct rs232c_cm4_mng_info_priv {
	dev_t			major;
	struct cdev		cdev;
	struct class		*class;
	struct device		*dev;
	struct device		*idev;
	struct task_struct	*kthread_tsk;
	struct mutex		tx_state_lock;
	struct mutex		rx_state_lock;
	wait_queue_head_t	rxwait;
	u8			mode_enable;
	u8			mode_baudrate;
	u8			mode_format;
};

static struct rs232c_cm4_mng_info_priv *rs232c_cm4_drvdata;
static int rs232c_drv_power = RS232C_DRV_POWER_NORMAL;

/* Tx/Rx ring buffer */
#define RS232C_RING_BUFF_SIZE	(2048)
#define RS232C_RW_DATA_SIZE	(255)
struct rs232c_core_ring_buffer {
	u8	buffer[RS232C_RING_BUFF_SIZE];
	int	rptr;		/* Read pointer for ring buffer */
	int	wptr;		/* Write pointer for ring buffer */
};

static struct rs232c_core_ring_buffer rs232c_tx_ring;
static struct rs232c_core_ring_buffer rs232c_rx_ring;
static bool rs232c_tx_wait_data_sts;
static bool rs232c_rx_wait_data_sts;

/* baudrate change table */
const uint32_t rs232c_core_BaudRate[8] = {
	9600,			/*   9600bps:0 */
	19200,			/*  19200bps:1 */
	38400,			/*  38400bps:2 */
	57600,			/*  57600bps:3 */
	115200,			/* 115200bps:4 */
	230400,			/* 230400bps:5 */
	460800,			/* 460800bps:6 */
	921600,			/* 921600bps:7 */
};

/*******************************************************************************
 * Variables
*******************************************************************************/

/*******************************************************************************
 * Functions
*******************************************************************************/
static bool rs232c_chk_wait_data(int wptr, int rptr)
{
	bool ret;

	if (wptr != rptr)
		ret = true;
	else
		ret = false;

	return ret;
}

/*--- Access for Rx Ring buffer -------------------------------------------*/
/* Read from ring buffer */
static int rs232c_rx_ring_read(char __user *buf, int count)
{
	int wptr_temp, rptr_temp;
	int sz_total, sz;
	int rest_cap;

	wptr_temp = rs232c_rx_ring.wptr;
	rptr_temp = rs232c_rx_ring.rptr;
	sz_total = rs232c_rx_ring.buffer[rptr_temp];	/*1st byte=size*/
	sz = sz_total;
	rest_cap = RS232C_RING_BUFF_SIZE - rptr_temp;

	if (rptr_temp == wptr_temp)
		return 0;
	if ((sz <= 0) || (sz > RS232C_RW_DATA_SIZE)) {
		ERRLOG("pointer size error! sz:%d buf:0x%p\n",
					sz, buf);
		return RS232C_CM4_NG;
	}
	if (count < sz) {
		ERRLOG("RX_BUFF_SIZE_ERROR sz:%d count:%d\n", sz, count);
		return -RS232C_CM4_RX_BUFF_SIZE_ERROR;
	}
	if (rptr_temp >= RS232C_RING_BUFF_SIZE) { /* Pointer inside check */
		ERRLOG("rx_ring.rptr error! %d\n", wptr_temp);
		return RS232C_CM4_NG;
	}

	/* Read from ring buffer Rx */
	if ((sz + 1) >= rest_cap) {
		if (rptr_temp < wptr_temp) {
			ERRLOG("err! r:%d w:%d\n", rptr_temp, wptr_temp);
			return RS232C_CM4_NG;
		}
		rptr_temp++;
		if (copy_to_user(buf, &rs232c_rx_ring.buffer[rptr_temp],
						rest_cap-1)) {
			ERRLOG("RX copy_to_user\n");
			return -EFAULT;
		}
		rptr_temp = 0;
		sz -= (rest_cap - 1);
		buf = (char *)(buf + (rest_cap - 1));
		if (copy_to_user(buf, &rs232c_rx_ring.buffer[rptr_temp], sz)) {
			ERRLOG("RX copy_to_user\n");
			return -EFAULT;
		}
	} else {
		rptr_temp++;
		if (copy_to_user(buf, &rs232c_rx_ring.buffer[rptr_temp], sz)) {
			ERRLOG("RX copy_to_user\n");
			return -EFAULT;
		}
	}
	/* Read Pointer increment */
	barrier();
	rs232c_rx_ring.rptr = rptr_temp + sz;

	rs232c_rx_wait_data_sts = rs232c_chk_wait_data(rs232c_rx_ring.wptr,
						rs232c_rx_ring.rptr);

	return sz_total;
}

/* Write to ring buffer */
static int rs232c_rx_ring_write(char *buf, int count)
{
	int rptr_temp, wptr_temp;
	int rest_cap, rest;
	int sz;

	rptr_temp = rs232c_rx_ring.rptr;
	wptr_temp = rs232c_rx_ring.wptr;
	rest_cap  = RS232C_RING_BUFF_SIZE - wptr_temp;
	sz        = count;
	rest      = 0;

	if ((sz <= 0) || (sz > RS232C_RW_DATA_SIZE)) {
		ERRLOG("pointer size error! count:%d buf:0x%p\n",
					sz, buf);
		return RS232C_CM4_NG;
	}
	if (wptr_temp >= RS232C_RING_BUFF_SIZE) { /* Pointer inside check */
		ERRLOG("rx_ring.wptr error! %d\n", wptr_temp);
		return RS232C_CM4_NG;
	}

	/* rest writable buffer size */
	if (rptr_temp <= wptr_temp)
		rest = RS232C_RING_BUFF_SIZE - (wptr_temp - rptr_temp);
	else
		rest = rptr_temp - wptr_temp;
	/* Flow control */
	if (rest < (RS232C_RW_DATA_SIZE + 1)) {
		ERRLOG("BUFF_FULL! size:%d rest:%d\n", sz, rest);
		return -RS232C_CM4_BUFF_FULL;
	}
	/* Write to ring buffer Tx */
	if ((sz + 1) >= rest_cap) {
		/* Write to end edge */
		rs232c_rx_ring.buffer[wptr_temp] = sz;
		wptr_temp++;
		memcpy(&rs232c_rx_ring.buffer[wptr_temp], buf, rest_cap - 1);
		wptr_temp = 0;
		sz -= (rest_cap - 1);
		buf = (char *)(buf + (rest_cap - 1));
		memcpy(&rs232c_rx_ring.buffer[wptr_temp], buf, sz);
	} else {
		rs232c_rx_ring.buffer[wptr_temp] = sz;
		wptr_temp++;
		memcpy(&rs232c_rx_ring.buffer[wptr_temp], buf, sz);
	}
	/* Write Pointer increment */
	barrier();
	rs232c_rx_ring.wptr = wptr_temp + sz;

	rs232c_rx_wait_data_sts = rs232c_chk_wait_data(rs232c_rx_ring.wptr,
						rs232c_rx_ring.rptr);

	return count;
}

/*--- Access for Tx Ring buffer -------------------------------------------*/

/* Read from ring buffer */
static int rs232c_tx_ring_read(char *buf, int count)
{
	int wptr_temp, rptr_temp;
	int sz_total, sz;
	int rest_cap;

	wptr_temp = rs232c_tx_ring.wptr;
	rptr_temp = rs232c_tx_ring.rptr;
	sz_total  = rs232c_tx_ring.buffer[rptr_temp];	/*1st byte=size*/
	sz        = sz_total;
	rest_cap  = RS232C_RING_BUFF_SIZE - rptr_temp;

	if (rptr_temp == wptr_temp)
		return 0;
	if ((sz <= 0) || (sz > RS232C_RW_DATA_SIZE)) {
		ERRLOG("pointer size error! sz:%d buf:0x%p\n",
					sz, buf);
		return RS232C_CM4_NG;
	}
	if (sz > RS232C_RW_DATA_SIZE)
		return 0;	/* Not enter this frame */
	if (count < sz) {
		/* Don't fit in this frame */
		return -RS232C_CM4_RX_BUFF_SIZE_ERROR;
	}
	if (rptr_temp >= RS232C_RING_BUFF_SIZE) { /* Pointer inside check */
		ERRLOG("tx_ring.rptr error! %d\n", rptr_temp);
		return RS232C_CM4_NG;
	}

	/* Read from ring buffer Rx */
	if ((sz + 1) >= rest_cap) {
		if (rptr_temp < wptr_temp) {
			ERRLOG("err! r:%d w:%d\n", rptr_temp, wptr_temp);
			return RS232C_CM4_NG;
		}
		rptr_temp++;
		memcpy(buf, &rs232c_tx_ring.buffer[rptr_temp], rest_cap-1);
		rptr_temp = 0;
		sz -= (rest_cap - 1);
		buf = (char *)(buf + (rest_cap - 1));
		memcpy(buf, &rs232c_tx_ring.buffer[rptr_temp], sz);
	} else {
		rptr_temp++;
		memcpy(buf, &rs232c_tx_ring.buffer[rptr_temp], sz);
	}
	/* Read Pointer increment */
	barrier();
	rs232c_tx_ring.rptr = rptr_temp + sz;

	rs232c_tx_wait_data_sts = rs232c_chk_wait_data(rs232c_tx_ring.wptr,
						rs232c_tx_ring.rptr);

	return sz_total;
}

/* Write to ring buffer */
static int rs232c_tx_ring_write(const char __user *buf, int count)
{
	int wptr_temp, rptr_temp;
	int rest_cap, rest;
	int sz;

	rptr_temp = rs232c_tx_ring.rptr;
	wptr_temp = rs232c_tx_ring.wptr;
	rest_cap  = RS232C_RING_BUFF_SIZE - wptr_temp;
	sz        = count;
	rest      = 0;

	if ((sz <= 0) || (sz > RS232C_RW_DATA_SIZE)) {
		ERRLOG("pointer size error! count:%d buf:0x%p\n", sz, buf);
		return RS232C_CM4_NG;
	}
	if (wptr_temp >= RS232C_RING_BUFF_SIZE) { /* Pointer inside check */
		ERRLOG("rs232c_tx_ring.wptr error! %d\n", wptr_temp);
		return RS232C_CM4_NG;
	}

	/* rest writable buffer size */
	if (rptr_temp <= wptr_temp)
		rest = RS232C_RING_BUFF_SIZE - (wptr_temp - rptr_temp);
	else
		rest = rptr_temp - wptr_temp;
	/* Flow control */
	if (rest < (RS232C_RW_DATA_SIZE + 1)) {
		ERRLOG("BUFF_FULL! size:%d rest:%d\n", sz, rest);
		return -RS232C_CM4_BUFF_FULL;
	}

	/* Write to ring buffer Tx */
	if ((sz + 1) >= rest_cap) {
		/* Write to end edge */
		rs232c_tx_ring.buffer[wptr_temp] = sz;
		wptr_temp++;
		if (copy_from_user(&rs232c_tx_ring.buffer[wptr_temp],
					buf, (rest_cap - 1)) != 0)
			return -EFAULT;
		wptr_temp = 0;
		sz -= (rest_cap - 1);
		buf = (char *)(buf + (rest_cap - 1));
		if (copy_from_user(&rs232c_tx_ring.buffer[wptr_temp],
					buf, sz) != 0)
			return -EFAULT;
	} else {
		rs232c_tx_ring.buffer[wptr_temp] = sz;
		wptr_temp++;
		if (copy_from_user(&rs232c_tx_ring.buffer[wptr_temp],
					buf, sz) != 0)
			return -EFAULT;
	}
	/* Write Pointer increment */
	barrier();
	rs232c_tx_ring.wptr = wptr_temp + sz;

	rs232c_tx_wait_data_sts = rs232c_chk_wait_data(rs232c_tx_ring.wptr,
						rs232c_tx_ring.rptr);

	return count;
}

/*--- Access for cm4 send -------------------------------------------------*/
int rs232c_core_cm4_send(u8 type, u8 length, u8 id, u8 *data)
{
	int ret;

	if ((data == NULL) || (length < 1)) {
		ERRLOG("param error data=%p length=%d\n", data, length);
		return -EINVAL;
	}

	/* Substitute the id to seq area as a workaround for the data area */
	/* shortage issue. */
	ret = rpmsg_cm4_send(CM4_MESSAGE_ID_232C, type, length, id, data);
	if (ret != 0) {
		ERRLOG("rpmsg_cm4_send failed(%d)\n", ret);
		return -EINVAL;
	}
	return 0;
}

int rs232c_core_enable(u8 enable)
{
	struct rs232c_cm4_mng_info_priv *priv = rs232c_cm4_drvdata;
	u8 dmy_data = 0;

	DBGLOG("enable\n");

	if (enable != 1) {
		ERRLOG("enable select missing:%d\n", enable);
		return -EINVAL;
	}

	priv->mode_enable = enable;
	rs232c_core_cm4_send(CM4_MESSAGE_TYPE_REQ, 1,
			RS232C_ENABLE_REQ, &dmy_data);

	return 0;
}

int rs232c_core_change_baudrate(u8 baudrate)
{
	struct rs232c_cm4_mng_info_priv *priv = rs232c_cm4_drvdata;
	u8 data[5] = {0x00};
	u32 baud_value;

	if (baudrate > RS232C_CM4_BAUD_115200) {
		ERRLOG("baudrate select missing:%d\n", baudrate);
		return -EINVAL;
	}

	baud_value = rs232c_core_BaudRate[baudrate];

	priv->mode_baudrate = baudrate;
	data[1] =  baud_value        & 0xFF;
	data[2] = (baud_value >>  8) & 0xFF;
	data[3] = (baud_value >> 16) & 0xFF;
	data[4] = (baud_value >> 24) & 0xFF;
	rs232c_core_cm4_send(CM4_MESSAGE_TYPE_REQ, 5, RS232C_BAUD_REQ, data);

	return 0;
}

int rs232c_core_change_format(u8 format)
{
	struct rs232c_cm4_mng_info_priv *priv = rs232c_cm4_drvdata;
	u8 data = 0x00;

	if (format > RS232C_CM4_CIS_STX_CS_ON) {
		ERRLOG("format select missing:%d\n", format);
		return -EINVAL;
	}

	priv->mode_format = data = format;
	rs232c_core_cm4_send(CM4_MESSAGE_TYPE_REQ, 1, RS232C_FORMAT_REQ, &data);

	return 0;
}

void rs232c_core_request_data_send(u8 size, u8 *buf)
{
	struct rs232c_cm4_mng_info_priv *priv = rs232c_cm4_drvdata;
	u8 data[255] = {0x00};
	int snd_len;

	if (priv->mode_format) {
		snd_len = size + 1;
		memcpy(&data[1], buf, size);
	} else {
		snd_len = size;
		memcpy(data, buf, size);
	}
	rs232c_core_cm4_send(CM4_MESSAGE_TYPE_REQ, (u8)snd_len,
				RS232C_SEND_DATA, data);
}

void rs232c_core_request_testcmd(void)
{
	u8 dmy_data = 0;

	DBGLOG("req testcmd\n");

	rs232c_core_cm4_send(CM4_MESSAGE_TYPE_REQ, 1,
			RS232C_TESTCMD_REQ, &dmy_data);
}

static ssize_t rs232c_core_status_show(struct device *dev,
				struct device_attribute *attr,
				char *buf)
{
	struct rs232c_cm4_mng_info_priv *priv = dev_get_drvdata(dev);

	return snprintf(buf, PAGE_SIZE, "%d\n", priv->mode_enable);
}

static ssize_t rs232c_core_power_store(struct device *dev,
				struct device_attribute *attr,
				const char *buf, size_t size)
{
	struct rs232c_cm4_mng_info_priv *priv = dev_get_drvdata(dev);
	u8 dmy_data = 0;
	u8 power;

	if (kstrtou8(buf, 0, &power))
		return -EINVAL;

	if (power != 1)
		return -EINVAL;

	if (priv->mode_enable == 0) {
		priv->mode_enable = power;
		rs232c_core_cm4_send(CM4_MESSAGE_TYPE_REQ, 1,
					RS232C_ENABLE_REQ, &dmy_data);
	}

	return size;
}

static ssize_t rs232c_core_tstcmd_store(struct device *dev,
				struct device_attribute *attr,
				const char *buf, size_t size)
{
	struct rs232c_cm4_mng_info_priv *priv = dev_get_drvdata(dev);
	u8 data[255] = {0x00};
	int len = strlen(buf) - 1;
	int snd_len = 0;

	if (len%2)
		return -EINVAL;

	if (priv->mode_format) {
		if (len/2 > (RS232C_RW_DATA_SIZE-2)) {
			ERRLOG("failed overflow length = %d, cis_at\n", len/2);
			return -EINVAL;
		}

		if (hex2bin(&data[1], buf, len/2))
			return -EINVAL;
		snd_len = len/2+1;
	} else {
		if (len/2 > RS232C_RW_DATA_SIZE) {
			ERRLOG("failed overflow length = %d, cis_mt\n", len/2);
			return -EINVAL;
		}

		if (hex2bin(data, buf, len/2))
			return -EINVAL;
		snd_len = len/2;
	}

	MSGLOG("host send req data = %s", buf);
	rs232c_core_cm4_send(CM4_MESSAGE_TYPE_REQ, (u8)snd_len,
				RS232C_SEND_DATA, data);

	return size;
}

static ssize_t rs232c_core_test_store(struct device *dev,
				struct device_attribute *attr,
				const char *buf, size_t size)
{
	rs232c_core_request_testcmd();
	return size;
}

static ssize_t rs232c_core_format_store(struct device *dev,
				struct device_attribute *attr,
				const char *buf, size_t size)
{
	u8 format;

	if (kstrtou8(buf, 0, &format))
		return -EINVAL;

	(void)rs232c_core_change_format(format);

	return size;
}

static ssize_t rs232c_core_baudrate_store(struct device *dev,
				struct device_attribute *attr,
				const char *buf, size_t size)
{
	u8 baudrat;

	if (kstrtou8(buf, 0, &baudrat))
		return -EINVAL;

	(void)rs232c_core_change_baudrate(baudrat);

	return size;
}

static DEVICE_ATTR(status, 0600, rs232c_core_status_show, NULL);
static DEVICE_ATTR(rpower, 0600, NULL, rs232c_core_power_store);
static DEVICE_ATTR(testcommand, 0600, NULL, rs232c_core_tstcmd_store);
static DEVICE_ATTR(test, 0600, NULL, rs232c_core_test_store);
static DEVICE_ATTR(format, 0600, NULL, rs232c_core_format_store);
static DEVICE_ATTR(baudrate, 0600, NULL, rs232c_core_baudrate_store);

static struct attribute *attributes[] = {
	&dev_attr_status.attr,
	&dev_attr_rpower.attr,
	&dev_attr_testcommand.attr,
	&dev_attr_test.attr,
	&dev_attr_format.attr,
	&dev_attr_baudrate.attr,
	NULL,
};

static const struct attribute_group attr_group = {
	.attrs	= (struct attribute **)attributes,
};

/*--- System call ---------------------------------------------------------*/
static ssize_t rs232c_core_read(struct file *filp, char __user *buf,
					size_t count, loff_t *offset)
{
	struct rs232c_cm4_mng_info_priv *priv = rs232c_cm4_drvdata;
	ssize_t status = 0;

	DBGLOG("read\n");

	if (rs232c_rx_wait_data_sts) {
		mutex_lock(&priv->rx_state_lock);
		status = (ssize_t)rs232c_rx_ring_read(buf, count);
		mutex_unlock(&priv->rx_state_lock);
	}

	return status;
}

static ssize_t rs232c_core_write(struct file *filp, const char __user *buf,
					size_t count, loff_t *offset)
{
	struct rs232c_cm4_mng_info_priv *priv = rs232c_cm4_drvdata;
	ssize_t status = 0;

	DBGLOG("write\n");

	/* if count the exceeds data length max, result to error */
	if (priv->mode_format) {
		if (count > (RS232C_RW_DATA_SIZE - 2)) {
			ERRLOG("failed overflow length = %d, cis_at\n",
				(int)count);
			return -EINVAL;
		}
	} else {
		if (count > RS232C_RW_DATA_SIZE) {
			ERRLOG("failed overflow length = %d, cis_mt\n",
				(int)count);
			return -EINVAL;
		}
	}

	mutex_lock(&priv->tx_state_lock);
	status = (ssize_t)rs232c_tx_ring_write(buf, (int)count);
	mutex_unlock(&priv->tx_state_lock);

	return status;
}

static long rs232c_core_ioctl(struct file *filp, unsigned int cmd,
						unsigned long arg)
{
	struct rs232c_cm4_mng_info_priv *priv = rs232c_cm4_drvdata;
	int err = 0;
	u32 missing = 0;
	struct rs232c_cm4_ioctl_data_param_def st_param;

	DBGLOG("ioctl\n");

	/* Check type and command number */
	if (_IOC_TYPE(cmd) != RS232C_CM4_IOC_MAGIC) {
		ERRLOG("command missing\n");
		return -ENOTTY;
	}

	if (_IOC_DIR(cmd) & _IOC_READ)
		err = !access_ok(VERIFY_WRITE, (void __user *)arg,
							_IOC_SIZE(cmd));
	if (err == 0 && _IOC_DIR(cmd) & _IOC_WRITE)
		err = !access_ok(VERIFY_READ, (void __user *)arg,
							_IOC_SIZE(cmd));
	if (err) {
		ERRLOG("command read/write missing:%d\n", (unsigned int)err);
		return -EFAULT;
	}

	missing = copy_from_user(&st_param, (void *)arg, sizeof(st_param));
	if (missing != 0)
		return -EFAULT;

	switch (cmd) {
	case RS232C_CM4_IOC_WRITE_ENABLE:
		err = rs232c_core_enable(st_param.write_val);
		if (err != 0)
			return err;
		break;
	case RS232C_CM4_IOC_READ_ENABLE:
		st_param.read_val = priv->mode_enable;
		if (copy_to_user((void __user *)arg, &st_param,
						sizeof(st_param))) {
			return -EFAULT;
		}
		break;
	case RS232C_CM4_IOC_WRITE_BAUDRATE:
		err = rs232c_core_change_baudrate(st_param.write_val);
		if (err != 0)
			return err;
		break;
	case RS232C_CM4_IOC_READ_BAUDRATE:
		st_param.read_val = priv->mode_baudrate;
		if (copy_to_user((void __user *)arg, &st_param,
						sizeof(st_param))) {
			return -EFAULT;
		}
		break;
	case RS232C_CM4_IOC_WRITE_FORMAT:
		err = rs232c_core_change_format(st_param.write_val);
		if (err != 0)
			return err;
		break;
	case RS232C_CM4_IOC_READ_FORMAT:
		st_param.read_val = priv->mode_format;
		if (copy_to_user((void __user *)arg, &st_param,
						sizeof(st_param))) {
			return -EFAULT;
		}
		break;
	case RS232C_CM4_IOC_WRITE_TESTCMD:
		rs232c_core_request_testcmd();
		break;
	default:
		ERRLOG("unsupported command:%d\n", cmd);
		break;
	}

	return 0;
}

static unsigned int rs232c_core_poll(struct file *filp, poll_table *ptbl)
{
	struct rs232c_cm4_mng_info_priv *ddt = rs232c_cm4_drvdata;
	unsigned int mask = 0;

	DBGLOG("poll\n");

	if (ddt == NULL)
		return -EBADFD;

	if (!rs232c_rx_wait_data_sts)
		poll_wait(filp, &ddt->rxwait, ptbl);

	if (rs232c_rx_wait_data_sts)
		mask |= POLLIN | POLLRDNORM;

	return mask;
}

static const struct file_operations rs232c_core_fops = {
	.read    = rs232c_core_read,
	.write   = rs232c_core_write,
	.unlocked_ioctl = rs232c_core_ioctl,
	.compat_ioctl   = rs232c_core_ioctl,
	.poll    = rs232c_core_poll,
};

static void kthread_rs232c_core_main(void)
{
	struct rs232c_cm4_mng_info_priv *priv = rs232c_cm4_drvdata;
	int status;
	u8  buf[RS232C_RW_DATA_SIZE];

	DBGLOG("kthread_main\n");

	if (rs232c_tx_wait_data_sts) {
		memset(buf, 0, sizeof(buf));
		mutex_lock(&priv->tx_state_lock);
		status = (ssize_t)rs232c_tx_ring_read(buf, RS232C_RW_DATA_SIZE);
		mutex_unlock(&priv->tx_state_lock);
		if (status > 0)
			rs232c_core_request_data_send(status, buf);
	}
}

static int kthread_rs232c_core_func(void *arg)
{
	DBGLOG("kthread_func\n");

	while (!kthread_should_stop()) {
		set_current_state(TASK_INTERRUPTIBLE);
		schedule_timeout(msecs_to_jiffies(20));	/* cycle 20 msec */
		if (kthread_should_stop())
			break;

		if (rs232c_drv_power == RS232C_DRV_POWER_SUSPEND)
			continue; /* for suspend */

		kthread_rs232c_core_main();
	}

	return 0;
}

void rs232c_core_callback(u8 type, u8 length, u8 seq, u8 *data)
{
	struct rs232c_cm4_mng_info_priv *priv = rs232c_cm4_drvdata;
	int status;
	char str[512] = {};

	DBGLOG("callback\n");

	if ((data == NULL) || (length <= 0)) {
		ERRLOG("param error data=%p length=%d\n", data, length);
		return;
	}

	switch (seq) {
	case RS232C_HOST_RCV_NTF:
		if (length > RS232C_RW_DATA_SIZE) {
			ERRLOG("host recv data length fail(%d)\n", length);
		} else {
			bin2hex(str, data, length);
			MSGLOG("host recv data ntf = %s\n", str);
		}
		mutex_lock(&priv->rx_state_lock);
		status = rs232c_rx_ring_write(data, length);
		mutex_unlock(&priv->rx_state_lock);
		if (status > 0)
			wake_up_interruptible(&rs232c_cm4_drvdata->rxwait);
		break;
	case RS232C_HOST_SND_RESP:
		if (data[0])
			ERRLOG("host send resp ret = 0x%02x\n", data[0]);
		break;
	case RS232C_BAUDRATE_RESP:
		MSGLOG("baudrate change resp ret = 0x%02x\n", data[0]);
		break;
	default:
		ERRLOG("invalid command id =0x%02x\n", seq);
		break;
	}

}

static void rs232c_core_suspend_common(void)
{
	if (rs232c_drv_power == RS232C_DRV_POWER_SUSPEND)
		return;

	rs232c_drv_power = RS232C_DRV_POWER_SUSPEND;
	if (rs232c_cm4_drvdata->kthread_tsk) {
		kthread_stop(rs232c_cm4_drvdata->kthread_tsk);
		rs232c_cm4_drvdata->kthread_tsk = NULL;
	}
}

static int rs232c_core_resume_common(void)
{
	int status = 0;

	if (rs232c_drv_power != RS232C_DRV_POWER_SUSPEND)
		return status;

	rs232c_cm4_drvdata->kthread_tsk =
			kthread_run(kthread_rs232c_core_func,
				(void *)rs232c_cm4_drvdata, "rs232c_cm4_tsk");
	if (IS_ERR(rs232c_cm4_drvdata->kthread_tsk)) {
		ERRLOG("error! kthread_tsk: 0x%p\n",
			rs232c_cm4_drvdata->kthread_tsk);
		status = RS232C_CM4_NG;
		goto resume_common_err;
	}

	rs232c_drv_power = RS232C_DRV_POWER_NORMAL;

	return status;

resume_common_err:
	ERRLOG("%s exit error status=%d\n", __func__, status);
	return status;
}

static int rs232c_core_suspend(struct device *dev)
{
	DBGLOG("suspend\n");

	rs232c_core_suspend_common();

	return 0;
}

static int rs232c_core_resume(struct device *dev)
{
	int status = 0;

	DBGLOG("resume\n");

	status = rs232c_core_resume_common();

	return status;
}

static int rs232c_core_probe(struct platform_device *pdev)
{
	struct rs232c_cm4_mng_info_priv *priv;
	int ret = 0;

	DBGLOG("probe\n");

	priv = devm_kzalloc(&pdev->dev, sizeof(struct rs232c_cm4_mng_info_priv),
			     GFP_KERNEL);
	if (!priv) {
		ERRLOG("allocate failed\n");
		return -ENOMEM;
	}

	priv->dev = &pdev->dev;
	priv->mode_enable = 0;
	priv->mode_baudrate = 0;
	priv->mode_format = 0;


	rs232c_cm4_drvdata = priv;
	platform_set_drvdata(pdev, priv);

	memset(&rs232c_tx_ring, 0, sizeof(rs232c_tx_ring));
	memset(&rs232c_rx_ring, 0, sizeof(rs232c_rx_ring));
	rs232c_rx_wait_data_sts = false;
	rs232c_tx_wait_data_sts = false;

	init_waitqueue_head(&rs232c_cm4_drvdata->rxwait);
	mutex_init(&rs232c_cm4_drvdata->tx_state_lock);
	mutex_init(&rs232c_cm4_drvdata->rx_state_lock);

	ret = alloc_chrdev_region(&rs232c_cm4_drvdata->major, 0, 1, "rs232c");
	if (ret) {
		ERRLOG("alloc_chrdev_region() error:0x%x\n", ret);
		goto out_unregister;
	}

	cdev_init(&rs232c_cm4_drvdata->cdev, &rs232c_core_fops);
	rs232c_cm4_drvdata->cdev.owner = THIS_MODULE;

	ret = cdev_add(&rs232c_cm4_drvdata->cdev, rs232c_cm4_drvdata->major, 1);
	if (ret) {
		ERRLOG("cdev_add() error:0x%x\n", ret);
		goto out_unregister;
	}
	rs232c_cm4_drvdata->class = class_create(THIS_MODULE, DRV_NAME);
	if (IS_ERR(rs232c_cm4_drvdata->class)) {
		ret = PTR_ERR(rs232c_cm4_drvdata->class);
		ERRLOG("class_create() error:0x%x\n", ret);
		goto out_cdev_del;
	}

	rs232c_cm4_drvdata->idev = device_create(rs232c_cm4_drvdata->class,
			NULL, rs232c_cm4_drvdata->major, NULL, DRV_NAME);
	if (IS_ERR(rs232c_cm4_drvdata->idev)) {
		ret = PTR_ERR(rs232c_cm4_drvdata->idev);
		ERRLOG("device_create() error:0x%x\n", ret);
		goto out_cdev_del;
	}

	rs232c_cm4_drvdata->kthread_tsk = kthread_run(kthread_rs232c_core_func,
				(void *)rs232c_cm4_drvdata, "rs232c_cm4_tsk");
	if (IS_ERR(rs232c_cm4_drvdata->kthread_tsk)) {
		ret = PTR_ERR(rs232c_cm4_drvdata->kthread_tsk);
		ERRLOG("kthread_run() error:0x%x\n", ret);
		goto out_cdev_del;
	}

	ret = rpmsg_cm4_register_callback(CM4_MESSAGE_ID_232C,
					rs232c_core_callback);
	if (ret != 0) {
		ERRLOG("regist callback failed(%d)\n", ret);
		ret = -EPROBE_DEFER;
		goto out_kthread_stop;
	}

	ret = sysfs_create_group(&pdev->dev.kobj, &attr_group);
	if (ret) {
		ERRLOG("sysfs create failed(%d)\n", ret);
		goto out_kthread_stop;
	}

	return 0;

out_kthread_stop:
	kthread_stop(rs232c_cm4_drvdata->kthread_tsk);
out_cdev_del:
	cdev_del(&rs232c_cm4_drvdata->cdev);
out_unregister:
	devm_kfree(&pdev->dev, priv);
	rs232c_cm4_drvdata = NULL;

	return ret;
}

static int rs232c_core_remove(struct platform_device *pdev)
{
	DBGLOG("remove\n");

	sysfs_remove_group(&pdev->dev.kobj, &attr_group);
	cdev_del(&rs232c_cm4_drvdata->cdev);
	devm_kfree(&pdev->dev, rs232c_cm4_drvdata);
	kthread_stop(rs232c_cm4_drvdata->kthread_tsk);
	rs232c_cm4_drvdata = NULL;

	return 0;
}

static const struct of_device_id rs232c_core_dt_ids[] = {
	{ .compatible = "sony,rs232c_cm4", },
	{ },
};
MODULE_DEVICE_TABLE(of, rs232c_core_dt_ids);

static const struct dev_pm_ops rs232c_core_pmops = {
	.suspend = rs232c_core_suspend,
	.resume  = rs232c_core_resume,
};


static struct platform_driver rs232c_core_driver = {
	.driver = {
		.name = DRV_NAME,
		.owner = THIS_MODULE,
		.of_match_table = rs232c_core_dt_ids,
		.pm    = &rs232c_core_pmops,
	},
	.probe = rs232c_core_probe,
	.remove = rs232c_core_remove,
};

static int __init rs232c_core_init(void)
{
	int result = 0;

	DBGLOG("init\n");

	result = platform_driver_register(&rs232c_core_driver);
	if (result != 0) {
		ERRLOG("register driver failed(%d)\n", result);
		return result;
	}

	return 0;
}

static void __exit rs232c_core_exit(void)
{
	DBGLOG("exit\n");

	platform_driver_unregister(&rs232c_core_driver);
}

module_init(rs232c_core_init);
module_exit(rs232c_core_exit);


MODULE_AUTHOR("Sony Corporation");
MODULE_DESCRIPTION("rs232c_cm4");
MODULE_LICENSE("GPL");

