// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright 2020,2021,2022 Sony Corporation
 */

#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/of_gpio.h>

#include <misc/sw_pll.h>
#include <sound/sony-asoc-card.h>
#include <misc/simple_consumer.h>
#include <linux/gpio/consumer.h>
#ifdef CONFIG_SND_SOC_TAS5825
#include <sound/tas5825.h>
#endif
#ifdef CONFIG_SND_SOC_TWINCODEC
#include <sound/twincodec.h>
#include <sound/imx-pcm-dma-v2.h>
#endif
#include <sound/pcm5102a_mute.h>

/* #define TRACE_PRINT_ON */
#define TRACE_TAG "####### "

/* trace print macro */
#ifdef TRACE_PRINT_ON
	#define print_trace(fmt, args...) pr_info(TRACE_TAG "" fmt, ##args)
#else
	#define print_trace(fmt, args...) pr_debug(TRACE_TAG "" fmt, ##args)
#endif

#define SONY_HW_MUTE_ON       1
#define SONY_HW_MUTE_OFF      0

enum mute_factor_index {
	FS_CHANGE,
	ALSA_TRIG,
	FCT_NUM,
};

struct mute_factor {
	unsigned int mute;
	unsigned int enable;
};

struct sony_hw_mute_priv {
	struct device *dev;
	struct mutex mute_mutex;
	struct mute_factor fct[FCT_NUM];
	struct workqueue_struct *mute_wq;
	struct delayed_work mute_off_work;
	struct gpio_desc *gpio_soc_mute;

	struct mute_factor zone_fct[FCT_NUM];
	struct workqueue_struct *zone_soc_mute_wq;
	struct delayed_work zone_soc_mute_off_work;
	struct gpio_desc *gpio_zone_soc_mute;

	unsigned int swpll_mute_req;
	unsigned int mute_off_delay;
	unsigned int zone_soc_mute_off_delay;
	unsigned int audio_codec_type;	/* 0=tas5825, 1=twincodec */
};

/* only used in callback */
static struct sony_hw_mute_priv *sony_hw_mute_drvdata;

static void mute_off_work_func(struct work_struct *work)
{
	struct delayed_work *dwork = to_delayed_work(work);
	struct sony_hw_mute_priv *priv = container_of(
		dwork, struct sony_hw_mute_priv, mute_off_work);

	gpiod_set_value(priv->gpio_soc_mute, 0);
	pr_debug("%s(): mute=0\n", __func__);
}

static void sony_hw_mute_update(struct sony_hw_mute_priv *priv)
{
	unsigned char i;
	unsigned int mute = 0;

	print_trace("%s()\n", __func__);

	for (i = 0; i < FCT_NUM; i++)
		if (priv->fct[i].enable)
			mute |= priv->fct[i].mute;

	if (mute == SONY_HW_MUTE_ON) {
		cancel_delayed_work(&priv->mute_off_work);
		gpiod_set_value(priv->gpio_soc_mute, 1);

	} else {
		queue_delayed_work(priv->mute_wq,
			&priv->mute_off_work,
			msecs_to_jiffies(priv->mute_off_delay));
	}
	pr_debug("%s(): mute=%u\n", __func__, mute);
}

static void zone_soc_mute_off_work_func(struct work_struct *work)
{
	struct delayed_work *dwork = to_delayed_work(work);
	struct sony_hw_mute_priv *priv = container_of(
		dwork, struct sony_hw_mute_priv, zone_soc_mute_off_work);

	if (priv->gpio_zone_soc_mute)
		gpiod_set_value(priv->gpio_zone_soc_mute, 0);

	pr_debug("%s(): mute=0\n", __func__);
}

static void zone_soc_mute_update(struct sony_hw_mute_priv *priv)
{
	unsigned int mute = 0;

	print_trace("%s()\n", __func__);

	if (!priv->gpio_zone_soc_mute)
		return;

	if (priv->zone_fct[ALSA_TRIG].enable)
		mute = priv->zone_fct[ALSA_TRIG].mute;

	if (mute == SONY_HW_MUTE_ON) {
		cancel_delayed_work(&priv->zone_soc_mute_off_work);
		gpiod_set_value(priv->gpio_zone_soc_mute, 1);
	} else {
		queue_delayed_work(priv->zone_soc_mute_wq,
			&priv->zone_soc_mute_off_work,
			msecs_to_jiffies(priv->zone_soc_mute_off_delay));
	}
	pr_debug("%s(): mute=%u\n", __func__, mute);
}

static ssize_t hw_mute_status_show(struct device *dev,
				   struct device_attribute *attr,
				   char *buf)
{
	struct sony_hw_mute_priv *priv = dev_get_drvdata(dev);
	ssize_t rb = 0;

	rb += snprintf(buf + rb, PAGE_SIZE - rb,
			"[fs_change] mute:%d, enable:%d\n",
			priv->fct[FS_CHANGE].mute, priv->fct[FS_CHANGE].enable);
	rb += snprintf(buf + rb, PAGE_SIZE - rb,
			"[alsa_trig] mute:%d, enable:%d\n",
			priv->fct[ALSA_TRIG].mute, priv->fct[ALSA_TRIG].enable);
	rb += snprintf(buf + rb, PAGE_SIZE - rb,
			"swpll_mute_req:%d\n",
			priv->swpll_mute_req);
	rb += snprintf(buf + rb, PAGE_SIZE - rb,
			"soc_mute:%d\n",
			gpiod_get_value(priv->gpio_soc_mute));
	if (priv->gpio_zone_soc_mute) {
		rb += snprintf(buf + rb, PAGE_SIZE - rb,
				"[zone_alsa_trig] mute:%d, enable:%d\n",
				priv->zone_fct[ALSA_TRIG].mute,
				priv->zone_fct[ALSA_TRIG].enable);
		rb += snprintf(buf + rb, PAGE_SIZE - rb,
				"zone_soc_mute:%d\n",
				gpiod_get_value(priv->gpio_zone_soc_mute));
	}

	return rb;
}

static ssize_t fs_change_enable_store(struct device *dev,
				      struct device_attribute *attr,
				      const char *buf, size_t size)
{
	struct sony_hw_mute_priv *priv = dev_get_drvdata(dev);
	long enable;

	if (kstrtol(buf, 0, &enable))
		return -EINVAL;

	priv->fct[FS_CHANGE].enable = enable;
	sony_hw_mute_update(priv);

	return size;
}

static ssize_t alsa_trigger_enable_store(struct device *dev,
					 struct device_attribute *attr,
					 const char *buf, size_t size)
{
	struct sony_hw_mute_priv *priv = dev_get_drvdata(dev);
	long enable;

	if (kstrtol(buf, 0, &enable))
		return -EINVAL;

	priv->fct[ALSA_TRIG].enable = enable;
	sony_hw_mute_update(priv);

	return size;
}

static ssize_t soc_mute_store(struct device *dev,
			      struct device_attribute *attr,
			      const char *buf, size_t size)
{
	struct sony_hw_mute_priv *priv = dev_get_drvdata(dev);
	long mute;

	if (kstrtol(buf, 0, &mute))
		return -EINVAL;

	if (gpiod_get_value(priv->gpio_soc_mute) != mute)
		gpiod_set_value(priv->gpio_soc_mute, mute);

	return size;
}

static ssize_t zone_alsa_trigger_enable_store(struct device *dev,
					      struct device_attribute *attr,
					      const char *buf, size_t size)
{
	struct sony_hw_mute_priv *priv = dev_get_drvdata(dev);
	long enable;

	if (kstrtol(buf, 0, &enable))
		return -EINVAL;

	priv->zone_fct[ALSA_TRIG].enable = enable;
	zone_soc_mute_update(priv);

	return size;
}

static ssize_t zone_soc_mute_store(struct device *dev,
				   struct device_attribute *attr,
				   const char *buf, size_t size)
{
	struct sony_hw_mute_priv *priv = dev_get_drvdata(dev);
	long mute;

	if (kstrtol(buf, 0, &mute))
		return -EINVAL;

	if (priv->gpio_zone_soc_mute &&
	    gpiod_get_value(priv->gpio_zone_soc_mute) != mute)
		gpiod_set_value(priv->gpio_zone_soc_mute, mute);

	return size;
}

static DEVICE_ATTR(status, 0600, hw_mute_status_show, NULL);
static DEVICE_ATTR(fs_change_enable, 0600, NULL, fs_change_enable_store);
static DEVICE_ATTR(alsa_trigger_enable, 0600, NULL, alsa_trigger_enable_store);
static DEVICE_ATTR(soc_mute, 0600, NULL, soc_mute_store);
static DEVICE_ATTR(zone_alsa_trigger_enable, 0600, NULL,
		   zone_alsa_trigger_enable_store);
static DEVICE_ATTR(zone_soc_mute, 0600, NULL, zone_soc_mute_store);

static struct attribute *attributes[] = {
	&dev_attr_status.attr,
	&dev_attr_fs_change_enable.attr,
	&dev_attr_alsa_trigger_enable.attr,
	&dev_attr_soc_mute.attr,
	&dev_attr_zone_alsa_trigger_enable.attr,
	&dev_attr_zone_soc_mute.attr,
	NULL,
};

static const struct attribute_group attr_group = {
	.attrs  = (struct attribute **)attributes,
};

/* callback function which is called by sw_pll driver */
static int hw_mute_fs_change_notify(struct notifier_block *self,
				unsigned long notused, void *event)
{
	SW_PLL_STAT *stat = (SW_PLL_STAT *)event;
	struct sony_hw_mute_priv *priv = sony_hw_mute_drvdata;

	print_trace("%s() Mute_Req=%d\n", __func__, stat->Mute_Req);

	priv->swpll_mute_req = stat->Mute_Req;
	mutex_lock(&priv->mute_mutex);
	if (stat->Mute_Req == SONY_HW_MUTE_ON)
		priv->fct[FS_CHANGE].mute = 1;
	sony_hw_mute_update(priv);
	mutex_unlock(&priv->mute_mutex);

	return NOTIFY_OK;
}

static struct notifier_block hw_mute_swpll_nb = {
	.notifier_call = hw_mute_fs_change_notify
};

/* callback function which is called by sony-asoc-card and tas5825 */
static int hw_mute_alsa_trigger_notify(struct notifier_block *self,
				    unsigned long mute, void *notused)
{
	struct sony_hw_mute_priv *priv = sony_hw_mute_drvdata;

	print_trace("%s() mute=%lu\n", __func__, mute);

	if (mute == SONY_HW_MUTE_ON) {
		priv->fct[ALSA_TRIG].mute = 1;
	} else {
		priv->fct[ALSA_TRIG].mute = 0;
		priv->fct[FS_CHANGE].mute = 0;
	}
	sony_hw_mute_update(priv);

	return NOTIFY_OK;
}

static struct notifier_block hw_mute_alsa_trigger_nb = {
	.notifier_call = hw_mute_alsa_trigger_notify
};

/* callback function which is called by simple_consumer */
static int hw_mute_simple_consumer_notify(struct notifier_block *self,
				unsigned long mute, void *notused)
{
	struct sony_hw_mute_priv *priv = sony_hw_mute_drvdata;
	print_trace("%s() mute=%lu\n", __func__, mute);

	mutex_lock(&priv->mute_mutex);
	if (mute == SONY_HW_MUTE_ON) {
		gpiod_set_value(priv->gpio_soc_mute, 1);
		if (priv->gpio_zone_soc_mute)
			gpiod_set_value(priv->gpio_zone_soc_mute, 1);
	} else {
		gpiod_set_value(priv->gpio_soc_mute, 0);
		if (priv->gpio_zone_soc_mute)
			gpiod_set_value(priv->gpio_zone_soc_mute, 0);
	}

	mutex_unlock(&priv->mute_mutex);

	return NOTIFY_OK;
}

static struct notifier_block hw_mute_simple_consumer_nb = {
	.notifier_call = hw_mute_simple_consumer_notify
};

/* callback function which is called by sony-asoc-card and pcm5102a */
static int zone_soc_mute_alsa_trigger_notify(struct notifier_block *self,
				    unsigned long mute, void *notused)
{
	struct sony_hw_mute_priv *priv = sony_hw_mute_drvdata;

	print_trace("%s() mute=%lu\n", __func__, mute);

	if (mute == SONY_HW_MUTE_ON)
		priv->zone_fct[ALSA_TRIG].mute = 1;
	else
		priv->zone_fct[ALSA_TRIG].mute = 0;
	zone_soc_mute_update(priv);

	return NOTIFY_OK;
}

static struct notifier_block zone_soc_mute_alsa_trigger_nb = {
	.notifier_call = zone_soc_mute_alsa_trigger_notify
};

/* platform driver */
static int sony_hw_mute_probe(struct platform_device *pdev)
{
	struct sony_hw_mute_priv *priv;
	int rv;

	print_trace("%s()\n", __func__);

	priv = devm_kzalloc(&pdev->dev, sizeof(struct sony_hw_mute_priv),
			     GFP_KERNEL);
	if (!priv)
		return -ENOMEM;

	mutex_init(&priv->mute_mutex);
	priv->dev = &pdev->dev;

	priv->swpll_mute_req = 0;
	priv->fct[FS_CHANGE].mute = 0;
	priv->fct[FS_CHANGE].enable = 1;
	priv->fct[ALSA_TRIG].mute = 1;
	priv->fct[ALSA_TRIG].enable = 1;
	priv->gpio_soc_mute = devm_gpiod_get(&pdev->dev,
	                                     "hw_mute",
	                                     GPIOD_OUT_HIGH);
	if (IS_ERR(priv->gpio_soc_mute)) {
		dev_err(&pdev->dev, "Failed to get hw_mute-gpios\n");
		return PTR_ERR(priv->gpio_soc_mute);
	} else {
		sony_hw_mute_update(priv);
	}

	priv->zone_fct[FS_CHANGE].mute = 0;
	priv->zone_fct[FS_CHANGE].enable = 0;
	priv->zone_fct[ALSA_TRIG].mute = 1;
	priv->zone_fct[ALSA_TRIG].enable = 1;
	priv->gpio_zone_soc_mute = devm_gpiod_get(&pdev->dev,
						  "zone_soc_mute",
						  GPIOD_OUT_HIGH);
	if (IS_ERR(priv->gpio_zone_soc_mute)) {
		rv = PTR_ERR(priv->gpio_zone_soc_mute);
		if (rv != -ENOENT) {
			dev_err(&pdev->dev, "Failed to get zone_soc_mute-gpios\n");
			goto out1;
		}
		priv->gpio_zone_soc_mute = NULL;
	} else {
		zone_soc_mute_update(priv);
	}

	if (of_property_read_u32(pdev->dev.of_node,
				 "mute-off-delay",
				 &priv->mute_off_delay))
		priv->mute_off_delay = 0;
	if (of_property_read_u32(pdev->dev.of_node,
				 "zone_soc_mute-off-delay",
				 &priv->zone_soc_mute_off_delay))
		priv->zone_soc_mute_off_delay = 0;
	if (of_property_read_u32(pdev->dev.of_node,
				"audio-codec-type",
				&priv->audio_codec_type))
		priv->audio_codec_type = 0;

	sony_hw_mute_drvdata = priv;

	platform_set_drvdata(pdev, priv);

	rv = sysfs_create_group(&pdev->dev.kobj, &attr_group);
	if (rv)
		goto out1;

	priv->mute_wq = create_singlethread_workqueue("sony_hw_mute_wq");
	if (!priv->mute_wq) {
		rv = -ENOMEM;
		goto out2;
	}
	INIT_DELAYED_WORK(&priv->mute_off_work, mute_off_work_func);
	priv->zone_soc_mute_wq =
		create_singlethread_workqueue("zone_soc_mute_wq");
	if (!priv->zone_soc_mute_wq) {
		rv = -ENOMEM;
		goto out3;
	}
	INIT_DELAYED_WORK(&priv->zone_soc_mute_off_work,
			  zone_soc_mute_off_work_func);

	rv = register_sw_pll_notifier(&hw_mute_swpll_nb);
	if (rv)
		goto out4;

	rv = register_sony_asoc_card_notifier(&hw_mute_alsa_trigger_nb);
	if (rv)
		goto out5;

	switch (priv->audio_codec_type) {
#ifdef CONFIG_SND_SOC_TWINCODEC
	case 1:
		rv = register_twincodec_notifier(&hw_mute_alsa_trigger_nb);
		if (rv == 0) {
			rv = register_dsddrain_notifier(&hw_mute_alsa_trigger_nb);
			if (rv)
				unregister_twincodec_notifier(&hw_mute_alsa_trigger_nb);
		}
		break;
#endif
	default:
#ifdef CONFIG_SND_SOC_TAS5825
		rv = register_tas5825_notifier(&hw_mute_alsa_trigger_nb);
#else
		rv = 0;
#endif
	}
	if (rv)
		goto out6;

	rv = register_simple_consumer_notifier(&hw_mute_simple_consumer_nb);
	if (rv)
		goto out7;

	rv = register_sony_asoc_zone_notifier(&zone_soc_mute_alsa_trigger_nb);
	if (rv)
		goto out8;

	rv = register_pcm5102a_notifier(&zone_soc_mute_alsa_trigger_nb);
	if (rv)
		goto out9;

	return 0;
out9:
	unregister_sony_asoc_zone_notifier(&zone_soc_mute_alsa_trigger_nb);
out8:
	unregister_simple_consumer_notifier(&hw_mute_simple_consumer_nb);
out7:
	switch (priv->audio_codec_type) {
#ifdef CONFIG_SND_SOC_TWINCODEC
	case 1:
		unregister_twincodec_notifier(&hw_mute_alsa_trigger_nb);
		unregister_dsddrain_notifier(&hw_mute_alsa_trigger_nb);
		break;
#endif
	default:
#ifdef CONFIG_SND_SOC_TAS5825
		unregister_tas5825_notifier(&hw_mute_alsa_trigger_nb);
#endif
	}
out6:
	unregister_sony_asoc_card_notifier(&hw_mute_alsa_trigger_nb);
out5:
	unregister_sw_pll_notifier(&hw_mute_swpll_nb);
out4:
	destroy_workqueue(priv->zone_soc_mute_wq);
out3:
	destroy_workqueue(priv->mute_wq);
out2:
	sysfs_remove_group(&pdev->dev.kobj, &attr_group);
out1:
	return rv;
}

static int sony_hw_mute_remove(struct platform_device *pdev)
{
	struct sony_hw_mute_priv *priv = dev_get_drvdata(&pdev->dev);

	print_trace("%s()\n", __func__);

	gpiod_set_value(priv->gpio_soc_mute, 0);
	if (priv->gpio_zone_soc_mute)
		gpiod_set_value(priv->gpio_zone_soc_mute, 0);

	cancel_delayed_work_sync(&priv->mute_off_work);
	destroy_workqueue(priv->mute_wq);
	cancel_delayed_work_sync(&priv->zone_soc_mute_off_work);
	destroy_workqueue(priv->zone_soc_mute_wq);

	unregister_sw_pll_notifier(&hw_mute_swpll_nb);
	unregister_sony_asoc_card_notifier(&hw_mute_alsa_trigger_nb);
	switch (priv->audio_codec_type) {
#ifdef CONFIG_SND_SOC_TWINCODEC
	case 1:
		unregister_twincodec_notifier(&hw_mute_alsa_trigger_nb);
		unregister_dsddrain_notifier(&hw_mute_alsa_trigger_nb);
		break;
#endif
	default:
#ifdef CONFIG_SND_SOC_TAS5825
		unregister_tas5825_notifier(&hw_mute_alsa_trigger_nb);
#endif
	}
	unregister_simple_consumer_notifier(&hw_mute_simple_consumer_nb);
	unregister_sony_asoc_zone_notifier(&zone_soc_mute_alsa_trigger_nb);
	unregister_pcm5102a_notifier(&zone_soc_mute_alsa_trigger_nb);

	sysfs_remove_group(&pdev->dev.kobj, &attr_group);

	return 0;
}

static int sony_hw_mute_suspend(struct device *dev)
{
	struct sony_hw_mute_priv *priv = dev_get_drvdata(dev);

	print_trace("%s()\n", __func__);

	gpiod_set_value(priv->gpio_soc_mute, 0);

	return 0;
}

static int sony_hw_mute_resume(struct device *dev)
{
	print_trace("%s()\n", __func__);

	return 0;
}

static const struct dev_pm_ops sony_hw_mute_pm_ops = {
	.suspend = sony_hw_mute_suspend,
	.resume = sony_hw_mute_resume,
};

#if defined(CONFIG_OF)
static const struct of_device_id sony_hw_mute_ids[] = {
	{.compatible = "sony,hw-mute",},
	{}
};
#endif

static struct platform_driver sony_hw_mute_driver = {
	.driver = {
		.name = "sony-hw-mute",
		.owner = THIS_MODULE,
		.pm = &sony_hw_mute_pm_ops,
#if defined(CONFIG_OF)
		.of_match_table = sony_hw_mute_ids,
#endif
	},
	.probe = sony_hw_mute_probe,
	.remove = sony_hw_mute_remove
};

module_platform_driver(sony_hw_mute_driver);

/* Module information */
MODULE_DESCRIPTION("sony hw mute driver");
MODULE_AUTHOR("Sony Home Entertainment & Sound Products Inc.");
MODULE_LICENSE("GPL");
MODULE_VERSION("v1.00");
