/* SPDX-License-Identifier: GPL-2.0 */
/*
 * sony_obs_port.c - Sony software driver to observing port
 *
 * Copyright 2021 Sony Corporation
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/virtio.h>
#include <linux/err.h>
#include <linux/cdev.h>
#include <linux/irq.h>
#include <linux/interrupt.h>
#include <linux/platform_device.h>
#include <linux/gpio.h>
#include <linux/of_gpio.h>
#include <linux/of_irq.h>
#include <linux/delay.h>

#define DRV_NAME		"obs_port"
#define ERRLOG(fmt, ...)	pr_err(DRV_NAME ": Error: " fmt, ## __VA_ARGS__)
#define WRNLOG(fmt, ...)	pr_warn(DRV_NAME ": Warning: " fmt, ## __VA_ARGS__)
#define MSGLOG(fmt, ...)	pr_info(DRV_NAME ": " fmt, ## __VA_ARGS__)
#define DBGLOG(fmt, ...)	pr_debug(DRV_NAME ": %s(%d): " fmt, __func__, __LINE__, ## __VA_ARGS__)

/*******************************************************************************
 * Definition
*******************************************************************************/
#define EVT_STRING_SIZE_MAX (50) /* buff(80) - default(24) = Remain(56) */

struct obs_port_mng_info_priv {
	struct platform_device *pdev;
	struct workqueue_struct *workqueue;
	struct delayed_work delayed;
	struct gpio_desc *gpio;
	int irq;
	int uevt_sts;
	unsigned int debounce_interval;
	const char *event;
	bool emergency;
};

/*******************************************************************************
 * Variables
*******************************************************************************/

/*******************************************************************************
 * Functions
*******************************************************************************/
static ssize_t obs_port_status_show(struct device *dev,
				struct device_attribute *attr,
				char *buf)
{
	struct obs_port_mng_info_priv *priv = dev_get_drvdata(dev);

	return snprintf(buf, PAGE_SIZE, "%d\n", priv->uevt_sts);
}

static DEVICE_ATTR(status, 0640, obs_port_status_show, NULL);

static struct attribute *attributes[] = {
	&dev_attr_status.attr,
	NULL,
};

static const struct attribute_group attr_group = {
	.attrs	= (struct attribute **)attributes,
};

static void obs_port_evt_notify(struct work_struct *work)
{
	struct obs_port_mng_info_priv *priv =
		container_of(work, struct obs_port_mng_info_priv, delayed.work);
	char event[80] = "EVENT=";
	char *envp[] = { event, NULL };
	int port = gpiod_get_value_cansleep(priv->gpio);

	if (priv->uevt_sts != port) {
		strncat(event, priv->event,
			strnlen(priv->event, EVT_STRING_SIZE_MAX));
		if (port)
			strncat(event, ",STATE=DETECTED", 15);
		else
			strncat(event, ",STATE=UNDETECTED", 17);
		kobject_uevent_env(&priv->pdev->dev.kobj, KOBJ_CHANGE, envp);
		priv->uevt_sts = port;
		if (priv->emergency == true)
			MSGLOG("Emergency Warning : %s is %s\n", priv->event,
				port ? "detect" : "not detect");
	}
}

static irqreturn_t obs_port_irq(int irq_num, void *data)
{
	struct obs_port_mng_info_priv *priv = data;

	cancel_delayed_work(&priv->delayed);
	queue_delayed_work(priv->workqueue, &priv->delayed,
			msecs_to_jiffies(priv->debounce_interval));

	return IRQ_HANDLED;
}

static int obs_port_probe(struct platform_device *pdev)
{
	struct obs_port_mng_info_priv *priv;
	struct device_node *np = pdev->dev.of_node;
	int ret = 0;

	DBGLOG("probe\n");

	priv = devm_kzalloc(&pdev->dev, sizeof(struct obs_port_mng_info_priv),
			     GFP_KERNEL);
	if (!priv) {
		ret = -ENOMEM;
		ERRLOG("allocate failed\n");
		goto err_exit;
	}

	priv->pdev = pdev;
	priv->uevt_sts = INT_MAX;

	if (device_property_read_u32(&pdev->dev, "debounce-interval",
				&priv->debounce_interval))
		priv->debounce_interval = 0;
	if (device_property_read_string(&pdev->dev, "event", &priv->event)) {
		ret = -ENOMEM;
		ERRLOG("read_string() event setting error (%d)\n", ret);
		goto err_kfree;
	}
	if (strnlen(priv->event, EVT_STRING_SIZE_MAX+1) > EVT_STRING_SIZE_MAX) {
		ret = -ENOMEM;
		ERRLOG("event string size error\n");
		goto err_kfree;
	}
	priv->emergency = of_property_read_bool(np, "emergency");

	/* gpio : input */
	priv->gpio = devm_gpiod_get(&pdev->dev, NULL, GPIOD_IN);
	if (IS_ERR(priv->gpio)) {
		ret = PTR_ERR(priv->gpio);
		ERRLOG("Failed to request gpios ret = %d\n", ret);
		goto err_kfree;
	}

	priv->workqueue = create_singlethread_workqueue("obs_port");
	if (!priv->workqueue) {
		ret = -ENOMEM;
		ERRLOG("unable to alloc workqueue\n");
		goto err_kfree;
	}

	INIT_DELAYED_WORK(&priv->delayed, obs_port_evt_notify);

	/* pin interrupt setting */
	priv->irq = gpiod_to_irq(priv->gpio);
	if (priv->irq < 0) {
		ret = priv->irq;
		ERRLOG("gpiod_to_irq() error (%d)\n", priv->irq);
		goto err_workqueue;
	}
	ret = devm_request_any_context_irq(&pdev->dev, priv->irq,
			obs_port_irq,
			IRQF_TRIGGER_RISING | IRQF_TRIGGER_FALLING,
			dev_name(&pdev->dev), priv);
	if (ret < 0) {
		ERRLOG("devm_request_any_context_irq() error (%d)\n", ret);
		goto err_workqueue;
	}

	platform_set_drvdata(pdev, priv);

	ret = sysfs_create_group(&pdev->dev.kobj, &attr_group);
	if (ret) {
		ERRLOG("sysfs create failed (%d)\n", ret);
		goto err_irq;
	}

	queue_delayed_work(priv->workqueue, &priv->delayed, 0);

	return 0;

err_irq:
	devm_free_irq(&pdev->dev, priv->irq, priv);
err_workqueue:
	destroy_workqueue(priv->workqueue);
err_kfree:
	devm_kfree(&pdev->dev, priv);
err_exit:
	return ret;
}

static int obs_port_remove(struct platform_device *pdev)
{
	struct obs_port_mng_info_priv *priv = dev_get_drvdata(&pdev->dev);

	DBGLOG("remove\n");

	sysfs_remove_group(&pdev->dev.kobj, &attr_group);

	devm_free_irq(&pdev->dev, priv->irq, priv);

	cancel_delayed_work(&priv->delayed);
	flush_workqueue(priv->workqueue);
	destroy_workqueue(priv->workqueue);

	devm_kfree(&pdev->dev, priv);

	return 0;
}

static const struct of_device_id obs_port_dt_ids[] = {
	{ .compatible = "sony,obs_port", },
	{ },
};

MODULE_DEVICE_TABLE(of, obs_port_dt_ids);

static struct platform_driver obs_port_driver = {
	.driver = {
		.name = DRV_NAME,
		.owner = THIS_MODULE,
		.of_match_table = obs_port_dt_ids,
	},
	.probe = obs_port_probe,
	.remove = obs_port_remove,
};

static int __init obs_port_init(void)
{
	int result = 0;

	DBGLOG("init\n");

	result = platform_driver_register(&obs_port_driver);
	if (result != 0) {
		ERRLOG("register driver failed(%d)\n", result);
		return result;
	}

	return 0;
}

static void __exit obs_port_exit(void)
{
	DBGLOG("exit\n");

	platform_driver_unregister(&obs_port_driver);
}

module_init(obs_port_init);
module_exit(obs_port_exit);


MODULE_AUTHOR("Sony Corporation");
MODULE_DESCRIPTION("obs_port");
MODULE_LICENSE("GPL");

