// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright 2021, 2022 Sony Corporation
 */

#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/mux/consumer.h>
#include <linux/types.h>
#include <linux/stddef.h>
#include <linux/pm_runtime.h>
#include <linux/regulator/consumer.h>
#include <linux/delay.h>

#undef TRACE_PRINT_ON
#define TRACE_TAG "####### "

/* trace print macro */
#ifdef TRACE_PRINT_ON
	#define print_trace(fmt, args...) pr_info(TRACE_TAG "" fmt, ##args)
#else
	#define print_trace(fmt, args...) {}
#endif

struct sony_video_selector_priv {
	struct device *dev;
	unsigned int state;
	unsigned int num_of_state;
	struct mux_control *mux;
	struct regulator *power;
	bool flag_selected;
};

static int set_state(struct sony_video_selector_priv *priv, unsigned int state)
{
	int ret;

	print_trace("%s()\n", __func__);

	if (priv->flag_selected == true) {
		ret = mux_control_deselect(priv->mux);
		if (ret < 0) {
			pr_err("%s() failed to deselect mux: %d\n",
				__func__, ret);
			return ret;
		}

		priv->flag_selected = false;
	}

	ret = mux_control_try_select(priv->mux, state);
	if (ret < 0) {
		pr_err("%s() failed to select mux: %d\n", __func__, ret);
		return ret;
	}

	priv->flag_selected = true;

	return ret;
}

static ssize_t state_store(struct device *dev,
				  struct device_attribute *attr,
				  const char *buf, size_t size)
{
	struct sony_video_selector_priv *priv = dev_get_drvdata(dev);
	unsigned int state;
	int ret;

	print_trace("%s()\n", __func__);

	if (kstrtouint (buf, 0, &state))
		return -EINVAL;

	if (state >= priv->num_of_state)
		return -EINVAL;

	if (priv->state != state) {
		ret = set_state(priv, state);
		priv->state = state;
		if (ret < 0) {	/* TODO: Return force OK in case of Error. */
			pr_err("%s() failed to set_state\n", __func__);
		}
	}

	return size;
}

static ssize_t state_show(struct device *dev,
				   struct device_attribute *attr,
				   char *buf)
{
	struct sony_video_selector_priv *priv = dev_get_drvdata(dev);
	ssize_t rb;

	print_trace("%s()\n", __func__);

	rb = snprintf(buf, PAGE_SIZE, "%d\n", priv->state);

	return rb;
}

static ssize_t num_of_state_show(struct device *dev,
				   struct device_attribute *attr,
				   char *buf)
{
	struct sony_video_selector_priv *priv = dev_get_drvdata(dev);
	ssize_t rb;

	print_trace("%s()\n", __func__);

	rb = snprintf(buf, PAGE_SIZE, "%d\n", priv->num_of_state);

	return rb;
}

static DEVICE_ATTR(state, 0600, state_show, state_store);
static DEVICE_ATTR(num_of_state, 0600, num_of_state_show, NULL);

static struct attribute *attributes[] = {
	&dev_attr_state.attr,
	&dev_attr_num_of_state.attr,
	NULL,
};

static const struct attribute_group attr_group = {
	.attrs  = (struct attribute **)attributes,
};


/* platform driver */
static int sony_video_selector_probe(struct platform_device *pdev)
{
	struct sony_video_selector_priv *priv;
	int ret;

	print_trace("%s()\n", __func__);

	priv = devm_kzalloc(&pdev->dev, sizeof(struct sony_video_selector_priv),
				 GFP_KERNEL);
	if (!priv)
		return -ENOMEM;


	priv->dev = &pdev->dev;
	priv->state = 0;
	priv->mux = devm_mux_control_get(&pdev->dev, NULL);
	if (IS_ERR(priv->mux)) {
		ret = PTR_ERR(priv->mux);
		if (ret != -EPROBE_DEFER)
			pr_err("%s() failed to devm_mux_control_get: ret=%d\n",
				 __func__, ret);
		return ret;
	}

	priv->power = devm_regulator_get(&pdev->dev, "video_sel");
	if (IS_ERR(priv->power)) {
		ret = PTR_ERR(priv->power);
		pr_err("%s() failed to no power-supply (%d)\n",
			__func__, ret);
		return ret;
	}

	priv->num_of_state = mux_control_states(priv->mux);
	priv->flag_selected = false;

	platform_set_drvdata(pdev, priv);

	ret = set_state(priv, priv->state);

	if (ret < 0)
		pr_err("%s() failed to set_state: ret=%d\n", __func__, ret);

	ret = sysfs_create_group(&pdev->dev.kobj, &attr_group);
	if (ret) {
		pr_err("%s() failed to sysfs_create_group: ret=%d\n",
			 __func__, ret);
		return ret;
	}

	pm_runtime_enable(&pdev->dev);

	/* .../power/control = on and call runtime_resume */
	pm_runtime_forbid(&pdev->dev);

	return 0;
}

static int sony_video_selector_remove(struct platform_device *pdev)
{
	print_trace("%s()\n", __func__);

	if (pm_runtime_active(&pdev->dev)) {
		pr_err("%s() runtime still active\n", __func__);
		pm_runtime_allow(&pdev->dev);
	}
	pm_runtime_disable(&pdev->dev);

	sysfs_remove_group(&pdev->dev.kobj, &attr_group);

	return 0;
}

static int sony_video_selector_suspend_common(struct device *dev)
{
	int ret;
	struct sony_video_selector_priv *priv = dev_get_drvdata(dev);

	print_trace("%s()\n", __func__);

	ret = set_state(priv, 0);
	if (ret < 0) {
		pr_err("%s() failed to set suspend state ret=%d\n",
			__func__, ret);
	}

	ret = regulator_disable(priv->power);
	if (ret)
		pr_err("%s() failed to disable regulator ret=%d\n",
			__func__, ret);

	return ret;
}

static int sony_video_selector_resume_common(struct device *dev)
{
	int ret;
	struct sony_video_selector_priv *priv = dev_get_drvdata(dev);

	print_trace("%s()\n", __func__);

	ret = regulator_enable(priv->power);
	if (ret) {
		pr_err("%s() failed to enable regulator ret=%d\n",
			__func__, ret);
		return ret;
	}

	usleep_range(19000, 20000);
	ret = set_state(priv, priv->state);
	if (ret)
		pr_err("%s() failed to set resume state ret=%d\n",
			__func__, ret);

	return 0;
}

static int sony_video_selector_suspend(struct device *dev)
{
	int ret = 0;

	print_trace("%s()\n", __func__);

	if (pm_runtime_active(dev)) {
		ret = sony_video_selector_suspend_common(dev);
		if (ret < 0)
			pr_err("%s() failed to set suspend\n", __func__);
	}

	return ret;
}

static int sony_video_selector_resume(struct device *dev)
{
	int ret = 0;

	print_trace("%s()\n", __func__);

	if (pm_runtime_active(dev)) {
		ret = sony_video_selector_resume_common(dev);
		if (ret)
			pr_err("%s() failed to set resume\n", __func__);
	}

	return ret;
}

static int sony_video_selector_runtime_suspend(struct device *dev)
{
	int ret;

	print_trace("%s()\n", __func__);

	ret = sony_video_selector_suspend_common(dev);
	if (ret)
		pr_err("%s() failed to set runtime suspend\n", __func__);

	return ret;
}

static int sony_video_selector_runtime_resume(struct device *dev)
{
	int ret;

	print_trace("%s()\n", __func__);

	ret = sony_video_selector_resume_common(dev);
	if (ret)
		pr_err("%s() failed to set runtime resume\n", __func__);

	return ret;
}


static void sony_video_selector_shutdown(struct platform_device *pdev)
{
	print_trace("%s()\n", __func__);

	sony_video_selector_suspend(&pdev->dev);
}

static const struct dev_pm_ops sony_video_selector_pm_ops = {
	.suspend = sony_video_selector_suspend,
	.resume = sony_video_selector_resume,
	.runtime_suspend = sony_video_selector_runtime_suspend,
	.runtime_resume = sony_video_selector_runtime_resume,
};

static const struct of_device_id sony_video_selector_ids[] = {
	{.compatible = "sony,video_selector",},
	{}
};

static struct platform_driver sony_video_selector_driver = {
	.driver = {
		.name = "sony-video-selector",
		.owner = THIS_MODULE,
		.pm = &sony_video_selector_pm_ops,
		.of_match_table = sony_video_selector_ids,
	},
	.probe = sony_video_selector_probe,
	.remove = sony_video_selector_remove,
	.shutdown = sony_video_selector_shutdown
};

module_platform_driver(sony_video_selector_driver);

/* Module information */
MODULE_DESCRIPTION("sony video selector driver");
MODULE_AUTHOR("Sony Corporation");
MODULE_LICENSE("GPL");
MODULE_VERSION("v1.00");
