// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright 2021, 2022 Sony Corporation
 */

#include <linux/err.h>
#include <linux/i2c.h>
#include <linux/module.h>
#include <linux/mux/driver.h>
#include <linux/property.h>
#include <linux/delay.h>

#define NJW1327_NUM_OF_STATE	(ARRAY_SIZE(trans_table))
#define NJW1327_MAX_STATE	(NJW1327_NUM_OF_STATE - 1)
#define NJW1327_NUM_OF_REG	(5)
#define NJW1327_REG_00H		(0x00)
#define NJW1327_PWRSAV_STATE	(0)
#define NJW1327_MUTE_STATE	(1)
#define MASK_VOUT1		(0xf0)
#define MASK_VOUT2		(0x0f)
#define MUTE_SET_VOUT1		(0x0f)
#define MUTE_SET_VOUT2		(0xf0)
#define MUTE_INTERVAL		(200)	/* ms */

#undef TRACE_PRINT_ON
#define TRACE_TAG "####### "

/* trace print macro */
#ifdef TRACE_PRINT_ON
	#define print_trace(fmt, args...) pr_info(TRACE_TAG "" fmt, ##args)
#else
	#define print_trace(fmt, args...) {}
#endif

struct njw1327_priv {
	int current_state;
	int target_state;
	bool flag_muting_vout1;
	bool flag_muting_vout2;
	struct i2c_client *i2c;
	struct workqueue_struct *wq;
	struct delayed_work work;
};

struct reg {
	u8	data[NJW1327_NUM_OF_REG];
};

const struct reg trans_table[] = {
	{ {0x00, 0x00, 0x00, 0x00, 0x00} },
	{ {0x00, 0x00, 0x00, 0x02, 0x00} },
	{ {0x70, 0x00, 0x00, 0x02, 0x00} },
	{ {0x60, 0x00, 0x00, 0x02, 0x00} },
	{ {0x07, 0x00, 0x00, 0x02, 0x00} },
	{ {0x77, 0x00, 0x00, 0x02, 0x00} },
	{ {0x67, 0x00, 0x00, 0x02, 0x00} },
	{ {0x06, 0x00, 0x00, 0x02, 0x00} },
	{ {0x76, 0x00, 0x00, 0x02, 0x00} },
	{ {0x66, 0x00, 0x00, 0x02, 0x00} }
};

static int send_i2c_data(const struct i2c_client *i2s, const u8 *data, int len)
{
	int i, ret;

	for (i = 0; i < len; i++) {
		ret = i2c_smbus_write_byte_data(i2s, i, data[i]);
		if (ret != 0)
			break;
	}

	return ret;
}

static int set_target_state(struct njw1327_priv *priv)
{
	int ret;

	print_trace("%s()\n", __func__);

	ret = send_i2c_data(priv->i2c,
		trans_table[priv->target_state].data, NJW1327_NUM_OF_REG);
	if (ret == 0) {
		priv->flag_muting_vout1 = 0;
		priv->flag_muting_vout2 = 0;
		priv->current_state = priv->target_state;
	} else {
		dev_err(&priv->i2c->dev,
			"failed to change state. err=%d\n", ret);
	}

	return ret;
}

static void sequence_work_func(struct work_struct *work)
{
	struct delayed_work *dwork = to_delayed_work(work);
	struct njw1327_priv *priv = container_of(
		dwork, struct njw1327_priv, work);

	print_trace("%s()\n", __func__);

	(void)set_target_state(priv);
}

static int set_muting(struct njw1327_priv *priv)
{
	int vout1_target, vout1_current, vout2_target, vout2_current;
	struct reg tmp = trans_table[priv->current_state];
	int ret = 0;

	print_trace("%s()\n", __func__);

	/* Muting is applied only changed vout */
	vout1_target  = trans_table[priv->target_state].data[0]  & MASK_VOUT1;
	vout2_target  = trans_table[priv->target_state].data[0]  & MASK_VOUT2;
	vout1_current = trans_table[priv->current_state].data[0] & MASK_VOUT1;
	vout2_current = trans_table[priv->current_state].data[0] & MASK_VOUT2;

	if ((vout1_current != vout1_target) || priv->flag_muting_vout1) {
		print_trace("%s() vout1 muting\n", __func__);
		tmp.data[0] = tmp.data[0] & MUTE_SET_VOUT1;
		priv->flag_muting_vout1 = 1;
	}

	if ((vout2_current != vout2_target) || priv->flag_muting_vout2) {
		print_trace("%s() vout2 muting\n", __func__);
		tmp.data[0] = tmp.data[0] & MUTE_SET_VOUT2;
		priv->flag_muting_vout2 = 1;
	}

	if (priv->flag_muting_vout1 || priv->flag_muting_vout2)
		ret = send_i2c_data(priv->i2c, tmp.data, NJW1327_NUM_OF_REG);

	return ret;
}

static int njw1327_set(struct mux_control *mux, int state)
{
	struct njw1327_priv *priv = mux_chip_priv(mux->chip);
	int ret = 0;

	print_trace("%s() state=%d\n", __func__, state);

	switch (state) {
	case MUX_IDLE_AS_IS:
		return 0;
	case 0 ... NJW1327_MAX_STATE:
		break;
	default:
		dev_err(&mux->chip->dev, "invalid state %d\n", state);
		return -EINVAL;
	}

	priv->flag_muting_vout1 = 0;
	priv->flag_muting_vout2 = 0;
	priv->current_state = priv->target_state;
	cancel_delayed_work_sync(&priv->work);
	priv->target_state = state;
	if ((state == NJW1327_PWRSAV_STATE) || (state == NJW1327_MUTE_STATE)) {
		ret = set_target_state(priv);
	} else {
		ret = set_muting(priv);
		if (ret == 0)
			queue_delayed_work(priv->wq, &priv->work,
				msecs_to_jiffies(MUTE_INTERVAL));
	}

	return ret;
}

static const struct mux_control_ops njw1327_ops = {
	.set = njw1327_set,
};

static int njw1327_probe(struct i2c_client *i2c, const struct i2c_device_id *id)
{
	struct device *dev = &i2c->dev;
	struct mux_chip *mux_chip;
	struct mux_control *mux;
	struct njw1327_priv *priv;
	s32 idle_state;
	int ret;

	print_trace("%s()\n", __func__);

	if (!i2c_check_functionality(i2c->adapter, I2C_FUNC_SMBUS_BLOCK_DATA))
		return -ENODEV;

	mux_chip = devm_mux_chip_alloc(dev, 1, sizeof(struct njw1327_priv));
	if (IS_ERR(mux_chip))
		return PTR_ERR(mux_chip);

	priv = mux_chip_priv(mux_chip);
	priv->current_state = 0;
	priv->target_state = 0;
	priv->flag_muting_vout1 = 0;
	priv->flag_muting_vout2 = 0;
	priv->i2c = i2c;
	priv->wq = create_singlethread_workqueue("njw1327_wq");
	if (!priv->wq)
		return -ENOMEM;
	INIT_DELAYED_WORK(&priv->work, sequence_work_func);

	mux_chip->ops = &njw1327_ops;

	mux = mux_chip->mux;
	mux->states = NJW1327_NUM_OF_STATE;

	ret = device_property_read_u32(dev, "idle-state", (u32 *)&idle_state);
	if (ret < 0)
		idle_state = MUX_IDLE_AS_IS;

	switch (idle_state) {
	case MUX_IDLE_AS_IS:
	case 0 ... NJW1327_MAX_STATE:
		mux->idle_state = idle_state;
		break;
	default:
		dev_err(dev, "invalid idle-state %d\n", idle_state);
		ret = -EINVAL;
		goto out1;
	}

	ret = devm_mux_chip_register(dev, mux_chip);
	if (ret < 0)
		goto out1;

	return 0;
out1:
	destroy_workqueue(priv->wq);
	return ret;
}

static const struct i2c_device_id njw1327_id[] = {
	{ .name = "njw1327", },
	{ }
};
MODULE_DEVICE_TABLE(i2c, njw1327_id);

static const struct of_device_id njw1327_of_match[] = {
	{ .compatible = "njr,njw1327", },
	{ }
};
MODULE_DEVICE_TABLE(of, njw1327_of_match);

static struct i2c_driver njw1327_driver = {
	.driver		= {
		.name		= "njw1327 driver",
		.of_match_table = of_match_ptr(njw1327_of_match),
	},
	.probe		= njw1327_probe,
	.id_table	= njw1327_id,
};
module_i2c_driver(njw1327_driver);

MODULE_DESCRIPTION("NJW1327 driver");
MODULE_AUTHOR("Sony Corporation");
MODULE_LICENSE("GPL");
MODULE_VERSION("v1.00");
