/* SPDX-License-Identifier: GPL-2.0 */
/*
 * ifcon_drv.c - IFCON Communication Driver
 *
 * Copyright 2017, 2019, 2020, 2021, 2022 Sony Corporation
 *
 */

#include <linux/module.h>
#include <linux/cdev.h>
#include <linux/kernel.h>
#include <linux/ktime.h>
#include <linux/delay.h>
#include <linux/poll.h>
#include <linux/freezer.h>
#include <linux/dma-mapping.h>
#include <linux/spi/spi.h>
#include <linux/device.h>
#include <linux/gpio.h>
#include <linux/types.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/of_gpio.h>
#include <linux/random.h>
#include <linux/sched/signal.h>
#include <linux/pm_runtime.h>
#include <linux/string.h>
#include <linux/ratelimit.h>

#include "ifcon_drv.h"

//============================
/* For SPI Config	*/
#define IFCON_DRV_SPI_SPEED_HZ		(4000000)
#define IFCON_DRV_SPI_BITS		(32)

/* transfer cycle (ms) */
#define IFCON_DRV_NORMAL_CYCLE_MS	(20)
#define IFCON_DRV_TX_ARRIVAL_TIME	(60)	/* Measured value (ms) */

/* For Endian Config */
	/* use byte order swap */
#define CONFIG_BYTE_SWAP		(1)

//============================
#define GPIO_IFCON_BD_REQ_PIN		(ifcon_drv_dev->gpio_bd_req)
#define GPIO_IFCON_RESET_PIN		(ifcon_drv_dev->gpio_if_reset)
#define GPIO_WATCHDOG_PIN		(ifcon_drv_dev->gpio_watchdog)

#define GPIO_OUT_ZERO			(0)
#define GPIO_OUT_ONE			(1)

//============================
/* For Test		*/
#define TEST_MODE			(0)

/* For Debug	*/
#define DEBUG_LOG			(0)
#define DEBUG_RINGEDGE			(0)

//============================
#define DUMP_LIMIT_INTERVAL		(5 * HZ)
#define DUMP_LIMIT_BURST		(20)
#define LIMIT_INTERVAL			(10 * 60 * HZ)
#define LIMIT_BURST			(10)

#define LOG_ERR_LIMITED(fmt, ...)					\
do {									\
	static DEFINE_RATELIMIT_STATE(_rs, LIMIT_INTERVAL, LIMIT_BURST);\
	if (__ratelimit(&_rs))						\
		pr_err(fmt, ##__VA_ARGS__);				\
} while (0)
#define LOG_WAR_LIMITED(fmt, ...)					\
do {									\
	static DEFINE_RATELIMIT_STATE(_rs, LIMIT_INTERVAL, LIMIT_BURST);\
	if (__ratelimit(&_rs))						\
		pr_warn(fmt, ##__VA_ARGS__);				\
} while (0)
#define LOG_INFO_LIMITED(fmt, ...)					\
do {									\
	static DEFINE_RATELIMIT_STATE(_rs, LIMIT_INTERVAL, LIMIT_BURST);\
	if (__ratelimit(&_rs))						\
		pr_info(fmt, ##__VA_ARGS__);				\
} while (0)

#define LOG_DEV_ERR(fmt, args...) LOG_ERR_LIMITED("%s: [%s] " fmt,\
				IFCON_DRV_DEVNAME, __func__, ##args)
#define LOG_DEV_WAR(fmt, args...) LOG_WAR_LIMITED("%s: " fmt,\
				IFCON_DRV_DEVNAME, ##args)
#define LOG_DEV_INFO(fmt, args...) LOG_INFO_LIMITED("%s: " fmt,\
				IFCON_DRV_DEVNAME, ##args)
#define LOG_DEV_DBG(fmt, args...) pr_debug("%s: " fmt,\
				IFCON_DRV_DEVNAME, ##args)
#define LOG_INFO(fmt, args...) pr_info("%s: " fmt,\
				IFCON_DRV_DEVNAME, ##args)
#define LOG_ERR(fmt, args...) pr_err("%s: [%s] " fmt,\
				IFCON_DRV_DEVNAME, __func__, ##args)
#define LOG_PR_INFO(fmt, args...) pr_info(fmt, ##args)
#if DEBUG_LOG
#define LOG_DBG(fmt, args...) pr_info(fmt, ##args)
#define LOG_V(fmt, args...) pr_info(fmt, ##args)
#else
#define LOG_DBG(fmt, args...) {}
#define LOG_V(fmt, args...) {}
#endif


//============================

#define IFCON_DRV_MINORS		(32) /* total number of minor number */

#define IFCON_DRV_RING_BUFF_SIZE	(4096) /* size of ring buffer */

/* Self status */
#define IFCON_DRV_CTRL_START_CONDITION		(0x50)
#define IFCON_DRV_CTRL_NORMAL			(0x55)
#define IFCON_DRV_CTRL_RESENDING		(0x5E)
#define IFCON_DRV_CTRL_RESEND_REQ		(0x5F)
#define IFCON_DRV_CTRL_BROKEN_PACKET		(0xFF)
#define IFCON_DRV_CTRL_INVALID			(0x00)
/* Another side status */
#define IFCON_DRV_RX_START_CONDITION		(0x30)
#define IFCON_DRV_RX_NORMAL			(0x33)
#define IFCON_DRV_RX_RESENDING			(0x3E)
#define IFCON_DRV_RX_RESEND_REQUEST		(0x3F)
/* Flow ctrl */
#define IFCON_DRV_FLOW_FOR_CHECK		(0x01)
#define IFCON_DRV_FLOW_NOT_FULL			(0x00)
#define IFCON_DRV_FLOW_NEAR_FULL		(0x80)
#define IFCON_DRV_FLOW_FULL			(0xC0)
#define IFCON_DRV_FLOW_PACKET_LOST		(0xE0)
#define IFCON_DRV_FLOW_ALL			(0xF0)
/* Counter */
#define IFCON_DRV_COUNTER_MAX			(255)
#define IFCON_DRV_WARN_COUNT_MAX		(20)

/* Module setup status */
#define IFCON_DRV_SETUP_NG			(0)
#define IFCON_DRV_SETUP_OK			(1)

/* Module status */
#define IFCON_DRV_STAT_INIT			(0)
#define IFCON_DRV_STAT_CONNECT			(1)

/* Suspend status */
#define IFCON_DRV_POWER_NORMAL			(0)
#define IFCON_DRV_POWER_SUSPEND			(1)

/* If communication reset */
#define IFCON_DRV_READ_NO_BLOCK			(0)
#define IFCON_DRV_READ_BLOCKING			(1)

/* display frame data */
#define IFCON_DRV_SEL_RX			(0)
#define IFCON_DRV_SEL_TX			(1)
#define IFCON_DRV_SEL_TX_RX			(2)

/* ifcon_hw_reset_pin status */
enum ifcon_hw_reset_pin_status {
	IFCON_DRV_HW_RESET_OFF,
	IFCON_DRV_HW_RESET_ON,
	IFCON_DRV_HW_RESET_DISABLE,
	IFCON_DRV_HW_RESET_ONESHOT
};

/* ifcon_bd_req_pin status */
enum ifcon_bd_req_pin_status {
	IFCON_DRV_BD_REQ_OFF,
	IFCON_DRV_BD_REQ_ON,
};

//============================
#define IFCON_DRV_LOSS_OUT_COUNT	(100)		/* 100 times */

#define IFCON_DRV_RESERVED_SIZE		(5)
#define IFCON_DRV_T1_RESERVED_SIZE	(2)

#define IFCON_DRV_NEAR_FULL_SIZE	(2 * ifcon_drv_cur_total_data_size)

/* Module open wait time for probe */
#define IFCON_DRV_OPEN_WAIT		(100 * HZ / 1000)/* 100ms cycle */
#define IFCON_DRV_OPEN_WAIT_CNT		(50)		/* timeout count */

#define IFCON_DRV_SUSPEND_TIMEOUT	(IFCON_DRV_NORMAL_CYCLE_MS*5)

static DECLARE_BITMAP(minors, IFCON_DRV_MINORS);

struct ifcon_drv_data_t {
	dev_t			devt;
	dev_t			major;
	struct spi_device	*spi;
	struct cdev		cdev;
	struct class		*class;
	struct device		*dev;
	struct task_struct	*kthread_tsk;
	struct mutex		state_lock;
	wait_queue_head_t	rxwait;
	unsigned int		users;
	u8			count_stat;
	u8			count_flow;

	int			gpio_bd_req;
	int			gpio_if_reset;
	int			gpio_watchdog;
	u32			max_freq;
};

static struct ifcon_drv_data_t *ifcon_drv_dev;
static int ifcon_drv_setup_status;

/* Tx/Rx SPI com frame */
struct ifcon_drv_frame {
	/* common control area */
	u8	status;		/* status */
	u8	flow_ctrl;	/* flow control */
	u8	count;		/* Communication Counter[0-255] */
	u8	frame_type;	/* Frame type */
	u8	reserved[IFCON_DRV_T1_RESERVED_SIZE];
	u16	ctrl_check_sum;	/* Check Sum of up to here */
	u8	fixed_data[IFCON_DRV_FIXED_DATA_SIZE];	/* Fixed data area */
	union {
		struct frame_type0 { /* Frame type NORMAL */
			u8	data[IFCON_DRV_RW_DATA_SIZE];
			u16	check_sum;	/* Check sum */
		} t0;
		struct frame_type1 { /* Frame type LARGE  */
			u8	data[IFCON_DRV_T1_RW_DATA_SIZE];
			u16	check_sum;	/* Check sum */
		} t1;
	};
};
#define IFCON_SPI_FRAME_SIZE	(sizeof(struct ifcon_drv_frame))

static struct ifcon_drv_frame *tx_frame;	/* Access for SPI register */
static struct ifcon_drv_frame *tmp_frame;	/* Access for SPI register */
static struct ifcon_drv_frame *rx_frame;	/* Access for SPI register */
static dma_addr_t tx_phyadr;
static dma_addr_t tmp_phyadr;

/* Tx/Rx ring buffer */
struct ifcon_drv_ring_buffer {
	u8	buffer[IFCON_DRV_RING_BUFF_SIZE];
	int	rptr;		/* Read pointer for ring buffer */
	int	wptr;		/* Write pointer for ring buffer */
};

static struct ifcon_drv_ring_buffer tx_ring;
static struct ifcon_drv_ring_buffer rx_ring;
static	int	tx_xptr;	/* Tx - transmit pointer, for resend used */
static	int	tx_vptr;	/* Tx - verify pointer, for resend request */

static int ifcon_drv_status = IFCON_DRV_STAT_INIT;
static int ifcon_drv_power = IFCON_DRV_POWER_SUSPEND;
static int ifcon_drv_rblock = IFCON_DRV_READ_NO_BLOCK;
static int ifcon_drv_boot;	/* booted flag */
static int ifcon_drv_loss;	/* Not received count */

/* What will be finalized when opened */
static int ifcon_drv_cur_frame_type;
static int ifcon_drv_cur_rx_frame_type;
static int ifcon_drv_cur_total_data_size = IFCON_DRV_RW_DATA_SIZE;
static int ifcon_drv_cur_frame_all_size = IFCON_DRV_FRAME_BASE_SIZE
					+ IFCON_DRV_RW_DATA_SIZE;

static void ifcon_mem_dump(char *title, char *p_mem, int size);


/*--- Module parameters -----------------------------------------------------*/

/* echo 1 > /sys/module/ifcon_drv/parameters/ifcon_status */
static int ifcon_status;
module_param_named(ifcon_status, ifcon_status, int, 0644);
MODULE_PARM_DESC(ifcon_status,
	"if set 1 then dump ifcon internal status");

/* tx arrival time from write_func to IFCON */
unsigned short tx_arrival_time = IFCON_DRV_TX_ARRIVAL_TIME;
module_param(tx_arrival_time, ushort, 0644);

unsigned short cycle_msec_val = IFCON_DRV_NORMAL_CYCLE_MS;
module_param(cycle_msec_val, ushort, 0644);

#define IFCON_DRV_CYCLE_MSEC_AVG_T	(10000) /* Average every 10000ms */
long cycle_msec_avg;
module_param(cycle_msec_avg, long, 0644);

long cycle_msec_max;
module_param(cycle_msec_max, long, 0644);

static void ifcon_hw_reset_on(void);
static void ifcon_hw_reset_off(void);
static void ifcon_hw_reset(void);
static int ifcon_hw_reset_pin = IFCON_DRV_HW_RESET_OFF;
int param_set_ifcon_hw_reset(const char *val, const struct kernel_param *kp)
{
	if (!val)
		return -EINVAL;
	if (!strncmp(val, "oneshot", 7)) {
		ifcon_hw_reset_pin = IFCON_DRV_HW_RESET_OFF;
		ifcon_hw_reset();
	} else if (!strncmp(val, "on", 2)) {
		ifcon_hw_reset_pin = IFCON_DRV_HW_RESET_ON;
		ifcon_hw_reset_on();
	} else if (!strncmp(val, "off", 3)) {
		ifcon_hw_reset_pin = IFCON_DRV_HW_RESET_OFF;
		ifcon_hw_reset_off();
	} else if (!strncmp(val, "disable", 7)) {
		ifcon_hw_reset_pin = IFCON_DRV_HW_RESET_DISABLE;
		ifcon_hw_reset_off();
	}
	return 0;
}
static int param_get_ifcon_hw_reset(char *buff, const struct kernel_param *kp)
{
	int ret;

	switch (ifcon_hw_reset_pin) {
	case IFCON_DRV_HW_RESET_OFF:
		ret = snprintf(buff, 5, "off\n");
		break;
	case IFCON_DRV_HW_RESET_ON:
		ret = snprintf(buff, 4, "on\n");
		break;
	case IFCON_DRV_HW_RESET_DISABLE:
		ret = snprintf(buff, 9, "disable\n");
		break;
	default:
		ret = -EINVAL;
		break;
	}
	return ret;
}
static const struct kernel_param_ops param_ops_ifcon_hw_reset = {
	.set = param_set_ifcon_hw_reset,
	.get = param_get_ifcon_hw_reset,
};
module_param_cb(ifcon_hw_reset_pin,
	&param_ops_ifcon_hw_reset, &ifcon_hw_reset_pin, 0644);

static void ifcon_bd_req_on(void);
static void ifcon_bd_req_off(void);
static int ifcon_bd_req_pin = IFCON_DRV_BD_REQ_OFF;
int param_set_ifcon_bd_req(const char *val, const struct kernel_param *kp)
{
	if (!val)
		return -EINVAL;
	if (!strncmp(val, "on", 2))
		ifcon_bd_req_on();
	else if (!strncmp(val, "off", 3))
		ifcon_bd_req_off();

	return 0;
}
static int param_get_ifcon_bd_req(char *buff, const struct kernel_param *kp)
{
	int ret;

	switch (ifcon_bd_req_pin) {
	case IFCON_DRV_BD_REQ_OFF:
		ret = snprintf(buff, 5, "off\n");
		break;
	case IFCON_DRV_BD_REQ_ON:
		ret = snprintf(buff, 4, "on\n");
		break;
	default:
		ret = -EINVAL;
		break;
	}
	return ret;
}
static const struct kernel_param_ops param_ops_ifcon_bd_req = {
	.set = param_set_ifcon_bd_req,
	.get = param_get_ifcon_bd_req,
};
module_param_cb(ifcon_bd_req_pin,
	&param_ops_ifcon_bd_req, &ifcon_bd_req_pin, 0644);

ulong transfers;
module_param(transfers, ulong, 0644);

ulong rx_errors;
module_param(rx_errors, ulong, 0644);

int rx_remain_bytes;
module_param(rx_remain_bytes, int, 0644);

int tx_wait_send_bytes;
module_param(tx_wait_send_bytes, int, 0644);

/* setting frame type */
int frame_type;
module_param(frame_type, int, 0644);

/* rx frame type */
int rx_frame_type;
module_param(rx_frame_type, int, 0444);

/* dump_mode */
#define DUMP_MODE_W		0x0001	/* write dump          */
#define DUMP_MODE_R		0x0002	/* read  dump          */
#define DUMP_MODE_SPKT		0x0004	/* snd packet all dump */
#define DUMP_MODE_RPKT		0x0008	/* rcv packet all dump */
#define DUMP_MODE_SPKT_P1	0x0010	/* snd data part dump  */
#define DUMP_MODE_RPKT_P1	0x0020	/* rcv data part dump  */
#define DUMP_MODE_ERR		0x8000	/* error dump          */
int dump_mode;
module_param(dump_mode, int, 0644);

#if TEST_MODE
#define TEST_MODE_TX		0x0001	/* Tx test randomly destroy */
#define TEST_MODE_RX		0x0002	/* Rx test randomly destroy */
int test_mode;
module_param(test_mode, int, 0644);
#endif

/*--- Byte order swap -------------------------------------------------------*/
static inline void ifcon_drv_cpu_to_be32(void *dst, const void *src,
	__kernel_size_t len)
{
	#if CONFIG_BYTE_SWAP
		u32 *p32dst;
		const u32 *p32src;

		p32dst = (u32 *)dst;
		p32src = (const u32 *)src;
		while (len >= sizeof(u32)) {
			*p32dst++ = cpu_to_be32(*p32src++);
			len -= sizeof(u32);
		}
	#endif
}
static inline void ifcon_drv_be32_to_cpu(void *dst, const void *src,
	__kernel_size_t len)
{
	#if CONFIG_BYTE_SWAP
		u32 *p32dst;
		const u32 *p32src;

		p32dst = (u32 *)dst;
		p32src = (const u32 *)src;
		while (len >= sizeof(u32)) {
			*p32dst++ = cpu_to_be32(*p32src++);
			len -= sizeof(u32);
		}
	#endif
}

/*--- frame get / set util -----------------------------------------*/

static u16 get_frame_size(void)
{
	u16 size;

	size = (ifcon_drv_cur_frame_type == IFCON_DRV_FRAME_TYPE1)
		? IFCON_DRV_T1_FRAME_ALL_SIZE : IFCON_DRV_FRAME_ALL_SIZE;
	return size;
}

static u8 *get_frame_data_p(struct ifcon_drv_frame *frame)
{
	u8 *p;

	p = (ifcon_drv_cur_frame_type == IFCON_DRV_FRAME_TYPE1) ?
		&frame->t1.data[0] : &frame->t0.data[0];

	return p;
}

static void clear_frame_data(struct ifcon_drv_frame *frame)
{
	if (ifcon_drv_cur_frame_type == IFCON_DRV_FRAME_TYPE1)
		memset(frame->t1.data, 0, sizeof(frame->t1.data));
	else
		memset(frame->t0.data, 0, sizeof(frame->t0.data));

}

static u16 get_idx_package_size(struct ifcon_drv_frame *frame, int idx)
{
	u16 size;

	if (ifcon_drv_cur_frame_type == IFCON_DRV_FRAME_TYPE1) {
		size = frame->t1.data[idx]
		 + (((u16)(frame->t1.data[idx + 1])) << 8);
	} else
		size = frame->t0.data[idx];

	return size;
}

static u16 get_ring_package_size(char *buf, int idx)
{
	u16 size;

	if (idx + 1 < IFCON_DRV_RING_BUFF_SIZE) {
		size = (u16)buf[idx] + (((u16)buf[idx + 1]) << 8);
		//LOG_INFO("ring_size:%04x\n", size);
	} else {
		size = (u16)buf[idx] + (((u16)buf[0]) << 8);
	}
	return size;
}

static inline u16 get_buff_size(const char __user *buf)
{
	u16 size;

	size = (((u16)buf[1]) << 8) + buf[0];

	return size;
}

/*--- Check sum --------------------------------------------*/
static u16 calc_check_sum(u8 *buf)
{
	int i = 0;
	int size = ifcon_drv_cur_frame_all_size - IFCON_DRV_CHECK_SUM_SIZE;
	u16 sum = 0;

	for (i = 0; i < size; i++)
		sum += buf[i];
	/* complement on two */
	sum = ~sum + 1;
	/* check sum value */
	return sum;
}

static inline int compare_check_sum(u8 *buf, u16 checksum)
{
	int status = IFCON_DRV_OK;
	u16 sum = calc_check_sum(buf);
	/* checksum value check */
	if (checksum != sum) {
		LOG_DEV_ERR("check sum error! 0x%x -> calc: 0x%x\n",
							checksum, sum);
		status = IFCON_DRV_NG;
	}
	return status;
}

static u16 get_check_sum(struct ifcon_drv_frame *frame)
{
	u16 checksum;

	checksum = (ifcon_drv_cur_frame_type == IFCON_DRV_FRAME_TYPE1)
		? frame->t1.check_sum : frame->t0.check_sum;
	return checksum;
}

static void set_check_sum(struct ifcon_drv_frame *frame)
{
	u16 checksum;

	checksum = calc_check_sum((u8 *)frame);
	if (ifcon_drv_cur_frame_type == IFCON_DRV_FRAME_TYPE1)
		frame->t1.check_sum = checksum;
	else
		frame->t0.check_sum = checksum;
}

/*--- Dump Tx/Rx buffer --------------------------------------------*/

/* for error case */
static void display_frame_data(u8 mode)
{
	if ((mode == IFCON_DRV_SEL_TX_RX) || (mode == IFCON_DRV_SEL_TX))
		ifcon_mem_dump("TX frame:",
			(char *)tx_frame, ifcon_drv_cur_frame_all_size);
	if ((mode == IFCON_DRV_SEL_TX_RX) || (mode == IFCON_DRV_SEL_RX))
		ifcon_mem_dump("RX frame:",
			(char *)rx_frame, ifcon_drv_cur_frame_all_size);
}

static void warn_rx_status(s8 *str)
{
	if (ifcon_drv_dev->count_stat < IFCON_DRV_WARN_COUNT_MAX) {
		ifcon_drv_dev->count_stat++;
		LOG_DEV_WAR("%s rx status:0x%02x size:0x%03x cnt:0x%02x\n", str,
		rx_frame->status, get_idx_package_size(rx_frame, 0),
							rx_frame->count);
	}
}

static void warn_rx_tx_flow(s8 *str)
{
	if (ifcon_drv_dev->count_flow < IFCON_DRV_WARN_COUNT_MAX) {
		ifcon_drv_dev->count_flow++;
		LOG_DEV_WAR("%s flow rx:0x%02x tx:0x%02x cnt:0x%02x\n", str,
		    rx_frame->flow_ctrl, tx_frame->flow_ctrl, rx_frame->count);
	}
}

static void display_internal_state(void)
{
	LOG_INFO("ifcon_drv_status = %02x\n", ifcon_drv_status);
	LOG_INFO("ifcon_drv_power  = %02x\n", ifcon_drv_power);
	LOG_INFO("ifcon_drv_rblock = %02x\n", ifcon_drv_rblock);
	LOG_INFO("rx_frame.status  = %02x\n", rx_frame->status);
	LOG_INFO("rx_frame.flow_.. = %02x\n", rx_frame->flow_ctrl);
	LOG_INFO("rx_wptr:%03x rx_rptr:%03x\n", rx_ring.wptr, rx_ring.rptr);
	LOG_INFO("tx_frame.status  = %02x\n", tx_frame->status);
	LOG_INFO("tx_frame.flow_.. = %02x\n", tx_frame->flow_ctrl);
	LOG_INFO("tx_wptr:%03x tx_rptr:%03x\n", tx_ring.wptr, tx_ring.rptr);
	LOG_INFO("tx_xptr:%03x tx_vptr:%03x\n", tx_xptr, tx_vptr);

	ifcon_mem_dump("RX_RING:", rx_ring.buffer, IFCON_DRV_RING_BUFF_SIZE);
	ifcon_mem_dump("TX_RING:", tx_ring.buffer, IFCON_DRV_RING_BUFF_SIZE);
}

static void ifcon_mem_dump(char *title, char *p_mem, int size)
{
	int i, len, xpos;
	char buf[80];
	static DEFINE_RATELIMIT_STATE(_rs,
			DUMP_LIMIT_INTERVAL, DUMP_LIMIT_BURST);

	if (!__ratelimit(&_rs))
		return;

	len = 0;
	if (title != NULL)
		len += snprintf(&buf[len], 32, "%s", title);
	if (len)
		LOG_PR_INFO("%s: %s\n", IFCON_DRV_DEVNAME, buf);

	len = 0;
	for (i = 0; i < size; i++) {
		xpos = i % 16;
		if (xpos == 0)
			len += snprintf(&buf[len], 7, "[%04x]", i);
		len += snprintf(&buf[len], 4, " %02x", p_mem[i]);
		if (xpos == 15) {
			LOG_PR_INFO("%s\n", buf);
			len = 0;
		}
	}
	if (len)
		LOG_PR_INFO("%s\n", buf);
}

static void ifcon_display_frame(char *title, struct ifcon_drv_frame *frame)
{
	int i, len;
	char *p_mem;
	char buf[128];
	u16 frame_size;

	frame_size = get_frame_size();
	len = 0;
	if (title != NULL)
		len += snprintf(&buf[len], 16, "%s", title);

	p_mem = (char *)frame;
	/* ctrl area */
	for (i = 0; i < IFCON_DRV_PACKET_CTRL_AREA_SIZE; i++)
		len += snprintf(&buf[len], 4, " %02x", p_mem[i]);
	/* data */
	len += snprintf(&buf[len], 4, "...");
	for (i = IFCON_DRV_PACKET_DATA_OFFSET;
		i < IFCON_DRV_PACKET_DATA_OFFSET + 4; i++) {
		len += snprintf(&buf[len], 4, " %02x", p_mem[i]);
	}
	/* check sum */
	len += snprintf(&buf[len], 4, "...");
	for (i = frame_size - 4; i < frame_size; i++)
		len += snprintf(&buf[len], 4, " %02x", p_mem[i]);
	LOG_DEV_INFO("%s\n", buf);
}

/******************************************************************************/
/*!	Request Resend
 *	The request of Resend is requested from another side.
 *	When word status is IFCON_DRV_RX_RESEND_REQUEST.
 */
/******************************************************************************/
static void resend_req(void)
{
	tx_ring.rptr = tx_vptr;
	tx_xptr = tx_ring.rptr;
}

/******************************************************************************/
/*!	Verifying OK
 *	This is verify the sending data is OK
 *	When word 1 is not 0x5F(resend request)
 */
/******************************************************************************/
static void verify_ok(void)
{
	tx_vptr = tx_xptr;
	tx_xptr = tx_ring.rptr;
}

/* Get flow control code */
static u8 get_flow_code(struct ifcon_drv_ring_buffer *ring)
{
	int rest = 0;

	/* rest writable buffer size */
	if (ring->rptr <= ring->wptr)
		rest = IFCON_DRV_RING_BUFF_SIZE - (ring->wptr - ring->rptr);
	else
		rest = ring->rptr - ring->wptr;
	/* Flow control code */
	if (rest < ifcon_drv_cur_total_data_size)
		return IFCON_DRV_FLOW_FULL;
	else if (rest < IFCON_DRV_NEAR_FULL_SIZE)
		return IFCON_DRV_FLOW_NEAR_FULL;
	else
		return IFCON_DRV_FLOW_NOT_FULL;
}

/*--- Check ctrl area --------------------------------------------*/
static void ifcon_hw_reset_on(void)
{
	gpio_set_value(GPIO_IFCON_RESET_PIN, GPIO_OUT_ONE);
}

static void ifcon_hw_reset_off(void)
{
	gpio_set_value(GPIO_IFCON_RESET_PIN, GPIO_OUT_ZERO);
}

static void ifcon_hw_reset(void)
{
	if (ifcon_hw_reset_pin == IFCON_DRV_HW_RESET_DISABLE) {
		LOG_ERR("Ifcon force reset disable!\n");
	} else {
		LOG_ERR("Ifcon force reset!\n");
		ifcon_hw_reset_on();
		usleep_range(10000, 11000);
		ifcon_hw_reset_off();
	}
}

static void ifcon_bd_req_on(void)
{
	gpio_set_value(GPIO_IFCON_BD_REQ_PIN, GPIO_OUT_ONE);
	LOG_INFO("GPIO_IFCON_BD_REQ_PIN=1\n");
	ifcon_bd_req_pin = IFCON_DRV_BD_REQ_ON;
}

static void ifcon_bd_req_off(void)
{
	gpio_set_value(GPIO_IFCON_BD_REQ_PIN, GPIO_OUT_ZERO);
	LOG_INFO("GPIO_IFCON_BD_REQ_PIN=0\n");
	ifcon_bd_req_pin = IFCON_DRV_BD_REQ_OFF;
}

static u16 calc_ctrl_check_sum(u8 *buf)
{
	int i = 0;
	int size = IFCON_DRV_PACKET_CTRL_AREA_SIZE - IFCON_DRV_CHECK_SUM_SIZE;
	u16 sum = 0;

	for (i = 0; i < size; i++)
		sum += buf[i];
	/* complement on two */
	sum = ~sum + 1;
	/* check sum value */
	return sum;
}

static int ifcon_ctrl_is_valid(void)
{
	int ret = 0;

	/* check status */
	switch (rx_frame->status) {
	case IFCON_DRV_RX_START_CONDITION:
	case IFCON_DRV_RX_NORMAL:
	case IFCON_DRV_RX_RESENDING:
	case IFCON_DRV_RX_RESEND_REQUEST:
		break;
	case IFCON_DRV_CTRL_INVALID:		/* 0x00 */
		/* All 0x00 */
		ret |= 0x01;
		break;
	case IFCON_DRV_CTRL_BROKEN_PACKET:	/* 0xff */
		/* All 0xff */
		ret |= 0x01;
		break;
	default:
		ret |= 0x01;
		break;
	}

	/* check frame type */
	switch (rx_frame->frame_type) {
	case IFCON_DRV_FRAME_TYPE0:
		if (rx_frame->ctrl_check_sum != 0)
			ret |= 0x02;
		break;
	case IFCON_DRV_FRAME_TYPE1:
		if (calc_ctrl_check_sum((u8 *)rx_frame)
						!= rx_frame->ctrl_check_sum)
			ret |= 0x02;
		break;
	default:
		ret |= 0x02;
		break;
	}

	if (ret == 0) {
		if (ifcon_drv_cur_rx_frame_type != rx_frame->frame_type) {
			rx_frame_type = rx_frame->frame_type;
			ifcon_drv_cur_rx_frame_type = rx_frame->frame_type;
		}
		return IFCON_DRV_OK;
	}

	ifcon_display_frame("ctrlNG:", rx_frame);
	return IFCON_DRV_NG;
}

static int check_ctrl_data(void)
{
	int ret = IFCON_DRV_NG;
	u32 total_size = 0;
	u16 size;

	if (ifcon_ctrl_is_valid() != IFCON_DRV_OK) {

		/* If communication reset */
		if ((rx_frame->status == IFCON_DRV_CTRL_INVALID) &&
		    (rx_frame->count == 0) &&
		    (get_idx_package_size(rx_frame, 0) == 0) &&
		    (get_check_sum(rx_frame) == 0)) {
			if (ifcon_drv_boot == 0
			 && ifcon_drv_loss == IFCON_DRV_LOSS_OUT_COUNT) {
				ifcon_hw_reset();
				ifcon_drv_boot = 1;
			}
			ifcon_drv_loss++;
			LOG_DEV_ERR("Ifcon spi reset!");
			tx_frame->status = IFCON_DRV_CTRL_START_CONDITION;
			tx_frame->flow_ctrl = IFCON_DRV_FLOW_NOT_FULL;
			ifcon_drv_status = IFCON_DRV_STAT_INIT;
			ifcon_drv_rblock = IFCON_DRV_READ_NO_BLOCK;
			wake_up_interruptible(&ifcon_drv_dev->rxwait);
			ret = IFCON_DRV_NG;
		} else {
			ifcon_drv_boot = 1;
		}

		return ret;
	}
	if (ifcon_drv_cur_rx_frame_type != ifcon_drv_cur_frame_type) {
		LOG_DEV_WAR("frame type mismatch!");
		return ret;
	}

	while (total_size <=
		(ifcon_drv_cur_total_data_size - IFCON_DRV_CTL_SIZE)) {
		size = get_idx_package_size(rx_frame, total_size);
		if (size > 0)
			total_size += size;
		else
			break;
	}
	/* Check data size */
	if (total_size > ifcon_drv_cur_total_data_size) {
		warn_rx_status("size over!:");
		return ret;
	}

	ret = IFCON_DRV_OK;
	/* Controll Area: status */
	switch (rx_frame->status) {
	case IFCON_DRV_RX_START_CONDITION:
		ifcon_drv_dev->count_stat = 0;
		tx_frame->status = IFCON_DRV_CTRL_NORMAL;
		if (ifcon_drv_status != IFCON_DRV_STAT_CONNECT) {
			ifcon_drv_dev->count_stat = 0;
			LOG_DEV_INFO("Connect frame type=%d",
					ifcon_drv_cur_rx_frame_type);
		}
		ifcon_drv_status = IFCON_DRV_STAT_CONNECT;
		break;
	case IFCON_DRV_RX_NORMAL:
		tx_frame->status = IFCON_DRV_CTRL_NORMAL;
		break;
	case IFCON_DRV_RX_RESENDING:
		tx_frame->status = IFCON_DRV_CTRL_NORMAL;
		warn_rx_status("Resending:");
		break;
	case IFCON_DRV_RX_RESEND_REQUEST:
		tx_frame->status = IFCON_DRV_CTRL_RESENDING;
		warn_rx_status("Resend req:");
		break;
	case IFCON_DRV_CTRL_BROKEN_PACKET:
		tx_frame->status = IFCON_DRV_CTRL_RESEND_REQ;
		ret = IFCON_DRV_NG;	/* discard data. */
		warn_rx_status("Broken!:");
		break;
	default:
		ret = IFCON_DRV_NG;	/* discard data. */
		warn_rx_status("Status Invalid!:");
		break;
	}
	/* Another side status: request resend ? */
	if (ret == IFCON_DRV_OK) {
		if (rx_frame->status == IFCON_DRV_RX_RESEND_REQUEST)
			resend_req();
		else
			verify_ok();
	}
	/* Self status: request resend ? */
	if (tx_frame->status == IFCON_DRV_CTRL_RESEND_REQ)
		ret = IFCON_DRV_NG;	/* discard data. */
	/* invalid data ? */
	if ((rx_frame->flow_ctrl & IFCON_DRV_FLOW_FOR_CHECK) != 0) {
		ret = IFCON_DRV_NG;	/* discard data. */
		warn_rx_status("FLOW_FOR_CHECK!:");
	}
	/* check rx sum */
	if ((ret == IFCON_DRV_OK)
	 && (ifcon_drv_status > IFCON_DRV_STAT_INIT)) {
		ret = compare_check_sum((u8 *)rx_frame,
					get_check_sum(rx_frame));
		if (ret != IFCON_DRV_OK) {
			rx_errors++;
			if (get_idx_package_size(rx_frame, 0) != 0) {
				tx_frame->status =
					IFCON_DRV_CTRL_RESEND_REQ;
				if (dump_mode & DUMP_MODE_ERR)
					display_frame_data(IFCON_DRV_SEL_RX);
			}
		}
	}

	return ret;
}

/*--- Calc remain bytes for ring buffer -----------------------------------*/
static int calc_remain_bytes(int wptr, int rptr)
{
	int rest;

	if (rptr <= wptr)
		rest = IFCON_DRV_RING_BUFF_SIZE - (wptr - rptr);
	else
		rest = rptr - wptr;

	return rest;
}

static int calc_wait_bytes(int wptr, int rptr)
{
	int count;

	if (rptr <= wptr)
		count = wptr - rptr;
	else
		count = IFCON_DRV_RING_BUFF_SIZE - (rptr - wptr);

	return count;
}

/*--- Access for Rx Ring buffer -------------------------------------------*/

/* Read from ring buffer */
static int rx_ring_read(char __user *buf, int count)
{
	int wptr_temp, rptr_temp;
	int sz_total, sz;
	int rest_cap;

	wptr_temp = rx_ring.wptr;
	rptr_temp = rx_ring.rptr;
	sz_total = get_ring_package_size(rx_ring.buffer, rptr_temp);
	sz = sz_total;
	rest_cap = IFCON_DRV_RING_BUFF_SIZE - rptr_temp;

	if (rptr_temp == wptr_temp)
		return 0;
	if ((sz <= 0) || (sz > ifcon_drv_cur_total_data_size)) {
		LOG_DEV_ERR("pointer size error! sz:%d buf:0x%p\n",
					sz, buf);
		return IFCON_DRV_NG;
	}
	if (count < sz) {
		LOG_DEV_ERR("RX_BUFF_SIZE_ERROR sz:%d count:%d\n", sz, count);
		return -IFCON_DRV_RX_BUFF_SIZE_ERROR;
	}
	if (rptr_temp >= IFCON_DRV_RING_BUFF_SIZE) { /* Pointer inside check */
		LOG_DEV_ERR("rx_ring.rptr error! %d\n", wptr_temp);
		return IFCON_DRV_NG;
	}

	/* Read from ring buffer Rx */
	if (sz >= rest_cap) {
		if (rptr_temp < wptr_temp) {
			LOG_DEV_ERR("err! r:%d w:%d\n", rptr_temp, wptr_temp);
			return IFCON_DRV_NG;
		}
#if DEBUG_RINGEDGE
		LOG_INFO("rx rd (sz >= rest_cap)sz:%d rest_cap:%d\n",
							sz, rest_cap);
#endif
		if (copy_to_user(buf, &rx_ring.buffer[rptr_temp], rest_cap)) {
			LOG_DEV_ERR("RX copy_to_user\n");
			return -EFAULT;
		}
		rptr_temp = 0;
		sz -= rest_cap;
		buf = (char *)(buf + rest_cap);
	}
	if (copy_to_user(buf, &rx_ring.buffer[rptr_temp], sz)) {
		LOG_DEV_ERR("RX copy_to_user\n");
		return -EFAULT;
	}
	/* Read Pointer increment */
	barrier();
	rx_ring.rptr = rptr_temp + sz;

	rx_remain_bytes = calc_remain_bytes(rx_ring.wptr, rx_ring.rptr);

	return sz_total;
}

/* Write to ring buffer */
static int rx_ring_write(char *buf, int count)
{
	int rptr_temp, wptr_temp;
	int rest_cap, rest;
	int sz, size;

	rptr_temp = rx_ring.rptr;
	wptr_temp = rx_ring.wptr;
	rest_cap = IFCON_DRV_RING_BUFF_SIZE - wptr_temp;
	sz = count;
	rest = 0;

	size = get_buff_size(buf);
	if (count != size) {
		LOG_DEV_ERR("(c!=sz) error! c:%04x sz:%04x\n", count, size);
		return IFCON_DRV_NG;
	}
	if ((sz <= 0) || (sz > ifcon_drv_cur_total_data_size)) {
		LOG_DEV_ERR("pointer size error! count:%d buf:0x%p\n",
					sz, buf);
		return IFCON_DRV_NG;
	}
	if (wptr_temp >= IFCON_DRV_RING_BUFF_SIZE) { /* Pointer inside check */
		LOG_DEV_ERR("rx_ring.wptr error! %d\n", wptr_temp);
		return IFCON_DRV_NG;
	}

	/* rest writable buffer size */
	if (rptr_temp <= wptr_temp)
		rest = IFCON_DRV_RING_BUFF_SIZE - (wptr_temp - rptr_temp);
	else
		rest = rptr_temp - wptr_temp;
	/* Flow control */
	if (rest < ifcon_drv_cur_total_data_size) {
		LOG_DEV_ERR("BUFF_FULL! size:%d rest:%d\n", sz, rest);
		return -IFCON_DRV_BUFF_FULL;
	}
	/* Write to ring buffer Tx */
	if (sz >= rest_cap) {
#if DEBUG_RINGEDGE
		LOG_INFO("rx wr (sz >= rest_cap)sz:%d rest_cap:%d\n",
							sz, rest_cap);
#endif
		/* Write to end edge */
		memcpy(&rx_ring.buffer[wptr_temp], buf, rest_cap);
		wptr_temp = 0;
		sz -= rest_cap;
		buf = (char *)(buf + rest_cap);
	}
	memcpy(&rx_ring.buffer[wptr_temp], buf, sz);
	/* Write Pointer increment */
	barrier();
	rx_ring.wptr = wptr_temp + sz;

	rx_remain_bytes = calc_remain_bytes(rx_ring.wptr, rx_ring.rptr);

	return count;
}

/* Write to ring buffer Rx */
static int copy_from_frame_to_rx_ring(char *buf)
{
	int sz_total, sz, wr_sz;
	char *buf_next;

	/* Read from ring buffer Tx */
	sz_total = 0;
	while (sz_total <= (ifcon_drv_cur_total_data_size
					- IFCON_DRV_CTL_SIZE)) {
		buf_next = buf + sz_total;
		wr_sz = get_buff_size(buf_next);
		if (wr_sz == 0)
			break;
		if ((sz_total + wr_sz) > ifcon_drv_cur_total_data_size) {
			LOG_DEV_ERR("total size error! :%d\n",
							sz_total + wr_sz);
			break;
		}
		sz = rx_ring_write(buf_next, wr_sz);
		if (sz <= 0)
			break;
		sz_total += sz;
	}

	if (sz < 0) {	/* error case */
		sz_total = sz;
		if (sz != -IFCON_DRV_BUFF_FULL) {
			if (dump_mode & DUMP_MODE_ERR)
				display_frame_data(IFCON_DRV_SEL_RX);
		}
	}

	if (rx_ring.rptr != rx_ring.wptr)
		wake_up_interruptible(&ifcon_drv_dev->rxwait);

	return sz_total;
}

static int get_rx_frame(void)
{
	int status = 0;

	/* Process for loading Rx data */
	status = check_ctrl_data(); /* check rx data */
	/* Write to ring buffer Rx */
	if (status == IFCON_DRV_OK)
		status = copy_from_frame_to_rx_ring(get_frame_data_p(rx_frame));

	return status;
}

/*--- Access for Tx Ring buffer -------------------------------------------*/

/* Read from ring buffer */
static int tx_ring_read(char *buf, int count, int used_sz)
{
	int wptr_temp, rptr_temp;
	int sz_total, sz;
	int rest_cap;

	wptr_temp = tx_ring.wptr;
	rptr_temp = tx_ring.rptr;
	sz_total = get_ring_package_size(tx_ring.buffer, rptr_temp);
	sz = sz_total;
	rest_cap = IFCON_DRV_RING_BUFF_SIZE - rptr_temp;

	if (rptr_temp == wptr_temp)
		return 0;
	if ((sz <= 0) || (sz > ifcon_drv_cur_total_data_size)) {
		LOG_DEV_ERR("pointer size error! sz:%d buf:0x%p\n",
					sz, buf);
		return IFCON_DRV_NG;
	}
	if (sz + used_sz > ifcon_drv_cur_total_data_size)
		return 0;	/* Not enter this frame */
	if (count < sz) {
		/* Don't fit in this frame */
		return -IFCON_DRV_RX_BUFF_SIZE_ERROR;
	}
	if (rptr_temp >= IFCON_DRV_RING_BUFF_SIZE) { /* Pointer inside check */
		LOG_DEV_ERR("tx_ring.rptr error! %d\n", rptr_temp);
		return IFCON_DRV_NG;
	}

	/* Read from ring buffer Rx */
	if (sz >= rest_cap) {
		if (rptr_temp < wptr_temp) {
			LOG_DEV_ERR("err! r:%d w:%d\n", rptr_temp, wptr_temp);
			return IFCON_DRV_NG;
		}
#if DEBUG_RINGEDGE
		LOG_INFO("tx rd (sz >= rest_cap)sz:%d rest_cap:%d\n",
							sz, rest_cap);
#endif
		memcpy(buf, &tx_ring.buffer[rptr_temp], rest_cap);
		rptr_temp = 0;
		sz -= rest_cap;
		buf = (char *)(buf + rest_cap);
	}
	memcpy(buf, &tx_ring.buffer[rptr_temp], sz);
	/* Read Pointer increment */
	barrier();
	tx_ring.rptr = rptr_temp + sz;

	tx_wait_send_bytes = calc_wait_bytes(tx_ring.wptr, tx_ring.rptr);

	return sz_total;
}

/* Write to ring buffer */
static int tx_ring_write(const char __user *buf, int count)
{
	int wptr_temp, vptr_temp;
	int rest_cap, rest;
	int sz, size;

	wptr_temp = tx_ring.wptr;
	vptr_temp = tx_vptr;
	rest_cap = IFCON_DRV_RING_BUFF_SIZE - wptr_temp;
	sz = count;
	rest = 0;

	size = get_buff_size(buf);
	if (count != size) {
		LOG_DEV_ERR("c!=sz err c:%04x sz:%04x\n", count, size);
		return IFCON_DRV_NG;
	}
	if ((sz <= 0) || (sz > ifcon_drv_cur_total_data_size)) {
		LOG_DEV_ERR("pointer size error! count:%d buf:0x%p\n",
					sz, buf);
		return IFCON_DRV_NG;
	}
	if (wptr_temp >= IFCON_DRV_RING_BUFF_SIZE) { /* Pointer inside check */
		LOG_DEV_ERR("tx_ring.wptr error! %d\n", wptr_temp);
		return IFCON_DRV_NG;
	}

	/* rest writable buffer size */
	if (vptr_temp <= wptr_temp)
		rest = IFCON_DRV_RING_BUFF_SIZE - (wptr_temp - vptr_temp);
	else
		rest = vptr_temp - wptr_temp;
	/* Flow control */
	if (rest < ifcon_drv_cur_total_data_size) {
		LOG_DEV_ERR("BUFF_FULL! size:%d rest:%d\n", sz, rest);
		return -IFCON_DRV_BUFF_FULL;
	}
	/* Check size over */
	if (sz > rest) {
		LOG_DEV_ERR("size over! size:%d > rest:%d\n", sz, rest);
		return -IFCON_DRV_BUFF_FULL;
	}

	/* Write to ring buffer Tx */
	if (sz >= rest_cap) {
#if DEBUG_RINGEDGE
		LOG_INFO("tx wr (sz >= rest_cap)sz:%d rest_cap:%d\n",
							sz, rest_cap);
#endif
		/* Write to end edge */
		memcpy(&tx_ring.buffer[wptr_temp], buf, rest_cap);
		wptr_temp = 0;
		sz -= rest_cap;
		buf = (char *)(buf + rest_cap);
	}
	memcpy(&tx_ring.buffer[wptr_temp], buf, sz);
	/* Write Pointer increment */
	barrier();
	tx_ring.wptr = wptr_temp + sz;

	tx_wait_send_bytes = calc_wait_bytes(tx_ring.wptr, tx_ring.rptr);

	return count;
}

/* Read from ring buffer Tx */
static int copy_from_tx_ring_to_frame(char *buf)
{
	int sz_total, sz;
	char *buf_next;

	sz = ifcon_drv_cur_total_data_size;
	sz_total = 0;
	buf_next = buf;

#if DEBUG_RINGEDGE
	LOG_INFO("tx_wptr:%d tx_rptr:%d\n", tx_ring.wptr, tx_ring.rptr);
	LOG_INFO("rx_wptr:%d rx_rptr:%d\n", rx_ring.wptr, rx_ring.rptr);
	LOG_INFO("tx_xptr:%d tx_vptr:%d\n", tx_xptr, tx_vptr);
#endif
	/* Read from ring buffer Tx */
	while (sz > 0) {
		sz = tx_ring_read(buf_next,
			(ifcon_drv_cur_total_data_size - sz_total), sz_total);
		if (sz > 0)
			sz_total += sz;
		buf_next = buf + sz_total;
	}

	return sz_total;
}

static int set_tx_frame(void)
{
	int status = 0;
	u8 flow_check = 0;	/* flow check */
	u8 rx_flow;

	rx_flow = rx_frame->flow_ctrl & IFCON_DRV_FLOW_ALL; /*Another side */

	/* Set controll Area: status ->	check_ctrl_data) */
	/* Set controll Area: count  ->	set_counter() */
	/* Set controll Area: flow */
	if (rx_flow == IFCON_DRV_FLOW_FULL) {	/* (0xC0) */
		flow_check = IFCON_DRV_FLOW_FOR_CHECK;	/* (0x01) */
		warn_rx_tx_flow("detect FLOW_FULL!:");
	} else if (rx_flow == IFCON_DRV_FLOW_NEAR_FULL) {	/* (0x80) */
		if ((tx_frame->flow_ctrl & IFCON_DRV_FLOW_FOR_CHECK) == 0)
			flow_check = IFCON_DRV_FLOW_FOR_CHECK; /* alternately */
		warn_rx_tx_flow("detect FLOW_NEAR_FULL!:");
	} else if (rx_flow == IFCON_DRV_FLOW_PACKET_LOST) {	/* (0xE0) */
		resend_req();
		tx_frame->status = IFCON_DRV_CTRL_RESENDING;
		warn_rx_tx_flow("detect FLOW_PACKET_LOST!:");
	} else { /* if (rx_flow == IFCON_DRV_FLOW_NOT_FULL) { */ /* (0x00) */
		if (tx_frame->flow_ctrl == 0)
			ifcon_drv_dev->count_flow = 0;
	}
	tx_frame->flow_ctrl = get_flow_code(&rx_ring) | flow_check;
	if ((tx_frame->flow_ctrl & IFCON_DRV_FLOW_ALL) != 0)
		warn_rx_tx_flow("Tx FLOW_CTRL:");

	tx_frame->frame_type = ifcon_drv_cur_frame_type;
	switch (tx_frame->frame_type) {
	case IFCON_DRV_FRAME_TYPE0:
	default:
		tx_frame->ctrl_check_sum = 0;
		break;
	case IFCON_DRV_FRAME_TYPE1:
		tx_frame->ctrl_check_sum = calc_ctrl_check_sum((u8 *)tx_frame);
		break;
	}

	/* Set data from ring buffer Tx */
	clear_frame_data(tx_frame);
	if ((flow_check != IFCON_DRV_FLOW_FOR_CHECK)
		&& (ifcon_drv_status > IFCON_DRV_STAT_INIT))
		status = copy_from_tx_ring_to_frame(get_frame_data_p(tx_frame));
	if (status < 0)
		LOG_DEV_ERR(" Fatal error! status:%d\n", status);

	/* Set check sum tx */
	set_check_sum(tx_frame);

	return status;
}

/*--- Access for SPI device --------------------------------------------------*/
#if TEST_MODE
static int random(void)
{
	int rd;

	get_random_bytes(&rd, sizeof(int));
	return rd;
}
#endif

/* The method of transferring SPI data that using spi_***() APIs	*/
/* are based on kernel/drivers/misc/mediatek/spi/mt8590/spi-dev.c	*/
static inline int
sync_write_and_read(struct ifcon_drv_data_t *spidev, int len)
{
	struct spi_transfer trn = {
	/*
	 * from Documentation/spi/spidev
	 * To make a full duplex request, provide both rx_buf
	 * and tx_buf for the same transfer.
	 * It's even OK if those are the same buffer.
	 */
		.tx_buf	= tx_frame,
		.rx_buf	= tmp_frame,
		.len	= len,
		.tx_dma	= tx_phyadr,
		.rx_dma	= tmp_phyadr,
		.speed_hz	= IFCON_DRV_SPI_SPEED_HZ,
	};
	struct spi_message msg;
	int status;

#if TEST_MODE
	u8 *pos = NULL;
	u8 bup;
	int eps;

	if (test_mode == TEST_MODE_TX) {
		if ((random() & 0xFF) == 0) {
			LOG_DBG("emulation : send data broken !!!\n");
			eps = (random() & 0x7fff) % IFCON_SPI_FRAME_SIZE;
			pos = (u8 *)tx_frame;
			bup = pos[eps];
			pos[eps] = (random() & 0xff);
		}
	}
#endif

	if (spidev->spi == NULL)
		status = -ESHUTDOWN;
	else {
		/* byte order convert */
		ifcon_drv_cpu_to_be32(tx_frame, tx_frame, IFCON_SPI_FRAME_SIZE);
		spi_message_init(&msg);
		msg.is_dma_mapped = 1;
		spi_message_add_tail(&trn, &msg);

		dma_sync_single_for_device(NULL,
			tx_phyadr, IFCON_SPI_FRAME_SIZE, DMA_BIDIRECTIONAL);
		status = spi_sync(spidev->spi, &msg);
		dma_sync_single_for_cpu(NULL,
			tmp_phyadr, IFCON_SPI_FRAME_SIZE, DMA_BIDIRECTIONAL);

		/* byte order convert */
		ifcon_drv_be32_to_cpu(tmp_frame, tmp_frame,
							  IFCON_SPI_FRAME_SIZE);
		ifcon_drv_be32_to_cpu(tx_frame, tx_frame, IFCON_SPI_FRAME_SIZE);
		if (status == 0) {
			memcpy(rx_frame, tmp_frame, IFCON_SPI_FRAME_SIZE);
			status = msg.actual_length;
		}
	}

#if TEST_MODE
	if (test_mode == TEST_MODE_RX) {
		if (pos != NULL)
			pos[eps] = bup;

		if ((random() & 0xFF) == 0) {
			LOG_DBG("emulation : receive data broken !!!\n");
			eps = (random() & 0x7fff) % IFCON_SPI_FRAME_SIZE;
			pos = (__u8 *)rx_frame;
			pos[eps] = (random() & 0xff);
		} else if ((random() & 0x3FFF) == 0) {
			LOG_DBG("emulation : no receive data !!!\n");
			memset((__u8 *)rx_frame, 0, IFCON_SPI_FRAME_SIZE);
		}
	}
#endif

	return status;
}

static void set_counter(void)
{
	if (tx_frame->count >= IFCON_DRV_COUNTER_MAX)
		tx_frame->count = 0;
	else
		tx_frame->count++;
}

static void spi_communication(struct ifcon_drv_data_t *ifcon_drv_dev)
{
	int status = 0;

	status = set_tx_frame();

	if (dump_mode & DUMP_MODE_SPKT)
		ifcon_mem_dump("Tx", (char *)tx_frame,
						ifcon_drv_cur_frame_all_size);
	else if (dump_mode & DUMP_MODE_SPKT_P1)
		ifcon_mem_dump("Tx dt...", (char *)&tx_frame->t0.data[0], 0x10);

#if DEBUG_LOG
	if (tx_frame->status == IFCON_DRV_CTRL_RESENDING)
		ifcon_display_frame("SNDRESENDING:", tx_frame);
	if (tx_frame->status == IFCON_DRV_CTRL_RESEND_REQ)
		ifcon_display_frame("SNDRESENDREQ:", tx_frame);
	if (tx_frame->flow_ctrl > 0)
		ifcon_display_frame("SNDFLW:", tx_frame);
#endif

	status = sync_write_and_read(ifcon_drv_dev,
		ifcon_drv_cur_frame_all_size);
	if (status < 0) {
		/*
		 * spi transfer errors
		 * -EINVAL: include status of 'sdma firmware not ready!'
		 *          ='failed to transfer one message from queue'
		 */
		return;
	} else if (status != ifcon_drv_cur_frame_all_size)
		LOG_DEV_ERR("transfer size error! status:%d\n", status);

	if (dump_mode & DUMP_MODE_RPKT)
		ifcon_mem_dump("Rx", (char *)rx_frame,
						ifcon_drv_cur_frame_all_size);
	else if (dump_mode & DUMP_MODE_RPKT_P1)
		ifcon_mem_dump("Rx dt...", (char *)&rx_frame->t0.data[0], 0x10);

	status = get_rx_frame();

	set_counter();	/* Process after Tx */
	transfers++;
}

/*---  main routine  ----------------------------------------------*/
static int spi_comm_thread(void *arg)
{
	ktime_t start_time;
	long delta_msec;
	long sum_msec = 0;
	long sum_count = 0;

	LOG_INFO("%s start\n", __func__);
	while (!kthread_should_stop()) {
		start_time = ktime_get();
		set_current_state(TASK_INTERRUPTIBLE);
		schedule_timeout(msecs_to_jiffies(cycle_msec_val));
		if (kthread_should_stop())
			break;

		if (ifcon_status) {
			ifcon_status = 0;
			display_internal_state();
		}

		if (ifcon_drv_power == IFCON_DRV_POWER_SUSPEND)
			continue; /* for suspend */

		mutex_lock(&ifcon_drv_dev->state_lock);
		spi_communication((struct ifcon_drv_data_t *)arg);
		mutex_unlock(&ifcon_drv_dev->state_lock);

		delta_msec = ktime_ms_delta(/* round up */
			ktime_add_us(ktime_get(), 500), start_time);
		if (delta_msec > cycle_msec_max)
			cycle_msec_max = delta_msec;
		sum_msec += delta_msec;
		sum_count++;
		if (sum_msec >= IFCON_DRV_CYCLE_MSEC_AVG_T) {
			cycle_msec_avg = sum_msec / sum_count;
			sum_msec = 0;
			sum_count = 0;
		}
	}
	LOG_INFO("%s end\n", __func__);
	return 0;
}

static void spi_comm_reset(void)
{
	mutex_lock(&ifcon_drv_dev->state_lock);

	/* Initialize ring buffer */
	tx_xptr = 0;
	tx_vptr = 0;
	memset(&tx_ring, 0, sizeof(tx_ring));
	memset(&rx_ring, 0, sizeof(rx_ring));
	tx_wait_send_bytes = 0;
	rx_remain_bytes = sizeof(rx_ring.buffer);

	/* init communication */
	tx_frame->status = IFCON_DRV_CTRL_START_CONDITION;
	tx_frame->flow_ctrl = IFCON_DRV_FLOW_NOT_FULL;
	ifcon_drv_status = IFCON_DRV_STAT_INIT;
	ifcon_drv_boot = 0;
	ifcon_drv_loss = 0;

	mutex_unlock(&ifcon_drv_dev->state_lock);
}

/*--- System call ---------------------------------------------------------*/
static ssize_t ifcon_drv_read(struct file *filp, char __user *buf,
					size_t count, loff_t *offset)
{
	ssize_t status = 0;

	//LOG_V("%s enter\n", __func__);
	ifcon_drv_rblock = IFCON_DRV_READ_BLOCKING;
	while ((rx_ring.rptr == rx_ring.wptr
	     || ifcon_drv_status < IFCON_DRV_STAT_CONNECT)
	    && ifcon_drv_rblock == IFCON_DRV_READ_BLOCKING
	    && ifcon_drv_power != IFCON_DRV_POWER_SUSPEND) {
		usleep_range(5000, 6000);
	}
	ifcon_drv_rblock = IFCON_DRV_READ_NO_BLOCK;

	if (ifcon_drv_power == IFCON_DRV_POWER_SUSPEND)
		return 0;

	/* If communication reset */
	if ((ifcon_drv_status < IFCON_DRV_STAT_CONNECT) &&
	    (ifcon_drv_dev->count_stat < IFCON_DRV_WARN_COUNT_MAX)) {
		ifcon_drv_dev->count_stat++;
		LOG_DEV_ERR(" com reset :IFCON_DRV_NOT_CONNECT error!\n");
		return -IFCON_DRV_NOT_CONNECT;
	}

	if (frame_type != ifcon_drv_cur_rx_frame_type) {
		LOG_DEV_ERR("Bad frame type error! rx type=%d\n",
						ifcon_drv_cur_rx_frame_type);
		return -IFCON_DRV_BADTYPE;
	}

	/* Read from ring buffer Rx */
	status = (ssize_t)rx_ring_read(buf, (int)count);

	if ((dump_mode & DUMP_MODE_R) && (status > 0))
		ifcon_mem_dump("rd:", buf, (int)status);

	//LOG_V("%s status:%ld\n", __func__, status);
	return status;
}

static ssize_t ifcon_drv_write(struct file *filp, const char __user *buf,
					size_t count, loff_t *offset)
{
	ssize_t status = 0;
	u8 tmp[IFCON_DRV_T1_RW_DATA_SIZE];

	if (ifcon_drv_power == IFCON_DRV_POWER_SUSPEND)
		return -ENETDOWN;

	//LOG_V("%s count 0x%lx\n", __func__, count);
	if (count > ifcon_drv_cur_total_data_size) {
		LOG_DEV_ERR("count > RW_DATA_SIZE error!\n");
		return -EMSGSIZE;
	}

	if (frame_type != ifcon_drv_cur_rx_frame_type) {
		LOG_DEV_ERR("Bad frame type error! rx type=%d\n",
						ifcon_drv_cur_rx_frame_type);
		return -IFCON_DRV_BADTYPE;
	}

	if (copy_from_user(tmp, (void *)buf, count)) {
		LOG_DEV_ERR("copy_from_user error\n");
		return -EFAULT;
	}

	status = (ssize_t)tx_ring_write(tmp, (int)count);

	if ((dump_mode & DUMP_MODE_W) && (status > 0))
		ifcon_mem_dump("wr:", tmp, (int)count);

	//LOG_V("%s exit status:%ld\n", __func__, status);
	return status;
}


static long ifcon_drv_ioctl(struct file *filp, unsigned int cmd,
						unsigned long arg)
{
	int err = 0;
	u32 missing = 0;
	struct st_ifcon_fixed_data_param_def st_param;

	LOG_V("%s enter\n", __func__);
	/* Check type and command number */
	if (_IOC_TYPE(cmd) != IFCON_DRV_IOC_MAGIC)
		return -ENOTTY;

	if (_IOC_DIR(cmd) & _IOC_READ)
		err = !access_ok(VERIFY_WRITE, (void __user *)arg,
							_IOC_SIZE(cmd));
	if (err == 0 && _IOC_DIR(cmd) & _IOC_WRITE)
		err = !access_ok(VERIFY_READ, (void __user *)arg,
							_IOC_SIZE(cmd));
	if (err)
		return -EFAULT;

	missing = copy_from_user(&st_param, (void *)arg, sizeof(st_param));
	if (missing != 0)
		return -EFAULT;
	/* Check size */
	if ((st_param.offset + st_param.sz) > IFCON_DRV_FIXED_DATA_SIZE)
		return -EFAULT;

	switch (cmd) {
	case IFCON_DRV_IOC_WRITE_FIXED:
		missing = copy_from_user(
			&tx_frame->fixed_data[st_param.offset],
						st_param.data, st_param.sz);
		break;
	case IFCON_DRV_IOC_READ_FIXED:
		missing = copy_to_user(st_param.data,
			&rx_frame->fixed_data[st_param.offset], st_param.sz);
		break;
	default:
		break;
	}

	if (missing != 0) {
		LOG_DEV_DBG("exit missing:%d\n", (unsigned int)missing);
		return -EFAULT;
	}
	return 0;
}

static long ifcon_drv_compat_ioctl(struct file *filp, unsigned int cmd,
							unsigned long arg)
{
	return ifcon_drv_ioctl(filp, cmd, arg);
}

static unsigned int ifcon_drv_poll(struct file *filp, poll_table *ptbl)
{
	struct ifcon_drv_data_t *ddt;
	unsigned int mask = 0;

	ddt = filp->private_data;

	//LOG_V("%s enter\n", __func__);
	if (ddt == NULL)
		return -EBADFD;

	if (rx_ring.rptr == rx_ring.wptr)
		poll_wait(filp, &ddt->rxwait, ptbl);

	if (rx_ring.rptr != rx_ring.wptr)
		mask |= POLLIN | POLLRDNORM;

	if (ifcon_drv_status == IFCON_DRV_STAT_CONNECT)
		mask |= POLLOUT | POLLWRNORM;

	//LOG_V("%s status:0x%x\n", __func__, mask);
	return mask;
}

static int ifcon_drv_open(struct inode *inode, struct file *filp)
{
	int status = 0;
	int timeout_cnt;

	LOG_DBG("%s enter wait\n", __func__);
	timeout_cnt = IFCON_DRV_OPEN_WAIT_CNT;
	while (ifcon_drv_setup_status != IFCON_DRV_SETUP_OK) {
		if (timeout_cnt > 0) {
			timeout_cnt--;
			set_current_state(TASK_INTERRUPTIBLE);
			schedule_timeout(IFCON_DRV_OPEN_WAIT);
		} else {
			LOG_ERR("ifcon_drv_setup_status = NG\n");
			status = IFCON_DRV_NG;
			return status;
		}
	}

	LOG_DBG("%s wait end\n", __func__);

	/* Only 1 user */
	if (ifcon_drv_dev->users > 0) {
		LOG_ERR("Already open! %d\n", ifcon_drv_dev->users);
		status = IFCON_DRV_NG;
		goto ifcon_drv_open_err;
	}

	ifcon_drv_dev->users++;

	ifcon_drv_cur_frame_type = frame_type;
	ifcon_drv_cur_rx_frame_type = frame_type;
	LOG_INFO("Open frame type=%d", frame_type);
	if (frame_type == IFCON_DRV_FRAME_TYPE1) {
		ifcon_drv_cur_total_data_size = IFCON_DRV_T1_RW_DATA_SIZE;
		ifcon_drv_cur_frame_all_size = IFCON_DRV_FRAME_BASE_SIZE
					+ ifcon_drv_cur_total_data_size;
	} else {
		frame_type = 0;
		ifcon_drv_cur_total_data_size = IFCON_DRV_RW_DATA_SIZE;
		ifcon_drv_cur_frame_all_size = IFCON_DRV_FRAME_BASE_SIZE
					+ ifcon_drv_cur_total_data_size;
	}

	filp->private_data = ifcon_drv_dev;
	nonseekable_open(inode, filp);

	/* reset spi tx/rx buffer */
	spi_comm_reset();

	/* Create & wakeup own thread */
	ifcon_drv_dev->kthread_tsk = kthread_run(spi_comm_thread,
					(void *)ifcon_drv_dev, "ifcon_drv_tsk");
	if (IS_ERR(ifcon_drv_dev->kthread_tsk)) {
		LOG_ERR("error! ifcon_drv_dev->kthread_tsk: 0x%p\n",
				ifcon_drv_dev->kthread_tsk);
		status = IFCON_DRV_NG;
		goto ifcon_drv_open_err;
	}

	LOG_DBG("%s exit\n", __func__);
	return status;

ifcon_drv_open_err:
	LOG_ERR("exit error\n");
	return status;
}

static int ifcon_drv_release(struct inode *inode, struct file *filp)
{
	LOG_DBG("%s enter\n", __func__);

	ifcon_drv_dev->users--;
	if (ifcon_drv_dev->kthread_tsk) {
		kthread_stop(ifcon_drv_dev->kthread_tsk);
		ifcon_drv_dev->kthread_tsk = NULL;
	}
	filp->private_data = NULL;

	LOG_DBG("%s exit\n", __func__);
	return 0;
}

/*-------------------------------------------------------------------------*/
static const struct file_operations ifcon_drv_fops = {
	.owner =	THIS_MODULE,
	.read =		ifcon_drv_read,
	.write =	ifcon_drv_write,
	.unlocked_ioctl = ifcon_drv_ioctl,
	.compat_ioctl = ifcon_drv_compat_ioctl,
	.poll =		ifcon_drv_poll,
	.open =		ifcon_drv_open,
	.release =	ifcon_drv_release,
};

/*-------------------------------------------------------------------------*/
static void ifcon_drv_suspend_common(void)
{
	int ret;
	unsigned long timer_expires;

	if (ifcon_drv_power == IFCON_DRV_POWER_SUSPEND)
		return;

	timer_expires = jiffies
	 + msecs_to_jiffies(IFCON_DRV_SUSPEND_TIMEOUT);
	while (tx_wait_send_bytes > 0) {
		/*
		 * Wait for the time when transmission in the buffer
		 * is expected to finish
		 */
		usleep_range(5000, 15000);
		if (time_after(jiffies, timer_expires))
			break;
	}
	if (tx_wait_send_bytes > 0)
		LOG_DEV_WAR("LOST tx data len=%d\n", tx_wait_send_bytes);

	ifcon_drv_power = IFCON_DRV_POWER_SUSPEND;
	if (ifcon_drv_dev->kthread_tsk) {
		kthread_stop(ifcon_drv_dev->kthread_tsk);
		ifcon_drv_dev->kthread_tsk = NULL;
	}

	ret = spi_master_suspend(ifcon_drv_dev->spi->master);
	if (ret)
		LOG_ERR("error! spi_master_suspend: %d\n", ret);

	/* to IFCON stop mode */
	ifcon_bd_req_off();
}

static int ifcon_drv_suspend(struct device *dev)
{
	LOG_DBG("%s enter\n", __func__);

	LOG_DBG("%s exit\n", __func__);
	return 0;
}

static int ifcon_drv_resume_common(void)
{
	int status = 0;
	int ret;

	if (ifcon_drv_power != IFCON_DRV_POWER_SUSPEND)
		return status;

	/* to IFCON wake mode */
	ifcon_bd_req_on();

	/* reset spi tx/rx buffer */
	spi_comm_reset();

	if (ifcon_drv_dev->spi->master->running == false) {
		ret = spi_master_resume(ifcon_drv_dev->spi->master);
		if (ret)
			LOG_ERR("error! spi_master_resume: %d\n", ret);
	}

	if (ifcon_drv_dev->users > 0) {
		ifcon_drv_dev->kthread_tsk = kthread_run(spi_comm_thread,
					(void *)ifcon_drv_dev, "ifcon_drv_tsk");
		if (IS_ERR(ifcon_drv_dev->kthread_tsk)) {
			LOG_ERR("error! ifcon_drv_dev->kthread_tsk: 0x%p\n",
				ifcon_drv_dev->kthread_tsk);
			status = IFCON_DRV_NG;
			goto ifcon_drv_resume_common_err;
		}
	}

	ifcon_drv_power = IFCON_DRV_POWER_NORMAL;

	return status;

ifcon_drv_resume_common_err:
	LOG_ERR("exit error status=%d\n", status);
	return status;
}

static int ifcon_drv_resume(struct device *dev)
{
	LOG_DBG("%s enter\n", __func__);

	LOG_DBG("%s exit\n", __func__);
	return 0;
}

/*--- runtime power management ----------------------------------------------*/

static int ifcon_drv_runtime_suspend(struct device *dev)
{
	LOG_DBG("%s enter\n", __func__);

	ifcon_drv_suspend_common();

	LOG_DBG("%s exit\n", __func__);
	return 0;
}

static int ifcon_drv_runtime_resume(struct device *dev)
{
	int status;

	LOG_DBG("%s enter\n", __func__);

	status = ifcon_drv_resume_common();

	LOG_DBG("%s exit\n", __func__);
	return 0;
}

/*-------------------------------------------------------------------------*/

static int ifcon_drv_probe(struct spi_device *spi)
{
	int status = 0;
	u32 minor = 0;
	int ret;

	LOG_V("%s enter\n", __func__);
	ifcon_drv_dev->spi = spi;

	#ifndef CONFIG_OF
		#error CONFIG_OF=Y required.
	#endif /* CONFIG_OF */

	/* get platform data */
	if (spi->dev.of_node) {

		ret = of_dma_configure(&spi->dev, spi->dev.of_node);
		if (ret) {
			LOG_ERR("of_dma_configure error! %d\n", ret);
			goto out_err_exit;
		}

		tx_frame = (struct ifcon_drv_frame *)devm_kzalloc(&spi->dev,
			IFCON_SPI_FRAME_SIZE * 3, GFP_KERNEL);
		if (!tx_frame) {
			rx_frame = tx_frame;
			tmp_frame = rx_frame;
			LOG_ERR("devm_kzalloc error! %p\n", tx_frame);
			status = -ENOMEM;
			goto out_err_exit;
		} else {
			tx_phyadr = virt_to_phys(tx_frame);
			tmp_frame = (struct ifcon_drv_frame *)
				((void *)tx_frame + IFCON_SPI_FRAME_SIZE);
			tmp_phyadr = tx_phyadr + IFCON_SPI_FRAME_SIZE;
			rx_frame = (struct ifcon_drv_frame *)
				((void *)tx_frame + IFCON_SPI_FRAME_SIZE * 2);
			LOG_INFO("tx_frame alloc vadr %p padr %p\n",
				tx_frame, (void *)tx_phyadr);
			LOG_INFO("tmp_frame alloc vadr %p padr %p\n",
				tmp_frame, (void *)tmp_phyadr);
		}

		if (of_property_read_u32(spi->dev.of_node,
			"spi-max-frequency", &ifcon_drv_dev->max_freq)) {
			/* default: 1MHz (Low speed) */
			ifcon_drv_dev->max_freq = 1000000;
			LOG_ERR("spi-max-frequency not found!\n");
		}
		LOG_INFO("spi-max-frequency=%d\n", ifcon_drv_dev->max_freq);

		GPIO_IFCON_BD_REQ_PIN = of_get_named_gpio(spi->dev.of_node,
			"bd_req-gpios", 0);
		if (GPIO_IFCON_BD_REQ_PIN < 0) {
			LOG_ERR("bd_req-gpios not found error!\n");
			status = -ENOMEM;
		}
		LOG_INFO("bd_req-gpios=%d\n", GPIO_IFCON_BD_REQ_PIN);
		GPIO_IFCON_RESET_PIN = of_get_named_gpio(spi->dev.of_node,
			"if_reset-gpios", 0);
		if (GPIO_IFCON_RESET_PIN < 0) {
			LOG_ERR("if_reset-gpios not found error!\n");
			status = -ENOMEM;
		}
		LOG_INFO("if_reset-gpios=%d\n", GPIO_IFCON_RESET_PIN);
		GPIO_WATCHDOG_PIN = of_get_named_gpio(spi->dev.of_node,
			"watchdog-gpios", 0);
		if (GPIO_WATCHDOG_PIN < 0) {
			LOG_ERR("watchdog-gpios not found error!\n");
			status = -ENOMEM;
		}
		LOG_INFO("watchdog-gpios=%d\n", GPIO_WATCHDOG_PIN);
		if (status == -ENOMEM)
			goto out_err_exit;
	} else {
		LOG_ERR("device tree not found error!\n");
		status = -ENODEV;
		goto out_err_exit;
	}

	/* Initialize the driver data */
	init_waitqueue_head(&ifcon_drv_dev->rxwait);

	/*
	 * If we can allocate a minor number, hook up this device.
	 * Reusing minors is fine so long as udev or mdev is working.
	 */
	minor = find_first_zero_bit(minors, IFCON_DRV_MINORS);
	if (minor < IFCON_DRV_MINORS) {
		ifcon_drv_dev->devt =
			MKDEV(MAJOR(ifcon_drv_dev->major), minor);
		set_bit(minor, minors);

		spi_set_drvdata(spi, ifcon_drv_dev);
	} else {
		LOG_ERR("no minor number available error!\n");
		status = -ENODEV;
		goto out_err_exit;
	}

	/* spi settings */
	spi->bits_per_word = IFCON_DRV_SPI_BITS;
	spi->mode = SPI_MODE_0;
	spi->master->min_speed_hz = 1000;
	spi->master->max_speed_hz = ifcon_drv_dev->max_freq;
	status = spi_setup(spi);
	if (status < 0) {
		LOG_ERR("spi_setup error! status=%d\n", status);
		status = -ENODEV;
		goto out_err_exit;
	} else {
		LOG_INFO("speed_hz=%dHz %dbits\n",
			IFCON_DRV_SPI_SPEED_HZ, spi->bits_per_word);
	}

	/* reset spi tx/rx buffer */
	spi_comm_reset();

	ifcon_drv_dev->users = 0;
	ifcon_drv_power = IFCON_DRV_POWER_SUSPEND;
	ifcon_drv_setup_status = IFCON_DRV_SETUP_OK;

	pm_runtime_enable(&ifcon_drv_dev->spi->dev);
	/* .../power/control = on and call runtime_resume */
	pm_runtime_forbid(&ifcon_drv_dev->spi->dev);

	LOG_V("%s exit\n", __func__);
	return status;

out_err_exit:
	if (tx_frame != NULL)
		devm_kfree(&spi->dev, tx_frame);
	rx_frame  = NULL;
	tmp_frame  = NULL;
	tmp_phyadr = 0;
	tx_frame  = NULL;
	tx_phyadr = 0;
	LOG_ERR("exit error\n");
	return status;
}

static int ifcon_drv_remove(struct spi_device *spi)
{
	LOG_V("%s enter\n", __func__);

	ifcon_drv_setup_status = IFCON_DRV_SETUP_NG;

	if (ifcon_drv_dev->kthread_tsk) {
		kthread_stop(ifcon_drv_dev->kthread_tsk);
		ifcon_drv_dev->kthread_tsk = NULL;
	}

	if (pm_runtime_active(&ifcon_drv_dev->spi->dev)) {
		LOG_ERR("runtime still active\n");
		pm_runtime_allow(&ifcon_drv_dev->spi->dev);
		/* call runtime_suspend() */
	}
	pm_runtime_disable(&ifcon_drv_dev->spi->dev);

	spi_set_drvdata(spi, NULL);
	ifcon_drv_dev->spi = NULL;

	device_destroy(ifcon_drv_dev->class, ifcon_drv_dev->devt);
	clear_bit(MINOR(ifcon_drv_dev->devt), minors);

	if (tx_frame != NULL)
		devm_kfree(&spi->dev, tx_frame);
	rx_frame  = NULL;
	tmp_frame  = NULL;
	tmp_phyadr = 0;
	tx_frame  = NULL;
	tx_phyadr = 0;

	LOG_V("%s exit\n", __func__);
	return 0;
}

static void ifcon_drv_shutdown(struct spi_device *spi)
{
	LOG_INFO("shutdown called\n");

	ifcon_drv_suspend(&spi->dev);
}

/*-------------------------------------------------------------------------*/
static const struct of_device_id ifcon_dev_dt_ids[] = {
	{ .compatible = "sony,ifcon_dev", },
	{ /* sentinel */ }
};
MODULE_DEVICE_TABLE(of, ifcon_dev_dt_ids);

static const struct dev_pm_ops ifcon_dev_pmops = {
	.suspend = ifcon_drv_suspend,
	.resume  = ifcon_drv_resume,
	.runtime_suspend = ifcon_drv_runtime_suspend,
	.runtime_resume  = ifcon_drv_runtime_resume,
};

static struct spi_driver ifcon_driver_st = {
	.driver = {
		.name  = IFCON_DRV_DEVNAME,
		.owner = THIS_MODULE,
		.of_match_table = ifcon_dev_dt_ids,
		.pm    = &ifcon_dev_pmops,
	},
	.probe  = ifcon_drv_probe,
	.remove = ifcon_drv_remove,
	.shutdown = ifcon_drv_shutdown,
};

/*-------------------------------------------------------------------------*/
static int __init ifcon_drv_init(void)
{
	int status = 0;

	ifcon_drv_setup_status = IFCON_DRV_SETUP_NG;

	/* Allocate driver data */
	ifcon_drv_dev = kzalloc(sizeof(struct ifcon_drv_data_t), GFP_KERNEL);
	if (!ifcon_drv_dev) {
		status = -ENOMEM;
		goto out_alloc_err;
	}

	mutex_init(&ifcon_drv_dev->state_lock);
	rx_frame  = NULL;
	tmp_frame  = NULL;
	tmp_phyadr = 0;
	tx_frame  = NULL;
	tx_phyadr = 0;
	ifcon_drv_dev->users = 0;

	/* Register SPI device */
	status = alloc_chrdev_region(&ifcon_drv_dev->major, 0, 1, "spi");
	if (status) {
		LOG_ERR("alloc_chrdev_region() error:0x%x\n", status);
		goto out_free;
	}
	cdev_init(&ifcon_drv_dev->cdev, &ifcon_drv_fops);
	ifcon_drv_dev->cdev.owner = THIS_MODULE;

	status = cdev_add(&ifcon_drv_dev->cdev, ifcon_drv_dev->major, 1);
	if (status) {
		LOG_ERR("cdev_add() error:0x%x\n", status);
		goto out_unregister;
	}

	/* Create ifcon_drv device class */
	ifcon_drv_dev->class = class_create(THIS_MODULE, IFCON_DRV_DEVNAME);
	if (IS_ERR(ifcon_drv_dev->class)) {
		status = PTR_ERR(ifcon_drv_dev->class);
		LOG_ERR("class_create() error:0x%x\n", status);
		goto out_cdev_del;
	}

	ifcon_drv_dev->dev = device_create(ifcon_drv_dev->class, NULL,
				ifcon_drv_dev->major, NULL, IFCON_DRV_DEVNAME);
	if (IS_ERR(ifcon_drv_dev->dev)) {
		status = PTR_ERR(ifcon_drv_dev->dev);
		LOG_ERR("device_create() error:0x%x\n", status);
		goto out_class_destroy;
	}

	status = spi_register_driver(&ifcon_driver_st);
	if (status < 0) {
		LOG_ERR("spi_register_driver() error:0x%x\n", status);
		goto out_class_destroy;
	}

	return status;

out_class_destroy:
	class_destroy(ifcon_drv_dev->class);
out_cdev_del:
	cdev_del(&ifcon_drv_dev->cdev);
out_unregister:
	unregister_chrdev_region(ifcon_drv_dev->major, 1);
out_free:
	kfree(ifcon_drv_dev);
out_alloc_err:
	LOG_ERR("exit err status=%d\n", status);
	return status;
}

static void __exit ifcon_drv_exit(void)
{
	LOG_V("%s enter\n", __func__);

	spi_unregister_driver(&ifcon_driver_st);
	class_destroy(ifcon_drv_dev->class);
	cdev_del(&ifcon_drv_dev->cdev);
	unregister_chrdev_region(ifcon_drv_dev->major, 1);
	kfree(ifcon_drv_dev);

	LOG_V("%s exit\n", __func__);
}

/*-------------------------------------------------------------------------*/
module_init(ifcon_drv_init);
module_exit(ifcon_drv_exit);

MODULE_AUTHOR("Sony Corporation");
MODULE_DESCRIPTION("IfCon driver");
MODULE_VERSION("2.1.0");
MODULE_LICENSE("GPL");
MODULE_ALIAS("spi:" IFCON_DRV_DEVNAME);
